<?php // -*- mode: php -*-
/*
 * Slooze PHP Web Photo Album
 * Copyright (c) 2000 Matthew Kendall
 *
 * Container (Ct) class holding the photo data. Exactly where the data is
 * stored depends on which Ct class you use. In this implementation, the data
 * is held in an RDBMS reached via an encapsulating class.
 *
 * Function naming convention:
 * getBars()          SELECT * FROM Bars
 * getBarsInFoo($foo) SELECT * FROM Bars WHERE (Foo = $foo)
 * getBar($barID)     SELECT * FROM Bars WHERE (BarID = $barID) BarID is primary key
 *
 * getBars...() returns $bar[rows][fields]
 * getBar()     returns $bar[fields]
 *
 * addBar($bar)       INSERT INTO Bars VALUES $bar
 * updateBar($bar)    UPDATE Bars SET * = $bar WHERE (BarID = $bar[BarID])
 * deleteBar($barID)  DELETE FROM Bars WHERE (BarID = $barID)
 *
 */

define("TOF_NOT_FOUND",1);
define("TOF_ACCESS_DENIED",2);

include_once('query.php');

class SloozeCtSql {

  /* Redefine this parameter by deriving your own class */
  var $sqlClass = "DB_Sql";  /* type of RDBMS data store */
  var $db;                   /* database access object */
  var $errString;            /* if an error occurs, set this to the error */
  var $errCode = 0;
  var $sortMethod = 1;	/* Default sort method is by average rate */
  var $orderbyArray = 
	  array(SORT_AVGRATE => 'pictures.AverageRate desc',
			SORT_NBCLICK_DESC => 'pictures.NbClick desc',
			SORT_ENTRYDATE_DESC => 'pictures.EntryDate desc',
			SORT_FRAMEID => 'pictures.FrameID',
			SORT_ROLLID => 'pictures.RollID',
			SORT_DATE => 'pictures.Date',
			SORT_DATE_DESC => 'pictures.Date desc',
			SORT_TOPICINDEX => 'topicscontent.Index');
  var $distinctonArray = 
	  array(SORT_AVGRATE => 'pictures.AverageRate',
			SORT_NBCLICK_DESC => 'pictures.NbClick',
			SORT_ENTRYDATE_DESC => 'pictures.EntryDate',
			SORT_FRAMEID => 'pictures.FrameID',
			SORT_ROLLID => 'pictures.RollID',
			SORT_DATE => 'pictures.Date',
			SORT_DATE_DESC => 'pictures.Date',
			SORT_TOPICINDEX => 'topicscontent.Index');
  var $pkeys = 
	  array('topics' => 'TopicID',
			'rolls' => 'RollID',
			'pictures' => 'PictureID',
			'users' => 'UserID',
			'groups' => 'GroupID',
			'imageelements' => array('TopicID','PageNum','Id'),
			'textelements' => array('TopicID','PageNum','Id'));
  var $cachedTables = 
	  array('topics'=>true);
  var $cache = array();
  var $tableDefs;

  var $users = array();

  var $dico = array ('topicid' => 'TopicID',
					 'description' => 'Description',
					 'rollid' => 'RollID',
					 'frameid' => 'FrameID',
					 'parenttopicid' => 'ParentTopicID',
					 'pictureid' => 'PictureID',
					 'filename' => 'Filename',
					 'owner' => 'Owner',
					 'entrydate' => 'EntryDate',
					 'date' => 'Date',
					 'nbclick' => 'NbClick',
					 'nbrates' => 'NbRates',
					 'maxrate' => 'MaxRate',
					 'minrate' => 'MinRate',
					 'averagerate' => 'AverageRate',
					 'sumrates' => 'SumRates',
					 'summary' => 'Summary',
					 'commentid' => 'CommentID',
					 'comment' => 'Comment',
					 'ip' => 'IP',
					 'userid' => 'UserID',
					 'username' => 'Username',
					 'password' => 'Password',
					 'name' => 'Name',
					 'email' => 'Email',
					 'selfgroup' => 'SelfGroup',
					 'upload' => 'Upload',
					 'selection' => 'Selection',
					 'admin' => 'Admin',
					 'quota' => 'Quota',
					 'diskusage' => 'DiskUsage',
					 'notifycomment' => 'NotifyComment',
					 'showemptytopics' => 'ShowEmptyTopics',
					 'disabletooltips' => 'DisableTooltips',
					 'groupid' => 'GroupID',
					 'note' => 'Note',
					 'typeid' => 'TypeID',
					 'value' => 'Value',
					 'type' => 'Type',
					 'libelle_fr' => 'Libelle_fr',
					 'picturesize' => 'PictureSize',
					 'displayratio' => 'DisplayRatio',
					 'pictureinterval' => 'PictureInterval',
					 'direct' => 'Direct',
					 'index' => 'Index',
					 'book' => 'Book',
					 'pagenum' => 'PageNum',
					 'id' => 'Id',
					 'box' => 'Box', 
					 'width' => 'Width', 
					 'height' => 'Height', 
					 'text' => 'Text',
					 'size' => 'Size',
					 'font' => 'Font',
					 'align' => 'Align');

  /* public: constructor */
  function SloozeCtSql() {
	  $name = $this->sqlClass;
	  $this->db = new $name;
	  $this->tableDefs = 
		   array('users' => array('Quota'=>'integer',
								  'PictureInterval' => 'integer', 
								  'ShowEmptyTopics'=>'boolean',
								  'DisableTooltips'=>'boolean',
								  'Admin'=>'boolean',
								  'Selection'=>'boolean',
								  'NotifyComment'=>'boolean'),
				 'topics' => array('ParentTopicID'=>'integer',
								   'Book' => 'boolean'),
				 'pageelements' => array('Box' => 'box'),
				 'imageelements' => array('Box' => 'box'),
				 'textelements' => array('Box' => 'box'));
	  global $sortMethod;
	  if (isset($sortMethod)) {
		  $this->sortMethod = $sortMethod;
	  }
  }

  function begin() {
	  $this->db->query("BEGIN");
  }

  function commit() {
	  $this->db->query("COMMIT");
  }

  function newDB() {
	  $name = $this->sqlClass;
	  return new $name;
  }

  function ownerFilter() {
	  return ($this->authorFilter!='')?
		  (" AND (Owner=".$this->authorFilter." OR pictures.PictureID IS NULL)"):
		  "";
  }

  function orderBy() {
	  if ($this->sortMethod!=0) {
		  return " order by ".
			  $this->orderbyArray[$this->sortMethod].
			  ",Pictures.PictureID ";
	  } else {
		  return " order by Pictures.PictureID ";
	  }
  }

  function distinctOn() {
	  if ($this->sortMethod!=0) {
		  return " distinct on (".
			  $this->distinctonArray[$this->sortMethod].
			  ",Pictures.PictureID) ";
	  } else {
		  return " distinct on (Pictures.PictureID) ";
	  }
  }

  /* public: resetError() makes errString empty so errors don't carry over */
  function resetError() {
	  $this->errString = "";
  }

  /* public: getError() returns errString */
  function getError() {
	  return $this->errString;
  }

  /**
   * Builds SQL statement for a query
   */
  function getSQL(&$query) {
		$sql = "SELECT ";
		if (count($query->distincts)>0)
			$sql .= 'DISTINCT ON ('.join(',',$query->distincts).') ';
		$sql .= join(',',$query->columns);
		if ($query->into)
			$sql .= " INTO TEMP ".$query->into;
		$sql .= " FROM ".$query->table;
		foreach($query->joins as $table => $column) {
			if (!is_array($column))
				$sql .= " JOIN $table USING ($column)";
			else
				$sql .= " JOIN $table ON ({$column[0]}={$column[1]})";
		}
		if (count($query->filters)>0)
			$sql .= " WHERE (".join(') AND (',$query->filters).')';
		if (count($query->groups)>0)
			$sql .= " GROUP BY ".join(',',$query->groups);
		if (count($query->order)>0)
			$sql .= " ORDER BY ".join(',',$query->order);
		if ($query->hasLimit)
			$sql .= " LIMIT ".$query->limit." OFFSET ".$query->offset;
		return $sql;
  }

  function queryAllRecords($table,$columns,$filters=array(),$order=array()) {
	  $query = new Query($table,$columns,$filters,$order);
	  $this->query($query);
	  return $this->getAllRecords($table);
  }

  function querySingleRecord($table,$columns,$filters=array(),$order=array()) {
	  $query = new Query($table,$columns,$filters,$order);
	  $this->db->query($this->getSQL($query));
	  return $this->getSingleRecord($table);
  }

  function queryRecordByKey($table,$keyValue) {
	  $doCache = $this->cachedTables[$table];
	  $record = false;
	  $pkey = $this->pkeys[$table];
	  if ($doCache) {
		  $record = $this->cache[$table][$keyValue];
	  }
	  if (!$record) {
		  $query = new Query($table,'*',"$pkey = '$keyValue'");
		  $this->db->query($this->getSQL($query));
		  $record =  $this->getSingleRecord($table);
		  if ($doCache) {
			  $this->cache[$table][$keyValue] = $record;
		  }
	  }
	  return $record;
  }

  function invalidateCacheEntry($table,$keyValue) {
	  if ($this->cachedTables[$table]) {
		  unset($this->cache[$table][$keyValue]);
	  }
  }

  function getAll($table,$columns='*',$order=array()) {
	  $this->query(new Query($table,$columns,array(),$order));
	  return $this->getAllRecords($table);
  }

  function getWhere($table,$columns='*',$filters=array(),$order=array()) {
	  $this->query(new Query($table,$columns,$filters,$order));
	  return $this->getAllRecords($table);
  }

  function getKey($table,$key) {
	  return $this->queryRecordByKey($table,$key);
  }

  /**
   * Returns a query for all the pictures a user can see
   */
  function picturesQuery($userID,$columns='pictures.*',$sort=true) {
	  $query = new Query('pictures',$columns);
	  $query->addFilter($this->buildPermsCondition($userID));
	  if ($sort && $this->sortMethod!=0) {
		  $query->addOrder($this->orderbyArray[$this->sortMethod]);
	  }
	  if ($sort)
		  $query->addOrder('pictures.PictureID');
	  if ($this->authorFilter!='')
		  $query->addFilter("Owner={$this->authorFilter}");
	  return $query;
  }

  /**
   * Returns all the topics of a picture. 
   * @param pictureID the picture
   */
  function getTopics($pictureID,$userID) {
	  $this->db->query(
		  "select * from Topics join topicscontent using (topicid) ".
		  "where PictureID=$pictureID and Direct=true and ".
		  $this->buildTopicPermsCondition($userID).
		  " order by Description");
	  return $this->getAllRecords('topics');
  }

  /* public: getTopicIDs() returns all the topic IDs of a picture. */
  function getTopicIDs($pictureID) {
	  $this->db->query(
		  "select TopicID from Topics join topicscontent using (topicid) ".
		  "where PictureID=$pictureID and Direct=true ".
		  "order by Description");
	  return $this->getAllSingleValues();
  }

  /**
   * Returns the most used keywords of a topic
   */
  function getTopicKeywords($topicID,$type = 0) {
	  if ($type != 0) {
		  $this->db->query(
			  "select type,value,count(value) as count from keywords,topicscontent".
			  " where keywords.pictureid = topicscontent.pictureid and topicscontent.topicid=$topicID".
			  " and type=$type".
			  " group by value,type order by count desc");
	  } else {
		  $this->db->query(
			  "select type,value,count(value) as count from keywords,topicscontent".
			  " where keywords.pictureid = topicscontent.pictureid and topicscontent.topicid=$topicID".
			  " group by value,type order by count desc");
	  }
	  return $this->getAllRecords();
  }

  /**
   * Returns the most used keywords of a topic
   */
  function getTopicKeywordValues($topicID,$type = 0) {
	  if ($type != 0) {
		  $this->db->query(
			  "select distinct value as count from keywords,topicscontent".
			  " where keywords.pictureid = topicscontent.pictureid and topicscontent.topicid=$topicID".
			  " and type=$type".
			  " order by value");
	  } else {
		  $this->db->query(
			  "select distinct value as count from keywords,topicscontent".
			  " where keywords.pictureid = topicscontent.pictureid and topicscontent.topicid=$topicID".
			  " order by value");
	  }
	  return $this->getAllSingleValues();
  }

  /**
   * Returns all the topics with the given parent TopicID. 0 is a
   * magic TopicID meaning root. 
   */
  function getSubTopics($parentTopicID = 0) {
	  $this->db->query("select * from Topics where ".
					   "(ParentTopicID = $parentTopicID) ".
					   "and (TopicID != 0) order by Description");
	  return $this->getAllRecords('topics');
  }

  /* public: getTopic() returns a single topic. */
  function getTopic($topicID) {
	  return $this->queryRecordByKey('topics',$topicID);
  }

  function nextVal($sequence) {
	  $this->db->query("SELECT nextval('$sequence')");
	  $this->db->query("SELECT currval('$sequence')");
	  return $this->getSingleValue();
  }

  /* public: addTopic() adds one single topic 
   * @return the TopicID of the new topic
   */
  function addTopic($topic) {
	  $topicID = $this->nextVal('topics_topicid_seq');
	  $topic['TopicID'] = $topicID;
	  $cols = $this->makeCols($topic);
	  $vals = $this->makeVals($topic);
	  $this->db->query("INSERT INTO Topics ($cols) VALUES ($vals)");
	  return $topicID;
  }

  /* public: deleteTopic() deletes one single topic */
  function deleteTopic($topicID,$force) {
	  if ($force || 
	      count($this->getAllPicturesInTopic($topicID,true,'PictureID')) == 0) 
	  {
		  $this->db->query("DELETE FROM Topics WHERE TopicID = '$topicID'");
		  $this->db->query("DELETE FROM TopicsContent WHERE TopicID = '$topicID'");
		  return true;
	  } else {
		  $this->errString = _('Cannot delete: there are pictures in this topic.');
		  return false; /* failed */
	  }
  }
  
  /* public: updateTopic() updates one single topic */
  function updateTopic($topic) {
	  if ($topic['TopicID']==$topic['ParentTopicID']) {
		  $this->error(_('Topic not updated: parent topic cannot be the topic itself'));
		  return false;
	  }
	  $topicID = $topic['TopicID'];
	  unset($topic['TopicID']);
	  $oldTopic = $this->getTopic($topicID);
	  $update = $this->makeUpdate($topic,'topics');
	  $this->db->query("update Topics set $update where (TopicID = '$topicID' )");
	  $this->invalidateCacheEntry('topics',$topicID);
	  if ($oldTopic['ParentTopicID'] != $topic['ParentTopicID']) {
		  // update topics for all pictures in that topic
		  $pictures = $this->getAllPicturesInTopic($topicID,true,'PictureID');
		  reset($pictures);
		  while (list($key,$picture) = each($pictures)) {
			  $topics = $this->getTopicIDs($picture['PictureID']);
			  $this->updateTopics($picture['PictureID'],$topics);
		  }
	  }
	  return true;
  }

  /* public: getRolls() returns all the rolls. */
  function getRolls($userID="") {
	  return $userID !== '' ? 
		  $this->getWhere('rolls','*',"Owner=$userID",'RollID') :
		  $this->getAll('rolls','*','RollID');
  }

  /* public: getRoll() return a single roll. */
  function getRoll($rollID) {
	  return $this->queryRecordByKey('rolls',$rollID);
  }

  /* public: addRoll() adds one single roll */
  function addRoll($roll) {
	  $cols = $this->makeCols($roll);
	  $vals = $this->makeVals($roll);
	  $this->db->query("INSERT INTO Rolls ($cols) VALUES ($vals)");
	  return true;
  }

  /* public: deleteRoll() deletes one single roll */
  function deleteRoll($rollID) {
	  $this->db->query("DELETE FROM Rolls WHERE (RollID = '$rollID')");
	  if ($this->db->Errno==0) {
		  return true;
	  } else {
		  $this->errString = _('Cannot delete: there are pictures in this roll.');
		  return false; /* failed */
	  }
  }

  /* public: updateRoll() updates one single roll */
  function updateRoll($roll) {
	  $update = $this->makeUpdate($roll);
	  $this->db->query(
		  "update Rolls set $update".
		  " where (RollID = '".$roll['RollID']."' )");
	  return true;
  }

  /* public: getPictures() returns all the pictures. */
  function getPictures($fields = '*') {
	  $this->db->query("select $fields from Pictures");
	  return $this->getAllRecords('pictures');
  }

  /**
   * Returns all the Pictures in a given Roll that a user can see
   *
   * @param rollID id of the roll
   * @param userID id of user
   * @param columns columns to fetch
   * @param offset skip this many pictures before fetching
   * @param limit only fetch this many pictures
   * @return array(totalCount,pictures)
   */
  function getPicturesInRoll($rollID, $userID, $columns='pictures.*', $offset=0, $limit=0) {
	  $query = $this->picturesQuery($userID,$columns,false);
	  $query->addFilter("pictures.RollID='$rollID'");
	  /*
	  if (func_num_args()>3)
		  $count = $this->countPicturesInQuery($query);
	  */
	  $query->addOrder('pictures.FrameID');
	  /*
	  if (func_num_args()>3)
		  $query->setLimit($offset,$limit);
	  */
	  $this->query($query);
	  /*
	  if (func_num_args()>3)
		  return array($count,$this->getAllRecords());
	  else
		  return array($this->db->num_rows(),$this->getAllRecords());
	  */
	  if (func_num_args()>3)
		  return array($this->db->num_rows(),$this->getRecords($offset,$limit));
	  else
		  return array($this->db->num_rows(),$this->getAllRecords('pictures'));
  }

  function queryPicturesInTopic($userID,$topicID,$recursive,$sort,$columns='pictures.*') {
	  $query = $this->picturesQuery($userID,$columns,$sort);
	  if (!($topicID==0 && $recursive)) {
		  $query->addJoin('topicscontent','PictureID');
		  $query->addFilter("topicscontent.TopicID=$topicID");
		  if (!$recursive)
			  $query->addFilter("TopicsContent.Direct=true");
	  }
	  return $query;
  }

  function countPicturesInQuery($query) {
	  $query->columns = array('count(*)');
	  $query->order = array();
	  $this->db->query($this->getSQL($query));
	  return $this->getSingleValue();	  
  }

  /**
   * Returns pictures visible by user in a given topic.
   *
   * @param topicID the topic to list
   * @param recursive wether to recursivley list pictures of subtopics
   * @param userID id of user
   * @param offset skip this number of pictures
   * @param limit only return this number of pictures
   * @param count store the total number of pictures in this variable
   */
  function getPicturesInTopic($topicID, $recursive, $userID, $offset, $limit, &$count) {
	  $query = $this->queryPicturesInTopic($userID,$topicID,$recursive,'Pictures.*');
	  $count = $this->countPicturesInQuery($query);
	  if ($limit>0)
		  $query->setLimit($offset,$limit);
	  $this->query($query);
	  return $this->getAllRecords('pictures');
  }

  /**
   * Returns the pictures that can be used as front page for a topic
   * @param topicID id of topic
   * @param recursive wether to include picture from subtopics
   * @param userID only return pictures this user can see
   */
  function getTopicFrontPage($topicID, $recursive, $userID) {
	  // I can't seem to be able to optimize this in one SQL query
	  $this->db->query("select PictureID from TopicsFrontPage where TopicID=$topicID");
	  $pictures = $this->getAllSingleValues();
	  foreach($pictures as $pictureID) {
		  $this->db->query(
			  "SELECT * FROM pictures WHERE pictureid=$pictureID".
			  " AND pictureid IN (SELECT PictureID FROM permissions JOIN groupsdef USING(GroupID)".
			  " WHERE groupsdef.UserID = $userID)");
		  $frontPage = $this->getSingleRecord();
		  if ($frontPage)
			  return $frontPage;
	  }
	  return false;
  }

  /**
   * Returns the topics a picture is a front page of.
   */
  function getPictureFrontPages($pictureID) {
	  $this->db->query("SELECT * FROM topics JOIN topicsfrontpage USING(TopicID)".
					   " WHERE PictureID=$pictureID");
	  return $this->getAllRecords('topics');
  }

  function getBestPictureInTopic($topicID, $recursive, $userID) {
	  /*
	  $query = $this->picturesQuery($userID,"Pictures.*",false);
	  if (!($topicID==0 && $recursive)) {
		  $query->addJoin('topicscontent','PictureID');
		  $query->addFilter("topicscontent.TopicID=$topicID");
		  $query->addOrder("Pictures.averagerate desc");
		  if (!$recursive)
			  $query->addFilter("TopicsContent.Direct=true");
	  }
	  $query->setLimit(0,1);
	  $this->query($query);
	  return $this->getSingleRecord();
	  */
	  $this->db->query(
		  "SELECT Pictures.* FROM pictures".
		  " JOIN topicscontent USING (PictureID)".
		  " WHERE (pictures.PictureID IN (SELECT PictureID FROM permissions JOIN groupsdef USING(GroupID) WHERE groupsdef.UserID = $userID))".
		  " AND (topicscontent.TopicID=$topicID)".
		  ($recursive ? "" : " AND topicscontent.Direct=true").
		  " ORDER BY Pictures.AverageRate desc LIMIT 1 OFFSET 0");
	  return $this->getSingleRecord();
  }

  /**
   * Counts pictures visible by a user in the subtopics of a topic 
   * @param topicID the topic whose subtopics to scan
   * @param userID id of the user
   * @param perms an array of permissions the cuser must have on the topics
   */
  function countPicturesInSubtopics($topicID, $userID, $perms='') {
	  /*
	  $query = 
		   $this->picturesQuery(
			   $userID,
			   array('count(PictureID)','topics.TopicID','topics.Description'),
			   false);

	  $query->addJoin('topicscontent','PictureID');
	  $query->addJoin('topics','TopicID');
	  $query->addFilter("topics.ParentTopicID=$topicID");

	  $query->addGroup('topics.TopicID');
	  $query->addGroup('topics.Description');

	  $query->addOrder('topics.Description');

	  $this->db->query($this->getSQL($query));
	  */

	  // We also want empty topics, so we must formulate the query like this
	  $query = "SELECT count(PictureID),topics.TopicID,topics.Description,topics.Book FROM topics".
		   " LEFT OUTER JOIN topicscontent USING (TopicID)".
		   " LEFT OUTER JOIN pictures USING (PictureID)".
		   " WHERE (topics.ParentTopicID=$topicID)".
		   " AND (".$this->buildTopicPermsCondition($userID,$perms).")".
		   " AND (".$this->buildPermsCondition($userID)." OR PictureID IS NULL)".
		   $this->ownerFilter().
		   " GROUP BY topics.TopicID,topics.Description,topics.Book".
		   " ORDER BY topics.Description";
	  $this->db->query($query);
	  return $this->getAllRecords('topics');
  }

  /**
   * Returns the permissions of user on a topic
   */
  function getTopicPerms($topicID,$userID) {
	  $this->db->query(
		  "SELECT Access FROM topicperms JOIN groupsdef USING(GroupID)".
		  " WHERE TopicID=$topicID AND groupsdef.UserID = $userID");
	  return $this->getAllSingleValues();
  }

  /**
   * Returns groups who have a perm on a topic
   */
  function getTopicGroups($topicID,$perm) {
	  $this->db->query(
		  "SELECT groups.* FROM groups JOIN topicperms USING(GroupID)".
		  " WHERE TopicID=$topicID AND access='$perm'");
	  return $this->getAllRecords('groups');
  }

  function removeTopicPerm($topicID,$groupID,$perm) {
	  $this->db->query(
		  "DELETE from topicperms WHERE TopicID=$topicID AND GroupID=$groupID AND Access='$perm'");
  }

  function addTopicPerm($topicID,$groupID,$perm) {
	  $this->db->query(
		  "SELECT Access FROM topicperms WHERE TopicID=$topicID AND GroupID=$groupID");
	  $perms = $this->getAllSingleValues();
	  if (!in_array($perm,$perms)) {
		  $this->db->query(
			  "INSERT INTO topicperms (TopicID,GroupID,Access) VALUES ($topicID,$groupID,'$perm')");
	  }
  }

  function countPicturesInTopic($topicID, $userID, $recurse=true) {
	  if ($topicID==0 && $recurse) {
		  $query = $this->picturesQuery($userID,'count(*)',false);
		  $this->db->query($this->getSQL($query));
	  } else
		  $this->db->query(
			  "select count(Pictures.PictureID)".
			  " from Pictures,TopicsContent".
			  " where (Pictures.PictureID = TopicsContent.PictureID) and ".
			  "(TopicsContent.TopicID = '$topicID') and".
			  ($recurse ? "" : " TopicsContent.direct=true and ").
			  " exists ".$this->buildPermsSelect($userID));
	  /*
		  $this->db->query(
			  "select count(*) from userpictures2 join topicscontent using(PictureID)".
			  " where TopicID=$topicID and UserID=$userID");
	  */
	  return $this->getSingleValue();
  }

  /** 
   * Returns really all the Pictures in a given Topic, regardless of
   * access rights.
   *
   * @param topicID which topic to get pictures from
   * @param recursive wether to recursively get pictures from subtopics
   * @param fields comma separated list of field to fetch for each pictures
   */
  function getAllPicturesInTopic($topicID, $recursive, $fields='*') {
	  $this->db->query(
		  "select $fields from Pictures join TopicsContent using (PictureID) ".
		  "where TopicsContent.TopicID = $topicID ".
		  ($recursive?"":"and TopicsContent.Direct = true")); 
	  return $this->getAllRecords('pictures');
  }

  /** 
   * getPicture() returns a single picture by pictureID. 
   * @param pictureID id of requested picture
   * @param userID only return picture if this user's ID is allowed to see it.
   *               Use -1 to always return picture.
   */
  function getPicture($pictureID , $userID = -1) {
	  if ($userID != -1) {
		  $this->db->query("select distinct on (Pictures.PictureID) ".
						   "* from Pictures,Permissions,Groupsdef where " . 
						   "(Pictures.PictureID = '$pictureID') and " . 
						   $this->buildPermsWhere($userID));
	  } else {
		  $this->db->query("select * from Pictures where " . 
						   "(PictureID = '$pictureID')");
	  }
	  return $this->getSingleRecord();
  }

  /* getPictureWhere() returns a single picture by pictureID. */
  function getPictureWhere($where) {
	  $this->db->query("select * from Pictures where $where");
	  return $this->getSingleRecord();
  }

  /* increment the NbClick of the picture */
  function clickPicture($pictureID) {
	  $this->db->query("update Pictures set NbClick=NbClick+1 ".
					   "where PictureID = '$pictureID'");
  }

  /* pictureExists() returns true if picture already exists. */
  function pictureExists($rollID, $frameID) {
	  $this->db->query("select * from Pictures where " . 
					   "RollID = '".addslashes($rollID)."' and ".
					   "FrameID = '".addslashes($frameID)."'");
	  return( $this->db->num_rows()>0 );
  }

  /* getPictureInTopic() returns a picture by its position, 
     as well as the previous and next pictures in the topic. */
  function getPictureInTopic($pictureID , $topicID, $recursive, 
                             $position, $userID) {
	  $query = "select ".$this->distinctOn().
		   " * from Pictures,TopicsContent,Permissions,Groupsdef  ".
		   "where Pictures.PictureID=TopicsContent.PictureID ".
		   ($recursive?"":"and TopicsContent.Direct=true ").
		   "and TopicsContent.TopicID=$topicID and " . 
		   $this->buildPermsWhere($userID). $this->orderBy();
	  return $this->getPictureInQuery($query, $pictureID, $position);
  }

  /* getPictureInRoll() returns a picture by its position, 
     as well as the previous and next pictures in the roll. */
  function getPictureInRoll($pictureID , $rollID, $position, $userID) {
	  $query = "select distinct on (Pictures.FrameID) * ".
		   "from Pictures,Permissions,Groupsdef ".
		   "where Pictures.RollID='$rollID' and " . 
		   $this->buildPermsWhere($userID)." order by Pictures.FrameID";
	  return $this->getPictureInQuery($query, $pictureID, $position);
  }

  function rateSortQuery($userID) {
	  return "select distinct on (Pictures.AverageRate,Pictures.PictureID) * ".
		  "from Pictures,Permissions,Groupsdef where " . 
		  $this->buildPermsWhere($userID).
		  " order by Pictures.AverageRate desc,Pictures.PictureID";
  }

  function searchQuery($search,$userID) {
	  return "select distinct on (Pictures.PictureID) * ".
		  "from Pictures,Permissions,Groupsdef where ".
		  "(Pictures.Description ~* '.*$search.*') and ".
		  $this->buildPermsWhere($userID);
  }

  /* getPicturesInSearch() returns all the Pictures where the given
   * search term occurs in the Description. */
  function getPicturesInSearch( $search,$userID,$offset,$count ) {
	  $this->db->query($this->searchQuery($search,$userID));
	  return( array($this->db->num_rows(),
	                $this->getRecords($offset,$count) ) );
  }

  function getPictureInSearch($pictureID,$search,$position,$userID) {
	  return $this->getPictureInQuery(
		  $this->searchQuery($search,$userID), $pictureID, $position);
  }

  function getRateSort($userID,$offset,$count) {
	  $query = $this->picturesQuery($userID);
	  $query->order = array('pictures.AverageRate desc','pictures.PictureID');
	  return $this->getPicturesInQuery($this->getSQL($query),$offset,$count);
  }

  function getPictureInRateSort($pictureID,  $position, $userID) {
	  $query = $this->picturesQuery($userID);
	  $query->order = array('pictures.AverageRate desc','pictures.PictureID');
	  return $this->getPictureInQuery($this->getSQL($query), $pictureID, $position);
  }

  function dateSortQuery($userID) {
	  return "select distinct on (Pictures.EntryDate,Pictures.PictureID)".
		  " * from Pictures,Permissions,Groupsdef where " . 
		  $this->buildPermsWhere($userID)." order by Pictures.EntryDate desc,Pictures.PictureID";
  }

  function getDateSort($userID,$offset,$count) {
	  return $this->getPicturesInQuery($this->dateSortQuery($userID),$offset,$count);
  }

  function getPictureInDateSort($pictureID,  $position, $userID) {
	  return $this->getPictureInQuery($this->dateSortQuery($userID), $pictureID, $position);
  }

  function UnindexedQuery($userID) {
	  return "select ".$this->distinctOn()." * ".
		  "from Pictures,Permissions,Groupsdef where ".
		  "not exists (select PictureID from Keywords where ".
		  "Pictures.PictureID=Keywords.PictureID) ".
		  "and ".$this->buildPermsWhere($userID).
		  $this->orderBy()  ;
  }

  function getUnindexedPictures($userID,$offset,$count) {
	  $this->db->query($this->unindexedQuery($userID));
	  return( array($this->db->num_rows(),
					$this->getRecords($offset,$count)) );
  }

  function getPictureInUnindexed( $pictureID,  $position, $userID ) {
	  return $this->getPictureInQuery($this->unindexedQuery($userID), $pictureID, $position);
  }

  function unratedQuery($userID) {
	  return "select ".$this->distinctOn()." * ".
		  "from Pictures,Permissions,Groupsdef where ".
		  "Pictures.NbRates=0 and ".
		  $this->buildPermsWhere($userID) . $this->orderBy();
  }

  function getUnratedPictures($userID,$offset,$count) {
	  $this->db->query($this->unratedQuery($userID));
	  return array(
		  $this->db->num_rows(),
		  $this->getRecords($offset,$count));
  }

  function getPictureInUnrated($pictureID,  $position, $userID) {
	  return $this->getPictureInQuery($this->unratedQuery($userID), $pictureID, $position );
  }

  function getPictureInMultipleKeywordSearch( $pictureID, $keywords, $matchAll, $position, $userID ) {
	  $this->doMultipleKeywordSearch($keywords, $matchAll, $userID);
	  return $this->getPictureInQuery2($pictureID, $position);
  }

  function getPictureInKeywordSearch($pictureID, $keyword, $typeID, $position, $userID) {
	  $keyword = addslashes($keyword);
	  if ($typeID==0) {
		  $query = "select ".$this->distinctOn()." * from Pictures,Permissions,Groupsdef where ".
			   "Pictures.PictureID in ".
			   "(select PictureID from Keywords where ".
			   "Type is null and Value='$keyword') and ".
			   $this->buildPermsWhere($userID).$this->orderBy();
	  } else { 
		  $query = "select ".$this->distinctOn()." * from Pictures,Permissions,Groupsdef where ".
			   "Pictures.PictureID in ".
			   "(select PictureID from Keywords where ".
			   "Type=$typeID and Value='$keyword') and ".
			   $this->buildPermsWhere($userID).$this->orderBy();
	  }
	  return $this->getPictureInQuery($query, $pictureID, $position );
  }

  function getPictureInQuery($query, $pictureID, $position) {
	  $this->db->query($query);
	  return $this->getPictureInQuery2($pictureID, $position);
  }

  function query(&$query) {
	  $this->db->query($this->getSQL($query));
  }

  function getPictureInQuery2($pictureID, $position) {
	  if ($position > $this->db->num_rows() - 1)
		  $position = $this->db->num_rows() - 1;
	  $this->db->seek($position);
	  $this->nextRecord();
	  if (!$pictureID || $this->db->Record['PictureID'] == $pictureID) {
		  $pictures = array();
		  $pictures['current'] = $this->db->Record;
		  $pictures['last'] = $this->db->num_rows() - 1;
		  if ($position>0) {
			  $this->db->seek($position-1);
			  $this->nextRecord();
			  $pictures['prev'] = $this->db->Record;
		  }
		  if ($position<$this->db->num_rows()) {
			  $this->db->seek($position+1);
			  $this->nextRecord();
			  $pictures['next'] = $this->db->Record;
		  }
		  return $pictures;
	  } else {
		  // in case the position was not correct, 
		  // try to find the picture in the list
		  $this->db->seek(0);
		  $position = 0;
		  while($this->nextRecord()) {
			  if ($this->db->Record['PictureID'] == $pictureID) {
				  $pictures = array();
				  $pictures['last'] = $this->db->num_rows() - 1;
				  $pictures['current'] = $this->db->Record;
				  $this->db->seek($position-1);
				  $this->nextRecord();
				  $pictures['prev'] = $this->db->Record;
				  $this->db->seek($position+1);
				  $this->nextRecord();
				  $pictures['next'] = $this->db->Record;
				  return $pictures;
			  } else {
				  $position = $position + 1;
			  }
		  }
		  return false;
	  }
  }
  
  /* public: addPicture() adds one single picture. 
   * @return the PictureID of the new picture
   */
  function addPicture($picture, $topics, $groups) {
	  if (!isset($picture['NbRates'])) {
		  $picture['NbRates'] = 0;
	  }
	  $pictureID = $this->nextVal('pictures_pictureid_seq');
	  $picture['PictureID'] = $pictureID;
	  $cols = $this->makeCols($picture);
	  $vals = $this->makeVals($picture);
	  $this->db->query("insert into Pictures ($cols) values ($vals)");
	  $user = $this->getUser($picture['Owner']);
	  $groupID = $user['SelfGroup'];
	  $newPicture = $this->getPictureWhere(
		  "RollID='".addslashes($picture['RollID'])."' and ".
		  " FrameID='".addslashes($picture['FrameID'])."'");
	  $pictureID = $newPicture['PictureID'];
	  $this->db->query("insert into permissions (GroupID,PictureID) ".
					   "values ($groupID,$pictureID)");

	  // set initial permissions
	  $this->setPermissions($pictureID, $groups, $user['UserID']);

	  // set initial topics
	  reset($topics);
	  if (is_array($topics)) {
		  while( list($key, $topicID) = each($topics) ) {
			  $this->insertTopic($pictureID,$topicID);
		  }
	  }
	  return $pictureID;
  }

  /* public: updateTopics() updates the topics of a picture */
  function updateTopics($pictureID, $topics) {
	  $this->db->query("delete from topicscontent where PictureID=$pictureID");
	  reset($topics);
	  while ( list ($key,$topicID) = each($topics) ) {
		  $this->insertTopic($pictureID,$topicID);
	  }
  }
  
  /* private: insertTopic() inserts a picture into a topic */
  function insertTopic($pictureID, $topicID) {
	  $this->begin();
	  if (!$this->topicContains($topicID, $pictureID)) {
		  $this->db->query("insert into topicscontent (TopicID,PictureID,Direct) ".
						   "values ($topicID,$pictureID,true)");
	  }
	  $this->updateParentTopics($pictureID,$topicID);
	  $this->commit();
  }

  function removePictureFromTopic($pictureID, $topicID) {
	  $this->begin();
	  $this->db->query(
		  "DELETE FROM topicscontent".
		  " WHERE PictureID=$pictureID AND TopicID=$topicID");
	  $this->db->query(
		  "DELETE FROM topicscontent".
		  " WHERE PictureID=$pictureID AND Direct=false");
	  $this->db->query("SELECT TopicID FROM topicscontent WHERE PictureID=$pictureID");
	  $topics = $this->getAllSingleValues();
	  foreach($topics as $topicID) {
		  $this->updateParentTopics($pictureID, $topicID);
	  }
	  $this->commit();
  }

  function topicContains($topicID,$pictureID) {
	  $this->db->query("select * from topicscontent ".
					   "where TopicID=$topicID and pictureID=$pictureID");
	  if ($this->db->next_record())
		  return true;
	  else
		  return false;
  }

  /* private: updateParentTopics() */
  function updateParentTopics($pictureID, $topicID) {
	  if ($topicID!=0) {
		  if (!$this->topicContains(0, $pictureID)) {
			  $this->db->query("insert into topicscontent (TopicID,PictureID,Direct) ".
							   "values (0,$pictureID,false)");
		  }
		  $this->db->query("select ParentTopicID from topics ".
						   "where TopicID=$topicID");
		  $topicID = $this->getSingleValue();
		  while($topicID!=0) {
			  if (!$this->topicContains($topicID, $pictureID)) {
				  $this->db->query("insert into topicscontent (TopicID,PictureID,Direct) ".
								   "values ($topicID,$pictureID,false)");
			  }
			  $this->db->query("select ParentTopicID from topics ".
							   "where TopicID=$topicID");
			  $topicID = $this->getSingleValue();
		  }
	  }
  }

  /* public: deletePicture() deletes one single picture */
  function deletePicture( $pictureID ) {
    $picture = $this->getPicture($pictureID);
    $this->db->query("delete from Keywords where PictureID = '$pictureID'");
    $this->db->query("delete from Comments where PictureID = '$pictureID'");
    $this->db->query("delete from Permissions where PictureID = '$pictureID'");
    $this->db->query("delete from Notes where PictureID = '$pictureID'");
    $this->db->query("delete from Pictures where PictureID = '$pictureID'");
    return true;
  }

  function setDate($picture) {
	  $this->db->query(
		  "update Pictures set Date='".$picture['Date']."'".
		  " where PictureID = ".$picture['PictureID']);	  
  }

  function updatePictureFields($picture) {
	  $pictureID = $picture['PictureID'];
	  unset($picture['PictureID']);
	  $update = $this->makeUpdate($picture,'pictures');
	  $this->db->query("update Pictures set $update where ".
					   "PictureID = $pictureID");
  }

  /* public: updatePicture() updates one single picture */
  function updatePicture($picture, $topics, $groups, $userID) {
	  $pictureID = $picture['PictureID'];
	  $oldPicture = $this->getPicture($pictureID);
	  // only the owner of a picture is allowed to modify it.
	  /*
	  if ($oldPicture['Owner']!=$userID) {
		  return 0;
	  }
	  */
	  // update groups
	  if (is_array($groups)) {
		  $this->setPermissions($pictureID, $groups, $oldPicture['Owner']);
	  }
	  // update topics
	  if (is_array($topics)) {
		  $this->updateTopics($pictureID,$topics);
	  }
	  // update other fields
	  $update = $this->makeUpdate($picture);
	  $this->db->query("update Pictures set $update where ".
					   "PictureID = $pictureID ".
					   "and Owner=$userID");
	  return $this->db->affected_rows();
  }

  /**
   * @param $pictureID ID of picture to modify
   * @param $groups ID of the groups to add
   */
  function addGroupsToPicture($pictureID, $groups) {
	  // only the owner of a picture is allowed to modify it.
	  foreach($groups as $groupID) {
		  $this->db->query(
			  "select * from permissions".
			  " where PictureID=$pictureID and GroupID=$groupID");
		  if (!$this->db->next_record())
			  $this->db->query(
				  "insert into permissions (GroupID,PictureID)".
				  " values ($groupID,$pictureID)");
	  }
  }

  /**
   * @param $pictureID ID of picture to modify
   * @param $groups ID of the groups to remove
   */
  function removeGroupsFromPicture($pictureID, $groups) {
	  // only the owner of a picture is allowed to modify it.
	  foreach($groups as $groupID) {
		  $this->db->query(
			  "delete from permissions where PictureID=$pictureID and GroupID in ".
			  $this->makeList($groups));
	  }
  }

  /* set the permissions of a picture 
   * @param owner userID of the owner of the picture
   */
  function setPermissions($pictureID, $groups, $owner) {
	  // move all permissions
	  $this->db->query("delete from permissions where ".
					   "PictureID=$pictureID");
	  // owner should always have permission
	  $user = $this->getUser($owner);
	  $groups[] = $user['SelfGroup'];
	  $groups = array_unique($groups);
	  // insert new permissions
	  while( list($key, $groupID) = each($groups) ) {
		  if ($groupID != $selfGroupID) {
			  $this->db->query("insert into permissions (GroupID,PictureID) ".
							   "values ($groupID,$pictureID)");
		  }
	  }
  }
  
  /* private: updateAllPicturesInTopic() sets the group for all pictures in a topic */
  /*  function updateAllPicturesInTopic( $topicID, $groupID, $userID ) {
    $this->db->query("update Pictures set Group=$groupID where " .
		     "(ParentTopicID = '$topicID' and Owner=$userID)");
    return true;
  }
  */

  function updateUsers($users) {
	  foreach($users as $user) {
		  $this->updateUser($user['UserID'],$user);
	  }
  }

  /* public: updates a user info */
  function updateUser($userID,$user) {
	  unset($this->users[$userID]);
	  if ($user['Quota']==='') 
		  $user['Quota']='NULL';
	  if ($user['PictureInterval']==='') 
		  $user['PictureInterval']='NULL';
	  $update = $this->makeUpdate($user,'users');
	  $this->db->query("update users set $update where UserID=$userID");
	  /* Also update the SelfGroup's Name if available */
	  if ($user['Name'])
		  $this->db->query(
			  "update groups set Name='".$user['Name']."'".
			  " where users.UserID=$userID and".
			  " users.SelfGroup=GroupID");
	  return true;
  }

  /* public: getComments() returns all the comments. */
  function getComments() {
    $this->db->query("select * from Comments");
    return $this->getAllRecords();
  }
  
  /* getCommentsInPicture() returns all the Comments for the given picture. */
  function getCommentsInPicture($pictureID) {
    $this->db->query("select * from Comments where (PictureID = '$pictureID')");
    return $this->getAllRecords();
  }

  /* public: addComment() adds one single comment */
  function addComment($comment) {
    $cols = $this->makeCols($comment);
    $vals = $this->makeVals($comment);
    $this->db->query("insert into Comments ($cols) values ($vals)");
    return true;
  }

  /* private: getAllRecords() returns an array of all records in the current resultset */
  function getAllRecords($table='') {
	  $arr = array();
	  while ($this->nextRecord()) {
		  if (is_array($table)) {
			  foreach($table as $t) {
				  $this->convertValues($this->db->Record,$t);
			  }
		  } else {
			  $this->convertValues($this->db->Record,$table);
		  }
		  $arr[] = $this->db->Record;
	  }
	  return $arr;
  }

  /* private: getAllSingleValues() returns an array of all 
	 single values in the current resultset */
  function getAllSingleValues($selfKey=false) {
	  $arr = array();
	  while ($this->db->next_record()) {
		  if ($selfKey)
			  $arr[$this->db->Record[0]] = $this->db->Record[0];
		  else
			  $arr[] = $this->db->Record[0];
	  }
	  return $arr;
  }

  function nextRecord() {
	  if ($this->db->next_record()) {
		  $this->fixColumnNames($this->db->Record);
		  return true;
	  } else {
		  return false;
	  }
  }

  /* private: getRecords() returns an array of records in the current resultset */
  function getRecords($offset,$count) {
	  $arr = array();
	  $this->db->seek($offset);
	  while ($this->nextRecord() && $count>0) {
		  $arr[] = $this->db->Record;
		  $count--;
	  }
	  return $arr;
  }

  /* private: getAllRecordsByIndex() returns an array of all records in the current resultset */
  function getAllRecordsByIndex($index) {
	  $arr = array();
	  while ($this->nextRecord()) {
		  $arr[$this->db->f($index)] = $this->db->Record;
	  }
	  return $arr;
  }
  
  /* private: getSingleRecord() returns the single record in the current resultset
   * or false if the resultset is empty */
  function getSingleRecord($table='') {
	  if ($this->nextRecord()) {
		  $this->convertValues($this->db->Record,$table);
		  return $this->db->Record;
	  }
	  return false;
  }
  
  /* private getSingleValue() */
  function getSingleValue() {
	  if ($this->db->next_record()) {
		  return $this->db->Record[0];
	  }
	  return false;
  }

  function getBooleanValue() {
	  if ($this->db->next_record()) {
		  return $this->db->Record[0]=='t';
	  }
	  return false;
  }

  function hasValue() {
	  return $this->db->next_record();
  }

  function makeList($array) {
	  return '('.join(',',$array).')';
  }

  /* private: makeCols($arr) makes a comma delimited list of the keys of $arr */
  function makeCols($arr) {
	  $keys = array();
	  reset($arr);
	  while (list($key, ) = each($arr)) {
		  $keys[] = "$key";
	  }
	  return (join(",", $keys));
  }

  /**
   * Makes a comma and single-quote delimited list
   * of the values of $arr, with special characters escaped 
   */
  function makeVals($arr) {
	  $values = array();
	  reset($arr);
	  while (list( ,$value) = each($arr)) {
		  $values[] = "'" . addslashes($value) . "'";
	  }
	  return (join(",", $values));
  }

  function getType($table, $column) {
	  return $this->tableDefs[$table][$column];
  }

  function convertValues(&$values,$table) {
	  foreach($values as $column => $value) {
		  $type = $this->getType($table,$column);
		  switch ($type) {
			  case 'boolean':
				  $values[$column] = $value=='t' || $value=='1'; break;
			  case 'box':
				  $values[$column] = parseBox($value); break;
		  }			  
	  }
  }

  /* private: makeUpdate($arr) makes a comma delimited list of the name/value
   * pairs of $arr, in the form name='value', with special characters escaped 
   * and primary key omitted
   */
  function makeUpdate($arr,$table='') {
	  $pairs = array();
	  foreach ($arr as $column => $value) {
		  $type = $this->getType($table,$column);
		  if ($value=='' && $type=='integer')
			  $value = 'NULL';
		  if ($type=='boolean') {
			  $value = $value ? 'true' : 'false';
		  } else if ($type=='box') {
			  $value = "'".formatBox($value)."'";
		  } else if ($value!='NULL'|| $type!='integer') {
			  $value = "'".addslashes($value)."'";
		  }
		  $pairs[] =  "$column" . " = " . $value;
	  }
	  return (join(",", $pairs));
  }

  function makeWhere($arr) {
	  $pairs = array();
	  while (list($key, $value) = each($arr)) {
		  $pairs[] =  "$key" . " = '" . addslashes($value) . "'";
	  }
	  return (join(" and ", $pairs));
  }

  /* public: deleteComment() deletes one single comment */
  function deleteComment( $commentID ) {
	  $this->db->query("delete from Comments where (CommentID = '$commentID')");
	  return true;
  }

  /* public: getUser() returns a single user */
  function getUser($userID) {
	  $user = $this->users[$userID];
	  if (!is_array($user)) {
		  $this->db->query("select * from Users where (UserID = '$userID')");
		  $user = $this->getSingleRecord('users');
		  $this->users[$userID] = $user;
	  }
	  return $user;
  }

  function getUserByName($username) {
	  $this->db->query("select * from Users where Username = '$username'");
	  return $this->getSingleRecord('users');
  }

  function getUserByEmail($email) {
	  $this->db->query("select * from Users where Email = '$email'");
	  return $this->getSingleRecord('users');
  }

  /* public: getGroups() returns the groups owned by a user, 
   * or all groups if $userID=="" */
  function getGroups($userID = "") {
	  if ($userID=="") {
		  $this->db->query(
			  "select GroupID,Name,count(userid)".
			  " from Groups left join groupsdef using (groupid)".
			  " group by groupid,name order by name");
		  return $this->getAllRecords('groups');
	  } else {
		  $this->db->query(
			  "select GroupID,Name,count(userid)".
			  " from Groups left join groupsdef using (groupid)".
			  " where (Owner = '$userID') group by groupid,name order by name");
		  return $this->getAllRecords('groups');
	  }
  }

  function getRealGroups() {
	  $this->db->query(
		  "select GroupID,Name from Groups".
		  " where Owner != 1 order by name");
	  return $this->getAllRecords('groups');
  }

  function getIndividualGroups() {
	  $this->db->query(
		  "select GroupID,Name from Groups".
		  " where Owner = 1 order by name");
	  return $this->getAllRecords('groups');
  }

  /* public: getGroupsForPicture() returns the groups allowed to view a picture */
  function getGroupsForPicture($pictureID) {
	  $this->db->query(
		  "select * from Groups,Permissions where".
		  " groups.GroupID=permissions.GroupID and".
		  " permissions.PictureID=$pictureID");
	  return $this->getAllRecordsByIndex("GroupID");
  }

  /* public: getGroup() returns a single group */
  function getGroup($groupID) {
	  return $this->queryRecordByKey('groups',$groupID);
  }

  function getGroupByName($name) {
	  $name = addslashes($name);
	  $this->db->query("select * from Groups where (name = '$name')");
	  return $this->getSingleRecord('groups');
  }

  /* public: addGroup() adds one single group */
  function addGroup($group, $users=array()) {
	  $cols = $this->makeCols($group);
	  $vals = $this->makeVals($group);
	  $this->db->query("insert into Groups ($cols) values ($vals)");
	  $this->db->query("select currval('\"groups_groupid_seq\"')");
	  $groupID = $this->getSingleValue();
	  foreach($users as $userID) {
		  $this->db->query("insert into GroupsDef (UserID,GroupID) ".
						   "values ('$userID','$groupID')");
	  }
	  return true;
  }

  /* public : deleteGroup() delete a group */
  function deleteGroup($groupID) {
	  $this->db->query("delete from groups where GroupID=$groupID");
	  $this->db->query("delete from groupsdef where GroupID=$groupID");
	  $this->invalidateCacheEntry('groups',$groupID);
	  return true;
  }

  function getUsersInGroup($groupID) {
	  $this->db->query(
		  "select * from Users,GroupsDef ".
		  "where (Users.UserID=GroupsDef.UserID and GroupID='$groupID') ".
		  "order by Name");
	  return $this->getAllRecords('users');
  }

  function getUsers() {
	  $this->db->query("select * from Users order by Name");
	  return $this->getAllRecords('users');
  }

  /* public: getAuthors() returns all users who own pictures */
  function getAuthors() { 
	  $this->db->query(
		  "select * from Users where userid in".
		  " (select distinct owner from pictures) order by Name");
	  return $this->getAllRecords('users');
  }

  function getKeywordTypes() {
	  $this->db->query("select * from KeywordTypes");
	  return $this->getAllRecords();
  }

  function getKeywords($pictureID) {
	  $this->db->query("select Value from Keywords where PictureID=$pictureID");
	  return $this->getAllRecords();
  }

  function getTypedKeywords($pictureID) {
	  $this->db->query(
		  "select Value,Libelle_fr,Type from Keywords,KeywordTypes ".
		  "where PictureID=$pictureID and Type=TypeID order by Type,Value");
	  return $this->getAllRecords();
  }

  function getUntypedKeywords($pictureID) {
	  $this->db->query(
		  "select Value from Keywords".
		  " where PictureID=$pictureID and Type is null order by Value");
	  return $this->getAllRecords();
  }

  function getKeywordValues($typeID,$userID) {
	  $this->db->query(
		  "select distinct count(*),Value from keywords".
		  " join pictures using(PictureID)".
		  " where ".($typeID != "" ? "Type=$typeID" : "Type is null").
		  " and ".$this->buildPermsCondition($userID).
		  $this->ownerFilter().
		  " group by Value order by Value");
	  return $this->getAllRecords();
  }

  function addType( $type ) {
	  $cols = $this->makeCols($type);
	  $vals = $this->makeVals($type);
	  $this->db->query("insert into KeywordTypes ($cols) values ($vals)");
	  return true;
  }

  function updateGroup($group, $users) {
	  $groupID = $group['GroupID'];
	  $update = "Name='" . addslashes($group['Name']) . "'";
	  $this->db->query("update Groups set $update where GroupID=$groupID");
	  $this->invalidateCacheEntry('groups',$groupID);
	  $this->db->query("delete from GroupsDef where GroupID=$groupID");
	  foreach($users as $userID) {
		  $this->db->query("insert into GroupsDef (UserID,GroupID) ".
						   "values ('$userID','$groupID')");
	  }
  }

  function addUserInGroup($userID, $groupID) {
	  $this->db->query("select * from GroupsDef where GroupID='$groupID' and UserID='$userID'");
	  if ($this->db->num_rows()==0) {
		  $this->db->query("insert into GroupsDef (UserID,GroupID) ".
						   "values ('$userID','$groupID')");
	  }
  }

  /* create a new user 
   * Takes into account the creation of the SelfGroup
   */
  function addUser($user) {
	  if ($user['Upload']!='true') {
		  $user['Upload']='false';
	  }
	  if ($user['Admin']!='true') {
		  $user['Admin']='false';
	  }
	  if ($user['NotifyComment']!='true') {
		  $user['NotifyComment']='false';
	  }
	  
	  /* insert new user into table */
	  $user['UserID'] = $this->getSequenceNextValue("users_userid_seq");
	  $user['SelfGroup'] = $this->getSequenceNextValue("groups_groupid_seq");
	  $cols = $this->makeCols($user);
	  $vals = $this->makeVals($user);
	  $this->db->query("insert into Users ($cols) values ($vals)");
	  
    /* create the SelfGroup for the user */
	  $this->addGroup(array('GroupID' => $user['SelfGroup'],
							'Name' => $user['Name'],
							'Owner' => 1));
    
	  /* insert the user into his SelfGroup */
	  $this->addUserInGroup($user['UserID'],$user['SelfGroup']);
	  
	  /* insert the new user in the "everybody" group */
	  $this->addUserInGroup($user['UserID'],1);
	  return true;
  }

  function deleteUser($userID,$removePictures) {
	  if ($removePictures) {
		  $this->db->query("delete from pictures where owner = $userID");
		  $this->log(sprintf(_('Removed %d pictures of user %s'),$this->db->affected_rows,$userID));
	  } else {
		  $this->db->query("update pictures set Owner=0 where Owner = $userID");
		  $this->db->query("update rolls set Owner=0 where Owner = $userID");
	  }
	  $this->db->query("delete from groups where groupid = (select selfgroup from users where UserID=$userID)");
	  $this->db->query("delete from users where UserID=$userID");
  }

  /**
   * Tells wether a user has the right to see a picture
   */
  function checkUser($userID,$pictureID) {
	  $this->db->query(
		  "select 1 from pictures where PictureID=$pictureID".
		  " and ".$this->buildPermsCondition($userID));
	  return $this->hasValue();
  }

  function addKeyword($keyword,$userID) {
	  if ($this->checkUser($userID,$keyword['PictureID'])) {
		  $where = $this->makeWhere($keyword);
		  if (! $keyword['Type']) {
			  $where .= " and Type is null";
		  }
		  $this->db->query("select * from Keywords where $where");
		  if ($this->db->num_rows()==0) {
			  $cols = $this->makeCols($keyword);
			  $vals = $this->makeVals($keyword);
			  $this->db->query("insert into Keywords ($cols) values ($vals)");
		  }
		  return true;
	  } else {
		  return false;
	  }
  }

  function deleteKeyword($pictureID,$value,$typeID,$userID) {
	  if ($this->checkUser($userID,$pictureID)) {
		  $value = addslashes($value);
		  if ($typeID) {
			  $this->db->query("delete from keywords ".
							   "where PictureID=$pictureID and ".
							   "Type=$typeID and Value='$value'");
		  } else {
			  $this->db->query("delete from keywords ".
							   "where PictureID=$pictureID and ".
							   "Type is null and Value='$value'");
		  }
		  return true;
	  } else {
		  return false;
    }
  }

  /* Returns pictures which have a keyword that exactly match the given type and value */
  function getKeywordSearch($typeID,$value,$userID,$offset,$count) {
	  $accessWhere = " and " . $this->buildPermsWhere($userID);
	  $value = addslashes($value);
	  if ($typeID==0) {
		  $this->db->query(
			  "select ".$this->distinctOn()." * from Pictures,Permissions,Groupsdef where ".
			  "Pictures.PictureID in ".
			  "(select PictureID from Keywords where ".
			  "Type is null and Value='$value')".
			  $accessWhere.$this->orderBy());
	  } else { 
		  $this->db->query(
			  "select ".$this->distinctOn()." * from Pictures,Permissions,Groupsdef where ".
			  "Pictures.PictureID in ".
			  "(select PictureID from Keywords where ".
			  "Type=$typeID and Value='$value')".
			  $accessWhere.$this->orderBy());
	  }
	  return array(
		  $this->db->num_rows(),
		  $this->getRecords($offset,$count));
  }

  function getMultipleKeywordSearch($values,$matchAll,$userID,$offset,$count) {
	  $this->doMultipleKeywordSearch($values,$matchAll,$userID);
	  return array(
		  $this->db->num_rows(),
		  $this->getRecords($offset,$count));
  }

  /**
   * Query pictures which have a keyword that exactly match one
   * value in the given array 
   *
   * @param values keyword values to search
   * @param matchAll wether to search for pictures that match all given values
   * @param userID id of user doing the search
   */
  function doMultipleKeywordSearch($values,$matchAll,$userID) {
	  if (count($values)==0)
		  return;
	  $accessWhere = $this->buildPermsWhere($userID);
	  reset($values);
	  $lValues = array();
	  while(list($key,$value)=each($values)) {
		  $value = addslashes($value);
		  $lValues[] = "Keywords.value='$value'";
	  }
	  $valueClause = join(" or ",$lValues);
	  $nbValues = count($lValues);
	  $tmp_select ="select PictureID,count(PictureID) as count ".
		   "from Keywords where $valueClause group by PictureID".
		   ($matchAll?" having count(PictureID)=$nbValues":"");
	  $this->db->query(
		  "select distinct Pictures.*,tmp.count from ".
		  "Pictures,($tmp_select) as tmp,Permissions,Groupsdef where ".
		  "pictures.PictureID=tmp.PictureID and ($accessWhere) ".
		  "order by count desc".
		  ($this->sortMethod!=0 ? ",".$this->orderbyArray[$this->sortMethod] : ''));
  }

  function getPicturesOwnedBy($userID) {
	  $this->db->query("select * from pictures where owner=$userID");
	  return $this->getAllRecords('pictures');
  }

  /**
   * Gets all the pictures a user can see.
   */
  function getAllPictures($userID,$offset,$count,$order) {
	  $accessWhere = $this->buildPermsWhere($userID);
	  $this->db->query(
		  "select distinct on (Pictures.$order,Pictures.RollID,Pictures.FrameID)".
		  " * from Pictures,Permissions,GroupsDef where ".
		  $accessWhere." order by Pictures.$order desc,".
		  "Pictures.RollID,Pictures.FrameID");
	  return array($this->db->num_rows(),
                   $this->getRecords($offset,$count));
  }

  function getPicturesInQuery($query,$offset,$count) {
	  $this->db->query($query);
	  return array($this->db->num_rows(),
                   $this->getRecords($offset,$count));
  }

  function getPicturesInQuery2($query,$offset,$limit) {
	  $count = $this->countPicturesInQuery($query);
	  $query->setLimit($offset,$limit);
	  $this->db->query($this->getSQL($query));
	  return array($count,
                   $this->getAllRecords());
  }

  function addRate($rate) {
	  if ($rate['UserID'] != 0) {
		  $this->db->query("select * from Notes where ".
						   "PictureID=".$rate['PictureID']." and ".
						   "UserID=".$rate['UserID']);
		  if ($this->db->num_rows()==0) {
			  $cols = $this->makeCols($rate);
			  $vals = $this->makeVals($rate);
			  $this->db->query("insert into Notes ($cols) values ($vals)");
		  } else {
			  $this->nextRecord();
			  $previous = $this->db->f('Note');
			  $this->db->query("update Notes set Note=".$rate['Note']." where ".
							   "PictureID=".$rate['PictureID']." and ".
							   "UserID=".$rate['UserID']);
		  }
	  } else {
		  $cols = $this->makeCols($rate);
		  $vals = $this->makeVals($rate);    
		  $this->db->query("insert into Notes ($cols) values ($vals)");
	  }
	  // update rate summary info for picture
	  $this->db->query(
		  "update pictures set ".
		  (isset($previous) ? "" : "NbRates=NbRates+1, ").
		  (isset($previous) ? 
		   "SumRates = (SumRates - $previous + ".$rate['Note']."),  "
		   : "SumRates = (SumRates + ".$rate['Note']."), ").
		  "MaxRate = (select max(Note) from Notes where PictureID=".$rate['PictureID']."), ".
		  "MinRate = (select min(Note) from Notes where PictureID=".$rate['PictureID'].")".
		  "where PictureID=".$rate['PictureID']);
	  $this->db->query(
		  "update pictures set ".
		  "AverageRate=float4(SumRates) / NbRates ".
		  "where PictureID=".$rate['PictureID']);
  }
  
  function getRate($pictureID,$userID) {
	  $this->db->query(
		  "select Note from notes where".
		  " PictureID=$pictureID and UserID=$userID");
	  return $this->getSingleValue();
  }

  /* SELECT DISTINCT should be used with this where clause 
   * since it may return duplicates rows.
   *
   * A picture can be seen by a user if :
   *   1. He is the owner of that picture
   *   2. He belongs to the group of that picture
   *   3. The group of the picture is `1' (everybody) */
  function buildPermsWhere($userID) {
	  return  "(pictures.PictureID=permissions.PictureID and".
		  " groupsdef.UserID=$userID and".
		  " groupsdef.GroupID=permissions.GroupID) ".
		  $this->ownerFilter();
  }

  function buildPermsCondition($userID) {
	  return  "pictures.PictureID IN ".
		  "(SELECT PictureID FROM permissions JOIN groupsdef USING(GroupID)".
		  " WHERE groupsdef.UserID = $userID)";
  }

  function buildTopicPermsCondition($userID,$access='') {
	  return  "topics.TopicID IN ".
		  "(SELECT TopicID FROM topicperms JOIN groupsdef USING(GroupID)".
		  " WHERE groupsdef.UserID = $userID".
		  ($access?" AND topicperms.Access IN ('".implode("','",$access)."')" : '').")";
  }

  function buildPermsSelect($userID) {
	  return  "(select permissions.PictureID from permissions,groupsdef ".
		  "where permissions.PictureID=Pictures.PictureID and ".
		  "groupsdef.UserID=$userID and ".
		  "groupsdef.GroupID=permissions.GroupID) ".
		  (($this->authorFilter!='')?("and Owner=".$this->authorFilter):"");
  }

  function getSequenceLastValue($sequence) {
	  $this->db->query("select last_value from $sequence");
	  $this->db->next_record();
	  return $this->db->f("last_value");
  }

  function getSequenceNextValue($sequence) {
	  $this->db->query("select nextval('$sequence')");
	  $this->db->next_record();
	  return $this->db->f("nextval");
  }

  /* this is just a hack to fix postgresql broken case sensitiveness */
  function fixColumnNames(&$record) {
	  if (is_array($record)>0) {
		  reset($record);
		  while (list($key,$value) = each($record)) {
			  if ($this->dico[$key]!="") {
				  $record[$this->dico[$key]] = $value;
			  }
		  }
	  }
  }

  function isSelected($pictureID,$userID) {
	  $this->db->query(
		  "select * from selections".
		  " where PictureID=$pictureID and UserID=$userID");
	  if ($this->db->next_record()) {
		  return true;
	  } else {
		  return false;
	  }
  }

  function selectedQuery($userID) {
	  return 
		  "select * from pictures ".
		  "where PictureID in (select PictureID from selections where UserID=$userID)".		  
		  $this->ownerFilter().
		  $this->orderBy();
  }

  function getSelectedPictures($userID, $offset=-1, $count=-1) {
	  $this->db->query($this->selectedQuery($userID));
	  return array(
		  $this->db->num_rows(),
		  $offset==-1 ? $this->getAllRecords() : $this->getRecords($offset, $count));
  }

  /** Returns the pictureID of selected pictures */
  function getSelection($userID) {
	  $selection = $this->selections[$userID];
	  if (!is_array($selection)) {
		  $this->db->query(
			  "select PictureID from selections where UserID=$userID");
		  $selection = $this->getAllSingleValues(true);
		  $this->selections[$userID] = $selection;
	  }
	  return $selection;
  }

  function getPictureInSelected($pictureID,  $position, $userID) {
	  return 
		  $this->getPictureInQuery(
			  $this->selectedQuery($userID), $pictureID, $position);
  }

  function addToSelection($pictureID,$userID) {
	  if (!$this->isSelected($pictureID,$userID)) {
		  $this->db->query(
			  "insert into selections (PictureID,UserID) ".
			  "values ($pictureID,$userID)");
		  unset($this->selections[$userID]);
	  }
  }

  function clearTopics($pictureID) {
	  $this->db->query("DELETE FROM topicscontent WHERE PictureID=$pictureID");
  }

  function addTopicToSelection($topicID,$recurse,$userID) {
	  $this->db->query(
		  "insert into selections (PictureID,UserID)".
		  " select PictureID,$userID from topicscontent ".
		  " where TopicID=$topicID".
		  ($recurse ? "" : " and Direct=true").
		  " and PictureID not in (select PictureID from selections where userid=$userID)");
  }

  function removeFromSelection($pictureID,$userID) {
	  $this->db->query(
		  "delete from selections".
		  " where PictureID=$pictureID and UserID=$userID");
	  unset($this->selections[$userID]);
  }

  function clearSelection($userID) {
	  $this->db->query("delete from selections where UserID=$userID");
  }

  /**
   * Adds a picture to the front page of a topic
   */
  function addToFrontPage($pictureID,$topicID) {
	  $this->db->query("select * from topicsfrontpage where TopicID=$topicID and PictureID=$pictureID");
	  if (!$this->db->next_record()) {
		  $this->db->query("insert into topicsfrontpage (topicid,pictureid)".
						   " values ($topicID,$pictureID)");
	  }
  }

  /**
   * Remove a picture from the front page of a topic
   */
  function removeFromFrontPage($pictureID,$topicID) {
	  $this->db->query("DELETE FROM topicsfrontpage where TopicID=$topicID and PictureID=$pictureID");
  }

  function log($str) {
	  global $myRender;
	  if ($myRender)
		  $myRender->log($str);
  } 

  function moveUp($pictureID, $topicID, $offset = 1) {
	  $this->begin();
	  $this->db->query(
		  "SELECT Index FROM topicscontent WHERE".
		  " PictureID=$pictureID AND TopicID = $topicID AND Direct = true");
	  $pos = $this->getSingleValue();
	  $newpos = $pos - $offset;
	  $this->db->query(
		  "UPDATE topicscontent SET Index = Index+1 WHERE".
		  " TopicID = $topicID AND Index >= $newpos AND Index < $pos AND Direct = true");
	  $this->db->query(
		  "UPDATE topicscontent SET Index = $newpos WHERE".
		  " TopicID = $topicID AND PictureID = $pictureID AND Direct = true");
	  $this->commit();
  }

  function moveDown($pictureID, $topicID, $offset = 1) {
	  $this->begin();
	  $this->db->query(
		  "SELECT Index FROM topicscontent WHERE".
		  " PictureID=$pictureID AND TopicID = $topicID AND Direct = true");
	  $pos = $this->getSingleValue();
	  $newpos = $pos + $offset;
	  $this->db->query(
		  "UPDATE topicscontent SET Index = Index-1 WHERE".
		  " TopicID = $topicID AND Index <= $newpos AND Index > $pos AND Direct = true");
	  $this->db->query(
		  "UPDATE topicscontent SET Index = $newpos WHERE".
		  " TopicID = $topicID AND PictureID = $pictureID AND Direct = true");
	  $this->commit();	  
  }

  function movePageDown($topicID, $pageNum, $offset = 1) {
	  $newpos = $pageNum + $offset;
	  $this->begin();
	  $this->db->query(
		  "UPDATE pageelements SET PageNum = -1".
		  " WHERE TopicID = $topicID AND PageNum = $pageNum");
	  $this->db->query(
		  "UPDATE pageelements SET PageNum = PageNum-1".
		  " WHERE TopicID = $topicID AND PageNum <= $newpos AND PageNum > $pageNum");	  
	  $this->db->query(
		  "UPDATE pageelements SET PageNum = $newpos".
		  " WHERE TopicID = $topicID AND PageNum = -1");
	  $this->commit();
  }

  function movePageUp($topicID, $pageNum, $offset = 1) {
	  $newpos = $pageNum - $offset;
	  $this->begin();
	  $this->db->query(
		  "UPDATE pageelements SET PageNum = -1".
		  " WHERE TopicID = $topicID AND PageNum = $pageNum");
	  $this->db->query(
		  "UPDATE pageelements SET PageNum = PageNum+1".
		  " WHERE TopicID = $topicID AND PageNum >= $newpos AND PageNum < $pageNum");	  
	  $this->db->query(
		  "UPDATE pageelements SET PageNum = $newpos".
		  " WHERE TopicID = $topicID AND PageNum = -1");
	  $this->commit();
  }

  function getPage($topicID,$num) {
	  $this->db->query(
		  "SELECT * FROM pages_view WHERE TopicID=$topicID AND PageNum=$num");
	  if (!$this->nextRecord())
		  return false;
	  $page = $this->db->Record;
	  $this->db->query(
		  "SELECT * FROM imageelements JOIN pictures USING (PictureID)".
		  " WHERE TopicID=$topicID AND PageNum=$num ORDER BY Id");
	  $page['Images'] = $this->getAllRecords(array('pictures','pageelements'));

	  $this->db->query(
		  "SELECT * FROM textelements ".
		  " WHERE TopicID=$topicID AND PageNum=$num ORDER BY Id");
	  $page['Texts'] = $this->getAllRecords('pageelements');
	  return $page;
  }

  function getPageCount($topicID) {
	  $this->db->query(
		  "SELECT count(PageNum) from pages WHERE TopicID=$topicID");
	  return $this->getSingleValue();
  }

  function addPage($topicID) {
	  $this->db->query(
		  "INSERT INTO pages (TopicID,PageNum)".
		  " Values ($topicID,(select count(PageNum) FROM pages WHERE topicid=$topicID))");
  }

  function getTextElement($topicID,$num,$id) {
	  $this->db->query(
		  "SELECT * FROM textelements".
		  " WHERE TopicID=$topicID AND PageNum=$num AND Id=$id");
	  return $this->getSingleRecord('pageelements');
  }

  function getImageElement($topicID,$num,$id) {
	  $this->db->query(
		  "SELECT * FROM imageelements JOIN pictures USING (PictureID)".
		  " WHERE TopicID=$topicID AND PageNum=$num AND Id=$id");
	  return $this->getSingleRecord('imageelements');
  }

  function addImageElement($image) {
	  $this->db->query(
		  "SELECT COUNT(Id) FROM imageelements".
		  " WHERE TopicID=${image['TopicID']} AND PageNum=${image['PageNum']}");
	  $image['Id'] = $this->getSingleValue();
	  $image['Box'] = formatBox($image['Box']);
	  $this->insertObject('imageelements',$image);
	  return $image['Id'];
  }

  function addTextElement($text) {
	  $this->db->query(
		  "SELECT COUNT(Id) FROM textelements".
		  " WHERE TopicID=${text['TopicID']} AND PageNum=${text['PageNum']}");
	  $text['Id'] = $this->getSingleValue();
	  $text['Box'] = formatBox($text['Box']);
	  $this->insertObject('textelements',$text);
	  return $text['Id'];
  }

  /**
   * Generic update method
   * @table table to update, must have an entry in $pkeys
   * @param field values
   */
  function updateObject($table,$object) {
	  $where = '';
	  $pkey = $this->pkeys[$table];
	  if (is_array($pkey)) {
		  $sep = '';
		  foreach ($pkey as $key) {
			  $where .= "$sep $key = '${object[$key]}'";
			  $sep = " AND ";
		  }
	  } else {
		  $where = "$pkey = '{$object[$pkey]}'";
	  }
	  $this->db->query(
		  "UPDATE $table SET ".$this->makeUpdate($object,$table).
		  " WHERE $where");
  }

  function insertObject($table,$object) {
	  $columns = $this->makeCols($object);
	  $values = $this->makeVals($object,$table);
	  $this->db->query("INSERT INTO $table ($columns) VALUES ($values)");
  }

  function upgrade($major,$minor) {
	  $this->db->query("SELECT * FROM pg_class where relkind = 'r' AND relname = 'params'");
	  if (!$this->db->next_record()) {
		  $this->db->query("CREATE TABLE params (Name varchar PRIMARY KEY, Value varchar)");
		  $this->setParameter('version','0.0');
		  $this->setParameter('dbversion','0');
	  }
	  $current = $this->getParameter('dbversion');
	  if ($current < 1) {
		  // add topicperms
		  $this->db->query("START TRANSACTION");
		  $this->db->query(
			  "CREATE TABLE topicperms (".
			  "TopicID integer NOT NULL REFERENCES topics ON DELETE CASCADE,".
			  "GroupID integer NOT NULL REFERENCES groups ON DELETE CASCADE,".
			  "Access character,".
			  "PRIMARY KEY (TopicID,GroupID,Access))");
		  $this->db->query("INSERT INTO groups VALUES (-1,'Admin',1)");
		  $this->db->query("INSERT INTO GroupsDef (UserID,GroupID) VALUES (1,-1)");
		  $this->db->query("UPDATE users SET selfgroup = -1 where UserID = 1");
		  $topics = $this->getAll('topics','TopicID');
		  foreach($topics as $topic) {
			  // Give read access to everybody on all topics
			  $this->db->query("INSERT INTO topicperms VALUES (${topic['TopicID']},'1','".READ."')");
			  // Admin owns every topics
			  $this->db->query("INSERT INTO topicperms VALUES (${topic['TopicID']},'-1','".OWNER."')");
		  }
		  $this->setParameter('dbversion',1);
		  $this->db->query("COMMIT");
	  }
	  if ($current < 2) {
		  // Add topicscontent.index
		  $this->db->query("START TRANSACTION");
		  $this->db->query("ALTER TABLE topicscontent ADD Index integer");
		  $this->db->query("SELECT * FROM topicscontent WHERE Direct = true ORDER BY TopicID");
		  $db = $this->newDB();
		  $topicID = '';
		  while($this->nextRecord()) {
			  $pictureID = $this->db->Record['PictureID'];
			  if ($this->db->Record['TopicID'] != $topicID) {
				  $pos = 0;
				  $topicID = $this->db->Record['TopicID'];
			  } else {
				  $pos++;
			  }
			  $db->query(
				  "UPDATE topicscontent SET Index = $pos WHERE".
				  " TopicID = $topicID AND PictureID = $pictureID AND Direct = true");
		  }
		  $this->setParameter('dbversion',2);
		  $this->db->query("COMMIT");
	  }

	  if ($current < 3) {
		  // Add topicscontent.index
		  $this->db->query("START TRANSACTION");
		  $this->db->query("ALTER TABLE topics ADD Book boolean");
		  $this->db->query("UPDATE topics SET Book = false");
		  $this->db->query("ALTER TABLE topics ALTER Book set NOT NULL");
		  $this->db->query("ALTER TABLE topics ALTER Book set DEFAULT false");

		  $this->db->query("DELETE FROM active_sessions");
		  $this->db->query("ALTER TABLE active_sessions DROP changed");
		  $this->db->query("ALTER TABLE active_sessions ADD changed bigint");
		  $this->db->query("ALTER TABLE active_sessions ALTER changed SET NOT NULL");
		  $this->db->query("CREATE INDEX session_changed ON active_sessions (changed)");

		  $this->setParameter('dbversion',3);
		  $this->db->query("COMMIT");
	  }

	  if ($current < 4) {
		  $this->db->query("START TRANSACTION");
		  $this->db->query(
			  "CREATE TABLE pages (".
			  "TopicID integer NOT NULL REFERENCES topics,".
			  "PageNum integer NOT NULL,".
			  "PRIMARY KEY (TopicID,PageNum))");
		  $this->db->query("ALTER TABLE topics ADD Width smallint");
		  $this->db->query("ALTER TABLE topics ADD Height smallint");
		  $this->db->query(
			  "CREATE VIEW pages_view AS".
			  " SELECT pages.*,topics.width,topics.height".
			  " FROM pages JOIN topics USING(TopicID)");
		  $this->setParameter('dbversion',4);
		  $this->db->query("COMMIT");
	  }
	  if ($current < 5) {
		  $this->db->query("START TRANSACTION");
		  $this->db->query("ALTER TABLE users DROP COLUMN picturesize");
		  $this->db->query("ALTER TABLE users DROP COLUMN pictureinterval");
		  $this->db->query("ALTER TABLE users DROP COLUMN showemptytopics");
		  $this->setParameter('dbversion',5);
		  $this->db->query("COMMIT");
	  }
	  $this->setParameter('version',$major.'.'.$minor);
  }

  function getParameter($name) {
	  $this->db->query("SELECT Value FROM params WHERE Name = '$name'");
	  return $this->getSingleValue();
  }

  function setParameter($name,$value) {
	  $this->db->query("SELECT * FROM params WHERE Name = '$name'");
	  if ($this->db->next_record()) {
		  $this->db->query("UPDATE params SET Value = '$value' WHERE Name = '$name'");
	  } else {
		  $this->db->query("INSERT INTO params VALUES ('$name','$value')");
	  }
  }
} /* end of class SloozeCtSql */

function parseBox($box) {
	if (ereg('\(([0-9]+),([0-9]+)\),\(([0-9]+),([0-9]+)\)',$box,$res)) {
		return array(
			'left' => min($res[1],$res[3]),
			'top' => max($res[2],$res[4]),
			'right' => max($res[1],$res[3]),
			'bottom' => min($res[2],$res[4]),
			'width' => abs($res[1]-$res[3]),
			'height' => abs($res[2]-$res[4]));
	} else {
		return nil;
	}
}

function formatBox($b) {
	return "(${b['left']},${b['top']}),(${b['right']},${b['bottom']})";
}

?>
