/*
  Top 10, a racing simulator
  Copyright (C) 2000, 2001  Deneux Johann
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Deneux Johann:       deneux@ifrance.com
*/

#ifndef LWREAD_H
#define LWREAD_H

#include <string>
#include <iostream>
#include <vector>
#include <map>

#include "math/Vertex.hh"
#include "util/error.hh"
#include "PolygonSet.hh"

namespace top10 {
  namespace helpers {
    using top10::math::Vector;

    /**
       Loads a LWO file (lightwave object) to a set of polygons
    */
    class LWRead: public PolygonSet
    {
    public:
      DECL_ERROR(Error);
      DECL_ERROR_FROM(ParseError, Error);

    public:
      /**
	 Loads a LWO file to set of triangles
      */
      LWRead(std::istream& lwo_file_stream);

    private:
      // TODO: These types depend on the host type
      typedef unsigned int ID4;

      typedef signed char I1;
      typedef signed short int I2;
      typedef signed int I4;

      typedef unsigned char U1;
      typedef unsigned short int U2;
      typedef unsigned int U4;

      typedef float F4;
      typedef char* S0;

      struct COL4 {
	U1 red, green, blue, pad;
      };

      struct VEC12 {
	F4 X, Y, Z;
      };

      // Integer percentage [0, 256]
      typedef I2 IP2;
      // Fractionnal part [0.0, 1.0]
      typedef F4 FP4;
      // Degrees
      typedef F4 DEG4;
      // Filename
      typedef S0 FNAM0;

      /* File TO Host conversion functions (byte-order) */
      static inline void ftoh(I4& x) {
#ifndef WORDS_BIGENDIAN
	x = ((x>>24) & 0xFF) | ((x>>8) & 0xFF00) | ((x<<8) & 0xFF0000) | ((x<<24) & 0xFF000000);
#endif 
      }

      static inline void ftoh(U4& x) {
#ifndef WORDS_BIGENDIAN
	x = ((x>>24) & 0xFF) | ((x>>8) & 0xFF00) | ((x<<8) & 0xFF0000) | ((x<<24) & 0xFF000000);
#endif 
      }

      static inline void ftoh(I2& x) {
#ifndef WORDS_BIGENDIAN
	x = ((x>>8) & 0xFF) | ((x<<8) & 0xFF00);
#endif
      }

      static inline void ftoh(U2& x) {
#ifndef WORDS_BIGENDIAN
	x = ((x>>8) & 0xFF) | ((x<<8) & 0xFF00);
#endif
      }

      static inline bool equals(ID4 x, const char tag[4]) {
	const ID4* xtag = (const ID4*)tag;
	return *xtag == x;
      }

      static inline void ftoh(FP4& x) {
	U4* hack = (U4*)&x;
	ftoh(*hack);
      }

    private:
      std::map<std::string, COL4> lw_surfaces;
      std::vector<VEC12> vertices;
      std::istream& in;

    private:
      /**
	 Gets next id tag from LWO file.
	 The value returned has been translated to host byte order.
      */
      ID4 getNextTag();

      int readPolygons();
  
      int ignoreChunk(const char* reason);

      int readSurface();

      int readSurfaceNames();

      int readVertices();

      //! Build PolygonSet::surfaces from lw_surfaces.
      void fixateSurfaces();
    };
  };
};

#endif
