#ifndef TOP10_HELPERS_POLYGONSET
#define TOP10_HELPERS_POLYGONSET

#include <string>
#include <vector>
#include <list>
#include "math/Matrix.hh"
#include "math/Triangle.hh"
#include "physX/Surface.hh"

namespace top10 {
  namespace helpers {

    /**
       One texture coordinate.
    */
    struct TexCoord
    {
      float x;
      float y;
    };

    /**
       Appearance of a face.
    */
    struct Surface
    {
      std::string name;
      unsigned char r,g,b,a;   // a = opacity
      std::string texture_filename;
      top10::physX::Surface phys_surface;   // Physical properties of the surface
    };

    //! Used to assign physical properties to surfaces.
    class SurfaceFilter
    {
    public:
      //! Returns true if there is a match. In that case phys_surface contains the mathing physical surface
      virtual bool match(top10::helpers::Surface, top10::physX::Surface* phys_surface) =0;

      //! Write the properties of the filter to a stream ("serialize" it)
      virtual void writeOut(std::ostream& out) const =0;

      virtual ~SurfaceFilter() {}
    };

    //! Surface filter based on the name of the surface
    class Substring_SurfaceFilter: public SurfaceFilter {
    public:
      Substring_SurfaceFilter(std::string substring, top10::physX::Surface phys_surface);
      bool match(top10::helpers::Surface, top10::physX::Surface* phys_surface);
      void writeOut(std::ostream& out) const;

    private:
      std::string substring;
      top10::physX::Surface phys_surface;
    };

    /**
       Class describing a single polygon
    */
    struct Polygon
    {
      std::vector<top10::math::Vector> p;
      std::vector<TexCoord> texels;
      int surface_name;

      void rotate(const top10::math::Matrix3&);
      void translate(top10::math::Vector);
    };

    //! A Mesh is a made of several polygons
    struct Mesh
    {
      //! Index of the first polygon in the PolygonSet
      int begin;
      //! Index after the last polygon in the PolygonSet
      int end;
      //! Name of this object
      std::string name;
      bool hidden;
    };

    /**
       Class describing a set of polygons
    */
    struct PolygonSet: public std::vector<Polygon>
    {
      std::vector<std::string> surf_names;
      std::vector<Surface> surfaces;

      std::vector<std::string> mesh_names;
      std::vector<Mesh> meshes;

      void transform(top10::math::Matrix4 m);
      void setPhysSurfaces(SurfaceFilter& filter);

      static PolygonSet* loadNew(std::string filename);

      //! Check if this ray intersects a mesh. Returns the index of the mesh or -1 if no intersection
      int intersectRay(top10::math::Vector origin, top10::math::Vector direction) const;

      //! Check if this ray intersects a polygon. Returns the index of the polygon or -1.
      int intersectRayPolygon(top10::math::Vector origin, top10::math::Vector direction) const;

      int findMeshIndex(std::string mesh_name) const;
    };

    template<typename Shape>
    bool intersect(const Polygon& polygon, const Shape& shape) {
      bool overlap = false;
      if (polygon.p.size() >= 3) {
	for (int i=polygon.p.size()-1; !overlap && i>1; --i) {
	  top10::math::Triangle triangle;
	  triangle.p[0] = polygon.p[0];
	  triangle.p[1] = polygon.p[i-1];
	  triangle.p[2] = polygon.p[i];
	    
	  overlap = top10::math::intersect(shape, triangle);
	}
      }
      return overlap;
    }

    template<typename Shape>
    bool intersect(const std::vector<Polygon>& polygons, const Shape& shape) {
      bool overlap = false;
      for (std::vector<Polygon>::const_iterator poly = polygons.begin();
	   !overlap && poly != polygons.end();
	   ++poly) {
	overlap = intersect(*poly, shape);
      }
      return overlap;
    }

  };
};

#endif
