/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_PHYSX_OCTREE_HH
#define TOP10_PHYSX_OCTREE_HH

#include <string>
#include "helpers/GenericOctree.hh"
#include "helpers/PolygonSet.hh"
#include "math/intersections.hh"

namespace top10 {

  namespace helpers {
    //! Function used when building a GenericOctree<ShapeT>
    void GenericOctree__getBoundingBox(const top10::helpers::Polygon& shape,
				       double& minX, double& maxX,
				       double& minY, double& maxY,
				       double& minZ, double& maxZ);

    //! Function used when using a GenericOctree<ShapeT>
    inline OverlapType GenericOctree__overlap(const top10::math::AxisAlignedBox b, const top10::math::Box& client_volume) {
      if (top10::math::intersect(b, client_volume)) return Intersect;
      else return None;
    }

    //! Function used when using a GenericOctree<ShapeT>
    inline OverlapType GenericOctree__overlap(const top10::math::AxisAlignedBox b, const top10::math::AxisAlignedBox& client_volume) {
      if (top10::math::intersect(b, client_volume)) return Intersect;
      else return None;
    }

    //! Function used when building a GenericOctree<ShapeT>
    inline bool GenericOctree__intersect(const top10::math::AxisAlignedBox b, const top10::helpers::Polygon& shape) {
      return top10::helpers::intersect(shape, b);
    }
  };

  namespace physX {
    class Octree: public top10::helpers::GenericOctree<top10::helpers::Polygon> {
    public:
      //! Create an octree of (some) polygons
      /*!
	\param polys pointers to a PolygonSet
	\param colliding_mesh_names Set of names of meshes to be included in this octree
      */
      Octree(const top10::helpers::PolygonSet* polys,
	     std::set<std::string> colliding_mesh_names);

      //! Create an octree of polygons
      Octree(const top10::helpers::PolygonSet*);

      //! Default constructor
      Octree(): top10::helpers::GenericOctree<top10::helpers::Polygon>() {}

    private:
      ShapeVec poly_vec;
    };
  };
};
#endif
