/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include <cmath>

#include "track/Track.hh"
#include "Wheel.hh"
#include "World.hh"

using namespace top10::physX;
using top10::math::OrthoNorm3;

Wheel::Wheel(top10::track::Track* _track,
	     double _k, double _side_k,
	     double _radius,
	     const char* name, bool debug):
  top10::util::Debug(name, debug),
  origin(0.0, 0.0, 0.0),
  height(0.0),
  track(_track), angle(0),
  radius(_radius),
  collision_speed(-0.5),
  long_k(_k),
  side_k(_side_k),
  drive_torque(0.0),
  w_speed(0.0), 
  ground_collision(None)
{
  grip_curve.addXY(0, 1);
  grip_curve.addXY(0.1, 1);
  grip_curve.addXY(0.15, 0.8);
  grip_curve.addXY(0.5, 0.7);

  slip_angle_k_curve.addXY(0 ,0.8);
  slip_angle_k_curve.addXY(0.5, 1);
  slip_angle_k_curve.addXY(2, 0.75);
}

void Wheel::dumpTo(std::ostream& out) const
{
  assert(0); // Not yet implemented
}

void Wheel::loadFrom(std::istream& in)
{
  assert(0); // Not yet implemented
}

void Wheel::collideTrack(Vector speed, double dt)
{
  if (is_debug()) printDebugStart();
  if (is_debug()) {
    debug_stream<<"Origin:"<<origin<<std::endl;
    debug_stream<<"Height:"<<height<<std::endl;
    debug_stream<<"Speed:"<<speed<<std::endl;
  }

  impulse_speed = Vector(0.0, 0.0, 0.0);

  // Find the triangle the wheel lies over
  if (!track->getSurfacePatch(origin, surface)) {
    surface = SurfacePatch(Vector(0,-10000.0,0), Vector(0,1,0), SurfaceSlip());
  }

  // Distance between wheel center and ground
  double d = (origin+orientation*Vector(0, height, 0)) * surface.normal - surface.d;

  // Handle bumps
  d += surface.bumps*sin(origin.x)*sin(origin.z);

  if (is_debug())
    debug_stream<<"d = "<<d<<std::endl;

  Vector up = orientation*Vector(0,1,0);
  double vertical_speed = speed * surface.normal;
  double upXnorm = up*surface.normal;

  // Adjust height
  double dh = 0.0;
  if (upXnorm > 1e-4) {
    dh = (radius-d)/upXnorm;
    double h2 = height + dh;
    if (h2 < 0) h2 = 0;
    if (h2 > 0.2) h2 = 0.2;
    dh = h2 - height;
    height = h2;
  }
  else if (upXnorm < -1e-4) {
    dh = (radius-d)/upXnorm;
    double h2 = height + dh;
    if (h2 > 0) h2 = 0;
    if (h2 < -0.2) h2 = -0.2;
    dh = h2 - height;
    height = h2;
  }

  // We don't want a wheel under the ground, do we?  
  if (d < radius) {
    if (upXnorm > 0.8)
      ground_collision = Contact;
    else ground_collision = Collision;
    impulse_speed = up*dh/dt;
  }
  else if ((d < radius + 0.01) &&
	   ((vertical_speed < 0 && vertical_speed > collision_speed) ||
	    (vertical_speed >= 0 && vertical_speed < -collision_speed))) {
    if (upXnorm > 0.8)
      ground_collision = Contact;
    else ground_collision = Collision;
    impulse_speed = up*dh/dt;
  }
  else {
    ground_collision = None;
  }

  if (is_debug()) {
    debug_stream<<"vertical_speed="<<vertical_speed<<std::endl;
    switch(ground_collision) {
    case None: debug_stream<<"No collision"; break;
    case Collision: debug_stream<<"Collision"; break;
    case Contact: debug_stream<<"Contact"; break;
    }
    debug_stream<<std::endl;
  }

  if (is_debug()) printDebugEnd();
}

void Wheel::computeForces(Vector speed, double dt)
{
  if (is_debug())
    printDebugStart();

  if (is_debug()) {
    debug_stream<<"center = "<<origin<<std::endl;
    debug_stream<<"height = "<<height<<std::endl;
    debug_stream<<"v_speed = "<<speed*(orientation*Vector(0,1,0))<<std::endl;
    debug_stream<<"w_speed = "<<w_speed<<std::endl;
    debug_stream<<"force = "<<external_force<<std::endl;
  }

  // Default init
  friction_force = Vector(0,0,0);
  global_force = Vector(0,0,0);
  sliding_speed = Vector(0,0,0);
  torque = 0;

  // Compute vectors relative to the wheel (see doc/wheel.png)
  //  Vector s1, s2, normal;
  {
    Vector v1, v2, v3;
    v1 = orientation*Vector(0,0,1);
    v3 = orientation*Vector(-1,0,0);
    v2 = orientation*Vector(0,1,0);
   
    normal = v1*cos(angle) - v3*sin(angle);
    s2 = v1*sin(angle) + v3*cos(angle);
    s1 = (projectV(surface, s2)); s1/=s1.size();
  }

  // Size of the vertical component of the external forces
  double vertical_force_size = external_force * surface.normal;
  if (vertical_force_size < 0) {
    // Vector from the center of the wheel to the plane containing the triangle
    Vector r = -surface.normal * radius;

    if (ground_collision != None) {
      // Normal reaction from the ground
      global_force += -surface.normal*vertical_force_size;

      // Drag
      Vector surface_speed = projectV(surface, speed);
      double sliding_speed_size = surface_speed.size();
      
      if (sliding_speed_size > SMALL_VECTOR) {
	// Reactive friction from the ground, simplified expression
	Vector reaction = surface.drag*(-surface_speed/sliding_speed_size) * -vertical_force_size;
	
	global_force += reaction;
	torque += (-r ^ reaction)*normal;
      }
    }

    if (ground_collision == Collision) {
      Vector surface_speed = projectV(surface, speed);
      double sliding_speed_size = surface_speed.size();
      
      if (sliding_speed_size > SMALL_VECTOR) {
	// Reactive friction from the ground, simplified expression
	Vector reaction = (-surface_speed/sliding_speed_size) * -vertical_force_size;
	
	global_force += reaction;
	torque += (-r ^ reaction)*normal;
      }
    }

    // Handle friction
    else if (ground_collision == Contact) {
        /* Sliding speed = speed of the contact patch on the wheel relatively
	 to the ground */
      if (fabs(w_speed) > 0.05)
	sliding_speed = projectV(surface, speed) - (((w_speed*normal)^r)*s1)*s1;
      else sliding_speed = projectV(surface, speed);
      
      Vector surface_speed = projectV(surface, speed);
      
      // Friction
      double sliding_speed_size = sliding_speed.size();
      if (is_debug()) debug_stream<<"Sliding speed = "<<sliding_speed<<" ("<<sliding_speed_size<<")"<<std::endl;
      if (sliding_speed_size > SMALL_VECTOR) {
	// Reactive friction from the ground, simplified expression
	Vector reaction = (-sliding_speed/sliding_speed_size) * -vertical_force_size;
	
	// Adjust it according to stickyness coeffs and to the surface's grip
	Vector flong = ((reaction * s1) * s1) * long_k * surface.grip
	  *grip_curve.getY(fabs((sliding_speed*s1)/(surface_speed*s1)));
	
	//	double grip_factor = gripFactor(sliding_speed, surface_speed);
	Vector fside = ((reaction * normal) * normal) * side_k * surface.grip
	  /** grip_factor*/;
	
	// Actual reaction of ground on the wheel
	friction_force = flong + fside;
	
	if (is_debug()) debug_stream<<"friction = "<<friction_force<<" ("<<flong<<" + "<<fside<<")"<<std::endl;
	
	global_force += friction_force;
	torque += (-r ^ friction_force)*normal;
      }
    }
    
  }
  
  // Angular speed is no longer updated here. Kart takes care of that.
  //  w_speed += dt*torque/inertia;

  if (is_debug())
    printDebugEnd();
}

double Wheel::getFriction() const
{
  double side_slide = fabs(sliding_speed * normal);
  return surface.grip*surface.grip*side_slide/10.0;
}

double Wheel::getLoad() const
{
  if (ground_collision != None) {
    double res = -external_force * surface.normal;
    if (res < 0) return 0.0;
    return res;
  }
  else {
    return 0.0;
  }
}

double Wheel::gripFactor(const Vector& sliding_speed, const Vector& speed) const
{
  double alpha;  // tan value of slip angle

  if (sliding_speed.size() < SMALL_VECTOR) {
    alpha=0.0;
  }
  else {
    if ((sliding_speed^speed).size() < SMALL_VECTOR) {
      alpha=1e6;
    }
    else {
      double speed_size = speed.size();
      if (speed_size < SMALL_VECTOR) {
	alpha=1e6;
      }
      else {
	Vector speed_norm = speed/speed.size();
	double y = sliding_speed * speed_norm;
	double x = (sliding_speed - y*speed_norm).size();

	alpha = y/x;
      }
    }
  }

  return slip_angle_k_curve.getY(fabs(alpha));
}

void Wheel::setAngle(double alpha)
{
  angle = alpha;
}

double Wheel::getAngle() const
{
  return angle;
}

void Wheel::setDriveTorque(double m)
{
  drive_torque = m;
}

double Wheel::getDriveTorque() const
{
  return drive_torque;
}

double Wheel::getAngularSpeed() const
{
  return w_speed;
}

Vector Wheel::getFrictionForce() const
{
  return friction_force;
}

double Wheel::getBumpFactor() const
{
  if (ground_collision == Contact) return surface.bumps;
  else return 0.0;
}
