/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "MeshEditor.hh"
#include "helpers/OrientMatrix.hh"

using namespace top10::tracked;

MeshEditor::MeshEditor():
  scaling(1.0), axis(top10::math::Y), axis_negate(false), polys(0), gl_polys(0), camera(0)
{
}

void MeshEditor::loadMesh(std::string filename)
{
  top10::helpers::PolygonSet* new_polys;
  new_polys = top10::helpers::PolygonSet::loadNew(filename);
  model_filename=filename;

  if (gl_polys) { delete gl_polys; gl_polys=0; }
  if (polys) { delete polys; polys=0; }

  polys = new_polys;
  gl_polys = new top10::ui_interactive::TriangleSet(polys);
}

void MeshEditor::reset()
{
  //Do nothing
}

std::string MeshEditor::getPickedMesh() const
{
  using top10::math::Vector;
  assert(camera);
  if (polys == 0) return std::string();

  Vector dir = camera->getDirection();

  // Transform the ray from global to local coordinates
  top10::math::Matrix4 M4 = top10::helpers::makeInverse(axis, axis_negate, scaling);
  Vector new_pos = M4*camera->getCenter();
  Vector new_dir = M4*dir;
  double new_dir_size = new_dir.size();
  assert(new_dir_size > SMALL_VECTOR);
  new_dir /= new_dir_size;

  // Compute intersection with the mesh
  int idx = polys->intersectRay(new_pos, new_dir);
  if (idx <0) return std::string();
  else return polys->mesh_names[idx];
}

std::string MeshEditor::getPickedSurface() const
{
  using top10::math::Vector;
  assert(camera);
  if (polys == 0) return std::string();

  Vector dir = camera->getDirection();

  // Transform the ray from global to local coordinates
  top10::math::Matrix4 M4 = top10::helpers::makeInverse(axis, axis_negate, scaling);
  Vector new_pos = M4*camera->getCenter();
  Vector new_dir = M4*dir;
  double new_dir_size = new_dir.size();
  assert(new_dir_size > SMALL_VECTOR);
  new_dir /= new_dir_size;

  // Compute intersection with the mesh
  int idx = polys->intersectRayPolygon(new_pos, new_dir);
  if (idx <0) return std::string();
  else return polys->surf_names[polys->at(idx).surface_name];
}

top10::math::Vector MeshEditor::getPickedPoint() const
{
  using top10::math::Vector;
  Vector pt;
  Vector dir = camera->getDirection();
  Vector pos = camera->getCenter();
  if (fabs(dir.y) < 0.01) throw std::string("Aiming too far");
  double t = -pos.y/dir.y;
  if (t < 0) throw std::string("Aiming away from the ground");
  pt = pos+t*dir;
  return pt;
}

void MeshEditor::setTransformGL() const
{  
  top10::math::Matrix4 tmpM4 = top10::helpers::makeOrientation(axis,
							       axis_negate,
							       scaling);
  GLdouble m[16];
  tmpM4.toGL(m);
  glMultMatrixd(m);
}

void MeshEditor::drawGL_impl() const
{
  assert(camera);
  
  // Draw the track
  // Rotate the track so that its upward direction points to +Y
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  setTransformGL();

  glEnable(GL_LIGHTING);
  if (gl_polys)
      gl_polys->drawGL(/**camera*/);  //TODO: transform the view volume as well
  glDisable(GL_LIGHTING);

  // Draw the aiming mark
  glDisable(GL_DEPTH_TEST);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glColor3f(1, 1, 0);
  glBegin(GL_LINES);
  glVertex3f(-0.2, 0, -1.0);
  glVertex3f(0.2, 0, -1.0);
  glVertex3f(0, -0.2, -1.0);
  glVertex3f(0, 0.2, -1.0);
  glEnd();  
  glEnable(GL_DEPTH_TEST);

  glPopMatrix();
}
