/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "QGLView.hh"
#include "math/Vertex.hh"
#include "math/Matrix.hh"

#include <cassert>
#include <math.h>

using top10::math::Vector;

QGLView::QGLView(QWidget* parent, const char* name):
  QGLWidget(parent, name), old_x(-1), old_y(-1), camera(0)
{
  setMouseTracking(true);
}

void QGLView::initializeGL()
{
  // set a directional light
  GLfloat light_position[] = {5.0, 10.0, 5.0, 1.0};
  glLightfv(GL_LIGHT0, GL_POSITION, light_position);

  glClearColor(.4, .4, 1, 0);
  glMatrixMode(GL_MODELVIEW);
  //  glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
  glEnable(GL_LIGHT0);
  glEnable(GL_DEPTH_TEST);

  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
}

void QGLView::resizeGL(int w, int h)
{
  glViewport(0, 0, w, h);
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(90.0, ((GLdouble)w)/h, 0.2, 1000.0);
  if (camera) {
      camera->setFOV(90.0);
      camera->setNear(0.2);
      camera->setFar(1000.0);
      camera->setRatio((double)w/h);
      camera->update();
  }
}

void QGLView::paintGL()
{
  glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
  if (!camera) return;

  //! Setup the view
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  camera->setViewGL();

  // Draw the grid
  for (int c = -500; c <= 500; c+=10) {
    glColor3f(0.0, 0.0, 0.2);
    glBegin(GL_LINES);
    glVertex3i(-500, 0, c);
    glVertex3i(500, 0, c);
    glVertex3i(c, 0, -500);
    glVertex3i(c, 0, 500);
    glEnd();
  }

  // Draw all drawable objects
  GLfloat u=0.0;
  for (std::vector<top10::tracked::Drawable*>::const_iterator it_drawable = drawables.begin();
       it_drawable != drawables.end();
       ++it_drawable) {
    glPolygonOffset(0.0, u);
    glEnable(GL_POLYGON_OFFSET_FILL);
    glEnable(GL_POLYGON_OFFSET_LINE);
    (*it_drawable)->drawGL();
    u = u-1.0;
  }

  //! Restore the view
  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();
}

void QGLView::mouseMoveEvent(QMouseEvent* e)
{
  using top10::math::Vector;
  // Rotate the camera
  if (e->state() & RightButton) {
    if (old_x >= 0) {
      double f_x = 180.0/width();
      double f_y = 90.0/height();

      if (camera) {
	camera->rotateX((e->globalX() - old_x)*f_x);
	camera->rotateY((e->globalY() - old_y)*f_y);
	camera->update();
	updateGL();
      }
    }
  }
  // Move the camera (pan)
  else if (e->state() & LeftButton) {
    if (old_x >= 0) {
      double f_x = 50.0/width();
      double f_y = 50.0/height();
      if (camera) {
	Vector flat_dir = camera->getDirection();
	flat_dir.y = 0.0;
	double s = flat_dir.size();
	if (s > SMALL_VECTOR)
	  flat_dir /= s;
	else {
	  flat_dir = camera->getUp();
	  flat_dir.y = 0;
	  s = flat_dir.size();
	  assert(s > SMALL_VECTOR);
	  flat_dir /= s;
	}
	camera->translate((e->globalX()-old_x)*f_x*camera->getRight() - (e->globalY()-old_y)*f_y*flat_dir);
	camera->update();
	updateGL();
      }
    }
  }
  else if (e->state() & MidButton) {
    if (old_x >= 0) {
      double f_y = 50.0/height();
      if (camera) {
	camera->translate(Vector(0.0, (e->globalY() - old_y) * f_y, 0.0));
	camera->update();
	updateGL();
      }
    }
  }

  old_x = e->globalX();
  old_y = e->globalY();
}
