/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "Frustum.hh"
#include "math/intersections.hh"
#include <cassert>
#include <GL/gl.h>

using namespace top10::ui_interactive;
using top10::math::Vector;
using top10::math::Plane;
using top10::math::AxisAlignedBox;
using top10::math::Box;

Frustum::Frustum(): center(0,0,0), e0(1,0,0), e1(0,1,0), e2(0,0,1), ratio(1.0), fov(M_PI/3.0), dist_near(1.0), dist_far(1e3)
{
  //  update();
}

void Frustum::setDirection(Vector dir, Vector up)
{
  if (dir.size() < SMALL_VECTOR || (dir^up).size() < SMALL_VECTOR) return;

  e2 = -1.0*dir;
  double s=e2.size();
  e2/=s;

  e0 = up^e2;
  s = e0.size();
  e0/=s;

  e1 = e2^e0;
}

void Frustum::getGLMatrix(double out[4][4]) const
{
  Vector e[3] = {e0, e1, e2};
  top10::math::Matrix3 M3(e);
  top10::math::Matrix4 M4(M3);
  M4 = M4.transpose();
  top10::math::Translation4 T4(-center);
  M4 = M4*T4;
  M4.toGL(out);

  //  std::cerr<<"e:"<<e0<<std::endl<<e1<<std::endl<<e2<<std::endl;
  //  std::cerr<<"M4: "<<M4<<std::endl;
}

void Frustum::update()
{
  // Compute the vertices of the view volume
  double h = fabs(tan(fov)/2.0);
  double w = h*ratio;
  nul = center+dist_near*(-e0*w+e1*h-e2);
  nur = center+dist_near*(e0*w+e1*h-e2);
  nll = center+dist_near*(-e0*w-e1*h-e2);
  nlr = center+dist_near*(e0*w-e1*h-e2);
  ful = center+dist_far*(-e0*w+e1*h-e2);
  fur = center+dist_far*(e0*w+e1*h-e2);
  fll = center+dist_far*(-e0*w-e1*h-e2);
  flr = center+dist_far*(e0*w-e1*h-e2);

  left = Plane(fll, ful, center);
  right = Plane(fur, flr, center);
  upper = Plane(ful, fur, center);
  lower = Plane(flr, fll, center);
  far_plane = Plane(center-e2*dist_far, e2);
  near_plane = Plane(center-e2*dist_near, -e2);

  using top10::helpers::Polygon;

  // transform this frustum to polygons
  polygons.clear();
  Polygon poly;

  // Left
  poly.p.push_back(nul);
  poly.p.push_back(nll);
  poly.p.push_back(fll);
  poly.p.push_back(ful);
  polygons.push_back(poly);

  // Right
  poly.p.clear();
  poly.p.push_back(nur);
  poly.p.push_back(nlr);
  poly.p.push_back(flr);
  poly.p.push_back(fur);
  polygons.push_back(poly);

  // Upper
  poly.p.clear();
  poly.p.push_back(nur);
  poly.p.push_back(nul);
  poly.p.push_back(ful);
  poly.p.push_back(fur);
  polygons.push_back(poly);

  // Lower
  poly.p.clear();
  poly.p.push_back(nlr);
  poly.p.push_back(nll);
  poly.p.push_back(fll);
  poly.p.push_back(flr);
  polygons.push_back(poly);

  // Far
  poly.p.clear();
  poly.p.push_back(flr);
  poly.p.push_back(fur);
  poly.p.push_back(ful);
  poly.p.push_back(fll);
  polygons.push_back(poly);

  // Near
  poly.p.clear();
  poly.p.push_back(nlr);
  poly.p.push_back(nur);
  poly.p.push_back(nul);
  poly.p.push_back(nll);
  polygons.push_back(poly);
}

top10::helpers::OverlapType Frustum::overlap(const AxisAlignedBox& box) const
{
  if (top10::helpers::intersect(polygons, box)) return top10::helpers::Intersect;

  // No intersection with the polygons: check if the box is inside the frustum
  Vector vertices[8];
  box.getVertices(vertices);
  for (int i=0; i<8; ++i) {
    if (!left.isBelow(vertices[i]) || !right.isBelow(vertices[i]) ||
	!upper.isBelow(vertices[i]) || !lower.isBelow(vertices[i]) ||
	!far_plane.isBelow(vertices[i]) || !near_plane.isBelow(vertices[i])) return top10::helpers::None;
  }
  
  return top10::helpers::Contains;
}

top10::helpers::OverlapType Frustum::overlap(const Box& box) const
{
  if (top10::helpers::intersect(polygons, box)) return top10::helpers::Intersect;

  // No intersection with the polygons: check if the box is inside the frustum
  Vector vertices[8];
  box.getVertices(vertices);
  for (int i=0; i<8; ++i) {
    if (!left.isBelow(vertices[i]) || !right.isBelow(vertices[i]) ||
	!upper.isBelow(vertices[i]) || !lower.isBelow(vertices[i]) ||
	!far_plane.isBelow(vertices[i]) || !near_plane.isBelow(vertices[i])) return top10::helpers::None;
  }
  
  return top10::helpers::Contains;
}

static inline void glVertex(Vector v)
{
  glVertex3f(v.x, v.y, v.z);
}

void Frustum::drawGL() const
{
  glColor3f(1.0, 0.0, 0.0);

  glBegin(GL_LINE_STRIP);
  glVertex(fll);
  glVertex(ful);
  glVertex(fur);
  glVertex(flr);
  glEnd();

  glBegin(GL_LINE_STRIP);
  glVertex(nll);
  glVertex(nul);
  glVertex(nur);
  glVertex(nlr);
  glEnd();

  glBegin(GL_LINES);
  glVertex(ful);
  glVertex(nul);
  glVertex(fll);
  glVertex(nll);
  glVertex(flr);
  glVertex(nlr);
  glVertex(fur);
  glVertex(nur);
  glEnd();
}


void Frustum::setViewGL() const
{
  GLdouble m[16];
  getGLMatrix(m);
  glLoadMatrixd(m);
}
