/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "MainMenu.hh"
#include "DebugMenu.hh"
#include "ControlsMenu.hh"
#include "util/UserFile.hh"
#include "util/Parse.hh"
#include "util/PathFinder.hh"
#include <fstream>
#include <time.h>
#include <sstream>

using namespace top10::ui_interactive;

MainMenu::MainMenu(UIPanel* p, bool* q, ActionMapping* actions,
		   Simulation** sim_ref, AudioDevice* audio):
  UIPanel::Menu(p), track_notifier(this), replay_notifier(this), wet_notifier(sim_ref), quit_notifier(q),
  kart_notifier(this), track(0),
  record_in(0), record_out(0),
  simulation_ref(sim_ref), audio_device(audio),
  kart_gl("data/karts/default_model")
{
  top10::util::UserFile::makePath("records");

  entries.push_back("Training session");
  links.push_back(new FileMenu(p, &track_notifier, "data/tracks", ".trk"));
  entries.push_back("Replay recorded session");
  links.push_back(new FileMenu(p, &replay_notifier, "records", ".rec"));
  entries.push_back("Load Kart (next session)");
  links.push_back(new FileMenu(p, &kart_notifier, "data/karts", ".mdl"));
  entries.push_back("Debug");
  links.push_back(new DebugMenu(p));
  entries.push_back("Controls");
  links.push_back(new ControlsMenu(p, actions));
  entries.push_back("Wet weather");
  links.push_back(new YesNoMenu(p, &wet_notifier));  
  entries.push_back("Quit");
  links.push_back(new YesNoMenu(p, &quit_notifier));
}

MainMenu::~MainMenu()
{
  while (!links.empty()) {
    if (links.back()) {
      delete links.back();
    }
    links.pop_back();
  }
}

void MainMenu::play(std::string track_file)
{
  std::string empty;
  playOrReplay(track_file, empty);
}

void MainMenu::replay(std::string replay_file)
{
  std::string empty;
  playOrReplay(empty, replay_file);
}

void MainMenu::playOrReplay(std::string track_file, std::string rec_in_file)
{
  assert(track_file.empty() || rec_in_file.empty());

  SDL_PauseAudio(1);
  SDL_LockAudio();

  // Record file
  if (record_out) {
    delete record_out;
    record_out = 0;
  }
  if (record_in) {
    delete record_in;
    record_in = 0;
  }

  top10::track::Track* new_track(0);

  // rec_in_file empty -> it's a new training session
  if (rec_in_file.empty()) {
    std::string record_filename;
#if defined(HAVE_TIME) && defined(HAVE_CTIME)
    time_t t = time(0);
    std::ostringstream buf;
    buf<<t<<".rec";
    record_filename = buf.str();
#endif
    
    if (!record_filename.empty()) {
      record_out = new std::ofstream((top10::util::UserFile::getPath()+"/records/"+record_filename).c_str());
      if (*record_out) (*record_out)<<"\""<<track_file<<"\""<<std::endl;
      else {
	std::cerr<<"Could not open record out file: "<<
	  (top10::util::UserFile::getPath()+"/records/"+record_filename)<<std::endl;
	delete record_out;
	record_out = 0;
      }
    }

    new_track = top10::track::loadTrack(track_file);
  }
  else {
    // rec_in_file not empty -> it's a replay session
    std::string filename = "records/"+rec_in_file;
    std::string path = top10::util::PathFinder::find(filename);
    if (path.empty()) throw std::string("Could not find replay file ") + filename;
    record_in = new std::ifstream(path.c_str());
    if (*record_in) {
      track_file = top10::util::parseString(*record_in);
      new_track = top10::track::loadTrack(track_file);
    }
    else {
      std::cerr<<"Could not open record in file: "<<track_file<<std::endl;
      delete record_in;
      record_in = 0;
    }
  }
  if (new_track) {
    if (*simulation_ref) delete *simulation_ref;
    *simulation_ref = new Simulation(new_track, record_in, record_out, kart_gl);
    (*simulation_ref)->resizeEvent(panel->getWindowWidth(), panel->getWindowHeight());
    if (audio_device)
      audio_device->setKart((*simulation_ref)->karts[0]);
    if (track) delete track;
    track = new_track;
  }

  SDL_UnlockAudio();
  SDL_PauseAudio(0);
}

void MainMenu::loadKartGL(std::string filename)
{
  try {
    KartGL new_kart_gl(filename);
    kart_gl = new_kart_gl;
  }
  catch(std::string err) {
    std::cerr<<"Warning: could not load kart description: "<<err<<std::endl;
  }
}

MainMenu::TrackFileNotifier::TrackFileNotifier(MainMenu* m):
  menu(m)
{
}

void MainMenu::TrackFileNotifier::fileSelected(std::string file)
{
  menu->play(file);
}

MainMenu::ReplayFileNotifier::ReplayFileNotifier(MainMenu* m):
  menu(m)
{
}

void MainMenu::ReplayFileNotifier::fileSelected(std::string file)
{
  menu->replay(file);
}

MainMenu::KartNotifier::KartNotifier(MainMenu* m):
  menu(m)
{
}

void MainMenu::KartNotifier::fileSelected(std::string file)
{
  menu->loadKartGL(file);
}
