/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_SIMULATION_H
#define TOP10_SIMULATION_H

#ifdef WIN32
#include <windows.h>
#include <wingdi.h>
#endif

//C/C++ Stuff
#include <math.h>
#include <vector>
#include <iostream>
#include <fstream>
#include <queue>

//OpenGL
#include <GL/gl.h>
#include <GL/glu.h>

//SDL + PLIB
#include <SDL.h>
#include <plib/fnt.h>

//My stuff
#include "math/Curve.hh"
#include "math/Vertex.hh"
#include "physX/World.hh"
#include "physX/Kart.hh"
#include "util/error.hh"
#include "util/Debug.hh"
#include "racing/KartTimer.hh"
#include "racing/CheckPoint.hh"
#include "racing/StartingArea.hh"
#include "racing/LapRecord.hh"
#include "track/Track.hh"

#include "Controls.hh"
#include "Kart.hh"
#include "Event.hh"

namespace top10 {
  namespace ui_interactive {

    class Simulation: public top10::util::Debug
    {
    public:
      DECL_ERROR(BadTrack);

    public:
      Simulation(top10::track::Track* track, std::istream* rep_in,
		 std::ostream* rep_out, const KartGL& kgl);

      /**
	 Issues the GL commands to draw the world
      */
      void paintGL();

      /**
	 Advances the simulation
	 \parameter Time (in ms) since the start of the simulation
       */
      void update(unsigned int t);

      //! Tell this simulation about the viewport
      void resizeEvent(int w, int h);

      /**
	 An event is issued by an user to interact with the world
       */
      void sendEvent(Action, Sint16 value);

      /**
	 Tells to use the next camera to draw the scene.
      */
      void nextCamera();

      /**
	 Tells to use the previous camera to draw the scenne.
      */
      void previousCamera();

      /**
	 Enable/disable debugging
      */
      inline bool toggleDebug()
      {
	return track->toggleDebug();
      }


      inline void setWet(bool b) {
	isWet = b;
	if (!track) return;
	if (b) track->setGripFactor(0.85);
	else track->setGripFactor(1.0);
      }

      inline bool getWet() const { return isWet; }

      ~Simulation();

      /**
	 The set of all karts
      */
      std::vector<top10::ui_interactive::Kart*> karts;

      void reinit();

    private:
      void initItems();

      void setFont(std::string font);

      /**
	 Dumps the current state of the world to the output stream.
	 Does nothing if no output stream was specified.
      */
      void dumpState();

      void setCam();

    private:
      int screen_w;
      int screen_h;

      //! Description of the track
      top10::track::Track* track;

      //! Is the track wet?
      bool isWet;

      //! The object issuing openGL instructions to draw karts
      KartGL kart_gl;

      /**
	 Points to an objet that registers every object in the world
      */
      top10::physX::World* world;

      //! Camera currently used to view the world
      Frustum cam;

      /**
	 Timers for each kart
      */
      std::vector<top10::racing::KartTimer> timers;

      //! Best lap
      top10::racing::LapRecord best_lap_record;

      //! Current laps for each kart
      std::vector<top10::racing::LapRecord> current_lap_records;

      //! Iterator on the state of the ghost kart
      /*! The ghost kart is an image of the kart during the fastest lap */
      top10::racing::LapRecord::const_iterator ghost_it;
      //! Iterator on the next state of the ghost kart
      top10::racing::LapRecord::const_iterator next_ghost_it;

      /**
	 Check points of the track
      */
      const top10::racing::CheckPoints* checkpoints;

      /**
	 The area where karts are originaly placed
      */
      const top10::racing::StartingArea* grid;

      /**
	 The steering angle of the controled kart
      */
      double steer_angle;
      //! The throttle % of the controled kart
      double throttle;
      //! The braking % of the controled kart
      double braking;

      /// Stream events are read from. Can be null (no playback)
      std::istream* rep_in;
      /// Strean events and states are written to. Can be null (no recording)
      std::ostream* rec_out;

      /// Time inside the world
      Uint32 current_t;
      /// Time when the world was created
      Uint32 initial_t;

      /// Number of current camera
      unsigned int cam_num;

      /// Used for playback
      std::queue<Event> events_to_replay;

      /// The shape of the speed gauge
      top10::math::Curve2D speed_gauge;
      /// The color of the bar when speed is 0
      const top10::math::Vector low_speed_col;
      /// The color of the bar when the speed is high
      const top10::math::Vector hi_speed_col;

      /// The drawable mesh of the track
      TriangleSet track_mesh;

      /// GL font stuff
      fntRenderer texout;
      fntTexFont font;
    };
  };
};
#endif
