/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "UIPanel.hh"
#include "util/PathFinder.hh"

#include <GL/gl.h>
#include <iostream>

using namespace top10;
using namespace ui_interactive;

/*
 * Class: UIPanel
 */

UIPanel::UIPanel(): currentMenu(0), frontMenu(0)
{
  setFont("fonts/default.txf");
}

void UIPanel::setFont(std::string font_name)
{
  std::string font_path = top10::util::PathFinder::find(font_name);
  if (font_path.empty()) throw std::string("Could not load font") + font_name;

  font.load(font_path.c_str());
  texout.setFont(&font);
  texout.setPointSize(1);
}

void UIPanel::event(SDL_Event event)
{
  if (event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_ESCAPE) {
    if (currentMenu) currentMenu = 0;
    else currentMenu = frontMenu;
  }
  else {
    if (currentMenu) currentMenu->event(event);
  }
  // Enable UNICODE translation while the menu is shown
  if (currentMenu) SDL_EnableUNICODE(1);
  else SDL_EnableUNICODE(-1);
}

void UIPanel::updateGL()
{
  if (currentMenu == 0) return;
  currentMenu->updateGL(texout, font);
}

/*
 * Class: UIPanel::Menu
 */

UIPanel::Menu::Menu(UIPanel* _panel):
  panel(_panel),
  entry_n(0)
{
}

void UIPanel::Menu::updateGL(fntRenderer& texout, const fntTexFont& font)
{
  // Mark the current entry
  float y = -(getEntryN()*(1.0 + font.getGap())-0.5) * texout.getPointSize();

  // Draw the marker to the left of the current entry
  glDisable(GL_TEXTURE_2D);
  glBegin(GL_TRIANGLES);
  glVertex2f(-1.2f, y-0.4f);
  glVertex2f(-1.2f, y+0.4f);
  glVertex2f(-0.2f, y);
  glEnd();

  texout.begin();
  y = 0;
  for (std::vector<std::string>::const_iterator p = entries.begin();
       p != entries.end();
       ++p) {
    texout.start2f(0, y);
    y -= (1.0 + font.getGap())*texout.getPointSize();
    texout.puts(p->c_str());
  }
  texout.end();
}

void UIPanel::Menu::selected()
{
  assert(entry_n >= 0 && entry_n < (int)entries.size() && entry_n < (int)links.size());
  panel->setMenu(links[entry_n]);
}

void UIPanel::Menu::event(SDL_Event event)
{
  if (event.type == SDL_KEYDOWN) {
    switch (event.key.keysym.sym) {
    case SDLK_SPACE:
    case SDLK_RETURN:
      selected();
      break;

    case SDLK_UP:
      prevEntry();
      break;

    case SDLK_DOWN:
      nextEntry();
      break;

    default: break;
    }
  }
}

int UIPanel::Menu::nextEntry()
{
  if (entries.empty()) {
    assert(entry_n==0);
    return entry_n;
  }

  if (entry_n == ((int)entries.size())-1) entry_n = 0;
  else entry_n++;

  return entry_n;
}

int UIPanel::Menu::prevEntry()
{
  if (entries.empty()) {
    assert(entry_n==0);
    return entry_n;
  }

  if (entry_n == 0) entry_n = entries.size()-1;
  else entry_n--;

  return entry_n;
}
