/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

//C++ stuff
#include <iostream>
#include <fstream>

//OpenGL
#include <GL/gl.h>
#include <GL/glu.h>

//SDL Stuff
#include <SDL/SDL.h>

//My stuff
#include "UIPanel.hh"
#include "ControlsMenu.hh"
#include "FileMenu.hh"
#include "YesNoMenu.hh"
#include "util/PathFinder.hh"
#include "util/UserFile.hh"

using namespace top10::ui_interactive;
using namespace std;

/*
  SDL video + openGL init
*/
SDL_Surface* initGL(int depth, int w, int h, bool full_screen)
{
  SDL_Surface* screen;
  int bits_per_channel(5);

  /* Video init */
  if (depth == 0) {
    // Try 24 bpp
    if (SDL_VideoModeOK( w, h, 24, SDL_OPENGLBLIT | (full_screen?SDL_FULLSCREEN:0))) {
      depth = 24;
      bits_per_channel = 8;
    }
    // Try 16 bpp
    else if (SDL_VideoModeOK( w, h, 16, SDL_OPENGLBLIT | (full_screen?SDL_FULLSCREEN:0))) {
      depth = 16;
      bits_per_channel = 5;
    }
    else depth = -1;
  }

  if (depth != -1) {
    cout<<"depth = "<<depth<<" bpc = "<<bits_per_channel<<endl;
    SDL_GL_SetAttribute( SDL_GL_RED_SIZE, bits_per_channel );
    SDL_GL_SetAttribute( SDL_GL_GREEN_SIZE, bits_per_channel );
    SDL_GL_SetAttribute( SDL_GL_BLUE_SIZE, bits_per_channel );
    SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, depth );
    SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );      
    screen = SDL_SetVideoMode( w, h, depth, SDL_OPENGLBLIT | (full_screen?SDL_FULLSCREEN:0));
    if (screen == NULL) {
      throw string("Could not set GL mode: ") + string(SDL_GetError());
    }
  }
  else {
    throw string("No true color video mode available.");
  }

  return screen;
}

class ChangeMenuFont: public FileMenu::Notifier {
public:
  ChangeMenuFont(UIPanel* p): panel(p) {};

  void fileSelected(std::string font) { panel->setFont(std::string("fonts/")+font); }
private:
  UIPanel* panel;
};

class MainMenu: public UIPanel::Menu {
public:
  MainMenu(UIPanel* p, bool* quit, float* r, float* g, float* b, ActionMapping* d):
    UIPanel::Menu(p), red(r), green(g), blue(b), digs(d), quit_notifier(quit) {
    entries.push_back("Red");
    links.push_back(0);
    entries.push_back("Green");
    links.push_back(0);
    entries.push_back("Blue");
    links.push_back(0);
    entries.push_back("Reset");
    links.push_back(0);
    entries.push_back("Controls");
    links.push_back(new ControlsMenu(p, digs));
    entries.push_back("Fonts");
    links.push_back(new FileMenu(p, new ChangeMenuFont(p), "fonts", "txf"));
    entries.push_back("Quit");
    links.push_back(new YesNoMenu(p, &quit_notifier));
  }

  void selected() {
    switch (getEntryN()) {
    case 0:
      (*red) += 0.2;
      break;
    case 1:
      (*green) += 0.2;
      break;
    case 2:
      (*blue) += 0.2;
      break;
    case 3:
      (*red) = (*green) = (*blue) = 0.0;
      break;
    default:
      UIPanel::Menu::selected();
      break;
    }
    if (*red > 1) *red = 0.0;
    if (*green > 1) *green = 0.0;
    if (*blue > 1) *blue = 0.0;
  }
private:
  float *red, *green, *blue;
  ActionMapping* digs;
  top10::ui_interactive::YesNoMenu::BoolNotifier quit_notifier;
};

int main(int argc, char** argv)
{
  try {
    SDL_Surface* screen;
    int depth = 16;
    int w = 640;
    int h = 480;
    bool full_screen = false;

    top10::util::UserFile::makePath();
    top10::util::PathFinder::addPath(top10::util::UserFile::getPath());
    top10::util::PathFinder::addPath("data");
    top10::util::PathFinder::addPath("data/defaults");

    /* Initialize the SDL library */
    if ( SDL_Init(SDL_INIT_NOPARACHUTE | SDL_INIT_TIMER | SDL_INIT_VIDEO | SDL_INIT_JOYSTICK | SDL_INIT_AUDIO) < 0 ) {
      throw string("Couldn't initialize SDL: ") + string(SDL_GetError());
    }
    int n_sticks = SDL_NumJoysticks();
    std::vector<SDL_Joystick*> joysticks;
    for (int i=0; i<n_sticks; ++i) {

      SDL_Joystick* joy = SDL_JoystickOpen(i);
      joysticks.push_back(joy);
      if (joy) {
	std::cout<<"Opened joystick "<<i<<std::endl;

	std::cout<<"name: "<<SDL_JoystickName(i)<<" ";
	std::cout<<"Axes: "<<SDL_JoystickNumAxes(joy)<<" ";
	std::cout<<"Buttons: "<<SDL_JoystickNumButtons(joy)<<" ";
	std::cout<<"Balls: "<<SDL_JoystickNumBalls(joy)<<" ";
      }
      else std::cout<<"Failed top open joystick "<<i<<std::endl;
    }

    // Initialize GL stuff
    screen = initGL(depth, w, h, full_screen);
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

    glViewport(0, 0, w, h);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glScalef(0.05, 0.1, 1);
    glTranslatef(-10, 0, 0);

    // Init font lib
    fntInit();

    bool quit;
    float red(0.0), green(0.0), blue(0.0);
    UIPanel panel;
    ActionMapping digs;
    if (!top10::util::PathFinder::find("inputs").empty())
      digs = loadActionMapping("inputs");

    UIPanel::Menu* menu = new MainMenu(&panel, &quit, &red, &green, &blue, &digs);
    panel.setFrontMenu(menu);

    /* Main loop */
    SDL_Event event;
    for (quit=false; !quit;) {
      while (SDL_PollEvent(&event)) {
	panel.event(event);
      }

      glClearColor(red, green, blue, 0);      
      glClear(GL_COLOR_BUFFER_BIT);
      glColor4f(1.0f, 1.0f, 1.0f, 0.5f);
      glDisable(GL_TEXTURE_2D);
      glBegin(GL_TRIANGLES);
      glVertex2f(-20, -10);
      glVertex2f(20, -10);
      glVertex2f(-20, 10);
      glEnd();
      glColor4f(1.0f, 1.0f, 0.0f, 0.9f);
      panel.updateGL();

      glFinish();
      SDL_GL_SwapBuffers();
    }
    saveActionMapping(top10::util::UserFile::getPath()+"/inputs", digs);
  }
  catch (std::string& e) {
    cerr<<e<<endl;
  }
  catch (...) {
    cerr<<"Unknown error caught"<<endl;
  }

  return 0;
}
