/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifdef WIN32
#include <windows.h>
#endif

#include <GL/gl.h>
#include "MeshNode.hh"

using namespace top10::graphX;

MeshNode::MeshNode(const top10::math::Mesh* _mesh,
		   std::vector<top10::math::Vector> _normals,
		   std::vector<int> _normal_idxs,
		   std::vector<TextureCoord> _tex_coords,
                   std::string name):
  mesh(_mesh),
  normals(_normals),
  tex_coords(_tex_coords),
  normal_idxs(_normal_idxs)
{
  setName(name);
}

MeshNode::MeshNode(const top10::math::Mesh* _mesh,
		   std::vector<TextureCoord> _tex_coords,
                   std::string name):
  mesh(_mesh),
  tex_coords(_tex_coords)
{
  using top10::math::Vector;
  using top10::math::Mesh;

  setName(name);
  
  const std::vector<Vector>* vertices = mesh->getVertices();
  const std::vector<Mesh::Face>* faces = mesh->getFaces();
  
  // One normal for each vertex
  normals.resize(vertices->size());
  // Three for each triangle
  normal_idxs.resize(faces->size() *3);

  // Sum the normal at each vertex over all faces sharing this vertex
  std::vector<int>::iterator idx_it = normal_idxs.begin();
  for (std::vector<Mesh::Face>::const_iterator face = faces->begin();
       face != faces->end();
       ++face) {
    int i0 = face->idxs[0], i1 = face->idxs[1], i2 = face->idxs[2];
    Vector N = (vertices->at(i1) - vertices->at(i0)) ^ (vertices->at(i2) - vertices->at(i1));
    double Ns = N.size();
    if (Ns > SMALL_VECTOR) {
      N/=Ns;
      normals.at(i0) += N;
      normals.at(i1) += N;
      normals.at(i2) += N;
    }
    *idx_it = i0; ++idx_it;
    *idx_it = i1; ++idx_it;
    *idx_it = i2; ++idx_it;
  }

  // Normalize to get the average
  for (std::vector<Vector>::iterator normal = normals.begin();
       normal != normals.end();
       ++normal) {
    double Ns = normal->size();
    if (Ns > SMALL_VECTOR)
      *normal /= Ns;
    else
      *normal = Vector(0,1,0);
  }
}
  
void MeshNode::renderGL(const RenderingFeatures& unused, const RenderState&, const CameraNode&) const
{
  using top10::math::Mesh;
  using top10::math::Vector;

  const std::vector<Mesh::Face>* faces = mesh->getFaces();
  const std::vector<Vector>* vertices = mesh->getVertices();

  std::vector<int>::const_iterator normal_idx_it = normal_idxs.begin();
  std::vector<TextureCoord>::const_iterator texel_it = tex_coords.begin();
  glBegin(GL_TRIANGLES);
  for (std::vector<Mesh::Face>::const_iterator face_it = faces->begin();
       face_it != faces->end();
       ++face_it) {
        for (int i=0; i<3; ++i, ++normal_idx_it, ++texel_it) {
          top10::math::Vector normal = normals.at(*normal_idx_it);
          assert(normal.size2() > 0.9 && normal.size2() < 1.1);
          glNormal3d(normal.x, normal.y, normal.z);
          glTexCoord2d(texel_it->s, texel_it->t);
          top10::math::Vector v = vertices->at(face_it->idxs[i]);
          glVertex3d(v.x, v.y, v.z);
        }
  }
  glEnd();

}
