/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_GRAPHX_RENDERSTATE_HH
#define TOP10_GRAPHX_RENDERSTATE_HH

#include "math/Vertex.hh"
#include "math/Matrix.hh"
#include "Light.hh"
#include "RenderingFeatures.hh"

#include <vector>

namespace top10 {
  namespace graphX {

    /*! RenderState is used to keep track of the current rendering state to avoid costly state changes. */
    class RenderState {
    public:
      typedef std::vector<Light> Lights;
      enum Filtering { None=0, Linear, Bilinear, Trilinear, Anisotropic };
      enum Culling { NoCulling=0, Front=1, Back=2 };
      enum PolyFillMode { Full=0, Lines=1 };
      enum FixedBits { MaterialBit=1, TextureBit=2, CullingBit=4, LineBit=8, PolyBit=16, LightsBit=32 };
      typedef signed long FixedBitsField;
      
    public:
      RenderState();

      //! Check consistency between this render state and the actual OpenGL state
      void checkConsistency() const;
      
      inline void setTransform(const top10::math::Matrix4& M) { to_world = M; }
      inline const top10::math::Matrix4& getTransform() const { return to_world; }

      void addLight(Light);
      inline const Lights* getLights() const { return &lights; }
      inline void disableLights() { if ((fixed_flags & LightsBit) == 0) lights.clear(); }
      
      // Accessors
      inline void setAlpha(unsigned char c) {
        if ((fixed_flags & MaterialBit) == 0) alpha = c;
      }
      inline void setColor(unsigned char _r, unsigned char _g, unsigned char _b) {
        if ((fixed_flags & MaterialBit) == 0) { r=_r; g=_g; b=_b; }
      }
      inline void setGlow(unsigned char _r, unsigned char _g, unsigned char _b) {
        if ((fixed_flags & MaterialBit) == 0) { glow_r = _r; glow_b = _b; glow_g = _g; }
      }
      inline void setTextureId(unsigned int id) {
        if ((fixed_flags & TextureBit) == 0) { texture_id = id; }
      }
      inline void setFiltering(Filtering f) {
        if ((fixed_flags & TextureBit) == 0) { filtering = f; }
      }
      inline void setOffset(float f) {
        offset = f;
      }
      inline void setCulling(Culling c) {
        if ((fixed_flags & CullingBit) == 0) { culling = c; }
      }
      inline void setFront(bool b) {
        if ((fixed_flags & CullingBit) == 0) { front_is_direct = b; }
      }
      inline void setLineWidth(float f) {
        if ((fixed_flags & LineBit) == 0) { line_width = f; }
      }
      inline void setPolyFill(PolyFillMode m) {
        if ((fixed_flags & PolyBit) == 0) { poly_fill_mode = m; }
      }
      
      inline Culling getCulling() const { return culling; }
      
      inline void disableChanges(FixedBits bit) { fixed_flags |= bit; }
      inline void disableChanges() { fixed_flags = (signed long)-1; }
      
      inline void enableChanges(FixedBits bit) { fixed_flags &= ~bit; }
      inline void enableChanges() { fixed_flags = 0; }
      
      inline void setChanges(FixedBitsField f) { fixed_flags = f; }
      
      //! Comparison operator, useful to sort RenderList according to the state
      bool operator<(const RenderState&) const;

      //! Update the current OpenGL state
      /*! \param old_state old OpenGL state, used to make as few state changes as possible. */
      void setStateGL(const RenderState& old_state, const RenderingFeatures&, bool force_init = false) const;
      
    private:
      //! Specifies which attributes are not allowed to be set
      FixedBitsField fixed_flags;
      //! Opacity
      unsigned char alpha;
      //! ambient and diffuse color
      unsigned char r,g,b;
      //! Emission (glow)
      unsigned char glow_r, glow_g, glow_b;
      //! Texture id.
      unsigned int texture_id;
      //! Texture filtering
      Filtering filtering;
      //! Depth offset
      float offset;
      //! Which faces should be drawn
      Culling culling;
      //! What is the front face
      bool front_is_direct;
      //! Line width
      float line_width;
      //! If polygons should be filled, or drawn in wireframes
      PolyFillMode poly_fill_mode;
      
    private:
      top10::math::Matrix4 to_world;
      Lights lights;
    };
  };
};

#endif
