/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/
#include "Renderer.hh"
#include "Node.hh"
#include "CameraNode.hh"
#include <GL/glu.h>

namespace top10 {
namespace graphX {

Renderer::Renderer(): camera(0)
{
}

void Renderer::clearList()
{
  rl.clear();    
}

void Renderer::buildList(const Node* node)
{
  RenderState rs;
  node->buildRenderList(rs, &rl);
}

void Renderer::initGL(int w, int h)
{
  width = w;
  height = h;
  
  glClearColor(0.0, 0.0, 0.0, 1.0);
  glEnable(GL_DEPTH_TEST);
  glDepthFunc(GL_LESS);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_ALPHA_TEST);
  glAlphaFunc(GL_EQUAL, 1.0);
  glDisable(GL_LIGHTING);
  glColor4f(1.0, 1.0, 1.0, 1.0);
  glViewport(0, 0, width, height);
  
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
}

void Renderer::clearAllGL()
{
  glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT|GL_STENCIL_BUFFER_BIT);  
}

void Renderer::clearDepthGL()
{
  glClear(GL_DEPTH_BUFFER_BIT);  
}

void Renderer::clearColorGL()
{
  glClear(GL_COLOR_BUFFER_BIT);
}

void Renderer::clearStencilGL()
{
  glClear(GL_STENCIL_BUFFER_BIT);  
}

void Renderer::renderCommon(RenderMethod method)
{
  assert(camera.getPtr());
  assert(glGetError() == GL_NO_ERROR);
  
  // Set the initial state
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  if (method != Hud) {
    gluPerspective(camera->getView().getFOV(), camera->getView().getRatio(), camera->getView().getNear(), camera->getView().getFar());
  }
  else {
    gluOrtho2D(-1.0, 1.0, -1.0, 1.0);   
  }
  
  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();
  double M[16];
  switch (method) {
  case Skybox: camera->getView().getOrientGLMatrix(M); break;
  case Scene:  camera->getView().getGLMatrix(M); break;
  case Hud:    top10::math::Identity4().toGL(M); break;
  }
  glLoadMatrixd(M);
  glPushMatrix();  

  // Set the initial state
  RenderState initial, prev_state;
  initial.setStateGL(initial, features, true /* force_init */);
  prev_state = initial;
    
  for (RenderList::const_iterator it = rl.begin(); it != rl.end(); ++it) {
    // Set the state
    it->first.setStateGL(prev_state, features);
    
    // Render
    it->second->renderGL(features, it->first, *camera);
    
    // Update the previous state
    prev_state = it->first;
    
#ifndef NDEBUG
    prev_state.checkConsistency();
#endif
  }
  
  initial.setStateGL(initial, features, true);
}
    
void Renderer::setCamera(CameraNode*c)
{
  camera = c;
}

CameraNode* Renderer::getCamera() const
{
  return camera.getPtr();
}

void Renderer::enableStencil(GLenum rel, GLint value)
{
  glEnable(GL_STENCIL_TEST);
  glStencilFunc(rel, value, 0xff);    
}

void Renderer::setStencilOp(GLenum fail, GLenum zfail, GLenum zpass)
{
  glStencilOp(fail, zfail, zpass); 
}

void Renderer::disableStencil()
{
  glDisable(GL_STENCIL_TEST); 
}

void Renderer::toggleColorRendering(bool b)
{
  if (b) glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
  else glColorMask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE); 
}

void Renderer::toggleDepthRendering(bool b)
{
  if (b) glDepthMask(GL_TRUE);  
  else glDepthMask(GL_FALSE);
}

void Renderer::toggleDepthTest(bool b)
{
  if (b) glEnable(GL_DEPTH_TEST);
  else glDisable(GL_DEPTH_TEST);  
}

};
};
