/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "GenericOctree.hh"
#include "math/Box.hh"
#include "math/Vertex.hh"
#include <cassert>

template<typename ShapeT>
top10::helpers::GenericOctree<ShapeT>::GenericOctree():
  POLY_THRESHOLD(5),
  MAX_DEPTH(20)
{
  for (int i=0; i<8; ++i) {
    top_node.children[i] = 0;
  }
}

template<typename ShapeT>
top10::helpers::GenericOctree<ShapeT>::GenericOctree(const GenericOctree<ShapeT>& other):
  POLY_THRESHOLD(other.POLY_THRESHOLD),
  MAX_DEPTH(other.MAX_DEPTH)
{
  for (int i=0; i<8; ++i) {
    top_node.children[i] = duplicateNode(other.top_node.children[i]);
    top_node.box = other.top_node.box;
    top_node.shape_refs = other.top_node.shape_refs;
  }
  shapes = other.shapes;
}

template<typename ShapeT>
top10::helpers::GenericOctree<ShapeT>::GenericOctree(const ShapeVec& _shapes, int poly_threshold, int max_depth):
  POLY_THRESHOLD(poly_threshold),
  MAX_DEPTH(max_depth)
{
  shapes = _shapes;
  build();
}

template<typename ShapeT>
void top10::helpers::GenericOctree<ShapeT>::build()
{
  using top10::math::Vector;

  // Min and max coordinates
  double minX=HUGE_VAL, maxX=-HUGE_VAL;
  double minY=HUGE_VAL, maxY=-HUGE_VAL;
  double minZ=HUGE_VAL, maxZ=-HUGE_VAL;

  // For each shape
  for (int shape_idx = shapes.size()-1; shape_idx >= 0; shape_idx--) {
    top_node.shape_refs.push_back(shape_idx);
    GenericOctree__getBoundingBox(shapes.at(shape_idx), minX, maxX, minY, maxY, minZ, maxZ);
  }

  top_node.box = top10::math::AxisAlignedBox(Vector((minX+maxX)/2.0,
						    (minY+maxY)/2.0,
						    (minZ+maxZ)/2.0),
					     (maxX-minX)/2.0, (maxY-minY)/2.0, (maxZ-minZ)/2.0);

  splitNode(&top_node, 0);
}

template<typename ShapeT>
top10::helpers::GenericOctree<ShapeT>& top10::helpers::GenericOctree<ShapeT>::operator=(const GenericOctree<ShapeT>& other)
{
  POLY_THRESHOLD = other.POLY_THRESHOLD;
  MAX_DEPTH = other.MAX_DEPTH;
  deconstruct();
  top_node = other.top_node;
  shapes = other.shapes;
  for (int i=0; i<8; ++i) {
    if (top_node.children[i])
      top_node.children[i] = duplicateNode(top_node.children[i]);
  }
  return *this;
}

template<typename ShapeT>
top10::helpers::GenericOctree<ShapeT>::~GenericOctree()
{
  deconstruct();
}

template<typename ShapeT>
void top10::helpers::GenericOctree<ShapeT>::deconstruct()
{
  for (int i=0; i<8; ++i) {
    if (top_node.children[i])
      freeNode(top_node.children[i]);
  }
}

template<typename ShapeT>
typename top10::helpers::GenericOctree<ShapeT>::Node* top10::helpers::GenericOctree<ShapeT>::duplicateNode(const Node* node)
{
  Node* ret = new Node(*node);
  for (int i=0; i<8; ++i) {
    if (ret->children[i])
      ret->children[i] = duplicateNode(ret->children[i]);
  }
  return ret;
}

template<typename ShapeT>
void top10::helpers::GenericOctree<ShapeT>::freeNode(Node* node)
{
  for(int i=0; i<8; ++i) {
    if (node->children[i]) {
      freeNode(node->children[i]);
      node->children[i] = 0;
    }
  }
  delete node;
}

template<typename ShapeT>
void top10::helpers::GenericOctree<ShapeT>::splitNode(Node* node, int depth)
{
  using top10::math::AxisAlignedBox;
  using top10::math::Vector;

  double sizes[3];
  node->box.getEdgeSizes(sizes);
  Vector center = node->box.getCenter();

  // For each sub-cube
  for (int i=0; i<8; ++i) {
    Vector V;
    V[0] = i&1? 0.25 : -0.25;
    V[1] = i&2? 0.25 : -0.25;
    V[2] = i&4? 0.25 : -0.25;
    node->children[i] = buildNode(AxisAlignedBox(center +(V|Vector(sizes[0], sizes[1], sizes[2])),
						 sizes[0]/4.0, sizes[1]/4.0, sizes[2]/4.0),
				  node->shape_refs,
				  depth +1);
  }
}

template<typename ShapeT>
typename top10::helpers::GenericOctree<ShapeT>::Node* top10::helpers::GenericOctree<ShapeT>::buildNode(top10::math::AxisAlignedBox box,
												       const ShapeRefs& shape_refs,
												       int depth)
{
  if (shape_refs.size() < POLY_THRESHOLD || depth >= MAX_DEPTH) return 0;

  Node* res = new Node;
  res->box = box;
  for (ShapeRefs::const_iterator ref = shape_refs.begin();
       ref != shape_refs.end();
       ++ref) {
    if (GenericOctree__intersect(box, shapes.at(*ref))) res->shape_refs.push_back(*ref);
  }

  splitNode(res, depth);
  return res;
}

template<typename ShapeT> template <typename VolT>
typename top10::helpers::GenericOctree<ShapeT>::ShapeRefs top10::helpers::GenericOctree<ShapeT>::getVolume(const VolT& vol, int depth) const
{
  ShapeRefs res;
  
  // Keep track of which shapes are in the result, to avoid duplicates
  int n = 1+(shapes.size()/8);
  char *used_shapes = new char[n];
  for (int i=0; i<n; ++i) used_shapes[i] = 0;
  
  getVolume(vol, &top_node, depth, res, used_shapes);
  
  delete[] used_shapes;
  
  return res;
}

template<typename ShapeT> template <typename VolT>
void top10::helpers::GenericOctree<ShapeT>::getVolume(const VolT& vol, const Node* node,
						                                          int depth, ShapeRefs& res,
                                                      char* used_shapes)
{
#define CP_SHAPES \
    for (ShapeRefs::const_iterator it = node->shape_refs.begin(); it != node->shape_refs.end(); ++it) {\
      int cell = (*it)/8;\
      int bit = 1<<((*it) & 7);\
      if ((used_shapes[cell] & bit) == 0) {\
        res.push_back(*it);\
        used_shapes[cell] |= bit;\
      }\
    }

  if (depth <= 0) {
    CP_SHAPES;
    return;
  }

  OverlapType ov = GenericOctree__overlap(node->box, vol);
  if (ov == None) return;
  if (ov == Contains) {
    CP_SHAPES;
    return;
  }
  assert(ov == Intersect);

  bool is_leaf = true;
  for (int i=0; i<8; ++i) {
    if (node->children[i]) {
      is_leaf = false;
      getVolume(vol, node->children[i], depth-1, res, used_shapes);
    }
  }

  if (!is_leaf) return;
  else {
    CP_SHAPES;
    return;
  }
  
#undef CP_SHAPES
}

template<typename ShapeT>
std::list<top10::math::AxisAlignedBox> top10::helpers::GenericOctree<ShapeT>::getAllBlocks() const
{
  return getAllBlocks(&top_node);
}

template<typename ShapeT>
std::list<top10::math::AxisAlignedBox> top10::helpers::GenericOctree<ShapeT>::getAllBlocks(const Node* node) const
{
  std::list<top10::math::AxisAlignedBox> blocks;

  bool is_leaf = true;
  for (int i=0; i<8; ++i) {
    if (node->children[i]) {
      is_leaf = false;
      std::list<top10::math::AxisAlignedBox> tmp = getAllBlocks(node->children[i]);
      blocks.insert(blocks.end(), tmp.begin(), tmp.end());
    }
  }
  if (is_leaf && !node->shape_refs.empty())
    blocks.push_back(node->box);

  return blocks;
}

template<typename ShapeT> template<typename VolumeT>
std::list<top10::math::AxisAlignedBox> top10::helpers::GenericOctree<ShapeT>::getBlocks(const VolumeT& vol, int depth) const
{
  return getBlocks(vol, &top_node, depth);
}

template<typename ShapeT> template<typename VolumeT>
std::list<top10::math::AxisAlignedBox> top10::helpers::GenericOctree<ShapeT>::getBlocks(const VolumeT& vol, const Node* node, int depth) const
{
  if (depth <= 0) {
    blocks.push_back(node->box);
    return blocks;
  }

  std::list<top10::math::AxisAlignedBox> blocks;
  OverlapType ov = GenericOctree__overlap(node->box, vol);
  if (ov == None) return blocks;
  if (ov == Contains) {
    blocks.push_back(node->box);
    return blocks;
  }
  assert(ov == Intersect);

  bool is_leaf = true;
  for (int i=0; i<8; ++i) {
    if (node->children[i]) {
      is_leaf = false;
      std::list<top10::math::AxisAlignedBox> tmp = getBlocks(vol, node->children[i]);
      blocks.insert(blocks.end(), tmp.begin(), tmp.end());
    }
  }
  if (is_leaf ||
      blocks.empty()) /* If the intersection succeded for the big box, but not for the children,
			 there is a problem with GenericOctree__intersect (eg precision issues with small boxes).
			 To fix that, we return the bix box anyway */
    blocks.push_back(node->box);

  return blocks;
}
