/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_MATH_BOX_HH
#define TOP10_MATH_BOX_HH

#include "Vertex.hh"
#include "Matrix.hh"

namespace top10 {
  namespace math {
    class AxisAlignedBox {
    public:
      //! Unit box centerer in (0,0,0)
      AxisAlignedBox();

      //! Box with half-sizes
      AxisAlignedBox(Vector center, double s0, double s1, double s2);

      void getVertices(Vector out[8]) const;
      void getHalfEdges(Vector out[3]) const;
      void getEdgeSizes(double out[3]) const;
      inline Vector getCenter() const { return center; }

      inline void translate(Vector v) { center += v; }

      inline bool isInside(Vector p) const {
        Vector d = p-center;
        return (fabs(d.x) < s[0]) && (fabs(d.y) < s[1]) && (fabs(d.z) < s[2]);
      }
      
    private:
      Vector center;
      //! Size of each edge
      double s[3];      
    };

    class Box {
    public:
      //! Default box: center in (0,0,0), edges of length 1
      Box();
      //! Box with center and half-edges
      Box(Vector center, Vector e0, Vector e1, Vector e2);

      void getVertices(Vector out[8]) const;
      void getHalfEdges(Vector out[3]) const;
      void getEdgeSizes(double out[3]) const;
      inline Vector getCenter() const { return center; }

      //! Return an orthonormal matrix of the orientation of the box
      OrthoNorm3 getOrient() const;

      void translate(Vector v) { center += v; }
      //! Matrix must be orthonormal
      void rotate(const Matrix3&);

    private:
      Vector center;
      //! Half edges
      Vector e[3];
      //! Size of each edge
      double s[3];
    };
  };
};

#endif
