/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "Curve.hh"

using namespace top10::math;

double Curve2D::getY(double x) const
{
  std::map<double, double>::const_iterator b = ctrl_points.upper_bound(x);
  if (b == ctrl_points.end()) {
    b--;
    return b->second;
  }
  else if (b == ctrl_points.begin()) {
    return b->second;
  }
  else {
    std::map<double, double>::const_iterator a = b; a--;
    return (x - a->first)*(b->second - a->second)/(b->first - a->first) + a->second;
  }
}

bool Curve2D::addXY(double x, double y)
{
  return ctrl_points.insert(std::pair<double, double>(x,y)).second;
}

bool Curve2D::removeXY(double x)
{
  return ctrl_points.erase(x) == 1;
}

void Curve2D::dumpTo(std::ostream& str) const
{
  for (std::map<double, double>::const_iterator p = ctrl_points.begin();
       p != ctrl_points.end();
       ++p) {
    output<double>(str, p->first);
    output<double>(str, p->second);
  }
}

void Curve2D::loadFrom(std::istream& str)
{
  for (std::map<double, double>::iterator p = ctrl_points.begin();
       p != ctrl_points.end();
       ++p) {
    double first;
    input<double>(str, first);
    if (first != p->first) throw ParseError("Can not change the control points of a Curve2D in loadFrom()");
    input<double>(str, p->second);
  }
}

// ****************************************************************************

PointND CurveND::get(double t) const
{
  const_iterator b = upper_bound(t);
  if (b == end()) {
    b--;
    return b->second;
  }
  else if (b == begin()) {
    return b->second;
  }
  else {
    const_iterator a = b; a--;
    PointND ret;

    ret = (t - a->first)*(b->second - a->second)/(b->first - a->first) + a->second;

    return ret;
  }
}

bool CurveND::add(double t, const PointND& v)
{
  return insert(std::pair<double, PointND>(t,v)).second;
}
