/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "Plane.hh"

using namespace top10::math;

Plane::Plane(Vector p1, Vector p2, Vector p3)
{
  normal = ((p2-p1)^(p3-p1));
  double size = normal.size();
  if (size < SMALL_VECTOR) std::cerr<<"Plane::Plane: small triangle p1 p2 p3"<<std::endl;
  normal/=size;
  d = p1*normal;
}

Plane::Plane(Vector p, Vector n)
{
  normal = n;
  d = p*n;
}

void Plane::translate(Vector v)
{
  d += v*normal;
}

void Plane::rotate(const Matrix3& M)
{
  normal = M*normal;
}

bool top10::math::intersect(const Plane& plane, const Vector& A, const Vector& B, Vector& result)
{
  Vector IA, IB;
  Vector n = plane.normal;
  Vector I = n * plane.d;
  IA = A-I;
  IB = B-I;
  Vector AB = B-A;

  double nIA = n*IA;
  // Check if intersection
  if ((nIA)*(n*IB) > 0) return false;

  // Compute intersection
  double t = -nIA/(AB*n);

  result = A + t*AB;
  return true;
}

bool top10::math::intersectRay(const Plane& plane, const Vector& A, const Vector& V, Vector& result, double& d)
{
  double vn = V*plane.normal;
  if (fabs(vn) < SMALL_VECTOR) return false;

  double t = (plane.d - A*plane.normal)/vn;
  if (t<0) return false;

  result = A+t*V;
  d = t;
  return true;
}

Vector top10::math::project(const Plane& plane, const Vector& v)
{
  Vector p;
  Vector v2;
  const Vector& n = plane.normal;

  v2 = v - (n * plane.d);
  p = v2 - (v2*n)*n;
  return p + (n * plane.d);
}

Vector top10::math::projectV(const Plane& plane, const Vector& v)
{
  const Vector& n = plane.normal;
  return v - (v*n)*n;
}

void Plane::dumpTo(std::ostream& str) const
{
  normal.dumpTo(str);
  output<double>(str, d);
}

void Plane::loadFrom(std::istream& str)
{
  normal.loadFrom(str);
  input<double>(str, d);
}
