/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "Point.hh"


using namespace top10::physX;
using namespace std;

Point::Point(double x, double y, double z, double m):
  mass(m), pos(x, y, z)
{
}

Point::Point(Vector p, double m):
  mass(m), pos(p)
{
}

void Point::dumpTo(ostream& str) const
{
  old_pos.dumpTo(str);
  pos.dumpTo(str);
  next_pos.dumpTo(str);

  output<double>(str, mass);

  speed.dumpTo(str);
  next_speed.dumpTo(str);

  accel.dumpTo(str);
}

void Point::loadFrom(istream& str)
{
  old_pos.loadFrom(str);
  pos.loadFrom(str);
  next_pos.loadFrom(str);

  input<double>(str, mass);

  speed.loadFrom(str);
  next_speed.loadFrom(str);

  accel.loadFrom(str);
}

void Point::integrate(double dt)
{
  next_speed = speed + dt*accel;
  next_pos = pos + 0.5*dt*(speed + next_speed);
}

void Point::shift()
{
  assert(pos.size() < 1e5);
  assert(speed.size() < 1e8);
  assert(accel.size() < 1e11);

  old_pos = pos;
  pos = next_pos;
  speed = next_speed;

  next_speed = next_pos = Vector(0,0,0);
}

void Point::back()
{
  next_pos = pos;
  pos = old_pos;
  old_pos = Vector(0,0,0);
}

std::ostream& top10::physX::operator<<(std::ostream& out, const Point& p)
{
  out<<"pos = "<<p.pos<<std::endl;
  out<<"next_pos = "<<p.next_pos<<std::endl;

  out<<"speed = "<<p.speed<<std::endl;
  out<<"next_speed = "<<p.next_speed<<std::endl;

  out<<"accel = "<<p.accel<<std::endl;

  return out;
}
