/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef RIGID_BODY_H
#define RIGID_BODY_H

#include <vector>
#include <cassert>

#include "util/Dumpable.hh"
#include "util/Debug.hh"
#include "math/Matrix.hh"
#include "Point.hh"

namespace top10 {
  namespace physX {

    class World;

    class RigidBody: public top10::util::Dumpable, public top10::util::Debug
    {
    public:
      //! Builds an empty body. Before using it, you *have* to call setMasses
      RigidBody(const char* name="Unnamed rigid body", bool debug=false);
      //! Set masses
      void setMasses(const std::vector<Point>& points);

      void clearForces();
      //! Apply a force. Position and force expressed in global coordinates
      void applyForceAt(Vector position, Vector force);
      //! Apply a force. position expressed in local coords, force in global!
      void applyForceAtL(Vector position, Vector force);
      //! Apply a torque, local coordinates
      inline void applyTorqueL(Vector T) { central_torque += T; }
      //! Apply a torque, global coordinates
      inline void applyTorque(Vector T) { central_torque += orient_inv*T; }
      //! Update the orientation, position and speeds of the body
      void integrate(double dt);

      //! Modify the linear velocity because of a collision.
      void applyLinearImpulseL(Vector impulse);
      //! Modify velocities (linear + angular) because of a collision.
      void applyImpulseAtL(Vector pos, Vector impulse);

      //! Move back the body to its previous position (useful for collision response)
      inline void moveBack() { translation.back(); }

      //! Return the orientation matrix
      inline const top10::math::OrthoNorm3& getOrientation() const {return orient;}
      //! Set the orientation
      inline void setOrientation(const top10::math::Matrix3& M) { orient = M; matrix_count=0; }
      //! Set the position. Vertex originally in (0,0,0) is translated to pos.
      inline void setTranslation(Vector t) { translation.pos = t; }
      inline Vector getTranslation() const { return translation.pos; }
      //! Get the global coordinates of local coordinates
      Vector getGlobalFromLocal(Vector) const;
      //! Get the local coordinates of global coords
      Vector getLocalFromGlobal(Vector) const;

      //! Return the force applied on the body
      inline Vector getForce() const {return central_force;}
      //! Return the torque in the center, local coordinates
      inline Vector getTorqueL() const {return central_torque;}
      //! Return the center of mass
      inline Vector getMassCenter() const {return g_center + translation.pos;}
      //! Return the total mass
      inline double getMass() const {return mass;}
      //! Return the position of the center of mass in local coordinates.
      inline Vector getMassCenterL() const {return g_center;}

      //! Return the speed of a vertex in the body
      Vector getSpeedAt(Vector position) const;
      //! Return the speed of a vertex (using local coordinates) in the body
      Vector getSpeedAtL(Vector local_position) const;

      //! Return the angular velocity, local coordinates
      inline Vector getAngularSpeedL() const { return w_speed; }

      inline const top10::math::Matrix3& getInertiaInv() const { return inertia_inv; }

      bool isActive() const;
      
      void dumpTo(std::ostream&) const;
      void loadFrom(std::istream&);

    private:
      //! Copy operator disabled.
      RigidBody& operator=(const RigidBody& other) { assert(0); return *(RigidBody*)0; }

    private:
      /* Those are the characteristics of a rigid body. They never change */
      //! Position of the center of gravity relative to the initial origin
      Vector g_center;
      //! Mass of the rigid body
      double mass;
      //! mass moment of inertia in local coordinates
      top10::math::Matrix3 inertia;
      //! inverse of inertia
      top10::math::Matrix3 inertia_inv;

      /* Here are the variables of the state of the rigid body. They are updated every time step */
      //! Orientation of the body, relative to the center of gravity
      top10::math::OrthoNorm3 orient;
      //! Inverse of the orientation
      top10::math::OrthoNorm3 orient_inv;
      //! Translation. It's a point to make it easier to update it.
      Point translation;
      //! Angular velocity, local coordinates
      Vector w_speed;
      //! Sum of all force vectors, in global coordinates
      Vector central_force;
      //! Sum of all torques on the center of gravity, in local coordinates
      Vector central_torque;

      //! The orientation matrix is corrected every now and then.
      int matrix_count;      
    };
  };
};
#endif
