/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "Track.hh"

#include "math/intersections.hh"
#include "helpers/3dsRead.hh"
#include "helpers/OrientMatrix.hh"
#include "physX/Surface.hh"
#include "physX/SurfacePatch.hh"
#include "util/PathFinder.hh"
#include "util/Parse.hh"
#include "util/GlobalOptions.hh"
#include "graphX/NodeIterator.hh"
#include <fstream>

using namespace top10;
using namespace top10::track;

using top10::util::Error;
using top10::math::Vector;
using top10::helpers::Read3DS;
using top10::racing::CheckPoint;
using top10::racing::CheckPoints;
using top10::racing::StartingArea;

Track::Track(top10::graphX::Node* m, const std::set<std::string>& deco_mesh_names,
             const std::set<std::string>& road_surfaces,
             const std::set<std::string>& dusty_surfaces,
             const std::set<std::string>& border_surfaces,
             const std::set<std::string>& dirt_surfaces,
             const std::set<std::string>& grass_surfaces,
             const std::set<std::string>& sand_surfaces):
    top_node(m),
    friction_factor(1.0),
    trimap(1, SetRow(1)),
    min_v(1e6, 1e6, 1e6),
    max_v(-1e6, -1e6, -1e6)
{
  // Decorative triangles are duplicated and store in an octree, used for collision detection.
  std::vector<top10::math::Triangle> triangles;
  
  // The surface types
  surfaces[ROAD_IDX] = top10::physX::SurfaceConcrete();
  surfaces[BORDER_IDX] = top10::physX::SurfaceBorder();
  surfaces[SAND_IDX] = top10::physX::SurfaceSand();
  surfaces[DIRT_IDX] = top10::physX::SurfaceDirt();
  surfaces[GRASS_IDX] = top10::physX::SurfaceGrass();
  surfaces[DUST_IDX] = top10::physX::SurfaceDust();

  // Duplicate non-decorative meshes so that they all share the same transformation
  top10::graphX::LeafNodeIterator node_it(m);
  while (!node_it.atEnd()) {
    const top10::graphX::MeshNode* mesh_node = dynamic_cast<const top10::graphX::MeshNode*>(*node_it);
    if (mesh_node) {
      if (deco_mesh_names.find(mesh_node->getName()) == deco_mesh_names.end()) {
        std::cerr<<"Found ground object "<<mesh_node->getName()<<std::endl;
        // Create a new transformed mesh
        top10::math::Mesh* my_mesh = new top10::math::Mesh(*(mesh_node->getMesh()));
        my_mesh->transform(node_it.getTransform());
        meshes.push_back(my_mesh);
      
        // Find the index of the surface type according to the name of the material of this mesh
        int surf_idx = 0;
#define FINDSURF(a) (a.find(mesh_node->getMaterialName()) != a.end())
        if (FINDSURF(road_surfaces)) surf_idx = ROAD_IDX;
        else if (FINDSURF(dusty_surfaces)) surf_idx = DUST_IDX;
        else if (FINDSURF(border_surfaces)) surf_idx = BORDER_IDX;
        else if (FINDSURF(dirt_surfaces)) surf_idx = DIRT_IDX;
        else if (FINDSURF(grass_surfaces)) surf_idx = GRASS_IDX;
        else if (FINDSURF(sand_surfaces)) surf_idx = SAND_IDX;
#undef FINDSURF  
  
        // for each face in this mesh
        const std::vector<top10::math::Vector>* vertices = my_mesh->getVertices();
        for (int idx=0; idx<my_mesh->getFaces()->size(); ++idx)
        {
          top10::math::Mesh::Face face = my_mesh->getFaces()->at(idx);
          // Add a FaceRef to the TriMap
          trimap[0][0].push_back(FaceRef(my_mesh, idx, surf_idx));

          // Update the min and max points      
          for (int i=0; i<3; ++i) {
            top10::math::Vector p = vertices->at(face.idxs[i]);
            if (p.x < min_v.x) min_v.x = p.x;
            if (p.y < min_v.y) min_v.y = p.y;
            if (p.z < min_v.z) min_v.z = p.z;

            if (p.x > max_v.x) max_v.x = p.x;
            if (p.y > max_v.y) max_v.y = p.y;
            if (p.z > max_v.z) max_v.z = p.z;
          }
        }
      }
      // It is a decoration
      else {
        std::cerr<<"Found deco object "<<mesh_node->getName()<<std::endl;
        top10::math::Mesh tmp(*(mesh_node->getMesh()));
        tmp.transform(node_it.getTransform());
        const std::vector<top10::math::Vector>* vertices = tmp.getVertices();
        const std::vector<top10::math::Mesh::Face>* faces = tmp.getFaces();
        for (std::vector<top10::math::Mesh::Face>::const_iterator faces_it = faces->begin(); faces_it != faces->end(); ++faces_it) {
          top10::math::Triangle t(vertices->at(faces_it->idxs[0]),
                                  vertices->at(faces_it->idxs[1]),
                                  vertices->at(faces_it->idxs[2]));
          triangles.push_back(t);
        }
      }
    }
    
    ++node_it;
  }
  
  Vector _diag = max_v - min_v;
  // cell_size = max(_diax.x, _diag.y, _diag.z)
  cell_size = _diag.x;
  if (_diag.y > cell_size) cell_size = _diag.y;
  if (_diag.z > cell_size) cell_size = _diag.z;
  
  deco_octree = top10::helpers::GenericOctree<top10::math::Triangle>(triangles, getOptI("Track.Octree.Polygons"), getOptI("Track.Octree.Depth"));
}

Track::~Track()
{
  for (std::vector<const top10::math::Mesh*>::iterator mesh_it = meshes.begin(); mesh_it != meshes.end(); ++mesh_it) delete *mesh_it;
}

void Track::buildTriMap(int depth)
{
  for (int i=0; i<depth; ++i) {
      iterateBuild();
  }
}

void Track::iterateBuild()
{
  // Save current values
  TriMap old_trimap = trimap;
  double old_cell_size = cell_size;
  int old_n = old_trimap.size();

  // Update new values
  trimap.clear();
  int n=old_n*2;
  trimap = TriMap(n, SetRow(n));
  cell_size /= 2.0;

  double x = min_v.x;
  for (int col=0; col < old_n; ++col, x+=old_cell_size) {
    double z=min_v.z;
    for (int row=0; row < old_n; ++row, z+=old_cell_size) {
      subdivide(old_trimap[col][row], x, z, cell_size,
		trimap[2*col][2*row], trimap[2*col+1][2*row],
		trimap[2*col][2*row+1], trimap[2*col+1][2*row+1]);
    }
  }
}

void Track::subdivide(const FaceRefSet& faces, double x, double z, double size,
			  FaceRefSet& ll, FaceRefSet& lr,
			  FaceRefSet& ul, FaceRefSet& ur)
{
  for (FaceRefSet::const_iterator face_it=faces.begin();
       face_it!=faces.end();
       ++face_it) {
    if (cellIntersect(*face_it, x, z, size)) {
       ll.push_back(*face_it);
    }
    if (cellIntersect(*face_it, x+size, z, size)) {
       lr.push_back(*face_it);
    }
    if (cellIntersect(*face_it, x, z+size, size)) {
       ul.push_back(*face_it);
    }
    if (cellIntersect(*face_it, x+size, z+size, size)) {
       ur.push_back(*face_it);
    }
  }
}

static inline double cross(double x1, double y1, double x2, double y2)
{
  return x1*y2 - y1*x2;
}

bool Track::diag1_intersect(double x1, double y1, double x2, double y2, double X, double Y, double s)
{
  double V1 = (x1 - X - (y1 -Y));
  double V2 = (x2 - X - (y2 - Y));
  if (V1 > 0.0 && V2 > 0.0) return false;
  if (V1 < 0.0 && V2 < 0.0) return false;
  
  V1 = cross(X-x1, Y-y1, x2-x1, y2-y1);
  V2 = V1 + s*((y2 - y1) - (x2 - x1));
  if (V1 > 0.0 && V2 > 0.0) return false;
  if (V1 < 0.0 && V2 < 0.0) return false;
  
  return true;	
}

bool Track::diag2_intersect(double x1, double y1, double x2, double y2, double X, double Y, double s)
{
  double V1 = (-(x1 - X) - (y1 -Y));
  double V2 = (-(x2 - X) - (y2 - Y));
  if (V1 > 0.0 && V2 > 0.0) return false;
  if (V1 < 0.0 && V2 < 0.0) return false;
  
  V1 = cross(X-x1, Y-y1, x2-x1, y2-y1);
  V2 = V1 + s*((y2 - y1) + (x2 - x1));
  if (V1 > 0.0 && V2 > 0.0) return false;
  if (V1 < 0.0 && V2 < 0.0) return false;
  
  return true;	
}

bool Track::cellIntersect(FaceRef face, double x, double z, double size)
{
  top10::math::Vector p[3];
  // Check if one of the vertices lies inside the square
  for (int i=0; i<3; ++i) {
    p[i] = toVector(face, i);
    if (p[i].x >= x && p[i].x < x+size &&
	p[i].z >= z && p[i].z < z+size) {
      return true;
    }
  }
  
  // Check if one of the square vertices lies within the polygon
  if (inPolygon(face, x, z)) return true;
  if (inPolygon(face, x+size, z)) return true;
  if (inPolygon(face, x+size, z+size)) return true;
  if (inPolygon(face, x, z+size)) return true;
  
  // Check if polygon and square segments intersect
  // Trick: I test against the square's diagonals instead of its edges
  for (int i=0; i<3; ++i) {
    int i2 = (i==2)?0:(i+1);
    if (diag1_intersect(p[i].x, p[i].z, p[i2].x, p[i2].z, x, z, size)) return true;
    if (diag2_intersect(p[i].x, p[i].z, p[i2].x, p[i2].z, x, z+size, size)) return true;
  }
  
  return false;
}

bool Track::inPolygon(FaceRef face, double x, double z)
{
  top10::math::Vector p[3];
  toVectors(face, p);
  
  return inPolygon(p, x, z);
}

bool Track::inPolygon(top10::math::Vector* p, double x, double z)
{
  /* to check if the point is inside a triangle, I use the cross product
     method.
  */
#define CROSS(a,b) ((p[b].x - p[a].x)*(z-p[a].z) - (p[b].z - p[a].z)*(x-p[a].x))
  double sign = CROSS(0,1);
  // If the point is on the edge, I consider it is not in the polygon
  if (fabs(sign) < 1e-5) return false;

  for (int i=0; i<3; ++i) {
    int i2 = (i==2)?0:(i+1);
    if ((CROSS(i,i2) * sign) < 0.0) return false;
  }
#undef CROSS

  return true;
}

bool Track::getSurfacePatch(const Vector& p_orig, top10::physX::SurfacePatch& out)
  const
{
  int found = 0;

  // p_orig expressed in local base
  Vector p( p_orig.x - min_v.x,
	    0,
	    p_orig.z - min_v.z);
  int c = int(p.x/cell_size);
  int l = int(p.z/cell_size);
  Vector p1, p2, p3;

  if (c >= trimap.size()) return false;
  if (c < 0) return false;

  if (l >= trimap[c].size()) return false;
  if (l < 0) return false;

  for (FaceRefSet::const_iterator ptri = trimap[c][l].begin();
       ptri != trimap[c][l].end();
       ptri++) {
    top10::math::Vector tri[3];
    toVectors(*ptri, tri);
    if (inPolygon(tri, p_orig.x, p_orig.z)) {
      p1 = tri[0];
      p2 = tri[1];
      p3 = tri[2];
      // Check if normal points upward
      if (((p2-p1)^(p3-p1)).y < 0) {
	Vector temp = p2;
	p2 = p3;
	p3 = temp;
      }

      out = top10::physX::SurfacePatch(top10::math::Plane(p1, p2, p3), surfaces[ptri->surf_idx]);
      out.grip *= friction_factor;
      found++;
      return true;
    }
  }

  if (found > 1) {
    std::cerr<<"Found more than one triangle: "<<found<<std::endl;
  }

  return found > 0;
}

std::vector<top10::math::Triangle> Track::getPotentialCollisions(const top10::math::Box& box) const
{
  std::vector<top10::math::Triangle> ret;
  
  Octree::ShapeRefs refs = deco_octree.getVolume(box);
  const Octree::ShapeVec* tris = deco_octree.getShapeVec();
  
  for (Octree::ShapeRefs::const_iterator ref_it = refs.begin(); ref_it != refs.end(); ++ref_it) {
    ret.push_back(tris->at(*ref_it));
  }
  
  return ret;
}

#if 0
const Track::FaceRefSet* Track::debugTriMap(const Vector& p_orig) const
{
  // p_orig expressed in local base
  Vector p( p_orig.x - min_v.x,
	    0,
	    p_orig.z - min_v.z);
  int c = int(p.x/cell_size);
  int l = int(p.z/cell_size);

  if (c >= trimap.size()) return 0;
  if (c < 0) return 0;

  if (l >= trimap[c].size()) return 0;
  if (l < 0) return 0;

  return &trimap[c][l];
}
#endif

void Track::setStartingArea(StartingArea sa) { starting_area = sa; }
void Track::setCheckPoints(CheckPoints cp) { checkpoints = cp; }

//if partial=true, we already parsed the opening < before coming here
static Vector parseVector(std::istream& in, bool partial=false)
{
  std::string token;

  if (!partial) {
    in >> token;
    if (token != "<") throw Error("Expected <, got ") + token;
  }

  double x, y, z;
  in >> x >> y >> z;

  Vector v(x, y, z);

  in >> token;
  if (token != ">") throw Error("Expected >, got ") + token;

  return v;
}

static void writeVector(std::ostream& out, Vector v)
{
  out << " < ";
  for (int i=0; i<3; ++i) out<<v[i]<<" ";
  out << " > ";
}

Track* top10::track::loadTrack(std::string filename)
{
  using namespace top10::helpers;

  Track* track;

  std::string meshname;
  top10::math::AxisEnum axis;
  bool axis_neg;
  double scale;
  top10::racing::StartingArea area;
  top10::racing::CheckPoints cps;
  std::set<std::string> decos;
  std::set<std::string> road_surfaces;
  std::set<std::string> dusty_surfaces;
  std::set<std::string> border_surfaces;
  std::set<std::string> dirt_surfaces;
  std::set<std::string> grass_surfaces;
  std::set<std::string> sand_surfaces;

  loadTrackParts(filename, &meshname, &axis, &axis_neg, &scale, &area, &cps, &decos,
		 &road_surfaces, &dusty_surfaces, &border_surfaces,
		 &dirt_surfaces, &grass_surfaces, &sand_surfaces);

  // Load the mesh of the ground
  top10::util::PathFinder finder = top10::util::PathFinder::defaultPathFinder();
  std::string filename_path = finder.find(filename);
  finder.addPath(top10::util::PathFinder::getPath(filename_path));
  std::string mesh_path = finder.find(meshname);
  if (mesh_path.empty()) throw Error("Could not find ")+meshname;
  top10::helpers::Read3DS r3ds(mesh_path);
  top10::graphX::Node* mesh = r3ds.getTopNode();
  
  // Setup the top-level transform
  top10::graphX::TransformNode* top_transform = new top10::graphX::TransformNode;
  top_transform->setToWorld(makeOrientation(axis, axis_neg, scale));
  top_transform->addChild(mesh);
        
  // Instantiate the track
  track = new Track(top_transform, decos, road_surfaces, dusty_surfaces, border_surfaces, dirt_surfaces, grass_surfaces, sand_surfaces);
  
  // Set the starting area and checkpoints
  track->setStartingArea(area);
  track->setCheckPoints(cps);

  // Build the mapping from squares to triangles to accelerate collision detection
  track->buildTriMap();

  return track;
}

void top10::track::loadTrackParts(std::string filename,
				  std::string* meshname, top10::math::AxisEnum *axis,
				  bool* axis_neg, double* scale,
				  StartingArea* area, CheckPoints* cps,
				  std::set<std::string>* decorative_meshes,
				  std::set<std::string>* road_surfaces,
				  std::set<std::string>* dusty_surfaces,
				  std::set<std::string>* border_surfaces,
				  std::set<std::string>* dirt_surfaces,
				  std::set<std::string>* grass_surfaces,
				  std::set<std::string>* sand_surfaces)
{
  using top10::util::parseString;

  // We don't modify the output args directly, for atomicity.
  std::string tmp_meshname;
  top10::math::AxisEnum tmp_axis;
  bool tmp_axis_neg;
  double tmp_scale;
  top10::racing::StartingArea tmp_area;
  top10::racing::CheckPoints tmp_cps;
  std::set<std::string> tmp_decorative_meshes;
  std::set<std::string> tmp_road_surfaces;
  std::set<std::string> tmp_dusty_surfaces;
  std::set<std::string> tmp_border_surfaces;
  std::set<std::string> tmp_dirt_surfaces;
  std::set<std::string> tmp_grass_surfaces;
  std::set<std::string> tmp_sand_surfaces;

  top10::util::PathFinder my_path_finder = top10::util::PathFinder::defaultPathFinder();
  my_path_finder.addPath(top10::util::PathFinder::getPath(filename));
  std::string file_path = my_path_finder.find(filename);
  if (file_path.empty()) throw Error("Could not find ")+filename;
  std::ifstream in(file_path.c_str());
  if (!in) throw Error(std::string("Could not open ") + file_path);
  
  /* Expected:
     MESH mesh_filename
     UP <X, Y, Z, -X, -Y or -Z>
     SCALE 1.2345

  */
  std::string token;
  in >> token;

  if (token == "MESH") {
    tmp_meshname = parseString(in);
    // We only want to keep the basename of the file, not the path.
    std::string::size_type pos = tmp_meshname.rfind("/", tmp_meshname.size());
    if (pos != std::string::npos) {
      tmp_meshname = tmp_meshname.substr(pos+1);
    }

    in >> token;
    if (token != "UP") throw Error(std::string("Expected UP, got ") + token);
    in >> token;
    top10::math::Matrix4 trans;

    tmp_axis_neg = false;
    if (token == "X")       { tmp_axis = top10::math::X; tmp_axis_neg = false; }
    else if (token == "-X") { tmp_axis = top10::math::X; tmp_axis_neg = true;  }
    else if (token == "Y")  { tmp_axis = top10::math::Y; tmp_axis_neg = false; }
    else if (token == "-Y") { tmp_axis = top10::math::Y; tmp_axis_neg = true;  }
    else if (token == "Z")  { tmp_axis = top10::math::Z; tmp_axis_neg = false; }
    else if (token == "-Z") { tmp_axis = top10::math::Z; tmp_axis_neg = true;  }
    else throw Error(std::string("Expected X, Y or Z, got ") + token);

    in >> token;
    if (token != "SCALE") throw Error(std::string("Expected SCALE, got ") + token);
    in >> tmp_scale;
  }
  else throw Error(std::string("Expected MESH or VECTOR, got: ") + token);

  /* Read the checkpoints. Grammar:
     BEGIN
     checkpoints
     END
     
     checkpoints := empty | checkpoint checkpoints
     checkpoint := vector vector
     vector := <x y z>
  */
  in >> token;
  if (token != "BEGIN") throw Error("Expected BEGIN, got ") + token;
  for (;;) {
    in >> token;
    if (token == "END") break;
    if (in.eof()) throw Error("Early EOF");
    
    Vector p1 = parseVector(in, true);
    Vector p2 = parseVector(in);
    
    tmp_cps.push_back(CheckPoint(p1, p2));
  }
  
  /* Read the starting area. Expected:
     vector vector vector.
     (point1 point2 direction)
  */
  Vector p1 = parseVector(in);
  Vector p2 = parseVector(in);
  Vector dir = parseVector(in);
  
  tmp_area = StartingArea(p1, p2, dir);

  /* Read the surface properties
     BEGIN
     material_name surface_type
     ...
     END

     surface_type is: ROAD, DUSTY_ROAD, BORDER, DIRT, GRASS, SAND
  */
  in >> token;
  if (token != "BEGIN") throw Error("Expected BEGIN, got ") + token;
  for (;;) {
    token = parseString(in);
    if (token == "END") break;
    if (in.eof()) throw Error("Early EOF");

    std::string material_name = token;
    in>>token;
    if (token == "ROAD") tmp_road_surfaces.insert(material_name);
    else if (token == "DUSTY_ROAD") tmp_dusty_surfaces.insert(material_name);
    else if (token == "BORDER") tmp_border_surfaces.insert(material_name);
    else if (token == "DIRT") tmp_dirt_surfaces.insert(material_name);
    else if (token == "GRASS") tmp_grass_surfaces.insert(material_name);
    else if (token == "SAND") tmp_sand_surfaces.insert(material_name);
    else throw Error("Expected surface type, got ") + token;
  }

  /* Read the names of meshes that are not part of the ground. Expected:
     BEGIN
     mesh name
     ...
     END
  */
  in >> token;
  if (token != "BEGIN") throw Error("Expected BEGIN, got ") + token;
  for (;;) {
    token = parseString(in);

    if (token == "END") break;
    if (in.eof()) throw Error("Early EOF");

    tmp_decorative_meshes.insert(token);
  }

  // Everything went well, modify the output args now.
  *meshname = tmp_meshname;
  *axis = tmp_axis;
  *axis_neg = tmp_axis_neg;
  *scale = tmp_scale;
  *area = tmp_area;
  *cps = tmp_cps;
  *decorative_meshes = tmp_decorative_meshes;
  *road_surfaces = tmp_road_surfaces;
  *dusty_surfaces = tmp_dusty_surfaces;
  *border_surfaces = tmp_border_surfaces;
  *dirt_surfaces = tmp_dirt_surfaces;
  *grass_surfaces = tmp_grass_surfaces;
  *sand_surfaces = tmp_sand_surfaces;
}

void top10::track::saveMeshTrack(std::string filename,
				 std::string meshname, top10::math::AxisEnum axis, bool axis_neg, double scale,
				 StartingArea area, CheckPoints cps,
				 std::set<std::string> decorative_meshes,
				 std::set<std::string> road_surfaces,
				 std::set<std::string> dusty_surfaces,
				 std::set<std::string> border_surfaces,
				 std::set<std::string> dirt_surfaces,
				 std::set<std::string> grass_surfaces,
				 std::set<std::string> sand_surfaces)
{
  std::ofstream out(filename.c_str());
  if (!out) throw Error("Could not open ") + filename;
  
  /* Expected:
     MESH <mesh file type: LW or 3DS> mesh_filename
     UP <X, Y, Z, -X, -Y or -Z>
     SCALE 1.2345

  */
  // We only want to keep the basename of the file, not the path.
  std::string::size_type pos = meshname.rfind("/", meshname.size());
  if (pos != std::string::npos) {
    meshname = meshname.substr(pos+1);
  }
  out << "MESH " << meshname << std::endl;

  out << "UP " ;
  if (axis_neg) out<<"-";

  if (axis >=0 && axis < 3) {
    const char names[3] = {'X', 'Y', 'Z'};
    out << names[(int)axis];
  }
  else throw Error("Internal Error: invalid axis index");

  out<<std::endl;

  out << "SCALE " << scale << std::endl;

  /* Write the checkpoints. Grammar:
     BEGIN
     checkpoints
     END
     
     checkpoints := empty | checkpoint checkpoints
     checkpoint := vector vector
     vector := <x y z>
  */
  out << "BEGIN" << std::endl;
  for (CheckPoints::const_iterator cp = cps.begin();
       cp != cps.end();
       ++cp) {
    writeVector(out, cp->getPoint1());
    writeVector(out, cp->getPoint2());
  }
  out<< "END" << std::endl;

  /* Write the starting area. Expected:
     vector vector vector
     (point1 point2 direction)
  */

  writeVector(out, area.getPoint1());
  writeVector(out, area.getPoint2());
  writeVector(out, area.getDirection());

  /* Write the surface filters.
     BEGIN
     material_name surface_type
     ...
     END

     surface_type is: ROAD, DUSTY_ROAD, BORDER, DIRT, GRASS, SAND
  */
  out << "BEGIN" << std::endl;

  for (std::set<std::string>::const_iterator str = road_surfaces.begin();
       str != road_surfaces.end();
       ++str) {
    out<<"\""<<(*str)<<"\" ROAD"<<std::endl;
  }

  for (std::set<std::string>::const_iterator str = dusty_surfaces.begin();
       str != dusty_surfaces.end();
       ++str) {
    out<<"\""<<(*str)<<"\" DUSTY_ROAD"<<std::endl;
  }

  for (std::set<std::string>::const_iterator str = dirt_surfaces.begin();
       str != dirt_surfaces.end();
       ++str) {
    out<<"\""<<(*str)<<"\" DIRT"<<std::endl;
  }

  for (std::set<std::string>::const_iterator str = border_surfaces.begin();
       str != border_surfaces.end();
       ++str) {
    out<<"\""<<(*str)<<"\" BORDER"<<std::endl;
  }

  for (std::set<std::string>::const_iterator str = grass_surfaces.begin();
       str != grass_surfaces.end();
       ++str) {
    out<<"\""<<(*str)<<"\" GRASS"<<std::endl;
  }

  for (std::set<std::string>::const_iterator str = sand_surfaces.begin();
       str != sand_surfaces.end();
       ++str) {
    out<<"\""<<(*str)<<"\" SAND"<<std::endl;
  }

  out << "END" << std::endl;

  /* Write the names of decorative meshes (those that are not part of the ground) */
  out<<"BEGIN"<<std::endl;
  for (std::set<std::string>::const_iterator it = decorative_meshes.begin();
       it != decorative_meshes.end();
       ++it) {
    out<<"\""<<(*it)<<"\""<<std::endl;
  }
  out<<"END"<<std::endl;
}

/*
*
* Octree specialization for triangles
*
*/
#include "helpers/GenericOctree-template.cpp"

namespace top10 {
  namespace helpers {
    
template<>
void GenericOctree__getBoundingBox<top10::math::Triangle>(const top10::math::Triangle& tri,
                                                          double& minX, double& maxX,
                                                          double& minY, double& maxY,
                                                          double& minZ, double& maxZ)
{
  for (int i=0; i<3; ++i) {
#define CMP(coord, m, M) if (tri.p[i].coord < m) m=tri.p[i].coord; if (tri.p[i].coord > M) M=tri.p[i].coord;
    CMP(x, minX, maxX);
    CMP(y, minY, maxY);
    CMP(z, minZ, maxZ);
#undef CMP
  }   
}

template<>
OverlapType GenericOctree__overlap<top10::math::Box>(const top10::math::AxisAlignedBox& aabox, const top10::math::Box& box)
{
  if (!top10::math::intersect(aabox, box)) return None;
  top10::math::Vector v[8];
  box.getVertices(v);
  for (int i=0; i<8; ++i) if (aabox.isInside(v[i])) return Intersect;
  return Contains;
}

template<>
bool GenericOctree__intersect<top10::math::Triangle>(const top10::math::AxisAlignedBox& aabox, const top10::math::Triangle& triangle)
{
  return top10::math::intersect(aabox, triangle);
}

}
}
