/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "SurfacesEditor.hh"
#include "SurfaceTypeProxyNode.hh"

using namespace top10::tracked;

SurfacesEditor::SurfacesEditor():
  mesh_ed(0), proxy_node(0), offset_node(0)
{
  surfaces[Road] =      &road_surfaces;
  surfaces[DustyRoad] = &dusty_road_surfaces;
  surfaces[Border] =    &border_surfaces;
  surfaces[Grass] =     &grass_surfaces;
  surfaces[Sand] =      &sand_surfaces;
  surfaces[Dirt] =      &dirt_surfaces;

  alpha_node = new top10::graphX::AlphaNode;
  alpha_node->setAlpha(128);
  getNode()->addChild(alpha_node.getPtr());
  
  offset_node = new top10::graphX::DepthOffsetNode;
  offset_node->setOffset(-1.0);
  alpha_node->addChild(offset_node.getPtr());
  
  reset();
}

void SurfacesEditor::reset()
{
  current_surface_name.clear();
  current_surface_type = Road;
}

void SurfacesEditor::setMeshEditor(MeshEditor* p)
{
  mesh_ed = p;
  offset_node->removeChild(proxy_node.getPtr());
  proxy_node = new top10::graphX::TransformNodeProxy(p->getTransformNode(), new SurfaceTypeProxyOperation(this));
  offset_node->addChild(proxy_node.getPtr());
}
      
std::set<std::string> SurfacesEditor::getCurrentSurfaceSet() const
{
  switch (current_surface_type) {
  case Road:      return road_surfaces;       break;
  case DustyRoad: return dusty_road_surfaces; break;
  case Grass:     return grass_surfaces;      break;
  case Border:    return border_surfaces;     break;
  case Sand:      return sand_surfaces;       break;
  case Dirt:      return dirt_surfaces;       break;
  }
  abort();
}

void SurfacesEditor::pick()
{
  assert(mesh_ed);

  current_surface_name = mesh_ed->getPickedSurface();
}

void SurfacesEditor::add()
{
  if (current_surface_name.empty()) throw std::string("No surface name to add to the current surface type");

  int found=0;
  // 6 surface types... (ugh)
  for (int i=0; i<6; ++i) {
    if (i==current_surface_type) surfaces[i]->insert(current_surface_name);
    else {
      std::set<std::string>::iterator f=surfaces[i]->find(current_surface_name);
      if (f != surfaces[i]->end()) {
	surfaces[i]->erase(f);
	found++;
      }
    }
  }
  assert(found==0 || found==1);
}

void SurfacesEditor::remove()
{
  if (current_surface_name.empty()) throw std::string("No surface name to remove");

  int found=0;
  // 6 surface types... (ugh)
  for (int i=0; i<6; ++i) {
    std::set<std::string>::iterator f=surfaces[i]->find(current_surface_name);
    if (f != surfaces[i]->end()) {
      surfaces[i]->erase(f);
      found++;
    }
  }
  assert(found==0 || found==1);
}
