/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "Simulation.hh"
#include "util/SlowTime.hh"
#include "util/GlobalOptions.hh"

using namespace top10::ui_interactive;
using namespace top10::math;
using namespace top10::racing;
using namespace std;

Simulation::Simulation(top10::track::Track* _track, int n_karts):
  track(_track),
  isWet(false),
  current_lap_records(n_karts),
  steer_angle(0),
  throttle(0),
  braking(0),
  current_t(0)
{
  world.fixed_objects.push_back(track.getPtr());
  
  // Put the karts on the track
  for (int i=0; i<n_karts; ++i) {
    Vector pos;
    Vector dir;
    track->getStartingArea()->getStartingPos(i, pos, dir);
    top10::util::Ref<Kart> kart(new Kart(track.getPtr(), pos + Vector(0,0.5,0), dir));
    karts.push_back(kart.getPtr());
    timers.push_back(KartTimer(kart.getPtr(), track->getCheckPoints()));
    // Register the karts to the "physics engine"
    kart->getRegistered(world);
  }
  
  lap_record_period = 1000*getOptD("Record.Period");
  last_record_update = 0;
}


void Simulation::sendEvent(Action action, Sint16 value)
{
  switch (action) {
  case Steering:
    steer_angle = -value/32768.0;
    break;
  case Acceleration:
    throttle = (value+32768)/65536.0;
    break;
  case Braking:
    braking = (value+32768)/65536.0;
    break;
  case AccelerationAndBraking:
    if (value > 0) {
      throttle = value/32768.0;
      braking = 0.0;
    }
    else {
      throttle = 0.0;
      braking = -value/32768.0;
    }
    break;
  case BrakingAndAcceleration:
    if (value < 0) {
      throttle = -value/32768.0;
      braking = 0.0;
    }
    else {
      throttle = 0.0;
      braking = value/32768.0;
    }
    break;

  default: abort();  /* We should not receive other kinds of events */
  }
}

void Simulation::update(unsigned int now)
{
  assert(now >= current_t);
  
  unsigned int elapsed = now - current_t;
  
  karts[0]->steer(steer_angle);
  karts[0]->accel(throttle);
  karts[0]->brake(braking);
  
  while (current_t + top10::physX::World::time_slice < now) {
    world.simulate();
    current_t += top10::physX::World::time_slice;
  }
    
  std::vector<LapRecord>::iterator lap_it = current_lap_records.begin();
  std::vector<top10::racing::KartTimer>::iterator timer_it = timers.begin();
  int kart_i = 0;
  for( ;lap_it != current_lap_records.end(); ++lap_it, ++kart_i, ++timer_it) {
    assert(kart_i <= karts.size());
    assert(timer_it != timers.end());
    
    timer_it->update(elapsed);
    
    if (timer_it->timerRunning() && (now-last_record_update > lap_record_period || timer_it->justFinishedLap())) {
      KartState s = karts[kart_i]->getKartState();
      s.timestamp = now;
      
      lap_it->addState(s);
      last_record_update = now;
      
      if (timer_it->justFinishedLap()) {
        std::cerr<<"Just finished lap"<<std::endl;
        if (best_lap_record.size() == 0 || lap_it->getTime() < best_lap_record.getTime()) {
          std::cerr<<"New best lap "<<lap_it->getTime()<<" < "<<best_lap_record.getTime()<<std::endl;
          best_lap_record.swap(*lap_it);
        }
        else {
          std::cerr<<"Sucky lap"<<lap_it->getTime()<<" >= "<<best_lap_record.getTime()<<std::endl;
        }
        lap_it->clear(now);
      }
    }   
  }
}
