/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_SIMULATION_H
#define TOP10_SIMULATION_H

//C/C++ Stuff
#include <vector>

//My stuff
#include "math/Vertex.hh"
#include "physX/World.hh"
#include "physX/Kart.hh"
#include "util/Debug.hh"
#include "util/RefCount.hh"
#include "racing/KartTimer.hh"
#include "racing/LapRecord.hh"
#include "track/Track.hh"

#include "Controls.hh"
#include "Kart.hh"
#include "Event.hh"

namespace top10 {
  namespace ui_interactive {

    class Simulation: public top10::util::Debug, public top10::util::RefCount
    {
    public:
      Simulation(top10::track::Track* track, int n_karts=1);
      
      /**
	 Advances the simulation
	 \parameter Time (in ms) since the start of the simulation
       */
      void update(unsigned int t);

      /**
	 An event is issued by an user to interact with the world
       */
      void sendEvent(Action, Sint16 value);

      /**
        Change the weather conditions at once.
      */
      inline void setWet(bool b) {
	isWet = b;
	if (!track.isValid()) return;
	if (b) track->setGripFactor(0.85);
	else track->setGripFactor(1.0);
      }

      inline bool getWet() const { return isWet; }

      inline Kart* getKart(int i) {
        if (i<0 || i>= karts.size()) return 0;
        return karts[i];
      }
      
      const Kart* getKart(int i) const {
        if (i<0 || i>= karts.size()) return 0;
        return karts[i];        
      }

      inline int getNKarts() const {
        return karts.size();
      }

      inline const top10::racing::KartTimer* getTimer(int i) const {
        if (i<0 || i>=timers.size()) return 0;
        else return &timers.at(i);
      }
      
      inline const top10::racing::LapRecord& getBestLap() const { return best_lap_record; }
      
    private:
      //! Description of the track
      top10::util::Ref<top10::track::Track> track;
            
      top10::util::RefArray<Kart> karts;
      
      //! Is the track wet?
      bool isWet;

      top10::physX::World world;

      //! Timers for each kart
      std::vector<top10::racing::KartTimer> timers;

      //! Best lap
      top10::racing::LapRecord best_lap_record;

      //! Current laps for each kart
      std::vector<top10::racing::LapRecord> current_lap_records;

      //! The steering angle of the controled kart
      double steer_angle;
      //! The throttle % of the controled kart
      double throttle;
      //! The braking % of the controled kart
      double braking;

      /// Time inside the world
      Uint32 current_t;
      
      //! Last time we updated the lap records
      Uint32 last_record_update;
      
      //! How often we should update the records
      int lap_record_period;
    };
  };
};
#endif
