/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "audio.hh"
#include "Simulation.hh"
#include "util/PathFinder.hh"
#include "util/error.hh"

using std::cerr;
using std::endl;

void top10::ui_interactive::fill_audio(void* _audio_device, Uint8* stream, int len)
{
	AudioDevice* audio_device = reinterpret_cast<AudioDevice*>(_audio_device);
	audio_device->render(stream, len);
}

top10::ui_interactive::AudioDevice::AudioDevice():
	kart(0)
{
	loadSamples();
}

top10::ui_interactive::AudioDevice::AudioDevice(Kart* _kart):
	kart(_kart)
{
	loadSamples();
}

void top10::ui_interactive::AudioDevice::setKart(Kart* _kart)
{
	kart=_kart;
}

void top10::ui_interactive::AudioDevice::loadSamples()
{
  /* Load sound samples */
  SDL_AudioSpec wanted;
  wanted.freq = 22050;
  wanted.format = AUDIO_S16;
  wanted.channels = 1;    /* 1 = mono, 2 = stereo */
  Uint32 len;

  std::string path = top10::util::PathFinder::defaultPathFinder().find("tyre.wav");
  if (path.empty()) throw top10::util::Error("Could not find tyre.wav");

  if (!SDL_LoadWAV(path.c_str(), &wanted, &sdl_tyre_audio, &len)) {
    left_wheel_audio = right_wheel_audio = 0;
    throw top10::util::Error(SDL_GetError());
  }
  else {
    left_wheel_audio = new SoundBuffer<Sint16>((Sint16*)sdl_tyre_audio, len/2);
    right_wheel_audio = new SoundBuffer<Sint16>((Sint16*)sdl_tyre_audio, len/2);
  }

  path = top10::util::PathFinder::defaultPathFinder().find("engine.wav");
  if (path.empty()) throw top10::util::Error("Could not find engine.wav");
  if (!SDL_LoadWAV(path.c_str(), &wanted, &sdl_engine_audio, &len)) {
    engine_audio = 0;
    throw top10::util::Error(SDL_GetError());
  }
  else {
    engine_audio = new SoundBuffer<Sint16>((Sint16*)sdl_engine_audio, len/2);
  }

}

void top10::ui_interactive::AudioDevice::render(Uint8* stream, int len)
{
  /* Compute volume of sound emitted by each sliding tyre */
  double k_left = kart->getLeftLoad() * kart->getLeftFriction();
  k_left /= 1000.0;
  if (k_left>1.0) k_left = 1.0;
  double k_right = kart->getRightLoad() * kart->getRightFriction();
  k_right /= 1000.0;
  if (k_right>1.0) k_right = 1.0;
//  std::cerr<<kart->getLeftLoad()<<" "<<kart->getLeftFriction()<<std::endl;

  // Compute frequency of sliding tyres
  double freq_left = 1.0 + kart->getLeftFriction()/2.0;
  if (left_wheel_audio) left_wheel_audio->setSamplingK(freq_left);
  double freq_right = 1.0 + kart->getRightFriction()/2.0;
  if (right_wheel_audio) right_wheel_audio->setSamplingK(freq_right);

  // Compute the frequency
  /* Compute engine sound frequency.
     k_engine = number of original samples for one new sample */
  double k_engine = 0.01*kart->getRadPerSec() + 1;
  if (engine_audio)
    engine_audio->setSamplingK(k_engine);

  /* Mix sound for each channel */
  for (Sint16* p = (Sint16*)stream; len>=4; p+=2, len-=4) {
    p[0] = p[1] = 0;

    /* Tyres' samples */
    if (left_wheel_audio && k_left > 0.0) {
      Sint16 wheel_sample = left_wheel_audio->readNextSample();
      p[0] = (Sint16)(wheel_sample * k_left);
    }
    if (right_wheel_audio && k_right > 0.0) {
      Sint16 wheel_sample = right_wheel_audio->readNextSample();
      p[1] = (Sint16)(wheel_sample * k_right);
    }

    /* Resample engine sound */
    if (engine_audio) {
      Sint16 engine_sample = engine_audio->readNextSample();
      p[0] += (Sint16)(engine_sample*0.25);
      p[1] += (Sint16)(engine_sample*0.25);
    }
  }
}

top10::ui_interactive::AudioDevice::~AudioDevice()
{
  SDL_LockAudio();
  delete left_wheel_audio;
  left_wheel_audio = 0;
  delete right_wheel_audio;
  right_wheel_audio = 0;
  delete engine_audio;
  engine_audio = 0;

  SDL_FreeWAV(sdl_tyre_audio);
  sdl_tyre_audio = 0;

  SDL_FreeWAV(sdl_engine_audio);
  sdl_engine_audio = 0;
  SDL_UnlockAudio();
};
