/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/
#include "GlobalOptions.hh"
#include "error.hh"
#include "PathFinder.hh"

namespace top10 {
namespace util {

GlobalOptions* GlobalOptions::singleton =0;

void GlobalOptions::load(std::string fname)
{
  top10::util::PathFinder finder = top10::util::PathFinder::defaultPathFinder();
  std::string path = finder.find(fname);
  if (path.empty()) throw Error("Could not find option file "+fname);
  
  if (!document.LoadFile(path)) throw Error(fname+" is not a valid XML file");
  
  // Well-formed XML documents have a single root
  TiXmlElement* xml_node = document.RootElement();
  // Ill-formed XML documents generated by the first version of this code have several children
  if (xml_node == 0 || std::string(xml_node->Value()) != "options") {
    // Fix the poor thing
    TiXmlDocument new_doc(document.Value());
    TiXmlElement new_root("options");
    TiXmlNode* node_it;
    node_it = document.FirstChild();
    while (node_it != 0) {
      new_root.LinkEndChild(node_it->Clone());
      node_it = node_it->NextSibling();
    }
    new_doc.InsertEndChild(new_root);
    document = new_doc;
    xml_node = document.RootElement();
  }
  assert(std::string(xml_node->Value()) == "options");
  xml_node = xml_node->FirstChildElement();

  while (xml_node) {
    std::string opt_name = xml_node->Value();
    
    const char* str_val = xml_node->Attribute("string");
    if (str_val)
      str_options[opt_name] = str_val;
    
    double num_val;
    if (xml_node->Attribute("number")) {
      if (xml_node->QueryDoubleAttribute("number", &num_val) == TIXML_SUCCESS) num_options[opt_name] = num_val;
      else throw Error("Parse error for option "+opt_name+". Could not parse number");
    }
    
    xml_node = xml_node->NextSiblingElement();
  }

}

void GlobalOptions::save(std::string fname)
{
  TiXmlElement* root = document.RootElement();
  if (root == 0) {
    TiXmlElement new_root("options");
    document.InsertEndChild(new_root);
    root = document.RootElement();
  }

  // Update the XML document
  // String values
  for (std::map<std::string, std::string>::const_iterator it = str_options.begin(); it != str_options.end(); ++it) {
    TiXmlElement* xml_node = root->FirstChildElement(it->first);
    if (xml_node) {
      xml_node->SetAttribute("string", it->second);
    }
    else {
      TiXmlElement new_node(it->first);
      new_node.SetAttribute("string", it->second);
      root->InsertEndChild(new_node);
    }
  }
  
  // Numeric values
  for (std::map<std::string, double>::const_iterator it = num_options.begin(); it != num_options.end(); ++it) {
    TiXmlElement* xml_node = root->FirstChildElement(it->first);
    if (xml_node) {
      xml_node->SetDoubleAttribute("number", it->second);
    }
    else {
      TiXmlElement new_node(it->first);
      new_node.SetDoubleAttribute("number", it->second);
      root->InsertEndChild(new_node);
    }
  }

  // Save the xml file
  if (!document.SaveFile(fname)) throw Error("Failed to save "+fname);
}

GlobalOptions::GlobalOptions(): document("options.xml")
{ 
  setNumOption("Control.Steer.Factor", 0.5);
  setNumOption("Control.Aid.ABS.Limit", -50);
  setStrOption("Control.Aid.ABS", "Yes");
  
  setStrOption("Render.Font", "default");
  
  setNumOption("Render.Ghost.Alpha", 0.6);
  
  setStrOption("Render.HUD", "Yes");
  setNumOption("Render.HUD.Alpha", 0.8);
  setNumOption("Render.HUD.CurrentTime.Pos.X", 0);
  setNumOption("Render.HUD.CurrentTime.Pos.Y", 0);
  setNumOption("Render.HUD.CurrentTime.Font.Size", 50);
  setNumOption("Render.HUD.BestTime.Pos.X", 0);
  setNumOption("Render.HUD.BestTime.Pos.Y", 60);
  setNumOption("Render.HUD.BestTime.Font.Size", 50);
  setNumOption("Render.HUD.LastTime.Pos.X", 0);
  setNumOption("Render.HUD.LastTime.Pos.Y", 120);
  setNumOption("Render.HUD.LastTime.Font.Size", 50);
  
  setStrOption("Render.Skybox", "Yes");
  
  setStrOption("Render.Shadow.Method", "Volume");
  setNumOption("Render.Shadow.Alpha", 0.2);
  
  setNumOption("Render.Window.Width", 800.0);
  setNumOption("Render.Window.Height", 600.0);
  setNumOption("Render.Window.Fullscreen", 1.0);
  
  setStrOption("Render.Kart.Outline", "Wireframe");
  setNumOption("Render.Kart.Outline.Thickness", 3.0);
  setNumOption("Render.Kart.Outline.Color.Red", 0.0);
  setNumOption("Render.Kart.Outline.Color.Green", 0.0);
  setNumOption("Render.Kart.Outline.Color.Blue", 0.0);
  
  setNumOption("Track.Octree.Polygons", 5.0);
  setNumOption("Track.Octree.Depth", 7.0);
  
  setStrOption("Sound", "Yes");
  
  setNumOption("Record.Period", 0.1);
}

void GlobalOptions::setStrOption(std::string s1, std::string s2)
{
  str_options[s1] = s2;
}

void GlobalOptions::setNumOption(std::string s1, double x)
{
  num_options[s1] = x;
}

std::string GlobalOptions::getStrOption(std::string s) const
{
  std::map<std::string, std::string>::const_iterator f = str_options.find(s);
  if (f != str_options.end()) return f->second;
  
  throw Error("Unknown option "+s);
}

double GlobalOptions::getNumOption(std::string s) const
{
  std::map<std::string, double>::const_iterator f = num_options.find(s);
  if (f != num_options.end()) return f->second;
  
  throw Error("Unknown option "+s);
}

GlobalOptions* GlobalOptions::getGlobalOptions()
{
  if (!singleton) singleton = new GlobalOptions;
  
  return singleton;
}

}
}
