/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef ROUND_BUFFER_H
#define ROUND_BUFFER_H

#include "error.hh"

namespace top10 {
  namespace util {

    /**
       This class describes a pointer pointing to a round buffer.
       That means that the pointer will loop instead of leaving the buffer.
    */
    template<typename T>
    class RoundBufferPtr
    {
    public:
      DECL_ERROR(BadSize);

    public:
      typedef T CellT;

    public:
      /**
	 Initialize the buffer boundaries.
	 The buffer must have been previously allocated.
	 @param start pointeur to the first element of the buffer
	 @param nb_items number of items contained in the buffer
	 @exception BadSize nb_items can't be zero or negative
      */
      RoundBufferPtr(CellT* start, int nb_items):
	_start(start),
	_end(start + nb_items - 1),
	p(start)
      {
	if (_end < _start) throw BadSize();
      }

      CellT& operator*() {return *p;}
      const CellT& operator*() const {return *p;}

      RoundBufferPtr<CellT>& operator++()
      {
	if (p == _end) p = _start;
	else p++;
	return *this;
      }

    private:
      CellT* const _start;
      CellT* const _end;
      CellT* p;
    };
  };
};

#endif
