/* {{{1 GNU General Public License

sofea - the Stack Operated Finite Element Analysis program
Copyright (C) 2004  Al Danial <al.danial@gmail.com>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
}}}1 */
#include <stdio.h>
#include <stk.h>
#include <main.h>
#include <sparse.h>
#include <malloc.h>   /* malloc, free    */
#include "fea_assemble.h"

int    N_NODES      = 0;  /* number of nodes in the model */

int constrain_nodes()   /* ( hKgg hNode_list --- hKxx ) {{{1 */
/*
 * man entry:  constrain_nodes {{{2
 * ( hKgg hNode_list --- hKxx ) Remove from the sparse matrix Kgg rows and columns associated with the specified finite element nodes to create a constrained matrix.  Kxx will be lower triangular if Kgg is.  The node list array contains renumbered node ID's beginning with 1 (regardless of xbase).
 * category: FEA
 * related: sparse, element_matrix, fea_assemble
 * 2}}}
 */
{
int DEBUG = 0;

    char   *name = "_Kxx";
    int     n_g_rows, n_g_cols, i, j, c, s, c_Kxx, N_ptr_Kxx, S_ptr_Kxx,
            offset, first_row, last_row, ID, max_rows_any_column,
            n_nodes_to_fix, row, n_rows_this_col, n_str_this_col, maxL,
            max_strings,
            ncols_Kxx = 0,
            n_str_Kxx = 0,
            n_nnZ_Kxx = 0,
            cmplx     = 0,  /* Future mod:  allow complex constraint */
           *dof_constrained_above, *row_list, *start_ind, *str_len;
    double *node_id_list, *num_data;
    SparseMatrix Kgg, Kxx;
#define size_T 160
    char    T[size_T+1];

if (DEBUG) gprintf("at top of constrain_nodes\n");

    /* type check the inputs  {{{2 */
    if (!is_sparse(tos-1)) {
        stkerr(" constrain_nodes:  Kgg ", SPARSENOT);
        return 0;
    }
    if (is_complex(tos-1)) {
        stkerr(" constrain_nodes:  Kgg ", "complex Kgg not yet supported");
        return 0;
    }
    Kgg = sparse_overlay(tos-1);
    if (Kgg.H[ROWS] != Kgg.H[COLS]) {
        stkerr(" constrain_nodes:  Kgg ", "is not square");
        return 0;
    }

    if (tos->typ != MAT) {
        stkerr(" constrain_nodes:  Node_list ", MATNOT);
        return 0;
    }
    n_nodes_to_fix = tos->row*tos->col;
    node_id_list   = tos->mat;
    if (!n_nodes_to_fix) {  /* nothing to constrain, just return Kgg */
        drop();
        return 1;
    }

    /*
     * dof_constrained_above keeps track of the number of rows above
     * the current row have already been eliminated.
     *   dof_constrained_above[j] = N   
     * means N rows above row j in Kgg have been constrained and won't
     * appear in the output matrix.  Row j itself is constrained if
     *   dof_constrained_above[j+1]-dof_constrained_above[j]
     * is non-zero. 
     */
    if ((dof_constrained_above = (int *) malloc(
                     (Kgg.H[ROWS] + 1) * sizeof(int))) == NULL) {
        stkerr(" constrain_nodes (dof_constrained_above): ",MEMNOT);
        return 0;
    }
    for (i = 0; i < Kgg.H[ROWS] + 1; i++)
        dof_constrained_above[i] = 0;

    for (i = 0; i < n_nodes_to_fix; i++) {
        ID = (int) node_id_list[i];
        if ((ID < 1) || ID > Kgg.H[ROWS]) {
            /* are the index values in range? */
            stkerr(" constrain_nodes:  Node_list ", MATNOT);
            snprintf(T, size_T, " Node_List[%d]=%d is not in range of 1 to %d (size of [Kgg])",
            i, ID, Kgg.H[ROWS]);
            stkerr(" constrain_nodes: ", T);
            return 0;
        }

        /* make a 0/1 list such that DOF j is free if d_c_a[j+1] = 0
         * and DOF j is constrained if d_c_a[j+1] = 1
         */
        for (j = 1; j <= 6; j++) 
            /* for now, fix all six dof on the given nodes */
            dof_constrained_above[(ID-1)*DOF_PER_NODE + j] = 1; 
    }
    /* integrate the 0/1 list to go from something like
     *    [ 0 1 0 0 0 1 1 1 0 1 ]
     * to
     *    [ 0 1 1 1 1 2 3 4 4 5 ]
     */
    for (j = 1; j < (Kgg.H[ROWS]+1); j++) {
        dof_constrained_above[j] += dof_constrained_above[j-1];
    }

    /* 2}}} */

    /* allocate arrays to hold string info for a sliced & diced column {{{2 */
    if ((row_list = (int *) malloc(Kgg.H[ROWS] * sizeof(int))) == NULL) {
        stkerr(" constrain_nodes (row_list): ",MEMNOT);
        return 0;
    }
    max_strings = (int) (Kgg.H[ROWS] + 1)/2;
if (DEBUG) gprintf("constrain_nodes max_strings=%d\n", max_strings);
    if ((start_ind = (int *) malloc(max_strings * sizeof(int))) == NULL) {
        stkerr(" constrain_nodes (start_ind): ",MEMNOT);
        return 0;
    }
    if ((str_len   = (int *) malloc(max_strings * sizeof(int))) == NULL) {
        stkerr(" constrain_nodes (str_len): ",MEMNOT);
        return 0;
    }
    /* 2}}} */

    /* pass 1:  determine how big Kxx will be {{{2 */
    max_rows_any_column = 0;
    for (c = 0; c < Kgg.H[COLS]; c++) {
        n_rows_this_col   = 0;
if (DEBUG) gprintf("dof_constrained_above[c=%d]=%d\n", c, dof_constrained_above[c]);
        if (dof_constrained_above[c+1] == dof_constrained_above[c]) {
            /* keep this column */
            ++ncols_Kxx;
            for (s = Kgg.S_start[c]; s < Kgg.S_start[c+1]; s++) {
                first_row = Kgg.S[s].start_row;
                last_row  = first_row + Kgg.S[s].len - 1;
                for (row = first_row; row <= last_row; row++) {
                    if (dof_constrained_above[row+1] ==
                        dof_constrained_above[row  ]) {
                        /* keep this row */
                        row_list[n_rows_this_col] = 
                            row - dof_constrained_above[row];
                        ++n_rows_this_col;
                    }
                }
            }
            strings_in_list(n_rows_this_col, /* in  length of list[]         */
                            row_list       , /* in                           */
                           &n_str_this_col , /* out number strings in list[] */
                            start_ind      , /* out index 1st string terms   */
                            str_len        , /* out length each string       */
                           &maxL);           /* unused */
            n_str_Kxx += n_str_this_col;
            n_nnZ_Kxx += n_rows_this_col;
            max_rows_any_column = MAX(max_rows_any_column, n_rows_this_col);
        }
    }
    /* 2}}} */

if (DEBUG) gprintf("Kxx will be %d x %d,  %d strings,  %d non-Zeros\n",
ncols_Kxx, ncols_Kxx, n_str_Kxx, n_nnZ_Kxx);

    /* declare memory for Kxx {{{2 */
    if (!sparse_stk(ncols_Kxx  , /* in (square; use # cols as # rows) */
                    ncols_Kxx  , /* in # cols */
                    n_str_Kxx  , /* in  number of strings       */
                    n_nnZ_Kxx  , /* in  number of nonzero terms */
                    cmplx      , /* in  0=real  1=complex       */
                    1          , /* in is internally indexed    */
                    name       , /* in  */
                   &Kxx)) {      /* out */
        return 0;
    }
    if (is_symm(tos-2))    set_symm(tos);
    if (is_low_tri(tos-2)) set_low_tri(tos);
    if (is_up_tri(tos-2))  set_up_tri(tos);

    /* 2}}} */

    /* pass 2:  populate Kxx {{{2 */
    c_Kxx     = 0;  /* column index for Kxx */
    N_ptr_Kxx = 0;
    S_ptr_Kxx = 0;
    if (cmplx) s = 2;
    else       s = 1;
    if ((num_data  = (double *) malloc(
                    s * max_rows_any_column * sizeof(double))) == NULL) {
        stkerr(" constrain_nodes (num_data): ",MEMNOT);
        return 0;
    }
    for (c = 0; c < Kgg.H[COLS]; c++) {
        n_rows_this_col   = 0;
        if (dof_constrained_above[c+1] == dof_constrained_above[c]) {
            /* keep this column */
            for (s = Kgg.S_start[c]; s < Kgg.S_start[c+1]; s++) {
                first_row = Kgg.S[s].start_row;
                last_row  = first_row + Kgg.S[s].len - 1;
                offset    = 0;
                for (row = first_row; row <= last_row; row++) {
                    if (dof_constrained_above[row+1] ==
                        dof_constrained_above[row  ]) {
                        /* keep this row */
                        row_list[n_rows_this_col] = 
                            row - dof_constrained_above[row];
                        num_data[n_rows_this_col]= Kgg.N[Kgg.S[s].N_idx+offset];
                        ++n_rows_this_col;
                    }
                    ++offset;
                }
            }
if (DEBUG) {
gprintf("constrain_nodes pass 2, column %d\nrow_list:", c);
for (s = 0; s < n_rows_this_col; s++) { gprintf(" %2d", row_list[s]); }
gprintf("\n");
}
            strings_in_list(n_rows_this_col, /* in  length of list[]         */
                            row_list       , /* in                           */
                           &n_str_this_col , /* out number strings in list[] */
                            start_ind      , /* out index 1st string terms   */
                            str_len        , /* out length each string       */
                           &maxL);           /* unused */
            Kxx.row_idx[c_Kxx]   = c;
            Kxx.col_idx[c_Kxx]   = c;
            Kxx.S_start[c_Kxx+1] = Kxx.S_start[c_Kxx] + n_str_this_col ;
            Kxx.N_start[c_Kxx+1] = Kxx.N_start[c_Kxx] + n_rows_this_col;
            for (s = 0; s < n_str_this_col; s++) {
                Kxx.S[S_ptr_Kxx].start_row = row_list[ start_ind[s] ];
                Kxx.S[S_ptr_Kxx].len       = str_len[s];
                Kxx.S[S_ptr_Kxx].N_idx     = N_ptr_Kxx;
                ++S_ptr_Kxx;
            }
            for (s = 0; s < n_rows_this_col; s++) {
                Kxx.N[ N_ptr_Kxx++ ] = num_data[s];
            }
            ++c_Kxx;
        }
    }
    /* 2}}} */

    free(str_len);
    free(start_ind);
    free(row_list);
    free(dof_constrained_above);
    free(num_data);

    lop();  /* dropping hNode_list */
    lop();  /* dropping hKgg       */

    return 1;

} /* 1}}} */
