/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

/* inpo.c  March 1999 

Copyright (c) 1999  D. R. Williamson
*/

#include <errno.h>
#include <ctype.h>
#include <math.h>
#include <stdarg.h>
 
#ifndef __USE_BSD
   #define __USE_BSD
#endif
#include <stdio.h>
extern FILE *fdopen (int __fd, __const char *__modes) __THROW;
#undef __USE_BSD

#ifndef __USE_MISC
   #define __USE_MISC
#endif
#include <stdlib.h>
#undef __USE_MISC

#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>

#ifndef __USE_BSD
   #define __USE_BSD
#endif
#include <dirent.h>
#undef __USE_BSD

#include "main.h"
#include "stk.h"

#include "ctrl.h"
#include "exe.h"
#include "inpo.h"
#include "key.h"
#include "lib.h"
#include "mat.h"
#include "math1.h"
#include "mem.h"
#include "sparse.h"
#include "sys.h"
#include "tag.h"
#include "tex.h"

/*--------------------------------------------------------------------*/

int FROM_ENDIAN,TO_ENDIAN; /* always changing */

/* Parameters for file functions */
double FMAGIC; /* magic number for file handle; set in inpoinit() */
enum _fTYP {ASCI,BINA}; /* types of files */
/* Fortran file errors: */
enum _forterr {FORTENDERR=1,FORTSEEKERR,FORTREADERR};

int LEN32=sizeof(int); /* for records in Fortran files */

/*--------------------------------------------------------------------*/

/* The following functions work together for output of characters to 
   a line:

      cr() - performs carriage return
      gprintf() - displays to sysout (works like printf)
      lf() - performs new-line feed
      nc() - performs new-line feed (no longer does carriage return too)
      out() - puts line character count of GOUT, chout, on the stack
      sp() - spaces to right one blank character
      tout() - pushes text string of displayed line, GOUT, to the stack

   The file for gprintf() can be changed any time using word set_sysout.

   Always using gprintf() (and these functions that use gprintf()) will
   keep everything synchronized.

   For example, while gprinf("...\n",...) can be used to perform a new-
   line feed, it is better to use nc() because the text output buffer
   will be reset to the beginning, and character-out count, chout, ac-
   cessible from high level word out, will be reset.

   See functions in file key.c for examples of character input and
   output. */

/*--------------------------------------------------------------------*/

int blank(char c)
/* Returns 1 if c is a character to be ignored as blank.  Character c 
   is presumed to be at ptr ip and tokenget() is running. */
{
   unsigned char *ip1;

   if(c==',') return 1; /* ignoring comma */

   if(c=='\n') { /* keeping track of newline in iprecl */
      iprecl=ip; 
      return 1;
   }
   if(c<=' ') return 1; /* ignoring blank and below */

   if(c=='\\' || c=='#') { /* advancing to newline char on \ or # */

   /* Advance to newline char only if \ or # does not have a char
      greater than blank (20h) on its left: */
      if(ip>(1+iprecl) && *(ip-1)>' ') return 0; 

      ip1=ip+stradv((char *)ip,"\n")-1;
      ip=MAX(ip1,ip+1); 
      iprecl=ip; 
      return 1;
   }
   return 0;
}

char *blockget(char *str)
/* Gets the block of text, starting from current source buffer pointer, 
   ip, that ends with str; returns NULL pointer if error. */
{
   char *block;
   int len;
   
   len=stradv1((char *)ip,str,32,32);

   if(len>0) { 
      if((block=malloc(len+1))==NULL) {
         stkerr(" blockget: ",MEMNOT);
         ip=ipend; /* jump to end of source buffer */
         return NULL;
      }
      else {
         memcpy(block,ip,len); *(block+len)='\0';
         ip+=len;
         iprecl=ip-strlen(str)-1;
         iprect=iprecl;
         return block;
      }
   }
   else {
      gprintf(" blockget: end of source reached without ending %s",str);
      nc();
      stkerr("","");
      ip=ipend; /* jump to end of source buffer */
      return NULL;
   }
}

int bootfile() /* ( --- qFile | empty) */
/* Returns boot file name on stack if found; otherwise, stack is
   empty and 0 is returned. 

   The file name is syspath/boot.v. */
{
   int f;

   _syspath();
   pushq2("/boot.v",7);
   /* (qPath qFile) */ catpath();
   dup1s();
   /* (qFile) */ fileq();
   popint(&f);
   if(f) return 1;

   if(sysout!=stderr) {
      fprintf(stderr," file not found: %s\n",tos->tex);
   }
   gprintf(" file not found: ");
   /* (qFile) */ dot();
   nc();
   return 0;
}

char braceskip()
/* Skips nested, paired braces with blanks surrounding each brace: 
      { ... { ... } ... } 
   Returns char } if successful, 0 if not. */
{
   int ok=0;

   ip=(unsigned char *)patskip((char *)ip,"{",32,32,"}",32,32,&ok);

   if(ok) return '}';
   gprintf(
      " braceskip: braces unpaired { } or not surrounded by blanks");
   nc();
   stkerr("","");
   return 0;
}

int copy4() /* copy4 (hFile1 qS1 hFile2 qS2 --- ) */
/* Copy a matrix from one put4-style file to another. */
{
   FILE *fp1,*fp2;
   double *A,*hFile1,*hFile2;

   unsigned int bytes,j=0,len;
   unsigned int header[8],NC=0,NC1=0,ND=1,NR,NF,NT,NW;
   int i=0;
   size_t items=1,red=1,rote=1;

   char N[16]="_",S1[9],S2[9];

   int filerr=1,found=0;

   if(tos->typ!=STR) {
      stkerr(" copy4: ",STRNOT);
      return 0;
   }
   if((tos-2)->typ!=STR) {
      stkerr(" copy4: ",STRNOT);
      return 0;
   }
   if((tos-1)->typ!=MAT ||
      !(fp2=filehand((tos-1)->mat,(tos-1)->row*(tos-1)->col))) {
      stkerr(" copy4: ",FILHNDNOT);
      return 0;
   }
   if((tos-3)->typ!=MAT ||
      !(fp1=filehand((tos-3)->mat,(tos-3)->row*(tos-3)->col))) {
      stkerr(" copy4: ",FILHNDNOT);
      return 0;
   }
   if(fp1==fp2) {
      stkerr(" copy4: ","cannot copy file to itself");
      return 0;
   }
   hFile2=(tos-1)->mat;
   hFile1=(tos-3)->mat;

   memset(S2,0,9);
   memcpy(S2,tos->tex,MIN(8,(tos->col)));

   memset(S1,0,9);
   memcpy(S1,(tos-2)->tex,MIN(8,((tos-2)->col)));

   while(!found) {

      /* MSC/Nastran V68 DMAP Dictionary, INPUTT4, p. 159:
         header items (0-6): NC NR NF NT NAME1 NAME2 FORMAT */
      filerr=fortread(fp1,hFile1,header,7*sizeof(int),&len);
      if(filerr) break;

      memcpy(1+N,header+4,8); /* NAME1+NAME2 */
      *(N+strlen1(N,9))='\0';
      NC=*(header);
      NC1=1+NC;

      if(strcmp(1+N,S1)==0 || strlen(S1)==0) {
         found=1;
         filerr=0;
         if(strlen(S1)==0) {
            memcpy(S1,1+N,(len=MIN(8,strlen(1+N))));
            *(S1+len)='\0';
         }
         if(strlen(S2)==0) {
            memcpy(S2,1+N,(len=MIN(8,strlen(1+N))));
            *(S2+len)='\0';
         }
         if(TRACE) {
            gprintf(" copy4: copying %s to %s",S1,S2);
            nc();
         }
      }
      else { /* skipping all columns, including the extra one: */
         j=0;
         while(j<NC1 && !filerr) {
            filerr=fortread(fp1,hFile1,&j,sizeof(int),&len);
         }
      }
   }
   if(found) { /* read the matrix and copy it */
      NR=*(header+1);
      NF=*(header+2); /* form */
      NT=*(header+3); /* type */
      if(NT==4) ND=2;

      if(TRACE) {
         gprintf(" copy4: NR=%d, NC=%d, NF=%d, NT=%d",NR,NC,NF,NT);
         nc();
      }
      if(!matstk(NR*ND,1,"_copy4")) return 0;
      A=tos->mat;

   /* Writing the header: NC NR NF NT NAME1 NAME2 */
      memcpy(header+4,S2,2*sizeof(int));
      fortwrite(fp2,hFile2,header,sizeof(int)*6);

      j=0;
      while(j<NC1 && red && rote) {

         memset(A,0,NR*ND*sizeof(double));

      /* Reading and writing a Fortran-style binary record: */

      /* Beginning byte count */
         red=fread(&bytes,(size_t)LEN32,items,fp1); /* rec len */
         rote=fwrite(&bytes,(size_t)LEN32,items,fp2);

         red+=fread(&j,(size_t)LEN32,items,fp1); /* ICOL */
         rote+=fwrite(&j,(size_t)LEN32,items,fp2);

         red+=fread(&i,(size_t)LEN32,items,fp1); /* IROW */
         rote+=fwrite(&i,(size_t)LEN32,items,fp2);

         red+=fread(&NW,(size_t)LEN32,items,fp1); /* NW */
         rote+=fwrite(&NW,(size_t)LEN32,items,fp2); 

         if(NW) {
            red+=fread((char *)A,(size_t)(NW*LEN32),items,fp1);
            rote+=fwrite((char *)A,(size_t)(NW*LEN32),items,fp2);
         }
         else {
            red+=fread(&len,(size_t)LEN32,items,fp1);
            rote+=fwrite(&len,(size_t)LEN32,items,fp2);
         }
      /* Ending byte count (for Fortran backspace to rec beginning) */
         red+=fread(&bytes,(size_t)LEN32,items,fp1); /* rec len*/
         rote+=fwrite(&bytes,(size_t)LEN32,items,fp2);

         red=(red==6);
         rote=(rote==6);
      }
      fupdate(hFile1,fp1); /* keeping file struct up to date */
      fupdate(hFile2,fp2); /* keeping file struct up to date */
       
      if(TRACE) {
         gprintf(" copy4: columns copied: %d",j-1);
         nc();
      }
      if(red && rote) return(drop() && drop2() && drop2());
      else {
         gprintf(" copy4: error while copying matrix %s",S1);
         nc();
         stkerr("","");
         drop();
         return 0;
      }
   }
   if(strlen(S1)) gprintf(" copy4: matrix %s not found",S1);
   else gprintf(" copy4: end of file");
   nc();
   stkerr("","");
   return 0;
}

int cr() /* cr ( --- ) */
/* Performing carriage return. */
{  
   chout=0;
   gprintf("\r");
   chout=0;
   *(GOUT+chout)='\0';
   return 1;
}

int dirq() /* dir? (qS --- f) */
/* Pushes true flag to stack if S is the name of a subdirectory.  If qS
   is a VOL handle (hT) instead of a string, then f is a column vector 
   of true and false flags.

   Flag f will be false if S is not a subdirectory. */
{
   char p[MAXNAMLEN+1],*T;
   double *A;
   int chars,i=0,len,rows;
   struct stat buf;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" dir?: ",STRORVOLNOT);
      return 0;
   }
   T=tos->tex;
   rows=tos->row;
   chars=tos->col;

   if(!matstk(rows,1,"_dir?")) return 0;
   A=tos->mat;

   for(;i<rows;i++) {
      memcpy(p,T,(len=strlen1(T,MIN(chars,MAXNAMLEN))));
      *(p+len)='\0';

      if(stat(p,&buf)==0 && S_ISDIR(buf.st_mode)) *A=xTRUE;
      else *A=xFALSE;
      A++;
      T+=chars;
   }
   if(rows==1) {
      A--;
      pushint(*A);
      lop();
   }
   return(lop());
}

int dirnames() /* dirnames (qDir --- hT) */
/* Table of names of files and subdirectories in Dir, including ./
   and ../. */
{
   char *name="_dirnames";
   struct dirent **NameList=NULL;
   int i=0,len=0,names;
   char *T;

   if(tos->typ!=STR) {
      stkerr(" dirnames: ",STRNOT);
      return 0;
   }
   unquoted();

   names=scandir(tos->tex,&NameList,NULL,NULL);

   for(;i<names;i++) len=MAX(len,strlen((*(NameList+i))->d_name));

   if(!volstk(MAX(0,names),len,name)) return 0;
   T=tos->tex;
   memset(T,' ',tos->row*tos->col);
 
   for(i=0;i<names;i++) {
      memcpy(T,(*(NameList+i))->d_name,strlen((*(NameList+i))->d_name));
      mallfree((void *)(NameList+i));
      T+=len;
   }
   mallfree((void *)&NameList);
   return(lop());
}

int dirsize() /* dirsize (qDir --- s) */
/* Pushing number of entries in Dir to the stack; 0 if no such Dir;
   number of entries includes files and subdirectories, including ./
   and ../. */
{
   if(tos->typ!=STR) {
      stkerr(" dirsize: ",STRNOT);
      return 0;
   }
   return(
      dirnames() &&
      rows()
   );
}

int dotreprint() /* reprint (qS n --- ) */
/* Reprinting string S on same text line, from its nth character to 
   rightmost. */
{
   unsigned int curs;
   unsigned long n;
   char p[TOKBUF+1];

   if(!popuint(&n)) return 0;

   if(tos->typ!=STR) {
      stkerr(" reprint: ",STRNOT);
      return 0;
   }
   n-=XBASE;
   if(n>(tos->col)-1 || n<0) {
      stkerr(" reprint: ",OUTCOL);
      return 0;
   }
   if(!chout) { /* just displaying it if line is empty */
      memmove(p,tos->tex,(curs=tos->col));
   }
   else {
      reprint(n,0);
      memmove(p,(tos->tex)+n,(curs=tos->col-n));
   }
   *(p+curs)='\0';
   gprintf("%s",p);
   return(drop());
}

int echo1() /* echo (hA --- ) */
/* Echo stack item to display without affecting GOUT.

   Here's the difference between echo and dot:
      [tops@gutter] ready > " this string" echo out .i
       this string 0
      [tops@gutter] ready > " this string" dot out .i 
       this string 12
      [tops@gutter] ready > 
*/
{
   int c0,ret;

   c0=chout;
   ret=dot();

   chout=c0;
   return ret;
}

int emit() /* emit (n or hV --- ) */
/* Emit character to display without affecting GOUT.

   Here's the difference between emit and putch:
      [tops@gutter] ready > 73 72 71 32 emit emit emit emit out .i
       GHI 0
      [tops@gutter] ready > 73 72 71 32 putch putch putch putch out .i
       GHI 4
      [tops@gutter] ready > */
{
   register double *V;
   register int k=0;

   int c0,n,ret=0;

   c0=chout;

   if(tos->typ==NUM) {
      popint(&n);
      gprintf("%c",n);
      ret=1;
   }
   else { 
      if(tos->typ==MAT) {
         V=tos->mat;
         for(;k<(tos->row*tos->col);k++) {
            n=(int)*V;
            gprintf("%c",n);
            V++;
         }
         ret=(drop());
      }
   }
   chout=c0;

   if(!ret) stkerr(" emit: ",STKNOT);
   return(ret);
}

int export(int size, int toendian) /* (hA --- hT) */
/* For size equal to 8, 4, or 2, convert 8-byte fp numbers in A to
   volume T containing size-byte elements of endian type toendian.

   If size=8, elements of T are 8 bytes wide and are fp numbers in
   toendian order; for size=4 or size=2, elements of T are 4 or 2
   bytes wide and contain 4 or 2 byte integers in toendian order.

   While returned T is type VOL, it really contains binary numbers
   of a matrix whose contents were stored by columns.  Type VOL is
   used because it allows element widths of varying size, while
   elements of type MAT (the type of incoming A) are exculsively
   8-byte numbers.

   Matrices like A are stored by columns, that is, adjacent values in
   A (ignoring endmost ones) correspond to rows in the same column.
   But volumes are stored by rows, like the text for which they are
   most often used.  Thus the adjacent values of A columns appear as
   adjacent values in rows of T, and the number of bytes in a row
   of T equals the number of rows in A times size bytes, while the
   number of rows in T equals the number of columns in A.

   T is intended for output to a file; it cannot be used for mathe-
   matical operations. */
{
   register char *Afrom,*Ato,*Aint;
   register double *Amat;
   double Aint8;
   register long *from=NULL,i=0,*to=NULL;
   int cols,k,terms,rows;
   char name[10];

   long d1234[8]={7,6,5,4,3,2,1,0}; /* little endian */
   long d4321[8]={0,1,2,3,4,5,6,7}; /* big endian */
   long d3412[8]={5,4,7,6,1,0,3,2}; /* pdp endian */

   long l1234[4]={3,2,1,0}; /* little endian */
   long l4321[4]={0,1,2,3}; /* big endian */
   long l3412[4]={1,0,3,2}; /* pdp endian */

   long s1234[2]={1,0}; /* little endian */
   long s4321[2]={0,1}; /* big endian */
   long s3412[2]={1,0}; /* pdp endian */

   switch(toendian) {
      case 1234: break;
      case 4321: break;
      case 3412: break;
      default:
         stkerr(" export: ","illegal endian value");
         return 0;
   }
   sprintf(name,"_export%d",size);
   *(name+8)='\0';
   if(!volstk((rows=tos->col),size*(cols=tos->row),name)) return 0;

   Aint=(char *)&Aint8;
   Amat=(tos-1)->mat;

   Afrom=(char *)(tos-1)->mat;
   Ato=(tos->tex);

   terms=rows*cols;
   memset(Ato,0,terms*size);

   k=byteorder(NULL);

   switch(size) {

      case 8:
         switch(k) {
            case 1234: from=d1234; break;
            case 4321: from=d4321; break;
            case 3412: from=d3412; break;
         }
         switch(toendian) {
            case 1234: to=d1234; break;
            case 4321: to=d4321; break;
            case 3412: to=d3412; break;
         }
         for(;i<terms;i++) {
            *(Ato+*(to  ))=*(Afrom+*(from  ));
            *(Ato+*(to+1))=*(Afrom+*(from+1));
            *(Ato+*(to+2))=*(Afrom+*(from+2));
            *(Ato+*(to+3))=*(Afrom+*(from+3));
            *(Ato+*(to+4))=*(Afrom+*(from+4));
            *(Ato+*(to+5))=*(Afrom+*(from+5));
            *(Ato+*(to+6))=*(Afrom+*(from+6));
            *(Ato+*(to+7))=*(Afrom+*(from+7));
            Ato+=8;
            Afrom+=8;
         }
      break;

      case 4:
         switch(k) {
            case 1234: from=l1234; break;
            case 4321: from=l4321; break;
            case 3412: from=l3412; break;
         }
         switch(toendian) {
            case 1234: to=l1234; break;
            case 4321: to=l4321; break;
            case 3412: to=l3412; break;
         }
         for(;i<terms;i++) {
            Aint8=_int4(*Amat);
            *(Ato+*(to  ))=*(Aint+*(from  ));
            *(Ato+*(to+1))=*(Aint+*(from+1));
            *(Ato+*(to+2))=*(Aint+*(from+2));
            *(Ato+*(to+3))=*(Aint+*(from+3));
            Ato+=4;
            Amat++;
         }
      break;

      case 2:
         switch(k) {
            case 1234: from=s1234; break;
            case 4321: from=s4321; break;
            case 3412: from=s3412; break;
         }
         switch(toendian) {
            case 1234: to=s1234; break;
            case 4321: to=s4321; break;
            case 3412: to=s3412; break;
         }
         for(;i<terms;i++) {
            Aint8=_int2(*Amat);
            *(Ato+*(to  ))=*(Aint+*(from  ));
            *(Ato+*(to+1))=*(Aint+*(from+1));
            Ato+=2;
            Amat++;
         }
      break;
   }
   return(lop());
}

int fallow() /* fallow (qFile --- f) */
/* Flag f is true if File is allowed to be opened for writing. */
{
   FILE *fp;
   int f=0;

   if(tos->typ!=STR) {
      stkerr(" fallow: ",STRNOT);
      return 0;
   }
   unquoted(); 

   dup1s();
   fileq();
   popint(&f);

   fp=fopen(tos->tex,"ab+"); /* can be opened for read and write? */

   if(fp) {
      fclose(fp);
      if(!f) remove(tos->tex); /* remove if file didn't exist */
      return(drop() && pushint(xTRUE));
   }
   else 
      return(drop() && pushint(xFALSE));
}
   
int fappend() /* append (hT qS --- ) */
/* Appends volume to file named S; creates S if not found. */
{
   register FILE *fp;
   register char *T;
   register int i=0,chars,rows;
   char *filename;

   if(!unquoted()) return 0;

   if((filename=peekq())==NULL) {
      stkerr(" fappend: ",STRNOT);
      return 0;
   }
   if((tos-1)->typ!=VOL && (tos-1)->typ!=STR) {
      stkerr(" fappend: ",STRORVOLNOT);
      return 0;
   }
   if((fp=fopen(filename,"a+b"))==NULL) {
      gprintf(" fappend: cannot create file %s", filename); nc();
      stkerr("","");
      return 0;
   }
   T=(tos-1)->tex;
   rows=(tos-1)->row;
   chars=(tos-1)->col;
  
   for(;i<rows;i++) {
      fwrite(T,strlen1(T,chars),1,fp);
      putc('\n',fp);
      T+=chars;
   }
   fclose(fp);
   return(drop2());
}

int feed() /* feed ( --- qS or hT or NULL(error)) */
/* Fetches the next text pattern in the source text and pushes it to
   the stack as a string or volume.  If end of source, pushes NULL
   to stack. */
{
   char *token[NBUF],*quote[NBUF];

   if((*(token+onbuf)=tokenget())!=NULL) {

      if(TRACE) {
         gprintf(" feed word: %s",*(token+onbuf)); nc();
      }
      /* From tokenget(), strings and text are in mallocked tokenq. */
      if((*(quote+onbuf)=tokenq)) {
         if(strmatch(*(token+onbuf),"{\"")) {
          
            tokenq=*(quote+onbuf);
            return(pushtex3()); /* volume */
         }
         tokenq=*(quote+onbuf);
         return(pushq3()); /* quote-string */
      }
      /* token string */
      return(pushq2(*(token+onbuf),strlen(*(token+onbuf)))); 
   }
   return(pushint(0)); /* null (error) */
}

int fileatime() /* fileatime (qS --- s) */
/* Access time for file S.  Time s is the number of seconds since 
   00:00:00 GMT, January 1, 1970.  If qS is a VOL handle (hT) instead 
   of a string, then s is a one-column vector.

   If S is not a regular file, then s=0. */
{
   char p[FILENAME_MAX],*T;
   double *A;
   int chars,i=0,len,rows;
   struct stat buf;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" fileatime: ",STRORVOLNOT);
      return 0;
   }
   T=tos->tex;
   rows=tos->row;
   chars=tos->col;

   if(!matstk(rows,1,"_fileatime")) return 0;
   A=tos->mat;

   for(;i<rows;i++) {
      memcpy(p,T,(len=strlen1(T,chars)));
      *(p+len)='\0';

      if(!stat(p,&buf))
         *A=(double)buf.st_atime;
      else *A=0;
      A++;
      T+=chars;
   }
   if(rows==1) {
      A--;
      pushint(*A);
      lop();
   }
   return(lop());
}

int fileclose() /* fclose (hFile --- ) */
/* Close a file and make handle structure invalid. */
{
   FILE *fp;

   if(tos->typ!=MAT ||
      !(fp=filehand(tos->mat,tos->row*tos->col))) {
      stkerr(" fclose: ",FILHNDNOT);
      return 0;
   }
   if(fclose(fp)) { /* error closing file */
      stkerr(" fclose: ",FILERR2);
      pushint(0);
      swap();
      pushint(fMAG+XBASE); /* destroy the magic number anyway */
      poke();
      return 0;
   }
   if(TRACE) {
      dup1s(); 
      named(); /* library name, including local tag */
      gprintf(" fclose: closing handle "); 
      dot(); nc();
   }
   return(
      pushint(0) &&
      swap() &&
      pushint(fMAG+XBASE) && /* destroy the magic number */
      poke()
   );
}

int filectime() /* filectime (qS --- s) */
/* Change status time for file S.  Time s is the number of seconds
   since 00:00:00 GMT, January 1, 1970.  If qS is a VOL handle (hT)
   instead of a string, then s is a one-column vector.

   If S is not a regular file, then s=0. */
{
   char p[FILENAME_MAX],*T;
   double *A;
   int chars,i=0,len,rows;
   struct stat buf;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" filectime: ",STRORVOLNOT);
      return 0;
   }
   T=tos->tex;
   rows=tos->row;
   chars=tos->col;

   if(!matstk(rows,1,"_filectime")) return 0;
   A=tos->mat;

   for(;i<rows;i++) {
      memcpy(p,T,(len=strlen1(T,chars)));
      *(p+len)='\0';

      if(!stat(p,&buf))
         *A=(double)buf.st_ctime;
      else *A=0;
      A++;
      T+=chars;
   }
   if(rows==1) {
      A--;
      pushint(*A);
      lop();
   }
   return(lop());
}

int filefound() /* filefound (qFile --- 0 or qFile1 f) */
/* Searches for file locally, then along usrpath, then along syspath,
   and then along docpath.

   Also tries usr/ and sys/.

   Does not search if qFile contains a path.

   Adds path to File if file is found and pushes true flag; otherwise, 
   just a false flag returns on the stack. */
{
   int f;
   char *usr="usrpath";
   char *doc="docpath";
   char *usr1="usr/";
   char *sys1="sys/";

   if(!unquoted()) return 0;

   dup1s();
   rmpath();

   if((tos->col)!=((tos-1)->col)) {
      /* qFile has a path.  Do not search. */
      drop();
      dup1s();
      fileq(); /* does qfile exist? */
      popint(&f);
      if(f) {
         pushint(xTRUE);
         return 1;
      }
      else {
         drop();
         pushint(xFALSE);
         return 1;
      }
   }
   drop();

   dup1s();
   fileq(); /* does qfile exist locally? */
   popint(&f);
   if(f) {
      pushint(xTRUE);
      return 1;
   }
   pushq2(usr,strlen(usr));
   xmain(0); /* usrpath string on stack */
      over();
      cat();
      dup1s();
      fileq(); /* does usr/qFile exist? */
      popint(&f);
      if(f) {
         lop();
         pushint(xTRUE);
         return 1;
      }
   drop();

   pushq2(SYSPATH,strlen(SYSPATH)); /* syspath string on stk */
      over();
      cat();
      dup1s();
      fileq(); /* does sys/qFile exist? */
      popint(&f);
      if(f) {
         lop();
         pushint(xTRUE);
         return 1;
      }
   drop();

   pushq2(doc,strlen(doc));
   xmain(0); /* docpath string on stack */
      over();
      cat();
      dup1s();
      fileq(); /* does doc/qFile exist? */
      popint(&f);
      if(f) {
         lop();
         pushint(xTRUE);
         return 1;
      }
   drop();

   pushq2(usr1,strlen(usr1)); 
      over();
      cat();
      dup1s();
      fileq(); /* does ../usr/qFile exist? */
      popint(&f);
      if(f) {
         lop();
         pushint(xTRUE);
         return 1;
      }
   drop();
 
   pushq2(sys1,strlen(sys1));
      over();
      cat();
      dup1s();
      fileq(); /* does ../sys/qFile exist? */
      popint(&f);
      if(f) {
         lop();
         pushint(xTRUE);
         return 1;
      }
   drop();

/* No file found */
   drop();
   pushint(xFALSE);
   return 1;
}

int fileget() /* fget (hFile bytes --- hT) */
/* Get bytes from file at current position.  Returned T is a text 
   volume of one row that is bytes long, or less.  If file pointer
   is already at the end of file, T has 1 row and 0 chars. */
{
   FILE *fp;
   double *hFile,dbytes;
   const double maxbytes=4294967295.;
   char *name="_fget",*p;
   unsigned long bytes;
   size_t items=1,red;

   if(!popd(&dbytes)) return 0;
   bytes=(unsigned long)MIN((double)dbytes,(double)maxbytes);

   if(tos->typ!=MAT ||
      !(fp=filehand(tos->mat,tos->row*tos->col))) {
      stkerr(" fget: ",FILHNDNOT);
      return 0;
   }
   hFile=tos->mat;

   bytes=MIN( /* cannot exceed file size less file position: */
      bytes,
      ((unsigned long)*(hFile+fSIZ)-(unsigned long)ftell(fp))
   );

   if(!volstk((int)items,(int)bytes,name)) return 0;
   p=tos->tex;

   if(bytes) {
      red=fread(p,(size_t)bytes,items,fp);

      if(red!=items) {
         stkerr(" fget: ",FILREADERR);
         drop();
         fupdate(hFile,fp); /* keeping file struct up to date */
         return 0;
      }
   }
   fupdate(hFile,fp); /* keeping file struct up to date */

   return(lop());
}

int filegetr() /* fgetr (hFile hR --- hT) */
/* Get records from random access file that are listed in R. 
   Number of records obtained equals the number of rows in R.

   Returned T is a volume with records rows, each of recsize bytes. 

   Offset rechead is assumed before the first valid record of recsize
   begins. */
{
   register FILE *fp;
   double *hFile;
   register double *hR;
   register int i=0,k;
   register char *p;
   char *name="_fgetr";
   size_t items=1,red=1;
   int rechead,recs,recsize;

   if(tos->typ!=MAT && tos->typ!=NUM) {
      stkerr(" fgetr: ",NUMORMATNOT);
      return 0;
   }
   if((tos-1)->typ!=MAT ||
      !(fp=filehand((tos-1)->mat,(tos-1)->row))) {
      stkerr(" fgetr: ",FILHNDNOT);
      return 0;
   }
   hFile=(tos-1)->mat;

   recsize=*(hFile+fREC);
   rechead=*(hFile+fHED);
   if(!recsize>0) {
      stkerr(" fgetr: ",FILRECNOT);
      return 0;
   }
   hand();
   hR=tos->mat;
   recs=tos->row;

   if(!volstk((int)recs,(int)recsize,name)) return 0;
   p=tos->tex;

   while(i<recs && red==items) {
      k=(int)*hR-XBASE;
      hR++;
      if(k<0) red=!items; 
      else {
         fseek(fp,rechead+k*recsize,SEEK_SET); 
         red=fread(p+loclin(i,recsize),(size_t)recsize,items,fp);
         i++;
      }
   }
   fupdate(hFile,fp); /* keeping file struct up to date */

   if(red!=items) {
      stkerr(" fgetr: ",FILREADERR);
      drop();
      return 0;
   }
   return(lop() && lop());
}

FILE *filehand(double *F, int len)
/* Fetch file handle from file struct after verifying magic number. */
{
   FILE *fp=NULL;

   if(len!=fLEN || *(F+fMAG)!=FMAGIC) return NULL;
   memcpy(&fp,(F+fHAN),sizeof(fp));
   return(fp);
}

int fileid() /* filetrue (hFile --- f) */
/* True flag to stack if hFile is truly a handle to open file.  hFile 
   has to be a matrix containing a file structure--see enum _file. */ 
{
   int isafile=xTRUE;

   if(tos->typ!=MAT) isafile=xFALSE; /* must be a column matrix */
   else if(!filehand(tos->mat,tos->row*tos->col)) isafile=xFALSE;

   return(
      drop() &&
      pushint(isafile)
   );
}

char *fileload(char *filename)
/* Loads a text file into memory and returns a pointer to it. */
{
   FILE *fp;
   char *buf;
   long bytes;
   size_t items=1,red;

   if((fp=fopen(filename,"rb"))==NULL) {
      gprintf(" fileload: cannot open file %s", filename); nc();
      stkerr("","");
      return NULL;
   }
   fseek(fp,0,SEEK_END);
   bytes=ftell(fp);

   if(TRACE) {
      gprintf(" fileload: loading file %s of %d bytes", \
         filename,bytes); nc();
   }
   fseek(fp,0,SEEK_SET); /* rewind */

   if((buf=(char *)malloc(2+bytes))==NULL) {
      stkerr(" fileload: ",MEMNOT);
      fclose(fp);
      return NULL;
   }
   *(buf+bytes)='\0';
   if(bytes) {

      red=fread(buf,(size_t)bytes,items,fp);

      if(red!=items) {
         gprintf(\
         " fileload: error reading %d bytes from\n    file %s",\
            bytes,filename);
         nc();
         stkerr("","");
         fclose(fp);
         mallfree((void *)&buf);
         return 0;
      }
      if(*(buf+bytes-1)!='\n') { /* add ending newline ch if missing */
         *(buf+bytes)='\n';
         *(buf+bytes+1)='\0';
      }
   }
   fclose(fp);
   return buf;
}

int fileopen() /* file (qFile qS --- ) */
/* Open a file, put file structure into catalog under name S.  If this 
   is a running word, handle S will go into its local library.  Other-
   wise, it goes into the main library. */
{
   FILE *fp=NULL;
   char *fname,*mode=NULL;
   double *hA;
   int f,fmode=0,ret;
   long pos=0;

   if(tos->typ!=STR || (tos-1)->typ!=STR) {
      stkerr(" file: ",STRSNOT2);
      return 0;
   }
   strchop(); swap(); unquoted(); swap();

   f=caton2(tos->tex);
   if(f) { /* found S in catalog; it cannot be an open file  */

      if(oncat->typ==MATR) { /* is S already a file struct? */

         fp=filehand(oncat->stk->mat,oncat->stk->row);

         if(fp) { /* error: must close file first */
            gprintf(" file: handle %s is open; must close first",\
               tagged(tos->tex,*(cattag+onbuf)));
            nc();
            stkerr("","");
            return 0;
         }
      }
   }
   fname=(tos-1)->tex;

/* Making the file handle MAT: */
   if(!matstk(fLEN,1,"_file")) return 0; /* matrix on stack */
   hA=tos->mat; /* file handle */

/* Booking the MAT for the handle into the catalog, with name S: */
   ret=(
      over() && /* hA qS */ book(TRACE) 
   );
   if(!ret) { 
      stkerr(" file: "," error booking file name into catalog");
      return 0;
   }
/* From this point, the stack is in order and the matrix handle named
   S is in the catalog.  If there is an error below here, in addition 
   to any error message the file handle that has been booked in the 
   catalog under name S will be purged.  Automations can check for 
   this (since they cannot read error messages). */

   switch(NEW) {

      case xTRUE: /* Create new file */
         /* Error if the file can be opened. */
         if((fp=fopen(fname,"rb"))) { 
            stkerr(" file: ",FILOLD);
            fclose(fp);
            fp=NULL;
         }
         else {
            /* Open new file for reading and writing. */
            if(BIN) mode="wb+"; /* create new r/w binary */
            else mode="w+"; /* create new r/w text */

            fp=fopen(fname,mode);
            fmode=1; /* read and write */
         }
      break;

      case xFALSE: /* open existing (old) file */
         /* Can the file be opened for reading and writing? */
         if(BIN) mode="rb+"; 
         else mode="r+"; 

         if((fp=fopen(fname,mode))) 
            fmode=1; /* read and write */

         /* Can the file be opened just for reading? */
         else { 
            if(BIN) mode="rb";
            else mode="r"; 

            if((fp=fopen(fname,mode))==NULL) { /* error if can't open */
               gprintf(" file: old file not found: %s",(tos-1)->tex);
               nc();
               stkerr("","");
               fp=NULL;
            }
            else fmode=0; /* read only */
         }
      break;
   }
   lop(); /* done using filename; drop from stack */

   if(fp==NULL) { /* error; marking the MAT catalog item as purged */
      caton2(tos->tex);
      oncat->stk->row=0; /* row==0 makes the file handle MAT purged */
      drop(); /* S off stack */
      return 0;
   }
   drop(); /* S off stack */

/* Setting the items in the file handle structure: */
   *(hA+fHAN)=0;
   memcpy((hA+fHAN),&fp,sizeof(fp)); /* game storing fp into double */

   if(BIN) *(hA+fTYP)=BINA;
   else *(hA+fTYP)=ASCI;

   if(!NEW) {
      fseek(fp,0,SEEK_END); /* seek to end of file */
      pos=ftell(fp); /* get file size */
      *(hA+fSTA)=0; /* status is old */
   }
   else *(hA+fSTA)=1; /* status is new */

   *(hA+fSIZ)=(double)pos;
   fseek(fp,0,SEEK_SET); /* rewind to beginning of file */

   *(hA+fREC)=0;
   *(hA+fHED)=0;
   *(hA+fPOS)=0;
   *(hA+fMOD)=fmode;
   *(hA+fMAG)=FMAGIC;
   
   return 1;
}

int fileopen1() /* filetemp (qFile qS --- qFile1) */
/* Open a temporary file with a unique name having characters that start
   with File, and put file handle structure into catalog under name S.
   If this is a running word, handle S will go into its local library.
   Otherwise, it goes into the main library. 

   Returns the unique file name that goes with S, which is required for
   deleting. */
{
   FILE *fp=NULL;
   char *fname,*mode=NULL;
   double *hA;
   int f,fd=-1,fmode=0,ret;
   long pos=0;

   if(tos->typ!=STR || (tos-1)->typ!=STR) {
      stkerr(" filetemp: ",STRSNOT2);
      return 0;
   }
   strchop(); 
   swap(); 
   unquoted(); 
   swap();

   f=caton2(tos->tex);
   if(f) { /* found S in catalog; it cannot be an open file  */

      if(oncat->typ==MATR) { /* is S already a file struct? */

         fp=filehand(oncat->stk->mat,oncat->stk->row);

         if(fp) { /* error: must close file first */
            gprintf(" file: handle %s is open; must close first",\
               tagged(tos->tex,*(cattag+onbuf)));
            nc();
            stkerr("","");
            return 0;
         }
      }
   }
   swap();
   pushstr("XXXXXX");
   cat();
   fname=tos->tex; /* file name will be on stack when return */
   swap();

/* Making the file handle MAT: */
   if(!matstk(fLEN,1,"_file")) return 0; /* matrix on stack */
   hA=tos->mat; /* file handle */

/* Booking the MAT for the handle into the catalog, with name S: */
   ret=(
      over() && /* hA qS */ book(TRACE)
   );
   if(!ret) {
      stkerr(" file: "," error booking file name into catalog");
      return 0;
   }
/* From this point, the stack is in order and the matrix handle named
   S is in the catalog.  If there is an error below here, in addition
   to any error message, the file handle that has been booked in the
   catalog under name S will be purged.  Automations can check for
   this (since they cannot read error messages). */

   if((fd=mkstemp(fname))>-1) { /* get file descriptor */
      if(BIN) mode="wb+"; /* create new r/w binary */
      else mode="w+";     /* create new r/w text */
      fp=fdopen(fd,mode);
      fmode=1; /* read and write */
   }
   else fp=NULL;

   if(fp==NULL) { /* error; marking the MAT catalog item as purged */
      caton2(tos->tex);
      oncat->stk->row=0; /* row==0 makes the file handle MAT purged */
      drop(); /* S off stack */
      return 0;
   }
   drop(); /* S off stack */

/* Setting the items in the file handle structure: */
   *(hA+fHAN)=0;
   memcpy((hA+fHAN),&fp,sizeof(fp)); /* game storing fp into double */

   if(BIN) *(hA+fTYP)=BINA;
   else *(hA+fTYP)=ASCI;

   *(hA+fSTA)=1; /* status is new */
   *(hA+fSIZ)=(double)pos;
   *(hA+fREC)=0;
   *(hA+fHED)=0;
   *(hA+fPOS)=0;
   *(hA+fMOD)=fmode;
   *(hA+fMAG)=FMAGIC;

   return 1;
}

int fileput() /* fput (hA hFile --- ) */
/* Put bytes from A into file starting at current file position. */
{
   FILE *fp;
   char *p;
   double *hFile;
   stkitem *A;
   long words;
   size_t items=1,rote=1;

   if(stkdepth()<2) {
      stkerr(" fput: ",NEEDTWO);
      return 0;
   }
   if(tos->typ!=MAT ||
      !(fp=filehand(tos->mat,tos->row))) {
      stkerr(" fput: ",FILHNDNOT);
      return 0;
   }
   if((tos-1)->typ==NUM || (tos-1)->typ==STR) {
      swap(); 
      hand(); /* convert number to 1x1 matrix, or string to volume */
      swap();
   }
   hFile=tos->mat;
   A=tos-1;
   
   if(A->typ==VOL) {
      words=(A->row)*(A->col);
      p=A->tex;
   }
   else {
      if(A->typ==MAT) {
         words=sizeof(double)*(A->row)*(A->col);
         p=(char *)A->mat;
      }
      else {
         stkerr(" fput: ",MATORVOLNOT);
         return 0;
      }
   }
   if(words) rote=fwrite(p,(size_t)words,items,fp);
   fupdate(hFile,fp); /* keeping file struct up to date */

   if(rote!=items) {
      stkerr(" fput: ",FILWROTERR);
      return 0;
   }
   return(drop2());
}

int fileputr() /* fputr (hA hFile hR --- ) */
/* Write elements of A into the records of random access file that are 
   listed in R.

   The number of records written is equal to the number of rows in R.

   If A is a volume, row i of A is written at record R(i).  Bytes
   written are the smallest of volume A row size and record size.

   If A is a matrix, records correspond to matrix colummns of 8-byte
   numbers, and the smaller of matrix A column bytes (rows*8) and 
   recsize bytes are written from column i of A to file record R(i).

   Offset rechead is assumed before the first record of recsize begins.

   An error occurs if number of rows in R exceeds the number of rows 
   in volume A or number of columns in matrix A. */
{
   stkitem *A;
   double *hFile;
   register FILE *fp;
   register char *p;
   register double *hR;
   register int i=0,k;
   size_t items=1,rote=1;
   int rechead,recs,rows;
   unsigned int bytes,chadv,recsize;

   if(tos->typ!=MAT && tos->typ!=NUM) {
      stkerr(" fputr: ",NUMORMATNOT);
      return 0;
   }
   if((tos-1)->typ!=MAT ||
      !(fp=filehand((tos-1)->mat,(tos-1)->row))) {
      stkerr(" fputr: ",FILHNDNOT);
      return 0;
   }
   hFile=(tos-1)->mat;

   recsize=*(hFile+fREC);
   rechead=*(hFile+fHED);
   if(!recsize>0) {
      stkerr(" fputr: ",FILRECNOT);
      return 0;
   }
   hand();
   hR=tos->mat;
   recs=tos->row;

   if((tos-2)->typ==NUM || (tos-2)->typ==STR) {
      rot();
      hand(); /* convert number to 1x1 matrix, or string to volume */
      rev();
   }
   A=(tos-2);

   rows=A->row;

   if(A->typ==VOL) {
      if(recs>rows) {
         stkerr(" fputr: ","number of records exceeds volume rows");
         return 0;
      }
      p=A->tex;
      chadv=A->col;
   }
   else {
      if(A->typ==MAT) {

         if(recs>A->col) {
            stkerr(" fputr: ", \
               "number of records exceeds matrix columns");
            return 0;
         }
         p=(char *)A->mat;
         chadv=sizeof(double)*rows;
      }
      else {
         stkerr(" fputr: ",MATORVOLNOT);
         return 0;
      }
   }
   bytes=MIN(chadv,recsize);

   while(i<recs && rote==items) {
      k=(int)*hR-XBASE;
      hR++;
      if(k<0) rote=!items; 
      else {
         fseek(fp,rechead+k*recsize,SEEK_SET);
         rote=fwrite(p+loclin(i,chadv),(size_t)bytes,items,fp);
         i++;
      }
   }
   fupdate(hFile,fp); /* keeping file struct up to date */

   if(rote!=items) {
      stkerr(" fputr: ",FILWROTERR);
      return 0;
   }
   return(drop2() && drop());
}

int fileq() /* file? (qS --- f) */
/* Pushes true flag to stack if S is the name of a file.  If qS is a
   VOL handle (hT) instead of a string, then f is a column vector of 
   true and false flags. 

   Flag f will be false if S is not a regular file. */
{
   char p[MAXNAMLEN+1],*T;
   double *A;
   int chars,i=0,len,rows;
   struct stat buf;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" file?: ",STRORVOLNOT);
      return 0;
   }
   unquoted();

   T=tos->tex;
   rows=tos->row;
   chars=tos->col;

   if(!matstk(rows,1,"_file?")) return 0;
   A=tos->mat;

   for(;i<rows;i++) {
      memcpy(p,T,(len=strlen1(T,MIN(chars,MAXNAMLEN))));
      *(p+len)='\0';

      if(stat(p,&buf)==0 && S_ISREG(buf.st_mode)) *A=xTRUE;
      else *A=xFALSE;
      A++;
      T+=chars;
   }
   if(rows==1) {
      A--;
      pushint(*A);
      lop();
   }
   return(lop());
}

int fileseek() /* fseek (hFile bytes --- ) */
/* Seek byte offset from beginning of file. */
{
   FILE *fp;
   unsigned long atpos,offset;
   double d;
   int ret;

   if(!popd(&d)) return 0;
   offset=(unsigned long)MAX((double)d,(double)0);

   if(tos->typ!=MAT ||
      !(fp=filehand(tos->mat,tos->row))) {
      stkerr(" fseek: ",FILHNDNOT);
      return 0;
   }
   atpos=ftell(fp); /* remember current position */

   ret=fseek(fp,offset,SEEK_SET); 

   if(ret) {
      stkerr(" fseek: ",FILSEEKNOT);
      fseek(fp,atpos,SEEK_SET); /* if error, put pointer back */
      return 0;
   }
   else
      fupdate(tos->mat,fp); /* keeping file struct up to date */

   return(drop());
}

int filesize() /* filesize (qS --- s) */
/* Size in bytes of file S.  If qS is a VOL handle (hT) containing a 
   file name in each row, then s is a one-column vector. */
{
   char p[FILENAME_MAX],*T;
   double *A;
   int chars,i=0,len,rows;
   struct stat buf;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" filesize: ",STRORVOLNOT);
      return 0;
   }
   T=tos->tex;
   rows=tos->row;
   chars=tos->col;

   if(!matstk(rows,1,"_filesize")) return 0;
   A=tos->mat;

   for(;i<rows;i++) {
      memcpy(p,T,(len=strlen1(T,chars)));
      *(p+len)='\0';

      if(!stat(p,&buf)) *A=(double)buf.st_size;
      else *A=-1;
      A++;
      T+=chars;
   }
   if(rows==1) {
      A--;
      pushint(*A);
      lop();
   }
   return(lop());
}

int filesize1() /* fsize (hFile --- bytes) */
/* File size in bytes for given file handle. */
{
   FILE *fp;
   long fpos,fend;
   double d;

   if(tos->typ!=MAT ||
      !(fp=filehand(tos->mat,tos->row))) {
      stkerr(" fsize: ",FILHNDNOT);
      return 0;
   }
   fpos=ftell(fp);

   fseek(fp,0,SEEK_END);
   fend=ftell(fp);
   d=fend;

   fseek(fp,fpos,SEEK_SET);

   return(
      drop() &&
      pushd(d)
   );
}

int filestep() /* fstep (hFile bytes --- ) */
/* Seek byte offset from current position in file.  Incoming bytes
   is negative to step backward. */
{
   FILE *fp;
   unsigned long atpos,pos;
   int bytes,ret;

   if(!popint(&bytes)) return 0;

   if(tos->typ!=MAT ||
      !(fp=filehand(tos->mat,tos->row))) {
      stkerr(" fstep: ",FILHNDNOT);
      return 0;
   }
   atpos=ftell(fp); /* remember current position */

   if(bytes>=0) ret=fseek(fp,(long)bytes,SEEK_CUR);

   else { /* stepping backward: */
      fseek(fp,0,SEEK_CUR);
      pos=ftell(fp);
      bytes=(unsigned long)pos+(int)bytes;
      if(bytes>=0) ret=fseek(fp,(long)bytes,SEEK_SET);
      else ret=1;
   }

   if(ret) {
      stkerr(" fstep: ",FILSEEKNOT);
      fseek(fp,atpos,SEEK_SET); /* if error, put pointer back */
      return 0;
   }
   else
      fupdate(tos->mat,fp); /* keeping file struct up to date */

   return(drop());
}

int filetime() /* filetime (qS --- s) */
/* Modification time for file S.  Time s is the number of seconds
   since 00:00:00 GMT, January 1, 1970.  If qS is a VOL handle (hT) 
   instead of a string, then s is a one-column vector.  

   If S is not a regular file, then s=0. */
{
   char p[FILENAME_MAX],*T;
   double *A;
   int chars,i=0,len,rows;
   struct stat buf;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" filetime: ",STRORVOLNOT);
      return 0;
   }
   T=tos->tex;
   rows=tos->row;
   chars=tos->col;

   if(!matstk(rows,1,"_filetime")) return 0;
   A=tos->mat;

   for(;i<rows;i++) {
      memcpy(p,T,(len=strlen1(T,chars)));
      *(p+len)='\0';

      if(!stat(p,&buf))
         *A=(double)buf.st_mtime;
      else *A=0;
      A++;
      T+=chars;
   }
   if(rows==1) {
      A--;
      pushint(*A);
      lop();
   }
   return(lop());
}

int fortmap() /* fortmap (hFile endian --- hMap) */
/* This function replaces the high level version of fortmap, which has
   been retired to the appendix of file.v.  
   
   This C version and the high level one have been verified to produce
   identical maps on a large and complex Fortran file of 398,000 records
   and with buffer size, Cols, set to 100, so memory and the running map
   pointer was reset over and over.  This C version is about four times
   faster than the high level version.

   Returned Map has 3 columns.  The number of rows in Map equals the
   number of Fortran records in File.

   The nth row of Map contains properties for record n as follows:
      Column 1 - record size (bytes)
      Column 2 - the first 4 bytes of the record interpreted as int
      Column 3 - the 0-based byte offset to the record

   If the record size in the first column of Map is integer-sized
   (4 bytes), then the value in the second column is its value, if
   it can be taken as an integer.  In such a case, there is no need
   to read the file again to obtain the record's sole value.

   Map is built column-wise as a 3-row matrix, and then transposed into
   a 3-column matrix.

   This function uses fortrecs() to go through the file and find all
   its Fortran records.  It gives fortrecs() its catalog name so fort-
   recs() will call it with each Fortran record it finds.

   When fortrecs() is finding records, this function is running at the
   else{*} branch below.  Finally, when fortrecs() finishes, it returns
   to the place in the if{*} branch where it was called, and this func-
   tion returns fortrec()'s return flag.

   Function fortrecs() cannot use the local stack (most C functions
   don't use it any) because this function uses it to pull and push
   Map while fortrecs() runs. */
{
/* To be reentrant, static is required for these: */
   static double *Map;
   static long fp=0;
   static int Cmax,never=1,nrec=0,nCols=100000;
   static char *name="_fortmap";

   int p=0,ret=0;

   if(never) {
      if(TRACE) {
         gprintf(" fortmap begin");
         nc();
      }
      if(!matstk(3,nCols,name)) return 0;
      Map=tos->mat;
      memset(Map,0,3*nCols*sizeof(double));
      lpush(); /* Map to local stack */
      Cmax=nCols;

      never=0;
      nrec=0;
      fp=0;

      pushq2("fortmap",7); /* pushq this function's catalog name */

      ret=fortrecs();

      if(TRACE) {
         gprintf(" fortmap end: %d records processed",nrec);
         nc();
      }
      /* Mapping is complete.  Pull Map from local stack and remove
      unused columns, then return its transpose. */
      lpull(); /* Map from local stack to stack */

      if(ret) {
         pushint(XBASE);
         pushint(nrec);
         items();
         catch();
         bend();
         pushq2(name,strlen(name));
         naming();
      }
      never=1; /* reset for next time */

      return(ret);
   }
   else {

   /* Function fortrecs() is calling with this stack diagram: (hT --- )
      T is a 1-row Fortran record of length tos->col at tos->tex;
      on return, T must be gone */

      if(tos->row) {
         fp+=sizeof(int); /* advance pseudo file pointer */
         nrec++;
         if(nrec==Cmax) {
            lpull(); /* Map from local stack */
            if(!matstk(3,nCols,name)) return 0;
            memset(tos->mat,0,3*nCols*sizeof(double));

            if(!park()) return 0;
            Map=(tos->mat)+3*(Cmax-1);
            Cmax=Cmax+nCols;
            lpush(); /* Map back to local stack */
         }
         /* Adding items to Map rows 1, 2, and 3: */
         *Map=tos->col; /* Map(1)=record length */
         Map++;

         _import4(1,FROM_ENDIAN,(int *)tos->tex,TO_ENDIAN,&p);
         *Map=p; /* Map(2)=int(record(1)) */
         Map++;

         *Map=fp; /* Map(3)=file byte offset to record */
         Map++;

         fp+=(tos->col+sizeof(int)); /* advance file ptr to next rec */
      }
      return(drop()); /* dropping, returning to fortrec() */
   }
}

int fortput() /* fortput (hA hFile --- ) */
/* Put bytes from A into one record of a Fortran binary file, starting 
   at current file position. */
{
   FILE *fp;
   char *p;
   double *hFile;
   stkitem *A;
   long words;

   if(stkdepth()<2) {
      stkerr(" fortput: ",NEEDTWO);
      return 0;
   }
   if(tos->typ!=MAT ||
      !(fp=filehand(tos->mat,tos->row))) {
      stkerr(" fortput: ",FILHNDNOT);
      return 0;
   }
   if((tos-1)->typ==NUM || (tos-1)->typ==STR) {
      swap();
      hand(); /* convert number to 1x1 matrix, or string to volume */
      swap();
   }
   hFile=tos->mat;
   A=tos-1;

   if(A->typ==VOL) {
      words=(A->row)*(A->col);
      p=A->tex;
   }
   else {
      if(A->typ==MAT) {
         words=sizeof(double)*(A->row)*(A->col);
         p=(char *)A->mat;
      }
      else {
         stkerr(" fortput: ",MATORVOLNOT);
         return 0;
      }
   }
   if(words) {
      if(fortwrite(fp,hFile,(unsigned int *)p,words)) {
         stkerr(" fortput: ",FILWROTERR);
         return 0;
      }
   }
   return(drop2());
}

int fortread(FILE *fp, double *hFile, unsigned int *array,\
   unsigned int lenmax, unsigned int *lenread)
/* Read next record from Fortran binary file; behaves like Fortran
   binary read by always skipping to end of record, no matter how
   many bytes are read.
   Returns 0 if no error. */
{
   size_t items=1,red;
   unsigned long bytes,fsize,len=0;
   long fpos,pos;
   int ret;

   pos=ftell(fp);

   red=fread(&bytes,LEN32,items,fp); /* read bytes len */
   if(red!=items) {
      fseek(fp,pos,SEEK_SET); /* keep pos */
      return(FORTENDERR);
   }
   if(lenmax) {
      red=fread(array,(len=MIN(bytes,(unsigned long)lenmax)),items,fp);
      if(red!=items) {
         fseek(fp,pos,SEEK_SET); /* keep pos */
         return(FORTREADERR);
      }
   }
   ret=fseek(fp,pos+bytes+2*LEN32,SEEK_SET);
   if(ret) {
      fseek(fp,pos,SEEK_SET); /* keep same pos */
      return(FORTSEEKERR);
   }
   fpos=ftell(fp);
   fsize=*(hFile+fSIZ);

   if(fpos<=fsize) {
      fupdate(hFile,fp); /* updating pos */
      *lenread=len;
      return 0; /* no error */
   }
   else { /* fpos is beyond end of file */
      fseek(fp,pos,SEEK_SET); /* keep pos */
      return(FORTENDERR);
   }
}

int fortrecs() /* fortrecs (hFile endian qWord --- ) */
/* Read each record of Fortran binary File, of word type endian, and
   run quoted Word with the Fortran record placed on the stack for it.

   Word has the following stack diagram: Word (hT --- ).  It receives
   the record, T (a 1-row VOL), on the stack, and consumes it.

   Global ints FROM_ENDIAN and TO_ENDIAN are in effect while this
   function runs, for possible use by Word if it is a C function.
   (If Word is high level, then endian can be handled any number
   of ways.)

   When this function finishes, it runs Word a final time with T of 
   zero rows and zero characters.  At this time, it is safe for Word
   to put something on the stack.  (Word should test for zero rows
   even if not using this 'feature.')

   For speed, this function reads large portions of File into memory,
   which is just a VOL on the stack, and then operates on records in
   memory.  On successive loadings, the starting file pointer is ad-
   justed backward to capture entirely the partial record at the end
   of the previous loading (and not yet processed).

   The initial size, fBUF, is increased (doubled) as larger records
   are encountered.  This trace shows fBUF, initially 32768 bytes,
   being doubled as bigger records are seen:

    next word: fortrecs
    fortrecs: buffer bytes doubled to 65536
    fortrecs: buffer bytes doubled to 131072
    fortrecs: buffer bytes doubled to 262144
    fortrecs: 398856 Fortran records processed in file of 21425832 bytes

   Using qWord="drop," fortrecs() takes about 1 second, on a RISC 6000
   workstation, to run through the 398,000 records of the 21.4 Mbyte
   file noted in the trace above. */
{
   FILE *fp;
   double *hFile,pWord;
   char *T,*T0;

   unsigned int delta=0,fBUF=32768,fp1=0,len=0,pos,sizefile;
   int grow=1,growmax=1024,more,nrec=0,Tchars,wrapup=0;
   int p[2],p0[2];

   if(!ptr()) return 0;
   if(!popd(&pWord)) return 0;

   if(!popint(&FROM_ENDIAN)) return 0;
   TO_ENDIAN=byteorder(NULL);

   if(tos->typ!=MAT ||
      !(fp=filehand(tos->mat,tos->row))) {
      stkerr(" fortrecs: ",FILHNDNOT);
      return 0;
   }
   pos=ftell(fp); /* saving initial position */

   fseek(fp,0,SEEK_END);
   sizefile=ftell(fp);

   fseek(fp,0,SEEK_SET); /* rewind */

   hFile=tos->mat;

   while(!wrapup) {
      dup1s(); /* hFile */
      pushint(fBUF*grow);
      fileget(); /* getting another chunk of bytes */

      T0=tos->tex;
      T=T0;
      Tchars=tos->col;

      len=0;
      more=1;

      while(more) {
         if(!wrapup) {
            memcpy(p0,T,2*sizeof(int));

            if(!*p0) {
               stkerr(" fortrecs: ","zero length Fortran record");
               return 0;
            }
            /* *p, the contents of p, is record length in bytes: */
            _import4(1,FROM_ENDIAN,p0,TO_ENDIAN,p);
            delta=*p+2*sizeof(int); /* bytes to next rec */
         }
         if((len+delta)<Tchars || wrapup) {

            /* Dummy-up stkitem parameters for Word: */
            tos->tex=T+sizeof(int); /* start of record */
            tos->col=*p; /* length of record */

            /* Running Word: */
            dup1s(); /* dup because Word drops incoming tos */
            exe1(pWord);

            /* Put back stkitem parameters: */
            tos->tex=T0;
            tos->col=Tchars;

            T+=delta;
            fp1+=delta;
            len+=delta;

            nrec++;
            if(wrapup) more=0;
         }
         else { /* current record is a partial record */
            if(ftell(fp)<sizefile) { 
               if(!len) { /* need bigger buffer */
                  grow+=grow; /* double the buffer size */
                  if(grow>growmax) {
                     stkerr("","");
                     gprintf(" fortrecs: %d byte buffer too small for",\
                        fBUF*grow);
                     gprintf(" Fortran record of %u bytes",\
                        *p+2*sizeof(int));
                     nc();
                     gprintf(" fortrecs: increase fBUF and/or growmax");
                     nc();
                     return 0;
                  }
                  if(TRACE) {
                     gprintf(" fortrecs: buffer bytes doubled to %d",\
                        fBUF*grow);
                     nc();
                  }
               }
               fseek(fp,fp1,SEEK_SET); /* set file ptr for next fget */
               fupdate(hFile,fp); /* updating fpos in handle */
               more=0;
            }
            else wrapup=1; /* at end of file; finish last bytes */
         }
      }
      drop(); /* dropping T */
   }
   if(TRACE) {
      gprintf(" fortrecs: %d Fortran records processed",nrec);
      gprintf(" in file of %d bytes",sizefile);
      nc();
   }
   fseek(fp,pos,SEEK_SET); /* put back the file pointer */
   fupdate(hFile,fp); /* updating fpos in handle */

   return(
      volstk(0,0,"_purged") &&
      exe1(pWord) &&
      drop() /* hFile off stack */
   );
}

int fortwrite(FILE *fp, double *hFile, unsigned int *p,\
   unsigned int bytes)
/* Write next record to Fortran binary file.
   Returns 0 if no error. */
{
   size_t items=1,rote;
   unsigned int len;
   
   if((len=bytes)) {
      rote=fwrite(&len,(size_t)LEN32,items,fp);
      rote+=fwrite(p,(size_t)len,items,fp);
      rote+=fwrite(&len,(size_t)LEN32,items,fp);
      if(rote!=3) {
         stkerr(" fortwrite: ",FILWROTERR);
         return 1;
      }
      fupdate(hFile,fp); /* keeping file struct up to date */
   }
   return 0;
}

int fpos1() /* fpos (hFile --- fpos) */
/* Current byte position of file pointer. */
{
   FILE *fp;
   long fpos;
   double d;

   if(tos->typ!=MAT ||
      !(fp=filehand(tos->mat,tos->row))) {
      stkerr(" fpos: ",FILHNDNOT);
      return 0;
   }
   fpos=ftell(fp);
   d=fpos;

   return(
      drop() &&
      pushd(d)
   );
}

int fsave() /* save (hT qS --- ) */
/* Saving volume to file named S.  Replaces S if it exists. */
{
   register FILE *fp;
   register char *T;
   register int i=0,chars,rows;
   char *filename;

   if(!unquoted()) return 0;

   if((filename=peekq())==NULL) {
      stkerr(" save: ",STRNOT);
      return 0;
   }
   if((tos-1)->typ!=VOL && (tos-1)->typ!=STR) {
      stkerr(" save: ",STRORVOLNOT);
      return 0;
   }
   if((fp=fopen(filename,"wb"))==NULL) {
      gprintf(" save: cannot open file %s", filename); nc();
      stkerr("","");
      return 0;
   }
   T=(tos-1)->tex;
   rows=(tos-1)->row;
   chars=(tos-1)->col;
   
   for(;i<rows;i++) {
      fwrite(T,strlen1(T,chars),1,fp);
      putc('\n',fp);
      T+=chars;
   }
   fclose(fp);
   return(drop2());
}

void fupdate(double *F, FILE *fp)
/* Keeping high level file struct in F up to date.  
   IMPORTANT: This function must be called every time the file pointer
   is repositioned. */
{  
   long fsize,pos;
   
   pos=ftell(fp);
   *(F+fPOS)=(double)pos;
   
   fsize=*(F+fSIZ);
   if(fsize<pos) *(F+fSIZ)=(double)pos;
}

int fxray()  /* (hFile bytes --- hT) */
/* Hex dump of bytes from File, starting at current file pointer. */
{
   FILE *fp;
   double *hFile,dbytes;
   const double maxbytes=4294967295.;
   unsigned long bytes,pos,size;
   int i=0,j=0,k=0,len,rows;

   #define perlin 16
   const int width=80;

   unsigned char c,line[1+perlin];
   char *L,*T;
   char *name="_fxray";
   char *heading="       0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F  0123456789ABCDEF";
   char *format1="%4d  %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X";
   char *format2="  %c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c";

   if(!popd(&dbytes)) return 0;
   bytes=MIN(dbytes,maxbytes);

   if(tos->typ!=MAT ||
      !(fp=filehand((hFile=tos->mat),tos->row))) {
      stkerr(" fxray: ",FILHNDNOT);
      return 0;
   }
   size=*(hFile+fSIZ);
   pos=(long)*(hFile+fPOS);
   bytes=MIN(bytes,size-pos);
   rows=1+(bytes%perlin!=0)+bytes/perlin;
   
   if(!volstk(rows,width,name)) return 0;

   T=tos->tex;
   memset(T,' ',rows*width);
   memcpy(T,heading,strlen(heading));
   memset(line,0,1+perlin);

   if(bytes) {

      while(fread(line,1,(size_t)MIN(perlin,bytes-k),fp)!=0 \
         && k<bytes) {

         L=(char *)mprintf(format1,2*i,*line,*(line+1),*(line+2),\
            *(line+3),*(line+4),*(line+5),*(line+6),*(line+7),\
            *(line+8),*(line+9),*(line+10),*(line+11),*(line+12),\
            *(line+13),*(line+14),*(line+15));

         if(!L) return 0;
         
         len=strlen(L);

         T+=width;
         memcpy(T,L,len);
         mallfree((void *)&L);

         for(j=0;j<perlin;j++) {
            c=*(line+j);
            if(c<32 || c>126) *(line+j)='.';
         }
         L=(char *)mprintf(format2,*line,*(line+1),*(line+2),\
            *(line+3),*(line+4),*(line+5),*(line+6),*(line+7),\
            *(line+8),*(line+9),*(line+10),*(line+11),*(line+12),\
            *(line+13),*(line+14),*(line+15));
         memcpy(T+len,L,strlen(L));
         mallfree((void *)&L);
         memset(line,0,perlin);

         i++;
         k+=perlin;
      }
      fupdate(hFile,fp); /* keeping file struct up to date */
      return(lop() && notrailing());
   }
   else return(lop());
   #undef perlin
}

int get() /* get (hFile qS --- hA) */
/* Get the stack item named S from a file written by put(). */
{
   FILE *fp;
   double *hFile,imag,real;
   char name1[256],name2[256],*p=NULL;
   long bytes=0,fsize,stpos;
   size_t items=1,red;
   int cols,end=0,found=0,len,type,rows;
   long long tag;
   static int second=0,secfsize=0;

   if(tos->typ!=STR) {
      stkerr(" get: ",STRNOT);
      return 0;
   }
   if((tos-1)->typ!=MAT ||
      !(fp=filehand((hFile=(tos-1)->mat),(tos-1)->row))) {
      stkerr(" get: ",FILHNDNOT);
      return 0;
   }
   stpos=ftell(fp); /* remember starting position */

   if(second) fsize=secfsize;
   else fsize=*(hFile+fSIZ);

   strchop();

   memcpy(name1,tos->tex,(len=MIN(255,(tos->col))));
   *(name1+len)='\0';

   while(!found && !end) {
   /* Read header: len, name, type, tag, real, imag, rows, cols */
      red=fread(&len,(size_t)sizeof(int),items,fp);

      if(ftell(fp)>=fsize) len=0;
      if(len==0 || red!=items) end=1;

      else {
         len=MIN((unsigned long)len,256); /* includes 0 termination */
         fread(name2,len,items,fp); 
         fread(&type,(size_t)sizeof(int),items,fp);
         fread(&tag,(size_t)sizeof(long long),items,fp);
         fread(&real,(size_t)sizeof(double),items,fp);
         fread(&imag,(size_t)sizeof(double),items,fp);
         fread(&rows,(size_t)sizeof(int),items,fp);
         fread(&cols,(size_t)sizeof(int),items,fp);

         switch(type) {

            case MAT:
               bytes=rows*cols*sizeof(double);
            break;

            case NUM:
               bytes=0;
            break;

            case STR:
            case VOL:
               bytes=rows*cols;
            break;

            default:
               stkerr(" get: ",STKNOT);
               return 0;
            break;
         }
         if(*(name1)==0 || strcmp(name1,name2)==0) found=1;
         else fseek(fp,bytes,SEEK_CUR);
      }
   }
   if(end) {
      if(second || stpos==0) {
         gprintf(" get: %s: %s",FILNAMNOT,name1);
         nc();
         stkerr("","");
         fseek(fp,0,SEEK_SET); /* rewind */
         fupdate(hFile,fp);
         second=0;
         return 0;
      }
      else { /* rewind and try from the beginning */
         fseek(fp,0,SEEK_SET); /* rewind */
         fupdate(hFile,fp); 
         second=1;
         secfsize=stpos;
         found=get();
         second=0;
         return(found);
      }
   }
   switch(type) {

      case MAT:
         if(!matstk(rows,cols,name2)) {
            fupdate(hFile,fp); 
            return 0;
         }
         p=(char *)tos->mat;
      break;

      case NUM:
         if(!pushdx(real,imag)) {
            fupdate(hFile,fp); 
            return 0;
         }
         p=(char *)&tos->real;
         tos->tok=(char *)memgetn(name2,strlen(name2));
      break;

      case STR:
         if(!strstk(cols,name2)) {
            fupdate(hFile,fp); 
            return 0;
         }
         p=(char *)tos->tex;
      break;

      case VOL:
         if(!volstk(rows,cols,name2)) {
            fupdate(hFile,fp); 
            return 0;
         }
         p=(char *)tos->tex;
      break;
   }
   tos->tag=tag;

   if(bytes) {
      red=fread(p,(size_t)bytes,items,fp);

      if(red!=items) {
         stkerr(" get: ",FILREADERR);
         fupdate(hFile,fp);
         drop();
         return 0;
      }
   }
   fupdate(hFile,fp); /* keeping file struct up to date */

   return(lop() && lop());
}

int get4() /* get4 (hFile qS --- hA) */
/* Get matrix named S from binary file written by put4 or the Nastran
   program output4 module.

   Groups of zeroes at the top and bottom of each column were not writ-
   ten, nor were null columns.

   One extra column (of zero length) follows the matrix, and is vital 
   when the last column is null and never written. */
{
   float *A;
   register float *Aj;
   FILE *fp;
   double *hFile;

   unsigned int bytes,j=0,len;
   unsigned int header[8],NC=0,NC1=0,NR,NF,NT,NW=1;
   int i=0;
   int filerr=1,found=0;

   size_t items=1,red=1;
   char N[16]="_",S[9];

   if(tos->typ!=STR) {
      stkerr(" get4: ",STRNOT);
      return 0;
   }
   if((tos-1)->typ!=MAT ||
      !(fp=filehand((tos-1)->mat,(tos-1)->row*(tos-1)->col))) {
      stkerr(" get4: ",FILHNDNOT);
      return 0;
   }
   strchop();

   memset(S,0,9);
   memcpy(S,tos->tex,MIN(8,(tos->col))); 

   hFile=(tos-1)->mat;

   while(!found) {

   /* MSC/Nastran V68 DMAP Dictionary, INPUTT4, p. 159:
      header items (0-6): NC NR NF NT NAME1 NAME2 FORMAT */
      filerr=fortread(fp,hFile,header,7*sizeof(int),&len);
      if(filerr) break;

      memcpy(1+N,header+4,8); /* NAME1+NAME2 */
      *(N+strlen1(N,9))='\0';
      NC=*(header);
      NC1=1+NC;

      if(strcmp(1+N,S)==0 || strlen(S)==0) {
         found=1;
         filerr=0;
         if(strlen(S)==0) {
            memcpy(S,1+N,(len=MIN(8,strlen(1+N))));
            *(S+len)='\0';
         }
      }
      else { /* skipping all columns, including the extra one: */
         j=0;
         while(j<NC1 && !filerr) {
            filerr=fortread(fp,hFile,&j,sizeof(int),&len);
         }
      }
   }
   if(found) { /* read the matrix */
      NR=*(header+1);
      NF=*(header+2); /* form */
      NT=*(header+3); /* type */
      if(!(NF==0 || NF==1 || NF==2 || NF==6)) {
         gprintf(" get4: matrix form is %d; must be 0, 1, 2, or 6",NF);
         nc();
         stkerr("","");
         return 0;
      }
      if(!(NT==1 || NT==2 || NT==4)) {
         gprintf(" get4: matrix type is %d; must be 1 or 2 or 4",NT);
         nc();
         stkerr("","");
         return 0;
      }
      if(TRACE) {
         gprintf(" get4: NR=%d, NC=%d, NF=%d, NT=%d",NR,NC,NF,NT);
         nc();
      }
      if(NT==4) NW=2;

      if(!matstk(NR*NW,NC,N)) {
         gprintf(" get4: cannot allocate %d by %d",NR*NW,NC);
         nc();
         stkerr("","");
         return 0;
      }
      if(NT==4) set_complex(tos);
      A=(float *)tos->mat;
      memset(A,0,NC*NR*NW*sizeof(double));

      j=0;
      while(j<NC1 && red) {

      /* Reading a Fortran-style binary record: */
         red=fread(&bytes,(size_t)LEN32,items,fp); /* rec len */

         red+=fread(&j,(size_t)LEN32,items,fp); /* ICOL */

         if(j<NC1) {
            red+=fread(&i,(size_t)LEN32,items,fp); /* IROW */
            red+=fread(&NW,(size_t)LEN32,items,fp); /* NW */

            Aj=A+(locvec((j-1),NR)+i-1)*NT; /* indices are 1 based */

            red+=fread((char *)Aj,(size_t)(NW*LEN32),items,fp); 

            red+=fread(&bytes,(size_t)LEN32,items,fp); 
         }
         else { /* bump fp past the extra column: */
            fseek(fp,(long)bytes,SEEK_CUR);
            red+=4;
         }
         fupdate(hFile,fp); /* keeping file struct up to date */
         red=(red==6);
      }
      if(NT==1) stretch((float *)A,(double *)A,NR*NC);

      if(TRACE) {
         gprintf(" get4: columns read: %d",j-1);
         nc();
      }
      if(red) return(lop() && lop());
      else {
         gprintf(" get4: error while reading matrix %s",S);
         nc();
         stkerr("","");
         drop();
         return 0;
      }
   }
   if(strlen(S)) gprintf(" get4: matrix %s not found",S);
   else gprintf(" get4: end of file");
   nc();
   stkerr("","");
   return 0;
}

int get4c() /* get4c (hFile qS hC --- hA) */
/* Get columns listed in C from matrix named S from binary file written
   by put4 or the Nastran program output4 module.

   Groups of zeroes at the top and bottom of each column were not writ-
   ten, nor were null columns.

   One extra column (of zero length) follows the matrix, and is vital 
   when the last column is null and never written. 

   Special behavior: After columns C have been collected, the file
   pointer is positioned back to the beginning of matrix S.  Because
   it never goes to the end of the matrix, this word can be used to
   read and reread a growing set of columns while a matrix is being 
   written, perhaps by another program. */
{
   FILE *fp;

   float *A;
   register float *Ak;
   double *C,*hFile;

   unsigned int bytes,len;
   unsigned int Crows,header[8],NC=0,NC1=0,NR,NF,NT,NW=1;
   unsigned int j=0,jpos,matstart,pos=0;
   unsigned int i=0,jlastpos,k=0;
   int jcol=-1,jlast=-1;
   int filerr=1,found=0;

   size_t items=1,red=1;
   char N[16]="_",S[9];

   if(tos->typ!=NUM && tos->typ!=MAT) {
      stkerr(" get4c: ",NUMORMATNOT);
      return 0;
   }
   if((tos-1)->typ!=STR) {
      stkerr(" get4c: ",STRNOT);
      return 0;
   }
   if((tos-2)->typ!=MAT ||
      !(fp=filehand((tos-2)->mat,(tos-2)->row*(tos-2)->col))) {
      stkerr(" get4c: ",FILHNDNOT);
      return 0;
   }
   swap();
   strchop();
   swap();

   hand();
   C=tos->mat;
   Crows=tos->row*tos->col;

   memset(S,0,9);
   memcpy(S,(tos-1)->tex,MIN(8,((tos-1)->col)));

   hFile=(tos-2)->mat;

   while(!found) {
      pos=ftell(fp);

   /* MSC/Nastran V68 DMAP Dictionary, INPUTT4, p. 159:
      header items (0-6): NC NR NF NT NAME1 NAME2 FORMAT */
      filerr=fortread(fp,hFile,header,7*sizeof(int),&len);
      if(filerr) break;
      memcpy(1+N,header+4,8); /* NAME1+NAME2 */
      *(N+strlen1(N,9))='\0';
      NC=*(header);
      NC1=1+NC;

      if(strcmp(1+N,S)==0 || strlen(S)==0) {
         found=1;
         filerr=0;
         if(strlen(S)==0) {
            memcpy(S,1+N,(len=MIN(8,strlen(1+N))));
            *(S+len)='\0';
         }
      }
      else { /* skipping all columns, including the extra one: */
         j=0;
         while(j<NC1 && !filerr) {
            filerr=fortread(fp,hFile,&j,sizeof(int),&len);
         }
      }
   }
   if(found) { /* read the matrix columns listed in C */
      matstart=pos;

      pos=ftell(fp); /* remember where found */
      NR=*(header+1);
      NF=*(header+2); /* form */
      NT=*(header+3); /* type */
      if(!(NF==0 || NF==1 || NF==2 || NF==6)) {
         gprintf(" get4c: matrix form is %d; must be 0, 1, 2, or 6",NF);
         nc();
         stkerr("","");
         return 0;
      }
      if(!(NT==1 || NT==2 || NT==4)) {
         gprintf(" get4c: matrix type is %d; must be 1 or 2 or 4",NT);
         nc();
         stkerr("","");
         return 0;
      }
      if(TRACE) {
         gprintf(" get4c: NR=%d, NC=%d, NF=%d, NT=%d",NR,NC,NF,NT);
         nc();
      }
      if(NT==4) NW=2;

      if(!matstk(NR*NW,Crows,N)) {
         gprintf(" get4c: cannot allocate %d by %d",NR*NW,Crows);
         nc();
         stkerr("","");
         return 0;
      }
      if(NT==4) set_complex(tos);

      A=(float *)tos->mat;
      memset(A,0,NR*NW*Crows*sizeof(double));

      k=0;
      while(k<Crows && red) {

         jcol=*(C+k)-XBASE;
         if(jcol<0 || jcol>NC-1) {
            gprintf(" get4c: column number %g is out of range",*(C+k));
            nc();
            stkerr("","");
            drop(); /* A off the stack */
            return 0;
         }
         if(jcol<=jlast) {
            fseek(fp,pos,SEEK_SET);
            jlast=-1;
         }
         j=-1;
         while(j!=jcol && red) {
            jpos=ftell(fp);
            jlastpos=jlast;
            filerr=fortread(fp,hFile,(unsigned int *)&jlast,\
               (unsigned int)sizeof(int),(unsigned int*)&len);
            red=(filerr==0);
            jlast--; /* jlast from Nastran is 1 based */
 
            if(jlast>=jcol && red) {
               fseek(fp,jpos,SEEK_SET); /* backspace */
               if(jlast==jcol) {

               /* Reading a Fortran-style binary record */
                  red=fread((char *)&bytes,(size_t)LEN32,items,fp);
 
                  red+=fread(&j,(size_t)LEN32,items,fp);
                  red+=fread(&i,(size_t)LEN32,items,fp); 
                  red+=fread(&NW,(size_t)LEN32,items,fp);

                  Ak=A+(locvec(k,NR)+i-1)*NT; /* indices are 1 based */

                  red+=fread((char *)Ak,(size_t)(NW*LEN32),items,fp);

                  red+=fread(&bytes,(size_t)LEN32,items,fp);
               }
               else { /* the column is null */
                  red=6;
                  jlast=jlastpos; /* to value before fortread */
               }
               red=(red==6);
               j=jcol;
            }
         }
         k++;
      }
      fseek(fp,matstart,SEEK_SET); /* position back to this mat */

      if(NT==1) stretch((float *)A,(double *)A,NR*Crows);

      if(TRACE) {
         gprintf(" get4c: columns read: %d",k);
         nc();
      }
      fupdate(hFile,fp); /* keeping file struct up to date */

      if(red) return(lop() && lop() && lop());
      else {
         gprintf(" get4c: error while reading matrix %s",S);
         nc();
         stkerr("","");
         drop();
         return 0;
      }
   }
   if(strlen(S)) gprintf(" get4c: matrix %s not found",S);
   else gprintf(" get4c: end of file");
   nc();
   stkerr("","");
   return 0;
}

int getn() /* getn (n hFile --- hA1 hA2 ... hAn) */
/* Get n stack items from a file written by put().  Items appear on 
   the stack in the order written to file: first on file is first on
   stack, last on file is on top of stack. */
{
   FILE *fp;
   double dn,*hFile,imag,real;
   char name[256],*p=NULL;
   unsigned long bytes=0,fsize,n;
   size_t items=1,red;
   int cols,found=0,len,type,rows;
   long long tag;

   if(tos->typ!=MAT ||
      !(fp=filehand((hFile=tos->mat),tos->row))) {
      stkerr(" getn: ",FILHNDNOT);
      return 0;
   }
   fsize=*(hFile+fSIZ);
   drop(); /* file handle off stack */

   if(!popd(&dn)) return 0;
   dn=MAX(0,dn);
   n=(unsigned long)MIN((double)dn,(unsigned long)(DEPSTK-stkdepth()));

   while(found<n) {
   /* Read header: len, name, type, tag, real, imag, rows, cols */
      red=fread(&len,(size_t)sizeof(int),items,fp);

      if(ftell(fp)>=fsize) break;

      if(len==0 || red!=items) break;
      else {
         len=MIN((unsigned long)len,256);
         fread(name,len,items,fp); /* reading terminator too */
         fread(&type,(size_t)sizeof(int),items,fp);
         fread(&tag,(size_t)sizeof(long long),items,fp);
         fread(&real,(size_t)sizeof(double),items,fp);
         fread(&imag,(size_t)sizeof(double),items,fp);
         fread(&rows,(size_t)sizeof(int),items,fp);
         fread(&cols,(size_t)sizeof(int),items,fp);

         switch(type) {

            case MAT:
               bytes=rows*cols*sizeof(double);
            break;

            case NUM:
               bytes=0;
            break;

            case STR:
            case VOL:
               bytes=rows*cols;
            break;

            default:
               stkerr(" getn: ",STKNOT);
               fupdate(hFile,fp); 
               return 0;
            break;
         }
      }
      switch(type) {

         case MAT:
            if(!matstk(rows,cols,name)) {
               fupdate(hFile,fp); 
               return 0;
            }
            p=(char *)tos->mat;
         break;

         case NUM:
            if(!pushdx(real,imag)) {
               fupdate(hFile,fp); 
               return 0;
            }
            p=(char *)&tos->real;
            tos->tok=(char *)memgetn(name,strlen(name));
         break;

         case STR:
            if(!strstk(cols,name)) {
               fupdate(hFile,fp); 
               return 0;
            }
            p=(char *)tos->tex;
         break;

         case VOL:
            if(!volstk(rows,cols,name)) {
               fupdate(hFile,fp); 
               return 0;
            }
            p=(char *)tos->tex;
         break;
      }
      tos->tag=tag;

      if(bytes) {
         red=fread(p,(size_t)bytes,items,fp);

         if(red!=items) {
            stkerr(" getn: ",FILREADERR);
            fupdate(hFile,fp);
            drop();
            return 0;
         }
      }
      found++;
   }
   fupdate(hFile,fp); /* keeping file struct up to date */
   return 1;
}

int gprintf(char *format, ...)
/* Displays text like printf() and writes to memory array GOUT, so
   GOUT matches the line showing on the screen. 

   WARNING: after each use, args is in an indefinite state and needs
   to be reinitialized again with va_start(). */
{
   int c0,len;
   va_list args;

   va_start(args,format);
      c0=chout;
      chout+=vfprintf(sysout,format,args);
   va_end(args);

/* Flushing sysout is required for a proper display when recovering 
   from EXIT or STOP calls made by functions imported to this program.  

   It is preferable to control EXIT or STOP calls in this program,
   rather than changing functions in imported packages.

   Examples of packages that exit on errors are: 

      BLAS Fortran functions in LAPACK call which xerbla.f to write an 
      error message and call STOP 

      A watch dog monitor in VGETTY that exits on failure to connect on
      the phone

   Attempts to stop the program are captured in the program's exit() 
   function, and interactive processing continues.  But the interactive
   display is affected (as when recovering from BLAS xerbla.f) and char-
   acters typed do not appear until after the NL key is hit.

   Flushing the output stream every time gprintf() is run clears all 
   this up.
*/
   fflush(sysout);

   if(GOMAX/2>chout) {
/*
      The following, which caused the program to hang or seg fault,
      now do not because of added cr in word console as the program
      prompt is displayed (key.v).

         9 36 random mtext . [Enter] [Enter] 
         1 36 random mtext . [Enter] [Enter]
         500 1 DO I .i LOOP  [Enter] [Enter]

      Function accept_keys() now always performs cr().  The cr added 
      clears chout after the first [Enter] key above when the prompt 
      is displayed and before the second [Enter].

      This problem only occurred during interactive keyboarding. 

      May 2006: 

      Using cr() to fix the problem described above has the side effect
      of producing 0D 00 in tty output that drives a terminal window. 
      Normally, 0D 0A come from a tty for CR LF to return the cursor to
      column 1 and go the the next line.  A single CR from a tty gives 
      not just 0D but also 00.

      0D 00 from tty occurs in full screen editing programs, and seems 
      to indicate "raw" character mode, or one character at a time; 
      0D 0A corresponds to "cooked" mode, or one line at a time.
 
      It is not desirable to have this program appear as needing to be
      in raw character mode, so the cr() fix described above has been
      changed.  Since setting chout=0 is really what fixes the problem,
      function accept_keys() now simply sets chout=0 instead of running
      cr() to do it.

      It was verified that the cases above work fine when chout=0 is in
      accept_keys, and that they hang when chout=0 is absent.
*/
      va_start(args,format);
      /* vsnprintf() does not work here: 
               len=vsnprintf(GOUT+c0,GOMAX,format,args);
         It returns len that is not consistent with len from vfprintf()
         and vsprintf(), and the interpretive keyboard is affected.  
         Example: run 1000 1 DO I .i LOOP, then recall its history
         line (Esc-k) and try traversing it. */

         len=vsprintf(GOUT+c0,format,args);
      va_end(args);

      *(GOUT+c0+len)='\0';
      rembks(GOUT);
      chout=strlen(GOUT);
   }
   return chout;
}

int import(int size, int fromendian) /* (hT --- hA) */
/* For size equal to 8, 4, or 2, convert size-byte numbers of endian
   type fromendian in T into 8-byte floating point numbers of the ma-
   chine's endian type, and put them into matrix A.

   If size=8, elements of T are 8 bytes wide and are fp numbers in
   fromendian order; for size=4 or size=2, elements of T are 4 or 2
   bytes wide and contain 4 or 2 byte integers in fromendian order.
   If size=-4, imports 4-byte floating point numbers.

   The number of columns in matrix A equals the number of rows in
   volume T, and the number of rows in A is the bytes per row of T
   divided by size.  This transposition of rows and columns is be-
   cause volumes, like T, store adjacent elements in a row while
   matrices, like A, store adjacent elements in a column--also see
   explanation in export(). */
{
   register char *Afrom,*Ato,*Aint;
   register double *Amat;
   int Aint4;
   short Aint2;
   register long *from=NULL,i=0,*to=NULL;
   int cols,k,opt=1,rows,terms;
   char name[10];
   union {
      char i[4];
      float f[4];
   } Aflo;

   long d1234[8]={7,6,5,4,3,2,1,0}; /* little endian */
   long d4321[8]={0,1,2,3,4,5,6,7}; /* big endian */
   long d3412[8]={5,4,7,6,1,0,3,2}; /* pdp endian */

   long l1234[4]={3,2,1,0}; /* little endian */
   long l4321[4]={0,1,2,3}; /* big endian */
   long l3412[4]={1,0,3,2}; /* pdp endian */

   long s1234[2]={1,0}; /* little endian */
   long s4321[2]={0,1}; /* big endian */
   long s3412[2]={1,0}; /* pdp endian */

   switch(fromendian) {
      case 1234: break;
      case 4321: break;
      case 3412: break;
      default:
         stkerr(" import: ","illegal endian value");
         return 0;
   }
   if(size<0) {
      opt=-1;
      size=ABS(size);
      sprintf(name,"_import%df",size);
      *(name+9)='\0';
   } 
   else {
      sprintf(name,"_import%d",size);
      *(name+8)='\0';
   }
   cols=tos->row;
   rows=(tos->col)/size;

   if(!matstk(rows,cols,name)) return 0;

   if(!rows) return(lop());

   terms=rows*cols;

   Afrom=(tos-1)->tex;
   Ato=(char *)tos->mat;
   Amat=tos->mat;

   k=byteorder(NULL);

   switch(size*opt) {

      case 8:
         switch(fromendian) {
            case 1234: from=d1234; break;
            case 4321: from=d4321; break;
            case 3412: from=d3412; break;
         }
         switch(k) {
            case 1234: to=d1234; break;
            case 4321: to=d4321; break;
            case 3412: to=d3412; break;
         }
         for(;i<terms;i++) {
            *(Ato+*(to  ))=*(Afrom+*(from  ));
            *(Ato+*(to+1))=*(Afrom+*(from+1));
            *(Ato+*(to+2))=*(Afrom+*(from+2));
            *(Ato+*(to+3))=*(Afrom+*(from+3));
            *(Ato+*(to+4))=*(Afrom+*(from+4));
            *(Ato+*(to+5))=*(Afrom+*(from+5));
            *(Ato+*(to+6))=*(Afrom+*(from+6));
            *(Ato+*(to+7))=*(Afrom+*(from+7));
            Afrom+=8;
            Ato+=8;
         }
      break;

      case 4:
         switch(fromendian) {
            case 1234: from=l1234; break;
            case 4321: from=l4321; break;
            case 3412: from=l3412; break;
         }
         switch(k) {
            case 1234: to=l1234; break;
            case 4321: to=l4321; break;
            case 3412: to=l3412; break;
         }
         Aint=(char *)&Aint4;
         for(;i<terms;i++) {
            *(Aint+*(to  ))=*(Afrom+*(from  ));
            *(Aint+*(to+1))=*(Afrom+*(from+1));
            *(Aint+*(to+2))=*(Afrom+*(from+2));
            *(Aint+*(to+3))=*(Afrom+*(from+3));
            *Amat=_real4(Aint4);
            Afrom+=4;
            Amat++;
         }
      break;

      case -4: /* four-byte floating point */
         switch(fromendian) {
            case 1234: from=l1234; break;
            case 4321: from=l4321; break;
            case 3412: from=l3412; break;
         }
         switch(k) {
            case 1234: to=l1234; break;
            case 4321: to=l4321; break;
            case 3412: to=l3412; break;
         }
         for(;i<terms;i++) {
            *(Aflo.i+*(to  ))=*(Afrom+*(from  ));
            *(Aflo.i+*(to+1))=*(Afrom+*(from+1));
            *(Aflo.i+*(to+2))=*(Afrom+*(from+2));
            *(Aflo.i+*(to+3))=*(Afrom+*(from+3));
            *Amat=(double)*Aflo.f;
            Afrom+=4;
            Amat++;
         }
      break;

      case 2:
         switch(fromendian) {
            case 1234: from=s1234; break;
            case 4321: from=s4321; break;
            case 3412: from=s3412; break;
         }
         switch(k) {
            case 1234: to=s1234; break;
            case 4321: to=s4321; break;
            case 3412: to=s3412; break;
         }
         Aint=(char *)&Aint2;
         for(;i<terms;i++) {
            *(Aint+*(to  ))=*(Afrom+*(from  ));
            *(Aint+*(to+1))=*(Afrom+*(from+1));
            *Amat=_real2(Aint2);
            Afrom+=2;
            Amat++;
         }
      break;
   }
   return(lop());
}

void _import4(int terms, int fromend, int *Afrom, int toend, int *Ato)
/* Transform 4-byte ints in Afrom of endian type fromend into 4-byte
   ints of endian type toend. */
{
   register int *from=NULL,i=0,*to=NULL;

   int l1234[4]={3,2,1,0}; /* little endian */
   int l4321[4]={0,1,2,3}; /* big endian */
   int l3412[4]={1,0,3,2}; /* pdp endian */

   switch(fromend) {
      case 1234: from=l1234; break;
      case 4321: from=l4321; break;
      case 3412: from=l3412; break;
   }
   switch(toend) {
      case 1234: to=l1234; break;
      case 4321: to=l4321; break;
      case 3412: to=l3412; break;
   }
   for(;i<terms;i++) {
      *((char *)Ato+*(to  ))=*((char *)Afrom+*(from  ));
      *((char *)Ato+*(to+1))=*((char *)Afrom+*(from+1));
      *((char *)Ato+*(to+2))=*((char *)Afrom+*(from+2));
      *((char *)Ato+*(to+3))=*((char *)Afrom+*(from+3));
      Afrom++;
      Ato++;
   }
}

void inpoinit()
{
   char* magic="file    ";

   memcpy(&FMAGIC,magic,sizeof(double));

   if(TRACE) {
      gprintf(" initializing input/output"); nc();
   }
   GOUT=(char *)memgetc(1,GOMAX);
   *GOUT='\0';

   /* used in key() for multitasker handoff */
   GOUT1=(char *)memgetc(1,GOMAX);

   chout=0;
   tokenq=NULL;
}

void insbks(char *str, int offset)
/* Inserts backspace character into null terminated str at offset+1,
   so the character at offset will be deleted.
   Assumes str can hold the additional byte and does not check that
   offset is less than str length. */
{
   register int i;

   i=strlen(str)+1;

   for(;i>offset;i--) {
      *(str+i)=*(str+i-1);
   }
   *(str+i+1)=8;
}

int lf() /* lf ( --- ) */
/* Line feed. */
{  
   chout=0;
   gprintf("\n");
   chout=0;
   *(GOUT+chout)='\0';
   return 1;
}

char *mprintf(char *format, ...)
/* Writing formatted text to memory and returning the pointer to it. 

   WARNING: returned character pointer must be freed by the calling
   function. */
{
   char buf[131072],*q;
   unsigned int len;
   va_list args;

   va_start(args,format);
      len=vsnprintf(buf,131071,format,args);
   va_end(args);

/* Moving to small buf: */
   if((q=malloc(1+len))==NULL) { 
      stkerr(" mprintf: ",MEMRESIZE);
      return NULL;
   }
   memcpy(q,buf,len);
   *(q+len)='\0';
   return q;
}

int nc() /* nl ( --- ) */
/* Line feed (new-line) and carriage return. */
{  
   if(KEYS) return(lf());

   chout=0;
   gprintf("\n");
   chout=0;
   *(GOUT+chout)='\0';
   return 1;
}

int number(char *str1, double *x)
/* Interprets a string as a double number, if possible; returns 0 if 
   str cannot be made into a number. 

   Number patterns can contain the characters
      + - . 0 1 2 3 4 5 6 7 8 9 D d E e
*/
{
   char *c,num[256],*str; 
   int len,len1,m=0;
   double n=0;

   len=strlen(str1);
   while(m<len && *(str1+m)==' ') m++; /* skip leading blanks */
   str=str1+m;
   if(*str=='\0') return 0; /* empty strings don't count. */

/* The first character in a number must be one of these:
      + - . 0 1 2 3 4 5 6 7 8 9 
   and the last character cannot be greater than 9: */
   if(*str<'+' || *str>'9' || *(str+strlen(str)-1)>'9') return 0;

/* Function strtod (stdlib.h) returns a pointer to the right end of the
   string that was not converted; if the entire string was converted, 
   then the returned pointer is a NULL and the number will be accepted.
*/ 
   c=str;
   n=strtod(str,&c);
   if(*c=='\0') { 
      *x=n; 
      return 1; 
   }
   len1=MIN(255,(strlen(c)));
   len=strlen(str)-len1;
   memcpy(num,str,MIN((255-len1),len));

   if(*c=='+' || *c=='-') { /* try strings like -1.5-10 as -1.5E-10 */
      *(num+len)='E';
      memcpy((num+len+1),c,len1);
      *(num+1+len+len1)='\0';
      return(number(num,x));
   }
   if(*c=='D' || *c=='d') { /* try replacing D or d with E */
      *(num+len)='E';
      memcpy((num+len+1),c+1,len1);
      *(num+len+len1)='\0';
      return(number(num,x));
   }
   m=0;
   while(m<strlen(c) && *(c+m)==' ') m++;
   if(m) { 
   /* Try with spaces between mantissa and exponent removed, such
      as "1.2 E10" expressed as "1.2E10" (Nastran allows such spaces 
      within a field on a bulk data card): */
      c+=m;
      if(*c=='E' || *c=='e' || *c=='D' || *c=='d') {
         *(num+len)='E';
         len1=strlen(c);
         memcpy((num+len+1),c+1,len1);
         *(num+len+len1)='\0';
         return(number(num,x));
      }
   }
   return 0;
}

int numbers() /* numbers (qS --- hV) */
/* All numbers from string S into vector V.  V(k) is INF if kth string
   pattern is not a number. */
{
   char *S;
   register char *p;
   register double *V;
   register int k=0;
   double x;
   int len;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" numbers: ",STRORVOLNOT);
      return 0;
   }
   if(tos->col==0) {
      return(
         drop() &&
         matstk(0,0,"_numbers")
      );
   }
   textput();
   asciify();

   if((S=(char *)malloc(1+(len=tos->col*tos->row)))==NULL) {
      stkerr(" numbers: ",MEMNOT);
      return 0;
   }
   /* Initial allocation is one 8-byte num for every char: */
   if(!matstk(len,1,"_numbers")) {
      mallfree((void *)&S);
      return 0;
   }
   V=tos->mat;

   memcpy(S,(tos-1)->tex,len); 
   *(S+len)='\0';
   
   p=(char *)strtok(S,", "); /* puts space and comma delimeters */
   if(p) {
      if(number(p,&x)) *(V+k)=x;
      else *(V+k)=INF;
   }
   k++;

   while(p) {
      p=(char *)strtok('\0',", ");
      if(p) {
         if(number(p,&x)) *(V+k)=x;
         else *(V+k)=INF;
         k++;
      }
   }
   mallfree((void *)&S);

   if(!matstk(k,1,"_numbers")) {
      stkerr(" numbers: ",MEMRESIZE);
      drop();
      return 0;
   }
   memcpy(tos->mat,V,k*sizeof(double));
   return(
      lop() && lop()
   );
}

int out() /* out ( --- c) */
/* Number of characters written to current line. */
{
   return(pushint(chout));
}

char parenskip()
/* Skips nested, paired parentheses:
      (...(...)...)..
   Returns char ) if successful, 0 if not. */
{
   int ok=0;

   ip=(unsigned char *)patskip((char *)ip,"(",255,255,")",255,255,&ok);

   if(ok) return ')';
   gprintf(" parenskip: parentheses unpaired"); nc();
   stkerr("","");
   return 0;
}

char *patskip(char *s, char *s1, char c1, char d1,
                       char *s2, char c2, char d2, int *ok)
/* Skips nested pairs of string patterns p1 and p2:

         p1 ... p1 ... p2 ... p2

   where p1=c1:s1:d1, p2=c2:s2:d2, and *s points to byte d1 of the
   initial pattern p1.

   Matching of x:sX:y to pN is successful if sX=pN, x<=cN, y<=dN.

   If successful, returns *patskip pointing to byte d2 that ends
   the last c2:p2:d2, and ok=1; otherwise ok=0 and *patskip points 
   to * (anything) that is the byte following the last x:p2:* if
   any such pattern, or to s+1 if not (whew). 

   Examples of patskip() are in braceskip() and parenskip().
*/
{
   static int pile=0;
   static char *s0=NULL;
   int adv1,adv2;

   if(s0==NULL) s0=s;
   s0++;

   adv2=stradv1(s0,s2,c2,d2);
   if(adv2==0) { /* not successful */
      s=s0+stradv1(s0,s2,255,255); /* trying for anything with s2 */
      s0=NULL;
      pile=0;
      *ok=0;
      return s;
   }
   adv1=stradv1(s0,s1,c1,d1);
   if(adv1!=0 && adv1<adv2) {
      s0+=adv1;
      pile++;
      return(patskip(s0,s1,c1,d1,s2,c2,d2,ok));
   }
   else {
      if(pile==0) { /* successful */
         s=s0+adv2;
         s0=NULL;
         *ok=1;
         return s;
      }
      else {
         s0+=adv2;
         pile--;
         return(patskip(s0,s1,c1,d1,s2,c2,d2,ok));
      }
   }
}

int printf1() /* printf (qS --- qS1) */
/* Writing character bytes in string S to memory and returning string
   S1 on stack. */
{
   char *S,*S1;
   int chars,k=0,len1=0;

   if(tos->typ!=STR) {
      stkerr(" printf: ",STRNOT);
      return 0;
   }
   S=tos->tex;
   chars=tos->col;

   if(!strstk(chars,"_printf")) {
      stkerr(" printf: ",MEMNOT);
      return 0;
   }
   S1=tos->tex;

   while(k<chars) {
      if(*S!=8) {
         *S1=*S;
         S++;
         S1++;
         len1++;
      }
      else { /* assumes backspace correction of the form 08h 20h 08h */
         S++;
         S1--;
         len1--;
      }
      k++;
   }
   if(len1<chars) {
      tos->col=len1;
      *(tos->tex+tos->col)='\0';
   }
   return(lop());
}

int put() /* put (hA hFile --- ) */
/* Put stack item A on file with header for retrieval by get or getn.  
   Item A will be appended to File. */
{
   FILE *fp;
   double *hFile;
   stkitem *A;
   long fsize,words=0;
   size_t items=1,rote=1;
   char *name,name1[256],*p=NULL,*putname="_put";
   int len;

   if(stkdepth()<2) {
      stkerr(" put: ",NEEDTWO);
      return 0;
   }
   if(tos->typ!=MAT ||
      !(fp=filehand(tos->mat,tos->row))) {
      stkerr(" put: ",FILHNDNOT);
      return 0;
   }
   hFile=tos->mat;
   fsize=*(hFile+fSIZ);

   A=tos-1;

   if(*(A->cntc) && A->nam) { /* item is in catalog--use stk.nam: */
      memcpy(name1,A->nam,MIN(255,(len=strlen(A->nam))));
      *(name1+len)='\0';
   }
   else { /* use its token name, which may have been set on-the-fly 
             using word naming */
      if(A->tok) memcpy(name1,A->tok,MIN(255,(len=strlen(A->tok))));
      else memcpy(name1,putname,(len=strlen(putname)));
      *(name1+len)='\0';
   }
   name=strtok(name1,", "); /* remove library tag if any */
   len=1+strlen(name);

   switch(A->typ) {

      case MAT:
         words=sizeof(double)*(A->row)*(A->col);
         p=(char *)A->mat;
      break;

      case NUM:
         words=0;
      break;

      case STR:
      case VOL:
         words=(A->row)*(A->col);
         p=A->tex;
      break;

      default:
         stkerr(" put: ",STKNOT);
         return 0;
      break;
   }
   fseek(fp,fsize,SEEK_SET); /* always appending to file */

/* Writing the header: len, name, type, tag, real, imag, rows, cols */
   fwrite(&len,(size_t)sizeof(int),items,fp);
   fwrite(name,(size_t)len,items,fp);
   fwrite(&A->typ,(size_t)sizeof(int),items,fp);
   fwrite(&A->tag,(size_t)sizeof(long long),items,fp);
   fwrite(&A->real,(size_t)sizeof(double),items,fp);
   fwrite(&A->imag,(size_t)sizeof(double),items,fp);
   fwrite(&A->row,(size_t)sizeof(int),items,fp);
   fwrite(&A->col,(size_t)sizeof(int),items,fp);

/* Writing the stack item: */
   if(words) rote=fwrite(p,(size_t)words,items,fp);
   fupdate(hFile,fp); /* keeping file struct up to date */

   if(rote!=items) {
      stkerr(" put: ",FILWROTERR);
      return 0;
   }
   return(drop2());
}

int put4() /* put4 (hA hFile --- ) */
/* From current file position, put matrix A onto Nastran-style binary
   output4 file. 
   Null columns are not written; an extra zero-length column, beyond 
   the last, is always written to allow seamless termination when the 
   last column is null and therefore not written. */
{
   stkitem *A;
   FILE *fp;
   double *hFile;
   register double *Aj;
   register int i=0,j=0;

   static unsigned int bytes,NC,ND,NF,NR,NT,NW;
   unsigned int header[8];
   static int k;

   static size_t items=1,rote=1;
   char name[32];

   if(tos->typ!=MAT ||
      !(fp=filehand(tos->mat,tos->row*tos->col))) {
      stkerr(" put4: ",FILHNDNOT);
      return 0;
   }
   if((tos-1)->typ!=MAT) {
      stkerr(" put4: ",MATNOT);
      return 0;
   }
   hFile=tos->mat;
   A=(tos-1);
   memset(name,' ',8); /* initialize name to blanks */

/* Use catalog name if in catalog: */
   if(*(A->cntc) && A->nam) { /* item is in catalog--use stk.nam: */
      memcpy(name,A->nam,MIN(31,(strcspn(A->nam,LIBID))));
   }
   else { /* use its token name; may be set on-the-fly using naming */
      memcpy(name,A->tok,MIN(31,(strlen(A->tok))));
   }
/* Writing the header: NC NR NF NT NAME1 NAME2 */
   NC=A->col;
   memcpy(header,(unsigned int *)&NC,sizeof(int));

   if(is_complex(A)) {
      NR=(A->row)/2;
      NT=4; /* type complex double precision */
      ND=2;
   }
   else {
      NR=A->row;
      NT=2; /* type double precision */
      ND=1;
   }
   memcpy(header+1,(unsigned int *)&NR,sizeof(int));

   if(NR==NC) NF=1; /* form 1 is square */
   else NF=2; /* form 2 is rectangular */
   memcpy(header+2,(unsigned int *)&NF,sizeof(int));
   memcpy(header+3,(unsigned int *)&NT,sizeof(int));
   memcpy(header+4,(unsigned int *)name,sizeof(int));
   memcpy(header+5,(unsigned int *)name+1,sizeof(int));

   fortwrite(fp,hFile,header,sizeof(int)*6);

   Aj=A->mat;

   while(j<NC && rote) {

      i=0;
      if(NT==4) { /* complex double */
         while(*(Aj+i)==0 && *(Aj+i+1)==0 &&
            i<NR*2) i+=2; /* trim (Re,Im) comin' */
         i/=2;
      }
      else { /* real double */
         while(*(Aj+i)==0 && i<NR) i++; /* trim comin' */
      }
      if(i<NR) { /* column is not null */

         if(NT==4) { /* complex double */
            k=2*(NR-1);
            while(*(Aj+k)==0 && *(Aj+k+1)==0 &&
               k>i*2) k-=2; /* and trim (Re,Im) goin' */
            k/=2;
         }
         else { /* real double */
            k=NR-1;
            while(*(Aj+k)==0 && k>i) k--; /* and trim goin' */
         }
         NW=(1+k-i)*NT; /* number of 4-byte words */
         bytes=(NW+3)*LEN32;

      /* Writing a Fortran-style binary record: */
         rote=fwrite(&bytes,(size_t)LEN32,items,fp);

         k=j+1;
         rote+=fwrite(&k,(size_t)LEN32,items,fp);
         k=i+1;
         rote+=fwrite(&k,(size_t)LEN32,items,fp);
         rote+=fwrite(&NW,(size_t)LEN32,items,fp);
         rote+=fwrite(Aj+i*ND,NW*LEN32,items,fp);

         rote+=fwrite(&bytes,(size_t)LEN32,items,fp);

         fupdate(hFile,fp); /* keeping file struct up to date */
         rote=(rote==6);
      }
      Aj+=ND*NR;
      j++;
   }
/* Writing the extra column: */
   *(header)=NC+1;
   *(header+1)=1;
   *(header+2)=2;
   *(header+3)=0xEFBEADDE;
   *(header+4)=0xEFBEADDE;
   k=fortwrite(fp,hFile,header,(5*LEN32));

   if(!rote || k) {
      stkerr(" put4: ",FILWROTERR);
      drop2();
      return 0;
   }
   return(drop2());
}

int putch1() /* putch (n or hV --- ) */
{
   register double *V;
   register int k=0;

   int n,ret=0;

   if(tos->typ==NUM) {
      popint(&n);
      gprintf("%c",n);
      ret=1;
   }
   else {
      if(tos->typ==MAT) {
         V=tos->mat;
         for(;k<(tos->row*tos->col);k++) {
            n=(int)*V;
            gprintf("%c",n);
            V++;
         }
         ret=(drop());
      }
   }
   if(!ret) stkerr(" putch: ",STKNOT);
   return(ret);
}

int putn() /* putn (hA1 ... hAn n hFile --- ) */
/* Put the n topmost stack items on File with headers for retrieval 
   by words get or getn.  Items are always appended to File. */
{
   FILE *fp;
   double dn,*hFile;
   stkitem *A;
   long n,words=0;
   size_t items=1,rote=1;
   char *name,name1[256],*p=NULL,*putname="_putn";
   int k=0,len;

   if(stkdepth()<3) {
      stkerr(" putn: ",STKNOT);
      return 0;
   }
   if(tos->typ!=MAT ||
      !(fp=filehand(tos->mat,tos->row))) {
      stkerr(" putn: ",FILHNDNOT);
      return 0;
   }
   hFile=tos->mat;
   drop(); /* file handle off stack */

   if(!popd(&dn)) return 0;
   dn=MAX(0,dn);
   n=(unsigned long)MIN((double)dn,(unsigned long)stkdepth());

   for(k=n-1;k>-1;k--) {
      pushint(k);
      roll(); /* first on stack is first on file */

      A=tos;

      if(*(A->cntc) && A->nam) { /* item is in catalog--use stk.nam: */
         memcpy(name1,A->nam,MIN(255,(len=strlen(A->nam))));
         *(name1+len)='\0';
      }
      else { /* use its token name, which may have been set on-the-fly 
                using word naming */
         if(A->tok) memcpy(name1,A->tok,MIN(255,(len=strlen(A->tok))));
         else memcpy(name1,putname,(len=strlen(putname)));
         *(name1+len)='\0';
      }
      name=strtok(name1,", "); /* remove library tag if any */
      len=1+strlen(name);

      switch(A->typ) {

         case MAT:
            words=sizeof(double)*(A->row)*(A->col);
            p=(char *)A->mat;
         break;

         case NUM:
            words=0;
         break;

         case STR:
         case VOL:
            words=(A->row)*(A->col);
            p=A->tex;
         break;

         default:
            stkerr(" put: ",STKNOT);
            return 0;
         break;
      }
      fseek(fp,(long)*(hFile+fSIZ),SEEK_SET); /* appending to file */

   /* Writing header: len, name, type, tag, real, imag, rows, cols */
      fwrite(&len,(size_t)sizeof(int),items,fp);
      fwrite(name,(size_t)len,items,fp);
      fwrite(&A->typ,(size_t)sizeof(int),items,fp);
      fwrite(&A->tag,(size_t)sizeof(long long),items,fp);
      fwrite(&A->real,(size_t)sizeof(double),items,fp);
      fwrite(&A->imag,(size_t)sizeof(double),items,fp);
      fwrite(&A->row,(size_t)sizeof(int),items,fp);
      fwrite(&A->col,(size_t)sizeof(int),items,fp);

   /* Writing the stack item: */
      if(words) rote=fwrite(p,(size_t)words,items,fp);

      if(rote!=items) {
         stkerr(" putn: ",FILWROTERR);
         return 0;
      }
      fupdate(hFile,fp); /* keeping file struct up to date */
      drop();
   }
   return 1;
}

char *quoteget(char *str)
/* Gets the current quote-string from the source buffer; returns NULL
   pointer if error. 
   Searches only current line, ending with \n. */
{
   char *quote;
   int len,lenn; 

   len=1+stradv((char *)ip+1,str);
   lenn=1+stradv((char *)ip+1,"\n"); 

   if(lenn>1 && (len>lenn || len==1)) {
      gprintf(" quoteget: expect ending %s on same line",str); nc();
      stkerr("","");
      ip+=lenn;
      iprecl=ip; 
      return NULL;
   }
   if((quote=malloc(len+1))==NULL) {
      stkerr(" quoteget: ",MEMNOT);
      ip+=len;
      return NULL;
   }
   memcpy(quote,ip,len); *(quote+len)='\0';
   ip+=len;
   return quote;
}

int rembks(char *str)
/* Removes from str each backspace character and the character--on its
   immediate left--that it deletes.  Returns the total number of char-
   acters that were removed from str.  Assumes the first character is 
   never a backspace, since there is no character on its left. */
{    
   register int acc=1,i;
   int len;

   len=strlen(str);
   i=len-1;
   for(;i>0;i--) {

      if(*(str+i)==8) memcpy((str+i-1),(str+i+1),acc);
      else acc++;
   }
   return(2*(len-acc)); /* total removed = two per backspace */
}

int reprint(unsigned int nchars, int chop)
/* Redesplaying the current output text line shown on the screen and
   stored in GOUT by function gprintf().

   The cursor will be placed after nchars characters, and if chop is
   true the character at the cursor, and all following, will be deleted
   from the screen and from stored image GOUT. */
{
   int wid;
   char GOUTsav[1+GOMAX]={0};

   wid=strlen(GOUT);
   if(nchars>wid || wid==0) return 0;

   if(chop) {
      fprintf(sysout,"%c",0x0); /* stable cursor on slow term */

      gprintf("%s",GOUT+chout); /* moving to rightmost */

      while(chout>nchars) { /* destructive backspace */
         gprintf("%c",8);
         gprintf("%c",' ');
         gprintf("%c",8);
      }
      return(nchars);
   }
   else {

      memcpy(GOUTsav,GOUT,1+wid);

      if(chout<nchars) {
         *(GOUT+nchars)='\0';
         gprintf("%s",GOUT+chout); /* moving right, to offset nchars */
      }
      while(chout>nchars) { /* nondestructive backspace */
         gprintf("%c",8);
      }
      memcpy(GOUT,GOUTsav,1+wid);
      return(wid);
   }
}

int sp() /* sp ( --- ) */
{
   gprintf(" ");
   return chout;
}

int stradv(char *s, char *s1) 
/* Returns the offset in s to the byte following substring s1;
   returns 0 if s1 not found. */
{
   char *s2;
   if((s2=strstr(s,s1))!=NULL) return(s2-s+strlen(s1)); 
   return 0;
}  

int stradv1(char *s, char *s1, unsigned char c, unsigned char d)
/* Returns the offset in s to the byte d that ends substring c:s1:d
   if s1 is surrounded by c (or less) and followed by d (or less);
   returns 0 otherwise. */
{
   register char *s2; 
   register int adv=0,lens1;

   lens1=strlen(s1);
   while((s2=strstr((s+adv),s1))!=NULL) {
      adv+=(s2-(s+adv)+lens1); 
      if(*(s2-1)<=c && *(s+adv)<=d) return adv;
   }      
   return 0;
}

int strlen1(char *str, int len)
/* Returns length without trailing blanks (and other chars < 32). */
{
   register int i;
   i=len-1; 
   while(i>-1 && (isspace((int)*(str+i)) || *(str+i)=='\0')) i--;
   return (i+1);
}

int strmatch(char *str, char *pat)
/* Returns 1 if lengths and bytes of str and pat are the same. */
{  
   register int i=0,lenpat;

   if(strlen(str)!=(lenpat=strlen(pat))) return 0;

   while(i<lenpat && *(str+i)==*(pat+i)) i++;
   return(i==lenpat);
}

int strnmatch(char *str, char *pat, int len)
/* Returns 1 if lengths and bytes of str and pat are the same up to
   len.  Does not presume null-terminated strings. */
{
   register int i=0;

   while(i<len && *(str+i)==*(pat+i)) i++;
   return(i==len);
}

int strsame(char *str1, char *str2)
/* Returns number of leftmost characters of str1 and str2 that are the 
   same. */
{
   register int k=0,len;

   len=MIN(strlen(str1),strlen(str2));
   while(k<len && (*(str1+k)==*(str2+k))) k++;
   return k;
}

int toc() /* toc (hFile --- ) */
/* Display names and header info from a file written by word put. */
{
   FILE *fp;
   double *hFile,imag,real;
   long atpos,pos;
   size_t items=1,red;
   long bytes=0,fsize;
   int cols,end=0,len,type,rows;
   long long tag;
   char name[32];
   char *formM=" matrix: %s  %d by %d";
   char *formN=" number: %s  %0.4E + i%0.4E";
   char *formC=" complx: %s  %d by %d";
   char *formV=" volume: %s  %d by %d";
   char *formS=" string: %s  %d characters";
   char *frmSP=" sparse: %s  %d by %d";
   char *frmSC=" spcplx: %s  %d by %d";
   char *special=" tag(%d): %s  %d by %d";
   int sphead[BPI*SPARSE_HDR_SIZE];

   if(tos->typ!=MAT || !(fp=filehand(tos->mat,tos->row*tos->col))) {
      stkerr(" toc: ",FILHNDNOT);
      return 0;
   }
   hFile=tos->mat;
   fsize=*(hFile+fSIZ);
   atpos=ftell(fp); /* remember current position */

   fseek(fp,0,SEEK_SET); /* rewind */

   while(!end) {
   /* Read header: len, name, type, tag, real, imag, rows, cols */
      red=fread(&len,(size_t)sizeof(int),items,fp);

      if(ftell(fp)>=fsize) len=0;

      if(len==0 || red!=items) end=1;
      else {
         len=MIN((unsigned long)len,32);

         fread(name,len,items,fp); /* reading null terminated */
         fread(&type,(size_t)sizeof(int),items,fp);
         fread(&tag,(size_t)sizeof(long long),items,fp);
         fread(&real,(size_t)sizeof(double),items,fp);
         fread(&imag,(size_t)sizeof(double),items,fp);
         fread(&rows,(size_t)sizeof(int),items,fp);
         fread(&cols,(size_t)sizeof(int),items,fp);

         switch(type) {

            case MAT:
               if((tag & TAG_COMPLEX)==TAG_COMPLEX) 
                  gprintf(formC,name,rows/2,cols);
               else gprintf(formM,name,rows,cols);
               nc();
               bytes=rows*cols*sizeof(double);
            break;

            case NUM:
               gprintf(formN,name,real,imag); 
               nc();
               bytes=0;
            break;

            case STR:
               gprintf(formS,name,cols); 
               nc();
               bytes=cols;
            break;

            case VOL:
               bytes=rows*cols; /* total size of VOL */

               if(tag==NOTAG) { /* ordinary VOL: */
                  gprintf(formV,name,rows,cols);
               }
               else { /* have a VOL with a tag */

            /* Assuming that any VOL with tag is the type having
               SPARSE_MAGIC_SIZE at the beginning.  This is not
               checked and could change in the future. */

                  pos=ftell(fp); /* save current pos */
                  fseek(fp,BPI*SPARSE_MAGIC_SIZE,SEEK_CUR);
                  red=fread(sphead,(size_t)BPI*SPARSE_HDR_SIZE,\
                     items,fp);
                  fseek(fp,pos,SEEK_SET); /* back to current */

                  rows=*(sphead+ROWS); /* reusing */
                  cols=*(sphead+COLS);

                  if((tag & TAG_SPARSE)==TAG_SPARSE) {

                     if((tag & TAG_COMPLEX)==TAG_COMPLEX) 
                        gprintf(frmSC,name,*(sphead+ROWS),\
                           *(sphead+COLS));
                     else
                        gprintf(frmSP,name,*(sphead+ROWS),\
                           *(sphead+COLS));
                  }
                  else 
                     gprintf(special,tag,name,*(sphead+ROWS),\
                        *(sphead+COLS));
               }
               nc();
            break;

            default:
               stkerr(" toc: ","file not written by put");
               end=1;
            break;
         }
         fseek(fp,bytes,SEEK_CUR);
      }
   }
   fseek(fp,atpos,SEEK_SET); /* putting pointer back where it was */

   fupdate(hFile,fp); /* keeping file struct up to date */
   return(drop());
}

int _toc4() /* _toc4 (hFile --- hT) */
/* Create a volume of matrix names in a Nastran-style binary output4 
   file. */
{
   FILE *fp;
   double *hFile;
   long atpos;

   unsigned int header[8],j=0,len,NC1;
   int filerr=0,found=0,rows=0;

   char *form=" %s  %d by %d  form %d  type %d",N[16],*p;

   if(tos->typ!=MAT ||
      !(fp=filehand(tos->mat,tos->row*tos->col))) {
      stkerr(" toc4: ",FILHNDNOT);
      return 0;
   }
   hFile=tos->mat;
   atpos=ftell(fp); /* remember current position */

   fseek(fp,0,SEEK_SET); /* rewind */

   while(!filerr) {
      
      /* MSC/Nastran V68 DMAP Dictionary, INPUTT4, p. 159:
         header items (0-6): NC NR NF NT NAME1 NAME2 FORMAT */
      filerr=fortread(fp,hFile,header,7*sizeof(int),&len);
      if(filerr) { 
         if(!found) stkerr(" toc4: ","not a valid output4 file");
         break;
      }
      memcpy(N,header+4,8); /* NAME1+NAME2 */
      *(N+strlen1(N,8))='\0';

      if(*(header+2)<0 || *(header+2)>6) { /* form must be 1 - 6 */
         stkerr(" toc4: ","not a valid output4 file");
         filerr=1;
         break;
      }
      found=1;
/*
      gprintf(form,N,*(header+1),*(header),*(header+2),*(header+3));
      nc();
*/
      p=mprintf(form,N,*(header+1),*(header),*(header+2),*(header+3));
      if(p) {
         rows++;
         pushq2(p,strlen(p));
         mallfree((void *)&p);
      }
      NC1=1+*(header);
      j=0;
      while(j<NC1 && !filerr) {
         filerr=fortread(fp,hFile,&j,sizeof(int),&len);
      }
   }
   fseek(fp,atpos,SEEK_SET); /* putting pointer back where it was */
   fupdate(hFile,fp); /* keeping file struct up to date */

   if(rows) {
      pushint(rows);
      pilen();
   }
   else {
      pushq2("",0);
   }
   return(lop());
}

int toc4() /* toc4 (hFile --- ) */
/* Display matrix names in a Nastran-style binary output4 file. */
{
   if(_toc4()) {
      return(dot() && nc());
   }
   else {
      return 0;
   }
}

int token0()
{
   stkerr(" token0: ",STRNULL);
   report(NULL); /* text contains numerical zero */
   return 1;
} 

char *tokenget1();

char *tokenget1()
/* For debugging.  Rename this function tokenget(), and rename 
   tokenget() to be tokenget1().  */
{
   char *token[NBUF];
   int bytes=24;

   *(token+onbuf)=tokenget1();

   pushq2((char *)iprect,bytes); dup1s(); dot(); nc(); dothex(); nc();

   if(*(token+onbuf)) {
      gprintf(" tokenget.  token,ip,iprect,ipend: %s,%d,%d,%d", \
         *(token+onbuf),ip,iprect,ipend); nc();
   } 
   else {
      gprintf(" tokenget source end.  ip,iprect,ipend: %d,%d,%d", \
         ip,iprect,ipend); nc();
   }
   return(*(token+onbuf));
}

char *tokenget()
/* Gets the next token from the source buffer; returns pointer to 
   empty token if error, and NULL pointer if end of source buffer. */
{
   register unsigned char *temp;
   int len;

   /* Do not free tokenq.  It may now be the mem pointer for an item on
   the stack or in the catalog.  Items from tokenq are always pushed to
   the stack, and drop() and catfree() take care of memory management
   from there. */

   tokenq=NULL; /* on return, tokenq will not be null if string or
                volume of text has been read; tokenq sent will be
                pushed to the stack, where it will live until it
                is freed or put into catalog */
 
   while(blank(*ip) && ip<ipend) ip++;

   if(*ip=='\0' || ip>=ipend) { /* end of source buffer */
      return NULL; /* the only time tokenget returns NULL */ 
   }
   iprect=iprecl; /* the line this token is on */

   if(*ip=='{' && *(ip+1)=='"') {
      if((tokenq=blockget("\"}"))!=NULL) { /* text is in tokenq */
         strcpy(token,"{\""); return token;
      }
      *token='\0'; return token;
   }
   if(*ip=='"') {
      if((tokenq=quoteget("\""))!=NULL) { /* text is in tokenq */
         strcpy(token,"\""); return token;
      }
      *token='\0'; return token;
   }
   if(*ip=='\'') {
      if((tokenq=quoteget("'"))!=NULL) { /* text is in tokenq */
         strcpy(token,"'"); return token;
      }
      *token='\0'; return token;
   }
   if(*ip=='(') {
      *token=parenskip(); 
      return(tokenget()); /* going back for another */
   }
   if(*ip=='{') {
      *token=braceskip(); 
      return(tokenget()); /* going back for another */
   }
   if(*ip==')' || *ip=='}') {
      gprintf(" tokenget: extraneous %c",*ip); nc();
      stkerr("","");
      ip++; *token='\0'; return token;
   }
   temp=ip;
   ip++;
   while(!blank(*ip)) ip++;
   len=ip-temp;
   ip++;
   if(len>TOKBUF) {
      gprintf(" tokenget: token exceeds size of buffer"); nc();
      stkerr("","");
      ip+=len;
      len=0;
   }
   memcpy(token,temp,len); *(token+len)='\0';
   return token;
}

char *tokengetc()
/* Gets the next token from the source buffer during control mode 

   Control mode is a mode during interpretation when words are being
   skipped in a branch not to be taken, as when words in the IF part 
   of the IF ... ELSE ... THEN structure are being skipped to get to 
   the ELSE part--see ctrl.c. 

   Returns pointer to empty token if error, and NULL pointer if end 
   of source buffer.

   Works same as tokenget except quotes, returned in tokenq, are 
   skipped. */
{
   char *token[NBUF],*temp=(char *)!NULL;

   while(temp) {
      *(token+onbuf)=tokenget();
      temp=tokenq;
      mallfree((void *)&tokenq);
   }
   return(*(token+onbuf));
}

int tout() /* tout ( --- qS) */
/* Pushing string containing current displayed line, of out number of
   characters, to the stack. */
{
   return(pushq2(GOUT,chout));
}

int uimport(int size, int fromendian) /* (hT --- hA) */
/* For size equal to 4, or 2, convert unsigned size-byte numbers of
   endian type fromendian in T into 8-byte floating point numbers of
   the machine's endian type, and put them into matrix A. 

   Elements of T are 4 or 2 bytes wide and contain 4 or 2 byte un-
   signed integers in fromendian order.

   The number of columns in matrix A equals the number of rows in
   volume T, and the number of rows in A is the bytes per row of T
   divided by size.  This transposition of rows and columns is be-
   cause volumes, like T, store adjacent elements in a row while
   matrices, like A, store adjacent elements in a column--also see
   explanation in export(). */
{
   register char *Afrom,*Ato,*Aint;
   register double *Amat;
   unsigned int Aint4;
   unsigned short Aint2;
   register long *from=NULL,i=0,*to=NULL;
   int cols,k,rows,terms;
   char name[10];

   long d1234[8]={7,6,5,4,3,2,1,0}; /* little endian */
   long d4321[8]={0,1,2,3,4,5,6,7}; /* big endian */
   long d3412[8]={5,4,7,6,1,0,3,2}; /* pdp endian */

   long l1234[4]={3,2,1,0}; /* little endian */
   long l4321[4]={0,1,2,3}; /* big endian */
   long l3412[4]={1,0,3,2}; /* pdp endian */

   long s1234[2]={1,0}; /* little endian */
   long s4321[2]={0,1}; /* big endian */
   long s3412[2]={1,0}; /* pdp endian */

   sprintf(name,"_uimport%d",size);
   *(name+9)='\0';

   cols=tos->row;
   rows=(tos->col)/size;

   if(!matstk(rows,cols,name)) return 0;

   if(!rows) return(lop());

   terms=rows*cols;

   Afrom=(tos-1)->tex;
   Ato=(char *)tos->mat;
   Amat=tos->mat;

   k=byteorder(NULL);

   switch(size) {

      case 8:
         switch(fromendian) {
            case 1234: from=d1234; break;
            case 4321: from=d4321; break;
            case 3412: from=d3412; break;
         }
         switch(k) {
            case 1234: to=d1234; break;
            case 4321: to=d4321; break;
            case 3412: to=d3412; break;
         }
         for(;i<terms;i++) {
            *(Ato+*(to  ))=*(Afrom+*(from  ));
            *(Ato+*(to+1))=*(Afrom+*(from+1));
            *(Ato+*(to+2))=*(Afrom+*(from+2));
            *(Ato+*(to+3))=*(Afrom+*(from+3));
            *(Ato+*(to+4))=*(Afrom+*(from+4));
            *(Ato+*(to+5))=*(Afrom+*(from+5));
            *(Ato+*(to+6))=*(Afrom+*(from+6));
            *(Ato+*(to+7))=*(Afrom+*(from+7));
            Afrom+=8;
            Ato+=8;
         }
      break;

      case 4:
         switch(fromendian) {
            case 1234: from=l1234; break;
            case 4321: from=l4321; break;
            case 3412: from=l3412; break;
         }
         switch(k) {
            case 1234: to=l1234; break;
            case 4321: to=l4321; break;
            case 3412: to=l3412; break;
         }
         Aint=(char *)&Aint4;
         for(;i<terms;i++) {
            *(Aint+*(to  ))=*(Afrom+*(from  ));
            *(Aint+*(to+1))=*(Afrom+*(from+1));
            *(Aint+*(to+2))=*(Afrom+*(from+2));
            *(Aint+*(to+3))=*(Afrom+*(from+3));
            *Amat=_ureal4(Aint4);
            Afrom+=4;
            Amat++;
         }
      break;

      case 2:
         switch(fromendian) {
            case 1234: from=s1234; break;
            case 4321: from=s4321; break;
            case 3412: from=s3412; break;
         }
         switch(k) {
            case 1234: to=s1234; break;
            case 4321: to=s4321; break;
            case 3412: to=s3412; break;
         }
         Aint=(char *)&Aint2;
         for(;i<terms;i++) {
            *(Aint+*(to  ))=*(Afrom+*(from  ));
            *(Aint+*(to+1))=*(Afrom+*(from+1));
            *Amat=_ureal2(Aint2);
            Afrom+=2;
            Amat++;
         }
      break;
   }
   return(lop());
}

int xray() /* xray (x or hFile bytes --- hT) */
/* Hex dump of bytes from File or stack item x. */
{
   int f,new,ret=0;
   char *fhandname="xray.tmp";

   if(tos->typ!=NUM) {
      stkerr(" xray: expect number of bytes","");
      return 0;
   }
   if(!over() || !fileid()) return 0;
   popint(&f);

   if(f) ret=fxray();

   else {

      if((tos-1)->typ!=NUM && (tos-1)->typ!=MAT && 
         (tos-1)->typ!=STR && (tos-1)->typ!=VOL) {
         stkerr(" xray: expect number, matrix, string or volume","");
         return 0;
      }
   /* These three lines make the temp file name: */
      pushq2("'_bin' 'tmppath' yank runid cat",31); xmain(0);
      pushq2(fhandname,strlen(fhandname));
      cat(); /* qPath+File */

      dup1s();
      fileq();
      popint(&f);
      if(f) { /* deleting file if it exists */
         dup1s();
         delete();
      }
      pushq2(fhandname,strlen(fhandname));
      new=NEW;
      NEW=xTRUE;
      ret=fileopen(); /* file handlle into local lib */
      NEW=new;

      if(ret) {
      /* File handle to stack: */
         pushq2(fhandname,strlen(fhandname)); libpry();

         rot(); /* x to top of stack */
         hand();

      /* Writing x to file: */
         over();
         fileput();

      /* Rewinding file: */
         dup1s();
         pushint(0);
         fileseek();

      /* Running fxray(): */
         swap(); /* (hFile bytes --- ) */
         ret=fxray();

      /* Closing tmp file: */
         pushq2(fhandname,strlen(fhandname));
         libpry();
         fileclose();

      /* Deleting tmp file: */
      /* These three lines make the temp file name: */
         pushq2("'_bin' 'tmppath' yank runid cat",31); xmain(0);
         pushq2(fhandname,strlen(fhandname));
         cat(); /* qPath+File */

         delete();
      }
   }
   return(
      ret &&
      pushq2("_xray",5) &&
      naming()
   );
}
