/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author and copyright holder of sql.c:  Al Danial <al.danial@gmail.com>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
}}}1 */
#ifdef SQLITE
#include <stdlib.h>
#include <string.h>   /* memset       */
#include <strings.h>  /* strncasecmp  */
#include <stdio.h>
#ifdef   __USE_ISOC99
#include <math.h>     /* INT32_MIN, NAN */
#else
#define INT32_MIN  (-(1 << 30) - 1)
#define NAN        (1.0E-309)
#endif
#include "tex.h"      /* strchop */
#include "mem.h"      /* memgetn */
#include "exe.h"      /* xmain   */
#include "stk.h"
#include "main.h"
#include "inpo.h"
#include "sql.h"      /* callback_data */
#include "tag.h"      /* is_dbh, set_dbh */

/* 
   sql.c

   Albert Danial  November 23 2003

   Words for interacting with relational databases via the
   Structured Query Language.
 */

int db_open()           /* db_open  ( qFilename --- dbh ) {{{1 */
/*
 * man entry:  db_open {{{2
 * ( qFilename --- dbh ) Open the SQLite database stored in the given filename and return a handle to the database.
 * category: db
 * related: sql, db_open2, db_close
 * 2}}}
 */
{
    char   *name = "_dbhandle";
    int     size = sizeof(sqlite3 *);
int DEBUG = 0;

if (DEBUG) gprintf("-> inside db_open size of sqlite3 db item=%d\n", size);

    if (tos->typ != STR) {
        stkerr(" db_open: ",STRSNOT);
        return 0;
    }
    strchop();  /* strip leading and trailing white space from db filename */

    if (!volstk(1, size, name)) return 0; /* ( qFilename --- qFilename dbh ) */
    set_dbh(tos);

    /*
     * Open the database; put the SQLite database handle into a stack item.
     */
if (DEBUG) {
gprintf("db_open db name is [%s]\n",   (tos-1)->tex);
}
    if (sqlite3OsFileExists((tos-1)->tex)) {
        if (sqlite3_open((tos-1)->tex, (sqlite3 **) &(tos->tex)) != SQLITE_OK) {
            stkerr(" db_open: ", "failed to open sqlite database");
            return 0;
        }
    } else {
        drop(); /* remove database handle */
        stkerr(" db_open: ", FILENOT);
        return 0;
    }
    lop();  /* remove filename from tos */

    return 1;
} /* 1}}} */
int db_open2()          /* db_open2 ( qFile1 qFile2 qAlias --- dbh ) {{{1 */
/*
 * man entry:  db_open2 {{{2
 * ( qFile1 qFile2 qAlias --- dbh ) Open the SQLite database file qFile1 and attach to it a second SQLite database file qFile2.  The attached database gets the alias qAlias.
 * category: db
 * related: sql, db_open, db_close
 * 2}}}
 */
{
    char    *name = "_dbhandle";
    int      i, rc = 0, size = sizeof(sqlite3 *);
    char     message[SQL_STR_SIZE-1];
    sqlite3 *sql_dbh;
int DEBUG = 0;
if (DEBUG) gprintf("top of db_open2\n");

    /* top three stack items are strings; clean them by trimming whitespace */
    for (i = 0; i < 3; i++) {
        if (tos->typ != STR) {
            stkerr(" db_open2: ",STRSNOT);
            return 0;
        }
        strchop();  /* strip leading and trailing white space from qAlias */
        pushstr("2 roll"); xmain(0); 
    }

    if (!volstk(1, size, name)) return 0; 
    /* ( qFile1 qFile2 qAlias --- qFile1 qFile2 qAlias dbh ) */
    set_dbh(tos);

if (DEBUG) {
gprintf("db_open2 db name 1 is [%s]\n",   (tos-3)->tex);
}
    /* open the primary database */
    if (sqlite3OsFileExists((tos-3)->tex)) {
        /* Have to pass in a locally defined database handle as the 2dn
         * argument to sqlite3_open otherwise the attach won't work.
         */
        if (sqlite3_open((tos-3)->tex, (sqlite3 **) &sql_dbh) != SQLITE_OK) {
            snprintf(message, SQL_STR_SIZE, 
                     "failed to open sqlite database '%s'", (tos-3)->tex);
            stkerr(" db_open2: ", message);
            return 0;
        }
        /* now manually put the database handle into the tos container */
        memcpy(&(tos->tex), &sql_dbh, size);
    } else {
        drop(); /* remove database handle */
        stkerr(" db_open2: primary db ", FILENOT);
        return 0;
    }

if (DEBUG) {
gprintf("db_open2 db name 2 is [%s]\n",   (tos-2)->tex);
}
    /* attach the secondary database to it */
    if (sqlite3OsFileExists((tos-2)->tex)) {
        snprintf(message, SQL_STR_SIZE, 
                 "attach '%s' as %s;", (tos-2)->tex, (tos-1)->tex);
        rc = sql_cmd(sql_dbh, "db_open2 ", message);
        if (!rc) return 0;
    } else {
        drop(); /* remove database handle */
        stkerr(" db_open2: secondary db ", FILENOT);
        return 0;
    }

    lop();  /* remove qAlias from tos */
    lop();  /* remove qFile1 from tos */
    lop();  /* remove qFile2 from tos */

    return 1;
} /* 1}}} */
int db_attach()         /* db_attach( dbh qFilename qAlias --- ) {{{1 */
/*
 * man entry:  db_attach {{{2
 * ( dbh qFilename qAlias --- ) Attach the SQLite database stored in the given filename and to the an already opened SQLite database whose handle is dbh.  Assign the name stored in qAlias as the new database's alias in SQLite.
NOTE:  db_attach won't work until an SQLite database handle can be fully serialized to the stack
 * category: db
 * related: sql, db_open, db_open2, db_detach, db_close
 * 2}}}
 */
{
    sqlite3 *sql_dbh;
    char attach_command[SQL_STR_SIZE-1]; 
    int rc = 0;
int DEBUG = 0;

    stkerr(" db_attach: ", "use db_open2 instead of db_attach.  db_attach won't work until an SQLite database handle can be fully serialized to the stack");
    return 0;

if (DEBUG) gprintf("-> inside db_attach\n");

    if (tos->typ != STR) {
        stkerr(" db_attach: ",STRSNOT);
        return 0;
    }
if (DEBUG) gprintf("-> inside db_attach a\n");
    strchop();  /* strip leading and trailing white space from alias */
    swap();
if (DEBUG) gprintf("-> inside db_attach b\n");

    if (tos->typ != STR) {
        stkerr(" db_attach: ",STRSNOT);
        return 0;
    }
if (DEBUG) gprintf("-> inside db_attach c\n");
    strchop();  /* strip leading and trailing white space from filename */
    swap();
if (DEBUG) gprintf("-> inside db_attach d\n");

    if (!is_dbh(tos-2)) {
        stkerr(" db_attach: ", "expecting an SQLite database handle");
        return 0;
    }
if (DEBUG) gprintf("-> inside db_attach e\n");
    sql_dbh = (sqlite3 *) &((tos-2)->tex);
if (DEBUG) gprintf("-> inside db_attach f\n");

    snprintf(attach_command, SQL_STR_SIZE, 
             "attach '%s' as %s;", (tos-1)->tex, tos->tex);
if (DEBUG) gprintf("-> inside db_attach h [%s]\n", attach_command);

   /* statement below will cause a segfault unless the 
    * database handle sql_dbh is opened here, or is fully
    * serialized on the stack
    */
    rc = sql_cmd(sql_dbh, "db_attach ", attach_command);
    if (!rc) return 0;
    return 1;

} /* 1}}} */
int db_detach()         /* db_detach( dbh qAlias --- ) {{{1 */
/*
 * man entry:  db_detach {{{2
 * ( dbh qAlias --- ) Detach the SQLite database qAlias from dbh.
 * category: db
 * related: sql, db_open, db_open2, db_attach, db_close
 * 2}}}
 */
{
    sqlite3 *sql_dbh;
    char detach_command[SQL_STR_SIZE-1]; 
int DEBUG = 0;

if (DEBUG) gprintf("-> inside db_detach\n");

    if (tos->typ != STR) {
        stkerr(" db_detach: ",STRSNOT);
        return 0;
    }
    strchop();  /* strip leading and trailing white space from alias */

    if (!is_dbh(tos-1)) {
        stkerr(" db_detach: ", "expecting an SQLite database handle");
        return 0;
    }
    sql_dbh = (sqlite3 *) &((tos-1)->tex);

    snprintf(detach_command, SQL_STR_SIZE, 
             "detach %s;", tos->tex);

    if (sql_cmd(sql_dbh, "db_detach ", detach_command)) {
        return 0;
    } else {
        drop();
        drop();
    }
    return 1;

} /* 1}}} */
int db_close()          /* db_close ( dbh --- ) {{{1 */
/*
 * man entry:  db_close {{{2
 * ( dbh --- ) Close the SQLite database defined by the given database handle.
 * category: db
 * related:  sql, db_open, db_open2
 * 2}}}
 */
{
int DEBUG = 0;
    sqlite3 *sql_dbh;
if (DEBUG) gprintf("-> inside db_close\n");

    if (!is_dbh(tos)) {
        stkerr(" db_close: ", "expecting an SQLite database handle");
        return 0;
    }
    sql_dbh = (sqlite3 *) &(tos->tex);
    sqlite3_close(sql_dbh);
    drop();
    return 1;
} /* 1}}} */
int sql()               /* sql ( dbh qSQL --- [col_N .. col_1 N] ) {{{1 */
/*
 * man entry:  sql {{{2
 * ( dbh qSQL --- [col_N .. col_1 N] ) Execute the SQL statement in qSQL on the database whose handle is dbh (use db_open to create a database handle from a filename).  If qSQL does not contain a SELECT, the command is performed and nothing is put on the stack.  If qSQL starts with SELECT, puts on the stack N data items returned by the query.  N = 0 if the query comes up empty.  If the query returns only a single row the stack items will be scalar values from that row.  If the query returns more than one row the stack items will be arrays of numbers or strings, ie, mat's or vol's, named after the database column prefixed with 'sql_'.
 * category: db
 * related:  db_open, db_open2, db_close
 * 2}}}
 */
{
    int           r = 0, c = 0, rc;
    sqlite3      *sql_dbh;
    callback_data SQL_data;
    double       *Num_Vector  = 0;
    char         *Char_Vector = 0;
    char          T[ERR_MSG_SIZE+1];
int DEBUG = 0;

    if (!is_dbh(tos-1)) {
        stkerr(" db_close: ", "expecting an SQLite database handle");
        return 0;
    }
    sql_dbh = (sqlite3 *) (tos-1)->tex;

    if (tos->typ != STR) {
        stkerr(" sql: ",STRSNOT);
        return 0;
    }
    lowercase();
    strchop();  /* strip leading and trailing white space from SQL command */

if (DEBUG) gprintf("-> inside sql expr=[%s]\n", tos->tex);
    if (strncmp(tos->tex, "select", 6)) {
        /* not a select, issue as a command */
        rc = sql_cmd(sql_dbh, "sql ", tos->tex);
        if (!rc) return 0;
        drop();     /* remove SQL command from tos     */
        drop();     /* remove database handle from tos */
        return 1;
    }

    rc = sql_do_query(sql_dbh, tos->tex, &SQL_data);
if (DEBUG) gprintf("-> inside sql rc from do_query=%d\n", rc);
    if ((rc == SQLITE_DONE) || (rc == SQLITE_OK)) {
        drop();     /* remove SQL command from tos     */
        drop();     /* remove database handle from tos */
        if (SQL_data.nRows == 1 && SQL_data.nCols == 1) {
            /* special case, returned a single scalar result */
            sql_push_scalar(r, c, SQL_data);
            pushint(1);  /* return a single stack item; N = 1 */
        } else {
            /* returned multiple columns or a single column w/multiple rows */
            /* each column produces one stack item */
            for (c = 0; c < SQL_data.nCols; c++) {
                if (SQL_data.nRows == 1) {
                    /* a scalar in every column */
                    sql_push_scalar(0, c, SQL_data);
                } else {
                    /* a vector in every column; need to allocate memory */
                    switch (SQL_data.type[c]) {
                    case SQLITE_INTEGER:
                    case SQLITE_FLOAT:
                        matstk(SQL_data.nRows,             1, SQL_data.name[c]);
                        Num_Vector  = tos->mat;
                        break;
                    case SQLITE_TEXT:
                        volstk(SQL_data.nRows,SQL_MAX_ID_LEN, SQL_data.name[c]);
                        Char_Vector = tos->tex;
                        break;
                    case SQLITE_NULL:  /* to store nan's */
                        matstk(SQL_data.nRows,             1, SQL_data.name[c]);
                        break;
                    default:
                        snprintf(T, ERR_MSG_SIZE,
                                 " unknown datatype %d", SQL_data.type[c]);
                        stkerr(" sql: ", T);
                        return 0;
                    }

                    /* now populate each row in the vector */
                    for (r = 0; r < SQL_data.nRows; r++) {
                        switch (SQL_data.type[c]) {
                        case SQLITE_INTEGER:
                            Num_Vector[r] = SQL_data.i[r][c];
                            break;
                        case SQLITE_FLOAT:
                            Num_Vector[r] = SQL_data.x[r][c];
                            break;
                        case SQLITE_NULL:
                            Num_Vector[r] = NAN;
                            break;
                        case SQLITE_TEXT:
                            strncpy(&Char_Vector[r*SQL_MAX_ID_LEN], 
                                     SQL_data.a[r][c], 
                                     SQL_MAX_ID_LEN);
                            break;
                        default: /* never gets here, condition trapped above */
                            break;
                        }
                    }
                }
            }
            /* flip order of returned columns from col_1 .. col_N to 
             * col_N .. col_1 */
            pushint(SQL_data.nCols); pushstr("revn"); xmain(0); 
            pushint(SQL_data.nCols);  /* N = number of columns */
        }
    } else {
        stkerr(" sql: statement : ", tos->tex);
        stkerr(" sql: gave error: ", SQL_data.err_msg);
        return 0;
    }

    return 1;
} /* 1}}} */

int sql_scalar_i(sqlite3 *dbh, char *query, char *error_string) {  /* {{{1 */
    /* Returns as an integer the first value produced by query.
     * This function is handy for getting table counts and other
     * SQL operations that return a scalar integer.  For example,
     *
     * n_nodes = sql_scalar_i("select count(seq_no) from node", 
     *                        "failed to count number of nodes")
     *
     */
    int    rc;
    callback_data buffer;

    buffer.row_index  = 0;
    rc = sql_do_query(dbh, query, &buffer);
    if ((rc == SQLITE_OK) || (rc == SQLITE_DONE)) {
        return buffer.i[0][0];
    } else {
        stkerr(" sql_scalar_i:", query);
        stkerr(" sql_scalar_i:", error_string);
        stkerr(" sql_scalar_i:", (char *) sqlite3_errmsg(dbh));
        return 0;
    }
} /* 1}}} */
int sql_do_query(sqlite3       *dbh  ,    /* in {{{1 */
                 char          *query,    /* in */
                 callback_data *sql_data) /* out */ 
{
    /*
     * Populates the global variable sql_data with the data
     * returned by the query.  Returns an sqlite3 rc value 
     * (SQLITE_OK is 0; values between 1 and < 100 are errors).
     */
int DEBUG = 0;  /* 0 -> no output; 1 -> debug; 2 -> vebose debug */
    int    rc, row = 0, col = 0, actual_type = 0, 
           have_undeclared_type = xFALSE;
    sqlite3_stmt *RowData;
    char          declared_type[SQL_MAX_ID_LEN+1];

    (*sql_data).nRows = 0;
    strncpy((*sql_data).err_msg, "no error", SQL_STR_SIZE);

    rc = sqlite3_prepare(dbh, query, -1, &RowData, 0);
    if (rc != SQLITE_OK) {
        snprintf((*sql_data).err_msg, SQL_STR_SIZE, 
                 "prepare failed for %s\n", query);
        return rc;
    }
if (DEBUG) gprintf("sql_do_query: top of query [%s], rc=%d\n", query, rc);
    (*sql_data).nCols      = sqlite3_column_count(RowData);
if (DEBUG) gprintf("sql_do_query ncolumns = %d\n", (*sql_data).nCols);

    /* Get the name and type of each column.  If the declared type
     * isn't an integer or float, call it a string.  If it has no
     * declared type, use the actual type later on. 
     */
    for (col = 0; col < (*sql_data).nCols; ++col) {
if (DEBUG) gprintf("sql_do_query col %d decltype=[%s]\n", 
col, sqlite3_column_decltype(RowData, col));
        if (sqlite3_column_decltype(RowData, col)) {
            strncpy(declared_type, 
                    sqlite3_column_decltype(RowData, col),
                    SQL_MAX_ID_LEN);
            if        (!strncasecmp(declared_type, "int"    , 3)) {
                (*sql_data).type[col] = SQLITE_INTEGER;
            } else if (!strncasecmp(declared_type, "float"  , 5)) {
                (*sql_data).type[col] = SQLITE_FLOAT;
            } else {
                (*sql_data).type[col] = SQLITE_TEXT ;
            }
        } else {
            /* No declared type; most likely an aggregate, join, etc.
             * Set the declared type to be the actual type based on values
             * from the first row of data.
             */
            (*sql_data).type[col] = 0;
            have_undeclared_type  = xTRUE;
        }
        strncpy((*sql_data).name[col], "sql_", SQL_MAX_ID_LEN);
        strncat((*sql_data).name[col], sqlite3_column_name(RowData, col),
                SQL_MAX_ID_LEN);
if (DEBUG) gprintf("sql_do_query column %2d  type=%d  name=[%s]\n", 
                   col                  , 
                   (*sql_data).type[col],
                   (*sql_data).name[col]);
    }

    /* Now loop through all the rows and extract the data. */
    while ((rc = sqlite3_step(RowData)) == SQLITE_ROW) {
if (DEBUG > 1) gprintf("sqlite3_step: start of row %d\n", row);

        if (!row && have_undeclared_type) {
            /* One of the columns had an undeclared type.  This is
             * the first row of data so go through each column and
             * assign the actual data type to each column which
             * didn't have a declared type.
             */
            for (col = 0; col < (*sql_data).nCols; ++col) {
                if (!(*sql_data).type[col]) {
                    (*sql_data).type[col] = sqlite3_column_type( RowData, col );
                }
            }
        }

        for (col = 0; col < (*sql_data).nCols; ++col) {
            actual_type = sqlite3_column_type( RowData, col );
            switch ( (*sql_data).type[col] ) { /* on *declared* type */
            case SQLITE_INTEGER:
if (DEBUG) gprintf("%d/i|", sqlite3_column_int(   RowData, col));
                if (actual_type == SQLITE_NULL) {
                    (*sql_data).i[row][col] = INT32_MIN;
                } else {
                    (*sql_data).i[row][col] = sqlite3_column_int(RowData, col);
                }
                break;
            case SQLITE_FLOAT:
                if (actual_type == SQLITE_NULL) {
                    (*sql_data).x[row][col] = NAN;
                } else {
if (DEBUG) gprintf("%e/f|", sqlite3_column_double(RowData, col));
                    (*sql_data).x[row][col] = sqlite3_column_double(RowData,col);
                }
                break;
            case SQLITE_TEXT:
if (DEBUG) gprintf("%s/t|", sqlite3_column_text(RowData, col));
                if (actual_type == SQLITE_NULL) {
                    (*sql_data).a[row][col][0] = '\0';     
                } else {
                    strncpy((*sql_data).a[row][col],     
                            (char *) sqlite3_column_text(RowData, col),
                            SQL_STR_SIZE);
                }
                break;
            default:
                /* Can get here legitimately if the column had no
                 * declared type and the actual type was null or blob.
                 * Dang.  Call it a null integer.
                 */
                (*sql_data).type[col]   = SQLITE_INTEGER;
                (*sql_data).i[row][col] = INT32_MIN;
            }
        }
if (DEBUG) gprintf("\n");
        ++row;
    }
    rc = sqlite3_finalize(RowData);

    (*sql_data).row_index += row;
    (*sql_data).nRows      = row;

    return rc;
} /* 1}}} */
int sql_push_scalar(int r, int c, callback_data buffer) { /* {{{1 */
    char T[ERR_MSG_SIZE+1];

    switch (buffer.type[c]) {
    case SQLITE_INTEGER:
        pushint(buffer.i[r][c]);
        break;
    case SQLITE_FLOAT:
        pushd(  buffer.x[r][c]);
        break;
    case SQLITE_TEXT:
        pushstr(buffer.a[r][c]);
        break;
    case SQLITE_NULL:
    case SQLITE_BLOB: /* not ready to handle blob; treat as null */
        pushint(0) && pushint(0) && null(); /* a purged matrix */
        break;
    default:
        stkerr(" sql: statement : ", tos->tex);
        snprintf(T, ERR_MSG_SIZE,
                 " unknown return datatype %d", buffer.type[c]);
        stkerr(" sql: ", T);
        return 0;
    }
    return 1;
} /* 1}}} */
int sql_cmd(sqlite3 *sql_dbh,  /* {{{1 */
            char    *caller_name,
            char    *command) {

int DEBUG = 0;
    sqlite3_stmt *sql_statement;
    int rc, error = 0;

if (DEBUG) gprintf("sql_cmd called by [%s]\n", caller_name);
    rc = sqlite3_prepare(sql_dbh, command, -1, &sql_statement, 0);
if (DEBUG) gprintf("sql_cmd: prepare  rc = %d\n", rc);
    if (rc != SQLITE_OK) {
        stkerr(" sql_cmd: ", "prepare failed");
        error = 1;
    } else {
        rc = sqlite3_step(sql_statement);
if (DEBUG) gprintf("sql_cmd: step     rc = %d\n", rc);
        if (rc != SQLITE_DONE) {
            stkerr(" sql_cmd: ", "step failed");
            error = 1;
        } else {
            rc = sqlite3_finalize(sql_statement);
if (DEBUG) gprintf("sql_cmd: finalize rc = %d\n", rc);
            if (rc != SQLITE_OK) {
                stkerr(" sql_cmd: ", "finalize failed");
                error = 1;
            }
        }
    }

    if (error) {
        stkerr(" sql_cmd SQL command: ", command);
        stkerr(caller_name, (char *) sqlite3_errmsg(sql_dbh));
        return 0;
    } else {
        return 1;
    }
} /* 1}}} */
#endif
