/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

#ifdef X11
/* xterm.c  October 1999

   Copyright (c) 1999   D. R. Williamson

   Some X11 elements adapted from:
   Gnuplot, gplt_x11.c, distributed with Red Hat Linux 5.2
   Copyright 1986 - 1993, 1998   Thomas Williams, Colin Kelley

information about one window/plot, from Gnuplot:
typedef struct plot_struct {
        Window window;
        Pixmap pixmap;
        unsigned int posn_flags;
        int x,y;
        unsigned int width, height; / window size /
        unsigned int px, py; / pointsize /
        int ncommands, max_commands;
        char **commands;
} plot_struct;
*/

#include <stdio.h>
#include <stdlib.h>

#include "main.h"
#ifndef DEPSTK
   #include "stk.h"
#endif

#include "exe.h"
#include "inpo.h"
#ifndef MAINLIB
   #include "lib.h"
#endif
#include "mat.h"
#include "math1.h"
#include "mem.h"
#include "term.h"
#include "tex.h"
#include "tag.h"

/*--------------------------------------------------------------------*/

/* wincreate() */
struct wcb *winlast=NULL;
struct wcb *wintable=NULL;

int bitmapf() /* bitmapf (hP qFile --- ) */
/* Read bitmap file and store elements into pixmap control block. */
{
   Pixmap pix;
   double *hP;
   unsigned int height,width;
   int ret,xhot,yhot;

   if(!Dpy) {
      stkerr(" bitmapf: ",TERMNOT);
      return 0;
   }
   if(tos->typ!=STR) {
      stkerr(" bitmapf: ",STRNOT);
      return 0;
   }
   strchop();

   ret=XReadBitmapFile(Dpy,Root,tos->tex,&width,&height,&pix,
      &xhot,&yhot);

   drop();
   if(tos->typ!=MAT) {
      stkerr(" bitmapf: ",MATNOT);
      XFreePixmap(Dpy,pix);
      return 0;
   }
   if(tos->row!=pcbsize) {
      gprintf(" bitmapf: require hP vector of %d rows",pcbsize); nc();
      stkerr("","");
      XFreePixmap(Dpy,pix);
      return 0;
   }
   dup1s();
   named(); /* name of hP onto stack */
   if(catfetch(tos->tex)==NULL) {
      stkerr(" bitmapf: ",PIXCATNOT);
      drop();
      XFreePixmap(Dpy,pix);
      return 0;
   }
   drop(); /* name qS off stack */
   if(TRACE) {
      gprintf(" bitmap address: %lX",pix); nc();
      gprintf(" bitmap width, height: %d-by%d",width,height); nc();
   }
   hP=tos->mat;
/* THIS IS NOT FINISHED */
   return 1;
}

int bounds(double x1, double y1, double x2, double y2, \
   double xMin, double xMax, double yMin, double yMax, \
   double *X, double *Y)
/* Clip point (x2,y2) of line (x1,y1)-(x2,y2) to keep within bounds
   of rectangle with opposing corners (xMin,yMin), (xMax,yMax). 

   Returns 1 if point is inside rectangle;

   Returns 0 if X and/or Y have been clipped. */
{
   *X=x2;
   *Y=y2;

   if(outside(Y,yMin,yMax)) 
      if(interpolate(x1,y1,x2,y2,X,Y)) return 0;

   if(outside(X,xMin,xMax)) 
      if(interpolate(y1,x1,y2,x2,Y,X)) return 0;

   return 1;
}

int colorpix() /* colorpix (qColor --- pix) */
{
   XColor pixscreen,pixexact;
   int ret;

   if(tos->typ!=STR) {
      stkerr(" color: ",STRNOT);
      return 0;
   }
   strchop();
   ret=XLookupColor(Dpy,DefaultColormap(Dpy,Scr),tos->tex,&pixexact,
      &pixscreen);
   if(!ret) {
      stkerr(" colorpix: ",COLORNOT);
      return 0;
   }
   XAllocNamedColor(Dpy,DefaultColormap(Dpy,Scr),tos->tex,&pixscreen,
      &pixexact);

   return(drop() && pushuint((unsigned long)pixscreen.pixel));
}

int defButtonPressedEvent() /* (hE --- ) */
{
/* From tests: button (BPbu): 1 left, 2 both (or middle), 3 right */

   gprintf(" defButtonPressedEvent"); nc();
   return 1;
}

int defButtonReleasedEvent() /* (hE --- ) */
{
/* From tests: button (BRbu): 1 left, 2 both or middle), 3 right */
   
   gprintf(" defButtonReleasedEvent"); nc();
   return 1;
}

int defClientMessageEvent() /* (hE --- ) */
{
   gprintf(" defClientMessageEvent"); nc();

/*
               if(event.xclient.message_type == WM_PROTOCOLS &&
                  event.xclient.format == 32 &&
                  event.xclient.data.l[0] == WM_DELETE_WINDOW) {
printf(" delete plot.  WM_PROTOCOLS,WM_DELETE_WINDOW: %X,%X\n\r",WM_PROTOCOLS,WM_DELETE_WINDOW);
                  plot_struct *plot = find_plot(event.xclient.window);
                  if (plot) delete_plot(plot);
               }
*/

   return 1;
}

int defConfigureEvent() /* (hE --- ) */
{
   gprintf(" defConfigureEvent"); nc();
   return 1;
}

int defExposeEvent() /* (hE --- ) */
{
   gprintf(" defExposeEvent"); nc();
   return 1;
}

int defKeyPressedEvent() /* (hE --- ) */
{
   gprintf(" defKeyPressedEvent"); nc();
   return 1;
}

int defKeyReleasedEvent() /* (hE --- ) */
{
   gprintf(" defKeyReleasedEvent"); nc();
   return 1;
}

int defMotionEvent() /* (hE --- ) */
{
   gprintf(" defMotionEvent"); nc();
   return 1;
}

int defVisibilityEvent() /* (hE --- ) */
{
/* From tests:
      state (VEst): 
           0 when fully visible,
           1 when partially covered 
           2 when fully covered */

   gprintf(" defVisibilityEvent"); nc();
   return 1;
}
   
int fontload() /* fontload (qS --- 0 | XFont fid) */
/* For given font name, S, loads font and returns XFontStruct pointer,
   XFont, and fontid, fid. */
{
   XFontStruct *font;
   Font fid;

   if(tos->typ!=STR) {
      stkerr(" fontload: ",STRNOT);
      return 0;
   }
   strchop();
   font=XLoadQueryFont(Dpy,tos->tex);
   if(font==NULL) {
      return(drop() && pushint(0));
   }
   fid=font->fid;

   return(
      drop() &&
      pushuint((unsigned long)font) &&
      pushuint((unsigned long)fid) 
   );
}

int GCcreate() /* GCcreate (hW hGC --- ) */
/* Create a graphics context.  Vector hGC is a graphics context block
   (gcb) structure, as defined in xterm.v.  
   Vector hW is a window control block (wcb) as defined in xterm.v.
   Both hW and hGC must be in the catalog.
   While wcb hW is provided here to make GC, this GC can be used for 
   drawing with any other hW of the same depth. */
{
   double *hA,*hGC,*hW;
   XGCValues gcv;
   unsigned long GCmask;
   GC GContext;
   Window win;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" GCcreate: ",MATNOT2);
      return 0;
   }
   hW=(tos-1)->mat;
   win=(Window)*(hW+WIN); /* WIN element of hW holds Window */
   if(!winfind(win,wintable)) {
      stkerr(" GCcreate: ",WINNOT);
      return 0;
   }
   if(tos->row!=gcbsize) {
      gprintf(" GCcreate: require hGC vector of %d rows",gcbsize); 
      nc();
      stkerr("","");
      return 0;
   } 
   dup1s(); /* hGC */
   named(); /* name of GC */
   if((catitem *)catfetch(tos->tex)==NULL) { /* not in catalog */
      stkerr(" GCcreate: ",GCCATNOT);
      drop();
      return 0;
   }
   drop(); /* name qS off stack */

   hGC=tos->mat;
   if(*(hGC+GCC)>0) {
      dup1s();
      GCfree();
   }
/* Graphics context attributes mask (GAM) and vector (GAV): */
   if(*(hGC+GAM)!=-INF) {
   /* Bit pattern of GAM mask element is already unsigned long, so
      just copy it into GCmask: */
      memcpy(&GCmask,(unsigned long *)(hGC+GAM),sizeof(long));

      if(*(hGC+GAV)!=-INF) {
         pushd(*(hGC+GAV));
         exe(); /* putting GC attributes vector on tos */
         if(tos->row!=GAVsize) {
            gprintf(" GCcreate: require attributes vector of %d rows",
               GAVsize); 
            nc();
            stkerr("","");
            return 0;
         }
         hA=tos->mat; /* graphics context attributes vector */
         gcv.function=(int)patint(*(hA+FUNC));
         gcv.plane_mask=patlong(*(hA+PLMASK));
         gcv.foreground=patlong(*(hA+FORE));
         gcv.background=patlong(*(hA+BACK));
         gcv.line_width=patint(*(hA+LINEWID));
         gcv.line_style=patint(*(hA+LINESTY));
         gcv.cap_style=patint(*(hA+CAPSTY));
         gcv.join_style=patint(*(hA+JOINSTY));
         gcv.fill_style=patint(*(hA+FILSTY));
         gcv.fill_rule=patint(*(hA+FILRUL));
         gcv.arc_mode=patint(*(hA+ARCMOD));
         gcv.tile=(Pixmap)patlong(*(hA+PIXTILE));
         gcv.stipple=(Pixmap)patlong(*(hA+PIXSTIP));
         gcv.ts_x_origin=patint(*(hA+TSX));
         gcv.ts_y_origin=patint(*(hA+TSY));
         gcv.font=(Font)patlong(*(hA+FONT));
         gcv.subwindow_mode=patint(*(hA+SUBMOD));
         gcv.graphics_exposures=(Bool)patint(*(hA+EXPOSE));
         gcv.clip_x_origin=patint(*(hA+CLIPX));
         gcv.clip_y_origin=patint(*(hA+CLIPY));
         gcv.clip_mask=(Pixmap)patlong(*(hA+CLIPMASK));
         gcv.dash_offset=patint(*(hA+DASHOFF));
         gcv.dashes=(char)patint(*(hA+DASHES));
         drop();
      }
      else {
         stkerr(" GCcreate: ",GCATTNOT);
         return 0;
      }
   }
   else {
      GCmask=GCFunction | GCBackground | GCForeground;
      gcv.function=GXcopy;
      gcv.background=XBlackPixel(Dpy,Scr);
      gcv.foreground=XWhitePixel(Dpy,Scr);
   }
/* Creating GC from GCmask and properties in gcv: */
   GContext=XCreateGC(Dpy,win,GCmask,&gcv);
   *(hGC+GCC)=(long)GContext;

   return(drop2());
}

int GCfore() /* GCfore (pix hGC --- ) */
/* Set the foreground color in GC. */
{
   GC GCon;
   double *hGC,*hGAV;
   long pix;

   pix=(long)(tos-1)->real;

   hGC=tos->mat;
   pushd(*(hGC+GAV));
   if(!exe()) return 0; /* putting GC attributes vector, gav, on tos */

   hGAV=tos->mat;
   *(hGAV+FORE)=(double)pix;

   if(*(hGC+GCC)!=-INF) {
      GCon=(GC)patlong(*(hGC+GCC)); /* GCC element of hGC holds GC */

      XSetForeground(Dpy,GCon,pix);
   }
   return(drop() && drop2());
}

int GCfree() /* GCfree (hGC --- ) */
/* Free a graphics context. */
{
   double *hGC;
   GC GContext;

   if(tos->typ!=MAT) {
      stkerr(" GCfree: ",MATNOT);
      return 0;
   }
   hGC=tos->mat;
   if(*(hGC+GCC)<=0) return(drop());
   
   GContext=(GC)patlong(*(hGC+GCC));
   XFreeGC(Dpy,GContext);
   *(hGC+GCC)=-INF;

   return(drop());
}

int GCline() /* GCline (linestyle hGC --- ) */
/* Set line style in GC. */
{
   GC GCon;
   double *hGC,*hGAV;
   int line_width,line_style,cap_style,join_style;

   hGC=tos->mat;

   pushd(*(hGC+GAV));
   if(!exe()) return 0; /* putting GC attributes vector, gav, on tos */

   hGAV=tos->mat;
   *(hGAV+LINESTY)=(tos-2)->real;

   if(*(hGC+GCC)!=-INF) {

      line_width=patint(*(hGAV+LINEWID));
      line_style=patint(*(hGAV+LINESTY)); /* new value */
      cap_style=patint(*(hGAV+CAPSTY));
      join_style=patint(*(hGAV+JOINSTY));

      GCon=(GC)patlong(*(hGC+GCC)); /* GCC element of hGC holds GC */

      XSetLineAttributes(Dpy,GCon,line_width,line_style,
         cap_style,join_style);
   }
   return(drop() && drop2());
}

int interpolate(double x1, double y1, double x2, double y2, \
   double *X2, double *Y2)
/* Interpolate X2 to go with Y2 on the line defined by end points
   (x1,y1) and (x2,y2). */
{
   if(y2==y1) return 1;
   *X2=x2 + (x2-x1)*(*Y2-y2)/(y2-y1);
   return 0;
}

int line() /* line (hW hGC hY hX --- ) */
/* Draw line segments from X(i),Y(i) to X(i+1),Y(i+1) in window W
   with graphics context GC. */
{
   register Window win;
   register GC GCon;
   register int k=0;
   register double *X,*Y;
   double *hW,*hGC;
   int lines;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || \
      (tos-2)->typ!=MAT || (tos-3)->typ!=MAT) {
      stkerr(" line: ",STKNOT);
      return 0;
   }
   hW=(tos-3)->mat;
   win=(Window)patlong(*(hW+WIN)); /* WIN element of hW holds Window */

   hGC=(tos-2)->mat;
   GCon=(GC)patlong(*(hGC+GCC)); /* GCC element of hGC holds GC */

   lines=(tos->row)-1;

/* Drawing lines from right to left */
   Y=((tos-1)->mat)+lines-1;
   X=(tos->mat)+lines-1;

   for(;k<lines;k++) {
      XDrawLine(Dpy,win,GCon,(short)*(X+1),(short)*(Y+1),
         (short)*(X),(short)*(Y));
      X--;
      Y--;
   }
   return(drop2() && drop2());
}

int linec() /* line (hW hGC hY hX hL--- ) */
/* In window W with graphics context GC, draw line segments from 
   X(i),Y(i) to X(i+1),Y(i+1) clipped to rectangle limits given 
   in L. */
{
   register Window win;
   register GC GCon;
   register int k=0;
   register double *X=0,*Y=0;
   double *hW,*hGC,X1,X2,Y1,Y2,Xmin,Xmax,Ymin,Ymax;
   int lines;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT || \
      (tos-3)->typ!=MAT || (tos-4)->typ!=MAT) {
      stkerr(" line: ",STKNOT);
      return 0;
   }
/* Loading the clip rectangle limits from L: */
   X=tos->mat;
   Xmin=MIN(*X,*(X+1));
   Xmax=MAX(*X,*(X+1));
   Ymin=MIN(*(X+2),*(X+3));
   Ymax=MAX(*(X+2),*(X+3));
   drop(); /* L off stack */

   hW=(tos-3)->mat;
   win=(Window)patlong(*(hW+WIN)); /* WIN element of hW holds Window */
   
   hGC=(tos-2)->mat;
   GCon=(GC)patlong(*(hGC+GCC)); /* GCC element of hGC holds GC */

/* Drawing lines from right to left */
   lines=tos->row-1;
   X=(tos->mat)+lines-1;
   Y=((tos-1)->mat)+lines-1;
   
   for(k=0;k<lines;k++) {
      if(bounds(*(X),*(Y),*(X+1),*(Y+1),
         Xmin,Xmax,Ymin,Ymax,&X2,&Y2)) {

         if(bounds(*(X+1),*(Y+1),*(X),*(Y),
            Xmin,Xmax,Ymin,Ymax,&X1,&Y1)) {

            if(!(X1==X2 && Y1==Y2)) 
               XDrawLine(Dpy,win,GCon,(short)X2,(short)Y2,
                  (short)X1,(short)Y1);
         }
      }
      X--;
      Y--;
   }
   return(drop2() && drop2());
}

int pminmax() /* pminmax (hA --- min max) */
/* Fetching smallest and largest terms in matrix, and ignoring UNDEF. */
{
   register double *A,max=-INF,min=INF;
   int k=0,len,rows;

   if(tos->typ!=MAT) {
      stkerr(" pminmax: ",MATNOT);
      return 0;
   }
   A=tos->mat;
   rows=tos->row;
   len=rows*tos->col;

   for(k=0;k<len;k++) {
      if((ABS(*A)) != (ABS(UNDEF))) {
         if(*A<min) min=*A;
         if(*A>max) max=*A;
      }
      A++;
   }
   return(
      drop() &&
      push(NUM,NULL,NOTAG,min,NULL,NULL,0,0,NULL) &&
      push(NUM,NULL,NOTAG,max,NULL,NULL,0,0,NULL)
   );
}

int wdemo() /* wdemo ( --- ) */
{
   Window win1;
   XEvent event;
   XSetWindowAttributes attributes;
   Cursor cursor_shape;
   XFontStruct *fontinfo;
   GC gr_context1,gr_context2;
   XGCValues gr_values;

   attributes.background_pixel=XWhitePixel(Dpy,Scr);
   attributes.background_pixel=25; /* red */
   attributes.border_pixel=XBlackPixel(Dpy,Scr);
   attributes.border_pixel=26; /* green */
 
   win1=XCreateWindow(Dpy,Root,0,0,
        XDisplayWidth(Dpy,Scr)-400,
        XDisplayHeight(Dpy,Scr)-200,
        30,Dep,InputOutput,Vis,
        CWBackPixel | CWBorderPixel,&attributes);
   XSelectInput(Dpy,win1,ExposureMask | KeyPressMask);
   
   gr_values.function=GXcopy;
   gr_values.plane_mask=AllPlanes;
   gr_values.foreground=BlackPixel(Dpy,Scr);
   gr_values.background=WhitePixel(Dpy,Scr);

   gr_context1=XCreateGC(Dpy,win1,
               GCFunction | GCPlaneMask | GCForeground | GCBackground,
               &gr_values);

   gr_values.function=GXxor;
   gr_values.foreground=WhitePixel(Dpy,Scr);
   gr_values.background=BlackPixel(Dpy,Scr);
   gr_context2=XCreateGC(Dpy,win1,
               GCFunction | GCPlaneMask | GCForeground | GCBackground,
               &gr_values);

   fontinfo=XLoadQueryFont(Dpy,"6x10");
   
   cursor_shape=XCreateFontCursor(Dpy,XC_heart);
   XDefineCursor(Dpy,win1,cursor_shape);

   XSetFont(Dpy,gr_context1,fontinfo->fid);
   XSetFont(Dpy,gr_context2,fontinfo->fid);

   XMapWindow(Dpy,win1);

   while(1) {
      XNextEvent(Dpy,&event);

      switch(event.type) {
      case Expose:
         XClearWindow(Dpy,win1);
         XDrawString(Dpy,win1,gr_context1,50,50,"Hello",5);
         XDrawImageString(Dpy,win1,gr_context2,20,20,"Hello",5);

         XFillRectangle(Dpy,win1,gr_context1,150,150,111,111);
         XFillRectangle(Dpy,win1,gr_context2,200,180,111,111);
         break;

      case KeyPress:
         return 1;
      }
   }
}

int wdemo1() /* wdemo ( --- ) */
/* Run a standalone X window demo */
{
   Window window;
   XSetWindowAttributes attributes;
   XGCValues gr_values;
   GC gr_context;
   XFontStruct *fontinfo;
   XEvent event;
   XColor color,dummy;
/*
   unsigned long width=350,height=200;
*/
   unsigned long width=50,height=50;

/* Testing pixmap: */
   int retpix,xhot,yhot;
   unsigned int hei,wid;
   Pixmap pix;

/* Testing XImage: */
   XImage *x;
   Drawable d;

   attributes.background_pixel=XWhitePixel(Dpy,Scr);

   window=XCreateWindow(Dpy,Root,
      200,200,width,height,5,Dep,InputOutput,Vis,
      CWBackPixel,&attributes);

   XSelectInput(Dpy,window,ExposureMask | KeyPressMask
       | ButtonPressMask);
   fontinfo=XLoadQueryFont(Dpy,"6x10");

   XAllocNamedColor(Dpy,DefaultColormap(Dpy,Scr),
      "black",&color,&dummy);

   gr_values.font=fontinfo->fid;
printf(" gr_values.font: %X\n\r",(unsigned int)gr_values.font);
   gr_values.foreground=color.pixel;
   gr_values.background=XWhitePixel(Dpy,Scr);
   gr_context=XCreateGC(Dpy,window,(GCForeground | GCBackground | GCFont),&gr_values);
   XFlush(Dpy);
   XMapWindow(Dpy,window);
   XFlush(Dpy);

/* reading a pixmap: */
   retpix=XReadBitmapFile(Dpy,Root,
      "/usr/include/X11/bitmaps/woman",&wid,&hei,&pix,&xhot,&yhot);
   printf(" retpix,wid,hei,xhot,yhot: %d,%d,%d,%d,%d\n\r",
      retpix,wid,hei,xhot,yhot);
   printf(" window,pix,: %X,%X\n\r",(unsigned int)window, \
      (unsigned int)pix);

/* get an image: */
/*
This bombs without displaying window first:
   ximage=XGetImage(Dpy,window,0,0,width,height,AllPlanes,XYPixmap);
   printf(" ximage.width,ximage.height: %d,%d\n\r",ximage->width,ximage->height);

But getting pixmap is ok:
   x=XGetImage(Dpy,pix,0,0,75,75,AllPlanes,XYPixmap);
   gprintf("    width, height: %d-by-%d",x->width,x->height); nc();
*/

   while(1) {
      XNextEvent(Dpy,&event);

      switch(event.type) {

      case ButtonPress:

/* XYPixmap uses 1 bit per pixel, ZPixmap uses 8 bits per pixel, and
   therefore has about 8 times as many bytes_per_line to hold them.

   Origin x, plus width-x must not be greater than width; probably
   true for y and height too.
 */

      d=window;
      x=XGetImage(Dpy,d,50,0,width-50,height,AllPlanes,XYPixmap);

   d=pix;
   x=XGetImage(Dpy,d,0,0,75,75,AllPlanes,XYPixmap);

      /* this code will go into word XImage: */
   gprintf(" Properties of drawable %X",d); nc();
   gprintf("    width, height: %d-by-%d",x->width,x->height); nc();
   gprintf("    xoffset: %d",x->xoffset); nc();
   gprintf("    format: %d",x->format); nc();
   gprintf("    (formats: %d=XYBitmap, %d=XYPixmap, %d=ZPixmap)", \
      XYBitmap,XYPixmap,ZPixmap); nc();
   gprintf("    data: %X",x->data); nc();
   gprintf("    byte_order: %d",x->byte_order); nc();
   gprintf("    bitmap_bit_order: %d",x->bitmap_bit_order); nc();
   gprintf("    (byte and bit order: %d=LSBFirst, %d=MSBFirst)", \
      LSBFirst,MSBFirst); nc();
   gprintf("    bitmap_pad: %d",x->bitmap_pad); nc();
   gprintf("    (bitmap pad: 8, 16, 32 either XY or ZFormat)"); nc();
   gprintf("    depth: %d",x->depth); nc();
   gprintf("    bytes_per_line: %d",x->bytes_per_line); nc();
   gprintf("    bits_per_pixel(Z): %d",x->bits_per_pixel); nc();
   gprintf("    Bits in Z arrangement:"); nc();
   gprintf("       red_mask(Z): %X",x->red_mask); nc();
   gprintf("       green_mask(Z): %X",x->green_mask); nc();
   gprintf("       blue_mask(Z): %X",x->blue_mask); nc();
   gprintf("    f.create_image: %X",x->f.create_image); nc();
   gprintf("    f.destroy_image: %X",x->f.destroy_image); nc();
   gprintf("    f.get_pixel: %X",x->f.get_pixel); nc();
   gprintf("    f.put_pixel: %X",x->f.put_pixel); nc();
   gprintf("    f.sub_image: %X",x->f.sub_image); nc();
   gprintf("    f.add_pixel: %X",x->f.add_pixel); nc();

   XDestroyImage(x);

   break;

      case Expose:
/*
         XCopyPlane(Dpy,pix,window,gr_context,0,0,wid,hei,50,50,1);
*/
         XCopyPlane(Dpy,pix,window,gr_context,0,0,wid,hei,0,0,1);
      break;

/* XCopyPlane copies up to wid-by-hei bits of pix, and no more; it
   does not stretch to fit larger rectangle. */


/*
      case Expose:
         XDrawLine(Dpy,window,gr_context,0,0,100,100);
         XDrawString(Dpy,window,gr_context,100,100,"hello",5);
      break;
*/
      case KeyPress:
/*
         XCloseDisplay(Dpy);
*/
      return(pushuint((unsigned long)window));
      }
   }
}

int winadd(struct wcb *new, struct wcb **table, struct wcb **end)
/* Adding a new element to linked list of windows being managed.

   Making a doubly linked list in sorted order, after:
      Schildt, H., "C: The Complete Reference,"
      Osborne McGraw-Hill, 1995

   Fancier than required, but useful as a model for general
   linked lists. */
{
   register struct wcb *old,*p;

   if(*end==NULL) { /* very 1st element in list */
      new->nex=NULL;
      new->pre=NULL;
      *end=new;
      *table=new;
      return 1;
   }
   p=*table;
   old=NULL;
   while(p) {
      if(strcmp(p->cat->nam,new->cat->nam)<0) {
         old=p;
         p=p->nex;
      }
      else { /* error if already have nam */
         if(strcmp(p->cat->nam,new->cat->nam)==0) {
            stkerr(" winadd: ",WINDUPE);
            return 0;
         }
         if(p->pre) { /* inserting in front of p */
            p->pre->nex=new;
            new->nex=p;
            new->pre=p->pre;
            p->pre=new;
            return 1;
         }
         new->nex=p; /* inserting in front of 1st element */
         new->pre=NULL;
         p->pre=new;
         *table=new;
         return 1;
      }
   }
   old->nex=new; /* putting at end */
   new->nex=NULL;
   new->pre=old;
   *end=new;
   return 1;
}

int winclear() /* winclear (hW x y w h) */
/* Clear a rectangle w-by-h at location x,y */
{
   Window win;
   struct wcb *wincb;
   int exposure=0,x,y;
   int h,w;

   if(!popint((int *)&h) || !popint((int *)&w) ||
      !popint(&y) || !popint(&x)) return 0;

   win=(Window)*((tos->mat)+WIN); /* WIN element holds Window */
   wincb=winfind(win,wintable); /* find win in linked list */

   if(wincb) {
      XClearArea(Dpy,win,x,y,w,h,(Bool)exposure);
      return(drop());
   }
   stkerr(" winclear: ",WINUNINIT);
   return 0;
}

int wincopy() /* wincopy (hWfrom hWto hGC --- ) */
/* Copy window Wfrom to window Wto. */
{
   Window Wfrom,Wto;
   GC GCon;
   double *hWto;

   GCon=(GC)patlong(*((tos->mat)+GCC)); /* GCC element holds GC */
   drop();

   hWto=tos->mat;
   Wto=(Window)*(hWto+WIN);

   Wfrom=(Window)*(((tos-1)->mat)+WIN);
   
   XCopyArea(Dpy,Wfrom,Wto,GCon,0,0,
      (unsigned int)*(hWto+W),(unsigned int)*(hWto+H),0,0);

   return(drop2());
}

int wincreate() /* wincreate (hW qTitle --- ) */
/* Create a window.  Vector hW is a window control block (wcb) struc-
   ture, as defined in xterm.v, that is booked in the catalog.
   If the window in hW exists, it is destroyed first. */
{
   double *hW,*hA;
   char *S,*Title;
   unsigned long CWmask;
   struct wcb *p;
   XSetWindowAttributes watt;

   if(tos->typ!=STR) {
      stkerr(" wincreate: ",STRNOT);
      return 0;
   }
   strchop();

   if((tos-1)->typ!=MAT) {
      stkerr(" wincreate: ",MATNOT);
      return 0;
   }
   if((tos-1)->row!=wcbsize) {
      gprintf(" wincreate: require hW vector of %d rows",wcbsize); 
      nc();
      stkerr("","");
      return 0;
   } 
   if((Title=(char *)memgetn(tos->tex,tos->col))==NULL) {
      return 0;
   }
   if(!*Title) {
      mallfree((void *)&Title);
   }
   drop(); /* qTitle off stack */

   dup1s(); /* hW */
   named(); /* S is name of hW: */
   if((S=(char *)memgetn(tos->tex,tos->col))==NULL) {
      drop();
      mallfree((void *)&Title);
      return 0;
   }
   drop(); /* name qS off stack */

   if(winfindname(S,wintable)) { 
      if(!( /* destroying existing window first: */
         dup1s() &&
         winfree()
      )) {
         stkerr(" wincreate: ",WINDESNOT);
         mallfree((void *)&Title);
         mallfree((void *)&S);
         return 0;
      }
   }
   if((p=malloc(sizeof(struct wcb)))==NULL) { /* item for linked wcb */
      stkerr(" wincreate: ",MEMNOT);
      mallfree((void *)&S);
      return 0;
   }
   p->cat=(catitem *)catfetch(S); /* hW must be in catalog */
   mallfree((void *)&S);
   if(p->cat==NULL) {
      stkerr(" wincreate: ",WINCATNOT);
      mallfree((void *)&Title);
      mallfree((void *)&p);
      return 0;
   }
   hW=tos->mat; /* structure of wcb (xterm.v) and enum _wcb (term.h) */
   p->vis=0; /* window has not been mapped */

   if(*(hW+W)==-INF) *(hW+W)=gW;
   if(*(hW+H)==-INF) *(hW+H)=gH;
   if(*(hW+X)==-INF) *(hW+X)=gX;
   if(*(hW+Y)==-INF) *(hW+Y)=gY;
   if(*(hW+FLAGS)==-INF) *(hW+FLAGS)=gFlags;

/* Window attributes mask (WAM) and attributes vector (WAV): */
   if(*(hW+WAM)!=-INF) {
   /* Bit pattern of WAM mask element is already unsigned long, so 
      just copy it into CWmask: */
      memcpy(&CWmask,(unsigned long *)(hW+WAM),sizeof(long));
      if(*(hW+WAV)!=-INF) {
         pushd(*(hW+WAV));
         exe(); /* putting attributes vector on tos */
         if(tos->row!=WAVsize) {
            gprintf(" wincreate: require attributes vector of %d rows",
               WAVsize);
            nc();
            stkerr("","");
            mallfree((void *)&Title);
            mallfree((void *)&p);
            return 0;
         }
         hA=tos->mat; /* window attributes vector */
         watt.background_pixmap=(Pixmap)patlong(*(hA+BACKPMAP));
         watt.background_pixel=patlong(*(hA+BACKPIX));
         watt.border_pixmap=(Pixmap)patlong(*(hA+BORDPMAP));
         watt.border_pixel=patlong(*(hA+BORDPIX));
         watt.bit_gravity=patint(*(hA+BGRAV));
         watt.win_gravity=patint(*(hA+WGRAV));
         watt.backing_store=patint(*(hA+BACKSTOR));
         watt.backing_planes=patlong(*(hA+BACKGPLANE));
         watt.backing_pixel=patlong(*(hA+BACKGPIX));
         watt.save_under=(Bool)patint(*(hA+SAVEU));
         watt.event_mask=(long)patint(*(hA+EVENTM));
         watt.do_not_propagate_mask=(long)patint(*(hA+PROPNOM));
         watt.override_redirect=(Bool)patint(*(hA+OVERRIDE));
         watt.colormap=(Colormap)patlong(*(hA+COLORM));
         watt.cursor=(Cursor)patlong(*(hA+CURSOR));
         drop();
      }
      else {
         stkerr(" wincreate: ",WINATTNOT);
         mallfree((void *)&Title);
         mallfree((void *)&p);
         return 0;
      }
   }
   else {
      CWmask=CWBackPixel;
      watt.background_pixel=XBlackPixel(Dpy,Scr);
   }
/* Creating window from properties in wcb, CWmask, and watt: */
   p->win=wcreate(Title,(unsigned int)*(hW+FLAGS),(int)*(hW+X),
      (int)*(hW+Y),(unsigned int)*(hW+W),(unsigned int)*(hW+H),
      CWmask,&watt);

   *(hW+WIN)=(long)p->win; /* Window ptr into win element of hW */

   drop(); /* hW off the stack */
   mallfree((void *)&Title);

   if(winadd(p,&wintable,&winlast)) return 1;
   else XDestroyWindow(Dpy,(Window)p->win);
   mallfree((void *)&p);
   return 0;
}

int windisp() /* windows ( --- ) */
/* Displaying window properties. */
{
   if(wintable) winlist(wintable);
   else {
      gprintf(WINSNOT);
      nc();
   }
   return 1;
}

struct wcb *winfind(Window window, struct wcb *table)
/* Finding window in linked list of windows. */
{
   register struct wcb *p;

   if(!window) return(struct wcb *)NULL;

   p=table;
   while(p) {
      if(p->win==window) break;
      p=p->nex;
   }
   return p;
}

struct wcb *winfindname(char *nam, struct wcb *table)
/* Finding named window in linked list of windows. */
{
   register struct wcb *p;
   char *nam1;

   p=table;
   nam1=tagged(nam,"CODE__"); 
   while(p) {
      if(strcmp(p->cat->nam,nam1)==0) break;
      p=p->nex;
   }
   if(p) return p;

   p=table;
   nam1=tagged(nam,"DATA__"); 
   while(p) { 
      if(strcmp(p->cat->nam,nam1)==0) break;
      p=p->nex;
   }
   return p;
}

int winfree() /* winfree (hW --- ) */
/* Destroy window and remove it from wintable linked list. */
{
   Window win;
   struct wcb *p;
   double *hW;

   if(tos->typ!=MAT) {
      stkerr(" winfree: ",MATNOT);
      return 0;
   }
   hW=tos->mat;
   win=(Window)*(hW+WIN); /* WIN element of hW holds Window */

   p=winfind(win,wintable);
   if(!p) {
      stkerr(" winfree: ",WINNOT);
      return 0;
   }
   if(p->win) { /* p->win is the same as win=(Window)*(hW+WIN) */
      XDestroyWindow(Dpy,p->win);
      *(hW+WIN)=-INF;
      p->vis=0;
   }
   winrem(p,&wintable,&winlast);
   return(drop());
}

void winlist(struct wcb *windows)
/* Displaying items in linked list of windows. */
{
   register struct wcb *p;

   p=windows;
   gprintf(" Windows:"); nc();

   while(p) {
      pushq2(p->cat->nam,strlen(p->cat->nam));
      notag();
      gprintf("  %s: Window=%lX",tos->tex,p->win);
      drop(); nc();
      p=p->nex;
   }
}

int winraise() /* winraise (hW --- ) */
{
   Window win;
   struct wcb *wincb;

   win=(Window)*((tos->mat)+WIN); /* WIN element holds Window */
   wincb=winfind(win,wintable); /* find win in linked list */

   if(wincb) {
      XRaiseWindow(Dpy,win);
      return(drop());
   }
   stkerr(" winraise: ",WINUNINIT);
   return 0;
}

void winrem(struct wcb *p, struct wcb **table, struct wcb **end)
/* Removing an item from a linked list. */
{
   if(*table==p) {
      *table=p->nex;
      if(*table) (*table)->pre=NULL;
      else *end=NULL;
   }
   else {
      p->pre->nex=p->nex;
      if(p!=*end) p->nex->pre=p->pre;
      else *end=p->pre;
   }
   mallfree((void *)&p);
}

int winresize() /* winresize (hW w h --- ) */
{ 
   Window win;
   struct wcb *wincb;
   int h,w;

   if(!popint((int *)&h) || !popint((int *)&w)) return 0;

   win=(Window)*((tos->mat)+WIN); /* WIN element holds Window */
   wincb=winfind(win,wintable); /* find win in linked list */

   if(wincb) {
      XResizeWindow(Dpy,win,w,h);
      return(drop());
   }
   stkerr(" winresize: ",WINUNINIT);
   return 0;
}

int winshow() /* winshow (hW --- ) */
{
   Window win;
   struct wcb *wincb;
   double *hW;

   if(tos->typ!=MAT) {
      stkerr(" winshow: ",MATNOT);
      return 0;
   }
   hW=tos->mat;
   win=(Window)*(hW+WIN); /* WIN element of hW holds Window */
   wincb=winfind(win,wintable); /* find win in linked list */

   if(!wincb) {
      stkerr(" winshow: ",WINNOT);
      return 0;
   }
   if(wincb->vis==1) return(drop()); /* window already visible */

   if(win) {
      XMapWindow(Dpy,win);
      wincb->vis=1; /* visible */
      return(drop());
   }
   stkerr(" winshow: ",WINUNINIT);
   return 0;
}

void XEventHandler(int event) /* (hE --- ) */
/* Handling an event that occurred in a window; hE on stack contains
   event info.
   Prototype file term.h contains the enumerations that define the 
   elements of E for various events (prototype for XEventHandler() 
   is also in term.h).
   The first element of E is always the window of the event. */
{
   Window win;
   XWindowAttributes watt;
   struct wcb *wincb;
   double *hW,*hH=NULL;
   #define NoHandlers -1
   int handler;
   
   win=(Window)*(tos->mat); /* from within, so no stack check first */

   if(!(wincb=winfind(win,wintable))) {
      drop();
      return;
   }
   if(WTRACE) {
      gprintf(" XEventHandler.  Event, window, wcb: %d, %s, %lX", \
         event,wincb->cat->nam,(unsigned int)win); nc();
   }
   oncat=wincb->cat; /* need this globally */
   (*(unsigned long (*)())oncat->exe)(); /* puts hW on stack */

   hW=tos->mat;
   if(*(hW+ECB)<0) { /* ptrs for exe() are always negative; see ptr() */
      handler=event;
      pushd(*(hW+ECB)); /* ptr to stack */
      exe(); /* puts hH on stack, list of event handlers */ 
      hH=tos->mat; 
      /* (hE hW hH) current stack elements */
   }
   else {
      handler=NoHandlers; 
   /* (hE hW) current stack elements */
   }
   switch(handler) { 

      case ButtonPress:
         pushd(*(hH+BP)); /* ptr */
      break;

      case ButtonRelease:
         pushd(*(hH+BR)); /* ptr */
      break;

      case ClientMessage:
         pushd(*(hH+CM)); /* ptr */
      break;

      case ConfigureNotify:
         pushd(*(hH+CE)); /* ptr */
      break;

      case Expose:

      /* Keep window size up to date for scaling */
         XGetWindowAttributes(Dpy,win,&watt);
         hW=(tos-1)->mat;
         *(hW+W)=(double)watt.width;
         *(hW+H)=(double)watt.height;

         pushd(*(hH+EE)); /* ptr */
      break;

      case KeyPress:
         pushd(*(hH+KP)); /* ptr */
      break;

      case KeyRelease:
         pushd(*(hH+KR)); /* ptr */
      break;

      case MotionNotify:
         pushd(*(hH+ME)); /* ptr */
      break;

      case VisibilityNotify: 
         pushd(*(hH+VE)); /* ptr */
      break;

      default:
         drop2(); drop(); /* dropping (hE hW hH) stack elements */
      return;

      case NoHandlers:
         drop2(); /* dropping (hE hW) stack elements */
      return;
   }
/* hE hW hH ptr */ lop(); 

/* hE hW ptr */ if(tos->real) exe(); /* run the handler */ 

                else { /* clear stk; no handler if ptr is zero: */
                   drop2();
                   drop();
                }
   return;
   #undef NoHandlers
}

int Ximageprops() /* XImage (d --- ) */
/* Display XImage structure for Drawable d. */
{
   unsigned long d;

   if(!popuint(&d)) return 0;

   gprintf(" Properties of drawable %lX",d); nc();
/*
   gprintf("    width, height: %d-by-%d",d.width,d.height); nc();
*/
return 1;

}

int XRegionFree() /* winregionfree (hR --- ) */
/* Destroy the region created by XRegionSet. */
{
   Region region;
   unsigned long R;

   if(!popuint(&R)) return 0;
   
   region=(Region)R;
   XDestroyRegion(region);
   return 1;
}

int XRegionSet() /* winregion (hEv hGC --- hR) */
/* Defines region for rectangle returned in expose event data, eventEE.
   Using region to redraw window has not been successful: clipped 
   regions are left undrawn. */
{
   Region region;
   XRectangle rect;
   double *Ev;
   GC GCon;

   region=XCreateRegion();
   GCon=(GC)patlong(*((tos->mat)+GCC)); /* GCC element holds GC */
   
   Ev=(tos-1)->mat;
   rect.x=(short)*(Ev+EEx);
   rect.y=(short)*(Ev+EEy);
   rect.width=(unsigned short)*(Ev+EEw);
   rect.height=(unsigned short)*(Ev+EEh);

   XUnionRectWithRegion(&rect,region,region);
   XSetRegion(Dpy,GCon,region);

   return(
      drop2() &&
      pushuint((unsigned long)region)
   );
}

/*
Creating wdraw, draw in window:

GC gc=(GC)0;

int wdraw()
{
Notes from display(plot), Xmain.c:

Using struct plot_struct *plot;

double xscale, yscale, pointsize;

    * set scaling factor between internal driver & window geometry *
   xscale=plot->width/4096.0;  yscale=plot->height/4096.0

   scale units of picture to the window
   px=(int)xscale*pointsize;
   py=(int)yscale*pointsize;

   create new GC (graphics context) and new pixmap:
   if(gc) XFreeGC(Dpy,gc);

   gc=XCreateGC(Dpy,plot->pixmap,0,(XGCValues *)0);
   XSetFont(Dpy,gc,font->fid);

   * set pixmap background *
   XSetForeground(Dpy,gc,colors[0]);
   XFillRectangle(Dpy,plot->pixmap,gc,0,0,plot->width,plot->height);
   XSetBackground(Dpy,gc,colors[0]);


In these, Raise and Clear should be set in wcreate:
   * top the window but don't put keyboard or mouse focus into it. *
   if(win->Raise) XMapRaised(Dpy,plot->window);
   * momentarily clear the window first if requested *
   if(win->Clear) {
      XClearWindow(Dpy,plot->window);
      XFlush(Dpy);
   }
*/
#endif
