/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

#if defined(ESSL) || defined(LAPACK)
/* essl.c  March 2001

   Copyright (c) 2001   D. R. Williamson

   Reference:
      Engineering and Scientific Subroutine Library, Version 2,
      Release 2, Second Edition (January 1994), IBM Corporation.

   LAPACK interface:  Albert Danial  May 2001
*/

#include <math.h>
#include <stdio.h>
#include <string.h>     /* strlen */
#include <stdlib.h>
#include <unistd.h>

#include "main.h"
#include "stk.h"
#include "essl.h"

#include "ctrl.h"
#include "exe.h"
#include "inpo.h"
#include "math1.h"
#include "mmath.h"
#include "mem.h"
#include "ordering.h"
#include "sys.h"
#include "tag.h"

int ceigSSL() /* ceigSSL (hC f --- hLr hLi hAr hAi) */
/* Eigenanalysis of real matrix C.  Resulting eigenvalues and eigen-
   vectors are complex. */
{
   double *Ai, *Ar, *Aux, *C, *Li, *Lr, *Mag;
#if   defined(ESSL)
   double *L, *A;
#elif defined(LAPACK)
   double *lapack_eigvec;
   char    chr_N       = 'N';
   char    chr_V       = 'V';
   double  dummy[1];
   double  double_0    =  0.0;
   double  double_m1   = -1.0;
   int     Aux_size, info;
#endif
   int     int_1       =   1;
   int     f,i=0,j=0,rows;
#ifdef ESSL
   int     int_2       =   2;
   int *sel;
#endif

   if(!popint(&f)) return 0; /* flag for balancing is not used yet */

   if(tos->typ!=MAT) {
      stkerr(" ceigSSL: ",MATNOT);
      return 0;
   }
   rows=tos->row;
   if(rows!=tos->col) {
      stkerr(" ceigSSL: ",SQUNOT);
      return 0;
   }
   cop(); /* C will be overwritten, so using a copy */
   C=tos->mat;

   if(!matstk(rows,   1,"_Lr")) return 0;
   Lr=tos->mat;
   if(!matstk(rows,   1,"_Li")) return 0;
   Li=tos->mat;

/* Computing complex eigenvalues and eigenvectors: */
#if   defined(ESSL)

   /* workspace array: */
   if(!matstk(rows,3,"_Aux")) return 0;
   Aux=tos->mat;

   /* array for complex eigenvalues: */
   if(!matstk(rows*2,1,"_L")) return 0;
   L=tos->mat;

   /* array for complex eigenvectors: */
   if(!matstk(rows*2,rows,"_A")) return 0;
   A=tos->mat;

   dgeev(1,C,rows,(dcmplx*) L,A,rows,sel,rows,Aux,3*rows);

   pushq2("5 roll drop",11); xmain(0); /* C no longer needed */
   pushq2("2 roll drop",11); xmain(0); /* Aux no longer needed */

   if(!matstk(rows,rows,"_Ar")) return 0;
   Ar=tos->mat;
   if(!matstk(rows,rows,"_Ai")) return 0;
   Ai=tos->mat;

   for (j = 0; j < rows; j++) {
       Lr[j] = L[2*j  ];
       Li[j] = L[2*j+1];
       DCOPY( rows, &A[2*j*rows  ], int_2, &Ar[j*rows], int_1);
       DCOPY( rows, &A[2*j*rows+1], int_2, &Ai[j*rows], int_1);
   }
   pushq2("2 roll drop",11); xmain(0); /* A no longer needed */
   pushq2("2 roll drop",11); xmain(0); /* L no longer needed */

#elif defined(LAPACK)

   /* workspace array: */
   if(!matstk(rows,4,"_Aux")) return 0;
   Aux_size = 4*rows; /* minimum value; need more for better performance */
   Aux=tos->mat;

   if(!matstk(rows,rows,"_lapack_eigvec"))    return 0;
   lapack_eigvec=tos->mat;

   DGEEV(&chr_N, &chr_V, &rows, C, &rows, Lr, Li,
         dummy, &int_1, lapack_eigvec, &rows, Aux, &Aux_size, &info);

   pushq2("4 roll drop",11); xmain(0); /* get rid of C,   no longer needed */
   swap(); drop();                     /* get rid of Aux, no longer needed */

   if(!matstk(rows,rows,"_Ar")) return 0;
   Ar=tos->mat;
   if(!matstk(rows,rows,"_Ai")) return 0;
   Ai=tos->mat;

   for (j = 0; j < rows; j++) {
       if (Li[j] == 0.0) {
          DCOPY( &rows, &lapack_eigvec[j*rows], &int_1, &Ar[j*rows], &int_1);
          /* zero out the imaginary part */
          DSCAL( &rows, &double_0, &Ai[j*rows], &int_1);
       } else {
          /* real part of eigenvector j */
          DCOPY(&rows,&lapack_eigvec[j*rows],    &int_1, &Ar[j*rows], &int_1);
          /* imag part of eigenvector j */
          DCOPY(&rows,&lapack_eigvec[(j+1)*rows],&int_1, &Ai[j*rows], &int_1);
          j++;

          /* real part of eigenvector j+1 */
          DCOPY(&rows,&lapack_eigvec[(j-1)*rows],&int_1, &Ar[j*rows],  &int_1);

          /* imag part of eigenvector j+1 */
          DCOPY(&rows,&lapack_eigvec[(j+0)*rows],&int_1, &Ai[j*rows],&int_1);
          DSCAL(&rows, &double_m1,                       &Ai[j*rows],&int_1);
       }
   }
   pushq2("2 roll drop",11); xmain(0);  /* get rid of lapack_eigvec */

#endif

   if(!matstk(rows,1,"_Mag")) return 0;
   Mag=tos->mat;

   for(i=0;i<rows;i++) {
      *(Mag+i)=pow(*(Lr+i)*(*(Lr+i)) + *(Li+i)*(*(Li+i)),0.5);
   }

   /* get the ascending sort sequence based on eigenvalue magnitudes */
   sort_seq(); /* stk: Lr Li Ar Ai S */

   /* now reorder each part of the eigensolution in-place */
   pushq2("1 pick reorder drop",19); xmain(0);    /* Ai */
   pushq2("2 pick reorder drop",19); xmain(0);    /* Ar */
   pushq2("3 pick reorder drop",19); xmain(0);    /* Li */
   pushq2("4 pick reorder drop",19); xmain(0);    /* Lr */

   /* finish off by dropping the sort term hS from the stack */
   drop(); 

   return(1);
}
 
int ceigSSL_OLD() /* ceigSSL (hC f --- hLr hLi hAr hAi) */
/* Eigenanalysis of real matrix C.  Resulting eigenvalues and eigen-
   vectors are complex. */
{
   double *A,*Ai,*Ar,*Aux,*C,*fptr,*L,*Li,*Lr,*Mag;
#ifdef LAPACK
   double *dummy, *lapack_eigvec, *lapack_eigval_Re, *lapack_eigval_Im;
   char   chr_N       = 'N';
   char   chr_V       = 'V';
   double double_0    =  0.0;
   double double_m1   = -1.0;
   int  Aux_size, info;
#endif
   int    int_1       =   1;
   int    int_2       =   2;
   int f,i=0,j=0,k=0,rows;
#ifdef ESSL
   int *sel;
#endif
   FILE *fpreal,*fpimag;
   size_t items=1;
   long fpos;
   unsigned char *filimag,*filreal;

   char *sorteig="(hLr hLi hfptr hMag) 4 parkn, yes 4 ndx sorton";

   if(!popint(&f)) return 0;

   if(tos->typ!=MAT) {
      stkerr(" ceigSSL: ",MATNOT);
      return 0;
   }
   rows=tos->row;
   if(rows!=tos->col) {
      stkerr(" ceigSSL: ",SQUNOT);
      return 0;
   }
   cop(); /* C will be overwritten, so using a copy */
   C=tos->mat;

   /* array for complex eigenvalues: */
   if(!matstk(rows*2,1,"_L")) return 0;
   L=tos->mat;

   /* array for complex eigenvectors: */
   if(!matstk(rows*2,rows,"_A")) return 0;
   A=tos->mat;

   /* workspace array: */
#if   defined(ESSL)
   if(!matstk(rows,3,"_Aux")) return 0;
#elif defined(LAPACK)
   if(!matstk(rows,4,"_Aux")) return 0;
   Aux_size = 4*rows; /* minimum value; need more for better performance */
#endif
   Aux=tos->mat;

/* Computing complex eigenvalues and eigenvectors: */
#if   defined(ESSL)
   dgeev(1,C,rows,(dcmplx*) L,A,rows,sel,rows,Aux,3*rows);
#elif defined(LAPACK)
   if(!matstk(   1,   1,"_dummy"))            return 0;
   dummy=tos->mat;
   if(!matstk(rows,rows,"_lapack_eigvec"))    return 0;
   lapack_eigvec=tos->mat;
   if(!matstk(rows,   1,"_lapack_eigval_Re")) return 0;
   lapack_eigval_Re=tos->mat;
   if(!matstk(rows,   1,"_lapack_eigval_Im")) return 0;
   lapack_eigval_Im=tos->mat;

   DGEEV(&chr_N, &chr_V, &rows, C, &rows, lapack_eigval_Re, lapack_eigval_Im,
         dummy, &int_1, lapack_eigvec, &rows, Aux, &Aux_size, &info);

   for (j = 0; j < rows; j++) {
       L[2*j  ] = lapack_eigval_Re[j];
       L[2*j+1] = lapack_eigval_Im[j];
       if (lapack_eigval_Im[j] == 0.0) {
          DCOPY( &rows, &lapack_eigvec[j*rows], &int_1, &A[2*j*rows], &int_2);
          /* zero out the imaginary part */
          DSCAL( &rows, &double_0, &A[2*j*rows+1], &int_2);
       } else {
          /* real part of eigenvector j */
          DCOPY(&rows,&lapack_eigvec[j*rows],    &int_1, &A[2*j*rows],  &int_2);
          /* imag part of eigenvector j */
          DCOPY(&rows,&lapack_eigvec[(j+1)*rows],&int_1, &A[2*j*rows+1],&int_2);

          j++;
          L[2*j  ] = lapack_eigval_Re[j];
          L[2*j+1] = lapack_eigval_Im[j];

          /* real part of eigenvector j+1 */
          DCOPY(&rows,&lapack_eigvec[(j-1)*rows],&int_1, &A[2*j*rows],  &int_2);

          /* imag part of eigenvector j+1 */
          DCOPY(&rows,&lapack_eigvec[(j+0)*rows],&int_1, &A[2*j*rows+1],&int_2);
          DSCAL(&rows, &double_m1,                       &A[2*j*rows+1],&int_2);
       }
   }

   drop(); /* dropping lapack_eigval_Im  from the stack */
   drop(); /* dropping lapack_eigval_Re  from the stack */
   drop(); /* dropping lapack_eigvec     from the stack */
   drop(); /* dropping dummy             from the stack */
#endif
   drop(); /* dropping Aux               from the stack */

   /* Writing eigenvector real and imaginary components to separate
   files: */
   filreal=(unsigned char *)mprintf("VEIGR.%d.tmp",getpid());
   fpreal=fopen((char *)filreal,"w+b");

   filimag=(unsigned char *)mprintf("VEIGI.%d.tmp",getpid());
   fpimag=fopen((char *)filimag,"w+b");

   if(fpreal==NULL || fpimag==NULL) {
      stkerr(" ceigSSL: ",FILNOTO);
      return 0;
   }
   if((fptr=malloc(1+rows*sizeof(double)))==NULL) {
      stkerr(" ceigSSL: ",MEMNOT);
      return 0;
   }
   for(j=0;j<rows;j++) {
      fpos=ftell(fpreal);
      *(fptr+j)=fpos;
      i=0;
      for(;i<rows;i++) {
         fwrite(A,sizeof(double),items,fpreal);
         A++;
         fwrite(A,sizeof(double),items,fpimag);
         A++;
      }
   }
   drop(); /* dropping A from the stack */

   /* Separating eigenvalue real and imaginary components: */
   if(!matstk(rows,1,"_Lr")) return 0;
   Lr=tos->mat;
   if(!matstk(rows,1,"_Li")) return 0;
   Li=tos->mat;

   k=0;
   for(i=0;i<rows;i++) {
      *(Lr+i)=*(L+k);
      *(Li+i)=*(L+1+k);
      k+=2;
   }
   /* Getting C and L off the stack: */
   pushq2("2 roll drop, 2 roll drop",24); xmain(0);

/* Sorting the eigenvector file pointers with the eigenvalues: */
   dup2s();
   if(!matstk(rows,1,"_fptr")) return 0;
   memcpy(tos->mat,fptr,rows*sizeof(double));
   free(fptr);

   if(!matstk(rows,1,"_Mag")) return 0;
   Mag=tos->mat;

   for(i=0;i<rows;i++) {
      *(Mag+i)=pow(*(Lr+i)*(*(Lr+i)) + *(Li+i)*(*(Li+i)),0.5);
   }
   /* Phrase sorteig makes the four-column matrix and sorts all rows
   using values of Mag in the 4th column: */
   pushq2(sorteig,strlen(sorteig)); xmain(0);

   /* Contents of the sorted four-column matrix on the stack: 
         col 1 - real part of eigenvalue
         col 2 - imaginary part of eigenvalue
         col 3 - file offset to eigenvector
         col 4 - eigenvalue magnitude (no longer needed) */

   memcpy(Lr,tos->mat,rows*sizeof(double));
   memcpy(Li,(tos->mat)+rows,rows*sizeof(double));
   fptr=(tos->mat)+2*rows;

   if(!matstk(rows,rows,"_Ar")) return 0;
   Ar=tos->mat;
   if(!matstk(rows,rows,"_Ai")) return 0;
   Ai=tos->mat;

/* Fetching the sorted eigenvectors: */
   fseek(fpreal,0,SEEK_SET); /* rewind */
   fseek(fpimag,0,SEEK_SET); /* rewind */

   for(j=0;j<rows;j++) {
      fpos=*(fptr+j);

      fseek(fpreal,fpos,SEEK_SET);
      fread(Ar,rows*sizeof(double),items,fpreal);
      Ar+=rows;

      fseek(fpimag,fpos,SEEK_SET);
      fread(Ai,rows*sizeof(double),items,fpimag);
      Ai+=rows;
   }
   fclose(fpreal);
   fclose(fpimag);

   pushq2((char *)filreal,strlen((char *)filreal));
   pushq2((char *)filimag,strlen((char *)filimag));

   free(filreal);
   free(filimag);

   return(
      delete() &&
      delete() &&
      pushq2("2 roll drop",11) &&
      xmain(0)
   );
}

int cmpySSL() /* cmpySSL (hAr hAi hBr hBi --- hCr hCi) \ C=A*B */
/* Complex matrix multiplication:
      Cr = Ar*Br - Ai*Bi
      Ci = Ar*Bi + Ai*Br

   Note: due to unwound loops (probably), things in ESSL are multiplied
   and added differently than in a straight-forward multiply-add.  This
   makes exact comparison improbable without filtering tiny terms. */
{
   double *Ar,*Ai,*Br,*Bi,*Cr,*Ci;
#if   defined(ESSL)
   double pos1=1,neg1=-1;
#endif
   int i,rA,cA,cB;
#ifdef LAPACK
   char   chr_N     = 'N';
   double double_0  =  0.0;
   double double_1  =  1.0;
   double double_m1 = -1.0;
#endif

   rA=(tos-3)->row;
   cA=(tos-3)->col;
   cB=(tos-1)->col;
   
   if(cA!=tos->row) {
      stkerr(" cmpySSL: ",MATSNOTC);
      return 0;
   }
   Ar=(tos-3)->mat;
   Ai=(tos-2)->mat;
   Br=(tos-1)->mat;
   Bi=(tos  )->mat;

   if(!matstk(rA,cB,"_Cr")) return 0;
   Cr=tos->mat;
   if(!matstk(rA,cB,"_Ci")) return 0;
   Ci=tos->mat;

   /* Cr = Ar*Br - Ai*Bi: */
#if   defined(ESSL)
   dgemul(Ar,rA,"N",Br,cA,"N",Cr,rA,rA,cA,cB);
   dgemm("N","N",rA,cB,cA,neg1,Ai,rA,Bi,cA,pos1,Cr,rA);
#elif defined(LAPACK)
   /* Cr = 1*Ar*Br + 0*Cr */
   DGEMM(&chr_N, &chr_N, &rA, &cB, &cA, &double_1,  Ar, &rA,
         Br, &cA, &double_0, Cr, &rA);
   /* Cr = -1*Ai*Bi + 1*Cr */
   DGEMM(&chr_N, &chr_N, &rA, &cB, &cA, &double_m1, Ai, &rA,
         Bi, &cA, &double_1,    Cr, &rA);
#endif

   /* Ci = Ar*Bi + Ai*Br: */
#if   defined(ESSL)
   dgemul(Ar,rA,"N",Bi,cA,"N",Ci,rA,rA,cA,cB);
   dgemm("N","N",rA,cB,cA,pos1,Ai,rA,Br,cA,pos1,Ci,rA);
#elif defined(LAPACK)
   /* Ci = 1*Ar*Bi + 0*Ci */
   DGEMM(&chr_N, &chr_N, &rA, &cB, &cA, &double_1,  Ar, &rA,
         Bi, &cA, &double_0, Ci, &rA);
   /* Ci = 1*Ai*Br + 1*Ci */
   DGEMM(&chr_N, &chr_N, &rA, &cB, &cA, &double_1,  Ai, &rA,
         Br, &cA, &double_1,    Ci, &rA);
#endif

   i=0;
   while(i<4) { /* dropping Ar, Ai, Br, Bi: */
      pushint(2);
      roll();
      drop();
      i++;
   }
   return 1;
}

int diagpostSSL() /* diagpostSSL (hA hV --- hB) */
/* Post-multiply A by diagonal matrix stored in vector V (column
   scaling of A). */
{
   register double *A,*B,d,*V;
   register int rA,cA,j=0;
#ifdef LAPACK
   int    rAcA;
   int    int_1       =   1;
   double regular_d;
   int    regular_rA;
#endif

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" diagpostSSL: ",MATNOT2);
      return 0;
   }
   if((cA=tos->row)!=(tos-1)->col) {
      stkerr(" diagpostSSL: ",
         " row size of vector V must match column size of A");
      return 0;
   }
   A=(tos-1)->mat;
   V=tos->mat;
   rA=(tos-1)->row;

   if(!matstk(rA,cA,"_B")) return 0;
   B=tos->mat;

#if   defined(ESSL)
   DCOPY(rA*cA,A,1,B,1);
#elif defined(LAPACK)
   rAcA = rA*cA;
   DCOPY(&rAcA, A, &int_1, B, &int_1);
#endif

   for(;j<cA;j++) {
      d=*(V+j);
#if   defined(ESSL)
      DSCAL(rA,d,B,1);
#elif defined(LAPACK)
      /* cannot use addresses of register variables */
      regular_rA = rA; 
      regular_d  =  d;
      DSCAL(&regular_rA, &regular_d, B, &int_1);
#endif
      B+=rA;
   }
   return(lop() && lop());
}

int dotprodSSL() /* *dotSSL (hA hB --- hV) */
/* Row j of vector V contains the dot product of column j of A and B. */
{
   double *A,*B,*V;
   int cols,rows, j;
#ifdef LAPACK
   int    int_1       =   1;
#endif

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" *dotSSL: ",MATNOT2);
      return 0;
   }
   if((rows=tos->row)!=(tos-1)->row || (cols=tos->col)!=(tos-1)->col) {
      stkerr(" *dotSSL: ",MATSNOTC);
      return 0;
   }
   if(!matstk(cols,1,"_*dot")) return 0;

   V=tos->mat;
   B=(tos-1)->mat;
   A=(tos-2)->mat;

#if   defined(ESSL)
   dndot(cols,rows,V,1,1,A,1,rows,B,1,rows);
#elif defined(LAPACK)
   for (j = 0; j < cols; j++) {
      V[j] = DDOT(&rows, &A[j*rows], &int_1, &B[j*rows], &int_1);
   }
#endif
   
   return(lop() && lop());
}

int movecSSL() /* movecSSL (hV hA c --- ) */
/* Move bytes of vector V into column c of matrix A.  No bounds and
   type checking are done. */ 
{
   register double *A,*V;
   int c,rows;
#ifdef LAPACK
   int    int_1       =   1;
#endif

   if(!popint(&c)) return 0;
   V=(tos-1)->mat;
   A=(tos->mat)+locvec((c-XBASE),(rows=tos->row));

#if   defined(ESSL)
   DCOPY(rows,V,1,A,1);
#elif defined(LAPACK)
   DCOPY(&rows, V, &int_1, A, &int_1);
#endif

   return(drop2());
}

int mpySSL() /* mpySSL (hA hB --- hC) \ C=A*B */
/* Matrix multiplication, A and B real, or A and B complex. */
{
   double *A,*B,*C;
   int cA,cB,nw=1,rA;
#ifdef LAPACK
   double double_1[2] = {1.0,0.0};
   double double_0[2] =  {0.0,0.0};
   char   chr_N    = 'N';
#endif

   rA=(tos-1)->row;
   cA=(tos-1)->col;
   cB=tos->col;

   if(is_complex(tos)) nw=2;

   if((nw*cA)!=tos->row) {
      stkerr(" mpySSL: ",MATSNOTC);
      return 0;
   }
   A=(tos-1)->mat;
   B=tos->mat;

   if(!matstk(rA,cB,"_C")) return 0;
   C=tos->mat;

#if   defined(ESSL)
   if(nw==1) {
      dgemul(A,rA,"N",B,cA,"N",C,rA,rA,cA,cB);
   }
   else {
      set_complex(tos);
      rA/=nw;
      zgemul(A,rA,"N",B,cA,"N",C,rA,rA,cA,cB);
   }
#elif defined(LAPACK)
   if(nw==1) {
   DGEMM(&chr_N, &chr_N, &rA, &cB, &cA, &double_1, A, &rA,
         B, &cA, &double_0,  C, &rA);
   }
   else {
      set_complex(tos);
      rA/=nw;
      ZGEMM(&chr_N, &chr_N, &rA, &cB, &cA, &double_1, A, &rA,
         B, &cA, &double_0,  C, &rA);
   }
#endif

   return(lop() && lop());
}

int plusSSL() /* plusSSL (hA hB --- hC) \ C=A+B */
/* Matrix addition, A and B real, or A and B complex. */
{
   double *A,*B,*C;
   int cA,rA;
#ifdef LAPACK
   int rAcA;
   double double_1    =  1.0;
   int    int_1       =   1;
#endif

   rA=(tos-1)->row;
   cA=(tos-1)->col;

   if(cA!=tos->col || rA!=tos->row) {
      stkerr(" plusSSL: ",MATSNOTC);
      return 0;
   }
   A=(tos-1)->mat;
   B=tos->mat;

   if(!matstk(rA,cA,"_C")) return 0;
   C=tos->mat;
   tos->tag=(tos-1)->tag;

#if   defined(ESSL)
   dgeadd(A,rA,"N",B,rA,"N",C,rA,rA,cA);
#elif defined(LAPACK)
   rAcA = rA*cA;
   DCOPY(&rAcA,            B, &int_1, C, &int_1);
   DAXPY(&rAcA, &double_1, A, &int_1, C, &int_1);
#endif

   return(lop() && lop());
}

#if defined(ESSL)
int reigSSL() /* reigSSL (hM hK --- hZ hw) */
/* Eigenvalues and eigenvectors of a real, symmetric eigensystem:
      K*Z=w*M*Z
   where K and M are real and symmetric, and M is positive definite. */
/*
{  Test case for reigSSL.

   Reference:
      Engineering and Scientific Subroutine Library, Version 2,
      Release 2, Second Edition (January 1994), IBM Corporation.

   Data from the Reference, Volume 2, Example 2, p. 792:

      list: -1 1 -1 ; list: 1 1 -1 ; list: -1 -1 1 ; 3 parkn into Kaa
      list:  2 1  0 ; list: 1 2  1 ; list:  0  1 2 ; 3 parkn into Maa

      [tops@bach] ready > Maa Kaa reigSSL

       stack elements:
             0 matrix: _w  3 by 1
             1 matrix: _Z  3 by 3
       [2] ok!
      [tops@bach] ready > 1E-8 filter .m nl nl 1E-8 filter .m
       Row 1:     -1.5
       Row 2:        0
       Row 3:        2
      
       Row 1:    0.866        0        0
       Row 2:  -0.5774  -0.4082  -0.7071
       Row 3:   0.2887  -0.4082   0.7071
}
*/
{
   double *Aux,*K,*M,*w,*Z;
   int one=1,row2,rows;
   int ierno=2115,inoal=255,inomes=255,izero=0;

   /* Set inoal=255 to allow unlimited errors and keep ESSL from shut-
      ting down the program on error 2115: leading minor having a 
      nonpositive determinant. */
   errset(&ierno,&inoal,&inomes,&izero,&izero,&izero);

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" reigSSL: ",MATNOT);
      return 0;
   }
   rows=tos->row;
   row2=rows+rows;
   if(rows!=tos->col || rows!=(tos-1)->col) {
      stkerr(" reigSSL: ",SQUNOT);
      return 0;
   }
   if(rows!=(tos-1)->row) {
      stkerr(" reigSSL: ",MATSNOTC);
      return 0;
   }
   if(rows==0) {
      stkerr(" reigSSL: ",MATPURG);
      return 0;
   }
   /* Copy M and K so originals stay intact: */
   cop();
   K=tos->mat;

   swap();
   cop();
   M=tos->mat;

   if(!matstk(2*rows,1,"_Aux")) return 0;
   Aux=tos->mat;

   if(!matstk(rows,rows,"_Z")) return 0;
   Z=tos->mat;

   if(!matstk(rows,1,"_w")) return 0;
   w=tos->mat;

   dsygv_(&one,K,&rows,M,&rows,w,Z,&rows,&rows,Aux,&row2);

   /* Dropping  Aux, K, M from stk: */
   return(
      lpush() &&
      lpush() &&
      drop2() && drop() &&
      lpull() &&
      lpull()
   );
}
#endif

int starfSSL() /* *fSSL (hA f --- hB) \ B(i,j)=A(i,j)*f */
/* Multiplying matrix A by scale factor f. */
{
   double *A,*B,f;
   int cA,len,rA;
#ifdef LAPACK
   int    int_1       =   1;
#endif

   if(!popd(&f)) return 0;

   if(tos->typ!=MAT) {
      stkerr(" *fSSL: ",MATNOT);
      return 0;
   }
   A=tos->mat;

   if(!matstk((rA=tos->row),(cA=tos->col),"_B")) return 0;
   B=tos->mat;

#if   defined(ESSL)
   DCOPY((len=rA*cA),A,1,B,1);
   DSCAL(len,f,B,1);
#elif defined(LAPACK)
   len = rA*cA;
   DCOPY(&len, A, &int_1, B, &int_1);
   DSCAL(&len, &f, B, &int_1);
#endif

   return(lop());
}

#if defined(ESSL)
int solveSSL() /* solveSSL (hb hA --- hx) */
/* Solve matrix equation A*x=b for unknown x, where A is a general 
   matrix.  A and b are both real or both complex. */
{
   double *A,*b,*Piv;
   int cols,nw=1,rows;
   char *trans="N";

   int ierno=2103,inoal=255,inomes=255,izero=0;

   /* Set inoal=255 to allow unlimited errors and keep ESSL from shut-
      ting down the program on error 2103: the matrix is singular */
   errset(&ierno,&inoal,&inomes,&izero,&izero,&izero);

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" solveSSL: ",MATNOT);
      return 0;
   }
   if(is_complex(tos)) nw=2;

   rows=tos->row;
   if(rows!=nw*tos->col) {
      stkerr(" solveSSL: ",SQUNOT);
      return 0;
   }
/* If b is purged, replace it with an identity matrix: */
   if((tos-1)->row==0 || (tos-1)->col==0) {
      lop();
      pushint(rows/nw);
      identity();
      if(nw==2) {
         pushint(rows/nw);
         dup1s();
         null();
         dblcmplx();
      }
      swap();
   }
   if(rows!=(tos-1)->row) {
      stkerr(" solveSSL: ",MATSNOTC);
      return 0;
   }
   /* Copy A and B so originals stay intact: */
   cop();
   A=tos->mat;

   swap();
   cop();
   b=tos->mat;
   cols=tos->col;

   if(!matstk(rows,1,"_Piv")) return 0;
   Piv=tos->mat;

   if(nw==1) {
      dgef(A,rows,rows,(int *)Piv);
      dgesm(trans,A,rows,rows,(int *)Piv,b,rows,cols);
   }
   else {
      rows/=nw;
      zgef(A,rows,rows,(int *)Piv);
      zgesm(trans,A,rows,rows,(int *)Piv,b,rows,cols);
   }
   return(
      drop() && lop() && pushq2("_x",2) && naming()
   );
}

int symSSL() /* symSSL (hb hA --- hx) */
/* Solve matrix equation A*x=b for unknown x, where matrix A is 
   symmetric.  A and b are both real or both complex. */
{
   double *A,*b,det,rcond;
   int cols,nw=1,rows;
   char *uplo="L";

   int ierno=2115,inoal=255,inomes=255,izero=0;

   /* Set inoal=255 to allow unlimited errors and keep ESSL from shut-
      ting down the program on error 2115: leading minor having a 
      nonpositive determinant. */
   errset(&ierno,&inoal,&inomes,&izero,&izero,&izero);

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" symSSL: ",MATNOT);
      return 0;
   }
   if(is_complex(tos)) nw=2;

   rows=tos->row;
   if(rows!=nw*tos->col) {
      stkerr(" symSSL: ",SQUNOT);
      return 0;
   }
   if(rows!=(tos-1)->row) {
      stkerr(" symSSL: ",MATSNOTC);
      return 0;
   }
   /* Copy A and B so originals stay intact: */
   cop();
   A=tos->mat;

   swap();
   cop();
   b=tos->mat;
   cols=tos->col;

   if(nw==1) {
      dpof(uplo,A,rows,rows);
      dposm(uplo,A,rows,rows,b,rows,cols);
   }
   else {
      rows/=nw;
      zpof(uplo,A,rows,rows);
      dposm(uplo,A,rows,rows,b,rows,cols);
   }
   return(
      lop() && pushq2("_x",2) && naming()
   );
}

int trilowSSL() /* trilowSSL (hA --- hL) */
/* For real symmetric matrix A, return lower triangluar matrix, L, such
   that L*L' = A. */
{
   double *A;
   int i,info,j=0,n;
   char uplo='L';
   int ierno=2115,inoal=255,inomes=255,izero=0;

/* Set inoal=255 to allow unlimited errors and keep ESSL from shutting 
   down the program on error 2115: matrix is not positive definite. */
   errset(&ierno,&inoal,&inomes,&izero,&izero,&izero);

   if(!(tos->typ==MAT)) {
      stkerr(" trilowSSL: ",MATNOT);
      return 0;
   }
   cop();
   n=tos->row;

   for(;j<n;j++) {
      A=(tos->mat)+locvec(j,n);
      for(i=0;i<j;i++) {
         *A=0;
         A++;
      }
   }
   A=tos->mat; /* on return, this will be L */
   dpof(&uplo,A,n,n);

   return(
      pushstr("_trilowSSL") && naming()
   );
}

#endif

#else
/*
 *  No code if neither ESSL nor LAPACK are available.
 */
#endif
