#include <stdlib.h>
/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author of metis.c: Al Danial <al.danial@gmail.com>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

#ifdef METIS
/* metis.c

   A. Danial January 2005

   Words to partition and renumber graphs.

----------------------------------------------------------------------*/

#include <stdio.h>
#include "stk.h"
#include "main.h"
#include "inpo.h"
#include "sparse.h"
#include "tag.h"       /* is_sparse() */
#include "mem.h"       /* matstk()    */
#include <unistd.h>    /* sleep */
#include <metis.h>
void GKfree(void *, ...);  /* should be in metis.h but isn't */
#define IDXSIZE sizeof(idxtype)
int  sparse_to_metis_graph(SparseMatrix G    ,
                           GraphType   *graph,
                           int         *wgtflag);
void ReadGraph(GraphType *graph, char *filename, int *wgtflag);
void WriteGraph( char *filename, int nvtxs, idxtype *xadj, idxtype *adjncy);
                                                                                
/* demo input  {{{1
                                                                                
sample graphs and two way partitions:
                                                                                
7 11
5 3 2
1 3 4
5 4 2 1
2 3 6 7
1 3 6
5 4 7
6 4
                                                                                
0
0
0
1
1
1
1
                                                                                
7 16
2 3 4
1 3 4 5 7
1 2 4 5 6 7
1 2 3 5 6
2 3 4 6 7
3 4 5 7
2 3 5 6
                                                                                
0
0
1
0
1
1
1
                                                                                
1}}} */
int pmet()    /* pmet ( hGraph n --- hPart ) {{{1 */
/*
 * man entry:  pmet {{{2
 * ( hGraph n --- hPart ) Partition a graph into n parts.  Indices of a sparse matrix convey the graph's connectivity while its numeric values contain optional vertex and edge weights.  The sparse matrix should be symmetric; only the upper triangle is used.  Real numerical values of the sparse matrix are used as vertex weights for the partitioning; complex numerical values are treated as edge weights.  Only one vertex weight is allowed for each vertex (ncon = 1) so the first upper triangular nonzero in the column is used for the vertex weight.  If all numerical values are zero an unweighted partitioning will be done.  Output is a vector denoting the partition each vertex belongs to.  Partition identifiers are integers on 0..(n-1).  For example,  Part[i] = j  means vertex i belongs to partition j.
 * category: math::matrix::partitioning
 * related: sparse
 * 2}}}
 */
{
    GraphType graph;
    int       i, nPart, edgecut, sp_conversion,
              wgtflag = 0, numflag = 0,
              options[10];
    char *filename = "input.pmet";
    idxtype *part;
    double  *partition;
    float lbvec[MAXNCON];
    SparseMatrix sp_graph;
int DEBUG = 0;
                                                                                
if (DEBUG) { gprintf("top of pmet\n"); }
    if (!popint(&nPart)) return 0;
    if (nPart <= 1) {
        stkerr("pmet: ", "asking for fewer than two partitions, nothing to do");        return 0;
    }
if (DEBUG) gprintf("pmet: requesting %d partitions\n", nPart);
    options[0] = 0;
                                                                                
if (0) {
    ReadGraph(&graph, filename, &wgtflag);
} else {
    if (!is_sparse(tos)) {
        stkerr("pmet: ",SPARSENOT);
        return 0;
    }
    sp_graph      = sparse_overlay(tos);
    sp_conversion = sparse_to_metis_graph(sp_graph, &graph, &wgtflag);
    if (!sp_conversion) {
        stkerr("pmet: ","sparse matrix to graph conversion error");
        return 0;
    }
}
                                                                                
    if (graph.nvtxs <= 0) {         stkerr("pmet: ", "empty graph");         return 0;     } else if (nPart > graph.nvtxs) {         stkerr("pmet: ", "requested more partitions than vertices");         return 0;     } if (DEBUG) { gprintf("pmet: after  ReadGraph nvtxs=%d ncon=%d\n", graph.nvtxs, graph.ncon); }                                                                                      part = (idxtype *) malloc(graph.nvtxs * IDXSIZE);
    if (!matstk(graph.nvtxs, 1, "_partition")) return 0;
    partition = tos->mat;
                                                                                
if (DEBUG) { gprintf("pmet: after  idxmalloc for part\n");}
    if (graph.ncon == 1) {
        METIS_PartGraphRecursive(  &graph.nvtxs  ,
                                    graph.xadj   ,
                                    graph.adjncy ,
                                    graph.vwgt   ,
                                    graph.adjwgt ,
                                   &wgtflag      ,
                                   &numflag      ,
                                   &nPart        ,
                                    options      ,
                                   &edgecut      ,
                                    part);
    } else {
        METIS_mCPartGraphRecursive(&graph.nvtxs  ,
                                   &graph.ncon   ,
                                    graph.xadj   ,
                                    graph.adjncy ,
                                    graph.vwgt   ,
                                    graph.adjwgt ,
                                   &wgtflag      ,
                                   &numflag      ,
                                   &nPart        ,
                                    options      ,
                                   &edgecut      ,
                                    part);
    }
if (DEBUG) {
    ComputePartitionBalance(&graph, nPart, part, lbvec);
    gprintf("  %d-way Edge-Cut: %7d, Balance: ", nPart, edgecut);
    for (i=0; i<graph.ncon; i++)
        gprintf("%5.2f ", lbvec[i]);
    gprintf("\n");
}
                                                                                
    for (i = 0; i < graph.nvtxs; i++) {
        partition[i] = (double) part[i];
    }
                                                                                
    GKfree(&graph.xadj  ,
           &graph.adjncy,
           &graph.vwgt  ,
           &graph.adjwgt,
           &part        ,
           LTERM);
                                                                                
    return 1;
} /* 1}}} */
                                                                                
int  sparse_to_metis_graph(  /* {{{1 */
                           SparseMatrix G    ,   /* in, better be symmetric */
                           GraphType   *graph,   /* out */
                           int         *wgtflag) /* out */
/* Note:  if the input matrix, G, is not symmetric the output will
 *        be a directed graph.  Metis cannot handle directed graphs!
 */
{
    int l, k, c, r, s, n_ptr, end_row,
        nEdges = 0, readew = 0, readvw = 0, ncon = 0;
    idxtype *xadj, *adjncy, *vwgt, *adjwgt;
int DEBUG = 0;
                                                                                
    InitGraph(graph);
                                                                                
    /*  pass 1:  count number of edges;  {{{2
     *           figure out if there are vertex and/or edge weights
     */
    for (c = 0; c < G.H[COLS]; c++) {
if (DEBUG) gprintf("Column %3d\n", c);
        n_ptr = 0;
        for (s = G.S_start[c]; s < G.S_start[c+1]; s++) {
            end_row = G.S[s].start_row + G.S[s].len - 1;
            for (r = G.S[s].start_row; r <= end_row; r++) {
                ++nEdges;
                if (G.N[ G.N_start[c] + n_ptr ]) {
                    /* non-zero real numerical value  at r,c */
                    readvw = 1;
                    ncon   = 1;
                }
if (DEBUG) {
gprintf("R %3d N[%3d]=% 12.4f",
r, G.N_start[c] + n_ptr, G.N[ G.N_start[c] + n_ptr ]);
}
                n_ptr++;
                if (G.H[COMPLX]) {
                    if (G.N[ G.N_start[c] + n_ptr ]) {
                        /* non-zero complex numerical value  at r,c */
                        readew = 1;
                    }
if (DEBUG) {
gprintf(", N[%3d]=% 12.4f",
G.N_start[c] + n_ptr, G.N[ G.N_start[c] + n_ptr ]);
gprintf("\n");
}
                    n_ptr++;
                }
            }
        }
    }
    /* 2}}} */
                                                                                
    /* initialize, allocate memory {{{2 */
    graph->nedges = nEdges;
    graph->nvtxs  = G.H[COLS];
    *wgtflag = 0;
    if (readew)
        *wgtflag += 1;
    if (readvw)
        *wgtflag += 2;
    ncon = graph->ncon = (ncon == 0 ? 1 : ncon);
                                                                                
    if (graph->nvtxs > MAXIDX) {
        errexit("\nThe matrix is too big: %d [%d %d]\n",
                graph->nvtxs, MAXIDX, sizeof(idxtype));
    }
                                                                                
    xadj   = graph->xadj   = (idxtype *) malloc((graph->nvtxs+1) * IDXSIZE);
    for (c = 0; c < graph->nvtxs+1; c++) xadj[c] = 0;
    adjncy = graph->adjncy = (idxtype *) malloc(graph->nedges * IDXSIZE);
                                                                                
    vwgt   = graph->vwgt   = (readvw ?
                             (idxtype *) malloc(ncon*graph->nvtxs * IDXSIZE) :
                             NULL);
    adjwgt = graph->adjwgt = (readew ?
                             (idxtype *) malloc(graph->nedges * IDXSIZE) :
                             NULL);
    /* 2}}} */
                                                                                
    /*  pass 2:  traverse and store the connectivity {{{2
     */
    n_ptr = 0;
    for (xadj[0]=0, k=0, c=0; c < graph->nvtxs; c++) {
                                                                                
        if (readvw) {
            for (l = 0; l < ncon; l++) {   /* limitation:  ncon <= 1 */
                /* vertex weight: only use first real term in the column */
                vwgt[c*ncon+l] = (int) G.N[ G.S[c].N_idx ];
            }
        }
                                                                                
        for (s = G.S_start[c]; s < G.S_start[c+1]; s++) {
            end_row = G.S[s].start_row + G.S[s].len - 1;
            for (r = G.S[s].start_row; r <= end_row; r++) {
                adjncy[k] = r;
                if (G.H[COMPLX]) {
                    n_ptr++;  /* skip the real part of a complex number */
                    if (readew && G.N[n_ptr]) {
                        /* there is a non-zero imaginary term;
                         * count it as an edge weight
                         */
                        adjwgt[k] = (int) G.N[n_ptr];
                    }
                    n_ptr++;  /* increment past the imaginary part */
                } else {
                    n_ptr++;  /* skip the real term */
                }
                k++;
            }
        }
        xadj[c+1] = k;
    }
    /* 2}}} */
                                                                                
    if (k != graph->nedges) {
        gprintf("sparse_to_metis_graph error:  k=%d "
                " graph->nedges=%d", k, graph->nedges);
        sleep(1000);
        return 0;
    }
    return 1;
} /* 1}}} */
void ReadGraph( /* {{{1 */
               GraphType *graph, char *filename, int *wgtflag)
/* based on metis-4.0/Programs/io.c */
{
    int i, k, l, fmt, readew, readvw, ncon, edge, ewgt = 0;
    idxtype *xadj, *adjncy, *vwgt, *adjwgt;
    char *line, *oldstr, *newstr;
    FILE *fpin;
int DEBUG = 0;
                                                                                
    InitGraph(graph);
                                                                                
    line = (char *)malloc(sizeof(char)*(MAXLINE+1));
                                                                                
    if ((fpin = fopen(filename, "r")) == NULL) {
      printf("Failed to open file %s\n", filename);
      exit(0);
    }
                                                                                
if (DEBUG) { gprintf("ReadGraph  A MAXLINE=%d\n", MAXLINE); }
    do {
      fgets(line, MAXLINE, fpin);
    } while (line[0] == '%' && !feof(fpin));
                                                                                
    if (feof(fpin)) {
      graph->nvtxs = 0;
      free(line);
      return;
    }
                                                                                
    fmt = ncon = 0;
    sscanf(line, "%d %d %d %d", &(graph->nvtxs), &(graph->nedges), &fmt, &ncon);                                                                                
if (DEBUG) { gprintf("ReadGraph  B\n"); }
    readew = (fmt%10 > 0);
    readvw = ((fmt/10)%10 > 0);
    if (fmt >= 100) {
      printf("Cannot read this type of file format!");
      exit(0);
    }
                                                                                
                                                                                
    *wgtflag = 0;
    if (readew)
      *wgtflag += 1;
    if (readvw)
      *wgtflag += 2;
                                                                                
    if (ncon > 0 && !readvw) {
      printf("----------------------------------------------------------\n");
      printf("***  I detected an error in your input file  ***\n\n");
      printf("You specified ncon=%d, but the fmt parameter does not "
             "specify vertex weights\n", ncon);
      printf("Make sure that the fmt parameter is set to either 10 or 11.\n");
      printf("----------------------------------------------------------\n");
      exit(0);
    }
if (DEBUG) { gprintf("ReadGraph  C\n"); }
                                                                                
    graph->nedges *=2;
    ncon = graph->ncon = (ncon == 0 ? 1 : ncon);
                                                                                
if (DEBUG) { gprintf("ReadGraph  D\n");
gprintf("%d %d %d %d %d [%d %d]\n",
fmt, fmt%10, (fmt/10)%10, ncon, graph->ncon, readew, readvw);
}
                                                                                
    if (graph->nvtxs > MAXIDX)
      errexit("\nThe matrix is too big: %d [%d %d]\n",
              graph->nvtxs, MAXIDX, sizeof(idxtype));
                                                                                
if (DEBUG) { gprintf("ReadGraph  E\n"); }
    xadj = graph->xadj = idxsmalloc(graph->nvtxs+1, 0, "ReadGraph: xadj");
if (DEBUG) { gprintf("ReadGraph  F\n"); }
    adjncy = graph->adjncy = malloc(graph->nedges * IDXSIZE);
if (DEBUG) { gprintf("ReadGraph  G\n"); }
                                                                                
    vwgt = graph->vwgt = (readvw ? malloc(ncon*graph->nvtxs * IDXSIZE) : NULL);
if (DEBUG) { gprintf("ReadGraph  H\n"); }
    adjwgt = graph->adjwgt = (readew ? malloc(graph->nedges * IDXSIZE) : NULL);
if (DEBUG) { gprintf("ReadGraph  I\n"); }
                                                                                
    /* Start reading the graph file */
    for (xadj[0]=0, k=0, i=0; i<graph->nvtxs; i++) {
      do {
        fgets(line, MAXLINE, fpin);
      } while (line[0] == '%' && !feof(fpin));
      oldstr = line;
      newstr = NULL;
                                                                                
      if (strlen(line) == MAXLINE)
        errexit("\nBuffer for fgets not big enough!\n");
                                                                                
      if (readvw) {
        for (l=0; l<ncon; l++) {
          vwgt[i*ncon+l] = (int)strtol(oldstr, &newstr, 10);
          oldstr = newstr;
        }
      }
if (DEBUG) { gprintf("ReadGraph  J i=%d\n",i); }
                                                                                
      for (;;) {
        edge = (int)strtol(oldstr, &newstr, 10) -1;
        oldstr = newstr;
                                                                                
        if (readew) {
          ewgt = (int)strtol(oldstr, &newstr, 10);
          oldstr = newstr;
        }
                                                                                
        if (edge < 0)
          break;
                                                                                
        adjncy[k] = edge;
        if (readew)
          adjwgt[k] = ewgt;
        k++;
      }
      xadj[i+1] = k;
    }
                                                                                
    fclose(fpin);
                                                                                
    if (k != graph->nedges) {
      printf("----------------------------------------------------------\n");
      printf("***  I detected an error in your input file  ***\n\n");
      printf("In the first line of the file, you specified that the "
             "graph contained\n%d edges. However, I only found %d edges "
             "in the file.\n", graph->nedges/2, k/2);
      if (2*k == graph->nedges) {
        printf("\n *> I detected that you specified twice the number of "
               "edges that you have in\n");
        printf("    the file. Remember that the number of edges specified "
               "in the first line\n");
        printf("    counts each edge between vertices v and u only once.\n\n");
      }
      printf("Please specify the correct number of edges in the first line "
             "of the file.\n");
      printf("----------------------------------------------------------\n");
      exit(0);
    }
if (DEBUG) { gprintf("ReadGraph  K k=%d\n",k); }
                                                                                
    free(line);
} /* 1}}} */
void WriteGraph( /* {{{1 */
                char *filename, int nvtxs, idxtype *xadj, idxtype *adjncy)
/* from metis-4.0/Programs/io.c */
{
    int i, j;
    FILE *fpout;
                                                                                
    if ((fpout = fopen(filename, "w")) == NULL) {
        printf("Failed to open file %s\n", filename);
        exit(0);
    }
                                                                                
    fprintf(fpout, "%d %d", nvtxs, xadj[nvtxs]/2);
                                                                                
    for (i=0; i<nvtxs; i++) {
        fprintf(fpout, "\n");
        for (j=xadj[i]; j<xadj[i+1]; j++)
            fprintf(fpout, " %d", adjncy[j]+1);
    }
                                                                                
    fclose(fpout);
} /* 1}}} */
#endif
