/*
 * C driver to test the function f_index_align. The functions
 * comp_int_pair(), get_sort_sequence(), and f_index_align() are direct
 * copies from the file src/sparse.c; once they are debugged here they
 * may be dropped directly back into src/sparse.c.
 *
 * A. Danial June 18 2005
 */
#include <stdio.h>
#include <stdlib.h>   /* random, RAND_MAX */
#include <malloc.h>   /* malloc, free    */
#include <sys/time.h> /* gettimeofday    */
#include <unistd.h>   /* sleep */

#define gprintf printf
int  comp_int_pair(const void *a, const void *b ) { /* {{{1 */
    /* sort based on the value in the first column: */
    return *(int *) a - *(int *) b;
    /* sort based on the value in the second column:
    return *(((int *)a) + 1) - *(((int *)b) + 1);
    */
} /* 1}}} */
void get_sort_sequence(int  N ,  /* {{{1 */
                       int *In,
                       int *Out) {
    int i;

    /* fill the working array */
    for (i = 0; i < N; i++) {
        Out[2*i  ] = In[i];
        Out[2*i+1] =    i ;
    }

    /* sort the working array */
    qsort(Out, N, 2*sizeof(int), &comp_int_pair);

    /* overwrite terms 0..N-1 with the sort order */
    for (i = 0; i < N; i++) {
        Out[i] = Out[2*i+1];
    }
} /* 1}}} */
int  f_index_align(int  X_size,  /* in  {{{1*/
                   int *X     ,  /* in  */
                   int  Y_size,  /* in  */
                   int *Y     ,  /* in  */
                   int *Map   ,  /* out */
                   int *err_cod) /* out */
/* {{{2
 * Given two arrays of integers, {X} and {Y}, returns {Map}, an array of indices which shows where the terms of {Y} fit in {X}.  The terms of {Map} satisfy Y[i] = X[ Map[i] ] for all Map[i] which are >= XBASE.  If Map[i] is negative, then B[i] does not appear in {A}.  The arrays may contain duplicate terms.
Example:
  X  = { 84019 39439 78310 79845 91165 19756 33523 76823 27778 55397 }
  Y  = { 78310 19756 76823 39439 }
produces
 Map = { 2 5 7 1 }
while
 {X} = { 84019 39439 78310 79845 91165 19756 33523 76823 27778 55397 }
 {Y} = { 78310 19756     4 39439 }
produces
 Map = { 2 5 -1 1 }
 *
 *  err_cod return value:
 *     0: no error
 *    -1: malloc X failed
 *    -2: malloc Y failed
 * 2}}} */ 
{
int DEBUG = 0;
    int  xi, yi, still_looking, *X_order, *Y_order;

if (DEBUG) gprintf("top of f_index_align\n");

    *err_cod = 0;  /* no error */

    if ((X_order = (int *) malloc(2*X_size*sizeof(int))) == NULL) {
        *err_cod = -1;
if (DEBUG) gprintf("f_index_align malloc X failed\n");
        return 0;
    }

    if ((Y_order = (int *) malloc(2*Y_size*sizeof(int))) == NULL) {
        *err_cod = -2;
if (DEBUG) gprintf("f_index_align malloc Y failed\n");
        return 0;
    }

    get_sort_sequence(X_size, X, X_order);
    get_sort_sequence(Y_size, Y, Y_order);
                                                                                
if (DEBUG) {
for (xi = 0; xi < X_size; xi++) 
gprintf("f_index_align X %2d. %5d %5d\n", xi, X[xi], X_order[xi]);
gprintf("f_index_align \n");
for (yi = 0; yi < Y_size; yi++)
gprintf("f_index_align Y %2d. %5d %5d\n", yi, Y[yi], Y_order[yi]);
gprintf("in sort order: \n");
for (xi = 0; xi < X_size; xi++) 
gprintf("f_index_align X %2d. %5d %5d\n", xi, X[X_order[xi]], X_order[xi]);
for (yi = 0; yi < Y_size; yi++)
gprintf("f_index_align Y %2d. %5d %5d\n", yi, Y[Y_order[yi]], Y_order[yi]);
}
    /* initialize the map with all mismatches */
    for (yi = 0; yi < Y_size; yi++) Map[yi] = -1;
                                                                                
    xi            = 0;
    yi            = 0;
    still_looking = 1;
    while (still_looking) {
if (DEBUG) gprintf("main:  yi=%d  xi=%d ", yi, xi);
        if ((xi >= X_size) || (yi >= Y_size)) {
if (DEBUG) gprintf("end of loop");
            still_looking = 0;
        } else if (Y[Y_order[yi]] == X[X_order[xi]]) {
if (DEBUG) gprintf("main:  Y[Y_order[yi=%d]=%d]=%d == X[X_order[xi=%d]=%d]=%d",
yi, Y_order[yi], Y[Y_order[yi]],
xi, X_order[xi], X[X_order[xi]]);
if (DEBUG) gprintf("\n");
            Map[ Y_order[yi] ] = X_order[xi];
if (DEBUG) gprintf("       Map[Y_order[yi=%d]=%d] = X_order[xi=%d] = %d",
yi, Y_order[yi], xi, X_order[xi]);
            ++yi;
            /* don't increment xi in case Y has duplicates and the  *
             * Y term after this one matches the current X term     */
        } else if (Y[Y_order[yi]] <  X[X_order[xi]]) {
if (DEBUG) gprintf("main:  Y[Y_order[yi=%d]=%d]=%d <  X[X_order[xi=%d]=%d]=%d",
yi, Y_order[yi], Y[Y_order[yi]],
xi, X_order[xi], X[X_order[xi]]);
            ++yi;
        } else {
if (DEBUG) gprintf("main:  Y[Y_order[yi=%d]=%d]=%d >  X[X_order[xi=%d]=%d]=%d",
yi, Y_order[yi], Y[Y_order[yi]],
xi, X_order[xi], X[X_order[xi]]);
            ++xi;
        }
if (DEBUG) gprintf("\n");
    }

    free(X_order);
    free(Y_order);

    return 1;
} /* 1}}} */
int  brute_force_index_align(int  X_size,  /* in  {{{1*/
                             int *X     ,  /* in  */
                             int  Y_size,  /* in  */
                             int *Y     ,  /* in  */
                             int *Map   )  /* out */
{
    int  xi, yi, found_it;

    for (yi = 0; yi < Y_size; yi++) {
        found_it = 0;
        for (xi = 0; xi < X_size; xi++) {
            if (Y[yi] == X[xi]) {
                Map[yi]  = xi;
                found_it = 1;
                break;
            }
        }
        if (!found_it) Map[yi] = -1;
    }

    return 1;
} /* 1}}} */

int main() { /* {{{1 */
    int   *X, *Y, *Map_1, *Map_2, err_cod, X_size, Y_size;
    int    i, iter, nAlign, all_match = 1;
    float  RANGE = 100.0;
    struct timeval timenow;

    for (iter = 0; iter < 100; iter++) {

        gettimeofday(&timenow, 0);

        srand(timenow.tv_sec);

        X_size = 1+(RANGE*rand()/(RAND_MAX+1.0));
        if ((X = (int *) malloc(X_size*sizeof(int))) == NULL) {
            printf(" X malloc of %d failed\n", X_size);
            return 0;
        }
        for (i = 0; i < X_size; i++) {
            X[i] = 1+(RANGE*rand()/(RAND_MAX+1.0));
        }

        Y_size = 1+(RANGE*rand()/(RAND_MAX+1.0));
        if ((Y = (int *) malloc(Y_size*sizeof(int))) == NULL) {
            printf(" Y malloc of %d failed\n", Y_size);
            return 0;
        }
        for (i = 0; i < Y_size; i++) {
            Y[i] = 1+(RANGE*rand()/(RAND_MAX+1.0));
        }

        if ((Map_1 = (int *) malloc(Y_size*sizeof(int))) == NULL) {
            printf(" Map_1 malloc of %d failed\n", Y_size);
            return 0;
        }
        if ((Map_2 = (int *) malloc(Y_size*sizeof(int))) == NULL) {
            printf(" Map_2 malloc of %d failed\n", Y_size);
            return 0;
        }

        /* O( n log n) */
        f_index_align(X_size,   /* in  */
                      X     ,   /* in  */
                      Y_size,   /* in  */
                      Y     ,   /* in  */
                      Map_1 ,   /* out */
                     &err_cod); /* out */

        /* O( n^2 ) */
        brute_force_index_align(X_size,  /* in  */
                                X     ,  /* in  */
                                Y_size,  /* in  */
                                Y     ,  /* in  */
                                Map_2 ); /* out */
        nAlign = 0;
        for (i = 0; i < Y_size; i++) {
            if (Map_1[i] != Map_2[i]) {
                all_match = 0;
                printf("mismatch  Map_1[%2d]=%5d  Map_1[%2d]=%5d\n", 
                        i, Map_1[i], i, Map_2[i]);
            }
            if (Map_2[i] != -1) {
                ++nAlign;
            }
        }
        printf("X_size=%3d  Y_size=%3d  nAlign=%3d  ", X_size, Y_size, nAlign);
        if (all_match) {
            printf("all terms match");
        }
        printf("\n");

        free(X);
        free(Y);
        free(Map_1);
        free(Map_2);

        sleep(2);    /* give clock a chance to roll over for next seed */
    }

    return 1;
} /* 1}}} */
