/* {{{1 GNU General Public License

sofea - the Stack Operated Finite Element Analysis program
Copyright (C) 2004  Al Danial <al.danial@gmail.com>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
}}}1 */
#include <stdio.h>
#include <stk.h>
#include <sqlite3.h>
#include "fea_assemble.h"
#include "sql.h"

int apply_spc() /* ( qModelDB qParnDB part_id --- hKeep_Ind ) {{{1 */
/*
 * man entry:  apply_spc {{{2
 * ( qModelDB qParnDB part_id --- hKeep_Ind ) Return a vector of degree of freedom indices for this partition (or the entire model if part_id is zero) corresponding to renumbered dofs which remain after single point constraints are removed.  This index list can be passed to submat to reduce mass and stiffness matrices.  (This implementation of SPC's is rudimentary.)
 * category: FEA
 * related: fea_assemble
 * 2}}}
 */
{
int DEBUG = 0;

    char   *name = "_Keep_indices";
    int     i, rc, partition_number, n_nodes, node, bits, good_dof,
            n_nonspc_dof = 0;
    double *all_dof, *nonspc_dof;
#define size_T 160
    char    T[size_T+1];
    callback_data sql_data;
    sqlite3 *dbh;
    char query[QUERY_STR_SIZE+1];

if (DEBUG)
gprintf("at top of apply_spc\n");
    query[0] = '\0';

    if (!popint(&partition_number)) return 0;
    if (partition_number) {  /* requested only a portion of the total model */
        pushstr("DB_PARTN"); /* add to stack alias for partition db */
        if (!db_open2()) {   /* open model db & attach partition db */
            stkerr(" apply_spc: ", "failed to open model and partn dbs");
            return 0;
        }
    } else {
        drop();             /* ignore whatever is in position of partn DB */
        if (!db_open()) {   /* open model db */
            stkerr(" apply_spc: ", "failed to open model db");
            return 0;
        }
    }
    dbh = (sqlite3 *) tos->tex; /* ( dbh ) */

    if (partition_number) {
        snprintf(query, QUERY_STR_SIZE,
                 "select count(id) from DB_PARTN.P%02d_canonical_nid ",
                 partition_number);
        n_nodes = sql_scalar_i(dbh, query, "apply_spc");

        snprintf(query, QUERY_STR_SIZE, 
                 "select new_id,dof from spc,sets,"
                 "  DB_PARTN.P%02d_canonical_nid,DB_PARTN.P%02d_renumbered_nid "
                 "  where spc.sid = sets.sid and "
                 "        DB_PARTN.P%02d_canonical_nid.nid = sets.entity_id and"
                 "        DB_PARTN.P%02d_canonical_nid.id = "
                 "        DB_PARTN.P%02d_renumbered_nid.orig_id"
                 " FIXME "
                 "  order by new_id;",
                partition_number,
                partition_number,
                partition_number,
                partition_number,
                partition_number);
    } else {  /* do the whole model */
        n_nodes = sql_scalar_i(dbh, "select count(seq_no) from node",
                                    "apply_spc");
        snprintf(query, QUERY_STR_SIZE, 
                 "select entity_id,dof from spc,sets "
                 "  where spc.sid = sets.sid "
                 "  order by entity_id;");
    }
    sql_data.nRows     = 0;
    sql_data.row_index = 0;
    rc = sql_do_query(dbh, query, &sql_data);  /* .i[][0] = renumbered node ID 
                                                * .i[][1] = eg, 456 = no rot's
                                                */
if (DEBUG) gprintf("apply_spc rc=%d\n", rc);
    if (rc != SQLITE_OK) {
        stkerr(" apply_spc: ","failure querying for constrained nodes");
        return 0;
    }

    /* store a working array to hold all dof; a simple numbered list
     * from 0 to nDOF-1
     */
    if (!matstk(n_nodes*DOF_PER_NODE, 1,  "_all_dof"))     return 0;
    all_dof = tos->mat;
    for (i = 0; i < n_nodes*DOF_PER_NODE; i++) { 
        all_dof[i] = i;
    }

    n_nonspc_dof = n_nodes*DOF_PER_NODE;
if (DEBUG) gprintf("apply_spc sql_data.nRows=%d\n", sql_data.nRows);
    for (i = 0; i < sql_data.nRows; i++) { 
        /* loop over constrained nodes; zero out constrained dof */
        node = sql_data.i[i][0];
        bits = bits_from_123456(sql_data.i[i][1]);
        if (bits & (1 << 0)) { 
            all_dof[node*DOF_PER_NODE + 0] = 0; --n_nonspc_dof;
        }
        if (bits & (1 << 1)) { 
            all_dof[node*DOF_PER_NODE + 1] = 0; --n_nonspc_dof;
        }
        if (bits & (1 << 2)) { 
            all_dof[node*DOF_PER_NODE + 2] = 0; --n_nonspc_dof;
        }
        if (bits & (1 << 3)) { 
            all_dof[node*DOF_PER_NODE + 3] = 0; --n_nonspc_dof;
        }
        if (bits & (1 << 4)) { 
            all_dof[node*DOF_PER_NODE + 4] = 0; --n_nonspc_dof;
        }
        if (bits & (1 << 5)) { 
            all_dof[node*DOF_PER_NODE + 5] = 0; --n_nonspc_dof;
        }
    }
if (DEBUG) gprintf("apply_spc 1 n_nonspc_dof=%d\n", n_nonspc_dof);

    /* populate the list of dof that are not constrained by SPC's */
    if (!matstk(n_nonspc_dof, 1, name))     return 0;
    good_dof   = 0;
    nonspc_dof = tos->mat;
    for (i = 0; i < n_nodes*DOF_PER_NODE; i++) { 
        if (all_dof[i] > 0) {
            nonspc_dof[good_dof++] = all_dof[i];
        }
        if (good_dof > n_nonspc_dof) {
            drop();
            drop();
            stkerr(" apply_spc: ", "dof count mismatch A");
            return 0;
        }
    }
if (DEBUG) gprintf("apply_spc 1 good_dof=%d\n", good_dof);
    if (good_dof != n_nonspc_dof) {
        stkerr(" apply_spc: ", "dof count mismatch B");
        return 0;
    }

    lop();      /* drop the working array */
    swap();     /* bring dbh to top so it can be closed */
    db_close();

    return 1;

} /* 1}}} */

int bits_from_123456(int constraint) { /* {{{1 */
    /* Set bits of return_value according to the string representation
     * of constraint.  Examples:
     *
     *     contraint     return_value 
     *                   (shown here in base 2)
     *
     *      123456          111111
     *         123          000111
     *        1456          111001
     *          24          001010
     *
     * The bit position count goes from right to left, ie, constraint = 3 
     * means set the 3rd bit from the right to 1.  Another way of looking
     * at it is constraint 1 = Tx, 2 = Ty, 3 = Tz, 4 = Rx, 5 = Ry, 6 = Rz
     * then the bits of the return value are in this order:
     *      Rz Ry Rx Tz Ty Tz
     */
    char string_rep[7] = { 0, 0, 0, 0, 0, 0, 0 };
    int  i, return_value = 0;
    snprintf(string_rep, 7, "%06d", constraint);
    for (i = 0; i < 6; i++) {
        switch ( (int) string_rep[i] ) {
            case '1':  return_value += (1 << 0); break;
            case '2':  return_value += (1 << 1); break;
            case '3':  return_value += (1 << 2); break;
            case '4':  return_value += (1 << 3); break;
            case '5':  return_value += (1 << 4); break;
            case '6':  return_value += (1 << 5); break;
            default: /* ignore anything outside 1:6 */ break;
        }
    }
    return return_value;
} /* 1}}} */
