/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

#ifdef KEYBOARD
/* key.c  June 1999

   Copyright (c) 1999   D. R. Williamson
*/

#define NOMACROS
#define _NO_PROTO
#include <curses.h>
#undef NOMACROS
#undef _NO_PROTO

#define _NO_PROTO
#include <string.h>
#undef _NO_PROTO

#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#include <time.h>
#ifndef __USE_BSD
   #define __USE_BSD /* for usleep */
#endif
#include <unistd.h>

#include "main.h"
#include "stk.h"
#include "lib.h"

#include "ctrl.h"
#include "exe.h"
#include "inpo.h"
#include "key.h"
#include "mat.h"
#include "mem.h"
#include "sys.h"
#include "term.h"
#include "tex.h"
#include "tag.h"

/* Key code notes:

   On AIX machines, Shift-Backspace sends the escape sequence for the
   Delete key, which is an escape (x27) followed by a pair of codes.

   Since the keyboard interface in this program does not service escape
   sequences, under AIX Backspace will not work while Shift is pressed.
   This matches AIX command line behavior, which also does not backspace
   while the shift key is pressed.

   Under Linux, Backspace works whether or not Shift is pressed.
*/

int KEVENT; /* 1 if key event is in progress: keys being hit until NL */

/* KEVENT note:
   Keyboard functions fired by high level words must test KEVENT for a 
   key event in progress to avoid doing anything that affects an active
   key event (a key event ends when the NL key is hit).  

   Examples are function khpush() fired by word history and function 
   story() fired by word story, where initialization of history arrays 
   is correctly skipped if KEVENT is true.

   It is rare that a high level word runs while keys are being pressed 
   prior to the NL key that sends a keyed phrase of high level words off
   to be run. 

   But one example is the action of the Tab key, which causes high level
   word tabfill (file key.v) to be run in the middle of a key event to 
   complete a partially typed string.  

   Word tabfill runs word history to obtain a set of prior keyed words 
   for matching, so khpush() (a keyboard function in this file) will be
   entered.  

   Under this condition, if khpush() were to initialize history arrays 
   (by running hset()) as if it were being run outside of a key event, 
   unpredictable results, including segmentation fault, can be expected.
*/

static char keyed[TOKBUF]={0}; /* latest keyed phrase (with prompt) */

int kmode; /* key mode hierarchy is: TEX -> ESC -> INS or APP */
int tabbed; /* count of consecutive tab hits */

int online=0; /* line being edited (always 0 for single line) */
int curs=0; /* position of cursor in line being edited */
int linewidth=0; /* current line width (and cursor rightmost loc) */
int chdel=' '; /* character deleted by DEL key */
int KBoff; /* when KBoff=1, all key input is ignored in key() */

/* History array note:
   There is a different history array for each run level of typing. 

   A new level of typing is reached through function main2(), via high 
   level word UP (not related to #define UP and UP1 in this file).
   ESC-q returns to the previous level. 

   For typing at new levels, see UP demo in key.v. */

enum {hAPP,hNEW,hREP,hROW,hSTK,hVOL,hZRO}; /* options for khis() */
int onhist=0; /* history line currently online */
char *hp; /* pointer to history array */
unsigned int hlines; /* lines in history text */
unsigned int hwidth; /* width of history text */

int KMAP=0;
/* Special keys when KMAP is true (used in amap()). */
#define UESC 0x2F /* char / key, above the 8 on keypad, enables ESC */
#define ULT  0x34 /* number 4 key moves left in ESC mode */
#define URT  0x36 /* number 6 key moves rt in ESC mode */
#define UAFT 0x38 /* number 8 key starts INS mode after space */
#define UREP 0x30 /* number 0 key does replace at cursor in ESC mode */
#define UDEL 0x2E /* char . key deletes at cursor in ESC mode */
#define UPLS 0x2B /* char + key same as NL */
 
time_t kTIME; /* time of last key hit */

/* List ignore[] is used in key() and key_ignore():
   Ignore certain ch values below 25 that can have non-display char-
   acters and can temporarily mess up the cursor position. */
   #define LEN_IGNORE 27

   unsigned int ignore[LEN_IGNORE] =  /* 1=ignore */
      {0,1,1,0,0,1,1,1,0,0,0,0,0,0,1,1,1,0,1,0,0,1,1,1,1,1,0};

/* At the ready prompt, use word getch to see what each key returns;
   for example, Ctrl-D puts 4 on the stack.  These are the ones that
   are ignored here, and word keyignore can be used to set others:
      key    ch
     Ctrl-A   1
     Ctrl-B   2
     Ctrl-E   5
     Ctrl-F   6
     Ctrl-G   7
     Ctrl-N   14
     Ctrl-O   15
     Ctrl-P   16
     Ctrl-R   18
     Ctrl-U   21
     Ctrl-V   22
     Ctrl-W   23
     Ctrl-X   24
     Ctrl-Y   25 */

   int chIGNORE;

static int KBYTES=0; /* remaining bytes to read from socket In, stdin */

/*--------------------------------------------------------------------*/

int accept_keys() /* accept_keys (qPrompt --- false or qS true) */
/* Accepting key input at Prompt on same line, with vi-style editing.

   Word console in file key.v uses this function to drive the inter-
   active keyboard.  Other uses, also in file key.v, are in words 
   numinp, query, and strinp. */
{
   int len,more=1,nlines=0;

   ESCq=0; /* ESC-q has not been hit to exit */

/* IMPORTANT: clear the count of characters, chout (see discussion in
   file inpo.c, gprintf()): */
   chout=0; 

   dot(); /* display the prompt */

#ifdef X11
/* Save window of focus, where the prompt has just been displayed */
   winfocus();
   WINKEY=tos->real;
   drop();
#endif

   while(more) {

   /* Leftmost margin, INDENT, is set here only, and reflects the 
      offset due to the prompt just displayed. */
      INDENT=MIN(chout,GOMAX);

      kset(kBEG);
      while(keys(0)); /* runs until NL key or ESC-q */
      kset(kEND);
      ESCq=BYE; /* global flag that ESC-q was used to exit */

   /* The keyed line image is in GOUT, the same as displayed on the
      screen--including the prompt. */
      len=MAX(0,(int)(strlen(GOUT)-INDENT));

      if(!len && !nlines)
         return(pushint(xFALSE)); /* return if NL key on empty line */

      pushq2(GOUT+INDENT,len); /* text with prompt to stack */
      if(nlines) cat();        /* concatenate lines */
      if(BYE) {                /* return if BYE (Esc-q) */
         drop();
         return(pushint(xFALSE));
      }
   /* Key input is done, unless the ending character is a backslash. */
      if(!len || *(tos->tex+tos->col-1)!='\\') more=0;
      else {
         nlines=1; 
         tos->col--;    /* ignore backslash by decrementing count */
         pushstr("> "); /* prompt for continuation lines */
         nc();          /* new line and carriage return */
         dot();         /* display the prompt */
      }
   }

   if(tos->col) return(pushint(xTRUE));
   else {
      return(
         drop() &&
         pushint(xFALSE)
      );
   }
}

int amap(int c) 
/* Remapping key c if KMAP is true; word kon activates KMAP, word koff
   (or -kon) deactivates it.
   This set of mappings adds functions to the keypad. */
{
   int ch; 
   static int NLtog=0;

   if(!KMAP) return(c);

/* Remapping selected keys: */
   ch=c;
   if(ch==UPLS) ch=NL;

   switch(ch) {

      case UESC: ch=ESC; break; /* switching to ESC mode */

      default: 
         if(kmode==ESC) { /* special keys in ESC mode: */

            switch(ch) {
               case ULT:  ch=LT;   break;
               case URT:  ch=RT;   break;
               case UREP: ch=REP;  break;
               case UDEL: ch=DEL;  break;

               case UAFT: ch=AFT1; break;
               case NL:   ch=AFT1; break;
            }
         } 
         else { 
         /* Special behavior of NL (Enter) key: 
               one NL is SB, two NL in a row make NL */
            if(ch==NL && !NLtog) {
               NLtog++;
               return(SB); /* space bar */
            }
         }
   }
   NLtog=0;
   return(ch);
}

int curscol() /* COLS ( --- n) */
/* Push to stack the number of columns in window. 

   This is an example of making an inlinex from within a C function,
   instead of sourcing a file.

   Word COLS is linked in word.p to this function, curscol().  The
   first time this function runs, it makes an inline called __COLS
   which is the high level word that COLS will actually run.

   This word is run in key.v at start up, to cause early creation of
   inline __COLS so it falls behind the fence (and does not show up
   when "whos" is run). */
{
   char *name="__COLS";
   int catmsg;

/* Here is the text of the inline: */
   char *text=" \
      SBIN 'stty size >' cat scratch cat shell \
      scratch asciiload scratch delete \
      chop numbers 2nd pry";

/* Here is the phrase that makes the inline: */
   char *make="'__COLS' inlinex";

   if(!caton(tagnative(name))) {
      catmsg=CATMSG;
      CATMSG=0;

      pushstr(text);
      pushstr(make);
      xmain(0); /* making the inline */
      caton(tagnative(name));

      CATMSG=catmsg;
   }
   return(exeinline());
}

int curslin() /* LINES ( --- n) */
/* Push to stack the number of lines (rows) in window.

   This is an example of making an inlinex from within a C function,
   instead of sourcing a file.

   Word LINES is linked in word.p to this function, curslin().  The
   first time this function runs, it makes an inline called __LINES
   which is the high level word that LINES will actually run.

   This word is run in key.v at start up, to cause early creation of
   inline __LINES so it falls behind the fence (and does not show up
   when "whos" is run). */
{
   char *name="__LINES";
   int catmsg;

/* Here is the text of the inline: */
   char *text=" \
      SBIN 'stty size >' cat scratch cat shell \
      scratch asciiload scratch delete \
      chop numbers 1st pry";

/* Here is the phrase that makes the inline: */
   char *make="'__LINES' inlinex";

   if(!caton(tagnative(name))) {
      catmsg=CATMSG;
      CATMSG=0;

      pushstr(text);
      pushstr(make);
      xmain(0); /* making the inline */
      caton(tagnative(name));

      CATMSG=catmsg;
   }
   return(exeinline());
}

int ESCq1() /* ESCq ( --- f) */
/* Flag is true if ESC-q was used to exit from accept_keys(). */
{
   return(pushint(ESCq*xTRUE));
}

int getchk() /* getch ( --- n) */
/* Get a keyboard character and push it to the stack. */
{
   int ignore;

   ignore=chIGNORE;
   chIGNORE=0; /* key() will not ignore any characters */

   pushint(key());

   chIGNORE=ignore;
   return 1;
}

int getcht() /* getcht ( --- t n) */
/* Get a keyboard character and the time the key was pressed, and push 
   them to the stack. */
{
   unsigned char n=0;
   int ignore;

   ignore=chIGNORE;
   chIGNORE=0; /* key() will not ignore any characters */

   n=key();
   timesys(); 
   pushint(n);

   chIGNORE=ignore;
   return 1;
}

int getkeys() /* getkeys ( --- qS) */
/* Wait for keyboard characters and push them to the stack in a 
   string. */
{
   unsigned char ch,*S;
   int ig_save;

   ig_save=chIGNORE; 

   chIGNORE=0; /* key() will not ignore any characters */

   ch=key();

   chIGNORE=ig_save;

   if(!strstk(1+KBYTES,"_getkeys")) return 0;
   S=(unsigned char *)tos->tex;
   *S=ch;

   if(KBYTES) { /* read the remaining bytes from socket In (stdin) */
      S++;
      read(In,S,KBYTES);
      KBYTES=0;

   /* Make ch replacements the same as in key(): */
      chrrpl((char *)S,tos->col,BKS1,BKS);
      chrrpl((char *)S,tos->col,CR,NL);
   }
   return 1;
}

void hset(int opt, int level)
/* Setting history array for run level: there is a different history
   array for each run level of typing, reached through main2(). */
{
   static char *hP[NBUF]; /* array of run level histories */
   static unsigned int hLINES[NBUF]; /* array of hlines */
   static unsigned int hWIDTH[NBUF]; /* array of hwidth */
   int i=0;

   switch(opt) {

      case kBEG:
      /* Initialization for khis() function: */
         hp=*(hP+level); /* history array for this run level */
         hlines=*(hLINES+level);
         hwidth=*(hWIDTH+level);
         break;

      case kEND:
      /* Saving settings of khis() history array: */
         *(hP+level)=hp;
         *(hLINES+level)=hlines;
         *(hWIDTH+level)=hwidth;
         break;

      case kZRO:
      /* Setting zeroes when start window (done once only) */
         for(;i<NBUF;i++) {
            *(hP+i)=NULL;
            *(hLINES+i)=0;
            *(hWIDTH+i)=0;
         }
         break;
   }
}

void kback()
/* Backspacing the cursor in the text line and deleting the character
   it lands on if not in command (ESC) mode. */
{
   if(linewidth<=INDENT) return;

   switch(kmode) {

      case TEX:
         curs=linewidth-1; /* backspacing */
         linewidth=reprint(curs,1);
         break;

      case ESC:
         if(curs<=INDENT) return;
         curs--; /* backspacing */
         linewidth=reprint(curs,0);
         break;

      case INS:
         if(curs<=INDENT) return;
         if(curs==linewidth) {
            curs=linewidth-1; /* backspacing */
            linewidth=reprint(curs,1);
         }
         else {
            curs--;
            kdel();
         }
         break;
   }
}

int KBtoggle() /* KBtoggle ( --- ) */
/* Toggle flag KBoff.  When KBoff=1, all key input ignored in key(). */
{
   if(KBoff!=0) KBoff=0;
   else KBoff=1;

   return 1;
}

void kdel()
/* Deleting the character at the cursor, keeping the cursor in the same
   location or on rightmost character. */
{
   if(linewidth<=INDENT) return; /* empty line */

   chdel=*(GOUT+curs); /* saving deleted character */

   *(GOUT+linewidth)=' ';
   *(GOUT+linewidth+1)='\0';

   insbks(GOUT,curs);

   linewidth=reprint(linewidth-1,1);

   curs=MIN(curs,linewidth-1);
   curs=MAX(curs,INDENT);
   reprint(curs,0);

   return;
}

int key()
/* Capturing the next character from the keyboard. */
{
   unsigned char ch=0;
   int have_key;

   if(!bufup()) return 0;

   setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */
   have_key=0;

   if(!*(jmpready+onbuf)) { /* falls through immediately */
      *(jmpready+onbuf)=KEY;

      terminal(); /* sitting here waiting for a key */
      ioctl(In,FIONREAD,&KBYTES); /* KBYTES>1 if pasted input */
      have_key=1;
   }
   bufdn();

   if(have_key) { /* coming from terminal() */
      read(In,&ch,1); /* drain a byte from socket In (stdin) */
      KBYTES--;
   }
   else { /* coming from longjmp to KEY */
      ch=KEY_IN; /* reading KEY_IN from key_in() */
      KEY_IN=NL;
   }
   if(ch<chIGNORE) { /* special keys */
      if(ch==4 && *(ignore+4)==0) { /* Ctrl-D will exit: */
         pushstr("exit");
         nc();
         main1();
      }
      if(*(ignore+ch)) ch=0;
   }
   if(KBoff) ch=0;
   if(ch==BKS1) ch=BKS;
   if(ch==CR) ch=NL;

   kHIT++;
   kTIME=time(&kTIME)+DELTA_T; /* true GMT */
   return(ch);
}

int keyboard() /* keyboard ( --- ) */
/* Setting up for interactive keys.  On return, the program is ready to
   receive input through function accept_keys().

   As an examples, word console in file key.v uses accept_keys() to
   to drive an interactive prompt, and word tinyc in file usr/tiny 
   uses prompter() in term.c that runs accept_keys() . */
{
   if(KEYS) {
      gprintf(" keyboard is running");
      nc();
      if(stkdepth()) return(drop());
      else return 1;
   }
   if(keysoff()) {
      KEYS=1;
      keywin(kBEG); /* on AIX, opens file descriptor 3 */

      tty();
      if(tos->col) TTY1=memgetn(tos->tex,tos->col);
      else {
         gprintf(\
            " keyboard: tty failed for interactive shell, and words");
         nc();
         gprintf(\
            "   running shell commands will display improperly;");
         nc();
         gprintf(\
            "   try running the tty command at the Unix prompt");
         nc();
      }
      drop();

      kHIT=0;
      kTIME=time(&kTIME)+DELTA_T; /* true GMT */
   }
   return 1;
}

int keyignore() /* keyignore (n --- ) */
/* Set nth key in list ignore, to be ignored.  See list ignore[] in 
   this file; offset n ranges from 0 to LEN_IGNORE-1. */
{
   int k;

   if(!popint(&k)) return 0;

   if(k<0 || k>LEN_IGNORE-1) {
      stkerr(" keyignore: ","n is out of range");
      return 0;
   }
   *(ignore+k)=1;
   return 1;
}

char *keying() 
/* Put present keyed input into global array keyed, and return the
    address of keyed. */
{
   int len;

   len=MAX(0,(int)(strlen(GOUT)));
   memcpy(keyed,GOUT,len);
   *(keyed+len)='\0';

   return keyed;
}

int keys(int c1)
/* Entering and editing characters in a line until NL key, when 0 is
   returned.  The text entered will be in GOUT, the memory image of
   displayed text used by gprintf().

   If incoming c1 is 0, gets next character from keyboard.

   Mimics a subset of Unix vi-style line editing functions.

   Uses nested switch() statements and recursion for three modes, or
   states, in the following hierarchy:

      TEX - low mode: basic text key entry
      ESC - middle mode: command key action and cursor movement
      INS - top mode: basic text key entry at current cursor position

   Input argument c1>0 allows this function to recurse on itself and
   implement its own series of key strokes.  For example, case DIN be-
   low performs keys(DEL) and keys(INS) to implement delete and insert.
   Cases AFT, APP, PUT, and REP also recurse with selected keys. */
{
   int c;
   static int retINS=1;

   static char *p;
   static int ch0,saved=0;

/* While waiting for a key, display tracking held in GOUT is isolated 
   from any display output by words that the multitasker might call,
   by substituting dummy GOUT1 to receive output from gprintf(): */
   p=GOUT;
   ch0=chout;

   GOUT=GOUT1; 
   *GOUT='\0';
   chout=0;
   saved=1;

   if(c1) c=c1;

   else 
      c=key(); /* sitting here until get character from keyboard;
                  meanwhile, the multitasker may be running words
                  and doing things */

   if(saved) { 
   /* Restoring display variables: */
      GOUT=p; /* now set to receive text of keyed characters */
      chout=ch0;
      saved=0;
   }
   c=amap(c);

   if(c==TAB) tabbed++;
   else tabbed=0;

   /*-----------------------------------------------------------------*/

   switch(c) { /* Outer.  Four cases of key: NL, BKS, ESC, default */

   case NL: /* newline key: done */

      if(linewidth>INDENT)
         khis(hREP,GOUT+INDENT,linewidth-INDENT,&online,NULL);
       
      else khis(hREP," ",1,&online,NULL);
       
      retINS=0; /* closing out INS mode if active */

      return 0;

   case BKS:  /* backspace */
      kback(); /* this function handles modes TEX, ESC, INS */
      break;

   case ESC: /* command mode key was just pressed */

      /*--------------------------------------------------------------*/

      switch(kmode) { /* Three cases of current state: ESC, INS, TEX */

      case ESC: /* already in command mode */
         beep1(); break;

      case INS: /* return to INS recursion and finish INS mode */
         return 0;

      default: /* TEX: going to command mode */
         kmode=ESC;
         kback();

         while(keys(0)); /* ESC recursion until key NL */

         kmode=TEX;
         return 0;
      }
      break;

   default: /* cases in outer level */

      /*--------------------------------------------------------------*/

      switch(kmode) { /* Middle.  Three mode states: TEX INS ESC */

      default: /* TEX: running in text mode */
         linewidth=newch(c);
         curs=linewidth;
         break;

      case INS: /* running in insert mode */
         kins(c); /* keep inserting */
         break;

      case ESC: /* running in command mode */

         /*-----------------------------------------------------------*/

         switch(c) { /* Inner ESC.  Control key cases in ESC mode */

         default: /* unknown key pressed */
            beep1(); break;

         case QT: /* quitting from ESC mode; no keys are returned */
            BYE=1;
            linewidth=INDENT; /* zero width so this line not in hist */
            return keys(NL); /* like hitting NL key */

         case DEL: /* deleting the character at the cursor */
            kdel(); /* deleted character is in chdel for PUT */
            break;

         case UP: /* reviewing earlier keyed history */
         case UP1:
            khscrol(UP);
            break;

         case DN: /* reviewing later keyed history */
         case DN1:
            khscrol(DN);
            break;

         case LT: /* moving cursor left */
         case LT1:
            kback();
            break;

         case RT: /* moving cursor right */
         case RT1:
         case SB:
            if(linewidth>INDENT) {
               if(curs<linewidth-1) curs++;
            }
            reprint(curs,0);
            break;

         case REP: /* replacing character at the cursor */
            if(linewidth<=INDENT) {
               beep1();
               break;
            }
            c=key();
            if(c<' ') break;
            krep(c);
            return(keys(LT));
            break;

         case PUT: /* appending character that was last deleted */
            if(linewidth>INDENT) curs++;
            kins(chdel);
            return(keys(LT));

         case DC: /* deleting characters from cursor to rightmost */
            linewidth=reprint(curs,1);
            kback();
            return 1;

         case DIN: /* delete and insert */
            keys(DEL);
            return keys(INS);

         case AFT1: /* like AFT but puts a space after text before 
                       appending */
            if(linewidth>INDENT) {
               curs=linewidth;
               linewidth=reprint(curs,0);
               kmode=TEX; keys(32); kmode=ESC;
            }
            linewidth=reprint(curs,0);
            return keys(INS);

         case AFT: /* jumping cursor to after text and appending */
            curs=MAX(INDENT,linewidth-1);
            return keys(APP);

         case APP: /* appending text past the cursor */
            if(linewidth>INDENT) {
               if(curs<linewidth) curs++;
               else {
                  curs=linewidth;
                  kmode=TEX; keys(32); kmode=ESC;
               }
            }
            linewidth=reprint(curs,0);
            return keys(INS);

         case INS: /* inserting text at cursor */
            kmode=INS; /* jumping from command mode to insert mode */
            retINS=1;
            while(keys(0)); /* INS recursion until ESC or NL */
            /* got ESC or NL key; initializing for return to ESC mode */
            kmode=ESC;
            if(curs>INDENT) kback();
            return retINS;
         }
      }
   }
   return 1;
}

int key_in() /* key_in (qS --- ) */
/* Send string S to be processed as if it had been received from the 
   keyboard.  See man key_in and man remotekeys. */
{
   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" key_in: ",STRORVOLNOT);
      return 0;
   }
   KEY_IN=*(tos->tex);

   return(drop() && bufunwind(KEY,KEY,1));
}

int keysoff()
{
   return(keywin(kEND));
}

int keywin(int opt)
/* Initializing or closing key window. 

   Runs curses newterm() to initialize the screen. 

   In Linux, man ncurses, the manual for the freely distributable form 
   of the curses library, gives a lot of information about curses.

   NOTE: 
      Using curses SCREEN enables non-display of Esc keys and operation
      of backspace.  

      To eliminate curses would require this program to take over these 
      functions in key(). */
{
   static SCREEN *KEYWIN=0; /* Ncurses screen */

   if(!KEYS) return 1; 

   switch(opt) {

   case kBEG: /* initializing: */

   /* On AIX, curses newterm() opens file descriptor 3: */
      KEYWIN=newterm(NULL,stdin,stdin); 
      stty1(); /* running word stty in key.v */

      kset(kZRO);

      return 1;

   case kEND:
      if(KEYWIN) {

      /* On AIX, closing file descriptor 3: */
         delscreen(KEYWIN);
         KEYWIN=0;

         KEYS=0;
         endwin(); /* restore tty modes to before newterm() */
      }
      return 1;

   default:
      stkerr(" keywin: "," invalid opt");
      return 0;
   break;
   }
}

char *khis(int opt, char *newlin, int linlen, int *lin, int *wid)
/* Interfacing with history array for these cases of opt:
      hAPP - appending string newlin to history volume
      hNEW - opening new line in history volume
      hREP - replacing history line defined in lin with string newlin
      hROW - returning pointer to selected row
      hSTK - pushing history volume to the stack
      hVOL - returning pointer to history volume, and its size: number
             of lines in lin, character width in wid
      hZRO - emptying the history volume */ 
{
   char *p=NULL;
   static char *r=NULL;
   int i=0,len,ret=0;

   switch(opt) {

      case hNEW: /* opening new, latest line for keying input: */
         *lin=0; /* line 0 is the latest line */

         if(!hlines || !hp || strlen1(hp,hwidth)) {
            return(khis(hAPP," ",1,lin,wid));
         }
         return hp;

      case hROW: /* returning pointer and wid for history lin: */
         p=hp+loclin(*lin,hwidth);
         *wid=MAX(1,strlen1(p,hwidth)); 
         mallfree((void *)&r);
         if((r=(char *)memgetc(1,*wid))==NULL) return NULL;
         memcpy(r,p,*wid);
         *(r+*wid)='\0';
         return r;

      case hREP: /* replacing history line, lin, with newlin: */

         if(!linlen) return NULL;

         if(!hlines) return(khis(hAPP,newlin,linlen,lin,wid));

         if(*lin>hlines-1) {
            stkerr(" khis: ",OUTROW);
            return NULL;
         }
         if(linlen>hwidth) {

            if((p=(char *)memgetc(hlines,linlen))==NULL) return NULL;
         
            for(;i<*lin;i++) 
               memcpy(p+loclin(i,linlen),hp+loclin(i,hwidth),hwidth);
 
            memcpy(p+loclin(*lin,linlen),newlin,linlen);

            for(i=1+*lin;i<hlines;i++) 
               memcpy(p+loclin(i,linlen),hp+loclin(i,hwidth),hwidth);

            mallfree((void *)&hp);
            hwidth=linlen;
            hp=p;
         }
         else {
            memset(hp+loclin(*lin,hwidth),' ',hwidth);
            memcpy(hp+loclin(*lin,hwidth),newlin,linlen);
         }
         return hp;

      case hVOL: /* returning pointer to hist volume, and its size: */
         *lin=hlines;
         *wid=hwidth;
         if(hlines) return hp;
         else return NULL;

      case hSTK: /* pushing hist volume to stack: */
         if((p=(char *)memgetc(hlines,hwidth))==NULL) return NULL;
         if(hp) memcpy(p,hp,hlines*hwidth);
         ret=(
            push(VOL,(char *)memgetn("_hist",5),NOTAG,0,NULL, \
               p,hlines,hwidth,NULL) 
         );
         if(ret) return p;
         else return NULL;

      case hZRO: /* initializing hist volume: */
         mallfree((void *)&hp);
         hlines=0;
         hwidth=0;
         return NULL;

      case hAPP: /* appending new line to hist volume */
         if(!linlen) return NULL;

         ret=pushq2(newlin,linlen); /* newest on top */
         if(!ret) return NULL;

         if(hlines) {
            ret=(
               typstr2vol() &&
               push(VOL,(char *)memgetn("_hist",5),NOTAG,0,NULL, \
                  hp,hlines,hwidth,NULL) &&
               pilev() && 
               asciify() /* clean out bad display characters */
            );
            if(!ret) {
               drop();
               return NULL;
            }
         }
         hwidth=tos->col;
         hlines=MIN(hlines+1,HLMAX);

         if((p=(char *)malloc(1+(len=hlines*hwidth)))==NULL) {
            drop();
            return NULL;
         }
         memcpy(p,tos->tex,len); /* copying volume from stack */
         *(p+len)='\0';

         drop(); 
        
         hp=p;
         return hp;

      default:
         return NULL;
   }
}

int khit() /* khit ( --- n) */
/* Number of keys hit since start up (includes copy-and-drop). */
{
   return(pushint(kHIT));
}

int khpush() /* history ( --- hT) */
/* Pushing history volume to the stack. 

   Assumes history for the run level of the last keyed input, set by
   kset(). */
{
   if(!KEVENT) hset(kBEG,RLKEY); /* initialization for khis() */

   return(
      (khis(hSTK,NULL,0,NULL,NULL)!=NULL) &&
      asciify() &&
      reversed() &&
      pushstr("_history") &&
      naming()
   );
}

void khscrol(int opt)
/* Scrolling through history array. */
{
   static char linsav[1+GOMAX];
   static int cursav;
   char *p=NULL;
   int len,lins;

   khis(hVOL,NULL,0,&lins,&len); /* getting size: lins-by-len */
   if(!lins) {
      beep1();
      return;
   }
   switch(opt) {

   case UP: /* to earlier */
      if(onhist==lins-1) { /* earliest one */
         beep1();
         return;
      }
      if(onhist==online) {
         memcpy(linsav,GOUT+INDENT,(len=MIN(GOMAX-INDENT,\
            linewidth-INDENT)));
         len=MAX(INDENT,len);
         *(linsav+len)='\0';
         cursav=curs;
      }
      else {
         if(linewidth>INDENT)
            khis(hREP,GOUT+INDENT,linewidth-INDENT,&onhist,NULL);
         else 
            khis(hREP," ",1,&onhist,NULL);
      }
      onhist++;
      p=khis(hROW,NULL,0,&onhist,&len);
      curs=INDENT;
      break;

   case DN: /* to latest */
      if(onhist==online) {
         if(linewidth!=INDENT) beep1(); /* only if latest not empty */
         return;
      }
      else { 
         if(linewidth>INDENT)
            khis(hREP,GOUT+INDENT,linewidth-INDENT,&onhist,NULL);
         else 
            khis(hREP," ",1,&onhist,NULL);
      }
      onhist--;
      if(onhist==online) {
         p=linsav;
         len=strlen(linsav);
         curs=cursav;
      }
      else {
         p=khis(hROW,NULL,0,&onhist,&len);
         curs=INDENT;
      }
      break;
   }
   khis(hREP,p,len,&online,NULL);

   reprint(INDENT,1);
/* erase1(INDENT); */
   gprintf("%s",p);
   linewidth=reprint(curs,0);
}

void kins(int c)
/* Inserting a character on the left side of the cursor. */
{
   if(linewidth==TOKBUF+INDENT) {
      beep1(); return;
   }
   if(c==TAB) {
      tabfill(keying());
      return;
   }
   if(linewidth>INDENT) {
      memmove((GOUT+curs+1),(GOUT+curs),linewidth-curs); /* spread */
   }
   *(GOUT+curs)=c; /* insert */
   linewidth++; /* bump length */
   linewidth=MIN(linewidth,GOMAX);
   *(GOUT+linewidth)='\0';

   reprint(linewidth,1);
   curs++;
   reprint(curs,0);
}

int koff() /* koff ( --- ) */
/* Turning key mapping off. */
{
   KMAP=0;
   return 1;
}

int kon() /* kon ( --- ) */
/* Turning on key mapping.  Algorithm defined in amap() will work. */
{
   KMAP=1;
   return 1;
}  

void krep(int c)
/* Replacing character at cursor with c. */
{
   if(linewidth==TOKBUF+INDENT) {
      beep1();
      return;
   }
   if(c==TAB) {
      tabfill(keying());
      return;
   }
   *(GOUT+curs)=c;
   curs++;
   reprint(curs,0);
}

void kset(int opt)
/* Making key settings depending on opt:

      kBEG: key input beginning 
      kEND: key input ending 
      kZRO: setting zeroes at start up

   There is little need for TRACE while a few keys are being hit, so
   it is turned off when typing a line begins, case kBEG, and turned
   back on when the newline--NL--key is hit, case kEND. 

   TRACE on during key functions produces output that disrupts the
   typed line image in GOUT and resets pointer chout, by calling 
   gprintf().  (For this same reason, debug output placed here should
   use printf() for output, not gprintf().)

   This problem arose with TRACE inside drop(), which is called by
   khis() when a typed line is updated for the history array.  It 
   produced a segmentation fault. */
{
   static int tsav;

   switch(opt) {

      case kBEG: /* key input beginning */
      /* Setting run level of keying, info for khpush(), story(),
         and xmain() (exe.c): */
         RLKEY=onbuf; 
         ILKEY=onrun;

         KEVENT=1; /* key event in progress */
         tsav=TRACE;
         TRACE=0;

         tabbed=0;

      /* Initialization for keys() function: */
         BYE=0;
         chIGNORE=LEN_IGNORE;
         curs=MIN(chout,GOMAX);
         kmode=TEX; /* initial mode */
         KEY_IN=NL;
         linewidth=curs;

      /* Initialization for khis() function: */
         hset(kBEG,RLKEY); /* for khis() */
         khis(hNEW,NULL,0,&online,NULL); /* new line in history */
         onhist=online;
      break;

      case kEND: /* key input ending */
         hset(kEND,RLKEY); /* end khis() */
         TRACE=tsav;
         KEVENT=0; /* key event ended */

      /* Saving phrase for typing() (run by word typing): */
         keying();

      break;

      case kZRO: /* setting zeroes at start up */
         hset(kZRO,0);
      break;
   }
}

int ktime() /* ktime ( --- s) */
/* Time of last key hit. */
{
   return(pushint(kTIME));
}

int newch(int c)
/* Writes new character c to end of current line. */
{
   if(chout<TOKBUF+INDENT) {
      if(c==0) return(chout); /* key() sets ignored to 0 */
      if(c==FF) chout--; /* keeps chout constant after gprintf() */
      if(c==TAB) {
         tabfill(keying());
         return(chout);
      }
      return(gprintf("%c",c)); /* displaying the key on screen */
   }
   else 
      beep1();
   return(chout);
}

int putkeys() /* putkeys (qS --- ) */
/* This is under development.  It does not work. */
/* Write key characters in string S to stdin, followed by nl. */
{
   char nl=0x0D;

   if(tos->typ!=STR) {
      stkerr(" putkeys: ",STRNOT);
      return 0;
   }
   /*unquoted(); */
   write(In,(unsigned char *)tos->tex,tos->col);
   write(In,&nl,1);

   return(drop());
}

int story() /* story (n --- hT) or (0 --- ) */
/* Puts a volume of the latest n entries from the history volume on
   the stack unless n is 0.  If n is 0, the history volume is emptied
   and nothing is pushed to the stack (as in phrase "no story"). 

   Assumes history for the run level of the last keyed input, set by
   kset(). */
{
   int lins=0;
   double d;

   if(!KEVENT) hset(kBEG,RLKEY); /* initialization for khis() */

   if(!popd(&d) || (khis(hSTK,NULL,0,NULL,NULL)==NULL)) return 0;

   if(d) {
      lins=MIN(fabs(d),(double)tos->row);
      return(
         pushint(XBASE) && 
         pushint(lins) && 
         items() && 
         reach() &&
         asciify() &&
         reversed() &&
         pushstr("_story") &&
         naming()
      );
   }
   else { /* emptying the history volume (but only if not KEVENT): */

      if(!KEVENT) {
         khis(hZRO,NULL,0,NULL,NULL);

         hset(kEND,RLKEY); /* end of story() */
      }
      return(drop()); /* drop hT from khis() */
   }
}

void tabfill(char *keyed)
/* Fill out end of current string if unique match is found using 
   high level word tabfill in key.v. */
{
   char *name="tabfill"; /* high level word to run */
   
   if(curs!=strlen(keyed)) return; /* if not at end of line */

   pushstr(keyed);            /* current string to stack */
   pushint(xTRUE*(tabbed>1)); /* indicate consecutive tab hits */
   pushstr(name); main1();    /* running tabfill */
}

int tty() /* tty ( --- qS) */
/* Get the name of the character special file in the /dev directory. 
   This file is used by function shell. */
{
   int f=0,wrt_ok;
   char temp[5];
   char *command=
      "'tty >' scratch cat shell scratch asciiload 1st quote \
               1st word drop strchop scratch deleteif";

   if(TTY1) return(pushstr(TTY1));

   if(KEYS) {
      pushstr(command);
      main1(); /* run tty shell command */

      anyq(); /* anything in tos STR from tty? */
      popint(&f);
      if(f) { /* see if 1st 4 chars are /dev: */
         memcpy(temp,tos->tex,4);
         *(temp+4)='\0';
         if(strcmp(temp,"/dev")==0) {

         /* Verify that /dev/name on stack is writable: */
            dup1s();
            fallow();
            popint(&wrt_ok);
            if(wrt_ok) return 1;

            drop();
         }
      }
      return(pushstr(""));
   }
   else
   return(pushstr("/dev/null")); /* no shell tty output when batch */
}

int typing() /* typing ( --- qS) */
/* Putting latest phrase typed onto the stack. */
{
   if(!*(keyed)) return(pushstr(""));

   return(pushq2(keyed+INDENT,MAX(0,strlen(keyed)-INDENT)));
}
#endif
