/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

/* mmath.c  October 2000

Copyright (c) 2000  D. R. Williamson

*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "main.h"
#include "stk.h"

#include "exe.h"
#include "inpo.h"
#include "math1.h"
#include "mat.h"
#include "mem.h"
#include "mmath.h"
#include "sparse.h"
#include "tag.h"

#ifdef LAPACK
   #include "lapack.h"
#endif

#ifdef ESSL
   #include "essl.h"
#endif

/*--------------------------------------------------------------------*/

/* Some of this code was generated from Fortran using gnu f2c. */

/* In some functions, this is used for indexing in double arrays: */
#define locf2c(col,rows) (col-1)*rows 

/*--------------------------------------------------------------------*/

int ceig() /* ceig (hC f --- hAr hAi) */
/* Interface to complex eigenvalue and eigenvector functions. */
{
   double *C,*C1;
   int bal,ret,rows;
   register double *Rr,*Ri,*Ri1,w;
   register int i;

   char *clear="2 roll drop";
/*
   char *filtering="swap 1e-10 filter, swap 1e-10 filter";
*/

/* Pointers for eigenvectors (function modald_()): */
   double *B,*A,*EAR,*EAI,*AR,*AI,*PIVOTS,*WORK;
   int imp,iout,*kWORK,nopt,rows1;
   float tol;

   if(!popint(&bal)) return 0;

   if(tos->typ!=MAT) {
      stkerr(" ceig: ",MATNOT);
      return 0;
   }
   if(!cop()) return 0;

   rows=tos->row;
   if(tos->col!=rows) {
      stkerr(" ceig: ","matrix is not square");
      return 0;
   }
   C=tos->mat;
   if(!matstk(rows,1,"_Rr")) return 0;
   Rr=tos->mat;
   if(!matstk(rows,1,"_Ri")) return 0;
   Ri=tos->mat;

   if((C1=(double *)memget(rows,rows))==NULL) return 0;
   memcpy(C1,C,sizeof(double)*rows*rows);

/* run balanc here */

   if(!eig_(C1,Rr,Ri,rows)) {
      mallfree((void *)&C1);
      drop2();
      return 0;
   } 
   mallfree((void *)&C1);

   if(rows>1) { /* sorting eigenvalues: */
      over();
      over();
      park(); /* Rr Ri into 2-column matrix: [Rr Ri] */
      pushint(xFALSE); /* sorting in descending order */
      pushint(XBASE); /* sorting first column, Rr */
      sorton(); 

      Ri1=(tos->mat)+rows; /* pointer to sorted Ri */
      for(i=1;i<rows;i++) { /* setting complex conjugates */
         Ri1++;
         if((w=fabs(*Ri1))!=0) {
            if(fabs(*(Ri1-1))==w) {
               *Ri1=w;
               *(Ri1-1)=-w;
            }
         }
      }
      memcpy(Rr,tos->mat,sizeof(double)*rows);
      memcpy(Ri,(tos->mat)+rows,sizeof(double)*rows);
      drop();
   }
/* Pointers for eigenvector calculations: */

   if((A=(double *)memget(1,1))==NULL) return 0; /* nopt=1, A not used*/
   if((B=(double *)memget0(rows,rows))==NULL) return 0;
   for(i=0;i<rows;i++) *(B+i+locvec(i,rows))=1; /* B=identity mat */

   if((AR=(double *)memget(rows+1,rows))==NULL) return 0;
   if((AI=(double *)memget(rows+1,rows))==NULL) return 0;

   if((PIVOTS=(double *)memget(rows+1,1))==NULL) return 0;
   if((WORK=(double *)memget(rows+1,3))==NULL) return 0;
   if((kWORK=(int *)malloc((rows+1)*sizeof(int)))==NULL) return 0;

   if(!matstk(rows+1,rows,"_Ar")) return 0;
   EAR=tos->mat;
   if(!matstk(rows+1,rows,"_Ai")) return 0;
   EAI=tos->mat;

/* Eigenvector calculations: 
     imp=2 for amp/phase form [not supported], otherwise real/imag form
     nopt=1 for state space form [used here], otherwise quadratic 
*/
   imp=0;
   nopt=1;
   rows1=rows+1;
   tol=1e-6;
   iout=0;

   ret=modald_(C,&rows,&rows,Rr,Ri,B,A,&rows,&rows,&rows,
              EAR,EAI,&rows1,AR,AI,&nopt,&imp,&tol,
              PIVOTS,kWORK,WORK,&iout);
   mallfree((void *)&A);
   mallfree((void *)&B);
   mallfree((void *)&AR);
   mallfree((void *)&AI);
   mallfree((void *)&PIVOTS);
   mallfree((void *)&WORK);
   mallfree((void *)&kWORK);

/* Stack contents at this point: hC hRr hRi hAi hAr */

   if(!ret) { /* if eigvec error, returning eigvals hRr hRi on stack */
      stkerr(" ceig: ","eigenvector error");
      drop2();
      pushq2(clear,strlen(clear));
      xmain(0);
      return 0;
   }
/* DON'T FILTER HERE
   else {
      pushq2(filtering,strlen(filtering));
      xmain(0);
   }
*/
   return( /* returning Ar Ai on stack; eigvals are the 1st row */
      pushq2(clear,strlen(clear)) && xmain(0) &&
      pushq2(clear,strlen(clear)) && xmain(0) &&
      pushq2(clear,strlen(clear)) && xmain(0) 
   );
}

int chmats_(double *rr, double *ri, double *ar, double *ai, int *mx, 
   double *a, double *b, double *c__, int *mda, int *mdb, int *mdc, 
   int *n, int *nopt)
/* chmats.f -- translated by f2c (version 19991025). */
{
    /* System generated locals */
    int ar_dim1, ar_offset, ai_dim1, ai_offset, a_dim1, a_offset, b_dim1, b_offset, c_dim1, c_offset, i__1, i__2;
    double d__1, d__2, d__3;

    /* Local variables */
    static double amod, vmax;
    static int j, k;
    static double rrmri;
    static int n2;

/*         ASSEMBLES THE CHARACTERISTIC MATRIX FOR A GIVEN EIGENVALUE */
/*         S = (RR,RI).  THE CHARACTERISTIC MATIX HAS THE FORM */
/*                     [AR,AI] = [(A*S + B)*S + C] */
/*         UNLESS NOPT = 1, IN WHICH CASE THE MATRIX HAS THE FORM */
/*                         [AR,AI] = [B*S - C] */

/*         AR AND AI ARE ROW SCALED BY THE MAXIMUM MODULUS OF THE ROW. */

/*         DECLARE DOUBLE PRECISION FUNCTIONS: */

    /* Parameter adjustments */
    ai_dim1 = *mx;
    ai_offset = 1 + ai_dim1 * 1;
    ai -= ai_offset;
    ar_dim1 = *mx;
    ar_offset = 1 + ar_dim1 * 1;
    ar -= ar_offset;
    a_dim1 = *mda;
    a_offset = 1 + a_dim1 * 1;
    a -= a_offset;
    b_dim1 = *mdb;
    b_offset = 1 + b_dim1 * 1;
    b -= b_offset;
    c_dim1 = *mdc;
    c_offset = 1 + c_dim1 * 1;
    c__ -= c_offset;

    /* Function Body */
    rrmri = *rr * *rr - *ri * *ri;
    n2 = *n;
    i__1 = n2;
    for (j = 1; j <= i__1; ++j) {
 vmax = 0.;

 i__2 = n2;
 for (k = 1; k <= i__2; ++k) {
     if (*nopt == 1) {
  goto L626;
     }
     ar[j + k * ar_dim1] = a[j + k * a_dim1] * rrmri + b[j + k * 
      b_dim1] * *rr + c__[j + k * c_dim1];
     ai[j + k * ai_dim1] = *ri * (b[j + k * b_dim1] + *rr * 2. * a[j + 
      k * a_dim1]);
/* Computing 2nd power */
     d__1 = a[j + k * a_dim1] * rrmri;
/* Computing 2nd power */
     d__2 = b[j + k * b_dim1] * *rr;
/* Computing 2nd power */
     d__3 = *ri * (b[j + k * b_dim1] + *rr * 2. * a[j + k * a_dim1]);
     amod = d__1 * d__1 + d__2 * d__2 + c__[j + k * c_dim1] * c__[k + 
      j * c_dim1] + d__3 * d__3;
     goto L627;
L626:
     ar[j + k * ar_dim1] = b[j + k * b_dim1] * *rr - c__[j + k * 
      c_dim1];
     ai[j + k * ai_dim1] = b[j + k * b_dim1] * *ri;
/* Computing 2nd power */
     d__1 = b[j + k * b_dim1] * *rr;
/* Computing 2nd power */
     d__2 = b[j + k * b_dim1] * *ri;
     amod = d__1 * d__1 + c__[j + k * c_dim1] * c__[j + k * c_dim1] + 
      d__2 * d__2;
L627:
     amod = sqrt(amod);
     vmax = MAX(amod,vmax);
/* L629: */
 }

 if (vmax == 0.) {
     goto L630;
 }
 i__2 = n2;
 for (k = 1; k <= i__2; ++k) {
     ar[j + k * ar_dim1] /= vmax;
     ai[j + k * ai_dim1] /= vmax;
/* L628: */
 }
L630:
 ;
    }
    return 0;
} /* chmats_ */

int conj1() /* conj (hA --- hB) */
/* The complex conjugate of matrix A is returned in B. */
{
   double *A,*B;
   int k=0,spars=0;

   if(tos->typ==NUM) {
      cop();
      if(is_complex(tos)) tos->imag=-tos->imag;
      else set_complex(tos);
      return 1;
   }
   if(is_sparse(tos)) {
      dense();
      spars=1;
   }
   if(tos->typ!=MAT) {
      stkerr(" conj: ",MATNOT);
      return 0;
   }
   if(!is_complex(tos)) {
      return(
         cop() &&
         dup1s() &&
         dims() &&
         null() &&
         dblcmplx()
      );
   }
   A=tos->mat;

   if(!matstk(tos->row,tos->col,"_B")) return 0;
   B=tos->mat;
   set_complex(tos);

   for(;k<(tos->row*tos->col)/2;k++) {
      *B=*A;
      A++;
      B++;
      *B=-(*A);
      A++;
      B++;
   }
   if(spars) sparse();

   return(lop());
}

int cross_correlation() /* cross-correlation (hX hY d --- hR) */
/* Cross-correlation, also called covariance, of X and Y, with delay
   d applied to Y.

   X and Y must have the same dimensions.  This is not checked.

   Each row in X and Y is a data record, and each column is a step
   in time.  Columns of Y are delayed for d steps and the cross-
   correlation is computed (Ref: Bendat and Piersol, p. 120):
      Rxy(k) = [X'(k) * Y(k-d)] */
{
   double *R,*R1,*X,*Y;
   int cols,d,i=1,j=0,rows;

   if((tos-1)->typ!=MAT || (tos-2)->typ!=MAT) {
      stkerr(" cross-correlation: ",MATNOT2);
      return 0;
   }
   if(!popint(&d)) return 0;

   rows=tos->row;
   cols=tos->col;

   Y=tos->mat; /* starting at first column */
   X=(tos-1)->mat + locvec(d,tos->row); /* starting at column d */

   if(!matstk(cols,1,"_R")) return 0;
   R=tos->mat;

/* The first d rows of R are null: */
   memset(R,0,MIN(d,cols)*sizeof(double)); 
   if(d>=cols) return(lop() && lop());

   R1=R+d;

#ifdef LAPACK
   for (; j < cols-d; j++) {
      R1[j] = DDOT(&rows, &X[j*rows], &i, &Y[j*rows], &i);
   }
#else
   #ifdef ESSL
      dndot(cols-d,rows,R1,i,i,X,i,rows,Y,i,rows);
   #else
      for(;j<cols-d;j++) {
         *R1=0;
         for(i=0;i<rows;i++) {
            *R1+=*X*(*Y);
            X++;
            Y++;
         }
         R1++;
      }
   #endif
#endif
   return(lop() && lop());
}

int diagpre() /* diagpre (hV hA --- hB) */
/* Pre-multiply A by matrix diagonals stored in columns of matrix 
   V (produces row scaling of A). */
{
   double *A,*A0,*B,*B0,*V;
   register double *A1,*B1,d;
   register int i,j,k=0,rA;
   int cA,cV;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" diagpre: ",MATNOT2);
      return 0;
   }
   if(is_complex(tos) || is_complex(tos-1)) return(diagpre_complex());

   if((rA=(tos-1)->row)!=tos->row) {
      stkerr(" diagpre: ","rows of V must match rows of A");
      return 0;
   }
   V=(tos-1)->mat;
   cV=(tos-1)->col;

   A0=tos->mat;
   cA=tos->col;

   if(!matstk(rA,cA*cV,"_B")) return 0;
   B0=tos->mat;

   for(;k<cV;k++) {
      A=A0;
      B=B0+locvec(cA*k,rA);
      for(i=0;i<rA;i++) {
         d=*V;
         A1=A;
         B1=B;
         for(j=0;j<cA;j++) {
            *B1=(*A1)*d;
            A1+=rA;
            B1+=rA;
         }
         A++;
         B++;
         V++;
      }
   }
   return(
      lop() && lop()
   );
}

int diagpre_complex() /* (hV hA --- hB) */
/* Called by diagpre when one or both matrices are complex.  

   This function separates things into real and imaginary and calls 
   diagpre() with real matrices like this high level word in mmath.v:

   inline: diagpre_complex (hVr hVi hAr hAi --- hBr hBi)
\     Pre-multiplication of complex A by a complex diagonal matrix
\     stored in vector V:
\        Br = Vr*Ar - Vi*Ai
\        Bi = Vi*Ar + Vr*Ai
\     where * denotes operation by word diagpre.

      "Ai" book "Ar" book "Vi" book "Vr" book
      Vr Ar diagpre, Vi Ai diagpre less (hBr) "_Br" naming
      Vi Ar diagpre, Vr Ai diagpre plus (hBi) "_Bi" naming
      freed is Ar, freed is Ai, freed is Vr, freed is Vi
   end
*/
{
   cmplxmatch();
   swap();
   cmplxdbl(); /* Vr Vi */
   rot();
   cmplxdbl(); /* Ar Ai */

/* stack: Vr Vi Ar Ai
          3  2  1  0  */
   pushint(2); pick(); pushint(2); pick(); /* Vi Ar */ 
   diagpre(); /* stack: Vr Vi Ar Ai Vi*Ar
                         4  3  2  1  0     */
   pushint(4); pick(); pushint(2); pick(); /* Vr Ai */ 
   diagpre(); /* stack: Vr Vi Ar Ai Vi*Ar Vr*Ai
                         5  4  3  2  1     0  */
   plusm(); lpush(); /* Bi = Vi*Ar + Vr*Ai to temp stack */

/* stack: Vr Vi Ar Ai
          3  2  1  0  */
   pushint(3); pick(); pushint(2); pick(); /* Vr Ar */ 
   diagpre(); /* stack: Vr Vi Ar Ai Vr*Ar
                         4  3  2  1  0     */
   pushint(3); pick(); pushint(2); pick(); /* Vi Ai */ 
   diagpre(); /* stack: Vr Vi Ar Ai Vr*Ar Vi*Ai
                         5  4  3  2  1     0  */
   minusm(); /* Br = Vr*Ar - Vi*Ai */

   lpull();  /* Bi = Vi*Ar - Vr*Ai */
   dblcmplx(); /* hB on tos */

/* Lop Ai Ar Vi Vr: */
   return(lop() && lop() && lop() && lop());
}

int diagpost() /* diagpost (hA hV --- hB) */
/* Post-multiply A by matrix diagonals stored in columns of matrix 
   V (produces column scaling of A). */
{
   register double *A,*A0,*B,*B0,d,*V;
   register int i,j,k=0;
   int cA,cV,rA;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" diagpost: ",MATNOT2);
      return 0;
   }
   if(is_complex(tos) || is_complex(tos-1)) return(diagpost_complex());

   if((cA=tos->row)!=(tos-1)->col) {
      stkerr(" diagpost: ","rows of V must match columns of A");
      return 0;
   }
   A0=(tos-1)->mat;
   rA=(tos-1)->row;

   V=tos->mat;
   cV=tos->col;

   if(!matstk(rA,cA*cV,"_B")) return 0;
   B0=tos->mat;

   for(;k<cV;k++) {
      A=A0;
      B=B0+locvec(cA*k,rA);
      for(j=0;j<cA;j++) {
         d=*V;
         for(i=0;i<rA;i++) {
            *B=(*A)*d;
            A++;
            B++;
         }
         V++;
      }
   }
   return(
      lop() && lop()
   );
}

int diagpost_complex() /* (hA hV --- hB) */
/* Called by diagpost when one or both matrices are complex.  

   This function separates things into real and imaginary and calls 
   diagpost() with real matrices like this high level word in mmath.v:

   inline: diagpost_complex (hAr hAi hVr hVi --- hBr hBi)
\     Post-multiplication of complex A by a complex diagonal matrix
\     stored in vector V:
\        Br = Ar*Vr - Ai*Vi
\        Bi = Ai*Vr + Ar*Vi
\     where * denotes operation by word diagpost.

      "Vi" book "Vr" book "Ai" book "Ar" book
      Ar Vr diagpost, Ai Vi diagpost less (hBr) "_Br" naming
      Ar Vi diagpost, Ai Vr diagpost plus (hBi) "_Bi" naming
      freed is Ar, freed is Ai, freed is Vr, freed is Vi
   end
*/
{
   cmplxmatch();
   swap();
   cmplxdbl(); /* Ar Ai */
   rot();
   cmplxdbl(); /* Vr Vi */

/* stack: Ar Ai Vr Vi
          3  2  1  0  */
   pushint(2); pick(); pushint(2); pick(); /* Ai Vr */ 
   diagpost(); /* stack: Ar Ai Vr Vi Ai*Vr
                         4  3  2  1  0     */
   pushint(4); pick(); pushint(2); pick(); /* Ar Vi */ 
   diagpost(); /* stack: Ar Ai Vr Vi Ai*Vr Ar*Vi
                         5  4  3  2  1     0  */
   plusm(); lpush(); /* Bi = Ai*Vr + Ar*Vi to temp stack */

/* stack: Ar Ai Vr Vi
          3  2  1  0  */
   pushint(3); pick(); pushint(2); pick(); /* Ar Vr */ 
   diagpost(); /* stack: Ar Ai Vr Vi Ar*Vr
                         4  3  2  1  0     */
   pushint(3); pick(); pushint(2); pick(); /* Ai Vi */ 
   diagpost(); /* stack: Ar Ai Vr Vi Ar*Vr Ai*Vi
                         5  4  3  2  1     0  */
   minusm(); /* Br = Ar*Vr - Ai*Vi */

   lpull();  /* Bi = Ai*Vr - Ar*Vi */
   dblcmplx(); /* hB on tos */

/* Lop Vi Vr Ai Ar: */
   return(lop() && lop() && lop() && lop());
}

void dot1(double *Ain, double *Bin, int len, double *dot)
/* Dot product of vectors A and B. */
{
   register double *A,*B,d=0;
   register int k=0;
   A=Ain;
   B=Bin;
   for(;k<len;k++) d+=*(A+k)*(*(B+k));
   *dot=d;
}

double dsign(double z, double p)
/* z receives the sign of p */
{
   z=fabs(z);
   if(p<0) return(-z);
   else return(z);
}

int eig_(double *a, double *rr, double *ri, int n)
{
   int i=0;

   elmhes_(a,n); /* converting a to Hessenberg upper form */

   if(hqr_(a,n,rr,ri)) { /* finding eigenvalues in Hessenberg matrix */

      for (;i<n;i++) { /* making tiny terms equal to zero: */
         if(fabs(rr[i])<1e-8) rr[i]=0.;
         if(fabs(ri[i])<1e-8) ri[i]=0.;
      }
      return 1;
   }
   return 0;
}

int elmhes_(double *a1, int n)
/* From Press, W. H., et al, "Numerical Recipes in FORTRAN," 
   Cambridge Univeristy Press, second edition, 1992. 

   Reduction to upper Hessenberg form by the elimination method.

   The real, nonsymmetric n-by-n matrix a, stored in n-by-n 
   array, is replaced by an upper Hessenberg matrix with identi-
   cal eigenvalues.  Recommended, but not required, is that this
   routine be preceded by balanc.f.  On output, the Hessenberg 
   matrix is in elements a(i,j) with i <= j+1.  Elements with
   i > j+1 are to be thought of as zero, but are returned with
   random values. */
{
/* Most of this code was generated from Fortran to C using gnu f2c. */

    /* System generated locals */
    double d__1;
    int a_dim1, i__1, i__2, i__3;

    /* Local variables */
    double *a;
    static int i__, j, m;
    static double x, y;

/*  Adjusting incoming pointer a1 for Fortran 1-based indexing 
    used below in the function body: */
    a=a1;
    --a;
    a_dim1 = n;

    /* Function Body */

    i__1 = n - 1;
/* m is called r+1 in Reference (p. 478) */
for (m = 2; m <= i__1; ++m) { 
   x = 0.;
   i__ = m;
   i__2 = n;

   for (j = m; j <= i__2; ++j) {
/* find the pivot */
      if((d__1 = a[j + locf2c((m - 1),a_dim1)], fabs(d__1)) > fabs(x)) {
         x = a[j + locf2c((m - 1),a_dim1)];
         i__ = j;
      }
   }
 if (i__ != m) {
/* interchange rows i__ and m: */
    i__2 = n;
    for (j = m - 1; j <= i__2; ++j) { 
       y = a[i__ + locf2c(j,a_dim1)];
       a[i__ + locf2c(j,a_dim1)] = a[m + locf2c(j,a_dim1)];
       a[m + locf2c(j,a_dim1)] = y;
    }
/* interchange columns i__ and m: */
    i__2 = n;
    for (j = 1; j <= i__2; ++j) {
       y = a[j + locf2c(i__,a_dim1)];
       a[j + locf2c(i__,a_dim1)] = a[j + locf2c(m,a_dim1)];
       a[j + locf2c(m,a_dim1)] = y;
    }
 }
 if (x != 0.) {
/* carry out the elimination */
    i__2 = n;
    for (i__ = m + 1; i__ <= i__2; ++i__) {
       y = a[i__ + locf2c((m - 1),a_dim1)];

       if (y != 0.) {
          y /= x;
          a[i__ + locf2c((m - 1),a_dim1)] = y;

          i__3 = n;
          for (j = m; j <= i__3; ++j) { 
             a[i__ + locf2c(j,a_dim1)] -= y * a[m + locf2c(j,a_dim1)];
          }

          i__3 = n;
          for (j = 1; j <= i__3; ++j) {
             a[j + locf2c(m,a_dim1)] += y * a[j + locf2c(i__,a_dim1)];
          }
       }
    }
 }

}    
   return 1; 
} /* elmhes_ */

int hqr_(double *a1, int n, double *wr1, double *wi1)
/* From Press, W. H., et al, "Numerical Recipes in FORTRAN,"
   Cambridge Univeristy Press, second edition, 1992.

   Find all eigenvalues of an n-by-n upper Hessenberg matrix a1
   that is stored in an n-by-n array.  On input, a1 can be
   exactly as output from elmhes_(); on output it is destroyed.
   The real and imaginary parts of the eigenvalues are returned
   in wr1 and wi1. */
{
/* Most of this code was generated from Fortran to C using gnu f2c. */

/* Initialized data */

/* levels of iteration */
    static int max1= 10;
    static int max2= 20;
    static int max3= 30;

    /* System generated locals */
    int a_dim1, i__1, i__2, i__3, i__4;
    double d__1, d__2, d__3, d__4;

    /* Local variables */
    static int i__, j, k, l, m;
    static double p, q, r__, s, t, u, v, w, x, y, z__, anorm;
    static int nn, its;
    double *a,*wr,*wi;

/*  Adjusting incoming pointer a1 for Fortran 1-based indexing 
    used below in the function body: */
    a=a1;
    --a;
    wr=wr1;
    --wr;
    wi=wi1;
    --wi;

    a_dim1 = n;

    /* Function Body */

/* compute matrix norm for possible use in locating single small 
    subdiagonal element: */
    anorm = (d__1 = a[1], fabs(d__1));
    i__1 = n;
    for (i__ = 2; i__ <= i__1; ++i__) {
 i__2 = n;
 for (j = i__ - 1; j <= i__2; ++j) {
     anorm += (d__1 = a[i__ + locf2c(j,a_dim1)], fabs(d__1));
 }
    }
    nn = n;
    t = 0.; /* gets changed only by an exceptional shift */
L1:
    if (nn >= 1) { /* begin search for next eigenvalue */
 its = 0;

L2: /* begin iteration: look for single small subdiagonal element: */
 for (l = nn; l >= 2; --l) { 
     s = (d__1 = a[l - 1 + locf2c((l - 1),a_dim1)], fabs(d__1)) + (d__2 = a[l 
      + locf2c(l,a_dim1)], fabs(d__2));
     if (s == 0.) {
  s = anorm;
     }
     if ((d__1 = a[l + locf2c((l - 1),a_dim1)], fabs(d__1)) + s == s) {
  goto L3;
     }
 }
 l = 1;
L3:
 x = a[nn + locf2c(nn,a_dim1)];
 if (l == nn) { /* one root found */
     wr[nn] = x + t;
     wi[nn] = 0.;
     --nn;
 } else {
     y = a[nn - 1 + locf2c((nn - 1),a_dim1)];
     w = a[nn + locf2c((nn - 1),a_dim1)] * a[nn - 1 + locf2c(nn,a_dim1)];
     if (l == nn - 1) { /* two roots found ... */
  p = (y - x) * .5; 
/* Computing 2nd power */
  d__1 = p;
  q = d__1 * d__1 + w;

  z__ = sqrt(fabs(q));
  x += t;
  if (q >= 0.) { /* ... a real pair */
      z__ = p + dsign(z__, p); /* abs z times sign of p */
      wr[nn] = x + z__;
      wr[nn - 1] = wr[nn];
      if (z__ != 0.) {
   wr[nn] = x - w / z__;
      }
      wi[nn] = 0.;
      wi[nn - 1] = 0.;
  } else { /* ... a complex pair */
      wr[nn] = x + p;
      wr[nn - 1] = wr[nn];
      wi[nn] = z__;
      wi[nn - 1] = -z__;
  }
  nn += -2;
     } else { /* no roots found; continue iteration */
  if (its >= max3) {
      stkerr(" hqr eigensolver: ","too many iterations");
      return 0;
  }
/* form exceptional shift */
  if (its == max1 || its == max2) {
      t += x;
      i__1 = nn;
      for (i__ = 1; i__ <= i__1; ++i__) {
   a[i__ + locf2c(i__,a_dim1)] -= x;
      }
      s = (d__1 = a[nn + locf2c((nn - 1),a_dim1)], fabs(d__1)) + (d__2 
       = a[nn - 1 + locf2c((nn - 2),a_dim1)], fabs(d__2));
      x = s * .75f;
      y = x;
/* Computing 2nd power */
      d__1 = s;
      w = d__1 * d__1 * -.4375f;
  }
  ++its;
  i__1 = l;
  for (m = nn - 2; m >= i__1; --m) {
/* form shift and then look for two consecutive small subdiagonal 
   elements */
      z__ = a[m + locf2c(m,a_dim1)];
      r__ = x - z__;
      s = y - z__;
      p = a[m + locf2c((m + 1),a_dim1)]; /* eq. 11.6.23 */
      if (a[m + 1 + locf2c(m,a_dim1)] != 0.) {
   p += (r__ * s - w) / a[m + 1 + locf2c(m,a_dim1)];
      }
      q = a[m + 1 + locf2c((m + 1),a_dim1)] - z__ - r__ - s;
      r__ = a[m + 2 + locf2c((m + 1),a_dim1)];
      s = fabs(p) + fabs(q) + fabs(r__);
/* scale to prevent overflow or underflow */
      p /= s;
      q /= s;
      r__ /= s;
      if (m == l) { /* eq. 11.6.26 */
   goto L4;
      }
      u = (d__1 = a[m + locf2c((m - 1),a_dim1)], fabs(d__1)) * (fabs(q) 
       + fabs(r__));
      v = fabs(p) * ((d__1 = a[m - 1 + locf2c((m - 1),a_dim1)], fabs(
       d__1)) + fabs(z__) + (d__2 = a[m + 1 + locf2c((m + 1),a_dim1)], fabs(d__2)));
      if (u + v == v) {
   goto L4;
      }
  }
L4:
  i__1 = nn;
  for (i__ = m + 2; i__ <= i__1; ++i__) {
      a[i__ + locf2c((i__ - 2),a_dim1)] = 0.;
      if (i__ != m + 2) {
   a[i__ + locf2c((i__ - 3),a_dim1)] = 0.;
      }
  }
  i__1 = nn - 1;
  for (k = m; k <= i__1; ++k) {
/* double QR step on rows l to nn and columns m to nn begin setup 
   of Householder vector: */
      if (k != m) {
   p = a[k + locf2c((k - 1),a_dim1)];
   q = a[k + 1 + locf2c((k - 1),a_dim1)];
   r__ = 0.;
   if (k != nn - 1) {
       r__ = a[k + 2 + locf2c((k - 1),a_dim1)];
   }
   x = fabs(p) + fabs(q) + fabs(r__);
   if (x != 0.) {
       p /= x;
/* scale to prevent overflow or overflow */
       q /= x;
       r__ /= x;
   }
      }
/* Computing 2nd power */
      d__2 = p;
/* Computing 2nd power */
      d__3 = q;
/* Computing 2nd power */
      d__4 = r__;
      d__1 = sqrt(d__2 * d__2 + d__3 * d__3 + d__4 * d__4);
      s = dsign(d__1, p);
      if (s != 0.) {
   if (k == m) {
       if (l != m) {
    a[k + locf2c((k - 1),a_dim1)] = -a[k + locf2c((k - 1),a_dim1)];
       }
   } else {
       a[k + locf2c((k - 1),a_dim1)] = -s * x;
   }
   p += s; /* eq. 11.6.24 */
   x = p / s;
   y = q / s;
   z__ = r__ / s;
   q /= p;
   r__ /= p;
   i__2 = nn;
   for (j = k; j <= i__2; ++j) { /* row modification */
       p = a[k + locf2c(j,a_dim1)] + q * a[k + 1 + locf2c(j,a_dim1)];
       if (k != nn - 1) {
    p += r__ * a[k + 2 + locf2c(j,a_dim1)];
    a[k + 2 + locf2c(j,a_dim1)] -= p * z__;
       }
       a[k + 1 + locf2c(j,a_dim1)] -= p * y;
       a[k + locf2c(j,a_dim1)] -= p * x;
   }
/* Computing MIN */
   i__3 = nn, i__4 = k + 3;
   i__2 = MIN(i__3,i__4);
   for (i__ = 1; i__ <= i__2; ++i__) { /* column modification */
       p = x * a[i__ + locf2c(k,a_dim1)] + y * a[i__ + locf2c((k + 1),a_dim1)];
       if (k != nn - 1) {
    p += z__ * a[i__ + locf2c((k + 2),a_dim1)];
    a[i__ + locf2c((k + 2),a_dim1)] -= p * r__;
       }
       a[i__ + locf2c((k + 1),a_dim1)] -= p * q;
       a[i__ + locf2c(k,a_dim1)] -= p;
   }
      }
  }
  goto L2; /* ... for next iteration of current eigenvalue */
     }
 }
 goto L1; /* ... for next eigenvalue */
    }
    return 1;
} /* hqr_ */

int inverd_(int *ndim, double *a, int *n, double *b, int *m, \
   double *determ, int *ising, int *index)
/* inverd.f -- translated by f2c (version 19991025). */
{
    /* System generated locals */
    int a_dim1, a_offset, b_dim1, b_offset, index_dim1, index_offset, 
     i__1, i__2, i__3, i__4;
    static int equiv_0[1], equiv_1[1];
    static double equiv_2[1];

    /* Local variables */
#define amax (equiv_2)
#define swap (equiv_2)
#define irow (equiv_0)
#define jrow (equiv_0)
    static double d__;
    static int i__, j, k, l;
#define t (equiv_2)
    static int l1;
    static double pivot;
#define icolum (equiv_1)
#define jcolum (equiv_1)

/* NDIM IS THE ACTUAL SIZE OF A IN CALLING PROGRAM, 
   E.G., A(NDIM,NDIM). 
   A IS SQUARE MATRIX TO BE INVERTED. 
   N IS SIZE OF UPPER LEFT PORTION BEING INVERTED. 
   B IS COLUMN OF CONSTANTS (OPTIONAL INPUT).  SUPPLY SPACE 
   B(NDIM,1) MINIMUM.
   M IS THE NUMBER OF COLUMNS IN B. 
   DETERM RETURNS THE VALUE OF DETERMINANT IF NON-SINGULAR 
   ISING RETURNS 2 IF MATRIX A(N,N) IS SINGULAR
                 1 IF MATRIX A(N,N) IS NON-SINGULAR.
   INDEX IS WORKING STORAGE (N,3).
   INVERSE RETURNS IN A.
   SOLUTION VECTORS RETURN IN B.
*/

/*     DOUBLE PRECISION A,B,AMAX,T,SWAP,DETERM,PIVOT */

/*         INITIALIZE */
    /* Parameter adjustments */
    b_dim1 = *ndim;
    b_offset = 1 + b_dim1 * 1;
    b -= b_offset;
    a_dim1 = *ndim;
    a_offset = 1 + a_dim1 * 1;
    a -= a_offset;
    index_dim1 = *n;
    index_offset = 1 + index_dim1 * 1;
    index -= index_offset;

    /* Function Body */
    *determ = 1.;
    i__1 = *n;
    for (j = 1; j <= i__1; ++j) {
       /* L10: */
       index[j + index_dim1 * 3] = 0;
    }
    i__1 = *n;
    for (i__ = 1; i__ <= i__1; ++i__) {

      /*         SEARCH FOR PIVOT */
       *amax = 0.;
       i__2 = *n;
       for (j = 1; j <= i__2; ++j) {
          if (index[j + index_dim1 * 3] == 1) {
             goto L40;
          }
          i__3 = *n;
          for (k = 1; k <= i__3; ++k) {
             if ((i__4 = index[k + index_dim1 * 3] - 1) < 0) {
                goto L20;
             } 
             else if (i__4 == 0) {
                goto L30;
             } 
             else {
               goto L190;
             }
             L20:
             d__ = a[j + k * a_dim1];
             if (fabs(d__) <= *amax) {
                goto L30;
             }
             *irow = j;
             *icolum = k;
             *amax = fabs(d__);
             L30: ; 
          }
          L40: ;
       }
       ++index[*icolum + index_dim1 * 3];
       index[i__ + index_dim1] = *irow;
       index[i__ + (index_dim1 << 1)] = *icolum;

/*         INTERCHANGE ROWS TO PUT PIVOT ELEMENT ON DIAGONAL */
       if (*irow == *icolum) {
          goto L70;
       }
       *determ = -(*determ);
       i__2 = *n;
       for (l = 1; l <= i__2; ++l) {
          *swap = a[*irow + l * a_dim1];
          a[*irow + l * a_dim1] = a[*icolum + l * a_dim1];
          /* L50: */
          a[*icolum + l * a_dim1] = *swap;
       }
       if (*m <= 0) {
          goto L70;
       }
       i__2 = *m;
       for (l = 1; l <= i__2; ++l) {
          *swap = b[*irow + l * b_dim1];
          b[*irow + l * b_dim1] = b[*icolum + l * b_dim1];
          /* L60: */
          b[*icolum + l * b_dim1] = *swap;
       }

/*         DIVIDE THE PIVOT ROW BY PIVOT ELEMENT */
L70:
       pivot = a[*icolum + *icolum * a_dim1];
       *determ *= pivot;
       if (pivot == 0.) {
          goto L190;
       }
       a[*icolum + *icolum * a_dim1] = 1.;
       i__2 = *n;
       for (l = 1; l <= i__2; ++l) {
          /* L80: */
          a[*icolum + l * a_dim1] /= pivot;
       }
       if (*m <= 0) {
          goto L100;
       }
       i__2 = *m;
       for (l = 1; l <= i__2; ++l) {
          /* L90: */
          b[*icolum + l * b_dim1] /= pivot;
       }

   /*         REDUCE NON-PIVOT ROWS */
       L100:
       i__2 = *n;
       for (l1 = 1; l1 <= i__2; ++l1) {
          if (l1 == *icolum) {
             goto L130;
          }
          *t = a[l1 + *icolum * a_dim1];
          a[l1 + *icolum * a_dim1] = 0.;
          i__3 = *n;
          for (l = 1; l <= i__3; ++l) {
             /* L110: */
             a[l1 + l * a_dim1] -= a[*icolum + l * a_dim1] * *t;
          }
          if (*m <= 0) {
             goto L130;
          }
          i__3 = *m;
          for (l = 1; l <= i__3; ++l) {
             /* L120: */
             b[l1 + l * b_dim1] -= b[*icolum + l * b_dim1] * *t;
          }
          L130: ;
       }
    }

/*         INTERCHANGE COLUMNS */
    i__2 = *n;
    for (i__ = 1; i__ <= i__2; ++i__) {
       l = *n + 1 - i__;
       if (index[l + index_dim1] == index[l + (index_dim1 << 1)]) {
          goto L150;
       }
       *jrow = index[l + index_dim1];
       *jcolum = index[l + (index_dim1 << 1)];
       i__1 = *n;
       for (k = 1; k <= i__1; ++k) {
          *swap = a[k + *jrow * a_dim1];
          a[k + *jrow * a_dim1] = a[k + *jcolum * a_dim1];
          a[k + *jcolum * a_dim1] = *swap;
          /* L140: */
       }
       L150: ;
    }
    i__2 = *n;
    for (k = 1; k <= i__2; ++k) {
       if (index[k + index_dim1 * 3] == 1) goto L160;
       *ising = 2;
       goto L190;
      L160: ;
    }
    *ising = 1; return 1;
    L190: *ising = 2; return 0;
} /* inverd_ */

#undef jcolum
#undef icolum
#undef t
#undef jrow
#undef irow
#undef swap
#undef amax

int mmax1() /* mmax1 (hV ht ntau --- hV1) */
/* Moving maximum.  V1(k) holds the maximum value of vector V in the 
   interval t(k) to t(k)-tau. */
{
   double max,*t,*t0,tau,*ti,tk,*V,*V1,*Vi;
   int f,i,k=0,rows;

   if(!popd(&tau)) return 0;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" mmax1: ",MATNOT2);
      return 0;
   }
   if(tos->row != (tos-1)->row) {
      stkerr(" mmax1: ",ROWSNOT);
      return 0;
   }
   rows=tos->row;
   t=tos->mat;
   t0=t;

   V=(tos-1)->mat;

   if(!matstk(rows,1,"_V1")) return 0;
   V1=tos->mat;

   tk=*t+tau;
   max=*V;
   Vi=V;
   while(*t<tk && k<rows) {
      max=MAX(max,*Vi);
      *V1=max;
      Vi++;
      V1++;
      t++;
      k++;
   }
   for(;k<rows;k++) {
      tk=*t-tau;
      f=bsearchd(tk,t0,rows,&i); /* f=1 if tk is found in t0 */
      i+=1-f; /* use next index if f=0 (tk not found) */

      Vi=V+i;
      ti=t0+i;
      max=*Vi;

      while(i<=k) {
         max=MAX(max,*Vi);
         Vi++;
         i++;
      }
      *V1=max;
      V1++;
      t++;
   }
   return(lop() && lop());
}

/*

This version of mmax1 is not suitable real time analysis because of the
backward running loop, which can hit different past values when future 
points are added, thus "revising" previous results that may have been 
used.

It has been replaced by the uncommented version of mmax1.

int mmax1() // mmax1 (hV ht ntau --- hV1) //
// Moving maximum.  V1(k) holds the maximum value of vector V in the
   interval t(k) to t(k)-tau. //
{
   double max,*t,t0,tau,*ti,tk,*V,*V1,*Vi;
   int i,rows;

   if(!popd(&tau)) return 0;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" mmax1: ",MATNOT2);
      return 0;
   }
   if(tos->row != (tos-1)->row) {
      stkerr(" mmax1: ",ROWSNOT);
      return 0;
   }
   rows=tos->row;
   ti=(tos->mat)+rows-1;
   t0=*(tos->mat);

   Vi=((tos-1)->mat)+rows-1;

   if(!matstk(rows,1,"_V1")) return 0;
   V1=(tos->mat)+rows-1;

   for(i=rows-1;i>-1;i--) {
      V=Vi;
      t=ti;
      max=*V;
      tk=MAX(*t-tau,t0);
      while(*t>tk) {
         t--;
         V--;
         max=MAX(max,*V);
      }
      *V1=max;
      V1--;
      Vi--;
      ti--;
   }
   return(lop() && lop());
}
*/

int mmin1() /* mmin1 (hV ht ntau --- hV1) */
/* Moving minimum.  V1(k) holds the minimum value of vector V in the 
   interval t(k) to t(k)-tau. */
{
   double min,*t,*t0,tau,*ti,tk,*V,*V1,*Vi;
   int f,i,k=0,rows;

   if(!popd(&tau)) return 0;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" mmin1: ",MATNOT2);
      return 0;
   }
   if(tos->row != (tos-1)->row) {
      stkerr(" mmin1: ",ROWSNOT);
      return 0;
   }
   rows=tos->row;
   t=tos->mat;
   t0=t;

   V=(tos-1)->mat;

   if(!matstk(rows,1,"_V1")) return 0;
   V1=tos->mat;

   tk=*t+tau;
   min=*V;
   Vi=V;
   while(*t<tk && k<rows) {
      min=MIN(min,*Vi);
      *V1=min;
      Vi++;
      V1++;
      t++;
      k++;
   }
   for(;k<rows;k++) {
      tk=*t-tau;
      f=bsearchd(tk,t0,rows,&i); /* f=1 if tk is found in t0 */
      i+=1-f; /* use next index if f=0 (tk not found) */

      Vi=V+i;
      ti=t0+i;
      min=*Vi;

      while(i<=k) {
         min=MIN(min,*Vi);
         Vi++;
         i++;
      }
      *V1=min;
      V1++;
      t++;
   }
   return(lop() && lop());
}

/*

This version of mmin1 is not suitable real time analysis because of the
backward running loop, which can hit different past values when future 
points are added, thus "revising" previous results that may have been 
used.

It has been replaced by the uncommented version of mmin1.

int mmin1() // mmin1 (hV ht ntau --- hV1) //
// Moving minimum.  V1(k) holds the minimum value of vector V in the 
   interval t(k) to t(k)-tau. //
{
   double min,*t,t0,tau,*ti,tk,*V,*V1,*Vi;
   int i,rows;

   if(!popd(&tau)) return 0;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" mmin1: ",MATNOT2);
      return 0;
   }
   if(tos->row != (tos-1)->row) {
      stkerr(" mmin1: ",ROWSNOT);
      return 0;
   }
   rows=tos->row;
   ti=(tos->mat)+rows-1;
   t0=*(tos->mat);

   Vi=((tos-1)->mat)+rows-1;

   if(!matstk(rows,1,"_V1")) return 0;
   V1=(tos->mat)+rows-1;

   for(i=rows-1;i>-1;i--) {
      V=Vi;
      t=ti;
      min=*V;
      tk=MAX(*t-tau,t0);
      while(*t>tk) {
         t--;
         V--;
         min=MIN(min,*V);
      }
      *V1=min;
      V1--;
      Vi--;
      ti--;
   }
   return(lop() && lop());
}
*/

int modald_(double *c__, int *n, int *m, double *rr, double *ri, 
   double *b, double *a, int *mdc, int *mdb, int *mda, 
   double *ear, double *eai, int *mx, double *ar, double *ai, 
   int *nopt, int *imp, float *tol, double *pivots, int *kwork, 
   double *work, int *w)
/* modald.f -- translated by f2c (version 19991025). */

/* Sizes: kwork needs 1 column; work needs 3 columns */
{
    /* System generated locals */
    int c_dim1, c_offset, b_dim1, b_offset, a_dim1, a_offset, ear_dim1, 
     ear_dim2, ear_offset, eai_dim1, eai_dim2, eai_offset, ar_dim1, 
     ar_offset, ai_dim1, ai_offset, work_dim1, work_offset, i__1, i__2;
    double d__1;

    /* Local variables */
    static int ncol, irel, nvct;
    static float zero, tiny;
    static int multiple;
    static int i__, j, ncalc, ipass, n2, jc, ji, ki, ir;
    static float ridiff;
    static float rrdiff;
    static int jcm, irm;

/*         GIVEN M EIGENVALUES S = (RR,RI), THIS S/R PRESIDES OVER */
/*         THE CALCULATION OF THE REAL AND COMPLEX PARTS (EAR,EAI) */
/*         OF THE MODAL MATRIX OF THE N-DIMENSIONAL QUADRATIC SYSTEM */

/*                   [(A*S + B)*S + C]*(EAR,EAI) = 0 */

/*         IF NOPT = 1, A IS CONSIDERED NULL (AND NOT USED) AND C */
/*         IS GIVEN A NEGATIVE SIGN, SO THAT THE DYNAMIC SYSTEM IS */
/*         ASSUMED TO BE IN STATE SPACE FORM: */

/*                        [B*S - C]*(EAR,EAI) = 0 */

/*         IF IMP = 2, AMP/PHASE FORM OF MODE SHAPES IS CALCULATED */

    /* Parameter adjustments */
    --rr;
    --ri;
    c_dim1 = *mdc;
    c_offset = 1 + c_dim1 * 1;
    c__ -= c_offset;
    b_dim1 = *mdb;
    b_offset = 1 + b_dim1 * 1;
    b -= b_offset;
    a_dim1 = *mda;
    a_offset = 1 + a_dim1 * 1;
    a -= a_offset;
    work_dim1 = *mx;
    work_offset = 1 + work_dim1 * 1;
    work -= work_offset;
    --kwork;
    --pivots;
    ai_dim1 = *mx;
    ai_offset = 1 + ai_dim1 * 1;
    ai -= ai_offset;
    ar_dim1 = *mx;
    ar_offset = 1 + ar_dim1 * 1;
    ar -= ar_offset;
    eai_dim1 = *mx;
    eai_dim2 = *m;
    eai_offset = 1 + eai_dim1 * (1 + eai_dim2 * 1);
    eai -= eai_offset;
    ear_dim1 = *mx;
    ear_dim2 = *m;
    ear_offset = 1 + ear_dim1 * (1 + ear_dim2 * 1);
    ear -= ear_offset;

    /* Function Body */
    tiny = (float)1e-6;
    zero = (float)1e-6;
    if (*tol > (float)0.) {
 zero = *tol;
    }

/*         INITIALIZE */
    ipass = 1;
    n2 = *n;
/*     WRITE(W,51) */
/*  51 FORMAT(1H1) */
    ncol = 1;
    ncalc = 0;

/*         TOP OF MAIN LOOP */
    i__ = 1;
    while(i__ <= *m) {
 irel = 0;
 nvct = 1;

/*     Check for multiple roots.  Assumes roots are sorted so multiple */
/*     ones are next to each other. */
 j = i__ + 1;
 multiple = 1;
 while(j <= *m && multiple) {
     rrdiff = (d__1 = rr[i__] - rr[j], fabs(d__1));
     ridiff = (d__1 = ri[i__] - ri[j], fabs(d__1));
     if (rrdiff < tiny && ridiff < tiny) {
  ++nvct;
/* bump vector count */
     } else {
  multiple = 0;
     }
     ++j;
 }
/*         TEST TO SEE IF THIS ROOT IS A COMPLEX CONJUGATE OF THE LAST */
 if (ri[i__] == 0.) {
     goto L625;
 }
 if (i__ == 1) {
     goto L625;
 }
 ridiff = (d__1 = ri[i__ - 1] + ri[i__], fabs(d__1));
 if (ridiff < tiny) {
     i__1 = nvct;
     for (ji = 1; ji <= i__1; ++ji) {
  i__2 = n2;
  for (ki = 1; ki <= i__2; ++ki) {
      ai[ki + ji * ai_dim1] = -ai[ki + ji * ai_dim1];
  }
     }
     goto L633;
 }
/*         SET UP THE COMPLEX MATRIX REQUIRED TO COMPUTE VECTOR(S) */
L625:
 chmats_(&rr[i__], &ri[i__], &ar[ar_offset], &ai[ai_offset], mx, &a[
  a_offset], &b[b_offset], &c__[c_offset], mda, mdb, mdc, n, 
  nopt);

/*         SET REAL FLAG, IREL, IF SYSTEM IS REAL (I.E., RR(I) = 0): */
 if (rr[i__] == 0.) {
     irel = 1;
 }
/*         CALCULATE THE VECTOR(S) */
 vectrd_(&ar[ar_offset], &ai[ai_offset], &n2, mx, mx, &irel, &nvct, &
  zero, &pivots[1], &kwork[1], &work[work_dim1 + 1], &work[(
  work_dim1 << 1) + 1], w);

/*         PLACE THE EIGENVALUES AND VECTORS INTO THE READOUT MATRIX */
L633:
 i__1 = nvct;
 for (jc = 1; jc <= i__1; ++jc) {
     jcm = jc + ncol - 1;
     ear[(jcm + ipass * ear_dim2) * ear_dim1 + 1] = rr[i__];
     eai[(jcm + ipass * eai_dim2) * eai_dim1 + 1] = ri[i__];
     i__2 = n2;
     for (ir = 1; ir <= i__2; ++ir) {
  irm = ir + 1;
  ear[irm + (jcm + ipass * ear_dim2) * ear_dim1] = ar[ir + jc * 
   ar_dim1];
  eai[irm + (jcm + ipass * eai_dim2) * eai_dim1] = ai[ir + jc * 
   ai_dim1];
/* L637: */
     }
/* L638: */
 }
 ncol += nvct;
 ++ncalc;

/*         WRITE THE PIVOTS */
 if (*w > 0) {
/*
Message goes here
*/
 }

 if (ncol > *m) {
     goto L680;
 }
/* L675: */
 i__ += nvct;
    }
/*         BOTTOM OF MAIN LOOP */

L680:
    --ncol;
    return 1;
} /* modald_ */

int s3coef() /* s3coef (hX ht --- hC) */
/* Create coefficients for cubic spline fit to data in X at points t.

   Each column in X defines a curve at points t, so N-by-M matrix X
   and corresponding N-by-1 vector t define M curves at N points.

   The columnwise structure of returned coefficient matrix C is as
   follows:

      0  1  2  3  4   5  6  7  8   ...
      t; x1 b1 c1 d1; x2 b2 c2 d2; ... xM bM cM dM

   where xk is column k of X, and bk ck dk are the spline coefficients
   for column k of X, and M is the number of columns of X.

   Note: If rows of incoming X and t do not match, it is assumed that
   X contains data stored by columns and X is transposed to align rows
   of X to match rows of t. */
{
   register double *X,*C,*t;
   register int bytes,cols,j=0,k=1,rows;
   int ccols;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" s3coef: ",MATNOT2);
      return 0;
   }
   if((rows=tos->row)!=(tos-1)->row) {
      if((tos-1)->col==rows) {
         swap(); bend(); swap(); /* assuming X cols match t rows */
      }
      else {
         stkerr(" s3coef: ",ROWSNOT);
         return 0;
      }
   }
   bytes=rows*sizeof(double);
   cols=(tos-1)->col;
   X=(tos-1)->mat;
   t=tos->mat;

   if(!matstk(rows,(ccols=1+4*cols),"_s3coef")) return 0;
   C=tos->mat;
   memcpy(C,t,bytes);

   for(;j<cols;j++) {
      memcpy(C+locvec(k,rows),X+locvec(j,rows),bytes);
      k++;
      spline3(rows,t,X+locvec(j,rows),C+locvec(k,rows));
      k+=3;
   }
   return(lop() && lop());
}

int s3eval() /* s3eval (hC ht --- hX hX' hX'') */
/* Evaluate cubic spline at points in t using coefficients C from
   s3coef().  Points in t need not match the original ones used to
   create coefficients C.

   The columnwise structure of coefficient matrix C is as follows:

      0   1   2   3   4    5   6   7   8    ...
      tf; Xf1 bf1 cf1 df1; Xf2 bf2 cf2 df2; ... XfM bfM cfM dfM

   where the first column, tf, contains points where Xf was fitted by
   s3coef(), Xfk is column k of fitted Xf, bfk cfk dfk are the spline
   coefficients of Xfk, and M is the number of columns (functions) in
   fitted Xf.

   Returned matrices X, X' and X'' will contain M columns for the M
   functions in C, and N rows for the N rows of incoming t. */
{
   double *bf,*cf,*df,dt,*t,*tf,*Xf;
   double *X,*Xd,*Xdd,*x,*xd,*xdd;
   int i=0,j,N,Nf,Nf4,M,r;
   char *roll4="4 roll";

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" s3eval: ",MATNOT2);
      return 0;
   }
   N=tos->row;
   M=(((tos-1)->col)-1)/4;
   Nf=(tos-1)->row;
   Nf4=Nf*4;

   t=tos->mat;

   tf=(tos-1)->mat; /* first column of C */

   if(!matstk(N,M,"_X")) return 0;
   X=tos->mat;
   if(!matstk(N,M,"_Xd")) return 0;
   Xd=tos->mat;
   if(!matstk(N,M,"_Xdd")) return 0;
   Xdd=tos->mat;

   for(;i<N;i++) {
      bsearchd(*(t+i),tf,Nf,&r); /* nearest-below ti in tf */
      dt=*(t+i)-*(tf+r);
      x=X;
      xd=Xd;
      xdd=Xdd;

      Xf=tf+Nf;
      bf=Xf+Nf;
      cf=bf+Nf;
      df=cf+Nf;

      /* Evaluation at ti:
          x = Xf(r) + dt*(bf(r) + dt*(cf(r) + dt*df(r)))
         xd = bf(r) + dt*(2*cf(r) + 3*dt*df(r))
        xdd = 2*cf(r) + 6*dt*df(r) */

      for(j=0;j<M;j++) {
         *x=*(Xf+r) + dt*(*(bf+r) + dt*(*(cf+r) + dt*(*(df+r))));
         *xd=*(bf+r) + dt*(*(cf+r)*2 + 3*dt*(*(df+r)));
         *xdd=2*(*(cf+r)) + 6*dt*(*(df+r));

         Xf+=Nf4;
         bf+=Nf4;
         cf+=Nf4;
         df+=Nf4;

         x+=N;
         xd+=N;
         xdd+=N;
      }
      X++;
      Xd++;
      Xdd++;
   }
   pushq2(roll4,strlen(roll4));
   main1();
   pushq2(roll4,strlen(roll4));
   main1();
   return(drop2());
}

int skyline() /* skyline (hA --- hA1) */
/* A1 is flat at all peaks of A columns.

   Peaks in columns of A are where:
      the value is greater than or equal to the previous
      the value is greater than or equal to the next
      and both of these are true.

   This word replaces high level skyline now in the Appendix of math.v
   and works the same except for differences due to using greater-than-
   or-equal tests rather than greater-than tests. */
{
   double *A,*R;
   int cols,i,j=0,rows;

   if(tos->typ!=MAT) {
      stkerr(" skyline: ",MATNOT);
      return 0;
   }
   cols=tos->col;
   rows=tos->row;
   lpush(); /* put A on local stack */

   for(;j<cols;j++) {
      lpeek();
      pushint(XBASE+j);
      catch(); /* column Aj on stack */
      A=tos->mat;
      A++;

      if(!matstk(rows,1,"_R")) return 0;
      R=tos->mat;
      *R=0;
      R++;

      for(i=1;i<rows-1;i++) {
         if(*A>=*(A-1) && *A>=*(A+1)) *R=1;
         else *R=0;
         A++;
         R++;
      }
      *R=0;
      looking(); /* (hAj hR --- hA1j) */
   }
   pushint(cols);
   return(parkn() && lpull() && drop());
}

int solve() /* solve (hb hA --- hx) */
/* Solve A*x=b for x. */
{
#if defined(LAPACK) || defined(ESSL)

 #ifdef LAPACK

   if(is_complex(tos)) return(zgesv1());
   else return(dgesv1());

 #endif

 #ifdef ESSL

   return(solveSSL()); /* b and A can be complex */

 #endif

#else

/* Solve A*x=b for x using gaussian elimination. */

   double *A,*b=NULL,determ;
   int cols,*index,ising,rows;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" solve: ",MATSNOT);
      return 0;
   }
   rows=tos->row;
   if(tos->col!=rows) {
      stkerr(" solve: ","matrix A is not square");
      return 0;
   }
   if((tos-1)->col>0 && (tos-1)->row!=rows) {
      stkerr(" solve: ","matrix b not compatible with A");
      return 0;
   }
   cols=(tos-1)->col;

   A=tos->mat;
   if(!matstk(rows,rows,"_A")) return 0;
   memcpy(tos->mat,A,sizeof(double)*rows*rows);
   A=tos->mat;

   if(cols) {
      b=(tos-2)->mat;
      if(!matstk(rows,cols,"_x")) return 0;
      memcpy(tos->mat,b,sizeof(double)*rows*cols);
      b=tos->mat;
   }
   if((index=(int *)malloc(1+rows*3*sizeof(int)))==NULL) {
      stkerr(" solve: ",MEMNOT);
      return 0;
   }

   if(!inverd_(&rows,A,&rows,b,&cols,&determ,&ising,index)) {
      gprintf(" solve: singular matrix"); nc();
      mallfree((void *)&index);
      if(cols) drop2();
      else drop();
      return 0;
   }
   mallfree((void *)&index);
   if(cols) return(lop() && lop() && lop());
   else return(lop() && lop());

#endif
}


void spline3(int n, double *x, double *y, double *coef)
/* Computing the coefficients for fitting a cubic spline to n data
   point pairs (x,y); points x are in ascending order.

   Reference: Forsythe, G. E. and Michael A. Malcolm, "Computer
   Methods for Mathematical Computations," Prentice-Hall, Inc.,
   1977.

   The spline representation s(x) is

      s(x) = y(i) + b(i)*(x - x(i)) + c(i)*(x - x(i))**2
             + d(i)*(x - x(i))**3

   for x(i) .ge. x .le. x(i+1). */
{
   register int i=0,nm1;
   register double *b,*c,*d,t;

   b=coef;
   c=b+n;
   d=c+n;

   if(n<2) {
      *b=0;
      *c=0;
      *d=0;
      return;
   }
   if(n<3) {
      *b=(*(y+1) - *y)/(*(x+1)- *x);
      *c=0;
      *d=0;
      *(b+1)=*b;
      *(c+1)=0;
      *(d+1)=0;
      return;
   }
   nm1=n-1;

/* Setting up tridiagonal system
      b = diagonal, d = offdiagonal, c = right hand side */
   *d=*(x+1)-*x;
   *(c+1)=(*(y+1)-*y)/(*d);

   for(i=1;i<nm1;i++) {
      *(d+i)=*(x+i+1)-*(x+i);
      *(b+i)=2*(*(d+i-1)+*(d+i));
      *(c+i+1)=(*(y+i+1)-*(y+i))/(*(d+i));
      *(c+i)=*(c+i+1)-*(c+i);
   }
/* End conditions; third derivatives at x(1) and x(n)
   obtained from divided differences (see reference): */
   *b=-*d;
   *(b+nm1)=-*(d+nm1-1);
   *c=0;
   *(c+nm1)=0;
   if(n>3) {
      *c=*(c+2)/(*(x+3)-*(x+1)) - *(c+1)/(*(x+2)-*x);
      *(c+nm1)=*(c+nm1-1)/(*(x+nm1) - *(x+nm1-2)) \
               - *(c+nm1-2)/(*(x+nm1-1)-*(x+nm1-3));
      *c=*c*((*d)*(*d))/(*(x+3)-*x);
      *(c+nm1)=-*(c+nm1)*(*(d+nm1-1)*(*(d+nm1-1))) \
               /(*(x+nm1)-*(x+nm1-3));
   }
/* Forward elimination: */
   for(i=1;i<n;i++) {
      t=*(d+i-1)/(*(b+i-1));
      *(b+i)=*(b+i) - t*(*(d+i-1));
      *(c+i)=*(c+i) - t*(*(c+i-1));
   }
/* Back substitution: */
   *(c+nm1)=*(c+nm1)/(*(b+nm1));
   for(i=nm1-1;i>=0;i--) {
      *(c+i)=(*(c+i) - *(d+i)*(*(c+i+1)))/(*(b+i));
   }
/* c(i) is now sigma(i) of the reference;
      computing polynomial coefficients: */
   *(b+nm1)=(*(y+nm1) - *(y+nm1-1))/(*(d+nm1-1)) \
            + *(d+nm1-1)*(*(c+nm1-1) + *(c+nm1)*2);
   for(i=0;i<nm1;i++) {
      *(b+i)=(*(y+i+1) - *(y+i))/(*(d+i)) \
             - *(d+i)*(*(c+i+1) + *(c+i)*2);
      *(d+i)=(*(c+i+1) - *(c+i))/(*(d+i));
      *(c+i)=*(c+i)*3;
   }
   *(c+nm1)=*(c+nm1)*3;
   *(d+nm1)=*(d+nm1-1);

   return;
}

int vectrd_(double *ar, double *ai, int *n, int *nrdim, int *nidim,
  int *irel, int *nvct, float *zro, double *pivots, int *ncol,
  double *wr, double *wi, int *iout)
/* vectrd.f -- translated by f2c (version 19991025). */
{
    /* System generated locals */
    int ar_dim1, ar_offset, ai_dim1, ai_offset, i__1, i__2, i__3;
    double d__1;

    /* Local variables */
    static float amax;
    static double temp, tsti, tstr;
    static int i__, j, k, l, m, icame;
    static float aiwas, arwas;
    static int ipass;
    static double pvtsz, fc, fd;
    static int nc, is;
    static float aitest;
    static int nczero;
    static float artest;
    static int km1, nvctnu;
    static int nmk, iro;

/*         THIS SUBROUTINE FINDS NVCT EIGENVECTORS FOR THE MATRIX */
/*         AR + I*AI.  THE VECTORS ARE RETURNED IN THE LEFT COLUMNS */
/*         OF AR AND AI.  ZERO PIVOT IS ENCOUNTERED EARLIER THAN */
/*         DEFINED BY NVCT. */
/*         IF THE PARTICULAR PROBLEM BEING SOLVED INVOLVES A DISTINCT */
/*         EIGENVALUE, THE INDEX NVCT WILL BE UNITY.  ZRO INDICATES */
/*         THE PRACTICAL SIZE OF ZERO.  DIFFERENCES TAKEN WHICH ARE */
/*         LESS THAN ZRO TIMES ONE OF THE SUBTRACTED VARIABLES ARE SET */
/*         TO ZERO. */
/*         THE ARRAY, PIVOTS, PRESENTS THE SQUARES OF THE MODULI OF THE */
/*         PIVOTS.  MATRICES AR AND AI ARE NXN, DIMENSIONED TO NRDIM */
/*         AND NIDIM ROWS, RESPECTIVELY, IN THE MAIN PROGRAM.  IF AI IS */
/*         NULL, IREL = 1 TO INDICATE THAT THE SYSTEM IS REAL. */


/*         DECLARE DOUBLE PRECISION FUNCTIONS: */

    /* Parameter adjustments */
    --wi;
    --wr;
    --ncol;
    --pivots;
    ar_dim1 = *nrdim;
    ar_offset = 1 + ar_dim1 * 1;
    ar -= ar_offset;
    ai_dim1 = *nidim;
    ai_offset = 1 + ai_dim1 * 1;
    ai -= ai_offset;

    /* Function Body */
    nvctnu = *nvct;
    i__1 = *n;
    for (i__ = 1; i__ <= i__1; ++i__) {
/* L100: */
 pivots[i__] = 0.;
    }
    i__1 = *n;
    for (ipass = 1; ipass <= i__1; ++ipass) {
 k = *n - ipass + 1;

/*         SCAN THE KXK SUBMATRIX FOR THE PIVOT */
 pvtsz = 0.;
 ncol[ipass] = 0;
 l = 0;
 i__2 = k;
 for (i__ = 1; i__ <= i__2; ++i__) {
     i__3 = k;
     for (j = 1; j <= i__3; ++j) {
  if (*irel == 1) {
      temp = ar[i__ + j * ar_dim1] * ar[i__ + j * ar_dim1];
  }
  if (*irel != 1) {
      temp = ar[i__ + j * ar_dim1] * ar[i__ + j * ar_dim1] + ai[
       i__ + j * ai_dim1] * ai[i__ + j * ai_dim1];
  }
  if (temp < pvtsz) {
      goto L149;
  }
  ncol[ipass] = j;
  pvtsz = temp;
  l = i__;
L149:
  ;
     }
/* L150: */
 }

/*         SAVE THE PIVOT */
 pivots[ipass] = pvtsz;
/* L155: */

/*         CHECK IF UPPER LEFT K BY K MATRIX IS NULL (I.E., IF THE */
/*         PIVOT HAS VANISHED). */
 amax = sqrt(pvtsz);
 if (amax <= (float)1e-6) {
     goto L1001;
 }

 if (k > *nvct) {
     goto L160;
 }

/*         IF HERE, THERE ARE LESS THAN NVCT VECTORS FOR THE REPEATED */
/*         EIGENVALUE.  WE WILL CONTINUE THE PROCESS UNTIL THE PIVOT */
/*         DOES VANISH, OR K = 1. */
/* moved here, below zero pivot test 7/7/ */

/* 
Message goes here 
 dmxout_(&ar[ar_offset], n, n, nrdim, iout, "AR  ", n, &k, &nvctnu, (
  ftnlen)4);
 dmxout_(&ai[ai_offset], n, n, nidim, iout, "AI  ", n, &k, &nvctnu, (
  ftnlen)4);
*/

 if (k == 1) {
     goto L1001;
 }
 --nvctnu;

L160:

/*         EXCHANGE ROWS AND COLUMNS TO MOVE THE L,NCOL(IPASS) ELEMENT */
/*         TO THE PIVOTAL POSITION - POSITION K,K.  EXCHANGE ROWS FIRST. */
 i__2 = k;
 for (j = 1; j <= i__2; ++j) {
     temp = ar[k + j * ar_dim1];
     ar[k + j * ar_dim1] = ar[l + j * ar_dim1];
     ar[l + j * ar_dim1] = temp;
     if (*irel == 1) {
  goto L170;
     }
     temp = ai[k + j * ai_dim1];
     ai[k + j * ai_dim1] = ai[l + j * ai_dim1];
     ai[l + j * ai_dim1] = temp;
L170:
     ;
 }

/*         EXCHANGE COLUMNS. */
 m = ncol[ipass];
 i__2 = *n;
 for (i__ = 1; i__ <= i__2; ++i__) {
     temp = ar[i__ + k * ar_dim1];
     ar[i__ + k * ar_dim1] = ar[i__ + m * ar_dim1];
     ar[i__ + m * ar_dim1] = temp;
     if (*irel == 1) {
  goto L180;
     }
     temp = ai[i__ + k * ai_dim1];
     ai[i__ + k * ai_dim1] = ai[i__ + m * ai_dim1];
     ai[i__ + m * ai_dim1] = temp;
L180:
     ;
 }

/*         DIVIDE THE PIVOTAL ROW BY THE PIVOT.  INSERT THE ROW INTO */
/*         THE WORKING MATRIX. */
 fc = ar[k + k * ar_dim1] / pvtsz;
 fd = ai[k + k * ai_dim1] / pvtsz;
 km1 = k - 1;
 i__2 = km1;
 for (j = 1; j <= i__2; ++j) {
     if (*irel == 1) {
  goto L185;
     }
     temp = ar[k + j * ar_dim1];
     ar[k + j * ar_dim1] = ar[k + j * ar_dim1] * fc + ai[k + j * 
      ai_dim1] * fd;
     ai[k + j * ai_dim1] = ai[k + j * ai_dim1] * fc - temp * fd;
     wr[j] = ar[k + j * ar_dim1];
     wi[j] = ai[k + j * ai_dim1];
     goto L190;
L185:
     ar[k + j * ar_dim1] *= fc;
     wr[j] = ar[k + j * ar_dim1];
L190:
     ;
 }
 ar[k + k * ar_dim1] = 1.;
 ai[k + k * ai_dim1] = 0.;

/*         REDUCE THE KTH COLUMN TO ZERO */
 i__2 = *n;
 for (i__ = 1; i__ <= i__2; ++i__) {
     if (i__ == k) {
  goto L220;
     }
     if (ar[i__ + k * ar_dim1] != 0.) {
  goto L197;
     } else {
  goto L195;
     }
L195:
     if (*irel == 1) {
  goto L220;
     }
     if (ai[i__ + k * ai_dim1] != 0.) {
  goto L197;
     } else {
  goto L220;
     }
L197:
     i__3 = km1;
     for (j = 1; j <= i__3; ++j) {
  tstr = (d__1 = ar[i__ + j * ar_dim1], fabs(d__1));
  if (*irel == 1) {
      goto L207;
  }
  tsti = (d__1 = ai[i__ + j * ai_dim1], fabs(d__1));
  ar[i__ + j * ar_dim1] = ar[i__ + j * ar_dim1] - ar[i__ + k * 
   ar_dim1] * ar[k + j * ar_dim1] + ai[i__ + k * ai_dim1]
    * ai[k + j * ai_dim1];
  ai[i__ + j * ai_dim1] = ai[i__ + j * ai_dim1] - ai[i__ + k * 
   ai_dim1] * ar[k + j * ar_dim1] - ar[i__ + k * ar_dim1]
    * ai[k + j * ai_dim1];
  temp = (d__1 = ar[i__ + j * ar_dim1], fabs(d__1));
  artest = *zro * tstr;
  arwas = ar[i__ + j * ar_dim1];
  if (temp < artest) {
      ar[i__ + j * ar_dim1] = 0.;
  }
  temp = (d__1 = ai[i__ + j * ai_dim1], fabs(d__1));
  aitest = *zro * tsti;
  aiwas = ai[i__ + j * ai_dim1];
  if (temp < aitest) {
      ai[i__ + j * ai_dim1] = 0.;
  }
  goto L209;
L207:
  ar[i__ + j * ar_dim1] -= ar[i__ + k * ar_dim1] * ar[k + j * 
   ar_dim1];
  temp = (d__1 = ar[i__ + j * ar_dim1], fabs(d__1));
  artest = *zro * tstr;
  arwas = ar[i__ + j * ar_dim1];
  if (temp < artest) {
      ar[i__ + j * ar_dim1] = 0.;
  }
L209:

  if (ar[i__ + j * ar_dim1] == 0. && arwas != 0. && *iout > 0) {

/* Message here */
  }
  if (*irel == 1) {
      goto L210;
  }

  if (ai[i__ + j * ai_dim1] == 0. && aiwas != 0. && *iout > 0) {
/* Message here */
  }

L210:
  ;
     }
     ar[i__ + k * ar_dim1] = 0.;
     ai[i__ + k * ai_dim1] = 0.;
L220:
     ;
 }

/* L1000: */
    }

L1001:

/*         K VECTORS CAN BE FORMED.  THEY WILL OCCUPY THE LEFT COLUMNS */
/*         OF MARTICES AR AND AI. */
    i__1 = k;
    for (j = 1; j <= i__1; ++j) {
 i__2 = *n;
 for (i__ = 1; i__ <= i__2; ++i__) {
     if (i__ == j) {
  ar[i__ + j * ar_dim1] = 1.;
     }
     if (i__ <= k) {
  ai[i__ + j * ai_dim1] = 0.;
     }
     if (i__ <= k && i__ != j) {
  ar[i__ + j * ar_dim1] = 0.;
     }
     if (i__ <= k) {
  goto L325;
     }
     ar[i__ + j * ar_dim1] = -ar[i__ + j * ar_dim1];
     ai[i__ + j * ai_dim1] = -ai[i__ + j * ai_dim1];
L325:
     ;
 }
/* L350: */
    }
    if (nvctnu == *nvct) {
 goto L355;
    }

/*         ZERO OUT THE OTHER COLUMNS, MAKING THEM TRIVIAL VECTORS. */
    nczero = *nvct - nvctnu;
    i__1 = nczero;
    for (j = 1; j <= i__1; ++j) {
 nc = nvctnu + j;
 i__2 = *n;
 for (i__ = 1; i__ <= i__2; ++i__) {
     ar[i__ + nc * ar_dim1] = 0.;
/* L352: */
     ai[i__ + nc * ai_dim1] = 0.;
 }
    }
L355:

/*         ARRANGE THE VECTORS IN PROPER ORDER (ACCOUNT FOR COLUMNAL */
/*         PIVOTING). */
    i__2 = k;
    for (j = 1; j <= i__2; ++j) {
 nmk = *n - k;
 i__1 = nmk;
 for (i__ = 1; i__ <= i__1; ++i__) {
     is = nmk - i__ + 1;
     icame = ncol[is];
     iro = k + i__;
     temp = ar[iro + j * ar_dim1];
     ar[iro + j * ar_dim1] = ar[icame + j * ar_dim1];
     ar[icame + j * ar_dim1] = temp;
     if (*irel == 1) {
  goto L360;
     }
     temp = ai[iro + j * ai_dim1];
     ai[iro + j * ai_dim1] = ai[icame + j * ai_dim1];
     ai[icame + j * ai_dim1] = temp;
L360:
     ;
 }
/* L370: */
    }
    return 0;
} /* vectrd_ */
