\ {{{1 GNU General Public License
{
Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} 
}

{ File file.v  December 1999

   Copyright (c) 1999   D. R. Williamson

   The appendix contains some examples.
}
\-----------------------------------------------------------------------

\  The file handle structure:

   "file" list:
      "handle"  \ handle from system
      "type"    \ 0=ascii, 1=binary
      "status"  \ 0=old, 1=new
      "size"    \ file size, bytes (0 if new)
      "recsize" \ byte-size of each random access record (set by frec)
      "rechead" \ byte-size of header in random access (set by fhead)
      "pos"     \ current file pointer position (fseek, fend, fput, ...)
      "mode"    \ 0=read, 1=read/write
      "mag"     \ unique id (magic) number is "file" str2num 
   end struct
\  Programming note: The order of these file struct items corresponds 
\  to the enumeration called _file in source prototype file inpo.h.

\-----------------------------------------------------------------------
{
   The words in this file are listed below.

   syspath "file.v" cat asciiload this " inline:" grepr reach dot

   inline: append4 (hA hFile --- ) \ append matrix A to a put4 file
   inline: archive (hFiles qDir --- hT) \ put files into an archive file
   inline: asciiread1 (qFile n --- hA) \ n numbers per row
   inline: asciiread4 (qFile n --- qOP4) \ n numbers per row to OP4 file
   inline: bcreate (qS n --- ) \ turn source file S into binary file
   inline: bsource (qS n --- ) \ load binary source file
   inline: bswap4 (qS hP --- qS1) \ swap 4-byte groups using pattern P
   inline: closeif (qS --- ) \ close file if S is name of handle open
   inline: crush (qFile --- ) \ create or crush existing File to 1 byte
   inline: dirfiles (qDir --- hNames hTimes) \ files in directory
   inline: fallow1 (hFiles --- hF) \ fallow flags for list of Files
   inline: fbook (hA qWord --- ) \ store A on file /tmp/Word
   inline: fbook2 (hA qWord --- ) \ store A as 2-byte uints on /tmp/Word
   inline: fbook4 (hA qWord --- ) \ store A as 4-byte uints on /tmp/Word
   inline: fcopy (qFile1 qFile2 --- ) \ copy File1 to File2
   inline: fend (hFile --- ) \ seek end of file
   inline: fendget (hFile --- etype or 0) \ endian type flag from file
   inline: fendlen ( --- n) \ length of endian tag appended to file
   inline: fendtag (etype hFile --- ) \ tag a file with endian type
   inline: fhead (rechead hFile --- ) \ set record head size in struct
   inline: fileblock (qFile --- f) \ block until File begins to grow
   inline: filejunk (qFile --- ) \ turn File into junk
   inline: filesearch (qFile --- no, or qFile1 yes) \ searches user PATH
   inline: file_tdiff (qAdir hAfiles qBdir hBfiles --- hDiff) time diffs
   inline: finitr (nrecs hFile --- ) \ initialize nrecs of random access
   inline: fload (hFile --- hA) \ load file into array A
   inline: fmap (hFile qS --- hMap) \ maps all strings S in File
   inline: fmapl (hFile hMapNL qS --- hMap) \ map to all lines with S
   inline: fmapleft (hFile hMapNL qS --- hMap) \ lines with S on left
   inline: fmapNL (hFile --- hM) \ maps all the new line chars in File
   inline: fname (qDir qFile --- f) \ f true if File is in Dir
   inline: forn (qS --- qS) \ file flag old or new depending upon S
   inline: fprops (hFile --- ) \ properties in file handle structure
   inline: frec (recsize hFile --- ) \ set record size in file struct
   inline: fstr (hFile hMapNL qS --- hT) \ lines with string S anywhere
   inline: fstr1 (hFile hMapNL qS --- hT) \ lines with S in 1st column
   inline: ftemp (qFil --- qFile) \ a unique file name from the system
   inline: ftempsys ( --- qFile) \ a unique file name from the system
   inline: ftouch (qFile --- ) \ touch File
   inline: fwrite (hA hFile --- toptr tolen) \ A to File, return where
   inline: matF ( --- hFile) \ handle to the file opened by word matfile
   inline: matfile (qP qS --- ) \ open the file handle called matF
   inline: matfile?  (hF --- f) \ true if F is handle to MAT-File
   inline: mget (hF qS --- hA) \ load matrix S from MAT-File F
   inline: mheader (hF --- ) \ xray of MAT-File header for F
   inline: mtoc (hF --- ) \ display the contents of MAT-File F
   inline: notafile (hFile --- ) \ display not-a-file-handle message
   inline: op4 ( --- hFile) \ handle to the file opened by word op4file
   inline: op4file (qP qS --- ) \ open the file handle called op4
   inline: put4q (hA qS --- ) \ put matrix A on binary file named S
   inline: rewind (hFile --- ) \ set File position to zero
   inline: seefiles ( --- hT) \ a table of all files the program can see
   inline: stkdelete ( --- ) \ delete stksave files
   inline: stkrestore (qFile --- ...) \ restore saved stack from File
   inline: stksave (... qFile --- ) \ save stack contents on File
   inline: voc (qFile --- hT) \ list of defined words in File

   Words for reading Fortran files:
   inline: fortfile (hFile --- ) \ set things up for a Fortran File
   inline: fortbackspace ( --- ) \ backspace one Fortran record
   inline: fortclose ( --- ) \ close file previously opened by fortfile
   inline: fortendian ( --- N) \ endian of Fortran file
   inline: fortendian_set (N --- ) \ set endian of Fortran file
   inline: fortrec ( --- hT) \ fetching the current record
   inline: fortrec? ( --- n) \ where the file record pointer is pointing
   inline: fortrewind ( --- ) \ reset record index to the beginning
   inline: fortseek (n --- ) \ setting Fortran record to 1-based index n
   inline: fortskip ( --- ) \ skip the current record
   inline: fortstep (n --- ) \ step record index by n records

}
\-----------------------------------------------------------------------

   private

   inline: append4 (hA hFile --- ) \ append matrix A to a put4 file
      this fend put4 ;

   inline: archive (hFiles qDir --- hT) \ put files into an archive file
\     Incoming volume, Files, lists some file names in Dir.  Make a tar
\     file archive of Files and return it in volume T on the stack.

      [ "HOME" env runid catpath "archive.tmp" + "TMP" book ]

      swap (hFiles) -path (hFiles) swap
      (hFiles qDir) TMP ctar \ write archive file

      TMP file?
      IF TMP old binary "BIN" file
         BIN INF fget (hT) \ archive file in VOL T on stack
         BIN fclose

         TMP delete \ delete archive file
      ELSE " archive: failed " date + . nl
         VOL tpurged (hT)
      THEN
      (hT) "_archive" naming
   end

   inline: asciiread1 (qFile n --- hA) \ n numbers per row
{     Reads n numbers per row from text File and builds numerical
      matrix A.

      Skips rows that contain text other than numbers, and skips rows
      that do not contain exactly n numbers.

      This word is more memory efficient than asciiread because the
      entire text File is not read into the program.

      This phrase runs the test in the Appendix:

         "file.v" "ASCIIREAD TEST" msource halt
}
      [ no "FILE" book, depthSTK 2 / "MAX" book, 100 "SHOW" book ]

      FILE filetrue IF FILE close THEN

      (n) "n" book
      (qFile) old ascii "FILE" open

      " asciiread1: mapping..." .
      FILE fmapNL "MAP" book \ MAP where all the NLs (new lines) are

    \ The first entry in MAP is -1 (see man fmapNL), so the number of
    \ rows in FILE is one less than the rows in MAP:
      "done;" . MAP rows nit .i " rows mapped" . nl

      SHOW 0> MAP rows SHOW > and
      IF " asciiread1: processing text rows:" . nl one .i 
         one "shown" book
      ELSE no "shown" book
      THEN

      zero n null (hA0) \ purged row on stk
      one "count" book \ one for A0 on stk

      MAP rows 2nd
      DO FILE MAP I pry MAP I 1- pry less fget \ get next line from file
         asciify 1st quote numbers any?        \ turn text into numbers
         IF these numbad totals ontop 0=       \ no bad ones?
            those rows n = and                 \ and have exactly n?
            IF bend (hA)                       \ put another row on stk
               one count bump                  \ bump the count

               count MAX =                     \ have MAX rows on stk?
               IF \ Pile the rows when there are MAX on stk:

                  (hA0 hA1 hA2 ... hA_MAX) MAX pilen (hA0)
                  one "count" book \ one for A0 on stk
               THEN

               I nit SHOW mod 0=
               IF I nit .i one shown bump
                  shown 8 = IF no "shown" book nl THEN
               THEN

            ELSE drop \ bad numbers, or not n of them; drop from stk
            THEN
         THEN
      LOOP
      (hA0 hA1 hA2 ... ) count pilen (hA) \ pile what remains on stk
      "_asciiread1" naming                \ give A a stk name
      FILE close                          \ close the file
   end

   inline: asciiread4 (qFile n --- qOP4) \ n numbers per row to OP4 file
{     Reads n columns per matrix row given on each line of text File,
      and writes each matrix column to a separate data block on a binary
      Nastran-style OP4 file.

      This word can be used for huge text data files, with millions of
      rows.

      The final matrix is on file OP4, with each column in a separate
      data block on the file.  Each column can be read with word get4.

      Skips rows that contain text other than numbers, and skips rows
      that do not contain exactly n numbers.

      This word is more memory efficient than asciiread and asciiread1
      because the text File is not read into the program and because
      only part of the matrix is in memory at any time.

      This phrase runs the test in the Appendix:

         "file.v" "ASCIIREAD TEST" msource halt
}
      [ depthSTK 2 / "MAX" book
        no "FILE" book
        no "FTMP" book
        no "FOP4" book
      ]
      FILE filetrue IF FILE close THEN
      FTMP filetrue IF FTMP close THEN
      FOP4 filetrue IF FOP4 close THEN

      (n) "n" book
      (qFile) this old ascii "FILE" open

      (qFile) -ext ".tmp" cat "Ftemp" book
      Ftemp deleteif
      Ftemp new binary "FTMP" open

      Ftemp -ext ".op4" cat "Fname" book
      Fname deleteif
      Fname new binary "FOP4" open

      " asciiread4: mapping..." .
      FILE fmapNL "MAP" book \ MAP where all the NLs (new lines) are

    \ The first entry in MAP is -1 (see man fmapNL), so the number of
    \ rows in FILE is one less than the rows in MAP:
      "done;" . MAP rows 1- .i " rows mapped" . nl

    \ How often to show progress:
      MAP rows 0.01 * log10 rounded 10^ "SHOW" book
      SHOW 100 >
      IF " asciiread4: processing text rows:" . nl one .i 
         one "shown" book
      ELSE no "SHOW" book \ no progress displayed
      THEN

      zero n null (hA0) \ purged row on stk
      one "count" book \ one for A0 on stk
      one "num" book

      MAP rows 2nd
      DO FILE MAP I pry MAP I 1- pry less fget \ get next line from file
         asciify 1st quote numbers any?        \ turn text into numbers
         IF these numbad totals ontop 0=       \ no bad ones?
            those rows n = and                 \ and have exactly n?
            IF bend (hA)                       \ put another row on stk
               one count bump                  \ bump the count

               count MAX =                     \ have MAX rows on stk?
               IF
                \ Pile the rows when there are MAX on stk:

                  (hA0 hA1 hA2 ... hA_MAX) MAX pilen (hA0)

                \ Write this portion to FTMP:
                  (hA0) "seg" num suffix naming FTMP put4
                  one num bump 

                  zero n null (hA0)   \ purged row on stk
                  one "count" book \ one for A0 on stk
               THEN

               I nit SHOW mod 0=
               IF I nit .i one shown bump
                \ Put nl so if this output goes to a file, vi limit
                \ on max line length won't be hit (about 2048 chars):
                  shown 8 = IF no "shown" book nl THEN
               THEN

            ELSE drop \ bad numbers, or not n of them; drop from stk
            THEN
         THEN
      LOOP SHOW 0> IF nl THEN

      (hA0 hA1 hA2 ... ) count pilen (hA) \ pile what remains on stk

    \ Write this final portion to FTMP:
      (hA) any? IF (hA) "seg" num suffix naming FTMP put4 THEN

    \ Put the pieces from FTMP together into FOP4, one column per data
    \ block:
      FTMP "toc4" >stk 1st word
      IF (hNames) push " asciiread4: writing file " . Fname . nl
         n 1st
         DO peek rows 1st

            DO FTMP this rewind peek I quote (name) J (col) get4c LOOP

            peek rows pilen "ch" I suffix naming FOP4 put4
            "   column" . I .i " to datablock " . "ch" I suffix . nl

         LOOP
         pull (hNames) drop

      ELSE " asciiread4: empty results file" ersys
      THEN

      FILE close, FTMP close, FOP4 close \ close the file handles
      Ftemp deleteif                     \ delete FTMP

      Fname \ return name of binary OP4 file
   end

   inline: bcreate (qS n --- ) \ turn source file S into binary file
\     Replaces extension of S with .bin; key number n is used to source
\     the file later using bsource.
      [ scalar "bc" book ] \ inline needs local ref for filetrue below
      true one NUM stkok and, two STR stkok and
      not IF "bcreate" stknot return THEN

      swap filefound not IF drop " file not found" . nl return THEN

      dup asciiload swap dup -ext ".bin" cat 2dup strmatch 0<>
      IF lop this deleteif, bc filetrue IF bc fclose THEN
         (qS) "bc" new binary file (hT)
         swap (hT key) bmake (hT1) bc fput, bc fclose
      ELSE drop lop swap " source file cannot have .bin extension" .
      THEN
   end

   inline: bsource (qS n --- ) \ load binary source file
\     Loads binary source file named S with key=n 
      [ scalar "bs" book ] \ inline needs local ref for filetrue below
      true one NUM stkok and, two STR stkok and
      not IF "bsource" stknot return THEN

      swap filefound not IF drop " file not found" . nl return THEN

      bs filetrue IF bs fclose THEN
      (qS) "bs" old binary file (hT)
      bs this file.size pry (hFile bytes) fget (hT), bs fclose
      swap (hT key) brun
   end

   inline: bswap4 (qS hP --- qS1) \ swap 4-byte groups using pattern P
\     Swap bytes in string of 4-byte patterns according to 1-based
\     sequence of the four indices in P.
\     Note: the four indices in P are 1-based values in range 1 to 4.

      ndx "P" book \ values in P into offsets for current index base

      four those chars four slash uniform push

      peek P 1st pry +d
      peek P 2nd pry +d
      peek P 3rd pry +d
      pull P 4th pry +d
      four parkn bend chain (hChars)

      (qS hChars) catch
   end

   inline: crush (qFile --- ) \ crush File to 1 byte
\     Equivalent to Unix command: echo>File
      "" swap save ;

   inline: closeif (qS --- ) \ close file if S is name of handle open
{     Will not work inside a word where the file handle is in the local
      library of the word (see op4file and put4q below for use of 
      filetrue and fclose within a word).     

      Example usage: starting fresh with new file and new handle:
         "t1.dat" this deleteif, "hT" this closeif, new ascii file

      Warning: if S is mistakenly the name of an inlinex, this word
      will not behave correctly.

}     no STR stkok not IF "closeif" stknot return THEN 

      (qS) this quoted " exists?" cat main \ looking for item in main
      IF (qS) main (hF), this filetrue     \ is it a file handle?
         IF (hF) fclose ELSE drop THEN 
      ELSE drop 
      THEN 
   end

   inline: dirfiles (qDir --- hNames hTimes) \ files in directory
\     List of file Names and column matrix of corresponding Times for 
\     files in directory Dir.
      (qDir) "" catpath "Dir" book
      Dir dirnames (hNames)
      (hNames) dup 1st "." cite rake lop   \ remove . files from list
      (hNames) dup Dir nose dir? rake drop \ remove directory names
      (hNames) dup Dir nose filetime (hTimes)
   end

   inline: fallow1 (hFiles --- hF) \ fallow flags for list of Files
      hand push
      list: peek rows 1st DO peek I quote fallow LOOP end
      pull drop
   end

   inline: fbook (hA qWord --- ) \ store A on file /tmp/Word
{     This word makes a word called Word, and puts A on a file called
      /tmp/Word.  Later, saying Word will run word Word that will get A
      from its file and place it on the stack.

      Incoming A is a matrix or volume, and Word will refer to it.

      Stack item A is stored on a file, not in the catalog.  But A
      appears on the stack by saying Word, so the fact that A is on a
      file is transparent--behavior is as if A had been booked into
      the catalog.

      If Word is in another word's library, its incoming string must
      include a library tag that can be obtained using localref().

      With no library tag, Word will go into the main library.

      If Word is in a local library, then it is fired using localrun(),
      just as when A is in a local library.

      Examples below show cases of main library and a word's library.

      Examples: In these examples, a matrix appears to be in the main 
      library or in the library of a word.  But it is actually stored
      on a file.

         File booking A into the main library:

            [tops@plunger] ready > 10 10 random "A" fbook

            [tops@plunger] ready > A \ saying A brings it to the stack 
             stack elements:
                   0 matrix: A  10 by 10
             [1] ok!
            [tops@plunger] ready >


         File booking M1 into the library of MATLIB, using localref()
         to apply a library tag:

            [tops@plunger] ready > "MATLIB" library
              word MATLIB,0:CODE__ into catalog

            [tops@plunger] ready > 3 2 random MATLIB "M1" localref fbook

            [tops@plunger] ready > MATLIB "M1" localrun .m
             Row 1:   0.2881   0.2295
             Row 2:    0.935   0.5886
             Row 3:   0.3633   0.5852
            [tops@plunger] ready >

         Here is this example's matrix on a file in /tmp.  This job's
         runid (20309) is in the name, so it will be deleted when the
         program exits:

         [dale@plunger] /tmp > llr *fbook*
         -rw-r--r-- 1 dale comm  91 Mar 17 22:52 T20309_fbook_MATLIB.M1


         File booking M1 into the library of a word (infix notation):

            >> library("MATLIB");
             word MATLIB,0:CODE__ into catalog

            >> fbook(random(3, 2), localref(MATLIB, "M1"));

            >> X = MATLIB.M1;

            >> .m(X);
             Row 1:   0.5098   0.2681
             Row 2:   0.3328   0.8089
             Row 3:  0.07094  0.06571
            >>

      Warning: Insufficient checking.  If there is a catalog name con-
      flict and Word cannot be created, stack item A will still be
      placed on a file and the file name will appear here in Fnames as
      if Word had been created.
}
      [ {" This is the text for Word to be created:
           ( --- hA) \ this is the stack diagram for Word 

       \ This will fail if there is a string !@#$% in the file name.

         [ purged "BIN_!@#$%" book ]
         BIN_!@#$% filetrue IF BIN_!@#$% fclose THEN

         "FILE" file?                        
         IF "FILE" old binary "BIN_!@#$%" file \ open file
            BIN_!@#$% "!@#$%" get (hA)         \ get matrix or volume
            BIN_!@#$% fclose                   \ close file
         ELSE " !@#$%: file " "FILE" + " not found" + . nl purged (hA)
         THEN (hA)
        "} "TEXT" book

        VOL tpurged "Fnames" book
        no "BIN" book

      \ The following defaults can be overridden with word bank:
        "_bin" "tmppath" yank (qS)
        (qS) "path" book \ path to files
        yes "cleanup" book \ word clean will delete fbooked files
      ]
      depth two <, one STR stkok not or
      IF "fbook" stknot return THEN

      (qWord) strchop "Word" book
      Word dup tagsep grepr rows 0>
      IF (qWord) tagsep chblank words
         dup 2nd quote strchop "B" book
         1st quote strchop "A" book  \ A in lib of B
      ELSE "" "B" book "A" book      \ A in main lib
      THEN

      path runid "T" "S" strp catpath "_fbook_" + (qS)
      B chars any
      IF B "." + A + ELSE A THEN (qS qS1) + (qS)
      (qS) "FNAME" book \ file name, like /tmp/T15975_fbook_B.A

    \ Adding FNAME to list of Fnames kept here in the local library:
      Fnames this FNAME grepe reach rows any not
      IF Fnames FNAME pile onto Fnames THEN

      CATMSG (f) no catmsg

      TEXT 
      "FILE" FNAME strp 
      "!@#$%" A strp (hT)
      (hT) Word macro \ make word A in main or lib B, depending on Word

      (f) catmsg

      FNAME deleteif \ delete Word's file

    \ Now put A on new file:
      FNAME new binary "BIN" file
      (hA) A naming BIN put, BIN fclose
   end

   inline: fbook2 (hA qWord --- ) \ store A as 2-byte uints on /tmp/Word
{     Use only when the terms of A are unsigned integers.

      This word makes a word called Word, that converts the terms of 
      matrix A into 2-byte ints and places them on a file called 
      /tmp/Word.

      Later, saying Word will run word Word that will get A from its 
      file, convert it back to an 8-byte floating point matrix and 
      place it on the stack.

      Incoming A is a matrix (type MAT), and Word will refer to it.

      Stack item A is stored on a file, not in the catalog.  But A
      appears on the stack by saying Word, so the fact that A is on a
      file is transparent--behavior is as if A had been booked into
      the catalog.

      Warning: Insufficient checking.  If there is a catalog name con-
      flict and Word cannot be created, stack item A will still be
      placed on a file and the file name will appear here in Fnames as
      if Word had been created.
}
      [ {" This is the text for Word to be created:
           ( --- hA) \ this is the stack diagram for Word 

       \ This will fail if there is a string !@#$% in the file name.

         [ purged "BIN_!@#$%" book ]
         BIN_!@#$% filetrue IF BIN_!@#$% fclose THEN

         "FILE" file?                        
         IF "FILE" old binary "BIN_!@#$%" file \ open file
            BIN_!@#$% "!@#$%" get (hT)         \ get volume of 2-bytes
            BIN_!@#$% fclose                   \ close file
            (hT) endian uimport2 (hA)          \ reclaim 8-byte A
         ELSE " !@#$%: file " "FILE" + " not found" + . nl purged (hA)
         THEN (hA)
        "} "TEXT" book

        VOL tpurged "Fnames" book
        no "BIN" book

      \ The following defaults can be overridden with word bank:
        "_bin" "tmppath" yank (qS)
        (qS) "path" book \ path to files
        yes "cleanup" book \ word clean will delete fbooked files
      ]
      depth two <, one STR stkok not or
      IF "fbook2" stknot return THEN

      (qWord) strchop "Word" book
      Word dup tagsep grepr rows 0>
      IF (qWord) tagsep chblank words
         dup 2nd quote strchop "B" book
         1st quote strchop "A" book  \ A in lib of B
      ELSE "" "B" book "A" book      \ A in main lib
      THEN

      path runid "T" "S" strp catpath "_fbook2_" + (qS)
      B chars any
      IF B "." + A + ELSE A THEN (qS qS1) + (qS)
      (qS) "FNAME" book \ file name, like /tmp/T15975_fbook2_B.A

    \ Adding FNAME to list of Fnames kept here in the local library:
      Fnames this FNAME grepe reach rows any not
      IF Fnames FNAME pile onto Fnames THEN

      CATMSG (f) no catmsg

      TEXT 
      "FILE" FNAME strp 
      "!@#$%" A strp (hT)
    \ Make word A in main or lib B, depending on Word--see examples in
    \ word fbook:
      (hT) Word macro 

      (f) catmsg

      FNAME deleteif \ delete Word's file

    \ Now convert 8-byte MAT to 2-byte int VOL and put VOL on new file:
      FNAME new binary "BIN" file
      (hA) endian export2 (hA)
      (hA) A naming BIN put, BIN fclose
   end

   inline: fbook4 (hA qWord --- ) \ store A as 4-byte uints on /tmp/Word
{     Use only when the terms of A are unsigned integers.

      This word makes a word called Word, that converts the terms of 
      matrix A into 4-byte ints and places them on a file called 
      /tmp/Word.

      Later, saying Word will run word Word that will get A from its 
      file, convert it back to an 8-byte floating point matrix and 
      place it on the stack.

      Incoming A is a matrix (type MAT), and Word will refer to it.

      Stack item A is stored on a file, not in the catalog.  But A
      appears on the stack by saying Word, so the fact that A is on a
      file is transparent--behavior is as if A had been booked into
      the catalog.

      Warning: Insufficient checking.  If there is a catalog name con-
      flict and Word cannot be created, stack item A will still be
      placed on a file and the file name will appear here in Fnames as
      if Word had been created.
}
      [ {" This is the text for Word to be created:
           ( --- hA) \ this is the stack diagram for Word 

       \ This will fail if there is a string !@#$% in the file name.

         [ purged "BIN_!@#$%" book ]
         BIN_!@#$% filetrue IF BIN_!@#$% fclose THEN

         "FILE" file?                        
         IF "FILE" old binary "BIN_!@#$%" file \ open file
            BIN_!@#$% "!@#$%" get (hT)         \ get volume of 4-bytes
            BIN_!@#$% fclose                   \ close file
            (hT) endian uimport4 (hA)          \ reclaim 8-byte A
         ELSE " !@#$%: file " "FILE" + " not found" + . nl purged (hA)
         THEN (hA)
        "} "TEXT" book

        VOL tpurged "Fnames" book
        no "BIN" book

      \ The following defaults can be overridden with word bank:
        "_bin" "tmppath" yank (qS)
        (qS) "path" book \ path to files
        yes "cleanup" book \ word clean will delete fbooked files
      ]
      depth two <, one STR stkok not or
      IF "fbook4" stknot return THEN

      (qWord) strchop "Word" book
      Word dup tagsep grepr rows 0>
      IF (qWord) tagsep chblank words
         dup 2nd quote strchop "B" book
         1st quote strchop "A" book  \ A in lib of B
      ELSE "" "B" book "A" book      \ A in main lib
      THEN

      path runid "T" "S" strp catpath "_fbook4_" + (qS)
      B chars any
      IF B "." + A + ELSE A THEN (qS qS1) + (qS)
      (qS) "FNAME" book \ file name, like /tmp/T15975_fbook4_B.A

    \ Adding FNAME to list of Fnames kept here in the local library:
      Fnames this FNAME grepe reach rows any not
      IF Fnames FNAME pile onto Fnames THEN

      CATMSG (f) no catmsg

      TEXT 
      "FILE" FNAME strp 
      "!@#$%" A strp (hT)
    \ Make word A in main or lib B, depending on Word--see examples in
    \ word fbook:
      (hT) Word macro 

      (f) catmsg

      FNAME deleteif \ delete Word's file

    \ Now convert 8-byte MAT to 4-byte int VOL and put VOL on new file:
      FNAME new binary "BIN" file
      (hA) endian export4 (hA)
      (hA) A naming BIN put, BIN fclose
   end

   inline: fcopy (qFile1 qFile2 --- ) \ copy File1 to File2
      [ no is F1, no is F2 ]

      true one STR stkok and, two STR stkok and not
      IF "fcopy" stknot return THEN

      F1 filetrue IF F1 fclose THEN
      F2 filetrue IF F2 fclose THEN

      "Fname2" book
      "Fname1" book

      Fname1 file? not
      IF " fcopy: file not found: " Fname1 cat ersys return THEN

      Fname2 this crush forn ascii "F2" file

      F2 rows any not
      IF " fcopy: invalid handle" . nl
         " fcopy: verify permissions for " Fname2 cat ersys return
      THEN

      Fname1 old ascii "F1" file

      F1 rows any not
      IF " fcopy: invalid handle" . nl
         " fcopy: verify permissions for " Fname1 cat ersys return
      THEN

      F1 INF fget (hT) F2 fput
      F1 fclose F2 fclose
   end

   inline: fend (hFile --- ) \ seek end of file
      this filetrue IF this file.size pry fseek return THEN notafile ;

   inline: fendget (hFile --- etype or 0) \ endian type flag from file
\     Tag was placed previously by word fendtag; 0 is returned if no
\     endian tag found.
      "File" book, File fpos push
      File its file.size pry fendlen less fseek 
      File fendlen fget 1st quote, File pull fseek
      this "L" strmatch 0= IF drop LITTLE_ENDIAN return THEN
      this "B" strmatch 0= IF drop BIG_ENDIAN return THEN
      this "P" strmatch 0= IF drop PDP_ENDIAN return THEN
      " fendget: endian type not found" . nl drop no
   end

   inline: fendlen ( --- n) \ length of endian tag appended to file
      one (byte) ;

   inline: fendtag (etype hFile --- ) \ tag a file with endian type
\     Use word fendget to retrieve the endian type flag written here;
\     the tag must be the last bytes of the file.
      [ list: LITTLE_ENDIAN PDP_ENDIAN BIG_ENDIAN ; yes sort
        "etype" book, "LPB" makes tag 
      ] etype rot bsearch
      IF tag swap character
      ELSE " fendtag: invalid endian type" . nl return
      THEN swap dup fend fput ;

   inline: fhead (rechead hFile --- ) \ set record head size in struct
      this filetrue IF file.rechead poke return THEN notafile ;

   inline: fileblock (qFile --- f) \ block until File begins to grow
\     Returned flag f is true if File is growing, false if timed out
\     after SEC wait.
      [ no "HAN" book
        30 "SEC" book
       {"
         File old binary "HAN" open HAN file.size pry SIZE > (f)
         HAN fclose
       "} "GROWING" macro
      ]
      (qFile) "File" book
      File file? not
      IF " fileblock: file " File + " not found" + ersys return THEN

      HAN filetrue IF HAN fclose THEN

      File old binary "HAN" open
      HAN rows 0=
      IF " fileblock: error opening " File + ersys return THEN

      HAN file.size pry "SIZE" book HAN fclose

      SEC "fileblock" "GROWING" localref WAIT_INIT
      10 (Hz) "WAITING" RATE

      WAIT_BEGIN GROWING (f)
   end

   inline: filejunk (qFile --- ) \ turn File into junk
\     Replace each File byte with a random byte.
      [ no "HAN" book ]
      "File" book
      HAN filetrue IF HAN fclose THEN

      File old binary "HAN" file HAN rows 0=
      IF " filejunk: error opening " File + ersys HALT THEN

      1 HAN file.size pry trandom HAN fput
      HAN fclose
   end

   inline: filesearch (qFile --- no, or qFile1 yes) \ searches user PATH
\     Searches in local, along PATH and at HOME, taking the first one
\     it finds.
      [  "" "more_places" book \ bank here more places to look
        {" 
           pwd
           "PATH" env ":" chblank words pile \ PATH
           "HOME" env pile 
           more_places words pile
           asciify noblanklines noq_alike 
        "} "places" inlinex

      ] 
      (qFile) push 

      "" places (qFile1 hT)

      these rows 1st
      DO this I quote (qPath) peek (qFile) catpath this file?
         IF swap rot drop (qFile1 hT) EXIT ELSE drop THEN (qFile1 hT)
      LOOP 
      pull (hT qfile) 2drop (qFile1) any?
   end

   inline: file_tdiff (qAdir hAfiles qBdir hBfiles -- hDiff) \time diffs
\     Any files of list Bfiles in Bdir that have different times or are
\     missing from list of Afiles in Adir, are listed in Diff.

      "Bfiles" book "Bdir" book
      "Afiles" book "Adir" book
      no "count" book

      Afiles Adir nose this filetime ctime (hKeys hVals)
      1000 "Ahash" hash_make

      Bfiles rows 1st
      DO Ahash Adir Bfiles I reach cat hash_lookup any?
         IF drop Bdir Bfiles I reach cat filetime ctime alike not
            IF Bfiles I reach one count bump THEN
         ELSE Bfiles I reach one count bump
         THEN
      LOOP count any?
      IF (count) pilen ELSE "" THEN
      Ahash hash_close
   end

   inline: finitr (nrecs hFile --- ) \ initialize nrecs of random access
      this filetrue not IF this notafile return THEN
      this file.status pry one <>
      IF drop " only for new file" ersys return THEN
      this file.recsize pry any?
      IF (nrecs hFile recsize) rot star  
         that file.rechead pry plus nit \ byte offset to end of file
         (hFile bytes) over swap fseek
         nullbyte that (0 hFile) fput rewind 
      ELSE " record size not specified; use frec first" . nl
      THEN
   end

   inline: fload (hFile --- hA) \ load file into array A
{     A is a volume if file is ascii, and a column matrix of 8-byte 
      numbers if file is binary.  If binary, assumes 8-byte floating 
      point numbers written by a machine of matching endian.
}     this filetrue 
      IF its zero fseek \ rewind
         again this file.size pry \ how many bytes?
         (hFile bytes) fget (hT) \ get 'em all
         swap (hFile) file.type pry 0= (ascii?) 
         IF (hT) textget
         ELSE (hT) endian import8
         THEN (hA) return
      THEN (hFile) notafile 
   end

   inline: fmap (hFile qS --- hMap) \ maps all strings S in File
\     Returned Map contains offsets to all strings S in File.
\     On return, File pointer is unchanged.

\     Uses a buffer to read just part of the file at a time, with 
\     negative stepping to catch desired strings that would be split 
\     because they straddle successive loadings of the buffer.

      [ 640000 is Bufsize 
        1000 is maxPile
        "number of items will exceed stack depth" is msg

        0 is bufsize, 0 is filesize \ for local inline made next:
        "bufsize, filesize File fpos less, min" "buflen" inlinex
      ]
      that filetrue not IF that notafile return THEN
      no STR stkok not IF "fmap" stknot return THEN

      depth maxPile plus depthSTK >
      IF " fmap: " msg cat ersys 2drop purged return THEN

      "S" book, "File" book

      S strlen (Slen) 
      Bufsize, that max, "bufsize" book
      (Slen) 1- negate, "-step" book

      File its fpos push, rewind
      File file.size pry "filesize" book
      no "Pile" book

      BEGIN
         File fpos (fp) 
         File buflen fget (hT)
         S smap rose (fp) +d (hMap)
         one Pile bump
         File fpos filesize < 
         IF File -step fstep
            Pile maxPile =
            IF Pile pilen, one "Pile" book THEN
            no
         ELSE yes
         THEN (f)
      UNTIL

      Pile pilen (hMap)
      File pull fseek
   end

   inline: fmapl (hFile hMapNL qS --- hMap) \ map to all lines with S
\     Returned two-column Map contains offsets in column 1 and lengths
\     in column 2 for all lines from File that contain string S.

\     On return, File pointer is unchanged.

      other filetrue, one STR stkok and, two MAT stkok and not
      IF 2drop drop "fmapl" stknot purged return THEN

      "S" book, bob "File" book File fpos push, (hMapNL) push

      File S fmap (hSmap) any?
      IF peek (hMapNL) rows any not
         IF pull drop File fmapNL push THEN (hSmap)

         peek again park swap (hXY hSmap) 
         look nodupes one +d (hPos)
         peek this -1 lag park over (hXY hPos) 
         look (hPos1) those less (hPos hLen) park (hMap)

      ELSE purged
      THEN pull drop, File pull fseek
   end

   inline: fmapleft (hFile hMapNL qS --- hMap) \ lines with S on left
\     Returned two-column Map contains offsets in column 1 and lengths
\     in column 2 for all lines from File that contain string S on the
\     left, not necessarily in the first column.

\     Incoming MapNL holds offsets to each new line character in File.
\     If incoming new line map MapNL is purged, one will be created
\     with word fmapNL.

\     On return, File pointer is unchanged.

      [ "(qS) this strlen Slen < IF no return THEN " 
        "1st Slen items catch S strmatch 0=" cat "match?" inlinex
      ]
      other filetrue not IF other notafile return THEN

      this strchop "S" book, other "File" book File fpos push
      (hFile hMapNL qS) fmapl (hMap) any? 
      IF (hMap) false those rows one fill push
         S strlen "Slen" book, these rows 1st 
         DO File that I 1st fetch fseek
            File that I 2nd fetch fget
            1st quote strchop match?
            IF true peek I poke THEN
         LOOP
         pull (hMap hRake) rake lop (hMap)
      ELSE purged (hMap)
      THEN File pull fseek
   end

   inline: fmapNL (hFile --- hM) \ maps all the new line chars in File
\     Returned M contains offsets to all NL characters in File.
\     On return, File pointer is unchanged.
\     First entry in M is -1, implying a NL before the first byte of
\     the file.
      NLch fmap (hM) -1 swap pile (hM) ; 

   inline: fname (qDir qFile --- f) \ f true if File is in Dir
    \ Sat Feb 20 09:30:10 PST 2010
      true 1 STR stkok and, 2 STR stkok and not
      IF "fname" stknot ELSE swap dirnames swap grepe rows 0> THEN  
   end
   
   inline: forn (qS --- qS) \ file flag old or new depending upon S
      no STR stkok not IF "forn" stknot return THEN
      strchop this file? IF old ELSE new THEN ;
   
   inline: fprops (hFile --- ) \ properties in file handle structure
      [ "ascii" "binary" pile onto type
        "old" "new" pile onto stat
        "read-only" "read/write" pile onto mode

        "(n) .u ' bytes' ." ".bytes" inlinex

        "(n) this 0> IF .bytes ELSE 'none' . drop THEN" ".rec" inlinex

        "(hFile n) that file.recsize pry 0> "
        "IF (n) .u ' bytes' . ELSE (n) .rec THEN" cat ".hed" inlinex

        "four spaces ." "dent" inlinex
      ]
      no MAT stkok not IF "fprops" stknot return THEN
      this filetrue not IF notafile return THEN

      " Properties of file handle " dot, this named notag dot nl

    \ Getting elements from file handle structure defined at the top
    \ of this file:
      this file.handle      pry "Handle: " dent . longhex . nl

      type that file.type   pry xbase + quote "Type:   " dent . . nl
      stat that file.status pry xbase + quote "Status: " dent . . nl
      mode that file.mode   pry xbase + quote "Mode:   " dent . . nl
      this file.size        pry "Size:   "            dent . .bytes nl
      this file.rechead pry "Random access head: " dent . .hed nl
      this file.recsize pry "Random access size: " dent . .rec nl
               file.pos pry "Pointer is at byte: " dent . .u nl
   end

   inline: frec (recsize hFile --- ) \ set record size in file struct
      this filetrue IF file.recsize poke return THEN notafile ;

   inline: fstr (hFile hMapNL qS --- hT) \ lines with string S anywhere
\     Returns all lines in File that contain string S.

\     Incoming MapNL holds offsets to each new line character in File.
\     If incoming new line map MapNL is purged, one will be created
\     with word fmapNL.

\     On return, File pointer is unchanged.

      [ 128 is pad, "number of items will exceed stack depth" is msg ]
      other filetrue not IF other notafile return THEN

      this "S" book, other "File" book File fpos push 
      (hFile hMapNL qS) fmapl (hMap) any?

      IF these rows depth pad plus plus, depthSTK >
         IF " fstr: " msg cat ersys 
            drop VOL tpurged
         ELSE (hMap) again push rows 1st
            DO File this peek I 1st fetch fseek
               peek I 2nd fetch fget (hTi)
            LOOP
            pull rows pilen (hT)
         THEN
      ELSE VOL tpurged
      THEN File pull fseek
   end

   inline: fstr1 (hFile hMapNL qS --- hT) \ lines with S in 1st column
\     Returns all lines in File with string S in the very first column.

\     Incoming MapNL holds offsets to each new line character in File.
\     If incoming new line map MapNL is purged, one will be created 
\     with word fmapNL.

\     On return, File pointer is unchanged.

      "S" book, bob "F" book F fpos push, (hMapNL) push

      \ Checking the very first bytes for a match:
      F this rewind S strlen fget, S strmatch 0= (f) 
      IF yes (-1) ELSE no one null THEN (hSmap0)

      F NLch S cat fmap (hSmap) 
      (hSmap0 hSmap) pile any?

      IF "Smap" book peek (hMapNL) rows any not
         IF pull drop F fmapNL push THEN

         Smap rows 1st
         DO peek (hMapNL) Smap I pry bsearch (r f)

            IF (r) peek that pry tic (fpos)
               peek two roll tic pry that less (bytes)
               (fpos bytes) F again three roll fseek, bob fget (hTi)

            ELSE (r) drop, purged "Smap" book 
               VOL tpurged (hT)
               \ MapNL does not have the NL found by fmap above;
               \ the file has changed and MapNL is not current:
               " fstr1: incoming new line map is not current" ersys 
               EXIT

            THEN
         LOOP 

         Smap rows any? IF (hT1 hT2 ... hTn n) pilen THEN

         purged "Smap" book
      ELSE VOL tpurged
      THEN pull drop, F pull fseek
   end

   inline: ftemp (qFil --- qFile) \ a unique file name from the system
{     Starting with incoming Fil, return file name File that is 
      unique in the machine file system.

      Upon return, File exists with zero bytes.

      Example: 
         If file is in /tmp and name includes runid, the program will 
         automatically delete it upon exit.  Here is making such a
         file for temporary use:
            "/tmp/" runid + ftemp
         
}     "HND" filetemp HND fclose 
   end

   inline: ftempsys ( --- qFile) \ a unique file name from the system
      [ "/tmp/" runid + "_" + "TMP" book ]
      TMP ftemp 
   end

   inline: ftouch (qFile --- ) \ touch File
    \ File must include path if it is not local.
      SBIN "touch " + swap + " &" + shell 
   end

   inline: fwrite (hA hFile --- toptr tolen) \ A to File, return where
      again rev this file.pos pry push fput
      file.pos pry pull again rev less
   end

   inline: matF ( --- hFile) \ handle to the file opened by word matfile
      "matfile" "matF" extract ;

   inline: matfile (qP qS --- ) \ open the file handle called matF 
\     Open handle matF for file named S at path P; file must exist.  
\     Word matF picks this word's library to supply the file handle.

\     Opens a file handle named matF, closing first the current handle.
      2dup cat file? not 
      IF 2dup cat " file not found: " swap cat ersys return THEN

      [ scalar "matF" book ]
      matF filetrue IF matF fclose THEN cat old binary "matF" file
   end

   inline: matfile?  (hF --- f) \ true if F is handle to MAT-File
\     Test handle F for valid binary MAT-File.
\     Reference: Matlab MAT-File Format for Matlab 5.3, June 1999.

      [ 128 is hbytes, 9 is idDOUB ]
      no MAT stkok not that filetrue not or
      IF drop " file handle required" ersys no return THEN
      this fpos push, this rewind, these six fget
      lowercase "matlab" alike
      IF yes ELSE no THEN swap pull fseek
   end

   inline: mget (hF qS --- hA) \ load matrix S from MAT-File F
{     Reading Matlab MAT-File, double precision matrix named S.
      Reference: Matlab MAT-File Format for Matlab 5.3, June 1999.

      Assumes file was written on machine of like endian.

      Used on big endian machine only; revision required for 
      little endian.

      Assumes values of integers in Matlab size and flags lists
      (Ref, p16) are less than 65536 and therefore can be expressed
      in two bytes even if stored in four.

}     [ "matfile?" "hbytes" extract into header_bytes,
        "matfile?" "idDOUB" extract into idDOUB ]
      that matfile? not
      IF " mget: not a MAT-F file" ersys return THEN
      into S, into F, no is found
      F fpos header_bytes < IF F header_bytes fseek THEN
      BEGIN
         eight F fpos plus, F fsize <
         found not and
      WHILE
         F eight fget endian import4 (hInts)
         this 1st pry into Endian
         2nd pry into Bytes
         F fpos into Pos

         F 40 fget endian import2 (hInts)
         \ Raking out and lopping the pairs of imported 00 00 bytes.
         \ If all imported are 2-byte values, want sixth, seventh and
         \ ninth; otherwise, behavior from here is unpredictable:
         (hInts) these rake lop
         this six ndx pry into Rows
         this seven ndx pry into Cols
         nine ndx pry these (namebytes)

         \ Step around Matlab complexity that appears only to save
         \ four bytes if name length is four or less:
         (namebytes) five <
         IF drop F -4 fstep, four
         ELSE eight slash rounded eight star
         THEN F swap fget this notrailing into S1

         (S1) S alike, S chars any not or
         IF F eight fget endian import4 this 2nd pry
            F swap (bytes) fget (hT)
            swap 1st pry this idDOUB =
            IF (id) drop (hT) endian import8 (hDoubles)
               Rows foldr, yes is found
            ELSE (id) int$ " mload: invalid matrix type "
               swap cat ersys drop F S return
            THEN
         ELSE F Pos fseek, F Bytes fstep
         THEN
      REPEAT
      found
      IF "_" S1 cat naming
      ELSE " mload: matrix " S " not found" cat cat ersys F S
      THEN
   end

   inline: mheader (hF --- ) \ xray of MAT-File header for F
\     hF is handle to file obtained with word file.
      [ "matfile?" "hbytes" extract into hsize ]
      this matfile? not
      IF " mheader: not a MAT-F file" ersys return THEN
      this fpos push, again rewind, this hsize fget
      these chars xray dot pull fseek
   end

   inline: mtoc (hF --- ) \ display the contents of MAT-File F
\     Reference: Matlab MAT-File Format for Matlab 5.3, June 1999.

\     Assumes file was written on machine of like endian.

\     Used on big endian machine only; revision required for 
\     little endian.

      [ "matfile?" "hbytes" extract into hsize
        "matfile?" "idDOUB" extract into idDOUB ]
      this matfile? not
      IF " mtoc: not a MAT-F file" ersys return THEN
      this into F fpos push, F hsize fseek
      BEGIN
         8 F fpos plus, F fsize <
      WHILE
         F 8 fget endian import4 (hInts)
         this 1st pry into Endian
         2nd pry into Bytes
         F fpos into Pos

         F 40 fget endian import2 (hInts)
         dup rake lop
         its 6 ndx pry into Rows
         its 7 ndx pry into Cols
         nine ndx pry these (namebytes)

         (namebytes) five <
         IF drop F -4 fstep, four
         ELSE eight slash rounded eight star
         THEN F swap fget notrailing
         sp dot, sp sp Rows .u " by " . Cols .u

         F eight fget endian import4 1st pry
         sp sp "type " . idDOUB =
         IF "double" ELSE "unsupported" THEN . nl

         F Pos fseek, F Bytes fstep
      REPEAT F pull fseek \ back to original position
   end

   inline: notafile (hFile --- ) \ display not-a-file-handle message
      sp named notag dot " is not a file handle" . nl ;

   inline: op4 ( --- hFile) \ handle to the file opened by word op4file
      "op4file" "op4" extract ;

   inline: op4file (qP qS --- ) \ open the file handle called op4
{     Open handle op4 for file named S at path P.

      Word op4 picks this word's library to supply the file handle.
      Opens a file handle named op4, closing first the current handle.

      DO NOT "UPGRADE" TO USE catpath IN THE PHRASE BELOW.  The form 
      here with cat allows:
         "" "filename" op4file
         "filename" "" op4file
      when the file is local, and people are using it and expect it.  

      Using catpath would mess this "feature" up by always requiring 
      some sort of path (or ./ if local).
}
      [ no "op4" book ] op4 filetrue IF op4 fclose THEN
      cat forn binary "op4" file
   end

   inline: put4q (hA qS --- ) \ put matrix A on binary file named S
\     File is compatible with the format of a Nastran output4 file
\     or a put4 file created by this program.
\     Appends A to other matrices on S.

      [ no "op4" book ] \ local ref for filetrue below
      op4 filetrue IF op4 fclose THEN 
      (qS) forn "op4" binary file, (hA) op4 append4, op4 fclose
   end

   inline: rewind (hFile --- ) \ set File position to zero
      this filetrue IF zero fseek return THEN notafile ;

   inline: seefiles ( --- hT) \ a table of all files the program can see
    \ These are all the directories:
         pwd \ local
         usrpath pile
         syspath pile
         docpath pile

         "PATH" env ":" chblank words \ PATH
         "HOME" env pile
         noq_alike

         pile (hDir)

    \ Weed out duplicate dirs--some may have / on the right, others
    \  may not; this removes right / and eliminates duplicates:
         (hDir) right justify these chars tic +trailing
         "/ " two spaces replace$ chop noq_alike (hDir)

    \ These are all the files in all the directories:
         depth nit push
         (hDir) dup push rows 1st
         DO peek I quote these dirnames catpath this file? rake lop
         LOOP pull drop
         depth pull less pilen

         (hFiles) dup file? rake lop 

      "_seefiles" naming
   end

   inline: stkdelete ( --- ) \ delete stksave files
      "stksave" "Fnames" yank any?
      IF dup rows 1st
         DO dup I quote deleteif LOOP drop
         VOL tpurged "stksave" "Fnames" bank
         0 1 null "stksave" "Depths" bank
      THEN
   end

   inline: stkrestore (qFile --- ...) \ restore saved stack from File
    \ Restore stack saved using word stksave.  After stack is restored,
    \ File is deleted.
    \ If File is not found, assumes stksave was called with empty
    \ stack, and returns with no error message.
      [ no "HAN" book ]
      (qFile) "File" book
      HAN filetrue IF HAN fclose THEN

      "stksave" "Fnames" yank "Fnames" book
      "stksave" "Depths" yank "Depths" book

      Fnames File grepr any?
      IF (hRows) 1st pry (row) "r" book
         Depths r pry (d)
      ELSE return \ assumes stksave saved nothing because stk was empty
      THEN (d)

      File old binary "HAN" open
      HAN rows 0= 
      IF (d) drop " stkrestore: error opening " File + ersys HALT THEN

      (d) any? IF HAN getn (...) THEN
      HAN fclose

      r Fnames rows teeth "R" book
      Fnames R rake "stksave" "Fnames" bank drop
      Depths R rake "stksave" "Depths" bank drop

      File deleteif
   end
      
   inline: stksave (... qFile --- ) \ save stack contents on File
\     Saves no file if stack is empty.
      [ no "HAN" book
        VOL tpurged "Fnames" book
        0 1 null "Depths" book
      ]
      (qFile) "File" book
      File deleteif

      depth (d) dup "d" book 0= IF return THEN

      HAN filetrue IF HAN fclose THEN
      File new binary "HAN" open
      HAN rows 0= 
      IF " stksave: error opening " File + ersys HALT THEN

      d Fnames File grepr any?
      IF (d hRows) 1st pry (row) Depths swap (d Depths row) poke
      ELSE (d) Depths swap pile "Depths" book
         Fnames File pile "Fnames" book
      THEN

      d HAN putn HAN fclose
   end 

   inline: tailf (qFile n --- hT) \ the endmost n lines of ascii File
      [ no is File ]
      File filetrue IF File fclose THEN
      that file?
      IF (n) push (qFile) old ascii "File" open
         File
            File
               File fmapNL these rows peek tic min endmost 1st pry
            fseek
         INF fget textget these rows pull min endmost 
         notrailing File fclose
      ELSE " tailf: file not found: " other cat ersys return
      THEN
      "_tailf" naming
   end

   inline: voc (qFile --- hT) \ list of defined words in File
\     Gathers words defined using 'define:' and 'inline:.'
\     Returned T is purged if no words found.

      [ no is File
      \ These are backward so fstr doesn't find them:
        " :enifed" backward is Str1
        " :enilni" backward is Str2
      ]
      filefound not IF " file not found" . nl return THEN

      File filetrue IF File fclose THEN old ascii "File" file

      File fmapNL push
      
      File peek Str1 fstr
      Str1 tug noblanklines
      Str1 strlen negate indent left justify (hT1)

      File pull Str2 fstr
      Str2 tug noblanklines
      Str2 strlen negate indent left justify (hT2)

      (hT1 hT2) pile noblanklines any?

      IF 1st word drop 
      ELSE VOL tpurged
      THEN
      (hT) noq_alike alphabetize "_voc" naming
   end

\-----------------------------------------------------------------------

{  Word fortfile contains the library for Fortran file operations, con-
   sisting of file handle, record map, and current record pointer.

   It is made here first, so words being made next can refer back to it
   and get ptrs to its library items.

}
   inline: fortfile (hFile --- ) \ set things up for a Fortran File
{     This word must be run before any Fortran utility words can be
      used for File.  It runs word fortmap to map all the records in
      the Fortran file, and keeps the resulting Map in its library.

      Important: If the file endian is different from the machine's,
      word fortendian_set must be run before running this word to
      initialize the file.

      Fortran record index in this library, called Rec, is 1-based.
}
      [ \ Utility Rec++ is run from other words to bump record index:
        "( --- ) Rec tic maxRec min 'Rec' book" "Rec++" inlinex
        purged is Map
        purged is File
        {" 
        fortfile: the Fortran unit is linked to a file handle; run word
          fortclose to unlink the current file handle--the handle will 
          remain open and valid
        "} -7 indent "msg" book
      ]
      File filetrue
      IF msg . nl "" ersys return THEN

      no MAT stkok not IF "fortfile" stknot return THEN

      " Mapping..." . (hFile) dup fortendian fortmap (hMap) "done" .

      (hMap) these rows "maxRec" book
      (hMap) "Map" book
      (hFile) "File" book

      fortrewind
   end

   inline: fortbackspace ( --- ) \ backspace one Fortran record
      [ "fortfile" "Rec" localref ptr "ptrRec" book ]
      ptrRec exe (nrec) nit fortseek
   end

   inline: fortclose ( --- ) \ close file previously opened by fortfile
\     Note: this will NOT close the file handle given previously to 
\     word fortfile.
      purged "fortfile" "File" bank
      purged "fortfile" "Map" bank
   end

   inline: fortendian ( --- N) \ endian of Fortran file
      [ endian "endian" book ] endian
   end

   inline: fortendian_set (N --- ) \ set endian of Fortran file
{     N is the Fortran file's byte order, perhaps from one of these
      words:
         BIG_ENDIAN, LITTLE_ENDIAN, PDP_ENDIAN

      If the file endian is different from the machine's, this word
      must be run before running word fortfile to initialize the file.

}     "fortendian" "endian" bank
   end

   inline: fortrec ( --- hT) \ fetching the current record
\     Upon return, record index has been bumped to the next.
      [ "fortfile" "File" localref ptr "pFile" book
        "fortfile" "Map" localref ptr "pMap" book
        "fortfile" "Rec" localref ptr "pRec" book
      ]
      pFile exe (hFile)
      pMap exe (hMap)
      pRec exe (n, 1-based)

      (hFile hMap n) ndx reach (h1x3)

      (hFile h1x3) 2dup 3rd pry (k, 0-based) fseek
      1st pry fget (hT)

      fortskip \ jumping Fortran file pointer to next record
   end

   inline: fortrec? ( --- n) \ where the file record pointer is pointing
      [ "fortfile" "Rec" localref ptr "pRec" book ]
      pRec exe (n, 1-based)
   end

   inline: fortrewind ( --- ) \ reset record index to the beginning
      [ "fortfile" "File" localref ptr "ptrFILE" book ]
\     Setting 1-based file pointer to one:
      ptrFILE exe (hFile) rewind, one fortseek
   end

   inline: fortseek (n --- ) \ setting Fortran record to 1-based index n
      [ "fortfile" "maxRec" localref ptr "maxRec" book
        "fortfile" "Rec" localref "Rec" book \ no ptr needed to book NUM
        "fortfile" "Map" localref ptr "pMap" book
        "fortfile" "File" localref ptr "pFile" book
      ]
      maxRec exe min, one max (n) dup Rec book

      (n) pFile exe (hFile)
      pMap exe, rot (hMap n) ndx 3rd (i j) fetch (k, 0-based)
      (hFile k) fseek
   end

   inline: fortskip ( --- ) \ skip the current record
      [ "fortfile" "Rec++" localref ptr "pRec++" book
        "fortfile" "Rec" localref ptr "pRec" book
      ] pRec++ exe, pRec exe (n) fortseek
   end

   inline: fortstep (n --- ) \ step record index by n records
      [ "fortfile" "Rec" localref ptr "Rec" book ]
      (n) Rec exe plus fortseek
   end

{  Notes on use of ptr and exe by words that use items from the library
   of word fortfile.

   Most of the fortfile words refer to items in the library of word
   fortfile.  Here is an example of a word that extracts library item
   Rec from the library of word fortfile:

  _inline: fortbackspace ( --- ) \ backspace one Fortran record
      "fortfile" "Rec" extract (nrec) nit fortseek
   end

   A faster way of using an extracted item is to just extract it once
   and obtain a ptr to it, and then simply "exe" the ptr later.  Even
   when an item changes, the ptr to it always is valid.

   To get a ptr, the full catalog name is required, something that word
   localref does.  For example, the full catalog name of Rec that is in
   the library of word fortfile is:

      [tops@steelee] ready > "fortfile" "Rec" localref

       stack elements:
             0 string: Rec,@fortfile  13 characters
       [1] ok!
      [tops@steelee] ready >

   As an alternate form of word fortbackspace, the following shows use
   of words localref and ptr during bracket mode as fortbackspace is
   being created:

  _inline: fortbackspace ( --- ) \ backspace one Fortran record
      [ "fortfile" "Rec" localref ptr "ptrRec" book ]
      ptrRec exe (nrec) nit fortseek
   end

   The ptr to Rec (an item in the library of word fortfile) is given
   the name ptrRec, and stored in the library of the new word.  When
   the word runs later, ptrRec is simply executed by word exe to pro-
   duce the same result as the version using extract shown above.

   Words using the library of fortfile have been written in this style
   of using ptrs to items in the library of word fortfile and exe to
   run them.
}

   private halt

\-----------------------------------------------------------------------
;  Appendix: Examples
\-----------------------------------------------------------------------

\  Examples using words fmap and fmapl with fseek and fget.  The two-
\  column map from fmapl is handy.

      syspath "file.v" cat old binary "F" dup closeif file
      F purged " inline: " fmapl "MAP" book
      depth push 10 1st
      DO MAP I reach (h1-by-2)
         (h1-by-2) F over 1st pry fseek
         (h1-by-2) F swap 2nd pry fget (hQ)
      LOOP F close
      depth pull less pilen notrailing left justify (hT)

      halt

      syspath "file.v" cat old binary "F" dup closeif file
      F fmapNL "L" book F " inline: " fmap "MAP" book
      depth push 10 1st
      DO L dup MAP I pry bsearch drop 2 items reach (hA)
         (hA) F that 1st pry fseek
         (hA) F swap delta 2nd pry fget (hQ)
      LOOP F close
      depth pull less pilen notrailing left justify (hT)

      halt

\  Making a random access text file:

      "ukey.v" filefound drop, asciiload asciify into A

      "ctest.dat" this deleteif, "ctest" this closeif, new ascii file

      A ctest fput \ write volume A, with fixed chars/row, to ctest

      A chars ctest frec \ set random record size to A chars/row

      ctest fprops \ show file handle properties

      ctest 1st 12 items reversed fgetr . \ get some rows in reverse

      ctest fclose, "ctest.dat" delete

\-----------------------------------------------------------------------
{
   Random access can be used for any file having uniform records.  A
   number of header bytes, not necessarily matching the uniform size 
   of each record, can be specified to provide the required offset to
   the first uniform record.

   In the following, phrases meant for the machine can be copied and 
   dropped to demonstrate random access and the use of fhead and frec:

   "headerbytes"
   "A 1       2.23456 3.23456 4.23456 A" pile
   "A 2       6.23456 7.23456 8.23456 A" pile
   "A 3       10.2345 11.2345 12.2345 A" pile
   "A 4       14.2345 15.2345 16.2345 A" pile
   "A 5       18.2345 19.2345 20.2345 A" pile

\  This phrase saves the example file, then creates a handle to it 
\  called f1:

   "fdirect.dat" this rev save, old binary "f1" this closeif file
 
   Here are the bytes of the file.  The first 12 are taken up by the
   header (including a 0Ah newline character), then uniform records of
   36 bytes follow:

[tops@gutter] ready > f1 INF xray .
       0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F  0123456789ABCDEF
   0  68 65 61 64 65 72 62 79 74 65 73 0A 41 20 31 20  headerbytes.A 1
   2  20 20 20 20 20 20 32 2E 32 33 34 35 36 20 33 2E        2.23456 3.
   4  32 33 34 35 36 20 34 2E 32 33 34 35 36 20 41 0A  23456 4.23456 A.
   6  41 20 32 20 20 20 20 20 20 20 36 2E 32 33 34 35  A 2       6.2345
   8  36 20 37 2E 32 33 34 35 36 20 38 2E 32 33 34 35  6 7.23456 8.2345
  10  36 20 41 0A 41 20 33 20 20 20 20 20 20 20 31 30  6 A.A 3       10
  12  2E 32 33 34 35 20 31 31 2E 32 33 34 35 20 31 32  .2345 11.2345 12
  14  2E 32 33 34 35 20 41 0A 41 20 34 20 20 20 20 20  .2345 A.A 4
  16  20 20 31 34 2E 32 33 34 35 20 31 35 2E 32 33 34    14.2345 15.234
  18  35 20 31 36 2E 32 33 34 35 20 41 0A 41 20 35 20  5 16.2345 A.A 5
  20  20 20 20 20 20 20 31 38 2E 32 33 34 35 20 31 39        18.2345 19
  22  2E 32 33 34 35 20 32 30 2E 32 33 34 35 20 41 0A  .2345 20.2345 A.
[tops@gutter] ready > 

\  Setting fhead and frec for random access of uniform records in f1:

   [tops@gutter] ready > 12 f1 fhead, 36 f1 frec, f1 fprops
    Properties of file handle f1
       Handle: 8477BF0
       Type: binary
       Status: old
       Mode: read/write
       Size: 192 bytes
       Random access head: 12 bytes
       Random access size: 36 bytes
       Current byte position: 192

   [tops@gutter] ready > 

   The current byte position of 192 is the result of running xray
   previously.
 
   Here is fetching 10 records in random order:

   First get word ranint if it is missing:

   [tops@gutter] ready > "ranint" missing IF math.v source THEN

   [tops@gutter] ready > f1 1st 5 10 1 ranint fgetr asciify .
   A 3       10.2345 11.2345 12.2345 A
   A 3       10.2345 11.2345 12.2345 A
   A 2       6.23456 7.23456 8.23456 A
   A 1       2.23456 3.23456 4.23456 A
   A 2       6.23456 7.23456 8.23456 A
   A 5       18.2345 19.2345 20.2345 A
   A 1       2.23456 3.23456 4.23456 A
   A 2       6.23456 7.23456 8.23456 A
   A 4       14.2345 15.2345 16.2345 A
   A 3       10.2345 11.2345 12.2345 A
   [tops@gutter] ready > 

   Here is fetching the header.  First rewind the file and get the
   length of the header (12 bytes):

   [tops@gutter] ready > f1 rewind, f1 its file.rechead pry

    stack elements:
          0 number: 12
          1 matrix: f1  9 by 1
    [2] ok!

   Fetching and displaying the header in hex and text forms:

   [tops@gutter] ready > fget, dup .hex, nl dot               
    68 65 61 64 65 72 62 79 74 65 73 0A
   headerbytes
   [tops@gutter] ready > 

   Getting rid of the file.  There is no way to delete a file using 
   its handle name--its system file name is used.

   This closes handle f1 so name f1 can be used for something else:

   [tops@gutter] ready > f1 (handle_name) fclose

   And this gets rid of the file that f1 was linked to:

   [tops@gutter] ready > "fdirect.dat" (file_name) deleteif
}
\-----------------------------------------------------------------------

\  Viewing a Forth screen file:

      "scr" closeif
      "GENET.SCR" old binary "scr" file

      64 (bytes/line) scr frec \ random access record size

      scr 1st that file.size pry 64 slash items (hRecs)
      (scr hRecs) fgetr "scrtext" book
   
      scrtext eview 
      scrtext "genet.4th" save

      scr fclose

\-----------------------------------------------------------------------

   Some data for testing words asciiread1 and asciiread4.
   Run this using phrase: "file.v" "ASCIIREAD TEST" msource halt

ASCIIREAD TEST

{" 
1. This might be a header
   101 201 301 401 501 601 701 801 901 1001 1101 1201
This text 1 2 3 is not all numbers
The next row has a bad number, X
 X 102 202 302 402 502 602 702 802 902 1002 1102 1202
The next row is blank

   102 202 302 402 502 602 702 802 902 1002 1102 1202
   103 203 303 403 503 603 703 803 903 1003 1103 1203
   104 204 304 404 504 604 704 804 904 1004 1104 1204
The next row has too few numbers
   105 205 305 405
   105 205 305 405 505 605 705 805 905 1005 1105 1205
   106 206 306 406 506 606 706 806 906 1006 1106 1206
   107 207 307 407 507 607 707 807 907 1007 1107 1207
"} 
(hT) chop "testread.dat" save \ writing the test file

\  Run a test:
   " Running asciiread1: " . nl
   "testread.dat" 12 asciiread1
   " Running asciiread4: " nl . nl
   "testread.dat" 12 asciiread4

   halt

\-----------------------------------------------------------------------

;  Obsolete words

  _inline: fmap (hFile qS --- hMap) \ maps all strings S in File
\     THIS VERSION IS FLAWED.  IT WILL MISS STRINGS THAT STRADDLE
\     THE BUFFER BOUNDARY.

\     Returned Map contains offsets to all strings S in File.
\     On return, File pointer is unchanged.
      [ 1048576 is Bufsize ]
      that filetrue not IF that notafile return THEN

      "S" book, "File" book

      xbase push
      0based \ 0-based is better for ?DO and offset calc, I*bufsize

    \ bufsize must be an exact multiple of string S length:
      S strlen Bufsize that max that /mod lop star "bufsize" book

      File fpos push, File rewind
      File file.size pry (bytes)
      (bytes) bufsize /mod (rem quot) "quot" book

      no one null (hMap), quot 1st
      ?DO File bufsize fget
         S smap I bufsize * +d
      LOOP quot tic pilen

      File rot (File rem) fget
      S smap quot bufsize * +d pile (hMap)
      File pull fseek, pull indexbase
   end


\  Word fortmap is now a native word.
  _inline: fortmap (hFile endian --- hMap) \ map records in Fortran file
{     Returned Map has 3 columns.  The number of rows in Map equals the
      number of Fortran records in File.
      The Nth row contains properties for record N as follows:
         The first column contains the record size (bytes), the second
         column contains the first 4 bytes of the record interpreted as
         an integer, and the third column holds the 0-based offset to
         the record.
      If the record size in the first column of Map is integer-sized
      (4 bytes), then the value in the second column is its value, if
      it can be taken as an integer.  In such a case, there is no need
      to read the file again to obtain the record's sole value.

      Note: Map is built column-wise as a 3-row matrix, and then
      transposed into a 3-column matrix.
}
      [ INF is cmax, 10000 "Cols" book ]

      (endian) fortendian_set

      xbase push 1based

      (hFile) "fil" book
      fil fsize "sizefile" book

      fil fpos push \ saving original position
      fil rewind

      one "Col" book
      Cols "Cmax" book

      three Cols null (hMap)
      BEGIN (hMap)
         Col Cmax =
         IF three Cols null park \ adding room for Cols more columns
            Cmax Cols plus "Cmax" book
         THEN

         fil fpos four plus (fpos+4) again (offset)
         other (offset hMap) 3rd Col (i j) store \ offset to rec

         (offset) sizefile < (f1)
         Col cmax < (f2)
         (f1 f2) and (f)
      WHILE
         fil eight fget fortendian import4 again (hV)
         other (hV hMap) Col movec \ reclen, 1st int

         fil swap ontop (reclen) fstep
         Col tic "Col" book

      REPEAT
      fil pull fseek \ restoring original position

      (hMap) 1st Col nit items catch transpose
      "_fortmap" naming (hMap)
   end

  _inline: fortint ( --- k) \ fetching integer from current record
\     Upon return, record index has been bumped to the next.
      fortrec "fortfile" "endian" extract import4 ontop
   end

\-----------------------------------------------------------------------


