\ {{{1 GNU General Public License
{
Program Tops - a stack-based computing environment
Copyright (C) 1999-2010  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} 
}

{ File plot.v  November 1999

   Copyright (c) 1999-2010  D. R. Williamson

   Drawing graphs.

   Requires xterm.v words to make window, event and graphics contexts.

   Words of xterm.v are automatically loaded at start up if the program
   has been compiled for X11 graphics by using flag -DX11 in make.inc.

   Listed below are some of the words in the manual that use plotting
   in their examples that is supported by words in this file, such as
   xyzoom:

      btt colorset detrend hanning integral lamp _lamp last linet
      load_demo logarea logistic lpcoef newt palign plot sine sine_rms
      sine_sweep sined skyline stepped sticks trend tshift tuning wCOVt
      wincreate wPSD

------------------------------------------------------------------------

   syspath "plot.v" + asciiload this " inline:" grepr reach dot
   syspath "plot.v" + asciiload this " function " grepr reach dot

   inline: clip (hA --- ) \ define curve A for clipping by word splot
   inline: last (hA N --- hA) \ make plot of last N steps; A remains
   inline: marg (n% --- ) \ set plot top and bottom margin in splot
   inline: noclip ( --- ) \ no clipping
   inline: scut (hP s1 k --- hP) \ plot P from step s1 for k steps
   inline: splot (s1 k hP --- ) \ plot P from step s1 for k steps
   inline: weave (hA hB --- hA1 hB1) \ weave curves

   inline: xyzoom (hP ht hXY --- hP ht) \ zoom to clipboard x,y values
   inline: xyzoom_init ( --- ) \ initialize word xyzoom
   inline: xzoom (hP ht hXY --- hP ht) \ zoom to clipboard x values

   inline: pColors ( --- hV) \ vector of X11 color values
   inline: pColors3 ( --- hColors) \ default line color triplets
   inline: blend (hR hG hB hW N --- qC) \ color value for N
   inline: colortest (qWord --- ) \ views of a color word
   inline: Blue (N --- qN) \ the Nth blue, N from 0 to 100
   inline: Blue1 (N --- qN) \ the Nth blue1, N from 0 to 100
   inline: Brown (N --- qN) \ the Nth Brown
   inline: Burlywood (N --- qN) \ the Nth Burlywood
   inline: Coral (N --- qN) \ the Nth Coral
   inline: DarkOrange (N --- qN) \ the Nth DarkOrange
   inline: Gray (N --- qN) \ the Nth Gray
   inline: Green (N --- qN) \ the Nth green, N from 0 to 100
   inline: Green1 (N --- qN) \ the Nth green1, N from 0 to 100
   inline: Khaki (N --- qN) \ the Nth khaki, N from 0 to 100
   inline: Magenta (N --- qN) \ the Nth Magenta
   inline: Plum (N --- qN) \ the Nth Plum
   inline: Purple (N --- qN) \ the Nth Plum
   inline: Red (N --- qN) \ the Nth red, N from 0 to 100
   inline: Sienna (N --- qN) \ the Nth Sienna
   inline: SlvrBlue (N --- qN) \ the Nth silver blue, N from 0 to 100
   inline: Tan (N --- qN) \ the Nth Tan
   inline: Tomato (N --- qN) \ the Nth Tomato
   inline: colors ( --- ) \ view colors from file color.txt
   inline: pLines ( --- hV) \ vector of X11 line types
   inline: colorset (hList --- hLines hColors) \ colors, lines for graph
   inline: graphdef ( --- ) \ default graph lines and colors
   inline: graphset (hLines hColors --- ) \ set colors, lines for graph

   inline: _plot (hY hX --- ) \ plot Y versus X in figure pfig
   inline: _plotclose ( --- ) \ close the plot window

   function (A1) = nudge(A, n, d1) // nudge lines to avoid overlap
   inline: pscale (hX hXr hWr --- hX1) \ X cols scaled to window axis
   inline: plotGCattrib (nLinWid nLinSty --- ) \ set file's GCB attrib

   inline: CB_CLR ( --- ) \ set all CLPBD values to UNDEF
   inline: CB_DEP ( --- n) \ depth of CLPBD stack
   inline: CB_POP ( --- hC) \ pop 3-by-1 top of stack from CLPBD
   inline: CB_PUSH (hXY --- ) \ push 2-by-1 matrix XY onto CLPBD
   inline: CB_SWEEP ( --- ) \ clear CLBPD stack if too old
   inline: CB_TIMEOUT (sec DTmax --- ) \ clear clipboard timed out
   inline: CLPBD ( --- hC) \ clipboard array, latest first

   inline: KP (hEV hW --- ) \ handler for key pressed event
   inline: KR (hEV hW --- ) \ handler for key released event
   inline: MEV (hEV hW --- ) \ handler for mouse motion event
   inline: pButton (hEV hW --- ) \ handler for button release event
   inline: pExpose (hEV hW --- ) \ handler for expose events in window

   inline: pGRID (hWCB sX sY --- ) \ draw grid lines on current plot
   inline: pGRID1 (hWCB sX sY --- ) \ draw grid lines on current plot
   inline: pgrid ( --- ) \ toggle the flag for placing grids on plot
   inline: pgrid_off ( --- ) \ turn off the display of grid lines
   inline: pgrid_space (nX nY --- ) \ number of grid spaces on plot

   inline: redraw (fig --- ) \ draw fig again
   inline: replot (fig --- ) redraw ; \ draw fig again

   inline: plotWCB ( --- hWCB) \ this file's window control block, WCB
   inline: plotBackPixel (qColor --- ) \ set plot background color

   inline: _Button1 (hXY --- ) \ display values for location in graph
   inline: _Button3 (hXY --- ) \ zoom in and out

   inline: z (hData --- hData) \ step a plot
   inline: zprompter ( --- )

   inline: plotdemo2 ( --- ) \ demo for graph animation

------------------------------------------------------------------------
}
   X11 not IF " X11 graphics required" . halt THEN

   CATMSG push no catmsg

\  General words for simple interactive plotting.

{  Word last is the simplest: given A on the stack, display the last N 
   steps for each column in A, as in: 

      1000 3 random (hA) 50 last

\     Appending a curve to clip upon:
      1000 1 random 0.2 *f dup clip park (hA)
      50 last \ show all with clipping

\     Now clip on the 1st original curve:
      (hA) its 1st catch clip
      100 marg 50 last \ marg works with clip on; zooming out

      noclip 30l \ remove clip (marg now doesn't work); show last 30
      (hA) drop plotclose
}

   inline: clip (hA --- ) \ define curve A for clipping by word splot
\     If A is purged, splot does not clip.
      "splot" "Y" bank
   end

   inline: last (hA N --- hA) \ make plot of last N steps; A remains
      [ 65 is Q ]
      true 
      one NUM stkok and 
      two MAT stkok and not
      IF "last" stknot return THEN

      INF that = IF drop these rows THEN

      "pGRID" "GRID" yank (f)
      IF (N) even tic dup (N) Q > 
         IF dup (N) two / integer 
         ELSE dup (N) nit 
         THEN 
         Q min (nX) "pGRID" "nX" bank
      THEN
      those rows over less tic one max ndx swap them splot
   end

   inline: marg (n% --- ) \ set plot top and bottom margin in splot
\     Top and bottom margins as % of top-to-bottom range.
      100 slash "splot" "margin" bank ;

   inline: noclip ( --- ) \ no clipping
      no "_plot" "Ybooked" bank
      purged clip purged setCLIP ;

   inline: scut (hP s1 k --- hP) \ plot P from step s1 for k steps
      other splot ;

   inline: splot (s1 k hP --- ) \ plot P from step s1 for k steps
{     Bank into Y the curve that dictates clipping, as in:
         C "splot" "Y" bank

      For no clipping, say:
         purged "splot" "Y" bank

      Note: margin (top and bottom) is used only when there is clipping;
      then word marg can be used to set it on-the-fly.
}
      [ purged "Y" book, 0.10 is margin ]
      true one MAT stkok and two NUM stkok and three NUM stkok and not
      IF "splot" stknot return THEN

      these rows "rmax" book, rev 2dup plus rmax >
      IF drop rmax over less tic THEN
      items (hX)
      Y rows any
      IF (hX) this 1st pry "x1" book
         its these rows ndx pry "x2" book
         Y those ndx reach dup minfetch 2drop "y1" book
         maxfetch 2drop "y2" book
         y2 y1 - margin * push
         list: x1 x2 y1 peek less y2 pull plus ;
      ELSE purged
      THEN
      setCLIP
      (hX) these rev ndx reach (hY)
      rose _plot
   end

   inline: weave (hA hB --- hA1 hB1) \ weave curves
{     Thu Aug 19 16:44:24 PDT 2010

      A and B are column vectors of curve Y(X), where X(k)>X(i), k>i.

      Place odd elements of A and even elements of B into A1, and even
      elements of A and odd elements of B into B1.

      Graphing lines A1 and B1 with the same color causes that color
      to fill in between original curves A and B, creating the effect
      of a polygon fill between A and B.

      Using line style LineRectangle causes a line to be plotted as the
      diagonal of a rectangle, and then the rectangle filled with the
      line color.  It was inspired by this word, and does a much better
      job of filling the lines created by this word; see "LineRectangle"
      in xterm.v.

      Example.  This shows creating a lines and colors list for word
      colorset() in an application; curves noted by G1, G2, G3 use
      style LineRectangle (LR) on single lines, while curves noted
      by (U1,U2), (U3,U4), (V1,V2) and (V3,V4) use style LR on line
      pairs created by word weave():

         "LineSolid" "LS" book     \ line type solid
         "LineOnOffDash" "LD" book \ line type dashed
         "LineRectangle" "LR" book \ line type rectangle

         list: \ a color and a line for each column of P
          \                        Curve  Key   Color
            "#236B8E" LS          \ P9     n   Steel Blue
            "#42647F" LS          \ P10    n   Blue Whale
            "#4A777A" LS          \ P11    n   Fenway Monster Blue

            "#213D30" LR          \ G1     w   Packer Green
            "#551011" LR          \ G2     e   Burnt Sienna
            "#22316C" LR          \ G3     i   Delft Blue

            "#83F52C" LS          \ B1     s   Neon Green
            "#00AF33" LS          \ B2     s   True Green
            30 Green  LS          \ B3     s   Medium Green

            "#05E9FF" LS          \ S1     j   Indiglo Blue
            "#1E90FF" LS          \ S2     j   Dodger Blue
            "#1464F4" LS          \ S3     j   Butterfly Blue

            "#004F00" LR 2dup     \ U1,U2  d   Dumpster Green
            "#006400" LR 2dup     \ U3,U4  f   Dark Green
 
            "#003F87" LR 2dup     \ V1,V2  k   Sign Blue
            "DodgerBlue4" LR 2dup \ V3,V4  l   Dark Dodger Blue
         end (hList)
         (hList) colorset (hLines hColors)
         (hLines hColors) "colors" book "lines" book
}
      (hA hB) 1st over rows items (hR)
      dup rows 2 spikes dup "R" book rake "ODD" book "EVEN" book

      (hA hB)  dup EVEN reach (hB_even)
              other ODD reach (hA_odd) R tier (hA1) rev
      (hA hB) swap EVEN reach (hA_even)
               swap ODD reach (hB_odd) R tier (hB1)

      0 "R" book 0 "EVEN" book 0 "ODD" book
   end

   inline: xyzoom (hP ht hXY --- hP ht) \ zoom to clipboard x,y values
{     Using mouse button clicks, zoom into a graph based on x, y values
      at the diagonal corners of a graph subregion.

      The description below references the left and right mouse buttons,
      but the mouse buttons to use are defined in the set up and not by
      this word.  An example set up is given below. 

      A set up using this word can save in the clipboard (word CLPBD) 
      two different points on a graph to define the diagonal corners of
      a subregion by successive clicks of the left mouse button with 
      the mouse arrow placed at the desired corner points.  

      Then clicking the right mouse button can cause zoom in to the de-
      fined region.  

      After a number of zooms in, done by defining more subregion cor-
      ners and clicking the right mouse button, continuing to click the
      right mouse button will successively zoom out through the former 
      zoom-in regions until the full graph is again displayed.
 
      Zoom-in regions are remembered on a local stack (note: local stack
      in this discussion refers to a local stack defined in this word, 
      and not the program's local stack where words like push and pull 
      apply), and zoom out in reverse order begins when there are no 
      subregion corners in the clipboard and subregion corners are in-
      stead popped from the top of the local stack.

      Zoom out continues until the local stack is empty and the full
      graph is again displayed.

      Incoming P and t are arrays exactly like the stack items sent to 
      word plot.  Line width, color, and clipping definitions used by 
      word plot and associated words are assumed to be set up as they
      would be for word plot.

      A note below in this word's macro replot discusses updating a time
      changing graph while it is zoomed.

      Example set up:

      Use of this word requires setting up two button functions.  One
      function saves mouse arrow XY positions to the clipboard and the 
      other fetches the plot arrays and runs this word.

      The following button word fetches plot arrays PG and tG and runs 
      this word, xyzoom; since the arrays remain on the stack after
      xyzoom runs (see stack diagram above), they are dropped to give
      the stack behavior of a button word, which is (hXY --- ):

         inline: ButtonZoom (hXY --- ) \ zoom in or out
            (hXY) PG tG rot (hP ht hXY) xyzoom \ zoom on graph P(t)
            (hP ht) 2drop \ drop the arrays
         end

      This assigns word ButtonZoom to Button3 (the right mouse button) 
      in pButton, the word that handles mouse button events for simple
      plot functions in file plot.v; note that the stack diagram of
      button words is (hXY --- ), which ButtonZoom adheres to:

      \  Set pButton.Button3 = ptr("ButtonZoom"):
         "ButtonZoom" ptr "pButton" "Button3" bank \ right button 
         CB_CLR \ initialize clipboard stack too

      Also required is having another button word add XY to the clip-
      board.  For this example, an existing word for the left mouse 
      button is given the additional task of pushing XY to the clip-
      board using word CB_PUSH:

         inline: Button1 (hXY --- ) \ display values in graph
          \ Display the values at the cursor location in graph.

            dup CB_PUSH \ add XY to CLPBD

            1 csr_str . nl \ display XY formatted to local time
            "auto" "LOCKED" yank not IF cprompt . THEN
         end

      This completes the example set up.  It makes Button 1 clicks add 
      graph XY pairs to the clipboard, and a Button 3 click run xyzoom.

      Also see below this file's default right button function, word
      _Button3, and examples in the man definitions that make graphs 
      allow zooming; see the list of words at the top of this file.
}
      [ -INF "x1" book
        INF  "x2" book
        -INF "y1" book
        INF  "y2" book

      \ Define a stack (called local stack in the discussion):
        0 "XSTKdep" book  \ current stack depth
        16 "XSTKmax" book \ max stack depth

        UNDEF 4 XSTKmax fill "XSTK" book \ 4 rows for each stack item 
        UNDEF 4 1 fill "X2" book

        0 XSTKmax 1- ones pile "RAKE" book

        purged "savCLIP" book

        {" ( --- f) \ f is true if graph is full (not zoomed)
           x1 -INF = x2 INF = and (f1)
           y1 -INF = y2 INF = and (f2)
           (f1 f2) and (f)
        "} "FULL" macro

        {" (hP ht --- ) \ replot graph P(t)
         {
           A note on updating a time changing, zoomed graph:

           In a real time system, P(t) grows as new data arrives.  A 
           word that automatically updates a real time graph can query 
           xyzoom.FULL by running 

              "xyzoom" "FULL" localrun 

           and then run xyzoom.replot if the graph is zoomed (that is,
           if xyzoom.FULL is false), as in

              (hP ht) "xyzoom" "replot" localrun 

           instead of running the ususal

              (hP ht) plot

           Here is an infix example function that takes advantage of
           infix dot notation to run the macros in xyzoom as described
           above:

              function autoplot(P, t) {
                 if(xyzoom.FULL) plot(P, t); /* make full plot */
                 else xyzoom.replot(P, t);   /* make zoomed plot */
              }

           New points will appear if the time axis of the graph has 
           already been sized to accommodate future points, and if the 
           zoomed region includes them.  Of course new points not in 
           the zoomed region will not be seen until zoom out.
         }
           (hP ht)
           FULL
           IF savCLIP purged "savCLIP" book \ use saved CLIP
           ELSE list: x1 x2 y1 y2 end       \ make CLIP for subregion
           THEN (hL) setCLIP

           (hP ht) _plot \ draw graph
        "} "replot" macro

        {" ( --- x1 x2 y1 y2) \ pop topmost zoom stack items
           "XSTK_POP" ERRset
           XSTK RAKE claw (hX hX1) X2 park "XSTK" book (hX)
           XSTKdep 1- 0 max "XSTKdep" book 
           (hX) dup push rows 1st
           DO peek I pry LOOP pull drop (x1 x2 y1 y2)
           ERR
        "} "XSTK_POP" macro

        {" (x1 x2 y1 y2 --- ) \ push items to top of zoom stack
           "XSTK_PUSH" ERRset
           4 listn
           XSTK park 1st XSTKmax items catch "XSTK" book
           XSTKdep 1+ XSTKmax min "XSTKdep" book
           ERR
        "} "XSTK_PUSH" macro

      ]
      "XY" book \ values from clipboard are used, not these

    \ Silently return if stack is not set up right or if MAT rows are 
    \ not compatible:
      (hP ht) true, one MAT stkok and, two MAT stkok and, not (f)
      (hP ht f) other rows other rows <> (hP ht f f1) or (f)
      IF (hP ht) return THEN

      FULL IF getCLIP "savCLIP" book THEN

      CB_DEP 2 <
      IF \ using values from local XSTK
         XSTKdep 0=
         IF FULL IF (hP ht) return THEN \ already at full size graph
            -INF INF -INF INF \ takes all rows
         ELSE 
            XSTK_POP (x1 x2 y1 y2) \ recall where we have been

          \ See if items on the stack match those of the current graph:
            push push (x1 x2)
            (x1 x2) dup     (x2) x2 = (f1)
            (x1 x2 f) other (x1) x1 = (f2)
            (x1 x2 f1 f2) and (fx)

            pull pull (y1 y2)
            (y1 y2) dup     (y2) y2 = (f1)
            (y1 y2 f) other (y1) y1 = (f2)
            (y1 y2 f1 f2) and (fy)

            (x1 x2 fx y1 y2 fy) 3 roll (fx fy) and (f)

            (x1 x2 y1 y2 f) 
            IF 2drop 2drop       \ x and y match the current graph
               (hP ht) XY xyzoom \ reenter and pop the next XSTK values
               return
            THEN
         THEN (x1 x2 y1 y2) "y2" book "y1" book "x2" book "x1" book
      ELSE \ using values popped from global CLPBD
         CB_POP dup 1st pry "x1" book 2nd pry "y1" book
         CB_POP dup 1st pry "x2" book 2nd pry "y2" book

         x1 x2 2dup min "x1" book max "x2" book
         y1 y2 2dup min "y1" book max "y2" book
      THEN
      CB_CLR                \ empty the clipboard

      x1 x2 y1 y2 XSTK_PUSH \ remember where we are going

      (hP ht) 2dup replot   \ run local macro replot

      (hP ht)               \ return with original incoming P(t)
   end

   inline: xyzoom_init ( --- ) \ initialize word xyzoom
      -INF "xyzoom" "x1" bank
      INF  "xyzoom" "x2" bank
      -INF "xyzoom" "y1" bank
      INF  "xyzoom" "y2" bank
      0 "xyzoom" "XSTKdep" bank
      purged "xyzoom" "savCLIP" bank
   end

{  This is a simpler version of xyzoom that just zooms on x, akin 
   to panning.  

   inline: xzoom (hP ht hXY --- hP ht) \ zoom to clipboard x values
{     Zoom into a graph based on x values, and let the program scale 
      the y values in the new subregion.  

      Zoom-in regions are remembered in a stack called XSTK, so zoom-
      out in reverse order to zoom-in is done when the clipboard is 
      empty, until the full graph is again displayed.

      Description for set up with left and right mouse buttons: 

         The left mouse button can be clicked to save two horizontal 
         t (or X) values in the clipboard, and then clicking the right 
         mouse button can cause the graph to zoom to their width.

         After a number of zooms in, continuing to click the right 
         mouse button will successively zoom back out until the full 
         graph is again displayed.
 
      Example set up: see example in word xyzoom.
}
      [ -INF "x1" book
        INF "x2" book

        purged "savCLIP" book

        0 "XSTKdep" book
        8 "XSTKmax" book

        UNDEF 2 XSTKmax fill "XSTK" book
        UNDEF 2 1 fill "X2" book

        0 XSTKmax 1- ones pile "RAKE" book

        {" ( --- x1 x2) \ pop topmost x1 and x2 from zoom stack
           "XSTK_POP" ERRset
           XSTK RAKE claw (hX hX1) X2 park "XSTK" book (hX)
           XSTKdep 1- 0 max "XSTKdep" book
           (hX) dup 1st pry swap 2nd pry (x1 x2)
           ERR
        "} "XSTK_POP" macro

        {" (x1 x2 --- ) \ push x1 and x2 to top of zoom stack
           "XSTK_PUSH" ERRset
           pile XSTK park 1st XSTKmax items catch "XSTK" book
           XSTKdep 1+ XSTKmax min "XSTKdep" book
           ERR
        "} "XSTK_PUSH" macro

        "x1 -INF = x2 INF = and IF yes ELSE no THEN" "FULL" macro
      ]
      "XY" book \ values from clipboard are used, not these

    \ Return if rows are not compatible:
      (hP ht) over rows over rows <> IF (hP ht) return THEN

      FULL IF getCLIP "savCLIP" book THEN

      CB_DEP 2 <
      IF \ using values from local XSTK
         XSTKdep 0=
         IF FULL IF (hP ht) return THEN \ already at full size graph
            -INF INF \ takes all rows
         ELSE XSTK_POP (x1 x2)
            (x1 x2) dup     (x2) x2 = (f1)
            (x1 x2 f) other (x1) x1 = (f2)
            (x1 x2 f1 f2) and (f)
            IF (x1 x2) 2drop    \ x1 and x2 match the current graph
               (hP ht) XY xzoom \ reenter and pop the next XSTK values
               return
            THEN
         THEN (x1 x2) "x2" book "x1" book
      ELSE \ using values from global CLPBD
         CB_POP 1st pry (x1)
         CB_POP 1st pry (x2)
         2dup min "x1" book max "x2" book
         x1 x2 XSTK_PUSH
      THEN
      CB_CLR \ empty the clipboard
      (hP ht)

      (ht) dup x1 bsearch drop "r1" book
      (ht) dup x2 bsearch drop "r2" book

      (hP ht)  over r1 r2 r1 - 1+ items reach (hP1)
      (ht hP1) over r1 r2 r1 - 1+ items reach (ht1)

      FULL IF savCLIP setCLIP purged "savCLIP" book ELSE noclip THEN
     
    \ Plot subset of rows P1(t1):
      (hP ht hP1 ht1) _plot

      (hP ht) \ return with orginal P(t)
   end

   End simpler version of xyzoom. } 

\-----------------------------------------------------------------------

\  Colors.

\  In RH 5.2, for an adjustable color window to create color values, 
\  under the MWM window manager open the menu: 
\     Start/Preferences/Background 
\  (left mouse button opens the Start menu).

\  Saying pColors puts the current color table on the stack.  
  
   inline: pColors ( --- hV) \ vector of X11 color values
      [ 0 'colorTable' book ] colorTable exe ;

\  Word colorTable stores into pColors the ptr of the word that will 
\  supply the colors when pColors is said.

   define: colorTable (qS --- ) \ store ptr to word S as color table
\     The behavior of word S is to place a list of colors on the stack:
\        S ( --- hList)
      0 STR stkok not IF "colorTable" stknot return THEN
      ptr "pColors" "colorTable" bank
   end

\  Here is the table of default colors:
   inline: pColors3 ( --- hColors) \ default line color triplets
      [ list: \ similar color (BGR) repeats every third one:
           list: 

              "DeepSkyBlue3" "DarkOliveGreen" "Firebrick3"

              "SlateBlue" "MediumSeaGreen" "IndianRed2" 

              "SkyBlue" "SeaGreen3" "Tomato1" 

           end 
           words (hT) again push rows 1st
           DO peek I quote colorpix (ncolor) LOOP pull drop
        end 100 repeat "colors" book
      ] colors
   end

   "pColors3" colorTable \ default

\  -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -

\  Experimenting with colors.

\  Word colors runs through all the colors in file doc/color.txt.

\  Some words, like Blue1, take four similar colors from color.txt and
\  then scale or interpolate R, G, and B colors.

\  Blending colors.
   inline: blend (hR hG hB hW N --- qC) \ color value for N
\     Interpolating N in levels W to blend R, G, B into a color.
      "N" book
      this rot park N lerp ontop no max 255 min push
      this rot park N lerp ontop no max 255 min push
      swap park N lerp ontop no max 255 min pull pull colorvalue
   end

   inline: colortest (qWord --- ) \ views of a color word
\     Word is a color word with stack diagram: (n --- )
\     Example usage: "Green" (qWord) colortest
      [ "Blue" "COLOR1" book ]
      "COLOR2" book

      X11 not IF " X11 graphics required" . halt THEN
      "sine" missing IF "mmath.v" source THEN
      "scrape" missing IF "mat.v" source THEN
      "_plot" missing IF "plot.v" source THEN
      list: 100 0 DO I COLOR1 main "LineSolid" 5 +LOOP end
      list: 100 0 DO I COLOR2 main "LineSolid" 5 +LOOP end cat
      colorset graphset purged setCLIP
      21 1 DO 1 .01 I * + rad/cyc 0 0.001 1001 sine drop LOOP
      21 1 DO 1 .01 I * + rad/cyc 180 0.001 1001 sine drop LOOP
      42 parkn .7 scrape .001 those rows uniform plot pause plotclose
      "pColors3" colorTable \ put default back
   end

   inline: Blue (N --- qN) \ the Nth blue, N from 0 to 100
{     Lower N is lighter.

      begin man Blue
         "colortest" missing IF "plot.v" source THEN
         "Green" "colortest" "COLOR1" bank
         "Blue" colortest
      end

      Old:
      [ list: 72  67  58  39  ; reversed "R" book
        list: 118 110 95  64 ; reversed "G" book
        list: 255 238 205 139 ; reversed "B" book
        list: 0 33 66 99 ; reversed "W" book
      ] R G B W four roll blend
}
      [ list: 0   0   0   ; "R" book
        list: 178 154 64  ; "G" book
        list: 238 205 205 ; "B" book
        list: 10  20  50  ; "W" book
      ] R G B W four roll blend

   end

   inline: Blue1 (N --- qN) \ the Nth blue1, N from 0 to 100
{     Lower N is lighter.

      To test, run: "plot.v" "begin man Blue1" msource

      begin man Blue1
         "colortest" missing IF "plot.v" source THEN
         "Blue" "colortest" "COLOR1" bank
         "Blue1" colortest
      end
}     [ {" R   B   B from color.txt
           72 118 255             RoyalBlue1
           67 110 238             RoyalBlue2
           58  95 205             RoyalBlue3
           39  64 139             RoyalBlue4
        "} 3 matread
        dup 1st catch 0.0 *f "R" book \ scaling R (more -> purple)
        dup 2nd catch 0.9 *f "G" book \ scaling G (more -> silver)
        3rd catch "B" book            \ interpolating B to new slopes
           list: 255 238 139 ;
           list: 255 220 130 ; 
           park (hXY) yes sort B lerp "B" book
 
        list: 0 25 75 100 ; "L" book
      ] "N" book
      L R park N lerp ontop no max 255 min
      L G park N lerp ontop no max 255 min
      L B park N lerp ontop no max 255 min
      colorvalue
   end

   inline: Brown (N --- qN) \ the Nth Brown
\     Lower N is lighter.
      [ list: 255 238 205 139 ; "R" book
        list: 165 154 133  90 ; "G" book
        list:  64  59  51  35 ; "B" book
        list: -30  10  30  80 ; "W" book
      ] R G B W four roll blend
   end

   inline: Burlywood (N --- qN) \ the Nth Burlywood
\     Lower N is lighter.
{
      begin man Burlywood
         "colortest" missing IF "plot.v" source THEN
         "Blue" "colortest" "COLOR1" bank
         "Burlywood" colortest
      end
}
      [ list: 255 238 205 139 ; "R" book
        list: 211 197 170 115 ; "G" book
        list: 155 145 125  85 ; "B" book
        list: -60 -40  60  80 ; "W" book
      ] R G B W four roll blend
   end

   inline: Coral (N --- qN) \ the Nth Coral
\     Lower N is lighter.
      [ list: 255 238 205 139 ; "R" book
        list: 114 106  91  62 ; "G" book
        list:  86  80  69  47 ; "B" book
        list:  20  40  60  80 ; "W" book
      ] R G B W four roll blend
   end

   inline: DarkOrange (N --- qN) \ the Nth DarkOrange
\     Lower N is lighter.
      [ list: 255 238 205 139 ; "R" book
        list: 127 118 102  69 ; "G" book
        list:   0   0   0   0 ; "B" book
        list:  20  40  60  80 ; "W" book
      ] R G B W four roll blend
   end

   inline: Gray (N --- qN) \ the Nth Gray
\     Lower N is darker.
      [ list: 0 255 ; "RGB" book
        list: -100 100  ; "W" book
      ] RGB dup dup  W four roll blend
   end

   inline: Green (N --- qN) \ the Nth green, N from 0 to 100
{     Lower N is lighter.

      begin man Green
         "colortest" missing IF "plot.v" source THEN
         "Green1" "colortest" "COLOR1" bank
         "Green" colortest
      end
}
      [ 0 "R" book
        list: 0 100 ; (hN)
        dup  list: 100 179 ; reversed park "G" book
        (hN) list: 0   116 ; reversed park "B" book
      ] 
      0 max 100 min push 
      R, G peek lerp, B pull lerp colorvalue
   end

   inline: Green1 (N --- qN) \ the Nth green1, N from 0 to 100
{     Lower N is lighter.

      begin man Green1
         "colortest" missing IF "plot.v" source THEN
         "Green" "colortest" "COLOR1" bank
         "Green1" colortest
      end
}     [ {"
           0 255 127             SpringGreen1
           0 238 118             SpringGreen2
           0 205 102             SpringGreen3
           0 139  69             SpringGreen4
        "} 3 matread
        dup 1st catch 1.0 *f "R" book
        dup 2nd catch "G" book
           list: 255 238 139 ;
           list: 225 170 80 ; park (hXY) yes sort G lerp "G" book
        3rd catch 1.0 *f "B" book

        list: 0 25 75 100 ; "L" book
      ] "N" book
      L R park N lerp ontop no max 255 min
      L G park N lerp ontop no max 255 min
      L B park N lerp ontop no max 255 min
      colorvalue
   end

   inline: Khaki (N --- qN) \ the Nth khaki, N from 0 to 100
{     Lower N is lighter.

      begin man Khaki
         "colortest" missing IF "plot.v" source THEN
         "Green" "colortest" "COLOR1" bank
         "Khaki" colortest
      end
}     [ {"
           255 246 143             khaki1
           238 230 133             khaki2
           205 198 115             khaki3
           139 134  78             khaki4
        "} 3 matread
        dup 1st catch 0.6 *f "R" book
        dup 2nd catch 0.8 *f "G" book
        3rd catch 0.9 *f "B" book

        list: 0 25 75 100 ; "L" book
      ] "N" book
      L R park N lerp ontop no max 255 min
      L G park N lerp ontop no max 255 min
      L B park N lerp ontop no max 255 min
      colorvalue
   end

   inline: Magenta (N --- qN) \ the Nth Magenta
\     Lower N is lighter.
      [ list: 255 238 205 139 ; "R" book
        list: 0   0   0   0   ; "G" book
        list: 255 238 205 139 ; "B" book
        list:   0  33  66  99 ; "W" book
      ] R G B W four roll blend
   end

   inline: Plum (N --- qN) \ the Nth Plum
\     Lower N is lighter.
      [ list: 255 238 205 139 ; "R" book
        list: 187 174 150 102 ; "G" book
        list: 255 238 205 139 ; "B" book
        list:   0  33  66  99 ; "W" book
      ] R G B W four roll blend
   end

   inline: Purple (N --- qN) \ the Nth Plum
\     Lower N is lighter.
      [ list: 155 145 125  85 ; "R" book
        list:  48  44  38  26 ; "G" book
        list: 255 238 205 139 ; "B" book
        list:   0  33  66  99 ; "W" book
      ] R G B W four roll blend
   end

   inline: Red (N --- qN) \ the Nth red, N from 0 to 100
{     Lower N is lighter.

      begin man Red 
         "colortest" missing IF "plot.v" source THEN
         "Tomato" "colortest" "COLOR1" bank
         "Red" colortest
      end
}     [ {"
           255 106 106             IndianRed1
           238  99  99             IndianRed2
           205  85  85             IndianRed3
           139  58  58             IndianRed4
        "} 3 matread
        dup 1st catch "R" book
           list: 255 238 139 ;
           list: 255 190 100 ; park (hXY) yes sort R lerp "R" book
        dup 2nd catch 0.0 *f "G" book
        3rd catch 0.0 *f "B" book

        list: 0 25 75 100 ; "L" book
      ] "N" book
      L R park N lerp ontop no max 255 min
      L G park N lerp ontop no max 255 min
      L B park N lerp ontop no max 255 min
      colorvalue
   end

   inline: Sienna (N --- qN) \ the Nth Sienna
\     Lower N is lighter.
      [ list: 255 238 205 139 ; "R" book
        list: 130 121 104  71 ; "G" book
        list:  71  66  57  38 ; "B" book
        list:   0  33  66  99 ; "W" book
      ] R G B W four roll blend
   end

   inline: SlvrBlue (N --- qN) \ the Nth silver blue, N from 0 to 100
\     Lower N is lighter.
      [ list: 135 126 108  74 ; "R" book
        list: 206 192 166 112 ; "G" book
        list: 255 238 205 139 ; "B" book
        list: -30  10  30  80 ; "W" book
      ] R G B W four roll blend
   end

   inline: Tan (N --- qN) \ the Nth Tan
\     Lower N is lighter.
      [ list: 255 238 205 139 ; "R" book
        list: 165 154 133  90 ; "G" book
        list:  79  73  63  43 ; "B" book
        list: -20  40  60  80 ; "W" book
      ] R G B W four roll blend
   end

   inline: Tomato (N --- qN) \ the Nth Tomato
\     Lower N is lighter.
      {
        begin man Tomato
           "colortest" missing IF "plot.v" source THEN
           "Green" "colortest" "COLOR1" bank
           "Tomato" colortest
        end
      }
      [ list: 255 238 205 139 ; "R" book
        list:  99  92  79  54 ; "G" book
        list:  71  66  57  38 ; "B" book
        list: -20  40  60  80 ; "W" book
      ] R G B W four roll blend
   end

   inline: colors ( --- ) \ view colors from file color.txt
\     A loop for viewing the colors defined in file color.txt.

      "colorset" missing IF "plot.v" source THEN
      "color.txt" filefound
      IF remtabf asciiload into F ELSE " no color file" . HALT THEN

      F 3 matread into C
      1st 1- "LINE" book

      BEGIN
         "[Enter] for next, - [Enter] to back up, Esc+q to exit > "
         accept_keys
         IF 1st catch "-" = IF -1 ELSE 1 THEN
         ELSE 1
         THEN LINE + 1st max "LINE" book

         LINE intstr ": " + F LINE quote asciify crowd + . nl

         C LINE reach push
         list: 3 1 DO peek I pry LOOP pull drop colorvalue "LineSolid"
         end colorset graphset

         1000 1 random 1st those rows items plot 

         ESCq
      UNTIL
      plotclose
   end

\-----------------------------------------------------------------------

\  Lines. 

\  Saying pLines puts the current line table on the stack.  

   inline: pLines ( --- hV) \ vector of X11 line types
      [ 0 'lineTable' book ] lineTable exe ;

\  Word lineTable stores into pLines the ptr of the word that will 
\  supply the line types when pLines is said.

   define: lineTable (qS --- ) \ store ptr to word S as line table
\     The behavior of word S is to place a list of lines on the stack:
\        S ( --- hList)
      0 STR stkok not IF "lineTable" stknot return THEN
      ptr "pLines" "lineTable" bank
   end

   define: pLinesSolid ( --- hLines) \ default all solid lines
      [ LineSolid pColors rows 1 fill into L ] L
   end

   "pLinesSolid" lineTable \ default

\-----------------------------------------------------------------------
{
   Setting colors and lines for current plot:

   Word colorset is used to get colors and lines during word creation, 
   since the system takes awhile to look up colors:
   (in brackets during word creation)
      [ list:
           "DarkGoldenrod2" "LineSolid"  \ angular velocity
           "FireBrick4" "LineSolid"      \ pointing vector
           "Green2" "LineSolid"          \ error
        end
        (hList) colorset "colors" book, "lines" book
      ]

   Then in the body of the word when the plot is being made later,  
   word graphset is run using the looked-up lines and colors:
      ... lines colors graphset ...
}
   inline: colorset (hList --- hLines hColors) \ colors, lines for graph
\     Incoming List holds text pairs of color and line descriptors.
\     Incoming List may be a quote string of paired color-line words.
\     Outgoing Lines and Colors are graphics numerical equivalents.

      words these rows two spikes rake (hLines hColors)
      (hColors) push
      list: peek rows 1st
         DO peek I quote colorpix LOOP pull drop
      end (hColors) push \ vector of colorpix values

      (hLines) push
      list: \ firing line style strings with word main:
         pull main
      end (hLines) \ vector of line style values
      pull (hColors)
   end

   inline: graphdef ( --- ) \ default graph lines and colors
      pLinesSolid pColors3 graphset
   end

   inline: graphset (hLines hColors --- ) \ set colors, lines for graph
\     Appends a batch of gray for extra curves that might be parked.
      [ {"
           20 is N \ data for N extra curves
           list:
              list: 1 N ; list: 0 100 ; park
              1 N items lerp integer (hColorIndices) push
              peek rows ndx 1st
              DO peek I pry Gray "LineSolid" LOOP
              pull drop
           end
           (hList) colorset "Grays" book, "Lines" book
        "} "grays" inlinex
        purged are Grays
      ]
      Grays rows any not IF grays THEN
      Grays pile "'xColor' book" main "xColor" colorTable
      Lines pile "'xLine' book" main "xLine" lineTable
   end

\-----------------------------------------------------------------------

\  Word graphpair displays the first curve (column) in Goldenrod,
\  and subsequent curves in alternating blue and green.

   define: colorpairs ( --- hC)
      [ list:
           "MediumBlue" "LineSolid"
           "DarkGreen" "LineSolid"

           "RoyalBlue" "LineSolid"
           "DarkOliveGreen4" "LineSolid"

           "DodgerBlue" "LineSolid"
           "SpringGreen4" "LineSolid"

           "SteelBlue" "LineSolid"
           "SpringGreen3" "LineSolid"

           "CornFlowerBlue" "LineSolid"
           "SpringGreen1" "LineSolid"

           "DeepSkyBlue3" "LineSolid"
           "DarkSeaGreen" "LineSolid"

           "DeepSkyBlue1" "LineSolid"
           "MediumSeaGreen" "LineSolid"

           "LightSkyBlue" "LineSolid"
           "LightSeaGreen" "LineSolid"
        end
        into colors
      ] colors
   end

   define: graphpair ( --- ) \ default colors and lines, master+pairs
\     Setting up colors and line styles:
      [
      \ Plotting these curves in this order:
        list: "Goldenrod" "LineSolid" ;  \ curve 1 color
        colorpairs \ blue, green pairs
        pile

        words these rows 2 spikes rake (hLines hColors)

        (hColors) push
        list: these rows 1st
           DO peek I quote colorpix LOOP pull drop
        end 60 repeat
        "xColor1" book \ vector of colorpix values

        (hLines) push \ firing line style strings with word main:
        list: pull main
        end 60 repeat
        "xLine" book \ vector of line style values

      ] xColor1 "'xColor' book" main, xLine "'xLine' book" main
      "xColor" colorTable, "xLine" lineTable
   end

   define: graphpair1 ( --- ) \ default colors and lines, pairs+master
\     Setting up colors and line styles:
      [ 
      \ Plotting these curves in this order:
        colorpairs \ blue, green pairs
        list: "Goldenrod" "LineSolid" ;  \ last curve color
        pile

        words these rows 2 spikes rake (hLines hColors)

        (hColors) push
        list: these rows 1st
           DO peek I quote colorpix LOOP pull drop
        end 60 repeat
        "xColor1" book \ vector of colorpix values

        (hLines) push \ firing line style strings with word main:
        list: pull main
        end 60 repeat
        "xLine" book \ vector of line style values

      ] xColor1 "'xColor' book" main, xLine "'xLine' book" main
      "xColor" colorTable, "xLine" lineTable
   end

\-----------------------------------------------------------------------

\  Making a plot.

   "1" "pfigmax" inlinex \ max plot figures before cycle back to 1st

   "[ pfigmax 2 null 'XY' book ] XY" "pXY" inlinex \ XY addr of fig data

   "[ scalar 'pfig' book ] pfig" "pfig" inlinex \ addr 1based fig count
   1 pfig !

   "[ scalar 'hold' book ] hold" "hold" inlinex \ address of hold flag

   inline: _plot (hY hX --- ) \ plot Y versus X in figure pfig
\     This word is fired by word plot in boot.v.
    \ Verify stack:
      true one MAT stkok and, two MAT stkok and, not 
      IF "plot" stknot return THEN 
{
    { NO LONGER REQUIRE LENGTHS OF X AND Y TO BE EQUAL.  
      ALLOW DIFFERENT LENGTHS, AND RESOLVE THEM WHEN DRAWING THE GRAPH.
      (SEE WORD pExpose.)
      THIS ALLOWS, FOR EXAMPLE, LEAVING BLANK SPACE ON THE RIGHT AS THE
      FOLLOWING DEMONSTRATES:
         list: 0 1 ; list: 0 1 2 ; plot

      NOTE THAT ANOTHER WAY TO DO THIS IS WITH setCLIP:
         list: 0 2 0 1 ; setCLIP list: 0 1 ; dup plot
    } 
      these rows them rows = not
      IF these rows them cols =
         IF swap bend swap \ transpose Y if its cols = rows of X
         ELSE " _plot: unequal X and Y rows" ersys return 
         THEN
      THEN
}
      these dims star 0> not 
      IF " X vector is purged" ersys return THEN

    \ Which figure:
      hold @ 
      IF pfig @ one max 
      ELSE pfig @ 1+ dup pfigmax > IF drop one THEN  
      THEN dup pfig ! into nfig

    \ Combine incoming with existing nfig array if necessary:
      hold @, pXY nfig ndx 1st fetch 0<> and
      IF swap park (hXY), pXY nfig ndx 1st fetch exe (hXn) dup push 
         those cols 1- clone lerp \ interpolate Y(X) to Y(Xn)
         pXY nfig ndx 2nd fetch exe (hYn) swap (hYn hY) park pull
      THEN (hY hXn)
      
      [ no "Ybooked" book 
        purged "saveCLIP" book

        no "USE_OUT_OF_CORE" book

        {" (hA1 hA2 --- f) \ f is true if bytes of A1 match bytes of A2
         \ Make sure CLIP region is still the same:
           getCLIP any?
           IF (hL1) saveCLIP (hL2) any?
              IF (hL1 hL2) - null? (f)
              ELSE (hL1) drop false
              THEN (f)
           ELSE saveCLIP rows 0= \ is saveCLIP purged too?
              IF true ELSE false THEN
           THEN (f)

           IF (hA1 hA2) over sizeof over sizeof =
              IF (hA1 hA2) xor nullc totals @ 0= \ true if all cols null
              ELSE (hA1 hA2) 2drop false
              THEN
           ELSE (hA1 hA2) 2drop false
           THEN
        "} "MATCH" macro
      ]

      "__X" nfig ndx suffix "Xname" book
      "__Y" nfig ndx suffix "Yname" book

      (hY hXn) USE_OUT_OF_CORE (hY hXn f)
      IF
       \ Book into lib with suffix nfig, replacing any former arrays
       \ Note: Here, word book is replaced by fbook to store out of 
       \ core and save memory:

         (hY hXn) Xname fbook
    
       \ If new Y on stack matches the one previously booked, just 
       \ return.
         (hY) Ybooked 
         IF (hY) dup Yname main MATCH (f) \ compare Y with prev booked
            IF (hY) drop return THEN
         THEN (hY)

         (hY) dup minfetch 2drop 10 >
         IF (hY) dup maxfetch 2drop 65536 (2 16 pow) < 
          \ If Y values are unsigned (> 10), allow them to be truncated
          \ and booked out of core as 2- or 4-byte ints:
            IF (hY) Yname fbook2
            ELSE (hY) Yname fbook4
            THEN
         ELSE
            (hY) Yname fbook
         THEN
         yes "Ybooked" book
         getCLIP "saveCLIP" book

      ELSE
         (hY hXn)

       \ Book into lib with suffix nfig, replacing any former arrays:
         (hY hXn) Xname book
         (hY) Yname book

      THEN

      Xname ptr pXY nfig ndx 1st store \ booked name ptr X into pXY
      Yname ptr pXY nfig ndx 2nd store \ booked name ptr Y into pXY

    \ Display the plot:
      plotWCB wcb.win pry UNDEF =
      IF pfig @ plotWCB wcb.usr poke
         plotWCB "plotWCB" "title" yank wincreate
         plotWCB winshow
      ELSE pfig @ redraw
      THEN
   end 

   public
   inline: _plotclose ( --- ) \ close the plot window
\     This word is fired by word plotclose in boot.v.
      "plotWCB" exists?
      IF plotWCB wcb.win pry UNDEF <> 
         IF "plotWCB" main winfree THEN 
      THEN
      no "_plot" "Ybooked" bank
      purged "_plot" "saveCLIP" bank
   end
   private

{" This word is written in infix.  Gather its text and run eval.

   function (A1) = nudge(A, n, d1) { // nudge lines to avoid overlap
   /* Nudge rows of column n of A by d, if necessary, so no value in
      any row r of column n matches any value in row r of the other
      columns of A.  To determine equality, integer values of A are 
      used.

      Intended for A values that are pixels in a graph.

      The name of a custom function that nudges ordinates Y so lines
      do not overlap can be booked into pExpose as follows: 
         "myNudge" "pExpose" "NUDGE" bank

      The stack diagram of function "myNudge" is (hY --- hY1). 

      To turn off NUDGE in pExpose, run the following to reset the 
      default:
         "noop" "pExpose" "NUDGE" bank 
   */
      {
         DIFF = 2; // nudge if difference is less than DIFF
      }
      if(cols(A) == 1 || n > cols(A)) return(A);

      if(d1 < 0) d = min(d1, -1); else d = max(d1, 1);

      R = teeth(n, cols(A)); 
      (N, D) = claw(A, R);  // separate column n from the rest

   /* The values to be compared are integers (do not use rounding): */
      N1 = integer(N);
      D1 = integer(D);

      BEGIN;
         E = (abs(D1 - clone(N1, cols(D1))) < DIFF); 
      WHILE(!null?(E));
         N += (eps = abs((across(E) != 0))*d); // nudge N(r) by d
         N1 += eps; 
      REPEAT;

      A1 = mesh(N, D, R); // reassemble A with nudged N

      D = D1 = E = eps = N = N1 = R = purged;
   }

"} eval \ parse infix into an inline and add to library

   inline: pscale (hX hXr hWr --- hX1) \ X cols scaled to window axis
{     Incoming X holds graph data for an axis.

      Incoming Xr and Wr are 2-by-1 matrices as follows:
         Xr holds min and max graph values for data in X
         Wr holds axis values corresponding to window min and max

      The two values in Xr must be in ascending order.  The two values
      in Wr may be in any order.
}
      (hX hXr hWr) them cols clone park (hXYtable)
      swap (hXYtable hX) lerp 
   end

\-----------------------------------------------------------------------

\  Graphics context block

\  Making GCB for plot curves

   GCBcreate "plotGCB" book \ graphics context block for plot curves

\  Setting GC attributes in graphics attributes vector (while setting
\  the corresponding mask bits in number on stack):

   GAVcreate "plotGAV" book

   public
   inline: plotGCattrib (nLinWid nLinSty --- ) \ set file's GCB attrib
      push push

      zero (initial mask on stack) \ making mask:
      pull (nLinWid)   plotGAV gav.linewidth  poke, GCLineWidth or
      pull (nLinSty)   plotGAV gav.linestyle  poke, GCLineStyle or
      one (Bool yes)   plotGAV gav.expose poke GCGraphicsExposures or
      GXcopy           plotGAV gav.function   poke, GCFunction or
      "Black" colorpix plotGAV gav.foreground poke, GCForeground or
      (mask) plotGCB gcb.gam poke \ mask bits number into plotGCB.gam

      "plotGAV" ptr plotGCB gcb.gav poke \ gav ptr into plotGCB.gav
   end
   private
   2 LineSolid plotGCattrib

\-----------------------------------------------------------------------

\  Clipboard

\  Word CLPBD and related CB_XXX implement a local clipboard for mouse
\  arrow events on XWindows graphs.  Word xyzoom in this file uses this
\  clipboard for zooming into and out of graph subregions.

   inline: CB_CLR ( --- ) \ set all CLPBD values to UNDEF
      UNDEF 3 "CB_DEP" "CBmax" yank fill, "CLPBD" "C" bank
      0 "CB_DEP" "DEP" bank
   end

   inline: CB_DEP ( --- n) \ depth of CLPBD stack
      [ 8 "CBmax" book \ max depth of clipboard
        0 "DEP" book   \ current depth of clipboard
      ] DEP
   end

   inline: CB_POP ( --- hC) \ pop 3-by-1 top of stack from CLPBD
\     If CLPBD is empty, 3-by-1 C will contain UNDEF values.

\     For usage, see notes in word CLPBD.

      [ 0 "CB_DEP" "CBmax" yank 1- ones pile "RAKE" book
        UNDEF 3 1 fill "C2" book ]

      CLPBD RAKE claw (hC hC1) C2 park "CLPBD" "C" bank (hC)
      "CB_DEP" "DEP" yank, 1- 0 max, "CB_DEP" "DEP" bank 
   end

   inline: CB_PUSH (hXY --- ) \ push 2-by-1 matrix XY onto CLPBD
{     Push the XY coordinates of cursor arrow onto the clipboard stack.

      A third row, the current time, is appended to XY to make a 3-row 
      vector that is pushed to the top of the clipboard stack.

      If the clipboard stack is full, the bottommost stack item is lost.

      If the current item on top of the clipboard stack is more than N
      seconds old, the clipboard stack is cleared before XY is pushed 
      to it.

      For usage, see notes in word CLPBD.
}
      [ "CB_DEP" "CBmax" yank "CBmax" book 
        30 "N" book \ seconds timeout
      ]
      0 MAT stkok not IF "CB_PUSH" stknot return THEN

      time N CB_TIMEOUT \ clear CLPBD if older than N

      (hXY) time pile (hC1) "CLPBD" "C" yank park (hC)

      (hC) dup cols CBmax > 
      IF 1st CBmax items catch \ bottommost item is lost
      THEN (hC)

      (hC) "CLPBD" "C" bank \ latest is column 1, top of stack

      "CB_DEP" "DEP" yank, 1+ CBmax min, "CB_DEP" "DEP" bank 
   end

   inline: CB_SWEEP ( --- ) \ clear CLBPD stack if too old
      [ 30 "DT" book ] 
      CB_DEP 0>
      IF time DT CB_TIMEOUT \ clear if older than DT seconds
      THEN
   end

   inline: CB_TIMEOUT (sec DTmax --- ) \ clear clipboard timed out
      "DTmax" book
      (sec) CLPBD 1st catch 3rd pry (csec) dup UNDEF <>
      IF (sec csec) - DTmax > IF CB_CLR THEN 
      ELSE 2drop 
      THEN
   end

   inline: CLPBD ( --- hC) \ clipboard array, latest first
{     Latest values are in column 1.  

      Each column of C corresponds to a mouse click within an XWindows
      graph window.  Rows of C are: 
         1 mouse arrow X
         2 mouse arrow Y
         3 time of click

      Usage:
      To use CLPBD, set pButton.ButtonX = ptr("CB_PUSH") where X = 1, 
      2 or 3, as in the following example for button 3:
         "CB_PUSH" ptr "pButton" "Button3" bank 

      This example will cause each right button (button 3) click to
      push XY of the mouse arrow onto the clipboard.  

      To use the clipboard later, word CB_POP fetches the data from 
      the clipboard.

      For example, CB_POP can fetch the topmost two XY pairs from the 
      clipboard stack after a user has clicked on a graph to define the
      upper left and lower right corners of a subregion, for use in 
      (fictional) word zoom (xyzoom in this file is a working example
      of using these clipboard words):
         CB_POP 1st 2 items reach (hXY1)
         CB_POP 1st 2 items reach (hXY2)
         (hXY1 hXY2) zoom \ zooming graph to a subregion
}
      [ 3 0 null "C" book ] C
   end
   CB_CLR
   1 60 / "CB_SWEEP" PLAY \ every 60 seconds

\-----------------------------------------------------------------------

\  Event control block

   ECBcreate "plotECB" book

   inline: KP (hEV hW --- ) \ handler for key pressed event
{     Sat Mar 20 16:10:17 PDT 2010 add macros for Alt, Ctrl and Shift.

      Record the code of the key being pressed while graphics window W 
      has focus.

      Elements of incoming EV are stored by dispatcher() in term.c.

      While a key is pressed, KP.NKEY will equal the key code.  For
      example: 
         keypressed = KR.NKEY;

      As soon as a key is released, KP.NKEY = -1, an invalid key code.

      From dispatcher() in file term.c, this shows that index KPkc in 
      EV is an event key's keycode:

         *(EV+KPkc)=event.xkey.keycode;

      and from file term.h, this shows that KPkc is the 7th item in an
      enum, making its value 6 since enums are zero-based:

         enum eventKP {KPwin,KPx,KPy,KPrx,KPry,KPst,KPkc,KPsize};

      Also see event handlers in the example for the manual in xterm.v.
}
      [ -1 "NKEY" book, 
        7 "KPkc" book \ 1-based index, gives 6 or 7 with ndx

       \ Flags for continuously pressed keys; these match the Names
       \ in KTABLE:
         no "ALT" book
         no "CTRL" book
         no "SHFT" book

       \ Table of names for keys that can remain pressed while other
       \ keys are pressed.  
         {" nKey  Name
            64    ALT
            113   ALT
            37    CTRL
            109   CTRL
            50    SHFT
            62    SHFT
         "} dup 1st word drop numerate
            swap 2nd word drop vol2mat bend park
            yes sort "KTABLE" book

         {" (nKey --- qName) \ Name from KTABLE for Key that is pressed
            KTABLE swap bsearch (r f)
            IF (r) KTABLE swap 2nd fetch num2str strchop (qName)
            ELSE (r) drop "" \ empty string, no KTABLE entry
            THEN (qKEY)
         "} "KHELD" macro
{
         Window ID is element wcb.win of the window control block, such
         as plotWCB[wcb.win] if using the default of this file; and the
         ID of the one window in the entire system that has focus is 
         given by word winfocus.

         Word MEV in this file senses mouse motion events, and runs
         macro KP.KHELD-CLR when the plot window loses focus.
 
         When window loses focus, macro KP.KHELD-CLR defined below can
         be run to clear flags for held keys.

         Examples of uppercase using flag KP.SHFT:

            (k) "KP" "SHFT" yank (f) IF (k) uppercase THEN (K)

         or (see word KR):

            if(KP.SHFT) k = uppercase(KR.LKEY(KR.NKEY));
            else k = KR.LKEY(KR.NKEY);

         See the appendix of sys/ukey.v for a utility that shows the
         key codes for graphics window key-pressed events; this phrase
         should run the utility, which was used to get the data for
         KTABLE above and LTABLE of word KR:

            syspath "ukey.v" + "KEYMAP" msource
}
         {" ( --- ) \ clear KHELD Names
           KTABLE 2nd catch nodupes bend mat2vol (hT)
           dup rows 1st DO no over I quote book LOOP drop
         "} "KHELD-CLR" macro

       \ These macros turn on and off the flags that have the names 
       \ given in KTABLE:
         "(nkey --- ) KHELD any? IF yes swap book THEN" "KHELD-ON" macro
         "(nkey --- ) KHELD any? IF no swap book THEN" "KHELD-OFF" macro
      ]
      (hW) drop 
      (hEV) KPkc ndx pry (nkey) \ get key code
      (nkey) dup KHELD-ON "NKEY" book
   end
   "KP" ptr plotECB ecb.kp poke \ KeyPressedEvent handler

   inline: KR (hEV hW --- ) \ handler for key released event
{     Sat Mar 20 16:10:17 PDT 2010 run macro KP.KHELD-OFF

      Key released while graphics window has focus; set KP.NKEY to
      invalid -1.  See notes in word KP.

      The latest key released is:

         keyreleased = KR.NKEY;
}
      [ "KP" "KPkc" yank "KPkc" book -1 "NKEY" book 

       \ Table of letters for keys:
         {" letter, nkey pairs:
            a 38 b 56 c 54 d 40 e 26 f 41 g 42 h 43 i 31
            j 44 k 45 l 46 m 58 n 57 o 32 p 33 q 24 r 27
            s 39 t 28 u 30 v 55 w 25 x 53 y 29 z 52
         "} words dice (hA hN) numerate swap vol2mat bend park
            yes sort "LTABLE" book

         {" (nKey --- qLetter) \ Letter from LTABLE for Key 
            (nKey) LTABLE swap bsearch (r f)
            IF (r) LTABLE swap 2nd fetch num2str strchop (qLetter)
            ELSE (r) drop "" \ empty string, no LTABLE entry
            THEN (qLetter)
         "} "LKEY" macro

       \ Table of symbols for keys; these are symbols that cannot be
       \ converted to upper case:
         {" symbol, nkey pairs when KP.SHFT = no:
            1 10 2 11 3 12 4 13 5 14 6 15 7 16 8 17 9 18 0 19 - 20 = 21
            [ 34 ] 35 ; 47 ' 48 , 59 . 60 / 61
         "} strings dice (hA hN) numerate swap vol2mat bend park
            yes sort "S1TABLE" book

         {" symbol, nkey pairs when KP.SHFT = yes:
            ! 10 @ 11 # 12 $ 13 % 14 ^ 15 & 16 * 17 ( 18 ) 19 _ 20 + 21
            { 34 } 35 : 47 " 48 < 59 > 60 ? 61
         "} strings dice (hA hN) numerate swap vol2mat bend park
            yes sort "S2TABLE" book

         {" (nKey --- qSymbol) \ Symbol from S1TABLE or S2TABLE for Key
             "KP" "SHFT" yank IF S2TABLE ELSE S1TABLE THEN (hTable)
            (nKey hTable) dup 2 roll bsearch (r f)
            IF (hTable r) 2nd fetch num2str strchop (qSymbol)
            ELSE (hTable r) 2drop "" \ empty string, no Table entry
            THEN (qSymbol)
         "} "SKEY" macro
{
         Example of macro KR.LKEY:

            k = KR.LKEY(KR.NKEY);

         k is an empty string if a letter for KR.NKEY is not found.

         Examples of uppercase using flag KP.SHFT:

            (k) "KP" "SHFT" yank (f) IF (k) uppercase THEN (K)

         or

            if(KP.SHFT) k = uppercase(KR.LKEY(KR.NKEY));
            else k = KR.LKEY(KR.NKEY);

         See the appendix of sys/ukey.v for a utility that shows the 
         key codes for graphics window key-pressed events; this phrase 
         should run the utility, which was used to get the data for
         KTABLE of word KP and LTABLE above:

            syspath "ukey.v" + "KEYMAP" msource
}
      ]
      (hW) drop
      -1 "KP" "NKEY" bank \ make KP.NKEY invalid

      (hEV) KPkc ndx pry (nkey) \ get key code
      (nkey) dup "KP" "KHELD-OFF" yank "NKEY" book
   end
   "KR" ptr plotECB ecb.kr poke \ KeyReleasedEvent handler

   inline: MEV (hEV hW --- ) \ handler for mouse motion event
{     Mon Mar 22 12:34:35 PDT 2010

      This motion event handler is used to sense when the plot window
      has lost focus, and then perform tasks related to losing focus. 

      These are some items related to motion events, but none of them 
      are required here:

         This shows that element MEwin is the window of the motion
         event, taken from dispatcher() in term.c:
            *(EV+MEwin)=event.xmotion.window;

         Enum in term.h shows the elements of the MotionEvent:
         /* MotionEvent (MotionNotify): */
            enum eventME {MEwin,MEx,MEy,MErx,MEry,MEst,MEh,MEsize};
}
      [ 1 "SEC" book
        0 "WIN" book

      \ This macro is on an alarm that is continually reset to SEC 
      \ seconds in the future as motion continues; it finally runs
      \ after motion has ceased for SEC seconds:
        {" ( --- )
           WIN winfocus <> \ has window lost focus?
           IF \ do tasks when focus is gone:
              "KP" "KHELD-CLR" yank \ clear the held key flags
              CB_CLR                \ clear the clipboard
           THEN 
        "} "MEV-TASKS" macro 
      ]

      (hEV hW) wcb.win pry "WIN" book drop
      SEC "MEV" "MEV-TASKS" localref ALARM \ run in SEC
   end
   "MEV" ptr plotECB ecb.me poke \ MotionEvent handler

   inline: pButton (hEV hW --- ) \ handler for button release event
{     Places on stack a 2-by-1 vector of X,Y coordinates at the cursor 
      arrow, and runs a user function that consumes the vector.

      Calls one of three user functions depending upon the button. 

      Example usage:

         Load the words of this file:
            "plot.v" source
            "pButton" wholib \ shows default Button ptrs

         Create a word to process a 2-by-1 vector of X,Y coordinates, 
         like the following:

            define: showXY (hXY --- ) nl dup 1st pry " X = " dot dot, \
               2nd pry " Y = " dot dot ;

         and verify that it works:

            list: 1 pi ; showXY

         Then bank a ptr to the new word in the local library here in
         word pButton, say for button 1 (left), by running the phrase:

            "showXY" ptr "pButton" "Button1" bank
            "pButton" wholib \ shows showXY Button1 ptr installed

         and make a sample plot:

            1000 1 random 1 1000 items plot

         Now whenever the left button is clicked with the cursor arrow 
         at a point in the plot window, user word showXY will display 
         the (X, Y) coordinates of the point at the ready prompt in the
         program window.

         This closes the plot window:

            plotclose

}     [ 
      \ 1-based indices that work with ndx for 0- or 1-based indexing:
        list: 2 1 ; is Yrev \ to reverse the two rows of Y-scale data
        2 is X, 3 is Y \ cursor coordinates in E structure

      \ See dispatcher(), term.c, where BRbu, button type, is the 7th 
      \ item:
      \    *(EV+BRbu)=event.xbutton.button;
        7 is BRbu \ location of button type in E structure

      \ Ptrs to user button functions:
        0 "Button1" book \ left button
        0 "Button2" book \ center button (left+right on 2-button mouse)
        0 "Button3" book \ right button

      \ Butts is a list of ptrs to ptrs for the three buttons:
        list: "Button1" ptr, "Button2" ptr, "Button3" ptr end 
        (hPtrs) into Butts
      ] 
      (hW) drop

    \ Extract the button ptr from Butts:
      Butts (hB) over 
      (hEV) BRbu ndx pry (num) \ *(EV+BRbu)=event.xbutton.button;
      (hB num) ndx pry (ptrN) 
      (ptrN) exe (nPtr) \ exe ptrN to get nPtr

      (nPtr) dup 0= IF (nPtr) drop (hEV) drop return THEN
      (nPtr) push \ user's function to local stack

      "pExpose" "sX" yank (hXY) \ window X range of values
      over (hEV) X ndx pry (hXY hx) lerp @ (X) \ interpolated X

      "pExpose" "sY" yank (hXY) \ window Y range of values
{
      Y rows are reversed because +Y axis is up (graph convention) while
      window +Y axis is down (text convention).  Must reverse rows so 
      1st col vals are ascending for lerp:
}     (hXY) Yrev ndx reach (hXY)
      rot (hEV) Y ndx pry (hXY hy) lerp @ (Y) \ interpolated Y

      (X Y) 2 listn (hXY) pull (nPtr) exe \ running user's function
   end
   "pButton" ptr plotECB ecb.br poke \ handler for ButtonReleasedEvent

   inline: pExpose (hEV hW --- ) \ handler for expose events in window
      \ Fri Aug 20 10:18:24 PDT 2010.  Add GRIDlast.
      [ scalar "sX" book, scalar "sY" book, no is hWCB 
        "noop" "NUDGE" book 

        no "GRIDlast" book \ false to draw grid lines first
      ]
      "plotGCB" main into hGCB \ allows changing plotGCB at any time
      (hW)  into hWCB 
      (hEV) into hEV \ info only; not used, but "redraw" yanks it

    \ Margin is a percentage of the minimum dimension:
      hWCB dup wcb.w pry, swap wcb.h pry min 
      0.05 * "Margin" book

    \ Which plot:
      hWCB wcb.usr pry (pfig) ndx (r)

    \ Scaling X:

      pXY over (r) 1st fetch exe (hX)

      getCLIP any?
      IF (hL) 1st two items reach (hXrange)
      ELSE (hX)  
         dup minfetch 2drop over maxfetch 2drop pile (hXrange)
      THEN (hXrange)

    \ Window horizontal range, left to right:
      Margin (wx1) hWCB wcb.w pry, Margin - (wx2) pile (hWrange)

      (hXrange hWrange) 2dup swap park "sX" book

      (hX hXr hWr) pscale "X" book

    \ Scaling Y:
      pXY swap (r) 2nd fetch exe (hY) 

      getCLIP any?
      IF (hL) 3rd two items reach (hYrange)
      ELSE (hY)
         dup minfetch 2drop over maxfetch 2drop pile (hYrange)
      THEN (hYrange) 

    \ Window vertical range: reversing window range to convert from 
    \ screen convention, Y axis down, to graph convention, Y axis up:
      hWCB wcb.h pry Margin - (wy2), Margin (wy1) pile (hWrange)

      (hYrange hWrange) 2dup swap park "sY" book
      (hY hYw hYr) pscale NUDGE main "Y" book

    \ Draw grid lines:
      GRIDlast not IF hWCB sX sY pGRID THEN

    \ Resolve different X and Y lengths:
      Y rows X rows <>
      IF Y rows X rows > 
         IF   Y 1st X rows items reach "Y" book
         ELSE X 1st Y rows items reach "X" book
         THEN
      THEN

    \ Graphics context:
      hWCB hGCB GCcreate

      xbase (base) push, 1based 

      getCLIP rows any
      IF sX 1st catch, sY 1st catch pile onto L
         Y cols 0> 
         IF 1st Y cols ndx \ drawing in reversed order so 1st is on top
            hWCB wcb.typ pry 1 = (trace)
            IF DO
                \ Color and line style of curve I:
                  pColors I pry hGCB GCfore
                  pLines  I pry hGCB GCline

                \ Draw curve I:
                  hWCB hGCB Y I catch X L linet \ X must be ascending
               LOOP
            ELSE
               DO
                \ Color and line style of curve I:
                  pColors I pry hGCB GCfore
                  pLines  I pry hGCB GCline

                \ Draw curve I:
                  hWCB hGCB Y I catch X L linec
               LOOP
            THEN
         THEN
      ELSE Y cols 0> 
         IF 1st Y cols ndx \ drawing in reversed order so 1st is on top
            DO
            \ Color and line style of curve I:
              pColors I pry hGCB GCfore
              pLines  I pry hGCB GCline

            \ Draw curve I:
              hWCB hGCB Y I catch X line

            LOOP 
         THEN
      THEN

      pull (base) indexbase 

    \ Draw grid lines:
      GRIDlast IF hWCB sX sY pGRID THEN

      hGCB GCfree
      purged "X" book
      purged "Y" book
   end
   "pExpose" ptr plotECB ecb.ee poke \ initial handler for ExposeEvent

   inline: pGRID (hWCB sX sY --- ) \ draw grid lines on current plot
\     This is a driver for a grid line word when its ptr is banked
\     here.
      [ no is ptrGRID 

      \ Initial values:
         yes "GRID" book
         4 is nX \ spaces along X
         4 is nY \ spaces along Y
      ]
      GRID not IF 2drop drop return THEN
      ptrGRID 0<> IF ptrGRID exe THEN
   end

   inline: pGRID1 (hWCB sX sY --- ) \ draw grid lines on current plot
{     Makes simple grid lines based on the window dimensions.

      The first column of incoming X and Y holds the window, and the
      second column holds the graph range.  For this simple word, only
      the first column is used.
}     [ \ Making graphics context:
         GCBcreate makes GCB \ graphics context block for grid lines
         GAVcreate makes GAV \ graphics attrib vec for grid lines

      \  Setting desired GC attributes in graphics attributes vector:
         0 (initial mask on stack) \ making mask:
         0              GAV gav.linewidth poke GCLineWidth or
         LineOnOffDash  GAV gav.linestyle poke GCLineStyle or
         1 (Bool yes)   GAV gav.expose    poke GCGraphicsExposures or
         GXcopy         GAV gav.function  poke GCFunction or
         "gray30" colorpix GAV gav.foreground poke GCForeground or
         (mask) GCB gcb.gam poke \ putting mask bits number into GCB.gam

         "GAV" ptr GCB gcb.gav poke \ gav ptr into GCB.gav
      ]
      "pGRID" "nX" yank 2 max "nX" book
      "pGRID" "nY" yank 2 max "nY" book

      (sY) 0 nY pile, swap (sY) 1st catch dup "Yrange" book park (hXY)
      1 nY nit items lerp dup park "Ylines" book

      (sX) 0 nX pile, swap (sX) 1st catch dup "Xrange" book park (hXY)
      1 nX nit items lerp dup park "Xlines" book

    \ Set graphics context this window:
      (WCB) GCB 2dup GCcreate

      Xlines rows 1st
      DO 2dup (WCB GCB) Yrange Xlines I reach bend line LOOP

      Ylines rows 1st
      DO 2dup (WCB GCB) Ylines I reach bend Xrange line LOOP

      2drop
   end

   inline: pgrid ( --- ) \ toggle the flag for placing grids on plot
\     This word toggles flag GRID in word pGRID.
      yes "pGRID" "GRID" yank
      xor "pGRID" "GRID" bank
   end
      
   inline: pgrid_off ( --- ) \ turn off the display of grid lines
      no "pGRID" "GRID" bank
   end
      
   inline: pgrid_space (nX nY --- ) \ number of grid spaces on plot
      (nY) "pGRID" "nY" bank
      (nX) "pGRID" "nX" bank
   end
\  Set initial grid handling:
   "pGRID1" ptr "pGRID" "ptrGRID" bank
   pgrid_off \ initial setting

   inline: redraw (fig --- ) \ draw fig again
    \ Fri Aug 20 11:03:31 PDT 2010.  Use pointer plotECB[ecb.ee] to the
    \    handler for expose events, rather than incorrectly calling
    \    default handler pExpose.

      pXY over 1st fetch 0= IF drop " No such figure" . nl return THEN

      "pExpose" "hWCB" yank (hW) any? not
      IF plotWCB \ default
      THEN
      swap (fig) over wcb.usr poke

    \ Clear the window, redraw it, and raise it to the top:
      dup (hW) 0 0 0 0 winclear 
      dup (hW) winraise 

      (hW) "pExpose" "hEV" yank (hEV) swap (hEV hW) 

      (hEV hW) "plotECB" main ecb.ee pry exe \ run expose event handler
   end

   inline: replot (fig --- ) redraw ; \ draw fig again

\-----------------------------------------------------------------------
{
   Window control block 
   This section makes a window control block called plotWCB to be used
   by the words in plot.v.  Many other window control blocks can be made
   the same way.
}
   _plotclose

\  Making plotWCB an inline means it can be seen by non-public words.
\  When it runs, plotWCB puts its window control block on the stack.

   inline: plotWCB ( --- hWCB) \ this file's window control block, WCB
      [ WCBcreate 'WCB' book 
        progname "@" host + + "title" book \ default name in frame
      ] WCB
   end

   "plotECB" ptr plotWCB wcb.ecb poke \ window event control block

   WAVcreate "plotWAV" book
   "plotWAV" ptr plotWCB wcb.wav poke \ window attributes vector

   0 (mask)
   CWBackPixel or 
   plotWCB wcb.wam poke \ window attributes mask

\  Background pixel for CWBackPixel:
   public
   inline: plotBackPixel (qColor --- ) \ set plot background color
\     Must close window and recreate for changed color to take effect.
      colorpix plotWAV wav.backpixel poke
   end
   private
   "Black" plotBackPixel
{
   "black" colorpix plotWAV wav.backpixel poke
   "bisque" colorpix plotWAV wav.backpixel poke
   "ivory" colorpix plotWAV wav.backpixel poke
   "LightSlateGray" colorpix plotWAV wav.backpixel poke
   "DarkSlateGray" colorpix plotWAV wav.backpixel poke
   "ForestGreen" colorpix plotWAV wav.backpixel poke
   "Antiquewhite" colorpix plotWAV wav.backpixel poke
}
\  Window size and placement:

   600 plotWCB wcb.w poke \ window width
   500 plotWCB wcb.h poke \ window height

   186 plotWCB wcb.x poke \ upper left corner at x on screen
   0 plotWCB wcb.y poke \ upper left corner at y on screen

 \ How to set the use of linec or linet in pExpose:
 \ 0 plotWCB wcb.typ poke \ 0 or UDEF, general (linec)
 \ 1 plotWCB wcb.typ poke \ 1, trace (linet; X must be ascending order)
{
   719 plotWCB wcb.w poke \ window width
   464 plotWCB wcb.h poke \ window height

   0  plotWCB wcb.x poke \ upper left corner at x on screen
   26 plotWCB wcb.y poke \ upper left corner at y on screen
}
\-----------------------------------------------------------------------

\  Default left and right button functions.

\  Left button function:
   inline: _Button1 (hXY --- ) \ display values for location in graph
\     Display the raw XY values corresponding to cursor location in
\     graph.  This word also pushes XY to the clipboard, for possible
\     use by xyzoom.

      dup CB_PUSH \ push XY to clipboard for possible use by xyzoom

      bend " x, y: %8.4f, %8.4f " format nl .
      cprompt sp . \ show console prompt
   end
\  Installing ptr to word _Button1 as the left button function in
\  word pButton of file plot.v:
   "_Button1" ptr "pButton" "Button1" bank \ left button function ptr

\  Right button function:
   inline: _Button3 (hXY --- ) \ zoom in and out
{     To zoom, require XY preceded on the stack by two matrices for
      word plot: (hY hX hXY).  These matrices, X and Y, are fetched
      from the library of _plot() assuming they are the ones just
      graphed as noted next.

      This word fetches plotted matrices called Yname and Xname from
      the library of word _plot (note the double reference below) that 
      were just graphed by word _plot (see file plot.v).  

      Also required to zoom are two zoom region corners in the two top-
      most elements of the CLPPB.  The action of word _Button1 is to 
      push XY queries to the clipboard in addition to its function of 
      displaying XY values at the cursor location, so two _Button1 XY
      queries followed by _Button3 should zoom in.  Pressing Button3
      again should zoom back out.
}
      "KP" "NKEY" yank (nKey)
      (nKey) dup 37 = swap 109 = or \ one of the Ctrl keys?
      IF CB_CLR THEN \ clear points from CLPBD if a Ctrl key is pressed

      (hXY) "_plot" dup "Yname" yank (__Yn) yank (hY)
            "_plot" dup "Xname" yank (__Xn) yank (hX)
            (hY hX) rot xyzoom 2drop
   end
\  Set pButton.Button3 = ptr("Button3"):
   "_Button3" ptr "pButton" "Button3" bank \ right button function ptr

\-----------------------------------------------------------------------

\  Plot animation.

   inline: z (hData --- hData) \ step a plot
{     This word runs under the multitasker, adding "step" to the ending
      row index of Data each time it is run, and making a graph of width
      "span."

      Also run is local word ztask (hData d --- ) whose ptr may be 
      banked here; the default is 2drop, to do nothing.
}
      [ \ Initial values:
        90 (steps) "span" book \ graph width
        span 1.25 * integer "d" book \ current step in data
        one (step) "step" book \ data index increment per time step

        "2drop" ptr "ztask" book \ default ztask (hData d --- )
      ]
      depth 0= IF zs return THEN
      this type MAT <> IF zs return THEN

      d step plus dup "d" book

      (d) span plus nit dup "dmax" book

      (hData dmax) those rows ndx >
      IF \ " end of data" nl . nl cprompt . 
         zs return 
      THEN

\     Executing ztask function with hData and latest index on stk:
      dup (hData) dmax ztask exe

\     Making the graph:
      d span other splot
   end

\  Defining word z in the multitasker:
      0.5 (Hz) "z" TASK

   define: z0 ( --- ) \ set the window for z plot to earliest
      "z" "span" yank, 1.25 * integer "z" "d" bank
   end

   define: zd (n --- ) \ set the step for z plot to index n
      ndx "z" "span" yank less "z" "d" bank ;

   define: zl (hData n --- hData) \ animate the last n steps
      those rows swap less tic zd zw ;

   inline: zprompter ( --- )
      ok @ push
      ok off \ so .s works
      nl
      BEGIN
         " ESC+q to exit > " accept_keys (0 or qT -1)
         IF main THEN nl ESCq
      UNTIL
      pull ok !
   end

   define: zrate (w --- ) \ set animation rate to w frames per second
      (w) "z" RATE ;
      
   define: zs (hData --- hData) "z" SLEEP ; \ zs [Enter] to stop

   define: zspan (n --- ) \ set the span of z plot to n steps
      "z" "d" yank, "z" "span" yank + (d+span) over (n) -
      1 max "z" "d" bank, "z" "span" bank
   end

   define: zw (hData --- hData) "z" WAKE ; \ zw [Enter] to awaken

\  End plot animation.

\-----------------------------------------------------------------------

   pull catmsg halt

\-----------------------------------------------------------------------

;  Appendix

   Example 1.

   "_plot" sourceof source X11 not IF halt THEN

\  This makes curves of random numbers:

   1000 1 random 100 *f is Y1
   1 Y1 rows items is X1

   Y1 50 +d is Y2, X1 is X2
   Y2 50 +d is Y3, X1 is X3

\  Y1 X1 plot 
\  Y2 X2 plot 
\  Y3 X3 plot 

   Y1 Y2 Y3 park park dup 175 +d park X1 plot

   private halt

   Example 2.

   "plotdemo2" sourceof source, X11 not IF halt THEN
   "load_demo" missing IF "mat.v" source THEN

   CATMSG push no catmsg

   inline: plotdemo2 ( --- ) \ demo for graph animation
\     Dogs chasing the Dow.
      [
        list: \ Setting up colors and lines for 7 curves.
          \ Dogs:
            00 Coral        "LineSolid" \ curve 1, S3
           -20 Green        "LineSolid" \ curve 2, S2
           -20 Blue         "LineSolid" \ curve 3, S1
            60 Coral        "LineSolid" \ curve 4, L3
           120 Green        "LineSolid" \ curve 5, L2
           100 Blue         "LineSolid" \ curve 6, L1
          \ Dow:
           "DarkGoldenrod2" "LineSolid" \ curve 7, Dow
        end
        (hList) colorset "colors" book, "lines" book
      ]
      load_demo any?
      IF lines colors graphset \ activate this word's lines and colors

       \ Setting clip and margin (this could be done later instead:
         (hC) dup one endcols clip \ using last curve for clipping
         200 marg \ 200% top and bottom margins on curve used for clip
      ELSE purged
      THEN
   end

   pull catmsg 

   plotdemo2 (hC) any?
   IF 
      90 zspan \ points in window
      2 (Hz) "z" RATE \ speed up the default 1 Hz
      " Dogs chasing the Dow" .
      zw pause 
      zs (hC) drop plotclose 'z' OMIT
   THEN
   noclip

   private halt

\  A loop for viewing the colors in file color.txt:

   "colorset" missing IF "plot.v" source THEN
   "color.txt" filefound 
   IF remtabf asciiload into F ELSE " no color file" . halt THEN

   F 3 matread into C
   C rows 1st 
   DO I .i sp F I quote . nl
      C I reach push
      list: 3 1 DO peek I pry LOOP pull drop colorvalue "LineSolid"
      end colorset graphset

      1000 1 random 1st those rows items plot pause
   LOOP
   halt
{

To do:

2008: Boy is this stuff old, dating back to 1999.  Consider a rewrite 
of _plot to make "hold" really work, and devise a way to work with 
curves defined by different X, Y pairs on the same graph.

Consider a plot structure for each curve that contains pointer to
data (X and Y), line color, line type and line width.  The word that
plots would have to change graphics context in its loop to accomodate
the changing line specs.  Perhaps the word could be "smart" so it
just plots as usual (a matrix containing columns, each column a curve)
if the stack item is just a matrix, or fancy plot if the stack item
is a plot struct.

Eliminate pfigmax.  Pile some kind of plot structure as more are
defined--make pfig, figure number, one of the structure elements.
WCB can hold a user pointer: make it the plot, pfig, to close the
loop (this has been started with wcb.usr in some code of this file).

Make plotWCB part of the structure, and allow different plotWCBs.

And use a word like matlab's "figure" to define the plots: "figure"
will add another row in the fig table. (allow NUM or STR, and convert
NUM to STR using int '%d' format: no, quotes make STR too much trouble)

Also flag in structure if log or linear plot.

On left or right key, scroll plot left or right (panning).
DONE: it isn't panning, but xyzoom zooms

Want to be able to specify color for each line.  DONE: pColor in pExpose

On mouse click, return XY value of plot-->also save plot range in the
structure and this is a snap. DONE: word pButton

Want to specify grid lines. DONE: word pGRID et al

}


