/* {{{1 GNU General Public License

sofea - the Stack Operated Finite Element Analysis program
Copyright (C) 2004  Al Danial <al.danial@gmail.com>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
}}}1 */
#include <stdio.h>
#include <stk.h>
#include "fea_assemble.h"
#include <sql.h>

int inertial_prop()   /* ( qModelDB hMgg --- hMxyz ) {{{1 */
/*
 * man entry:  inertial_prop {{{2
 * ( qModelDB hMgg --- hMxyz ) Computes inertial properties from the model's mass matrix.  If the model's inertial units are weights rather than masses, the outputs will be the model's weight in the three directions (terms 1,2,3) and center of mass (terms 4,5,6).  Mgg must be ordered according to the renumbered_nid.new_id sequence.
 * category: FEA
 * related: sparse, element_matrix, fea_assemble
 * 2}}}
 */
{
int DEBUG = 0;  /* 1 = regular debug, 2 = info on every dof */

    int     rc, dof, node, nDOF, n_found, n_remain, offset; 
    double  Mx, My, Mz, CMx, CMy, CMz, *Mxyz, *mass_diag;
    sqlite3 *dbh;
    char query[QUERY_STR_SIZE+1];
    callback_data sql_data;
    char    T[ERR_MSG_SIZE+1];

if (DEBUG) gprintf("top of inertial_prop\n");
    Mx = 0.0;    CMx = 0.0; 
    My = 0.0;    CMy = 0.0; 
    Mz = 0.0;    CMz = 0.0;
    if (!is_sparse(tos)) {
        stkerr(" inertial_prop: ", SPARSENOT);
        return 0;
    }
    if (!is_low_tri(tos) || !is_up_tri(tos)) { /* Mgg must be diagonal matrix */
        stkerr(" inertial_prop: ", "cannot yet handle coupled mass matrices");
        return 0;
    }
    spdiag();  /* extract the diagonal from the g-set mass matrix */
    swap();    /* ( Mgg_diag qModelDB ) */
    if (!db_open()) {   /* open model db ( Mgg_diag dbh ) */
        stkerr(" inertial_prop: ", "failed to open model db");
        return 0;
    }
    dbh = (sqlite3 *) tos->tex;
    swap(); /* ( dhb Mgg_diag ) */
    mass_diag = tos->mat;
    nDOF      = tos->row;  /* should be # dof in g-set */
if (DEBUG) gprintf("inertial_prop nDOF=%d\n", nDOF);

    n_remain           = 1;
    offset             = 0;
    sql_data.row_index = 0;
    while (n_remain) {
        sql_data.nRows = 0;
        snprintf(query, QUERY_STR_SIZE,
                 "select new_id,x1,x2,x3 from renumbered_nid,node "
                 "  where renumbered_nid.orig_id = node.seq_no "
                 "limit %d offset %d;" ,
                 SQL_BLOCK_SIZE, offset);
        rc = sql_do_query(dbh, query, &sql_data);
        /* the query returns
         *       renumbered node ID     = sql_data.i[][0]
         *       X in basic coordinates = sql_data.x[][1]
         *       Y in basic coordinates = sql_data.x[][2]
         *       Z in basic coordinates = sql_data.x[][3]
         */
        if ((rc != SQLITE_DONE) && (rc != SQLITE_OK)) {
            snprintf(T, ERR_MSG_SIZE,
                     " SQL error: rc=%d %s\n", rc, sql_data.err_msg);
            stkerr(" inertial_prop ", T);
            return 0;
        }
        offset  += SQL_BLOCK_SIZE;
        n_found  =  sql_data.nRows;
        n_remain = (n_found == SQL_BLOCK_SIZE ? 1 : 0);

        for (node = 0; node < n_found; node++) {
            dof = (sql_data.i[node][0]-1)*DOF_PER_NODE;
if (DEBUG > 1) gprintf("inertial_prop mass_diag node %2d [%3d,%3d,%3d] = % 12.5e  % 12.5e  % 12.5e\n", 
sql_data.i[node][0], dof, dof+1, dof+2, 
mass_diag[dof  ], mass_diag[dof+1], mass_diag[dof+2]);
            /* mass */
            Mx  += mass_diag[dof  ];
            My  += mass_diag[dof+1];
            Mz  += mass_diag[dof+2];
            /* center of mass */
            CMx += mass_diag[dof  ] * sql_data.x[node][1];
            CMy += mass_diag[dof+1] * sql_data.x[node][2];
            CMz += mass_diag[dof+2] * sql_data.x[node][3];
        }
    }
    if (sql_data.nRows*DOF_PER_NODE != nDOF) {
        snprintf(T, ERR_MSG_SIZE,
                 " size mismatch between SQL row count (%d*%d) and size "
                 "of input matrix (%d)",
                 sql_data.nRows, DOF_PER_NODE, nDOF);
        stkerr(" inertial_prop ", T);
        return 0;
    }
    drop();  /* ( dbh ) */

    db_close();

    if (!matstk(1, 6, "Mxyz")) return 0;
    Mxyz = tos->mat;
    Mxyz[0] =  Mx;
    Mxyz[1] =  My;
    Mxyz[2] =  Mz;
    Mxyz[3] = CMx/Mx;
    Mxyz[4] = CMy/My;
    Mxyz[5] = CMz/Mz;

    return 1;

} /* 1}}} */
