/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

/* tex.c  April 1999

Copyright (c) 1999  D. R. Williamson
*/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "main.h"
#include "stk.h"

#include "ctrl.h"
#include "exe.h"
#include "inpo.h"
#include "key.h"
#include "lib.h"
#include "mat.h"
#include "mem.h"
#include "sparse.h"
#include "sys.h"
#include "tag.h"
#include "term.h"
#include "tex.h"

int alike() /* alike (qS1 qS2 --- f) */
/* Returns true if strings S1 and S2 match, ignoring surrounding 
   blanks.  If volume, uses first row. */
{
   char *S1,*S2;
   int m,n1,n2,ret;

   if((tos->typ!=STR && tos->typ!=VOL) ||
      ((tos-1)->typ!=STR && (tos-1)->typ!=VOL)) {
      stkerr(" alike: ",STRSNOT2);
      return 0;
   }
   if((S2=(char *)memgetn(tos->tex,tos->col))==NULL) return 0;
   drop();
   n2=strspn(S2," "); /* offset to 1st char in S2*/
   m=strlen(S2)-1;
   while(m>-1 && isspace((int)*(S2+m))) m--;
   *(S2+m+1)='\0';
   
   if((S1=(char *)memgetn(tos->tex,tos->col))==NULL) return 0;
   drop();
   n1=strspn(S1," "); /* offset to 1st char in S1*/
   m=strlen(S1)-1;
   while(m>-1 && isspace((int)*(S1+m))) m--;
   *(S1+m+1)='\0';
   
   ret=strcmp(S1+n1,S2+n2);

   if(ret) ret=xFALSE;
   else ret=xTRUE;

   mallfree((void *)&S1);
   mallfree((void *)&S2);
   return(pushint(ret));
}  

/* A shorter version of alike() after strchop() and strmatch1() were
   written (probably not as fast as the original):

   int alike() */ /* alike (qS1 qS2 --- f) */
/* Returns true if strings S1 and S2 match, ignoring surrounding 
   blanks. */
/*
{
   int f;

   if((tos->typ!=STR && tos->typ!=VOL) ||
      ((tos-1)->typ!=STR && (tos-1)->typ!=VOL)) {
      stkerr(" alike: ",STRSNOT2);
      return 0;
   }
   strchop();
   swap();
   strchop();
   if(!strmatch1()) return 0;

   popint(&f);
   if(f) return(pushint(xFALSE));
   return(pushint(xTRUE));
}
*/

int asciify() /* asciify (hT --- hT1) */
/* Replacing characters less than blank with blank. */
{
   register unsigned char *T,*T1;
   register int bytes,i=0,rows;
   int isquote,ret=0;
   char *name="_asciify";

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" asciify: ",STRORVOLNOT);
      return 0;
   }
   isquote=(tos->typ==STR);
   hand();

   T=(unsigned char *)tos->tex;
   bytes=tos->col;
   rows=tos->row;
   
   if((T1=(unsigned char *)memgetc(rows,bytes))==NULL) {
      stkerr(" asciify: ", MEMNOT);
      return 0;
   }
   for(;i<rows*bytes;i++) *(T1+i)=MAX(*(T+i),(unsigned char)' ');

   ret=(
      drop() &&
      push(VOL,(char *)memgetn(name,strlen(name)),NOTAG,0,NULL, \
         (char *)T1,rows,bytes,NULL)
   );
   if(isquote) {
      ret=(ret && pushint(XBASE) && quote() &&
          pushq2(name,strlen(name)) && naming()
      );
   }
   return(ret);
}

int asciiload() /* asciiload (qFile --- hT) */
/* Loads text file, qFile, and puts volume on the stack. */
{
   char *s,*filename;

   if(!unquoted()) return 0;

   if((filename=peekq())==NULL) {
      stkerr(" asciiload: ",FILNOT);
      drop();
      pushint(0); pushint(0); blockofblanks(); /* empty VOL */
      return 0;
   }
   if((s=fileload(filename))==NULL) {
      stkerr(" asciiload: ",FILERR);
      drop();
      pushint(0); pushint(0); blockofblanks(); /* empty VOL */
      return 0;
   }
   return(
      pushq(s,strlen(s)) &&
      textget(0) &&
      pushstr("_") &&
      rot() &&
      cat() &&
      naming()
   );
}

int asciiread() /* asciiread (qFile n --- hA) */
/* Reads n numbers per row from ascii file and builds numerical
   matrix A. */
{
   unsigned long n;

   if(!popuint(&n)) {
      stkerr(" asciiread: ",NUMNOT);
      return 0;
   }
   if(!asciiload()) {
      stkerr(" asciiread: ",FILERR);
      return 0;
   }
   return(
      asciify() &&
      pushint(n) &&
      matread()
   );
}

char *bitpat(unsigned char c)
/* Converting byte c into an 8 character bit pattern for display. */
{
   static unsigned char buf[9]="        \0";
   unsigned char c1[8]={'0','0','0','0','0','0','0','0'};
   unsigned char mask[8]={128,64,32,16,8,4,2,1};
   register int i=0;

   for(;i<8;i++) if(c&*(mask+i)) *(c1+i)='1';
   memcpy(buf,c1,8);
   return (char *)buf;
}

int blpad() /* blpad (hT n --- hT1) */
/* Widen all rows of VOL T to n characters.  If T is already wider 
   than n characters, nothing is done. */ 
{
   char *name="_blpad",*T=NULL,*T1;
   int chars=0,i=0,n,rows=0;

   if((tos-1)->typ==STR) {
      if(!popint(&n)) return 0;

      if(n<tos->col) return(cop());

      T1=memgetn1(tos->tex,tos->col,n);
      return(
         push(STR,(char *)memgetn(name,strlen(name)),NOTAG,0,NULL,T1,\
            1,n,NULL) &&
         lop()
      );
   }
   if((tos-1)->typ!=VOL) {
      stkerr(" blpad: ",STRORVOLNOT);
      return 0;
   }
   if(!popint(&n)) return 0;

   if(n<tos->col) return(cop());

   T=tos->tex;
   chars=tos->col;
   rows=tos->row;

   if(!volstk(rows,n,name)) return 0;
   T1=tos->tex;

   for(;i<rows;i++) {
      memcpy(T1,T,MIN(n,chars));
      memset(T1+chars,' ',MAX(0,n-chars));
      T+=chars;
      T1+=n;
   }
   return(lop());
}

int cat() /* cat (qA qB --- qC) */
/* Concatenates strings A and B (or row-compatible volumes A and B) 
   into C. */
{
   register char *qA,*qB,*qC;
   stkitem *Avol,*Bvol;
   register int rA,cA,cB,cC,len,i=0;

   if((tos-1)->typ==STR && tos->typ==STR) {
      qA=(tos-1)->tex; qB=tos->tex;
      cA=(tos-1)->col; cB=tos->col;
      if((qC=(char *)memgetc(1,(cC=(cA+cB))))==NULL) return 0;
      memcpy(qC,qA,cA);
      memcpy((qC+cA),qB,cB);
      return(
         drop2() &&
         push(STR,(char *)memgetn("_cat",4),NOTAG,0,NULL,qC,1,cC,NULL)
      );
   }
   if((tos-1)->typ==VOL && tos->typ==VOL) {
      Avol=tos-1; Bvol=tos;
      qA=Avol->tex; cA=Avol->col; rA=Avol->row;
      qB=Bvol->tex; cB=Bvol->col;
      if(rA!=Bvol->row) { 
         stkerr(" cat: ",VOLSNOTC); return 0;
      }
      if((qC=(char *)memgetc(rA,(cC=cA+cB)))==NULL) return 0;
      for(;i<rA;i++) { 
         len=loclin(i,cC);
         memcpy(qC+len,qA+loclin(i,cA),cA);
         memcpy(qC+len+cA,qB+loclin(i,cB),cB);
      } 
      return(
         drop2() &&
         push(VOL,(char *)memgetn("_cat",4),NOTAG,0,NULL,qC,rA,cC,NULL)
      );
   }
   if((tos-1)->typ==STR && tos->typ==VOL) {
   /* Fix stack and reenter: */
      return(
         swap() && 
         typstr2vol() && 
         swap() && 
         cat() &&
         pushint(XBASE) &&
         quote() /* first row of VOL */
      );
   }
   if((tos-1)->typ==VOL && tos->typ==STR) {
      return(
         typstr2vol() && 
         cat() &&
         pushint(XBASE) &&
         quote() /* first row of VOL */
      );
   }
   stkerr(" cat: ",STKNOT); return 0;
}

int chblank() /* chblank (hT qS --- hT1) */
/* Returns T1 with any of the characters from T that match the 
   characters listed in S replaced by blanks. */
{
   char *S,*S1,*T,*T1;
   register int i,k=0,l;
   int chars=0,chars_S=0,rows;

   if(tos->typ!=STR) {
      stkerr(" chblank: ",STRNOT);
      return 0;
   }
   if((tos-1)->typ!=STR && (tos-1)->typ!=VOL) {
      stkerr(" chblank: ",STRORVOLNOT);
      return 0;
   }
   strchop();

   S=tos->tex;
   chars_S=tos->col;
   if(!chars_S) return(drop() && cop());

   T=(tos-1)->tex;
   chars=(tos-1)->col;
   rows=(tos-1)->row;
   if(!chars || !rows) return(drop() && cop());

   if((S=(char *)malloc(1+chars_S))==NULL) return 0;
   memcpy(S,tos->tex,chars_S);
   *(S+chars_S)='\0';

   if(!volstk(rows,chars,"_chblank")) return 0;
   T1=tos->tex;
   memset(T1,' ',rows*chars);

   if((S1=(char *)malloc(1+chars))==NULL) return 0;
   *(S1+chars)='\0';

   for(;k<rows;k++) {
      memset(S1,' ',chars);
      for(i=0;i<chars;i++) {
         l=strspn((T+i),S);
         i+=l;
         if(i<chars) *(S1+i)=*(T+i);
      }
      memcpy(T1,S1,chars);
      T+=chars;
      T1+=chars;
   }
   mallfree((void *)&S);
   mallfree((void *)&S1);

   if((tos-2)->typ==STR) tos->typ=STR;

   return(lop() && lop());
}

int chkeep() /* chkeep (hT qS --- hT1) */
/* Returns T1 with only the characters from T that match the
   characters listed in S; all others are replaced by blanks. */
{
   char *S,*S1,*T,*T1;
   register int k=0,l;
   int chars,rows;

   if(tos->typ!=STR) {
      stkerr(" chkeep: ",STRNOT);
      return 0;
   }
   if((tos-1)->typ!=STR && (tos-1)->typ!=VOL) {
      stkerr(" chkeep: ",STRORVOLNOT);
      return 0;
   }
   strchop();

   S=tos->tex;
   chars=tos->col;

   if((S=(char *)malloc(2+chars))==NULL) return 0;
   *S=' ';
   memcpy(S+1,tos->tex,chars);
   *(S+1+chars)='\0';

   T=(tos-1)->tex;
   chars=(tos-1)->col;
   rows=(tos-1)->row;

   if(!volstk(rows,chars,"_chkeep")) return 0;
   T1=tos->tex;
   memset(T1,' ',rows*chars);

   if((S1=(char *)malloc(1+chars))==NULL) return 0;
   *(S1+chars)='\0';

   for(;k<rows;k++) {
      memcpy(S1,T,chars);
      while((l=strspn(S1,S))<chars) *(S1+l)=' ';
      memcpy(T1,S1,chars);
      T+=chars;
      T1+=chars;
   }
   mallfree((void *)&S);
   mallfree((void *)&S1);

   if((tos-2)->typ==STR) tos->typ=STR;

   return(lop() && lop());
}

void chrrpl(char *T, int len, char C, char C1) 
/* In T, replace instances of char C with char C1. */
{
   int k=0;

   for(;k<len;k++) {
      if(*T==C) *T=C1;
      T++;
   }
}

enum dispform {DEC=101,INT,HEX,tBIN,PURG,UINT,STDOUT};

int display(int dispform) /* ., .bin, .hex, .i .u (hT --- )  */
/* Displays a stack item. */

/* This function tries to do too much.  It is way too big and complex.
   The wish now is that it won't ever need to be touched. */
{
   unsigned long (**p)();
   register stkitem *stk;
   char *buf,*s,*tag;
   register double *d,x;
   register int i=0,j=0,lines,width,rows,cols;
   int k,perptr=8;
   long n;
   unsigned long u;
   union { 
      double x; 
      unsigned char c[sizeof(double)]; 
   } xc={0};
   long long q;
   const int per=6,perh=4,perb=1,perint=10,perptr32=8,peruint=6;
   const int pertBIN=8,perHEX=24,perptr64=7;

   const int maxmat=8000;  /* max matrix terms */
   const int maxvol=8000;  /* max volume rows */
   const int maxstr=64000; /* max string bytes */

   #define TLEN 81
   char tbuf[TLEN];

   #define cformat0 "%c"
   #define eformat0 "%10.4E"
   #define eformat1 " %10.4E"
   #define eformat2 "  %10.4E"
   #define iformat0 "%7d"
   #define iformat1 "% 7d"
   #define sformat0 "%s"
   #define sformat1 " %s"
   #define uformat1 "% 11u"
   #define xformat0 "%02X"
   #define xformat1 " %02X"

   if((stk=(peek()))==NULL) {
      stkerr(" display: ",EMPTYSTK);
      return 0;
   }
   switch(stk->typ) {

      default:
         gprintf(" display: stack type %d not supported",stk->typ);
         nc();
         stkerr("","");
         return 0;
      break;

      case MAT:
         rows=stk->row;
         cols=stk->col;
         if(rows*cols>maxmat) {
            gprintf(" matrix is too big to display");
            break;
         }
         if(!rows || !cols) dispform=PURG;
         d=stk->mat;
         j=0; 
         switch(dispform) {
         default:
         case DEC:   
            while(j<cols) {
               if(j!=0) nc();
               gprintf(" Column %d:",j+XBASE); nc();
               k=locvec(j,rows);
               /* using procedure to not print -0 */
               if((x=*(d+k))>=0) gprintf(eformat2,fabs(x)); 
               else gprintf(eformat1,x); 
               i=1;
               while(i<rows) {
                  if(!(i%per)) nc();
                  if((x=*(d+k+i))>=0) gprintf(eformat2,fabs(x)); 
                  else gprintf(eformat1,x); 
                  i++;
               }
               j++;
            }
         break;
         case INT:
            while(j<cols) {
               if(j!=0) nc();
               gprintf(" Column %d:",j+XBASE); nc();
               k=locvec(j,rows);
               x=*(d+k); 
               if(fabs(x)>2147483647) {
                  s=mprintf("%30.0f",x);
                  if(!s) return 0;
                  n=strspn(s," ");
                  if(x>=0) gprintf("  %s",s+n);
                  else gprintf(" %s",s+n);
                  mallfree((void *)&s);
               }
               else {
                  n=x;
                  if(n>=0) gprintf(iformat1,ABS(n));
                  else gprintf(iformat0,n);
               }
               i=1;
               while(i<rows) {
                  if(!(i%perint)) nc();
                  x=*(d+k+i); 
                  if(fabs(x)>2147483647) {
                     s=mprintf("%30.0f",x);
                     if(!s) return 0;
                     n=strspn(s," ");
                     if(x>=0) gprintf(" %s",s+n);
                     else gprintf("%s",s+n);
                     mallfree((void *)&s);
                  }
                  else {
                     n=x;
                     if(n>=0) gprintf(iformat1,ABS(n));
                     else gprintf(iformat0,n);
                  }
                  i++;
               }
               j++;
            }
         break;
         case UINT:
            while(j<cols) {
               if(j!=0) nc();
               gprintf(" Column %d:",j+XBASE); nc();
               k=locvec(j,rows);
               x=fabs(*(d+k));
               if(x>2147483647) {
                  s=mprintf("%30.0f",x);
                  if(!s) return 0;
                  n=strspn(s," ");
                  gprintf(" %s",s+n);
                  mallfree((void *)&s);
               }
               else {
                  pushd(*(d+k));
                  popuint((unsigned long *)&u);
                  gprintf(uformat1,u);
               }
               i=1;
               while(i<rows) {
                  if(!(i%peruint)) nc();
                  x=fabs(*(d+k+i));
                  if(x>2147483647) {
                     s=mprintf("%30.0f",x);
                     if(!s) return 0;
                     n=strspn(s," ");
                     gprintf(" %s",s+n);
                     mallfree((void *)&s);
                  }
                  else {
                     pushd(*(d+k+i));
                     popuint((unsigned long *)&u);
                     gprintf(uformat1,u);
                  }
                  i++;
               }
               j++;
            }
         break;
         case HEX:
            while(j<cols) {
               if(j!=0) nc();
               gprintf(" Column %d:",j+XBASE); nc();
               sp();
               xc.x=*(d+locvec(j,rows));
               for(k=0;k<sizeof(double);k++) 
                  gprintf(xformat0,*(xc.c+k));
               i=1;
               while(i<rows) {
                  if(!(i%perh)) nc();
                  sp();
                  xc.x=*(d+i+locvec(j,rows));
                  for(k=0;k<sizeof(double);k++) 
                     gprintf(xformat0,*(xc.c+k));
                  i++;
               }
               j++;
            }
         break;
         case tBIN:
            while(j<cols) {
               if(j!=0) nc();
               gprintf(" Column %d:",j+XBASE); nc();
               xc.x=*(d+locvec(j,rows));
               for(k=0;k<sizeof(double);k++) 
                  gprintf(sformat1,bitpat(*(xc.c+k)));
               i=1;
               while(i<rows) {
                  if(!(i%perb)) nc();
                  xc.x=*(d+i+locvec(j,rows));
                  for(k=0;k<sizeof(double);k++) 
                     gprintf(sformat1,bitpat(*(xc.c+k)));
                  i++;
               }
               j++;
            }
         break;
         case PURG:
            if(cols==1) {
               gprintf(" vector is purged");
            }
            else {
               gprintf(" matrix is purged"); 
            } 
         break;
         }

      drop();
      return 1;
      break;

      case NUM:
      if(is_complex(tos)) {
         cmplxdbl();
         swap();
         display(dispform);

         if(dispform!=tBIN) {
            pushstr(",");
            dot();
         }
         return(display(dispform));
      }
      switch(dispform) {
         default:
         case DEC:   
            popd(&xc.x); gprintf(eformat1,xc.x); 
         break;
         case INT:
            popd(&xc.x);
            q=xc.x;
            if(ABS(q)>2147483647) {
               s=mprintf("%30.0f",xc.x);
               if(!s) return 0;
               k=strspn(s," ");
               gprintf(" %s",s+k); /* space for + sign */
               mallfree((void *)&s);
            }
            else {
               k=q;
               gprintf(" %d",k); /* space for + sign */
            }
         break;
         case UINT:
            popd(&xc.x);
            x=fabs(xc.x);
            if(x>2147483647) {
               s=mprintf("%30.0f",x);
               if(!s) return 0;
               n=strspn(s," ");
               gprintf("%s",s+n); /* no space for sign */
               mallfree((void *)&s);
            }
            else {
               pushd(xc.x);
               popuint((unsigned long *)&u);
               gprintf("%u",u); /* no space for sign */
            }
         break;
         case HEX:
            popd(&xc.x); 
            for(i=0;i<sizeof(double);i++) 
               gprintf(xformat1,*(xc.c+i));
         break;
         case tBIN:
            popd(&xc.x); 
            for(i=0;i<sizeof(double);i++) 
               gprintf(sformat1,bitpat(*(xc.c+i)));
         break;
      }
      return 1;
      break;

      case PTR:
      p=(unsigned long (**)())stk->ptr;
      rows=stk->row;
      if(!rows) dispform=PURG;

      if(sizeof(long)==4) perptr=perptr32;
      else perptr=perptr64;

      switch(dispform) {
         default:
         case HEX:
            i=0; 
            while(i<rows) {
               if(i>0 && !(i%perptr)) nc();
                  gprintf(" %lX",*(p+i));
               i++;
            }
         break;
         case PURG:
            gprintf(" vector has no pointers"); nc();
            drop();
         break;
      }
      popptr(&n,&tag);
      return 1;
      break;

      case STR:
      s=stk->tex;
      width=stk->col;
      if(!width) dispform=PURG;
      if(width>maxstr) {
         gprintf(" string is too big to display");
      }
      else {
         i=0;
         switch(dispform) {
            default:
            case DEC:
               while(i<width) {
                  gprintf(cformat0,(unsigned char)*(s+i)); 
                  i++;
               }
            break;

            case tBIN:
               while(i<width) {
                  gprintf(sformat1,bitpat(*(s+i))); 
                  i++; 
                  if(!(i%pertBIN)) nc();
               }
            break;

            case HEX:
               while(i<width) {
                  gprintf(xformat1,*(s+i)); 
                  i++; 
                  if(!(i%perHEX)) nc();
               }
            break;

            case PURG:
               /* Skipping this message: allowing empty strings
                  to print nothing, i.e. "" . is ok.
               gprintf(" string has no characters");
               */
            break;
         }
      }
      drop();
      return 1;
      break;

      case VOL:
         if(is_sparse(tos)) {
            rows=stk->row;
            cols=stk->col;
            if(rows*cols>maxmat) {
               gprintf(" sparse matrix is too big to display");
               break;
            }
            return(
               dense() &&
               dot()
            );
         }
         if(tos->row==1) {
            textget0();
            notrailing();
         }
         s=stk->tex;
         lines=stk->row;
         width=stk->col;
         if(!lines || !width) {
            gprintf(" volume is empty");
            drop();
         }
         else {
            if(lines>1) { /* showing text when more than one line: */
               if(lines>maxvol) {
                  gprintf(" volume is too big to display");
                  break;
               }
               else {
                  if(dispform==HEX || dispform==tBIN) {
                  gprintf(
                     " display: no hex or bin for multi-line volume");
                     nc();
                  }
                  if((buf=(char *)malloc(width+1))==NULL) {
                     stkerr(" display: ",MEMNOT);
                     return 0;
                  }
                  k=MAX(1,strlen1(s,width));
                  memcpy(buf,s,k); *(buf+k)='\0';
                  gprintf(sformat0,buf);
                  i=1;
                  while(i<lines) {
                     j=loclin(i,width);
                     k=MAX(1,strlen1(s+j,width));
                     memcpy(buf,s+j,k); *(buf+k)='\0';
                     nc();
                     gprintf(sformat0,buf);
                     i++;
                  } mallfree((void *)&buf);
               }
            } 
            else { /* displaying single line */
               pushint(XBASE); quote(); stk=peek();
               s=stk->tex;
               width=stk->col;
               switch(dispform) {
                  default:
                     case DEC:
                        if(width<TLEN) {
                        memcpy(tbuf,s,width);
                        *(tbuf+width)='\0';
                        gprintf(sformat0,tbuf);
                     }
                     else {
                        i=0; while(i<width) {
                        gprintf(cformat0,(unsigned char)*(s+i)); i++; }
                     }
                  break;
                  case tBIN:
                     i=0; 
                     while(i<width) {
                        gprintf(sformat1,bitpat(*(s+i))); 
                        i++; 
                        if(!(i%pertBIN)) nc();
                     }
                  break;
                  case HEX:
                     i=0; while(i<width) {
                        gprintf(xformat1,*(s+i)); i++; }
                  break;
            }
         }
         drop();
         return 1;
      break;
      }
   } 
   return 1;

   #undef TLEN
   #undef cformat0
   #undef eformat0
   #undef eformat1
   #undef eformat2
   #undef iformat0
   #undef iformat1
   #undef sformat0
   #undef sformat1
   #undef uformat1
   #undef xformat0
   #undef xformat1
}

int dot() { return display(DEC); } /* . (hA --- ) or dot (hA --- ) */

int dotbin() /* .bin (hA --- ) */
{
   int i=0,k,ret=1;

   if(tos->typ==VOL) {
      for(;i<(k=tos->row);i++) {
         dup1s();
         pushint(i+XBASE);
         quote();
         ret=(ret && display(tBIN));
         if((1+i)<k) nc();
      }
      return(ret && drop());
   }
   else return display(tBIN); 
} 

int dothex() /* .hex (hA --- ) */
{
   int i=0,k,ret=1;

   if(tos->typ==VOL) {
      for(;i<(k=tos->row);i++) {
         dup1s();
         pushint(i+XBASE);
         quote();
         ret=(ret && display(HEX));
         if((1+i)<k) nc();
      }
      return(ret && drop());
   }
   else return display(HEX); 
} 

int dotint() { return display(INT); } /* .i (hA --- ) */

int dotpair() /* dotpair (qA.B --- qA qB) */
/* Separate string A.B into strings A and B.  Returned A is an empty
   string if there is no A or no dot in A.B.  If A or dot are present,
   string B cannot be an empty string.

   This function is (better than and) nearly equivalent the following
   word:
   inline: dotpair (qA.B --- qA qB) \ string A.B into strings A and B
      "." " " replace$ strings dup 1st quote strchop swap
      dup rows one > IF 2nd quote strchop ELSE drop "" swap THEN
   end
*/
{
   char *S;
   int chars,ret=0;
   register int i=0;

   if(tos->typ!=STR) {
      stkerr(" dotpair: ",STRNOT);
      return 0;
   }
   strchop();
   if(!(chars=tos->col)) {
      return(
         drop() &&
         pushstr("") &&
         dup1s()
      );
   }
   if(!strstk(chars,"_d")) return 0;
   memcpy(tos->tex,(tos-1)->tex,chars);
   S=tos->tex;
   lop();

   while(i<chars && *(S+i)!='.') i++;

   if(i>chars-2) {
      if(i==chars-1) {
         stkerr(" dotpair: ","a second string must follow dot");
         return 0;
      }
      stkerr(" dotpair: ","no dot in pattern");
      return 0;
   }
   *(S+i)='\x20';

   ret=(
      strings() &&
      dup1s() &&
      pushint(XBASE) &&
      quote() && /* string A */
      strchop() &&
      swap()
   );
   if(tos->row>1) {
      return(
         ret &&
         pushint(XBASE+1) &&
         quote() && /* string B */
         strchop()
      );
   }
   return(
      ret &&
      drop() &&
      pushstr("")  && /* string A */
      swap() /* string B */
   );
}

int dotstr() /* dotstr (qS --- ) */
/* Fast output directly to stdout socket. */
{
   int len;

   if(Out>-1) {
      fflush(stdout);
      len=write(Out,tos->tex,tos->col);
      if(len!=tos->col) {
         stkerr(" dotstr: ","error writing all bytes");
         drop();
         return 0;
      }
      return(drop());
   }
   else 
      return display(DEC); 
} 

int dotuint() { return display(UINT); } /* .u (hA --- ) */

int format() /* format (hA qS --- hT) */
/* Formats every row of A according to format S, to create volume T. */
{
   char *name="_format";
   char *form,*L,*T=NULL;
   register int cols,j=0,rows,width=0;
   int ret=0;
   const int pad=32; /* width = pad plus width of first line */
   const int maxcols=64;
   double *A,B[64] = { 0, 0, 0, 0, 0, 0, 0, 0,
                       0, 0, 0, 0, 0, 0, 0, 0,
                       0, 0, 0, 0, 0, 0, 0, 0,
                       0, 0, 0, 0, 0, 0, 0, 0,
                       0, 0, 0, 0, 0, 0, 0, 0,
                       0, 0, 0, 0, 0, 0, 0, 0,
                       0, 0, 0, 0, 0, 0, 0, 0,
                       0, 0, 0, 0, 0, 0, 0, 0 };

   if(tos->typ!=STR) {
      stkerr(" format: ",STRNOT);
      return 0;
   }
   if(is_complex(tos-1)) {
      stkerr(" format: ",REALNOT);
      return 0;    
   }
/* This test avoids seg faults on bad formats: */
   while(!isalpha(*(tos->tex+j)) && j<tos->col) j++;
   if(j==tos->col) {
      stkerr(" format: ","no alpha specifier in format string");
      return 0;
   }
   j=0;

   if((form=(char *)memgetn(tos->tex,tos->col))==NULL) return 0;
   drop(); 
   hand();

   if(tos->typ==MAT) {
      if(tos->col>maxcols) {
         stkerr(" format: rows to format have more than 64 columns", \
            "");
         mallfree((void *)&form);
         return 0;
      }
      ret=bend();
      if(!ret) {
         mallfree((void *)&form);
         return 0;
      }
      cols=tos->col;
      rows=tos->row;
      A=tos->mat;

      for(;j<cols;j++) { /* copy to B in case A is smaller than 63 */

      /* Copying to B: if A < 64, using it in call to mprintf can give
         a seg fault. */
         memcpy(B,A,rows*sizeof(double));

         L=(char *)mprintf(form,*B,*(B+1),*(B+2),*(B+3),*(B+4),*(B+5),  
            *(B+6), *(B+7), *(B+8), *(B+9), *(B+10),*(B+11),*(B+12),    
            *(B+13),*(B+14),*(B+15),*(B+16),*(B+17),*(B+18),*(B+19),    
            *(B+20),*(B+21),*(B+22),*(B+23),*(B+24),*(B+25),*(B+26),    
            *(B+27),*(B+28),*(B+29),*(B+30),*(B+31),*(B+32),*(B+33),    
            *(B+34),*(B+35),*(B+36),*(B+37),*(B+38),*(B+39),*(B+40),    
            *(B+41),*(B+42),*(B+43),*(B+44),*(B+45),*(B+46),*(B+47),    
            *(B+48),*(B+49),*(B+50),*(B+51),*(B+52),*(B+53),*(B+54),    
            *(B+55),*(B+56),*(B+57),*(B+58),*(B+59),*(B+60),*(B+61),    
            *(B+62),*(B+63));

         if(!L) {
            mallfree((void *)&form);
            return 0;
         }
         if(j==0) { /* using first line to set width: */
            width=pad+strlen(L);
            if(!volstk(cols,width,name)) return 0;
            T=tos->tex;
            memset(T,' ',cols*width);
         }
         memcpy(T,L,MIN(strlen(L),width));
         mallfree((void *)&L);
         T+=width;
         A+=rows;
      }
      mallfree((void *)&form);

      if(tos->row==1) typvol2str(); /* return 1-row VOL as STR */

      return(
         lop() &&
         notrailing() &&
         pushq2(name,strlen(name)) &&
         naming()
      );
   }
   if(tos->typ==VOL) {
      stkerr(" format: VOL type not supported yet","");
      mallfree((void *)&form);
      return 0;
   }
   stkerr(" format: ",NOTSUPT);
   mallfree((void *)&form);
   return 0;
}

int grepe() /* grepe (hT qS --- hRows) */
/* List of row numbers of lines that exactly match string S.  Leading 
   and trailing blanks are ignored when matching. */
{
   char *s,*S,*T;
   register double *R;
   register int k=0,n=0;
   int chars,rows;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" grepe: ",STRORVOLNOT);
      return 0;
   }
   if((tos-1)->typ!=STR && (tos-1)->typ!=VOL) {
      stkerr(" grepe: ",STRORVOLNOT);
      return 0;
   }
   if((S=(char *)memgetn(tos->tex,tos->col))==NULL) return 0;
   strchop1(S);

   T=(tos-1)->tex;
   rows=(tos-1)->row;
   chars=(tos-1)->col;

   if(!matstk(rows,1,"_grepe")) {
      mallfree((void *)&S);
      return 0;
   }
   R=tos->mat;

   if((s=(char *)malloc(1+chars))==NULL) {
      mallfree((void *)&S);
      return 0;
   }
   *(s+chars)='\0';

   for(;k<rows;k++) {
      memcpy(s,T,chars);
      strchop1(s);

      if(strmatch(s,S)) {
         *(R+n)=k+XBASE;
         n++;
      }
      T+=chars;
   }
   mallfree((void *)&S);
   mallfree((void *)&s);
   tos->row=n;

   return(lop() && lop());
}

int grepr() /* grepr (hT qS --- hRows) */
/* List of row numbers of lines that contain string S somewhere. */
{
   unsigned char *p,*s,*S,*T;
   register double *R;
   register int k=0,n=0;
   int chars,rows;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" grepr: ",STRORVOLNOT);
      return 0;
   }
   if((tos-1)->typ!=STR && (tos-1)->typ!=VOL) {
      stkerr(" grepr: ",STRORVOLNOT);
      return 0;
   }
   S=(unsigned char *)tos->tex;
   T=(unsigned char *)(tos-1)->tex;
   rows=(tos-1)->row;
   chars=(tos-1)->col;

   if(!matstk(rows,1,"_grepr")) return 0;
   R=tos->mat;

   if((s=(unsigned char *)malloc(2+chars))==NULL) return 0;
   *(s+chars)=' ';
   *(s+1+chars)='\0';

   for(;k<rows;k++) {
      memcpy(s,T,chars);
      p=(unsigned char *)strstr((char *)s,(char *)S);
      if(p!=NULL) {
         *(R+n)=k+XBASE;
         n++;
      }
      T+=chars;
   }
   mallfree((void *)&s);
   tos->row=n;

   return(lop() && lop());
}

int hexbytes() /* hexbytes (qS --- qT) */
/* Convert hex byte strings within S into binary bytes.

   inline: hexbytes (qS --- hT) \ hex byte strings into binary bytes
{     Examples:

         "FF FB 18 FF FB 20 FF FB 23 FF FB 27" hexbytes (hT)
         (hT) uimport1 (hA) .u

         "FF FB 01 52 65 64 20 48 61 74 20 4C 69 6E 75 78" hexbytes (hT)
         (hT) INF xray (hT1) .
}
      hexuint export1
   end
*/
{
   return(
      qhexuint() &&
      export1()
   );
}

int html2text() /* html2text (hT or qS1 --- qS) */
/* Removes HTML strings from the VOL or STR on the stack, and does
   no formatting of the text that remains. */
{
   char *H,*T,*name="_html2text";
   int BL=32,count=0,j=0,k=0;
   char c;
 
   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" html2text: ",STRORVOLNOT);
      return 0;
   }
   textput();
   H=tos->tex;

   if(!strstk(tos->col,"_T")) return 0;
   T=tos->tex;

   while(k<tos->col) {
      c=*(H+k);
      k++;
      if(c=='<' && *(H+k)>BL) count++;

      if(!count) {
         *(T+j)=c;
         j++;
      }
      if(c=='>' && count)
      count--;
   }
   T=memgetn(tos->tex,j-1);

   return(
      push(STR,(char *)memgetn(name,strlen(name)),NOTAG,0,NULL,T, \
         1,j-1,NULL) &&
      lop() &&
      lop()
   );
}

int indent() /* indent (hT n --- hT1) */
/* Indenting string or volume n spaces.  Negative n moves left. */
{
   int isq,n,m,ret=0;

   isq=((tos-1)->typ==STR);

   if(!(popint(&n) && hand())) return 0;

   if(!n) return 1;

   if(n>0) { /* push right */
      ret=(
         pushint(tos->row) &&
         pushint(n) &&
         blockofblanks() &&
         swap() &&
         park()
      );
   }
   else { /* pull left */
      n=ABS(n);
      m=MAX(0,tos->col-n);
      ret=(
         pushint(n+XBASE) && /* starting index */
         pushint(m) && /* number of items */
         items() &&
         catch()
      );
   }
   if(isq) ret=(ret && pushint(XBASE) && quote());

   return(ret &&
      pushstr("_indent") && naming()
   );
}

int intstr() /* _int$ (x --- qS) intstr (x --- qS) */
/* Converting an integer to a text string. */
{
   double d;
   const double bigd=9007199254740992.;
   long long n;
   int i=0,len=0;

   char q[361];

   if(is_complex(tos)) {
      stkerr(" _int$: ",REALNOT);
      return 0;    
   }
   if(!popd(&d)) return 0;

   if(!(fabs(d)>bigd)) { /* value above bigd its just for fun */
      n=(long long)d;
      d=n;
   }
   sprintf(q,"%-360.0f",d);

   while(i<360 && len<360) {
      if(*(q+i)>0x20) {
         len++;
         i++;
      }
      else break;
   }
   return(pushq2(q,len));
}

int justify() /* justify (hT f --- hT1) */
/* Justifying text in T.  When f is true, text is right justified to
   the width of its widest line; when f is false, text is left justi-
   fied. */
{
   register char *T,*T1;
   register unsigned int bytes,i=0,k=0,rows;
   int f,isq,isqr,retr;

   if(!(tos-1)->col) { /* return if nothing to justify */
      drop();
      return 1;
   }
   isq=((tos-1)->typ==STR); /* 1 if have STR */

   if(!(popint(&f) && hand())) return 0;

   if(tos->typ!=VOL) {
      stkerr(" justify: ",STRORVOLNOT);
      return 0;
   }
   if(f) { /* right justify */
      isqr=isq; /* making safe for reentry */
      retr=asciify();
      bytes=tos->col;
      retr=(retr &&
         pushint(XBASE) && pushint(bytes) && items() &&

         reversed() && dup1s() && rev() &&

         catch() &&                  /* reversing characters */
         pushint(!f) && justify() && /* then left justifying them */
         swap() && catch()           /* and reversing them back */
      );
      if(isqr)
         return(retr &&
            pushint(XBASE) && quote() &&
            pushstr("_justify") && naming()
         );
      else
         return(retr &&
            pushstr("_justify") && naming()
         );
   }
   else { /* left justify */
      T=(char *)tos->tex;
      rows=tos->row;
      bytes=tos->col;
      if((T1=(char *)memgetc(rows,bytes))==NULL) {
         stkerr(" justify: ", MEMNOT);
         return 0;
      }
      for(;i<rows;i++) {

         k=MIN((bytes-1),(strspn(T+loclin(i,bytes)," ")));

         memcpy(T1+loclin(i,bytes),T+loclin(i,bytes)+k,bytes-k);
      }
      drop();
      if(isq)
         return(
            push(STR,(char *)memgetn("_justify",8),NOTAG,0,NULL,T1, \
               1,bytes,NULL)
         );
      else
         return(
            push(VOL,(char *)memgetn("_justify",8),NOTAG,0,NULL,T1, \
               rows,bytes,NULL)
         );
   }
}

int longhex() /* longhex (n --- qS) */
/* The eight bytes in n are binary long int.  Return their hex pattern
   in string S. */
{
   char *fX="%lX",name[18]={0};

   if(tos->typ!=NUM) {
      stkerr(" longhex: ",NUMNOT);
      return 0;
   }
   sprintf(name,fX,tos->real);

   return(
      drop() &&
      pushstr(name)
   );
}

int lowercase() /* lowercase (qS | hT --- qS1 | hT1) */
{
   register int cols,i=0,j=0,rows;
   register char *p,*pi;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" lowercase: ",STRORVOLNOT);
      return 0;
   }
   rows=tos->row;
   cols=tos->col;

   if((p=(char *)memgetn(tos->tex,rows*cols))==NULL) {
      stkerr(" lowercase: ",MEMNOT);
      return 0;
   }
   for(;i<rows;i++) {
      pi=p+loclin(i,cols);
      for(j=0;j<cols;j++) {
         *(pi+j)=tolower(*(pi+j));
      }
   }
   if(tos->typ==STR)
      return(
         drop() &&
         pushq(p,cols) &&
         pushstr("_lowercase") && naming()
      );
   else
      return(
         drop() &&
         push(VOL,(char *)memgetn("_lowercase",10),NOTAG,0,NULL,p, \
            rows,cols,NULL)
      );
}

int matread() /* matread (hT n --- hA) */
/* Reads n numbers per row from volume, skipping text, and builds
   numerical matrix A.  If there are fewer than n numbers in a row, 
   the row is skipped. */
{
   char *T,*T1,*p,*q;
   register double *A,*Rake;
   double x;
   register unsigned int cols,i=0,j=0,len,rows,wid;
   unsigned long n;

   if(!popuint(&n)) {
      stkerr(" matread: ",NUMNOT);
      return 0;
   }
   if(!hand() || tos->typ!=VOL) {
      stkerr(" matread: ",STRORVOLNOT);
      return 0;
   }
   T=tos->tex;
   wid=tos->col;
   rows=tos->row;
   cols=n;

   if((q=(char *)memgetc(1,wid))==NULL ||
      (!matstk(rows,cols,"_matread")) ||
      (!matstk(rows,1,"_rake"))) {
         stkerr(" matread: ",MEMNOT);
         return 0;
   }
   A=(tos-1)->mat;
   Rake=tos->mat;

   for(;i<rows;i++) {
      j=0;
      T1=T+loclin(i,wid);
      memcpy(q,T1,(len=strlen1(T1,wid)));
      *(q+len)='\0'; /* without 0 can get uninitialized q in strtok */
      p=(char *)strtok(q,", ");

      while(j<cols && p) {
         if(number(p,&x)) {
            *(A+i+locvec(j,rows))=x;
            j++;
         }
         p=(char *)strtok('\0',", ");
      }
      if(j<cols) *(Rake+i)=1;
      else *(Rake+i)=0;
   }
   mallfree((void *)&q);

   return(
      rake() &&
      drop() &&
      lop() &&
      pushstr("_matread") &&
      naming()
   );
}

int neat() /* neat (hT --- hT1) */
/* Sub-strings of T into neat columns.

   This function is a literal translation of this word from boot.v:
      inline: neat (hT --- hT1) \ sub-strings of T into neat columns
         push one into count
         BEGIN peek count ndx string
            IF one count bump
               left justify " " tail
               no
            ELSE yes
            THEN
         UNTIL count nit any? IF parkn ELSE "" THEN
         pull drop notrailing "_neat" naming
      end
*/
{
   int count=1,flag,more=1;

   if(tos->typ!=VOL && tos->typ!=STR) {
      stkerr(" neat: ",STRORVOLNOT);
      return 0;
   }
   lpush(); /* T on temp stack */
   while(more) {
      lpeek();
      pushint(count);
      ndx();
      string();
      popint(&flag);
      if(flag) {
         count++;
         pushint(0);
         justify();
         pushstr(" ");
         tail();
      }
      else more=0;
   }
   count--;
   if(count) {
      pushint(count);
      parkn();
   }
   else pushstr("");

   return( 
      lpull() &&
      drop() &&
      notrailing() &&
      pushstr("_neat") &&
      naming()
   );
 }

int noblanklines() /* noblanklines (hT --- hT1) */
/* Removes lines from T that are blank. */
{
   char *s,*T,*T1,*Ti;
   register int chars,i=0,j=0,k=0,rows,width;
   int ret,typ;

   if((typ=tos->typ)!=VOL && typ!=STR) {
      stkerr(" noblanklines: ",STRORVOLNOT);
      return 0;
   }
   if(typ==STR) {
      chars=strlen1((T=tos->tex),tos->col);
      if((s=(char *)memgetn(T,chars))==NULL) return 0;
      ret=(
         drop() &&
         pushstr(s) &&
         pushstr("_noblanklines") && naming()
      );
      mallfree((void *)&s);
      return(ret);
   }
   T=tos->tex;
   rows=tos->row;
   width=tos->col;

   if(!volstk(rows,width,"_noblanklines")) return 0;
   T1=tos->tex;
   Ti=T;

   for(;i<rows;i++) {
      j=0;
      while(j<width && *(Ti+j)==' ') j++;
      if(j!=width) {
         memcpy(T1+loclin(k,width),Ti,width);
         k++;
      }
      Ti+=width;
   }
   if(k<rows) {
      if(!volstk(k,width,"_noblanklines")) return 0;
      memcpy(tos->tex,T1,k*width);
      return(lop() && lop());
   }
   return(lop());
}

int noq_alike() /* noq_alike (hR --- hT) */
/* Returns T with no quotes (rows) from R that are alike.  Assumes
   no null bytes in R. */
{
   if(noq_alike1()) return(drop());
   else return 0;
}

int noq_alike1() /* noq_alike1 (hR --- hT hA) */
/* Returns T with no quotes (rows) from R that are alike, and column
   vector A with ones at rows matching those taken from R to make T.
   Assumes no null bytes in R. */
{
   char *R,*R0,*Ref,*s,*T;
   double *A;
   int chars,j=0,k,m=0,mid,rows;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" noq_alike1: ",STRORVOLNOT);
      return 0;
   }
   if(tos->typ==STR) {
      pushint(0);
      pushint(1);
      null();
      return 1;
   }
   notrailing();

   Ref=tos->tex; /* original VOL, R */
   rows=tos->row;
   chars=tos->col;

   dup1s();
   cop();
   R0=tos->tex; /* copy of original */

   dup1s();
   cop();
   T=tos->tex; /* new VOL, T */

   if(!matstk(rows,1,"_A")) return 0;
   A=tos->mat;
   lpush(); /* put A on local stack */
   memset(A,0,sizeof(double)*rows);

   s=Ref;
   R=R0;
   mid=chars/2;

   for(;j<rows;j++) {

      if(*(R+loclin(j,chars))) {

         for(k=0;k<rows;k++) {

            if(*R && *(R+mid)==*(s+mid) && *R==*s && \
               !memcmp(R,s,chars)) *R='\0';

            R+=chars;
         }
         R=R0;
         memcpy(T,s,chars);
         m++;
         T+=chars;
         *(A+j)=1;
      }
      s+=chars;
   }
   lop();
   lop();

   if(!volstk(m,chars,"_noq_alike1")) return 0;
   memcpy(tos->tex,(tos-1)->tex,m*chars);

   return(lop() && lpull());
}

int notrailing() /* notrailing (hT --- hT1) */
/* Removes trailing blanks (and less-than-blanks) from end of text. */
{
   char *s,*s1;
   register int rows,chars,width,i=0;
   int typ;

   if(tos->typ==MAT || tos->typ==NUM) return 1; /* just ignore */

   if((typ=tos->typ)!=VOL && typ!=STR) {
      stkerr(" notrailing: ",STRORVOLNOT);
      return 0;
   }
   if(typ==STR) {
      chars=strlen1(tos->tex,tos->col);
      return(pushq2(tos->tex,chars) && lop());
   }
   asciify();

   chars=0;
   s1=tos->tex;
   rows=tos->row;
   width=tos->col;

   for(;i<rows;i++) {
      chars=MAX(chars,strlen1(s1,width));
      s1+=width;
   }
   s1=tos->tex; /* reset s1 */

   if(!volstk(rows,chars,"_notrailing")) return 0;
   s=tos->tex;

   if(chars) {
      for(i=0;i<rows;i++) {
         memcpy(s,s1,chars);
         s1+=width;
         s+=chars;
      }
   }
   return(lop());
}

int nullbyte() /* nullbyte ( --- qS) */
{
   return(pushq2("\0",1));
}

int nulrp() /* nulrp (hT qS --- hT1) */
/* Tue Sep  6 18:20:17 PDT 2011

   Replace null bytes in T with byte S. */
{
   char S,*T;
   int k=0,len;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" nulrp: ",STRORVOLNOT);
      return 0;
   }
   if((tos-1)->typ!=STR && (tos-1)->typ!=VOL) {
      stkerr(" nulrp: ",STRORVOLNOT);
      return 0;
   }
   S=*(tos->tex); /* qS */
   drop();

   cop();
   T=tos->tex; /* hT */

   len=tos->col*tos->row;

   for(;k<len;k++) {
      if(*T=='\0') *T=S;
      T++;
   }
   return(1);
}

int pilev() /* pile (hT hU --- hV) */
/* Piles two volumes, padding the smaller one with trailing blanks. */
{
   stkitem *Tvol,*Uvol;
   register char *T,*U,*V;
   register int rT,cT,rU;
   int cU,rV;

   if((tos-1)->typ==STR) {
      if(!(swap() && typstr2vol() && swap())) {
          return 0;
      }
   }
   if(tos->typ==STR) {
      if(!typstr2vol()) return 0;
   }
   if(stkset(" pilev: ")!=VOL) return 0;

   if(is_sparse(tos) || is_sparse(tos-1)) {
      return(
         pushint(2) && sppilen()
      );
   }
   Tvol=tos-1; Uvol=tos;
   rT=Tvol->row; cT=Tvol->col;
   rU=Uvol->row; cU=Uvol->col;
   if(cT<cU) {
      if(!(
         swap() && 
         pushint(rT) && 
         pushint(cU-cT) &&
         blockofblanks() && 
         cat() && 
         swap()
      )) return 0;
      cT=cU;
   } 
   else {
      if(cT>cU) {
         if(!(
            pushint(rU) && 
            pushint(cT-cU) &&
            blockofblanks() && 
            cat()
         )) return 0;
      }
   }
   if((V=(char *)memgetc((rV=rT+rU),cT))==NULL) return 0;
   T=Tvol->tex; U=Uvol->tex;
   memcpy(V,T,rT*cT);
   memcpy(V+loclin(rT,cT),U,rU*cT);
   return(
      drop2() &&
      push(VOL,(char *)memgetn("_pile",5),NOTAG,0,NULL,V,rV,cT,NULL)
   );
}

int qhexuint() /* hexuint (qS --- hA) */
/* Converts elements of string S into base 10 unsigned numbers.
   Base 16 strings in S are separated by spaces or commas, and each 
   becomes the number in a row of matrix A.
   Example:
      [tops@clacker] ready > "BEE,  FF  FD" hexuint .m 
       Row 1:     3054
       Row 2:      255
       Row 3:      253
      [tops@clacker] ready > */
{
   char c,*q,*T;
   unsigned long h=0;
   int fac=1,i,j=0,k,len,rows,width;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" qhexuint: ",STRORVOLNOT);
      return 0;
   }
   words();
   T=tos->tex;
   rows=tos->row;
   width=tos->col;

   if((q=(char *)malloc(1+width))==NULL) {
      stkerr(" qhexuint: ",MEMNOT);
      return 0;
   }
   for(;j<rows;j++) {
      memcpy(q,T,width);
      *(q+(len=strlen1(q,width)))='\0';

      fac=1;
      h=0;

      for(i=1;i<1+len;i++) {
         k=len-i;
         c=toupper(*(q+k));
         if(c>'@') c-=('A'-10);
         else c-='0';
         if(c<0 || c>15) {
            stkerr(" qhexuint: ",HEXNOT);
            return 0;
         }
         h+=fac*c;
         fac<<=4;
      }
      pushuint(h);
      T+=width;
   }
   mallfree((void *)&q);

   if(rows>1) {
      pushint(rows);
      return(listfin() && lop());
   }
   else return(lop());
}

int qnumber() /* number (qS --- false | x true) */
/* Takes just the first line if S is a VOL. */
{
   double x;
   char *q;
   int width;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" qnumber: ", STRORVOLNOT);
      return 0;
   }
   width=tos->col;
   if((q=(char *)memgetc(1,width))==NULL) {
      stkerr(" qnumber: ", MEMNOT);
      return 0;
   }
   memcpy(q,tos->tex,width);
   *(q+strlen1(q,width))='\0'; /* no trailing blanks */

   if(number(q,&x)) {
      mallfree((void *)&q);
      return(
         drop() && pushd(x) && pushint(xTRUE)
      );
   }
   mallfree((void *)&q);
   return(
      drop() && pushint(xFALSE)
   );
}

int quote() /* quote (hT r --- qS) */
/* Extracting row r from volume T, and presenting it as a quote string.
   The quote returned has trailing blanks to the width of T; word no-
   trailing can be used to remove them. */
{
   int row,width;

   if((tos-1)->typ!=STR && (tos-1)->typ!=VOL) {

      if((tos-1)->typ==MAT) return pry();

      else {
         stkerr(" quote: ",STRORVOLNOT);
         return 0;
      }
   }
   if(!popint(&row)) return 0;

   row=row-XBASE;

   if(row<0 || row+1>tos->row) {
      gprintf(" quote: row %d is outside rows of given text",row+XBASE);
      nc();
      stkerr("","");
      return 0;
   }
   width=tos->col;
   if(!strstk(width,"_quote")) return 0;

   memcpy(tos->tex,(tos-1)->tex+(loclin(row,width)),width);
   return(lop());
}

int quoted1() /* quoted (qS --- q"S") */
/* Putting string S in quotes. */
{
   if(tos->typ!=STR) {
      stkerr(" quoted: ",STRNOT);
      return 0;
   }
   if(strchr(tos->tex,'\"')) pushstr("'");
   else pushstr("\"");
         
   return(
      swap() &&
      over() &&
      cat() &&
      cat() 
   );
}

int replace_str() /* replace$ (hT qS qS1 --- hT1) */
/* In T, replace instances of string S with string S1.  String S has
   more characters than S1, and T is wider than S.  Blanks make up
   the difference in lengths of S and S1. */
{
   char *S1,*T,*T1;
   double *Map;
   int chars,i=0,rows;

   if(tos->typ!=STR) {
      stkerr(" replace$: ",STRNOT);
      return 0;
   } 
   if((tos-1)->typ!=STR) {
      stkerr(" replace$: ",STRNOT);
      return 0;
   } 
   if((tos-2)->typ!=STR && (tos-2)->typ!=VOL) {
      stkerr(" replace$: ",STRORVOLNOT);
      return 0;
   } 
   if(tos->col<(tos-1)->col) {
      chars=(tos-1)->col;
      if(!strstk(chars,"")) return 0;
      memset(tos->tex,' ',chars);
      memcpy(tos->tex,(tos-1)->tex,(tos-1)->col);
      lop();
   }
/* Number of chars to replace: */
   chars=MIN(MIN(tos->col,(tos-1)->col),(tos-2)->col); 

   lpush(); /* S1 on local stack */

   swap();
   cop();
   dup1s();
   rot(); /* stk: (hT1 hT1 qS) */
   smap(); /* stk: (hT1 hMap); offsets to instances of S in T1 */

   Map=tos->mat;
   rows=tos->row;

   T1=(tos-1)->tex;

   lpull(); /* S1 off local stack */
   S1=tos->tex;

   /* stk: (hT1 hMap qS) */

   for(;i<rows;i++) {
      T=T1+(int)*(Map+i);
      memcpy(T,S1,chars);   
   }
   return(drop2());
}

int rmnull(char *str, int len)
/* Removes from str each null character.  Returns length of str with
   nulls removed.  
   If any nulls are removed, an ending null is placed after the last 
   character, making str a null terminated string. */
{
   register int i=0,k=0;

   while(k<len) {
      if(*(str+k)) {
         *(str+i)=*(str+k);
         i++;
      }
      k++;
   }
   if(k>i) *(str+i)='\0';
   return(i);
}

int rmpath() /* -path (qS --- qS1) */
/* From S, remove rightmost slash and all characters to its left. */
{
   char *S;
   int i=0,m,rows,width;

   if(tos->typ==STR || tos->typ==VOL) {

      rows=tos->row;
      if(!(rows>0)) return 1;

      lpush();

      for(;i<rows;i++) {

         lpeek();

         pushint(i+XBASE);
         quote();
         unquoted();

         S=tos->tex;
         m=width=tos->col;

         while(m>0 && (*(S+m-1)!='/')) m--;

         pushq2(S+m,width-m);
         lop();

      }
      pushint(rows);
      pilen();
      lpull();
      (tos-1)->typ=tos->typ;
      return(drop());
   }
   stkerr(" rmpath: ",STRORVOLNOT);
   return 0;
}

int rounded() /* rounded (hA --- hB) */
/* Values in A rounded to integers.  Even numbers with fractional 0.5
   are rounded down and odd numbers with fractional 0.5 are rounded up.
*/
{
   register double *A,*B;
   register int k=0;
   char *s;
   int cols,rows;

   if(tos->typ==MAT) {

      if(is_complex(tos)) {
         return(
            cmplxdbl() &&
            rounded() &&
            swap() &&
            rounded() &&
            swap() &&
            dblcmplx()
         );
      }
      rows=tos->row;
      cols=tos->col;

      if(!matstk(rows,cols,"_rounded")) return 0;

      A=(tos-1)->mat;
      B=tos->mat;

      for(;k<rows*cols;k++) {
         s=mprintf("%30.0f",*A);
         number(s,B);
         mallfree((void *)&s);
         A++;
         B++;
      }
      return(lop());
   }
   if(tos->typ==NUM) {
      hand();
      if(!rounded()) return 0;
      if(is_complex(tos)) 
         return(pushdx(*(tos->mat), *((tos->mat)+1)) && lop());
      else return(pushd(*(tos->mat)) && lop());
   }
   stkerr(" rounded: ",NUMORMATNOT);
   return 0;
}

int said() /* said (qS hT n --- ) */
/* Putting S into the nth row of volume T.  If T is not in the catalog,
   or elsewhere on the stack, or on the local stack, it will be lost 
   when it drops from the stack here. */
{
   unsigned int len;
   unsigned long n;

   if(!popuint(&n)) return 0;

   if(tos->typ!=VOL && tos->typ!=STR) {
      stkerr(" said: ",STRORVOLNOT);
      return 0;
   }
   if((tos-1)->typ!=VOL && (tos-1)->typ!=STR) {
      stkerr(" said: ",STRORVOLNOT);
      return 0;
   }
   n=n-XBASE;
   if(n>(tos->row)-1) {
      stkerr(" said: ",OUTROW);
      return 0;
   }
   len=MIN((tos->col),((tos-1)->col));
   memcpy((tos->tex)+loclin(n,tos->col),(tos-1)->tex,len);
   return(drop2());
}

int smap() /* smap (qS qW --- hMap) */
/* Returns list of offsets to every instance of string W in S. */
{
   char *p,*S,*S0,*Smax,*W;
   unsigned long *M,*M0;
   int i=0,k=0,len;
   double *Map;
   int chars;

   if(tos->typ!=STR || ((tos-1)->typ!=STR && (tos-1)->typ!=VOL)) {
      stkerr(" smap: ",STRSNOT2);
      return 0;
   }
   len=tos->col;
   W=tos->tex;

   chars=(tos-1)->col*(tos-1)->row;
   S=(tos-1)->tex;
   S0=S;
   Smax=S+chars;
   
   if((M0=(unsigned long *)malloc( \
      sizeof(unsigned long)*(1+chars)))==NULL) {
      stkerr(" smap: ",MEMNOT);
      return 0;
   }
   M=M0;

   *M=0;
   if(len) {
      while(S<Smax && k<chars) {
         p=(char *)strstr(S,W);
         if(p) {
            *M=p-S0; /* offset to W */ 
            S+=p-S+len; /* advancing S beyond this W */
            M++;
            k++;
         }
         else S=Smax;
      }
   }
   if(!matstk(k,1,"_smap")) return 0;
   Map=tos->mat;

   M=M0;
   for(;i<k;i++) {
      *Map=*M;
      Map++;
      M++;
   }
   mallfree((void *)&M0);
   return(lop() && lop());
}

int spaced() /* spaced (qS --- qS_) */
/* If this function or function strlen() is changed, to verify 
   continued operability test it using tabfill on a unique match, 
   like -ALARM:

      [tops@clacker] ready > man -ALA [tab tab]

   which should display

      [tops@clacker] ready > man -ALARM _

   where the cursor (shown above as _) follows a space after the 
   letter M.  

   This behavior also requires that strlen() not trim trailing 
   blanks. */
{
   if(tos->row==1) {
      return(
         hand() &&
         pushint(XBASE) &&
         quote() &&
         pushstr(" ") &&
         cat()
      );
   }
   else {
      return(
         pushstr(" ") &&
         tail()
      );
   }
}

int str0() /* str0 (qS k --- qSk) */
/* Extract the kth null terminated string from S.  Assumes exactly one 
   terminating null after each string, and no other nulls in S. */
{
   char *S,*S0;
   int i=0,k=-1,k0,len;

   if(!popint(&k0)) return 0;
   k0-=XBASE;
   k0=MAX(0,k0);

   if(tos->typ!=STR) {
      stkerr(" str0: ",STRNOT);
      return 0;
   }
   S0=tos->tex;
   len=tos->col;

   while(i<len && !(k==k0)) {
      k++;
      S=S0+i;
      if(k==k0) pushstr(S);
      else i+=1+strlen(S);
   }
   if(!(k==k0)) pushstr("");

   return(lop());
}

int strarray() /* strarray (qS --- qS1) */
/* Inserting 0 between words of S, making a string array in C.  See 
   xterminit() for a loop that sets pointers in string array. */
{
   char *p,*S;
   register int chars,i=0,rows;

   if(tos->typ!=STR) {
      stkerr(" strarray: ",STRNOT);
      return 0;
   }
   words();

   S=tos->tex;
   rows=tos->row;
   chars=tos->col;
   if((p=(char *)memgetc(1,chars))==NULL) {
      stkerr(" strarray: ",MEMNOT);
      return 0;
   }
/* 1 0 blockofblanks, 1st quote: */
   pushint(1); pushint(0); blockofblanks(); 
   pushint(XBASE); quote();
      
   for(;i<rows;i++) {
      memcpy(p,S+loclin(i,chars),chars);   
      *(p+strlen1(p,chars))='\0';
      pushq2(p,1+strlen(p));
      cat();
   }
   mallfree((void *)&p);
   return(lop());
}

int strarrayx() /* strarrayx (qS --- hT) */
/* Extract null terminated strings from S and stack them in volume T,
   one row per string.  Assumes exactly one terminating null after each
   string, and no other nulls in S. */
{
   char *S,*S0;
   int i=0,k=0,len;

   if(tos->typ!=STR) {
      stkerr(" strarrayx: ",STRNOT);
      return 0;
   }
   S0=tos->tex;
   len=tos->col;

   while(i<len) {
      S=S0+i;
      pushstr(S);
      i+=tos->col+1;
      k++;
   }
   return(pushint(k) && pilen() && lop());
}

int strchop() /* strchop (qS --- qS1) */
/* Removes surrounding blanks from string.  VOLs will be sent to 
   chop(). */
{
   char *S;
   int m,n=0;

   if(tos->typ==STR) {
      S=tos->tex;

      m=tos->col-1; /* offset to last char in S */
      while(m>-1 && isspace((int)*(S+m))) m--;
      
      if(m>0) n=strspn(S," "); /* offset to 1st char in S */

      return(pushq2(S+n,m-n+1) && lop());
   }
   if(tos->typ==VOL) return(chop());

   stkerr(" strchop: ",STRNOT);
   return 0;
}

void strchop1(char *S)
/* Removes surrounding blanks from string and places ending null after
   last nonblank character. */ 
{
   int m,n=0;

   m=strlen(S)-1; /* offset to last char in S */
   while(m>-1 && isspace((int)*(S+m))) m--;
   *(S+m+1)='\0';

   if(m>0) n=strspn(S," "); /* offset to 1st char in S */

   if (!n)
       return;

   memcpy(S,S+n,m-n+2);
}

int string() /* string (qS n --- false | qSn true) */
/* Pushes the nth space-delimited string from S and a true flag onto 
   the stack; pushes just a false flag if no nth string. 

   Also works if qS is hT, a VOL.  If VOL, returns a VOL with just
   string n on each line, and a true flag; if no string n, the line is 
   blank.  If there is no string n on any lines, returns just a false 
   flag. */
{
   int chars,i,k=0,n,ok=xTRUE,rows,vol=0;
   char *S,*Sn;
   char *buf,*p;
   char *name="_string";

   if(!popint(&n)) return 0;
   n-=XBASE;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" string: ",STRORVOLNOT);
      return 0;
   }
   S=tos->tex;
   if(!tos->col || !tos->row) {
      return(
         drop() && 
         pushint(0)
      );
   }
   vol=(tos->typ==VOL);

   if(!volstk((rows=tos->row),(chars=tos->col),name)) return 0;
   if(!vol) tos->typ=STR;

   Sn=tos->tex;
   memset(Sn,' ',rows*chars);

   if((buf=(char *)malloc(1+chars))==NULL) {
      stkerr(" string: ",MEMNOT);
      return 0;
   }
   while(k<rows && ok==xTRUE) {
      memcpy(buf,S,chars); 
      *(buf+chars)='\0';

      p=(char *)strtok(buf," "); /* puts space delimiters */
   
      i=1;
      while(i<=n) {
         p=(char *)strtok('\0'," ");
         i++;
      }
      if(p) memcpy(Sn,p,strlen(p));
      else if(!vol) ok=xFALSE; /* no string n in STR */

      S+=chars;
      Sn+=chars;
      k++;
   } 
   mallfree((void *)&buf);

   if(vol) { /* check VOL for all blanks */
      dup1s();
      noblanklines();
      if(!tos->row) ok=xFALSE;
      drop();
   } 
   if(ok) {
      return(
         lop() &&
         notrailing() &&
         pushstr(name) &&
         naming() &&
         pushint(ok)
      );
   }
   else {
      return(
         drop2() &&
         pushint(ok)
      );
   }
}

int strings() /* strings (qS --- hT) */
/* Each space delimited substring of S occupies a row in volume T. 
   This function always returns a volume (VOL). */
{
   char *p,*S,*T;
   int ret;

   if(tos->typ==VOL) typvol2str(); /* VOL to STR */
   else cop(); /* copy STR since strtok() will change contents */

   if(tos->typ!=STR) {
      stkerr(" strings: ",STRORVOLNOT);
      return 0;
   }
   asciify();
   S=tos->tex;

/* len+2 for added ending: 0A 00 */
   if(!volstk(1,2+tos->col,"_T")) return 0;
   T=tos->tex;

   p=strtok(S," ");

   while(p) {
      memcpy(T,p,strlen(p));
      T+=strlen(p);
      *T='\n';
      T++;
      p=(char *)strtok('\0'," ");
   }
   *T='\0'; /* textget() needs ending null byte */

   ret=textget(0); 

/* Force to be VOL--textget() has returned STR if input one row */
   tos->typ=VOL;

   return(
      ret &&
      notrailing() &&
      pushstr("_strings") &&
      naming() &&
      lop()
   );
}

int strlenq() /* strlen (qS | hT --- n | hA) */
/* Length of quote-string S including trailing blanks, or vector of 
   lengths of each row of T without trailing blanks. 

   Not trimming trailing blanks of S is desired behavior.  See note 
   in spaced(), this file.  

   One row VOL T is treated like STR S. */
{
   char *T;
   double *A;
   int chars,k=0,rows;

   if(tos->typ==VOL && tos->row>1) { 
      notrailing();
      T=tos->tex;
      chars=tos->col;
      rows=tos->row;

      if(!matstk(rows,1,"_strlen")) return 0;
      A=tos->mat;

      for(;k<rows;k++) {
         *A=(double)strlen1(T,chars);
         A++;
         T+=chars;
      }
      return(lop());
   }
   else {
      if(tos->typ==STR || tos->typ==VOL) {
      /* When string or 1 line VOL, do not trim trailing blanks: */
         chars = tos->col;
         return(
            drop() &&
            pushint(chars)
         );
      }
   }
   stkerr(" strlen: ",STRORVOLNOT);
   return 0;
}

int strmatch0() /* strmatch0 (qS1 qS2 len --- f) */
/* Matching strings to length len, including nulls.  Flag f is true
   if the strings match, and false if they do not. */
{
   double dlen;
   char *S1,*S2;
   int k=0,len1,len2,matching=1;

   if(tos->typ!=NUM) {
      stkerr(" strmatch0: ",NUMNOT);
      return 0;
   }
   if(((tos-2)->typ!=STR && (tos-2)->typ!=VOL) ||
      ((tos-1)->typ!=STR && (tos-1)->typ!=VOL)) {
      stkerr(" strmatch0: ",STRORVOLNOT);
      return 0;
   }
   popd(&dlen);

   S2=tos->tex;
   len2=tos->col*tos->row;

   S1=(tos-1)->tex;
   len1=(tos-1)->col*(tos-1)->row;

   len1=(int)MIN((double)len1,dlen);
   len1=MIN(len1,len2);

   while(matching && k<len1) {
      matching=(*S1==*S2);
      k++;
      S1++;
      S2++;
   }
   if(matching) matching=xTRUE;
   else matching=xFALSE;

   return(drop2() && pushint(matching));
}

int strmatch1() /* strmatch (qS1 qS2 --- n) */
/* Matching quote-strings: n=0 if match, n>0 if S1>S2. */
{
   int len,ret;

   if((tos->typ!=STR && tos->typ!=VOL) || 
      ((tos-1)->typ!=STR && (tos-1)->typ!=VOL)) { 
      stkerr(" strmatch: ",STRORVOLNOT);
      return 0;
   }
   len=MAX(tos->col*tos->row,(tos-1)->col*(tos-1)->row);
   ret=strncmp((tos-1)->tex,tos->tex,len);
   return(drop2() && pushint(ret));
}

int strplace() /* strplace (qB qA n --- ) */
/* Place all of string B into string A starting at index n. */
{
   int lenA,lenB,n;

   if((tos-2)->typ!=STR || (tos-1)->typ!=STR) {
      stkerr(" strplace: ",STRNOT);
      return 0;
   }
   if(!popint(&n)) return 0;
   n=n-XBASE;

   lenA=tos->col;
   lenB=(tos-1)->col;

   if(n>=lenA || n<0 || n+lenB>lenA) {
      stkerr(" strplace: ",OUTCHAR);
      return 0;
   }
   memcpy((tos->tex)+n,(tos-1)->tex,lenB);
   return(drop2());
}

int strp() /* strp (hT qS qS1 --- hT1) */
/* In T, replace instances of string S with string S1.  Strings S and
   S1 may be any size and may contain blanks. */
{
   char *p,*S,*S1,*T,*W,*W0;
   int cols,k=0,len,len1,more,rows;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" strp: ",STRORVOLNOT);
      return 0;
   }
   if((tos-1)->typ!=STR && (tos-1)->typ!=VOL) {
      stkerr(" strp: ",STRORVOLNOT);
      return 0;
   }
   if((tos-2)->typ!=STR && (tos-2)->typ!=VOL) {
      stkerr(" strp: ",STRORVOLNOT);
      return 0;
   }
   T=(tos-2)->tex;
   rows=(tos-2)->row;
   cols=(tos-2)->col;

   S=(tos-1)->tex;
   len=(tos-1)->col;
   if(!(len>0 && (tos-1)->row>0)) {
      stkerr(" strp: ","target string S is empty");
      return 0;
   }
   S1=tos->tex;
   len1=tos->col;

   if((W0=(char *)malloc(1+cols))==NULL) {
      stkerr(" strp: ",MEMNOT);
      return 0;
   }
   *(W0+cols)='\0';

   for(;k<rows;k++) {
      W=W0;
      memcpy(W,T,cols);

      more=1;
      pushstr("");

      while(more) {
         p=(char *)strstr(W,S);
         if(p==NULL) {
            pushstr(W);
            more=0;
         }
         else {
            pushq2(W,p-W);
            pushq2(S1,len1); /* S1 can have nulls */
            cat();
            W=p+len;
         }
         cat();
      }
      T+=cols;
   }
   mallfree((void *)&W0);

   return(
      pushint(rows) &&
      pilen() &&
      lop() &&
      lop() &&
      lop()
   );
}

int tail() /* tail (hT qS --- hT1) */
{
   return(
      over() &&
      hand() &&
      pushint(tos->row) &&
      lop() &&
      pileof() &&
      park()
   );
}

int textget(int flag) /* (qS --- hT) */
/* Creates volume T with WIDTH characters per line, from string S that 
   includes newline characters.  Returned T is a quote-string if there 
   is just one line (str2vol can be used to convert to volume).

   If incoming flag is nonzero, the first line of S is not part of T.  

   Assumes string S is null terminated.  

   Word asciify can be used to remove non-ascii characters, like new-
   line characters and tabs. */
{
   register char *string,*s;
   register int lines=0,i,len,minwidth;

   minwidth=WIDTH;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" textget: ",STRORVOLNOT);
      return 0;
   }
   string=tos->tex;
   if(flag) string+=stradv(string,"\n"); /* skipping first line */
   s=string;
   if(strlen(string)>0) { 
      while((i=stradv(s,"\n"))!=0) { 
         lines++; 
         s+=i;
         minwidth=MAX(minwidth,i);
      }
      if(lines==0) lines++;
   }
   else { 
      lines=1;
   }
   if((s=(char *)memgetc(lines,minwidth))==NULL) {
      stkerr(" textget: ",MEMNOT);
      drop();
      return 0;
   }
   i=0;
   while(i<lines) {
      len=stradv(string,"\n"); /* newline char will be in T too */
      if(len>0) {
         memcpy(s+loclin(i,minwidth),string,len);
      }
      string+=len;
      i++;
   }
   if(1 || lines>1) /* returning a VOL */
      return(
         drop() &&
         pushtex((char *)memgetn("_{\"",3),s,lines,minwidth)
      );
   else /* returning a STR */
      return(
         drop() &&
         pushq(s,minwidth) &&
         pushq2("_{\"",3) && naming()
      );
}

int textget0() /* textget (qS --- hT) */
/* Making volume from string or one-row volume with newline
   characters. */
{
   int len,ret=1;

   if(tos->typ==VOL)
      ret=(ret &&
         pushint(XBASE) &&
         quote()
      );
   if(!ret) return 0;

   if(tos->typ!=STR) {
      stkerr(" textget: ",STRORVOLNOT);
      return 0;
   }
/* Adding newline ch at end if one is missing */
   dup1s();
   strlenq();
   popint(&len);
   if(*(tos->tex+len-1)!='\n') {
      notrailing();
      pushstr("\n");
      cat();
   }
   ret=textget(0);

/* Force VOL--textget() will return STR if input one row */
   tos->typ=VOL;

   return(
      ret &&
      pushstr("_textget") &&
      naming()
   );
}

int textput() /* textput (hT --- qS) */
/* Making volume into one-row string with newline characters.  Does
   nothing if T is already a string. */
{
   register char *S,*Ti;
   register int i=0,len;
   char *nl="\n";
   int chars,rows;

   if(tos->typ==STR) {
      pushstr(nl);
      cat();
      return 1;
   }
   if(tos->typ!=VOL) {
      stkerr(" textput: ",STRORVOLNOT);
      return 0;
   }
   rows=tos->row;

   chars=tos->col;
   Ti=tos->tex;

   if(!strstk(rows+rows*chars,"_textput")) return 0;
   S=tos->tex;

   for(;i<rows;i++) {
      memcpy(S,Ti,(len=strlen1(Ti,chars))); /* no trailing blanks */
      Ti+=chars;
      S+=len;
      *S=*nl; /* newline character */
      S++;
   }
   *S='\0';
   S++;
   len=S-(tos->tex);

   if(TRACE) {
      gprintf(" textput: from volume bytes %d, to string bytes %d",\
         rows*chars,len-1);
      nc();
   }
   tos->col=len-1; /* override stack item character size */
   return(lop());
}

int textwidth() /* WIDTH (n --- ) */
/* Set minimum WIDTH used by textget.  Default is 80. */
{
   int W;

   if(!popint(&W)) return 0;

   WIDTH=MAX(1,W);

   return 1;
}

int totext() /* >text (qS c n --- qS1) or strput (qS c n --- qS1) */
/* Replacing character n in string S with ascii c */
{
   int c,len,n;
   char *S1;

   if(!popint(&n)) return 0;
   n=n-XBASE;
   if(!popint(&c)) return 0;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" strput: ",STRORVOLNOT);
      return 0;
   }
   len=tos->col*tos->row;
   if(n>=len || n<0) {
      stkerr(" strput: ",OUTCHAR);
      return 0;
   }
   if((S1=(char *)memgetn(tos->tex,len))==NULL) return 0;
   drop();
   *(S1+n)=c;

   return(push(STR,(char *)memgetn("_strput",7),NOTAG,0,NULL,S1, \
      1,len,NULL));
}

int tug() /* tug (qS qF --- qS1) */
/* Tugs phrase S to the left until string F is leftmost.  S1 is empty 
   string if F is not in S.  If S is a volume, does this for every 
   line; lines without F are blank. */
{
   char *F,*p,*s,*S,*S1;
   register int k=0;
   int chars,never=1,rows,typ;

   if(tos->typ!=STR || ((tos-1)->typ!=STR && (tos-1)->typ!=VOL)){ 
      stkerr(" tug: ",STRORVOLNOT);
      return 0;
   }
   F=tos->tex;
   S=(tos-1)->tex;
   typ=(tos-1)->typ;
   rows=(tos-1)->row;
   chars=(tos-1)->col;

   if(!volstk(rows,chars,"_tug")) return 0;
   S1=tos->tex;
   memset(S1,' ',rows*chars); /* blanks (cannot be nulls) */

   if((s=(char *)malloc(1+chars))==NULL) return 0;
   *(s+chars)='\0';

   for(;k<rows;k++) {
      memcpy(s,S,chars);
      p=(char *)strstr(s,F);
      if(p!=NULL) {
         memcpy(S1,p,strlen(p));
         never=0;
      }
      S+=chars;
      S1+=chars;
   }
   if(typ==STR) tos->typ=STR; 
   mallfree((void *)&s);

   if(never) { /* returning string of 0 characters */
      drop();
      volstk(1,0,"_tug");
   }
   return(lop() && lop());
}

int typmat2vol() /* mat2vol (hA --- hT) */
/* Columns of matrix A are rows in volume T.  Bytes are unchanged.

   Using cop() below means that if T is just one instance on the
   stack, and T is not in the catalog, no bytes had to be allocated 
   or moved. */
{
   int chars;

   if(tos->typ!=MAT) {
      stkerr(" mat2vol: ",MATNOT); return 0;
   }
   if(!cop()) return 0;

   tos->tex=(char *)tos->mat;
   tos->mat=NULL;
   tos->typ=VOL;

   chars=sizeof(double)*tos->row;
   tos->row=tos->col;
   tos->col=chars;

   return(
      pushstr("_mat2vol") &&
      naming() 
   );
}

int typnum2mat() /* num2mat (x --- hA) */
{
   double xr,xi;
   int complex=0;

   complex=is_complex(tos);
   if(!popdx(&xr,&xi)) return 0;

   if(complex) {
      if(!matstk(2,1,"_num2mat")) return 0;
      set_complex(tos);
      *(tos->mat)=xr;
      *(tos->mat+1)=xi;
   }
   else {
      if(!matstk(1,1,"_num2mat")) return 0;
      *(tos->mat)=xr;
   }
   return 1;
}

int typnum2str() /* num2str (x --- qx) */
{
   char *str;
   union { 
      double x; 
      unsigned char c[sizeof(double)]; 
   } xc;

   if(tos->typ!=NUM) {
      stkerr(" num2str: ",NUMNOT); 
      return 0; 
   }
   if(is_complex(tos)) {
      stkerr(" num2str: ",REALNOT);
      return 0;    
   }
   if((str=(char *)memgetc(1,sizeof(double)))==NULL) return 0;
   xc.x=pop()->real;
   memcpy(str,xc.c,sizeof(double));
   return(push(STR,(char *)memgetn("_num2str",8),NOTAG,0,NULL,str, \
      1,sizeof(double),NULL));
}

int typstr2num() /* str2num (qS --- x) */
{
   return(
      typstr2vol() &&
      typvol2mat() &&
      pushint(XBASE) &&
      pry()
   );
}

int typstr2mat() /* str2mat (qS --- hA) */
{
   char *S;
   double *A;
   int chars,rows,TAG;

   if(tos->typ!=STR) {
      stkerr(" str2mat: ",STRNOT); return 0;
   }
   S=tos->tex; 
   chars=tos->col;
   rows=1+chars/sizeof(double);
   TAG=tos->tag;
 
   if(!matstk(rows,1,"_str2mat")) return 0;

   A=tos->mat;
   tos->tag=TAG;

   *(A+rows-1)=0;
   memcpy((char *)A,S,chars);

   return(lop());
}

int typstr2vol() /* str2vol (qS --- hT) */
/* Type of S is changed in place.  Address to bytes of S remains the
   same. */
{
   if(tos->typ!=STR) {
      stkerr(" str2vol: ",STRNOT); return 0;
   }
   tos->typ=VOL;
   return(
      pushstr("_str2vol") &&
      naming()
   );
}

int typvol2mat() /* vol2mat (hT --- hA) */ 
/* Text volume into matrix, where a matrix column contains a text row.

   Bytes of T are unchanged, except blanks may be appended to its rows
   to give char width that is an integer multiple of sizeof(double) 
   bytes. 

   Using cop() below means that if T is just one instance on the 
   stack, and T is not in the catalog, no bytes had to be allocated 
   or moved. */
{
   int chars=0;

   if(tos->typ==STR) hand();

   if(tos->typ!=VOL) {
      stkerr(" vol2mat: ",VOLNOT); return 0;
   }
   if(tos->col%sizeof(double)) return(typvol2mat1());

   if(!cop()) return 0;

   tos->mat=(double *)tos->tex;
   tos->tex=NULL;
   tos->typ=MAT;

   chars=tos->col;
   tos->col=tos->row;
   tos->row=chars/sizeof(double);

   return(
      pushstr("_vol2mat") &&
      naming()
   );
}

int typvol2mat1() /* vol2mat (hT --- hA) */
/* Text volume into matrix, where a matrix column contains a text row.
   Bytes of T are unchanged, except blanks may be appended to its rows
   to give an integer multiple of sizeof(double) bytes per row. */
{
   double *A,*B;
   int chars,cols,del,rows,TAG;

   if(tos->typ!=VOL) {
      stkerr(" vol2mat: ",VOLNOT); return 0;
   }
   rows=tos->row;
   chars=tos->col;
   TAG=tos->tag;

   if((del=chars%sizeof(double))) del=sizeof(double)-del;

   if(!(pushint(rows) &&
        pushint(del) &&
        blockofblanks() &&
        cat())
     ) return 0;

   cols=tos->col/sizeof(double);
   A=(double *)tos->tex;

   if((B=(double *)memget(cols,rows))==NULL) return 0;

   memcpy(B,A,rows*cols*sizeof(double));

   return(
      drop() &&
      push(MAT,(char *)memgetn("_vol2mat",8),TAG,0,B,NULL, \
         cols,rows,NULL)
   );
}

int typvol2str() /* vol2str (hT --- qS) */
/* Volume T into string S.
   NL char at end of line from textput becomes a space with
   asciify that separates, in S, the ending word of one line
   from the beginning one of the next. */
{
   return(
      textput() && asciify()
   );
}

int unquoted() /* unquoted (qS --- qS1) */
/* Removing surrounding single- or double-quotes from S.  Does nothing
   if S is a VOL with more than one row. */
{
   char *s1,*s2;

   if(tos->typ==VOL) {
      if(tos->row==1) { /* take first row string if VOL */
         pushint(XBASE);
         quote();
      }
      else return 1; /* return if VOL has more than one row */
   }
   if(tos->typ!=STR) {
      stkerr(" unquoted: ",STRNOT);
      return 0;
   }
   if(!strchop()) return 0;
   if((tos->col)<2) return 1;

   s1=tos->tex;
   s2=s1+strlen(s1)-1;

   if( 
      (*(s1)=='\"' && (*(s2)=='\"')) ||
      (*(s1)=='\'' && (*(s2)=='\'')) 
     ) {
     s1++;
     *s2='\0';
   }
   return(pushq2(s1,strlen(s1)) && lop());
}

int uppercase() /* uppercase (qS | hT --- qS1 | hT1) */
{
   register int cols,i=0,j=0,rows;
   register char *p,*pi;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" uppercase: ",STRORVOLNOT);
      return 0;
   }
   rows=tos->row;
   cols=tos->col;

   if((p=(char *)memgetn(tos->tex,rows*cols))==NULL) {
      stkerr(" uppercase: ",MEMNOT);
      return 0;
   }
   for(;i<rows;i++) {
      pi=p+loclin(i,cols);
      for(j=0;j<cols;j++) {
         *(pi+j)=toupper(*(pi+j));
      }
   }
   if(tos->typ==STR)
      return(
         drop() &&
         pushq(p,cols) &&
         pushstr("_uppercase") && naming()
      );
   else
      return(
         drop() &&
         push(VOL,(char *)memgetn("_uppercase",10),NOTAG,0,NULL,p, \
            rows,cols,NULL)
      );
}

int word() /* word (qS n --- false | qSn true) */
/* Pushes the nth word from S and a true flag onto the stack; pushes
   just a false flag if no nth word. 

   Also works if qS is hT, a VOL.  If VOL, returns a VOL with just
   word n on each line, and a true flag; if no word n, the line is 
   blank.  If there is no word n on any lines, returns just a false 
   flag. */
{
   int chars,i,k=0,n,ok=xTRUE,rows,vol=0;
   char *S,*Sn;
   char *buf,*p;
   char *name="_word";

   if(!popint(&n)) return 0;
   n-=XBASE;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" word: ",STRORVOLNOT);
      return 0;
   }
   if(!tos->col || !tos->row) {
      return(
         drop() && 
         pushint(0)
      );
   }
   S=tos->tex;
   vol=(tos->typ==VOL);

   if(!volstk((rows=tos->row),(chars=tos->col),name)) return 0;
   if(!vol) tos->typ=STR;

   Sn=tos->tex;
   memset(Sn,' ',rows*chars);

   if((buf=(char *)malloc(1+chars))==NULL) {
      stkerr(" word: ",MEMNOT);
      return 0;
   }
   while(k<rows && ok==xTRUE) {
      memcpy(buf,S,chars); 
      *(buf+chars)='\0';

      p=(char *)strtok(buf,", "); /* puts space and comma delimiters */
   
      i=1;
      while(i<=n) {
         p=(char *)strtok('\0',", ");
         i++;
      }
      if(p) memcpy(Sn,p,strlen(p));
      else if(!vol) ok=xFALSE; /* no word n in STR */

      S+=chars;
      Sn+=chars;
      k++;
   } 
   mallfree((void *)&buf);

   if(vol) { /* check VOL for all blanks */
      dup1s();
      noblanklines();
      if(!tos->row) ok=xFALSE;
      drop();
   } 
   if(ok) {
      return(
         lop() &&
         notrailing() &&
         pushstr(name) &&
         naming() &&
         pushint(ok)
      );
   }
   else {
      return(
         drop2() &&
         pushint(ok)
      );
   }
}

int words() /* words (qS --- hT) */
/* Each word of string S occupies a row of volume T.
   Commas are eliminated, since words do not contain them.
   This function always returns a volume (VOL). */
{
   return(
      pushstr(",") &&
      chblank() &&
      strings() &&
      pushstr("_words") &&
      naming()
   );
}
