\ {{{1 GNU General Public License
{
Program Tops - a stack-based computing environment
Copyright (C) 1999-2012  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} 
}

\ File mmath.v  October 2000

\ Copyright (c) 2000-2012   D. R. Williamson

   CATMSG push no catmsg 

   "numbad" missing IF "math.v" source THEN

{ ----------------------------------------------------------------------

   Contents:

   Use these to obtain the following list of contents:
      syspath "mmath.v" + asciiload this " inline:" grepr reach dot
      syspath "mmath.v" + asciiload this " function " grepr reach dot

   inline: *c (hAr hAi hBr hBi --- hCr hCi) \ complex matrix mult
   inline: *cby (hAr hAi hBr hBi --- hCr hCi) \ complex el-by-el mult
   inline: *conj (hAr hAi --- hM) \ el-by-el A times complex conjugate
   inline: +c (hAr hAi hBr hBi --- hCr hCi) \ complex addition
   inline: /cby (hAr hAi hBr hBi --- hCr hCi) \ complex el-by-el div
   inline: acceleration (hAc hOmega dOmega hr hdr hddr --- hAp)
   inline: acos (hC --- hA) \ inverse cosine
   inline: asin (hC --- hA) \ inverse sine
   function (D) = btt(P, t, dt, f) // Bottom or top tracker
   inline: c_resultant (hA --- hR) \ resultant of complex components
   inline: caccel (hAr hAi hOmega --- hVr hVi) \ complex harmonic accel
   inline: cdisp (hqdd hOmega --- hq) \ complex displacement from accel
   inline: cmag (hAr hAi --- hM) cmagphase drop ; \ magnitude of A
   inline: colrm (hA hF Fmax --- hA1 hF1) \ remove columns of A
   inline: cosine (A w phi dt N --- hW ht) \ cosine wave time history
   inline: couple (hMaa hSVc hSVb hSVq hmaa hMSEc hMSEb hMSEq --- hMhh)
   inline: cov (hA --- hCov) \ covariance of terms in A
   inline: cov2 (hX hY hLAG --- hR) \ covariance between X and Y
   inline: crate (hAr hAi hOmega --- hVr hVi) \ complex harmonic rate
   inline: Cronin (Q Oct/min Hz --- %Resonance) \ Cronin's equation
   inline: EIVrows (hPhi r --- hRows) \ rows of Phi that influence EIV
   inline: db10 (hP --- hPdb) \ 10*log10(P): dB when P is a power ratio
   inline: db20 (hP --- hPdb) \ 20*log10(P): dB for pressure ratio P
   inline: dbcheatsheet ( --- ) \ some decibel examples
   inline: derivative (hX ht --- hdX) \ change in X by finite difference
   inline: diagpost_complex (hAr hAi hVr hVi --- hBr hBi)
   inline: diagpre_complex (hVr hVi hAr hAi --- hBr hBi)
   inline: dircos (hX hZ --- hC) \ create direction cosine matrices
   inline: dircos' (hC --- hC') \ transpose direction cosine matrices
   inline: dircos9 (hABC --- hC) \ creating direction cosine matrices
   inline: eigchk (C Lr Li Ar Ai --- Br Bi) \ check eigensolution
   inline: Frms (hP hF --- hRMS) \ rms integral of P over frequencies F
   inline: Frms_log (hP hF --- hRMS) \ integral of P over frequencies F
   inline: groups (hR g1 g2 --- hG) \ groups of consec numbers with gaps
   inline: IC+ (hX hV --- hX1) \ add init conditions V(k) to X(k,t)
   inline: integ (hY hf --- hIY) \ integral Y*df, from 1st to last f
   inline: integral (hY hf --- hIY) \ integral Y*df, all terms
   inline: Kdecomp (hF hK --- hx) \ solve K*x = F for x
   inline: Kinv (hK --- hKinv) \ inverse of symmetric stiffness matrix
   inline: locrect (hRTZ --- hXYZ) \ points from cyl into rect sys
   inline: lsq (b A --- x) \ least-squares solution, x=[inv(A'*A)]*A'*b
   inline: median (hC --- hCm) \ row k of Cm holds median across C(k)
   inline: Miles (hPSD hF nQ --- hGrms) \ Mile's equation for random PSD
   inline: mmax2 (hVm0 hV ht ntau --- hVm)
   inline: mmin2 (hVm0 hV ht ntau --- hVm) \ moving minimum
   inline: mnorm (hVL hVR hM --- hVL1 hVL2) \ normalize complex modes
   inline: modal_work (hPHI hFHz --- hWork) \ work in mode PHI of FHz
   inline: modes (hA --- hPhi hOmegSq) \ separate A into shape, freq
   inline: msigma (hA ht w d --- hM) \ moving mean and variance
   inline: mstats (hA n --- hB) \ moving statistics
   inline: nonesame (hA --- hA1) \ no terms in A are the same
   inline: nonneg (hV1 --- hV) \ make the changes in V nonnegative
   inline: normsig (hA --- hA1 nMean nSigma)
   inline: palign (hY1 ht1 hY2 ht2 --- hY1 hY2 ht) \ Y1 and Y2 at t
   inline: palign_log (hY1 ht1 hY2 ht2 --- hY1 hY2 ht) \ Y1 and Y2 at t
   inline: ploc (hCab A* hXYZb --- hXYZa) \ express sys B vecs in sys A
   inline: rats (hKxx hRBx --- hSx) \ force separation ratios
   inline: RBcyl (hTheta hXYZ --- Phi6Nx6) \ RB modes for cyl coords
   inline: RBphys (hPhi hSupport --- hPhi1) \ physical RB vectors
   inline: RBrect (hXYZ --- Phi6Nx6) \ RB modes for rect coords
   inline: refine (hA n --- hA1) \ steps between rows of A n times finer
   inline: Rmax (hV n --- hR) \ rows of V for session nmax highest 
   inline: rms (hY --- Yrms) \ rms of amplitudes in Y columns
   inline: rotating (hC0 hOmega hV ht --- hV1) \ inertial from rotating
   inline: rsi (hH hL hC --- hRSI) \ relative strength of C within H - L
   inline: rss (hY --- Yrss) \ rss of amplitudes in Y columns
   inline: s3curve (hC ht n --- hX hX' hX'')
   inline: s3deriv (hC ht n --- dC) \ derivative of C(t)
   inline: sine (A w phi dt N --- hW ht) \ sine wave time history
   inline: sine_rms (hA --- hR) \ total rms of sines
   inline: sine_rms_moving (hA1 hF1 Wfac dF --- hR hF)
   inline: sine_rms_running (hA --- hR) \ running rms of sines
   inline: sined (A w z phi dt N --- hW ht) \ damped sine wave time hist
   inline: steady_balance (hZ hW hQ hFrad hqr hqi --- hBr hBi)
   inline: syminv (hA --- hAinv) \ inverse of symmetric matrix
   inline: symmetric (hA --- hA1) \ force A to be symmetric
   inline: Tcyl (hT --- hCcb) \ transformation from rect to cyl
   inline: tilde (hA --- hA~) \ 3x3 tilde matrices from x,y,z vectors
   inline: tilts (hC --- hTheta) \ tilts to make chain of attitudes in C
   inline: transform (hM hT --- hT'MT) \ transform M to M1=T'*M*T
   inline: transform_diag (hM hT --- hT'MT) \ transform diagonal M
   inline: trap (hV ht --- hX) \ integrate V(t) using trapezoidal rule
   inline: trapC (hC0 hOmega t --- hC) \ integrate dir cos matrix
   inline: truncate (hPhim hOmega2m omeg2 --- hPhi hOmega2)
   inline: twang (hB hK dof dt steps --- hq ht) \ transient free-vib
   inline: velocity (hVc hOmega hr hdr --- hVp) \ inertial velocity
   inline: watching (hC0 hOmega hV ht --- hV1) \ rotating from inertial
   inline: wCOV (hX hY hLAG w d --- hR)
   inline: wCOVt (hX hY ht w d --- R) \ covariance R(t) for X(t), Y(t)
   inline: weighted (hA hW --- hB) \ weighted average
   inline: xformvec (hOp hRpb Xb type -- hXp) \ transform Xb to Xp
   inline: xortho (A Lr Li Pr Pi Gr Gi --- ) \ cross orthogonality

---------------------------------------------------------------------- }

   inline: *c (hAr hAi hBr hBi --- hCr hCi) \ complex matrix mult
\     Cr = Ar*Br - Ai*Bi, Ci = Ar*Bi + Ai*Br
      3 pick over (Ar Bi) *, 3 pick 3 pick (Ar*Bi Ai Br) * + (Ci) push
      3 roll rot (Ar Br) *, rev (Ar*Br Ai Bi) * - (Cr)  
      (Cr) "_Cr" naming, pull (Ci) "_Ci" naming ;
{
  _inline: *c1 (hAr hAi hBr hBi --- hCr hCi) \ complex matrix mult
\     Cr = Ar*Br - Ai*Bi; Ci = (Ar+Ai)*(Br+Bi) - Ar*Br - Ai*Bi
\     Experimental version of *c with three multiplies instead of four.

\     Note: for large matrices (600-by-600) *c1 is about 30% faster
\     than *c; for small matrices, improvement is much less.

      3 pick 3 pick (Ar Ai) +, other other + (Ar+Ai) (Br+Bi) * push
      3 roll 2 roll (Ar Br) *, rev (Ai Bi) *  2dup -
      (Cr) "_Cr" naming rev, + pull swap - (Ci) "_Ci" naming ;
}
   inline: *cby (hAr hAi hBr hBi --- hCr hCi) \ complex el-by-el mult
\     Cr = Ar*Br - Ai*Bi, Ci = Ar*Bi + Ai*Br 
\     where * = *by, element-by-element multiplication
      3 pick over (Ar Bi) *by, 3 pick 3 pick (Ar*Bi Ai Br) *by + (Ci)
      (Ci) push, 3 roll rot (Ar Br) *by, rev (Ar*Br Ai Bi) *by - (Cr)
      (Cr) "_Cr" naming, pull (Ci) "_Ci" naming ;

   inline: *conj (hAr hAi --- hM) \ el-by-el A times complex conjugate
\     Equivalent to (Ar+iAi)*(Ar-iAi) where * is element-by-element
\     multiplication.
      cmagphase drop dup *by ;

   inline: +c (hAr hAi hBr hBi --- hCr hCi) \ complex addition
      rot plus rev plus swap ;

   inline: /cby (hAr hAi hBr hBi --- hCr hCi) \ complex el-by-el div
\     Cr = (Ar*Br + Ai*Bi)/B2 
\     Ci = (Ai*Br - Ar*Bi)/B2 
\     B2 = Br*Br + Bi*Bi
\     where * = *by, element-by-element multiplication
      dup dup *by, other dup *by + (B2) push
      negate *cby peek /by swap pull /by
      (Cr) "_Cr" naming, swap (Ci) "_Ci" naming ;

   inline: acceleration (hAc hOmega dOmega hr hdr hddr --- hAp) 
{     Inertial accleleration of point p in a rotating frame:
         Ap = Ac + ddr + 2*Omega x dr + dOmega x r + Omega x (Omega x r)
      where:
         Ac = acceleration of point c fixed in rotating frame
         Omega = angular velocity of rotating frame
         dOmega = angular acceleration of rotating frame
         r = vector from point c to point p
         dr = rate of change of vector length from c to p
         ddr = derivative of rate of change of vector length from c to p

      All variables are vectors in 3 dimensions stored as 3x1 columns 
      within incoming 3xN matrices, where each column N pertains to a 
      point in time.
      The three elements of each vector are its components as measured 
      against the axes of the rotating frame.
}
      3 roll, 3 pick *cross plus (dOmega x r + ddr)
      3 pick, 2 roll *cross dup plus plus (+ 2*Omega x dr)  
      rev over *cross *cross less (- Omega x (r x Omega) )
      plus (+ Ac) ;

   inline: acos (hC --- hA) \ inverse cosine
      hand these rows ones, over dup *by, less sqrt swap atan2 ;
   
   inline: asin (hC --- hA) \ inverse sine
      hand these rows ones, over dup *by, less sqrt atan2 ;

{" This word is written in infix; gather its text and run eval:

   function (D) = btt(P, t, dt, f) { // Bottom or top tracker 
   /* For P(t) and interval dt, track bottoms if f .eq. 0 or track 
      tops if f .ne. 0.

      Examples:
         if(missing("sine")) source("mmath.v");               \

         (P, t) = sine(1, 0.01*2*pi, 0, (dt=3), (N=120));     \
         P += uniform(dt/100, N);                             \
         plot([btt(P, t, dt, 1) , btt(P, t, dt, 0) ,  P], t);

      // This shows tracker curves A and B computed inside btt: 
         if(missing("sine")) source("mmath.v");           \
         (P, t) = sine(1, 0.01*2*pi, 0, (dt=3), (N=120)); \
         P += uniform(dt/100, N);                         \

         btt.purgeAB = no;      \
         C0 = btt(P, t, dt, 0); \
         A0 = btt.A;            \
         B0 = btt.B;            \

         C1 = btt(P, t, dt, 1); \
         A1 = btt.A;            \
         B1 = btt.B;            \

         plot([A1 , B1 , P , A0 , B0], t); \
         pause; \
         plot([C1 , C0 , P], t);
   */
      { purgeAB = yes; }

      IF(f) // track tops
         A = mmax1(P, t, dt);
         B = mmin1(A, t, dt);
      ELSE // track bottoms
         A = mmin1(P, t, dt);
         B = mmax1(A, t, dt);
      THEN;
      C = looking(P, (A==B));
      D = looking(P, ((lag(C!=P, 1) && C==P)));   

      if(purgeAB) A = B = C = purged;
      else C = purged;
   }

"} eval \ parse infix into an inline and add to library

   inline: c_resultant (hA --- hR) \ resultant of complex components
{     For complex vector components in each column of matrix A, compute
      the complex resultant, R, of each.  

      Incoming complex matrix A has M rows and N columns.  

      Each column of A contains complex vector components.  The kth 
      row in N-by-1 vector R contains the vector resultant of the M
      complex terms in the kth column of A.
}
      this is_complex not
      IF "c_resultant" stknot return THEN

      hand \ if A is a complex NUM, make it into a 1-by-1 MAT

      (hA) real-imag (hAr hAi)
      (hAr hAi) mag (hRi) swap mag (hRr) swap (hRr hRi)
      (hRr hRi) complex (hR)
   end

   inline: caccel (hAr hAi hOmega --- hVr hVi) \ complex harmonic accel
\     Accelerations in complex harmonic response at imaginary fre-
\     quencies Omega.  See notes in word crate.
\        Vr + iVi = -Omega^2*(Ar + iAi)
      dup *by negate rot over diagpost, rot rot diagpost ;

   inline: cdisp (hqdd hOmega --- hq) \ complex displacement from accel
\     qdd and q are stored as complex; Omega is real.
      swap real-imag rot
      these rows ones swap /by negate caccel 
      complex
   end

   inline: cmag (hAr hAi --- hM) cmagphase drop ; \ magnitude of A

   inline: colrm (hA hF Fmax --- hA1 hF1) \ remove columns of A
{     Where row k of vector F is greater than Fmax, remove column k 
      of A and row k of F.

      Example:
         This should give about 5000 columns on average:

            10 10000 random 10000 1 random 0.5 colrm drop
 
}     (hA hF Fmax) that (hF) push
      (hA hF Fmax) those dims fill <= (hRake) this rev
      (hRake hA hRake) claw lop (hA1)
      (hRake hA1) pull rot (hF hRake) rake lop (hF1)
   end

   inline: cosine (A w phi dt N --- hW ht) \ cosine wave time history
\     Cosine wave of amplitude A, frequency w and phase phi:
\        W = A*cos(w*t - phi), t from 0 to (N-1)*dt
\     where w is rad/sec frequency, phi is deg phase angle.
      (dt N) uniform (t) push, (phi) rad/deg * "phi" book, (w) "w" book
      peek (t) w *f phi -d cos swap *f "_cos(t)" naming
      pull "_t" naming
   end

   inline: couple (hMaa hSVc hSVb hSVq hmaa hMSEc hMSEb hMSEq --- hMhh)
{     This is a general word to couple the mass, damping or stiffness
      matrices of two dynamic models formulated in terms of boundary 
      points and elastic modes when boundary points are held fixed.

      Example: The stack diagram above shows incoming matrix Maa for
      model SV and incoming matrix maa for model MSE.  Accompanying
      degree-of-freedom maps for model SV are:

         SVc - map of SV boundary points in Maa that couple to MSE
         SVb - map of the remaining SV boundary points in Maa
         SVq - map of SV fixed-boundary modal points in Maa

      and degree-of-freedom maps for model MSE are:

         MSEc - map of MSE boundary points in maa that couple to SV
         MSEb - map of the remaining MSE boundary points in maa
         MSEq - map of MSE fixed-boundary modal points in maa

      Note that the order of points listed in SVc and MSEc must be such
      that coupled points are correctly matched, and motions of coupled
      points must be referenced to the same coordinate system.

      Incoming Maa and maa have the following form (note that subma-
      trices, like Mcc, may actually be dispersed in various rows (and
      columns, by symmetry) of Maa according to map SVc):

                Mcc Mcb Mcq          mcc mcb mcq 
          Maa = Mbc Mbb Mbq    maa = mbc mbb mbq
                Mqc Mqb Mqq          mqc mqb mqq

      The returned coupled matrix Mhh has the following form:
  
               Mcc+mcc Mcb mcb Mcq mcq
                 Mbc   Mbb  0  Mbq  0
         Mhh =   mbc    0  mbb  0  mbq
                 Mqc   Mqb  0  Mqq  0
                 mqc    0  mqb  0  mqq 

      where submatrices are stacked exactly as shown with degrees-of-
      freedom as follows:

         coupled points c (common dofs in both SVc and MSEc)
         SV remaining boundary points (dofs in SVb)
         MSE remaining boundary points (dofs in MSEb)
         SV fixed-boundary modal points (dofs in SVq)
         MSE fixed-boundary modal points (dofs in MSEq)

      Note that incoming maps for b and q points should have 0 rows and
      1 column if there are none (as created with phrase 0 1 null).

}     into MSEq, into MSEb, into MSEc, into maa
      into SVq, into SVb, into SVc, into Maa

      Maa SVc catch dup SVc reach into Mcc
                    dup SVb reach into Mbc
                        SVq reach into Mqc
      Maa SVb catch dup SVb reach into Mbb
                        SVq reach into Mqb
      Maa SVq catch SVq reach into Mqq

      maa MSEc catch dup MSEc reach into mcc
                     dup MSEb reach into mbc
                         MSEq reach into mqc
      maa MSEb catch dup MSEb reach into mbb
                         MSEq reach into mqb
      maa MSEq catch MSEq reach into mqq

      Mcc mcc plus, Mbc pile, mbc pile, Mqc pile, mqc pile \ Mhh col 1
      Mbc bend, Mbb pile, mbb rows Mbb cols null pile,
         Mqb pile, mqq rows Mbb cols null pile, park       \ Mhh col 2
      mbc bend, Mbb rows mbb cols null pile, mbb pile,
         Mqq rows mbb cols null pile, mqb pile, park       \ Mhh col 3
      Mqc bend, Mqb bend pile, mbb rows Mqq cols null pile,
         Mqq pile, mqq rows Mqq cols null pile, park       \ Mhh col 4
      mqc bend, Mbb rows mqq cols null pile, mqb bend pile,
         Mqq rows mqq cols null pile, mqq pile, park       \ Mhh col 5
   end

   inline: cov (hA --- hCov) \ covariance of terms in A
\     Covariance matrix: Cov=C'*C, where C is A after normalizing
\     each column to have zero mean.
      these totals, those rows /f (hM) \ M(j) holds mean of col j
      those rows clone bend (hB)       \ B(i,j) holds mean of col j
      (hA hB) less (hC)                \ C(*,j) is A(*,j) less mean of j
      this bend swap star              \ Cov=C'*C
   end

   inline: cov2 (hX hY hLAG --- hR) \ covariance between X and Y
{     X and Y are column vectors, where each row is a step in time.  
      Row k of returned R contains the covariance of X and Y with 
      LAG(k).

      If d=LAG(k) is positive, Y is lagged by d rows: X'(n)*Y(n-d), and
      if d is negative, X is lagged by d rows: Y'(n)*X(n-d).

      Reference: Bendat, J. S. and A. G. Piersol, "Random Data Analysis
      and Measurement Procedures," Third Edition, p. 119.

      Local macro "norm" removes the mean and normalizes a vector by
      the variance.  To obtain normalized vectors, say for plotting,
      the macro can be run separately with expressions like:

          Xnorm = cov2.norm(X); // infix
          (hX) "cov2" "norm" localrun (hXnorm) \ postfix

      The mean and variance from last time are save in cov2.mean and
      cov2.sigma.
}
      [
      \ (hA --- hA1) subtract mean, normalize by variance
        "(hA) dup totals @ over rows / (hA nMean) - " \ A(i)=A(i)-Mean
        "dup dup *by totals @ (nVar) any? " + \ Var=sum(A(i)*A(i)
        "IF (nVar) sqrt / THEN (hA1)" + \ A1(i)=A(i)/sqrt(Var)
        "norm" macro
      ]
      hand "LAG" book 

    \ Subtract mean and normalize by variance; store transposed vectors
    \ for word cross-correlation:
      (hY) norm (hY) bend "Y" book 
      (hX) norm (hX) bend "X" book

      LAG rows 1st
      DO LAG I pry dup 0>
         IF X Y rot (hX hY d) 
         ELSE Y X rot abs (hY hX d)
         THEN (hX hY d) cross-correlation (hR)
      LOOP LAG rows parkn (hR) totals

      purged "X" book
      purged "Y" book
      purged "LAG" book
   end

   inline: crate (hAr hAi hOmega --- hVr hVi) \ complex harmonic rate 
{     Rates in complex harmonic response at imaginary frequencies 
      Omega, where:
         x = A*sin(Omega*t) = [Ar + j*Ai]*sin(Omega*t)
         xdot = Omega*A*cos(Omega*t) = [Vr + j*Vi]*cos(Omega*t)
      Omega is a complex diagonal matrix with zero real part, and the 
      imaginary part is stored as a vector in hOmega; harmonic rate is: 
         Vr + j*Vi = (0 +j*Omega)*(Ar + j*Ai) = -Omega*Ai + j*Omega*Ar
}
      dup push negate (hAi hOmega_neg) diagpost (hVr)
      swap pull (hAr hOmega) diagpost (hVr hVi)
   end

   inline: Cronin (Q Oct/min Hz --- %Resonance) \ Cronin's equation
{     Cronin's equation.  Fraction of oscillator steady state response 
      for low sweep rates:

         %Resonance = 1 - exp[-2.86*(ln2*SweepRate*Q*Q/Hz)^(-0.445)] 

      where SweepRate=(Oct/min)/60.

      Example.

         For Q of 20, sweep rate of 2 octaves per minute, 4 Hz oscil-
         lator:

            [tops@clacker] ready > 20 2 4 Cronin .
             8.6058E+01

         Says that a 4 Hz oscillator reaches 86% of resonant amplitude.
}
      [ 2 ln "ln2" book ]

      (Hz) "Hz" book
      (Oct/min) 60 / (Oct/sec) "SweepRate" book 
      (Q) "Q" book

      one
         ln2 SweepRate *
         Q Q * * Hz / -0.445 pow
         -2.86 * e^ 
      less 100 * (%)
   end

   inline: EIVrows (hPhi r --- hRows) \ rows of Phi that influence EIV
{     Defining rows of Phi that most influence the effective indepen-
      dence vector, EIV.

      The EIV gives a measure of the important rows in Phi that are 
      needed to discern orthogonality between the columns of Phi.

      Reference: Kammer, D. C., "Sensor Placement for On-Orbit Modal
         Identification and Correlation of Large Space Structures,"
         J. of Guidance, Control and Dynamics, Vol. 14, No. 2,
         March-April 1991.
 
      Factor r, less than one, constrains influential rows of Phi to
      ones that affect the EIV in the range from r*EIV(max) to EIV(max).

      The one row of Phi that influences EIV the least is removed on
      each iteration, and EIV is recomputed.  Iterating continues until
      only influential rows of Phi--those contributing r*EIV(max) and
      above--remain.
}
      [
       {" This local word provides the EIV on each iteration:

         \     EIV = diag[Phi * inv(Phi'*Phi) * Phi']

         [ "solveSSL" exists? IF "solveSSL" "solve" overload THEN
           "mpySSL" exists? IF "mpySSL" "star" overload THEN
         ]
         (hPhi) push peek (hPhi) dup bend dup
         pull (hPhi) star solve stardiag (hEIV)

       "} "EIV" inlinex

         1E-8 is eps \ negligible term in EIV

       \ Showing progress each iteration:
         no "tic" book \ to turn on, say: yes "EIVrows" "tic" bank

      ]
      true 1 NUM stkok and, 2 MAT stkok and not
      IF "EIVrows" stknot return THEN

      "r" book

    \ Removing rows of initial Phi that are null:
      (hPhi) dup nullr (hR0) push
      (hPhi) 1st those rows items peek (hR0) rake "Rows" book drop
      (hPhi) pull (hR0) rake "Phi" book drop

    \ Make zero those rows of Phi that contribute max*eps or less:
      Phi dup EIV dup maxfetch 2drop eps * filter 0<> abs
      those cols clone *by (hPhi)

    \ Removing rows of Phi that are null or negligible:
      (hPhi) dup nullr (hR0) push
      Rows peek (hR0) rake "Rows" book drop
      (hPhi) pull (hR0) rake "Phi" book drop

      BEGIN
         tic IF Phi rows .i THEN

         Phi EIV (hEIV) dup minfetch drop "row" book (min)
         swap maxfetch 2drop r * (max*r)
         (min max*r) <
      WHILE \ min < max*r, eliminate its row
         row Phi rows teeth Phi that rake "Phi" book drop
         Rows swap rake "Rows" book drop
      REPEAT
      Rows cop "_Rows" naming
      freed is Phi, freed is Rows
   end

   inline: db10 (hP --- hPdb) \ 10*log10(P): dB when P is a power ratio
      this type NUM = push hand abs log10 ten *f pull IF ontop THEN ;

   inline: db20 (hP --- hPdb) \ 20*log10(P): dB for pressure ratio P
      this type NUM = push hand abs log10 20 *f pull IF ontop THEN ;

   inline: dbcheatsheet ( --- ) \ some decibel examples
      [ {"
      The definition of dB is 10*log10(P/Pref) where P/Pref is a ratio 
      of power P to reference power Pref.  This means that ratios of 
      things derived from the square-root of power, like gRMS, must be 
      squared when computing dB--hence the use of 20*log10(R) which is 
      equivalent to 10*log10(R*R).

      Word db10 applies to power (like power spectral density, PSD), 
      and word db20 applies to measures proportional to square-root 
      of power (like gRMS).

        "} left justify 2 indent says blurb
        "     dB Cheat Sheet" "          db10    db20" pile
        "  Ratio  power  pressure" pile (hHead)
        list: .01 .1 3 / .05 .1 .125 .25 .5 .707 1 
              1.414 2,4 8 10 20 30 100 ;
        dup db10, over db20 park park "%7.3f%7.1f %7.1f" format pile
        20 indent makes cheat
      ] cheat blurb dot nl dot nl
   end

   inline: derivative (hX ht --- hdX) \ change in X by finite difference
{     Vector function X is given at N discrete points that match the N 
      rows of parametric, increasing vector t.  Derivative by finite 
      difference is:

            dX(t)/dt = [X(t+h) - X(t)]/h, as h approaches zero

      Spacing between points need not be uniform, but the value of h 
      given by t(N)*hfac (hfac is set below in brackets) is assumed to 
      be smaller than any step t(k+1) - t(k).

      For discontinuous data, X should be a curve fitted to the data 
      with extra points between.  Later, derivatives at just the data 
      points can be raked from output vector dX.  But note that inte-
      grating such curves can produce an integral that varies signifi-
      cantly from the original curve.  Spline fitting, using s3coef and
      s3eval may produce much better results but even they can drift 
      considerably.

}     its these rows pry, hfac star (h) push, 2dup swap park (tX)
      swap peek (h) +d lerp swap less pull /f [ 1e-6 is hfac ] ;

   inline: diagpost_complex (hAr hAi hVr hVi --- hBr hBi)
\     Post-multiplication of complex A by a complex diagonal matrix 
\     stored in vector V:
\        Br = Ar*Vr - Ai*Vi
\        Bi = Ai*Vr + Ar*Vi
\     where * denotes operation by word diagpost.

      "Vi" book "Vr" book "Ai" book "Ar" book
      Ar Vr diagpost, Ai Vi diagpost less (hBr) "_Br" naming
      Ar Vi diagpost, Ai Vr diagpost plus (hBi) "_Bi" naming
      freed is Ar, freed is Ai, freed is Vr, freed is Vi
   end

   inline: diagpre_complex (hVr hVi hAr hAi --- hBr hBi)
\     Pre-multiplication of complex A by a complex diagonal matrix 
\     stored in vector V:
\        Br = Vr*Ar - Vi*Ai
\        Bi = Vi*Ar + Vr*Ai
\     where * denotes operation by word diagpre.

      "Ai" book "Ar" book "Vi" book "Vr" book
      Vr Ar diagpre, Vi Ai diagpre less (hBr) "_Br" naming
      Vi Ar diagpre, Vr Ai diagpre plus (hBi) "_Bi" naming
      freed is Ar, freed is Ai, freed is Vr, freed is Vi
   end

   inline: dircos (hX hZ --- hC) \ create direction cosine matrices
{     Incoming X and Z have M columns to determine M direction cosine
      matrices.
      Successive 3x3 dir cos matrices are parked in C, making it 3x3M.

      Each direction cosine matrix, Ck = Cnm, is used in the form
                           xn = Cnm * xm
      to express vector xm in system m as vector xn in system n, given
      the following two vectors expressed in n:

         Xk = vector from origin of n to a point in the +(x-z)
              plane of m
         Zk = vector from origin of n to a point on the z-axis of m

      where Xk and Zk are the kth column in X and Z, respectively.
}     
      dup rot (Z X) *cross (Ym),
      (Z Ym) dup other (Ym Z) *cross (Xm), (Zm Ym Xm) rev swap
      (Xm Ym Zm) park park, norm 
      three those cows = IF return THEN
      these cows once over items swap
      three slash fold bend, chain ndx catch (hC) 
   end

   inline: dircos' (hC --- hC') \ transpose direction cosine matrices
{     Transpose the 3x3 submatrices within 3x3M matrix C.
      This demo shows that elements go where they are supposed to:

         1st 9 items 3 fold, dup push, 30 20 
         DO peek I +d park 10 +LOOP, pull drop (hC)
         mformat push "%4.0f" mformatset 
         dup nl .m nl dircos' (hC') nl .m nl pull mformatset

}     
      (hC) bend chain, three fold  
\     These are identical to the last two phrases in dircos:
      these cows once over items swap
      three slash fold bend, chain ndx catch (hC') 
   end

   inline: dircos9 (hABC --- hC) \ creating direction cosine matrices
{     Creating direction cosine matrices for N reference frames. 

      Each column of incoming 9-by-N matrix ABC contains three 3d 
      vectors: A (rows 1-3) defines the origin, B (rows 4-6) defines
      a point on the +Z axis, and C (rows 7-9) defines a point in the 
      +X+Z plane.

      Each column of outgoing matrix C contains the 9 elements of the
      direction cosine matrix (a 3-by-3 stored by columns) for the
      corresponding column of ABC.
}
      [ 1 3 items "A" book, 4 3 items "B" book, 7 3 items "C" book ]

      (hABC) this A ndx reach push \ origin A onto local stack
      (hABC) this C ndx reach peek less (X)
      swap (hABC) B ndx reach pull less norm (Zm) \ Z unit vec for C

\     Creating the Y and X unit vectors for C:
      (X Zm) this rot (Zm X) *cross (Y) norm (Ym) \ Zm cross X = Y
      (Zm Ym) this other (Ym Zm) *cross (Xm) \ Ym cross Zm = Xm

      (Zm Ym Xm) swap pile swap pile (hC)
   end

   inline: eigchk (C Lr Li Ar Ai --- Br Bi) \ check eigensolution
{     Ref: Luenberger, D. G., "Introduction to Dynamic Systems,"
      Wiley, 1979.
      The eigensolution check is complex matrix B given by
                B = C*A - A*L
      which should equal zero (Luenberger, p. 82).  In detail, have
                Br + iBi = C*(Ar + iAi) - (Ar +iAi)*(Lr + iLi)
      where Lr + iLi is the complex diagonal n-by-n eigenvalue matrix
      stored as an n-by-1 matrix.  The real and imaginary parts of the
      check, which should both be null, are written as
                Br = C*Ar - Ar*Lr + Ai*Li
                Bi = C*Ai - Ar*Li - Ai*Lr
}
      (Ai) into Ai, (Ar) into Ar
      (Li) into Li, (Lr) into Lr
      (C) into C

      C Ar *, Ar Lr *diagonal -, Ai Li *diagonal + (Br)
      C Ai *, Ar Li *diagonal -, Ai Lr *diagonal - (Bi)
   end

   inline: Frms (hP hF --- hRMS) \ rms integral of P over frequencies F
\     Row i of P goes with row i of frequencies in F.

\     Integrating P over all frequencies:
      swap bend (hP') \ row k of P' is the frequency history of PSD k
      swap (P' hF) trap (hR)

\     The last column of R is the value of integration up to the
\     last frequency; take the square root of it
      (hR) these cols ndx catch, sqrt (hRMS) "_RMS" naming
   end

   inline: Frms_log (hP hF --- hRMS) \ integral of P over frequencies F
\     Logarithmic trapezoidal integration.
      logarea totals (hA) sqrt (hRMS) "_Frmslog" naming
   end

   inline: groups (hR g1 g2 --- hG) \ groups of consec numbers with gaps
{     Determine groups of consecutive numbers from g1 to g2 that do not
      include the numbers listed in R.  Return in two-column matrix G 
      the pairs of numbers defining from-to ranges of the allowable 
      values.  It is assmued that g1 < min(R) and g2 > max(R)
      Example: R = [ 3 4 9 11 ], g1 = 1, g2 = 24.  Then from-to pairs 
      in G are:
               G = [(1 2), (5 8), (10 10), (12 24)]
      where the first number in each pair is in col 1, the second in 
      col 2.
      Here is the phrase to run this example: 
         list: 3 4 9 11 ; 1 24 groups .m
      See express program nas.voc for an example using groups to create 
      SESET and SEBSET cards.
}     makes g2 says g1, right sort says R
      R -1 +d, g2 pile onto H1, g1 R 1 +d, pile onto L1
      g1 R pile onto sH, R g2 pile onto sL
      L1 its sL - rake lop, H1 its sH - rake lop park ; 

   inline: IC+ (hX hV --- hX1) \ add init conditions V(k) to X(k,t)
{     Add the first element of V to all elements in the first row of X,
      the second element of V to all elements in the second row of X, 
      ..., .
      Test: 
         3 5 null, list: 1 2 3 ; IC+ .m (adds 1 to row 1, 2 to row 2, 
      ...)
}     hand chain those cows clone plus 
   end

   inline: integ (hY hf --- hIY) \ integral Y*df, from 1st to last f
\     Total integral of Y*df from first f to last f.
      integral one endmost
   end

   inline: integral (hY hf --- hIY) \ integral Y*df, all terms
\     All terms in integral Y*df from first f to last f.
      swap bend swap trap bend
   end
{
   Need word /sym before these words will work.
   inline: Kdecomp (hF hK --- hx) \ solve K*x = F for x
\     Stiffness matrix decomp with zero rows and columns removed,
\     and small k added to diagonals.
      [ 1e-6 is k ] dup nullr this push, dup partition lop lop lop
      these rows identity k *f plus (K + k),
      swap peek rake lop swap /sym
      peek rows over rows - over cols null swap pull tier ;

   inline: Kinv (hK --- hKinv) \ inverse of symmetric stiffness matrix
      these rows identity swap Kdecomp ;
}
   inline: locrect (hRTZ --- hXYZ) \ points from cyl into rect sys
\     Incoming 3xN matrix RTZ defines locations of N points in
\     cylindrical coordinates, where angles T are in degrees.
\     NOTE: angles in RTZ are in degrees, not radians
      this two reach rad/deg *f makes Theta
      this one reach (R) dup Theta cos *by (X) 
      swap (R) Theta sin *by (Y) 
      rot three reach (Z) 
      pile pile (hXYZ)
   end

   inline: lsq (b A --- x) \ least-squares solution, x=[inv(A'*A)]*A'*b
      back over bend again, rot star rev, swap star solve ;

   inline: median (hC --- hCm) \ row k of Cm holds median across C(k)
\     Best if number of cols in C is an odd number.
      bend colsort bend these cols two slash integer tic ndx catch
   end

   inline: Miles (hPSD hF nQ --- hGrms) \ Mile's equation for random PSD
\     Mile's equation for simplified Grms response to PSD at F: 
\        Grms = sqrt[(pi/2)*Q*F*PSD]
\     where PSD values are in units g^2/Hz, corresponding frequencies
\     F are in Hz, and Q is dynamic amplification factor (1/2*zeta).
      *f *by pi two slash *f sqrt (hGrms)
   end

   inline: mmax2 (hVm0 hV ht ntau --- hVm)
{     To overcome the delay of mmax1() when there are many points in
      V(t) (20,000 to 30,000), and it is run over and over in a real
      time system, this word receives previously computed moving max
      values, Vm0(t), saved by the calling word, and just computes 
      Vm1(t) for new, added rows and appends it to the old ones to 
      make Vm = [Vm0 ; Vm1].
}
      3 pick (hVm0) rows any
      IF (hVm0 hV ht ntau)
         (ntau) "tau" book "t" book "V" book

       \ Trim rows of Vm0 back to the point of the last local min:
         (hVm0) 1st over rows items (hRows)
         (hVm0 hRows) over delta 0< rake (hR0 hR1) lop
         (hR1) dup rows 1st max pry "r" book
         (hVm0) 1st r items reach (hVm0)
         (hVm0) V rows over rows - 0>

         IF (hVm0)
            1st over rows items (hRows) V rows teeth V swap
            rake lop (hVm0 hV2) pile "V" book

            1st r 1- 1 max items V rows teeth (hR) push

            V peek (hV hR) rake (hV0 hV1)
            t pull (ht hR) rake lop (ht)
            (hV0 hV1 ht) tau mmax1 (hV0 hV1) pile
         THEN (hVm)
         purged "t" book
         purged "V" book
      ELSE
         (hVm0 hV ht ntau) mmax1 (hVm0 hVm) lop (hVm)
      THEN
   end

   inline: mmin2 (hVm0 hV ht ntau --- hVm) \ moving minimum
{     To overcome the delay of mmin1() when there are many points in
      V(t) (20,000 to 30,000), and it is run over and over in a real
      time system, this word receives previously computed moving min
      values, Vm0(t), saved by the calling word, and just computes 
      Vm1(t) for new, added rows and appends it to the old ones to 
      make Vm = [Vm0 ; Vm1].
}
      3 pick (hVm0) rows any
      IF (hVm0 hV ht ntau)
         (ntau) "tau" book "t" book "V" book

       \ Trim rows of Vm0 back to the point of the last local max:
         (hVm0) 1st over rows items (hRows)
         (hVm0 hRows) over delta 0> rake (hR0 hR1) lop
         (hR1) dup rows 1st max pry "r" book
         (hVm0) 1st r items reach (hVm0)
         (hVm0) V rows over rows - 0>

         IF (hVm0)
            1st over rows items (hRows) V rows teeth V swap
            rake lop (hVm0 hV2) pile "V" book

            1st r 1- 1 max items V rows teeth (hR) push

            V peek (hV hR) rake (hV0 hV1)
            t pull (ht hR) rake lop (ht)
            (hV0 hV1 ht) tau mmin1 (hV0 hV1) pile
         THEN (hVm)
         purged "t" book
         purged "V" book
      ELSE
         (hVm0 hV ht ntau) mmin1 (hVm0 hVm) lop (hVm)
      THEN
   end

   inline: mnorm (hVL hVR hM --- hVL1 hVL2) \ normalize complex modes
\     Normalize left and right eigenvectors, VL and VR respectively, so
\     VL' * M * VR equals identity, where VL' is the conjugate transpose
\     of VL.

      swap push push (hVL)
      this (hVL) conj transpose pull (hM) star peek (hVR) stardiag
      -0.5 pow >square (hF)
      swap over (hVL hF) star pull (hVR) rot star
   end

   inline: modal_work (hPHI hFHz --- hWork) \ work in mode PHI of FHz
{     Work done at each row (dof) in mode PHI.  
      Mode PHI is assumed to be normalized to unity generalized mass, 
      so the summation of over all rows in Work (total work) should 
      equal the square of FHz (expressed in rad/sec instead of Hz).

      Modal work equations:

         K*PHI = F (forces at dofs due to deformations PHI)
         F#PHI = W (work done by F, where # is term-by-term multiply)

         Finding F and computing W:

            Strain energy in mode PHI is equal to frequency squared:

               PHI'*K*PHI = w^2

            Premultiplying by PHI gives:

               PHI*PHI'*K*PHI = PHI*w^2

            This equation is of the form A*F=b, where:

               A = PHI*PHI' (symmetric and probably singular)
               F = K*PHI
               b = PHI*w^2

            Solve for F using the conjugate gradient method (word cgm).

            Then the work done is:

               W = F#PHI

      Test case:

         1e5 seedset 10 1 random 3 1 null pile 
         "PHI" book 20 "FHz" book
         PHI dup FHz modal_work park dup .m

       \ Checking that total work equals w^2:
         (hWork) totals 2nd pry 
         FHz 2pi * dup * park .m

          cgm solution: max error: 3.6380E-12
           Row 1:   0.7788     2965
           Row 2:   0.5322     1385
           Row 3:  0.01319   0.8507
           Row 4:   0.7237     2560
           Row 5:   0.9186     4125
           Row 6:   0.4616     1042
           Row 7:   0.4717     1088
           Row 8:   0.6406     2006
           Row 9:   0.2896    409.9
           Row 10:  0.2077      211
           Row 11:       0        0
           Row 12:       0        0
           Row 13:       0        0
}
      "FHz" book
      "PHI" book
      PHI dup bend * "A" book         \ A=PHI*PHI'
      PHI FHz 2pi * dup * *f "b" book \ b=PHI*w^2

\     Matrix A (PHI*PHI') is symmetric but probably is singular.  The
\     conjugate gradient method will find a solution:
         b A cgm "F" book \ using cgm to solve A*F=b for F

\     Checking the solution:
         A F * b - abs maxfetch 2drop
         " cgm solution: max error:" . . nl

         purged "A" book \ get rid of huge A
         purged "b" book

\     Computing work done, F#PHI:
         F PHI *by "_W" naming

         purged "PHI" book
         purged "F" book
   end

   inline: modes (hA --- hPhi hOmegSq) \ separate A into shape, freq
{     Extract M-by-N matrix of eigenvectors and N-by-1 matrix of
      eigenvalues from matrix A.

      Matrix A from eig2 is (1+M)-by-N.  The first row contains the
      N eigenvalues; rows 2 through 1+M contain the eigenvectors of
      M degrees-of-freedom.
}
      1st those rows teeth rake 
      "_Phi" naming, swap bend "_OmegSq" naming ;

   inline: msigma (hA ht w d --- hM) \ moving mean and variance
{     Returned M(t) has four columns of data computed within a window 
      of width w steps that is moved over A(t) from latest to earliest 
      in steps of d.  

      Column 1 contains the mean of data within the window, column 2 
      contains its standard deviation, columns 3 and 4 hold the max-
      imum and minimum, respectively.

      After computing mean and standard deviation for each window posi-
      tion, they are linearly interpolated from window-to-window times
      (spaced at d), to times in t (spaced at 1), so that rows of M cor-
      respond to times in A.  Maximum and minimum data is presented for
      each window position and is not interpolated.

      This word contains a local macro, stats, for computing the mean
      and standard deviation of A.  It can be run from anywhere in the 
      program using phrases like:

         (Mean, Variance) = msigma.stats(A); // infix

         (hA) "msigma" "stats" localrun (hMean hVariance) \ postfix
}
      [
        {" (hA --- nMean nSigma) \ mean and standard deviation of A
          (hA) dup totals @ over rows / (hA nMean) 
          (hA nMean) dup push - (hA)             \ A(i)=A(i)-Mean
          dup dup *by totals @ (nVar) dup 0>     \ Var=sum(A(i)*A(i))
          IF (hA nVar) swap rows 1- 1 max / sqrt \ sigma=sqrt(Var/(N-1))
          ELSE (hA 0) lop
          THEN pull swap (nMean nSigma)
        "} "stats" macro
      ]
      "d" book "w" book
      (ht) push (hA) "A" book

      A rows w - 0 max 1+ A rows thrulist
      "L" book \ (rows(A)-d+1):rows(A)

      depth push

      A rows w - 0 max d / integer 1+ (Rcols) 1st
      DO A L reach (hA) 

         (hA) dup push stats (nM nS) 
         peek maxfetch 2drop (hMx) 
         pull minfetch 2drop (hMn) 4 listn bend (hM)
         
         L d - dup 1st pry 1st <
         IF (hL) dup 1st >= rake lop THEN "L" book
      LOOP

      depth peek - revn       \ reverse rows
      depth pull - pilen (hM) \ pile rows

      (hM) dup rows 2 <
      IF dup pile THEN (hM)

    \ Separate columns of M for different mapping methods.  Interpo-
    \ lation is used for mean and standard deviation, and a lookup 
    \ method is used for maximum and minumum to maintain the window 
    \ values in stepwise fashion:
      (hM) dup 1st 2 items catch "M1" book \ use lerp method
      (hM) 3rd 2 items catch "M2" book     \ use look (step) method

    \ Map rows in current M, spaced at d steps, to rows at times in t:
      peek (ht)
    \ Make a list of rows in time t that correspond to rows of M that
    \ are in steps of d:
      list: peek (ht) rows M1 rows 2 - (x2) any? (false or x2 true)
         IF (x2) 1st DO dup d - LOOP THEN 1st
      end reversed (hRows)
      (ht hRows) reach (ht1) \ window times t1

      (ht1) dup (ht1) M1 park (hXY) \ table for interpolation
      (hXY) peek (ht) M1 cols clone (hXY hx) lerp (hM1)

      (ht1 hM1) swap (ht1) M2 park (hXY) \ table for lookup
      (hXY) pull (ht) M2 cols clone (hXY hx) look (hM2)

      (hM1 hM2) park "_msigma" naming

      purged "M1" book
      purged "M2" book
      purged "A" book
      purged "L" book
   end

   inline: mstats (hA n --- hB) \ moving statistics
{     Fri Dec  2 04:56:54 PST 2011

      For column vector A, compute statistics on each set of n rows.

      Returned matrix B of four columns holds minimum, average, maximum
      and variance taken over each n rows of A.

      Returned "variance" is an unbiased estimate of the variance for
      n independent samples, obtained by dividing by n - 1 for n >= 2:
         v = (Sum[(x(k) - xmean)^2])/(n - 1)

      Reference: Sokolnikoff, I. S., and R. M. Redheffer, "Mathematics
         of Physics and Modern Engineering," 1958, pp. 669-670.

      Example.  This example matches the one in function stats1() in
      file src/math.c, which is correct by inspection.

         Function stats1() analyzes each row of A.

         Since mstats() works on a column vector, each row I of A is
         an input column to mstats().  A dummy first set of rows is
         needed to get things rolling, using [A[I,*]' ; A[I,*]'].

         The endmost row from mstats() is equivalent to stats1() out-
         put, and becomes a row of X.  Then stats1(A) is run to show
         matching results.

            >> A = [1 ; 2 ; 4]*[1 , 2 , 3];

            .m(A); nl;

            X = pilen( \
               (DO(rows(A), 1) \
                   endmost(mstats([A[I,*]' ; A[I,*]'], cols(A)), 1) \
                LOOP), rows(A)); \

            .m(X); nl;

            .m(stats1(A)); nl; <<

         These are the results from running the expressions above at
         the ready prompt (they can all be copied at once):

            [tops@plunger] ready > >> A = [1 ; 2 ; 4]*[1 , 2 , 3];

            >>             .m(A); nl;
             Row 1:        1        2        3
             Row 2:        2        4        6
             Row 3:        4        8       12

            >>             .m(X); nl;
             Row 1:        1        2        3        1
             Row 2:        2        4        6        4
             Row 3:        4        8       12       16

            >>             .m(stats1(A)); nl; <<
             Row 1:        1        2        3        1
             Row 2:        2        4        6        4
             Row 3:        4        8       12       16

            [tops@plunger] ready > 
}
      "n" book (hA) push
      peek n mmin (hMin) \ Min
      peek n ma (hMean)  \ Mean
      peek n mmax (hMax) \ Max

      pull (hMin hMean hMax hA) 2 pick (hA hMean)

      (hA hMean) - (hDev)
      (hDev) dup *by n mtot n 1- / \ (Sum[(A(k) - Mean(k))^2])/(n - 1)

      (hMin hMean hMax hVar) 4 parkn (hB)
      "_mstats" naming
   end

   inline: nonesame (hA --- hA1) \ no terms in A are the same
{     Wed Feb  8 02:00:26 PST 2012.  Add small numbers to the terms of
      A so none are the same.

      Assumes numbers in A are on the order of 1 and above.

      Add a small number to A(i) derived from its 0based index, i, and
      the number of rows R:

         A(i) = A(i) + i/(R^4)

      R is limited to 100 max to avoid losing digits.  Without this
      limit, the following example with 50,000 rows will fail to create
      values that are all different.
      
      Example:

         [tops@plunger] ready > 50000 1 random 1000 * integer

          stack elements:
                0 matrix: _byterm1  50000 by 1
          [1] ok!
         [tops@plunger] ready > dup nodupes
       
         From word nodupes, the following shows that there are only 
         1000 unique numbers (and you can safely bet that they range
         from 0 to 999):

          stack elements:
                0 matrix: _nodupes  1000 by 1
                1 matrix: _byterm1  50000 by 1 
          [2] ok!
         [tops@plunger] ready > swap nonesame

          stack elements:
                0 matrix: _nonesame  50000 by 1
                1 matrix: _nodupes  1000 by 1
          [2] ok!
         [tops@plunger] ready > nodupes

         The following shows that after running nonesame, all numbers 
         are unique, since no rows were lost with word nodupes:

          stack elements:
                0 matrix: _nodupes  50000 by 1
                1 matrix: _nodupes  1000 by 1
          [2] ok!

         [tops@plunger] ready > dup pminmax

         This shows the smallest and largest numbers:

          stack elements:
                0 number: 999.00048089
                1 number: 2.27E-06
                2 matrix: _nodupes  50000 by 1
                3 matrix: _nodupes  1000 by 1
          [4] ok!
         [tops@plunger] ready > 
}
      (hA) 1 over rows 10 max 100 min dup * dup * / (x)
      (hA x) over rows uniform + (hA1) "_nonesame" naming
   end

{ ---

   Because it uses sorting, the following version of nonesame does not
   guarantee the same outcome for earlier terms as new terms are added 
   to A in a real time system.  

   If a new term matches an earlier one, the earlier one may be bumped
   differently than it was in the past.  It depends upon the outcome of
   the sort--which rows are swapped--when there are duplicates.

   inline: nonesame (hA --- hA1) \ bump terms until none are the same
{     Tue Feb  7 19:24:53 PST 2012.  Terms in vector A are bumped until
      none are the same.

      Adapted from word time_vec in mfil.v.

      Values in A are bumped by diff=0.01. 

      Just bumping a value may get it above its equal partner but might
      make it equal to the neighbor above.  And more than two might be
      equal in the first place, so bumping just one is not enough.

      The BEGIN ... WHILE ... REPEAT loop below cycles around until
      no values are equal.  It uses the fact that a true flag equals
      -1, so "(-1) abs X *" gives the add-on 1*X needed when the match
      flag is true, and zero when it is false and equal to 0; +1 and
      -1 lag is used to align the rows going up and down.

      Not very fast for large vectors.  Might work faster for very large
      vectors if values are large and diff is large.

      Examples:

      1. List A has four equal values:

            [tops@plunger] ready > list: 0 1 0 0 0 ; "A" book A bend .m
             Row 1:        0        1        0        0        0 

         This word keeps adding diff until no values are equal.  When
         diff = 0.01, this new A is obtained:

            [tops@plunger] ready > 0.01 "nonesame" "diff" bank
            [tops@plunger] ready > A nonesame (hAnew) bend .m
             Row 1:     0.03        1     0.02        0     0.01 

      2. Large list:

            [tops@plunger] ready > 1 "nonesame" "diff" bank
            [tops@plunger] ready > 1000 1 random 1000 * integer "A" book

         Running nodupes shows 1000 - 649 duplicates in A:
            [tops@plunger] ready > A nodupes

             stack elements:
                   0 matrix: _nodupes  649 by 1 
             [1] ok!

         Now run nonesame on A and run nodupes on its output:
            [tops@plunger] ready > A nonesame

             stack elements:
                   0 matrix: _nonesame  1000 by 1
                   1 matrix: _nodupes  649 by 1
             [2] ok!

         No duplicates in the output from nonesame:
            [tops@plunger] ready > nodupes

             stack elements:
                   0 matrix: _nodupes  1000 by 1
                   1 matrix: _nodupes  649 by 1
             [2] ok!
            [tops@plunger] ready > 
}
      [ 0.01 "diff" book ]

      "nonesame" ERRset

      (hA) 1st over rows items park (hB) yes sort (hB)
      (hB) dup 1st catch (hA) 
{
      Temporarily affix a large negative number so the loop will
      not fail (and run forever) on bad data, such as all values
      being equal:
}     (hA) -INF swap pile (hA) \ append A to a large negative number

      (hA) \ process sorted A
      BEGIN (hA) dup dup -1 lag - 0= (hf)
         (hf) abs diff * (hdA) 1 lag
         (hA hdA) dup totals @ 0<> (f)
      WHILE (hA hdA) + (hA)
      REPEAT (hA hdA) drop

      2nd over rows 1- items reach (hA) \ remove the negative number

      (hB hA) swap 2nd catch (hR) \ make R1 to recover original order:
      (hR) 1st over rows items park yes sort 2nd catch (hR1)

      (hA hR1) reach (hA1) \ back in original order
      "_nonesame" naming

      ERR
   end
--- }
   inline: nonneg (hV1 --- hV) \ make the changes in V nonnegative
    \ Wed Feb 22 11:38:29 PST 2012.  Assumes V1 is close to being an
    \ ever-increasing function, like a logistic growth function.

      "V" book V rows true * "r" book
      BEGIN
         V dup delta 0< not (hf) dup totals @ r = not
         IF (hV hf) looking "V" book false
         ELSE 0 "V" book (hV hf) drop true (hV true)
         THEN (f)
      UNTIL (hV)
   end

   inline: normsig (hA --- hA1 nMean nSigma)
{     Remove the mean of A and then normalize by its standard devia-
      tion, sigma, to make A1.
 
      Reference: Bendat, J. S. and A. G. Piersol, "Random Data Analysis
      and Measurement Procedures," Third Edition, p. 395.
}
      (hA) dup totals @ over rows / (hA nMean) dup "mean" book - 
      dup dup *by totals @ (nVar) dup 0>
      IF (nVar) over rows 1- 1 max / sqrt dup rev / swap
      THEN (nvar) "sigma" book (hA1) "_normsig" naming
      (hA) mean sigma
   end

   inline: palign (hY1 ht1 hY2 ht2 --- hY1 hY2 ht) \ Y1 and Y2 at t
\     Align rows of discrete data in all columns of Y1(t1) and Y2(t) 
\     so each has the same number of rows for the same set of points, 
\     t=[t1, t2]: Y1(t), Y2(t)

\     Y1 and Y2 can have a different number of columns.

      other over pile yes sort nodupes push
      (hY2 ht2) bob park peek those cols nit clone lerp (hY2) rev
      (hY1 ht1) bob park peek those cols nit clone lerp (hY1) 
      bob pull (hY1 hY2 ht)
   end

   inline: palign_log (hY1 ht1 hY2 ht2 --- hY1 hY2 ht) \ Y1 and Y2 at t
\     Interpolation on logarithm values; does not return rows that hold
\     bad numbers due to bad logs.

\     Y1 and Y2 can have a different number of columns.

      four 1st DO log10 three roll LOOP
      palign
      three 1st DO 10^ rot LOOP
      other numbad across 0<>
      other numbad across 0<> or push
      three 1st DO peek rake trash rot LOOP 
      pull trash
   end

   inline: ploc (hCab A* hXYZb --- hXYZa) \ express sys B vecs in sys A
{     Each column in 3xn matrix XYZb is a vector in system B; return 
      corresponding vectors in system A, given by
            XYZa = Cab*(XYZb - A*)
      where Cab rotates B vectors into A, and A* is the vector from 
      the origin of B to the origin of A, expressed in system B
}     swap those cows, clone less star ;

   inline: rats (hKxx hRBx --- hSx) \ force separation ratios
{     For incoming stiffness matrix Kxx and corresponding rigid body
      modes RBx, compute force separation ratios Sx:
            Sx = Kxx*RBx/Kxx_diag
}
      [ 2 is Kmin ] \ for Kxx(i,i) below Kmin, returned S(i) is 0
      over swap (hKxx hRBx) star
      swap >vector abs Kmin filter push
      peek rake, peek dup rake, lop 6 clone /by
      swap dims null swap pull tier abs
   end

   inline: RBcyl (hTheta hXYZ --- Phi6Nx6) \ RB modes for cyl coords
{     XYZ is 3xN matrix of vector offsets from ref point to each of N 
      points.  

      Row 1 of XYZ holds the X, row 2 the Y, and row 3 the Z component.

      Theta is a vector of N angles (radians) about Z that define a 
      local cylindrical system at each point (see word Tcyl).  The
      number of rows in Theta matches the number of columns in XYZ.

      Note: as a check, the X row in Phi for each point should show 
      negligible motion in column 6, meaning ThetaZ rotations give no 
      radial motion; and the Y row in column 6 should equal the radius
      to the point.
}
      into XYZ, hand into Theta, Theta rows 1st 
      DO Theta I pry Tcyl three fold, three dup null, dup, other tile
         three identity, 
         three dup null, XYZ I catch tilde negate, other tile
         star
      LOOP XYZ cols pilen
   end

   inline: RBphys (hPhi hSupport --- hPhi1) \ physical RB vectors
\     Transform rigid body eigenvectors in Phi into physical vectors
\     using Support dofs in the eigenvectors as the support point.

      "SUP" book 
      "Phi" book

\     The first "SUP rows" columns of Phi are assumed to be rigid
\     body modes:
      Phi 1st SUP rows items catch "RB" book \ rigid body modes
      purged "Phi" book

      SUP RB rows teeth (hRake) \ taking 6 support dofs
      (hRake) push              \ rake of zeroes and ones to local stk

      RB peek (hRake) rake "RBlr" book "RBrr" book
      SUP rows identity RBlr purged RBrr solve star
      (I RBlr*RBrr-inv) pull (hRake) tier

      "_RBphys" naming

      purged "RB" book
      purged "RBlr" book
      purged "RBrr" book
   end

   inline: RBrect (hXYZ --- Phi6Nx6) \ RB modes for rect coords
{     XYZ is 3xN matrix of vector offsets from ref point to each of N
      points.  

      Row 1 of XYZ holds the X, row 2 the Y, and row 3 the Z component.

      Returned Phi6Nx6 contains 6 rigid body mode columns, each column 
      containing 3 translations and 3 rotations for each point in XYZ.
}
      into XYZ, XYZ cols 1st
      DO three identity, three dup null, XYZ I catch tilde negate, 
         other tile
      LOOP XYZ cols pilen
   end

   inline: refine (hA n --- hA1) \ steps between rows of A n times finer
{     Example:

         vector: 1 2 pi ; dup .m nl nl 4 refine .m nl
}
      this two < IF drop cop return THEN
      push once those rows, uniform back park
      once peek slash those rows, nit pull star tic
      uniform those cows, nit clone lerp 
   end

   inline: Rmax (hV n --- hR) \ rows of V for session nmax highest 
{     Fri Feb  3 12:47:13 PST 2012

      Incoming V is a vector of data in time order for all sessions,
      where each set of n rows corresponds to one session.

      Returned R is a vector of row numbers for the nmax highest values
      of V in each session.

      This word is more general than the context under which it was de-
      veloped.  Incoming n simply divides the rows of V into subsets,
      here called "sessions," where the rows of V must be divisible by 
      n with no remainder (so that word foldr() can be applied below).

      Sizes are:
         number of sessions = nsess = rows(V)/n
         rows(R) = nmax*nsess = nmax*rows(V)/n 
      where rows(R) cannot be greater than rows(V) (thus nmax cannot
      be greater than n).

      Row numbers in R are compatible with row numbers of V.  Within
      each session, the nmax row numbers of V in vector R are in order
      from highest V to lowest V (lowest of the nmax highest ones, that
      is).  This means that they probably are not in ascending order
      within R.

      Default nmax is nmax_def (10).  To change nmax beforehand, say
      to 100, run
         % Rmax.nmax = 100;
      or
         ready > 100 "Rmax" "nmax" bank
      Upon exit, nmax is set back to the default of 10 (nmax_def).

      Examples:
         Finding rows of C that are minimums during each session:
            Rows_min = Rmax(-C, n); // rows of minimum values in C

         Looking at C at times of higher session V, with nmax=100:
            X = teeth((Rmax.nmax=100), Rmax(V, n), rows(V));
            Cmax = looking(C, X==0);
}
      [ 10 "nmax_def" book, nmax_def "nmax" book ]

      (hV n) nmax over (nmax n) >
      IF " Rmax: size error: nmax is greater than n" ersys return THEN

      (hV n) "n" book
      1st n items "N" book              \ indices for one session

      (hV) n foldr (hA)                 \ A has n rows and nsess cols

      (hA) dup push cols 1st            \ loop over the cols of A
      DO peek I catch N park (hN)       \ append index vec N to Ith col
         (hN) no sort 2nd catch (hR)    \ indices of sorted Ith col
         (hR) 1st nmax items reach (hR) \ just nmax indices of highest
       \ (hR) yes sort                  \ DO NOT SORT HERE \ sorted
         (hR) I qdx 1- n * + (hR)       \ plus session offset
      LOOP
      (hR1 hR2 ... hRn) pull cols pilen (hR)

      nmax_def "nmax" book 0 "N" book
   end

   inline: rms (hY --- Yrms) \ rms of amplitudes in Y columns
\     For N columns in Y, k=1,N: Yrms(k)=sqrt[dot(Y(k),Y(k))/N]

\     Note: if Yrms has one row, then its value is returned.

      hand these rows swap this *dot swap /f sqrt
      (Yrms) these rows one = IF ontop THEN
   end

   inline: rotating (hC0 hOmega hV ht --- hV1) \ inertial from rotating
{     View inertial from rotating.
      A reference frame is rotating with rate Omega(t), where column k 
      of Omega contains the 3x1 angular rate vector components at the 
      time given in row k of t, and the rate vector components are re-
      solved to the rotating frame.

      Like Omega, column k of V(t) is a vector corresponding to the
      time contained in column k of t.  But the vector components de-
      fined in the three rows of V are resolved to another reference 
      frame, related to the rotating frame at the initial time by ro-
      tational transformation matrix C0.

      Matrix C0 is the direction cosine matrix defining attitude of 
      the rotating frame at the first time in t.  Its columns are unit 
      vectors in the rotating frame, and its rows are unit vectors in 
      the frame of V--usually but not necessarily--an inertial frame.

      This word transforms vector V, defined in the frame in which a 
      rotating frame is viewed--the one in which rotational rate Omega 
      is seen--into the rotating frame itself, producing vector V1.  Of
      course the magnitude and direction of V and V1 are identical--they
      are the same vector that is simply being viewed from different 
      reference frames.

      Transforming V(t) requires numerical integration of C0 under rates
      Omega, and word trapC is used.  Having the time history of direc-
      tion cosine matrices from trapC, beginning with C0 and ending with
      CN corresponding to the last time in t, velocity V1 is computed 
      as:
                           V1(t) = C(t) * V(t)
      Examples: 
         See word watching, where rotating transforms vectors to a ro-
         tating frame, then watching transforms them back to an inertial
         one.  Also see example 2 in word velocity for use of rotating.
}
      back push trapC, three no empty those cobs 1st 
      DO that I catch three fold (C), peek I catch (V) star (V1) park 
      LOOP lop pull drop ;

   inline: rsi (hH hL hC --- hRSI) \ relative strength of C within H - L
{     RSI = 100*(C - L)/(H - L)

      Note: If C(k) is bounded by H(k) and L(k), then RSI(k) is between
      0 and 100.
}
      push 2dup max rev min pull \ make sure H is max, L is min

      (hH hL hC)
      over - \ C - L
      rev -  \ H - L
      1 max  \ assumes if H - L equals zero, so does C - L, i.e. C = L.
      / 100 *
   end

   inline: rss (hY --- Yrss) \ rss of amplitudes in Y columns
\     For N columns in Y, k=1,N: Yrms(k)=sqrt[dot(Y(k),Y(k))]
      hand this *dot sqrt
   end

   inline: s3curve (hC ht n --- hX hX' hX'')
{     Using n previous time steps in a cubic spline fit to estimate
      curves of the columns of C at each time, t.  Uniform times in 
      t are assumed.

      Each column of C is a separate function of t, and each is evalu-
      ated and returned in a column of X, X', and X''. 
}
      "n" book, 1st n items reach "t" book, "C" book
      n ndx push

      C rows tic 1st n plus
      DO C I n less n items reach t s3coef
        t s3eval (hX hX' hX'') 
        rot peek reach 
        rot peek reach 
        rot peek reach 
      LOOP pull drop

      C rows tic n less three star pilen

      n nit three star C cols null swap pile (hX)

      (hX) these rows 3 spikes rake "_X" naming swap
      these rows 2 spikes rake "_X'" naming swap "_X''" naming

      purged "C" book purged "t" book
   end

   inline: s3deriv (hC ht n --- dC) \ derivative of C(t)
{     Using n previous time steps in a cubic spline fit to estimate
      derivatives of the columns of C at each time, t.  Uniform times
      in t are assumed.

      Each column of C is a separate function of t, and each is evalu-
      ated and returned in a column of dC.

      Note: this word is the same as: (hC ht n) s3curve drop lop

      Here is a test case, an offset sine wave, A, where the derivative
      is a cosine wave, Adot:
         "compare" sourceof source
         1 10 2pi * 0 0.001 1000 sine into t 10 +d into A
         1 10 2pi * 0 0.001 1000 cosine drop 10 2pi * *f into Adot
         0based A t 10 s3deriv Adot 2dup compare 3 parkn rview
         1based A t 10 s3deriv Adot 2dup compare 3 parkn rview
}
      "n" book, 1st n items reach "t" book, "C" book

      C rows tic 1st n plus
      DO C I n less n items reach t s3coef
        t s3eval drop lop n ndx reach
      LOOP
      C rows tic n less pilen

      n nit C cols null swap pile

      purged "C" book purged "t" book
      "_dC" naming
   end

   inline: sine (A w phi dt N --- hW ht) \ sine wave time history
\     Sine wave of amplitude A, frequency w and phase phi:
\        W = A*sin(w*t - phi), t from 0 to (N-1)*dt
\     where w is rad/sec frequency, phi is deg phase angle.
      (dt N) uniform (t) push, (phi) rad/deg * "phi" book, (w) "w" book
      peek (t) w *f phi -d sin swap *f "_sin(t)" naming
      pull "_t" naming
   end

   inline: sine_rms (hA --- hR) \ total rms of sines
\     Incoming matrix A holds k columns of sine amplitudes, zero-to-
\     peak.  Row k of vector R is the rms total of all amplitudes of
\     the sines in column k of A applied simultaneously and assuming
\     unequal frequencies and unequal phase angles.

\     Note: if R has one row, then its value is returned.

      sine_rms_running one endmost bend
      (hR) these rows one = IF ontop THEN
   end

   "logspace" missing IF "math.v" source THEN

   inline: sine_rms_moving (hA1 hF1 Wfac dF --- hR hF)
{     Moving RMS of sines A1(F1) in a window moving in steps dF.

      Window width at frequency f is

         Fw = 2 * Wfac * f

      Returned matrices R and F are the matrix of center-of-window
      RMS values, R, and the center-of-window frequencies, F.

      Example:
         1000 ones dup 2 * over 3 * 3 parkn \ R, sines of ampl 1, 2, 3
         .1 10 999 logspace                 \ F, .1 to 10 Hz, 100 points
         .01                                \ Window factor, Wfac
         .001                               \ dF
         sine_rms_moving swap park eview   
         halt
}
      [ 1E-10 "eps" book ]
      "dF" book, (Wfac) "Wfac" book, "F1" book, "A1" book

      F1 one endmost ontop "Fmax" book
      F1 1st pry "f1" book
      A1 cols 1st
      DO depth push
         A1 I catch "A" book
         f1 "f" book
         BEGIN
            f Fmax <
            IF A F1
               f Wfac star (Fw)          \ window width is 2*Fw
               f that less zero max (f1) \ f1=f-Fw, left side of window
               f rot plus (f2)           \ f2=f+Fw, right side of window


               (hA F1 f1 f2) inclusive \ points in window
               rake drop (hR) any?     \ rake points from A

               IF (hR) sine_rms (rms)  \ rms of all N sines in window
               ELSE eps (rss)          \ case of no freqs in window
               THEN
               dF f bump false

            ELSE true
            THEN (f)
         UNTIL
         depth pull less listn
      LOOP
      A1 cols parkn "_R" naming
      dF those rows uniform
      F1 1st pry plus "_F1" naming
   end

   inline: sine_rms_running (hA --- hR) \ running rms of sines
\     Each column of A contains sine amplitudes for a response point,
\     listed in order of ascending frequency.  Columns of returned R
\     contain the running rms for the sines in columns of A.

      hand         \ NUM into 1-by-1 MAT for word partials
      (hA) dup *by \ squared
      two slash    \ rms squared
      partials     \ sums of rms squared
      sqrt (R)     \ running rms = the sqrt of the summed rms^2 values
   end

   inline: sined (A w z phi dt N --- hW ht) \ damped sine wave time hist
{     Sine wave of amplitude A, frequency w, damping ratio z and phase 
      angle phi:
         W = A*e^(-z*w*t)*sin(w*(1-z^2)*t - phi), t from 0 to (N-1)*dt
      where w is rad/sec frequency, z is damping ratio (<1), phi is deg
      phase angle.

      Light damping is assumed (z < 1) and is not checked.
}
      (dt N) uniform (t) push, (phi) rad/deg * "phi" book, 
      (z) "z" book, (w) "w" book, (A)

      peek (t) 
      w 1 z z * - *                   \ damped frequency, w*(1-z^2)
      (A t wd) * phi - sin * (hS)     \ A*sin(w*(1-z^2)*t - phi)

      peek (t) z w * negate * e^ (hE) \ e^(-z*w*t)
      (hS hE) *by 

      "_sind(t)" naming
      pull "_t" naming
   end

\ _define: steady (hZ hW hQ hF --- hqr hqi) freqency response analysis
\     This is a native word; see wapp.c.  Below are some examples.

{  Examples using word steady for frequency response analysis:

      Reference: Thomson, W. T., "Vibration Theory and Applications,"
                 Prentice-Hall, 1965, pp. 51-56.

      steady (hZ hW hQ hF --- hqr hqi) freqency response analysis

      Word steady solves a number of scalar differential equations 
      given by

            qdd + 2*zeta*omega*qd + omega**2*q = Q*sin(Omega*t)

      for steady state response, q, after transients have decayed.  
      Most often, q can be thought of as a modal coordinate: each row 
      of q may correspond to a natural mode of an elastic system.

      Stack elements:
         hZ = vector of critical damping ratios, zeta
         hW = vector of natural frequencies, omega (rad/sec)
         hQ = matrix of generalized force amplitudes:
                 each row of Q matches a row (mode) in zeta and omega
                 each column of Q corresponds to a driving frequency 
                 listed in hF
         hF = vector of driving frequencies, Omega (rad/sec)
         hqr = real part of computed response
         hqi = imaginary part of computed response

      1) From the Reference, example 3.1-2, p. 56:

         Compute amplitude and phase at resonance of a machine on 
         springs.

         Copy and drop the following four lines to run this example:

         193 (lb) 386 / is M, 6 is C, 200 is Ks, 10 (lb) is Fo
         Ks M / sqrt is omega, C 2 / M / omega / is zeta, Fo M / is Q
         zeta, omega, Q, omega steady,
         cmagphase deg/rad *f pile bend .m

         Result, showing amplitude and phase angle at resonance 
         (Ref, p. 56):

            [tops@gutter] ready > cmagphase deg/rad *f pile bend .m
             Row 1:  0.08333       90 
            [tops@gutter] ready > 

      2) Single degree-of-freedom oscillator, response at three fre-
         quencies:

            zeta = 0.05 (5% of critical damping)
            omega = 1.0 rad/sec natural frequency
            Q = [ 1.0, 1.0, 1.0 ], flat unity input at 3 driving freqs
            F = [ 0.5, 1.0, 2.0 ], three driving frequencies (rad/sec)

         Copy and drop the following three lines to run this example 
         (bend is used to transpose Q into a row matrix: 1 mode by 4 
         frequencies):

            0.05 (zeta), 1.0 (omega) 
            list: 1 1 1 ; bend, list: 0.5, 1, 2 ;
            steady cmagphase deg/rad *f pile nl .m

         Results:
            F (rad/sec):     0.5       1.0     2.0
                  Row 1:     1.33       10   0.3326
                  Row 2:   -3.814      -90   -176.2
            [tops@gutter] ready > 

            In column 2, the amplitude 10.0 corresponds to resonant
            response of a 5% damped oscillator: 1/(2*0.05) = 10; and 
            the phase angle, shown below it, is -90 degrees as expected:
            the resonant response lags the force by a 90 degree phase 
            angle.

         (Previous results from the express program, with error in sign
         of phase angle, row 2:
            [express@bach] ready >
            F (rad/sec):   0.5         1.0          2.0
                  Row 1:  1.3304      10.000      .33260
                  Row 2:  3.8141      90.000      176.19
             [0] ok!
            [express@bach] ready >)
}

   inline: steady_balance (hZ hW hQ hFrad hqr hqi --- hBr hBi)
\     Checking the balance of the solution from word steady.
      complex "q" book
      "Frad" book
      "Q" book
      "W" book
      "Z" book

      q real-imag Frad caccel complex \ qdd

      W Z 2 *f *by q cols clone
      q real-imag Frad crate (2*Z*W qdr qdi)
      other *by swap rot *by swap complex \ 2*Z*W*qd

      W W *by q cols clone
      q real-imag
      other *by swap rot *by swap complex \ W*W*qd

    \ Balance B should equal 0:
    \    B = qdd + 2*Z*W*qd + W*W*q - Q
      plus plus Q 0i complex less

      real-imag "_Bi" naming swap "_Br" naming swap
   end

   inline: syminv (hA --- hAinv) \ inverse of symmetric matrix
      [ "symSSL" exists? IF "symSSL" "solve" overload THEN ]
      these cols identity, that is_complex IF 0i complex THEN
      swap solve ;

   inline: symmetric (hA --- hA1) \ force A to be symmetric
      this bend plus two /f ;

   inline: Tcyl (hT --- hCcb) \ transformation from rect to cyl
{     T is the angle (in radians) measuring the rotation about an axis
      Z from the X axis of system B to the X axis of system C.

      If T is an N-by-1 vector, then returned Ccb is 9-by-N matrix,
      and contains N 3-by-3 transformation matrices, each stored in
      a 9-by-1 column, for the N angles in T.

      For N=1, angle T can be just a number.

      Submatrices in Ccb are of the form:

               | cosT sinT  0|
         Ccb = |-sinT cosT  0|
               |  0    0    1|

      and transform 3-by-1 vectors from system B to system C;

         xc = Ccb * xb

      Note: use 3 fold to make 9-by-N Ccb into N parked 3-by-3s.

      Example: list: 30 45 ; rad/deg *f Tcyl 3 fold .m
}
      hand xbase push 1based dup into T rows 1st
      DO nine one null
         T I pry cos this other one poke, those five poke
         T I pry sin this negate them two poke, those four poke
         once those 9 poke
      LOOP T rows parkn
      "_Tcyl" naming pull indexbase
   end

   inline: tilde (hA --- hA~) \ 3x3 tilde matrices from x,y,z vectors
\     For N columns in A, N parked 3x3 tilde matrices are produced
      1st push, 1based
      dup push cows 1st
      DO peek I catch
         three dup empty (hnd) push
         ape one pry (x)
            ape peek (x hnd) three two stick
            negate peek (-x hnd) two three stick
         ape two pry (y)
            ape peek (y hnd) one three stick
            negate peek (-y hnd) three one stick
         three pry (z)
            ape peek (z hnd) two one stick
            negate peek (-z hnd) one two stick
         pull 
      LOOP pull cols parkn, pull indexbase
   end

   inline: tilts (hC --- hTheta) \ tilts to make chain of attitudes in C
{     For a set of N+1 3x3 direction cosine matrices contained in C, 
      determine a set of N 3x1 vectors that, when used as Omega in 
      Poisson's equation and integrated, will step from first to last 
      matrix in C (as in word trapC).

      Incoming C contains 9 rows and N+1 columns.  Each 9 element column
      of C is a 3x3 direction cosine matrix, chained and stored as de-
      scribed in word trapC, so each column in incoming C corresponds 
      to a point in time.  The following phrase will fetch and shape the
      Ith column of C, for the Ith time point, into its 3x3 matrix: 
      (hC) I reach (hC_9x1) 3 fold (hC_3x3).

      Neighboring columns in output 3xN matrix Theta are separated by a
      unit time step, and so need to be divided by the actual steps used
      to make C to create a rate vector:
              Omega = Theta/dt

      Testing shows that the magnitude of the rotating frame rate vec-
      tor multiplied by the time step should be less than one sixteenth
      radian:
              Theta = Omega*dt < 0.0625 radian (3.58 degrees)
      meaning that the variation between adjacent 3x3 direction cosine 
      matrices in a simulation should be less than 3.58 degrees.
 
      Methodology.  Poisson's equation (see trapC) can be written as:
              dCk = (Ck x Omega)*dt
      where Ck is a column from direction cosine matrix C.  Writing the
      cross product using a 3x3 tilde matrix for vector Ck gives
              dCk = (Ck~ * Omega)*dt
      Define 
              Theta = Omega*dt 
      and
              dCk = C2k - C1k 
      where C1 and C2 are two adjacent direction cosine matrices in a 
      set that defines a sequence of attitude changes, so that an equa-
      tion for Theta is:
              Ck~ * Theta = C2k - C1k
      Stacking the three vectors for k=1,3 gives a system of nine equa-
      tions for 3 unknown values in Theta:
              C~ * Theta = C2 - C1
      This can be solved using word lsq.  To later obtain Omega from 
      Theta, the above definition shows that
              Omega = Theta/dt

      Validation.  In the examples below, word trapC is used to inte-
      grate an initial identity matrix, and then run tilts to obtain 
      history Omega that would also recreate them.  If all is working 
      correctly, Omega from word tilts should match Omega that created 
      the matrices in trapC:

         Make a set of direction cosine matrices for rate Omega and 
         study the effect of various time steps.

         Results:

         For Omega = (0 0 1) rad/sec and dt = 1 sec, this is the 
         sequence of direction cosine matrices from trapC:

            "lsq" missing IF mmath.v source THEN   
            3 identity, vector: 0 0 1 ; dup park, 1 (step) 2 uniform
            trapC this 1 catch 3 fold nl .m, 
            this 2 catch 3 fold nl .m, tilts nl bend .m nl

               Row 1:        1        0        0
               Row 2:        0        1        0
               Row 3:        0        0        1

               Row 1:   0.7071  -0.7071        0
               Row 2:   0.7071   0.7071        0
               Row 3:        0        0        1

         and the tilt is .70711:

               Row 1:       -0       -0   0.7071 

         It is concluded that the step between each matrix is too big, 
         since the result should be 1.0, not .707; the unit step for 
         1 rad/sec is too large.

         Here is the effect of reducing the time step when from 1 in
         this example:
            Using 0.5 step, get .44721 from tilts, and 
               .44721 / .5 = .89442
            Using 0.25 step, get .24254 .25 / = .97016
            Using 0.125 step, get .12403 .125 / = .99224
            Using 0.0625 step, get 6.23783E-02 .0625 / = .99805

      Thus the product of Omega*step should be 0.0625 radians or less, 
      so if Omega is large, need a correspondingly small step for good 
      accuracy.

      This requirement corresponds to 0.0625*57.3 = 3.58 degrees

      Why this works can be implied from the results of the first case,
      where unit step and unit Omega gave changes of 0.707.  The change
      between steps is tan(theta) = step/Omega, meaning the step has to
      be sufficiently small to allow the approximation tan(theta) = 
      theta.  For 3.58 degrees, 
             tan(3.58) = 6.240166417E-02 
              3.58 deg = 6.248278722E-02
            difference = 8.112304662E-05
}
      push 3 no empty peek cobs one less 1st
      DO peek I catch, 3 fold (C1), again tilde bend (C1~)
         peek I tic catch 3 fold (C2), rot less chain (C1~ C2-C1) 
         swap lsq 
      LOOP pull cobs parkn
   end

   inline: transform (hM hT --- hT'MT) \ transform M to M1=T'*M*T
\     Transform M by T using the triple product T'*M*T.
      [ "mpySSL" exists? IF "mpySSL" "star" overload THEN ]
      again bend rot star swap star ;

   inline: transform_diag (hM hT --- hT'MT) \ transform diagonal M
\     Transform M by T using the triple product T'*M*T, where M is a
\     diagonal matrix stored as a column vector.
      [ "mpySSL" exists? IF "mpySSL" "star" overload THEN ]
      this bend rev diagpre star ;

   inline: trap (hV ht --- hX) \ integrate V(t) using trapezoidal rule
{     Column k of vectors V and X correspond to row k of the indepen-
      dent variable in vector t, that is, if t is time, then V and X 
      are stored columnwise by time: one column per time point in t 
      (see "the equation solution viewpoint," notes in doc/plug.doc).

      Each row in V(t) is integrated with respect to independent vari-
      able t to produce the corresponding row in X(t).  The trapezoidal
      rule is used, with steps given by increments between row elements
      in column matrix t.

      Reference: Sokolnikoff, I. S., and R. M. Redheffer, "Mathematics 
                 of Physics and Modern Engineering," McGraw-Hill, 1958.
 
      Examples:

      1. From the Reference, p. 718.  Given a table of values Y(X):

            list: 2.105 2.808 3.614 4.604 5.857 7.451 9.467 11.985 ; 
            is Y, 1 Y rows items makes X,

            find the integral of Y(X) from X = 1 to 7.

            Copy and drop the two lines above that make X and Y, then 
            the phrase below that runs trap and fetches the value from
            row 7, corresponding to the integral of Y(X) at X = 7:

               Y bend (Y(X) ), X trap, 7 (column) catch, nl .m 

            gives 30.120.

      2. Time for free-fall of 100 feet, to nearest .01 seconds (ignor-
         ing drag):
            Create 1000 points at steps of .01 to cover 10 seconds, and
            run trap twice to integrate constant one-g acceleration into
            distance.  Then park distance (d) and time (t) into an XY 
            table and interpolate t for 100 feet, using lerp.  Copy and
            drop these lines to run:

               32.17 (ft/sec/sec) 1000 1 fill bend,
               .01 those cols uniform (10 sec max) this push trap, 
               peek trap, bend pull (hd ht) park, 100 (ft) lerp, nl .m

            gives 2.49 seconds.

      3. Velocity at 2.49 seconds during free-fall:
            This is similar to the previous example, except one-g is 
            integrated once to obtain velocity.  Then time (t) and 
            velocity (V) are parked into an XY table (note that time 
            is first here, unlike the prevous) and interpolated for 
            velocity V at 2.49 seconds:

               0.01 300 uniform (3 sec max), 
               these rows 1 empty 32.17 +d
               (ht hG) bend over trap bend (ht hV)
               park, 2.49 (sec) lerp nl .m

            gives 80.1 ft/sec.

      4. Compare free-fall distance on earth and the moon up to 10 
         seconds and plot results:

               1 1001 uniform .01 *f (t), 
               32.17 those rows 1 fill, again 6 /f (hE hM), 
               park, bend that trap, that trap 
               bend swap plot pause plotclose

}     push, bend this dup, delta less, plus two /f
      pull delta those cobs clone *by, partials bend ;

   inline: trapC (hC0 hOmega t --- hC) \ integrate dir cos matrix
{     Integrate Poisson's equation for a reference frame rotating at 
      Omega:
                   dC = -Omega x C
      to find the frame's direction cosine matrix at each time step 
      given in one-column matrix t.

      Incoming 3x3 matrix C0 is a 3x3 rotational transformation ma-
      trix--also called a direction cosine matrix--that defines the 
      three dimensional attitude between two reference frames.  Omega 
      is a 3xN matrix where each column contains the three vector 
      components of rotational rate of one frame relative to the other.
      In detail, if we call frame B the one whose unit vectors are the 
      columns of matrix C0, and frame A the one whose unit vectors are 
      the rows of C0, then Omega is the rotational rate vector of B
      relative to A.  Furthermore, the components contained in Omega 
      are components of the rate vector as seen (measured) in frame B.

      Matrix C from this word contains a sequence of 3x3 direction 
      cosine matrices that each differ by a step in time t.  Each is 
      stored as a 9x1 matrix (word chain) and then all are stored as 
      sequential columns in C (word park), making C a 9xN matrix for 
      N elements in t. 

      Thus C has one column for each time point in t; for example, the 
      9 rows at column k of C contain the 3x3 direction cosine for time
      given in the kth row of one-column matrix t.  The example below 
      shows how matrices from C for particular times are extracted and 
      reformatted into 3x3 matrices (for column k, its just: C k catch,
      3 fold).

      Omega is the rate vector of the rotating frame.  Being a physical
      vector, it has three rows corresponding to vector components in 3
      dimensional space.  Each column in Omega, like each column in out-
      put matrix C, relates to a point in time contained at the corres-
      ponding row of one-column matrix t. 

      Methodology: Assume each row in frame rate vector, Omega--resolved
      to rotating frame axes--is constant over the (small) interval be-
      tween each element of t.  For typical step from t0 to t1, compute
      the rate at t0 by Poisson's equation:
                   dC0 = -Omega x C0
      then assume simple trapezoidal integration so that C1 at t1 is 
      initial value, C0, plus initial rate dC0 times the step size:
                   C1 = C0 + dC0 * (t1 - t0)

      Accuracy may be improved by chopping Omega and t into finer steps
      before firing trapC, using words refine and lerp1 (see lerp1 ex-
      ample in file boot.v).
 
      Each 3x3 direction cosine matrix in C--stored as a 9x1--is used 
      later to transform the rotating frame's vectors into the frame 
      where rotation is being observed.  The columns in typical 3x3 Ck 
      are unit vectors of the rotating frame axes as seen in the other 
      frame (usually, but not necessarily, inertial space), and the 
      rows of Ck are unit vectors in the other frame as they are seen 
      in the rotating frame.  Thus Ck is used as Cab to transform from 
      rotating system b to another system a in the transformation equa-
      tion:
                   xa = Cab * xb
      where xb is vector x when its components are measured in rotating
      frame b, and xa is the same vector x when its components are mea-
      sured in another frame, a.  To go the other way--from a to b--sim-
      ply transpose Cab, as in:
                   xb = Cab' * xa

      Testing word trapC (see also the testing of word tilts, where it 
      is shown that Omega*t should be 0.0625 or less for good accuracy):
            3 identity is C0
            list: 1 0 0 ; 100 clone says wX \ frame rate, 1 rad/sec
            list: 0 1 0 ; 100 clone says wY 
            list: 0 0 1 ; 100 clone says wZ 
            .01 100 uniform is t

         In these examples, C is integrated to 1 second using steps of 
         .01 sec and a frame rotation rate of 1 rad/sec.  The average 
         of matrices at steps 79 and 80 (.78 and .79 radians) is com-
         puted to approximate the matrix for 45 degrees (.785 radians),
         and we expect to see values of sines and cosines of 45 degrees,
         like .7071.

         Here's rotating about X (wX = 1 0 0):
            C0 wX t trapC, its 79 catch, swap 80 catch
            plus 2 /f, 3 fold nl .m nl
               Row 1:        1        0        0
               Row 2:        0   0.7074  -0.7068
               Row 3:        0   0.7068   0.7074

         Here's rotating about Y (wY = 0 1 0):
            C0 wY t trapC, its 79 catch, swap 80 catch
            plus 2 /f, 3 fold nl .m nl
               Row 1:   0.7074        0   0.7068
               Row 2:        0        1        0
               Row 3:  -0.7068        0   0.7074

         Here's rotating about Z (wZ = 0 0 1):
            C0 wZ t trapC, its 79 catch, swap 80 catch
            plus 2 /f, 3 fold nl .m nl
               Row 1:   0.7074  -0.7068        0
               Row 2:   0.7068   0.7074        0
               Row 3:        0        0        1
}
\ ******** THIS SHOULD BE MADE FASTER USING WORD parkn (see tilts)
      delta push over, chain rev, these cobs 2nd
      DO this I nit catch, dup dup park park, 
         other (C0 Omega3) *cross (dC0)
         (dC0) peek I pry (dC0 dt) *f rot plus norm (C1)
         rot over chain park rev back
      LOOP 2drop pull drop (hC) ;

   inline: truncate (hPhim hOmega2m omeg2 --- hPhi hOmega2) 
\     Keeping modes below omeg2
\     Frequencies are squared (rad/sec)
      xbase push one indexbase, over abs swap bsearch 
      drop 1st swap items
      two roll over, catch rev, reach pull indexbase ;

   inline: twang (hB hK dof dt steps --- hq ht) \ transient free-vib
{     Inputs on stack:
         B is square, coupled damping matrix
         K is vector of diagonal stiffness matrix (frequency squared)
         (mass matrix not required: it must be an identity matrix)
         dof is the degree-of-freedom to twang (initial condition = 1)
         dt is the time step
         steps is the number of time steps
      Outputs on stack:
         q is matrix of response displacements, one column per time 
           point
         t is column matrix of times corresponding to columns in q

      Example usage (twang dof 11 then plot it):

         Bhh Khh 11 .01 1500 twang, swap 11 reach swap mplot mgrid

}     rev (steps dof dt) push push push
      (B K) these rows pull (steps) null (hp),
      these rows 1 null (hqd0), these rows 1 null (hq0),
      once over pull (dof) poke (hq0, with dof = 1)
      peek (dt) left transient (hqdd hqd hq) lop lop (hq)
      pull (dt) those cols uniform (ht) ;

   inline: velocity (hVc hOmega hr hdr --- hVp) \ inertial velocity
\     Inertial velocity in rotating frame
{     Compute the inertial velocity of point p, given the inertial
      velocity of another point, c, and the varying vector r from c 
      to p.  The rotational velocity of the frame in which point c 
      is fixed (and thus the rotational velocity of c itself) is 
      given by Omega.

      Point c is fixed in the moving reference frame, but p may be 
      moving; if it is, then rate matrix dr gives the time rate of 
      change of the length of vector r from c to p.

      All vectors are resolved to the reference frame of c that is 
      rotating at angular rate Omega (rad/sec).  The inertial velocity 
      of p is then given by:
                        Vp = Vc + Omega x r + dr

      Each matrix of vectors has 3 rows for vector components in the 
      reference frame, and N columns that each correspond to a point 
      in time.

      Examples:

      1. Velocity at 2.49 seconds during free-fall:
            This is Example 3 in word trap and serves to show how 
            vectors in three dimensions are set up for word velocity.  

            0.01 300 uniform (3 sec max, t), 
            vector: 0 0 32.17 (in/sec/sec) ; \ acceleration in Z
            those rows clone (Ac), over trap (Vc), 
            these dims null (Omega),
            list: 1 0 0 ; those cols clone (r), 
            these dims null (dr)
            (hVc hOmega hr hdr) velocity (Vp),
            3 (Z) reach bend park, 2.49 (sec) lerp nl .m

         gives 80.1 ft/sec.

      2. Rotating and translating body:
            A body is accelerating in the inertial X direction and 
            rotating about its body-fixed Z axis with harmonic rate.  
            Inertial and body-fixed axes are initially aligned.  

            "graphdef" missing IF "plot.v" source THEN graphdef
            1 101 uniform .01 *f (t),
            these rows 2 swap null, that (t) 10 *f sin bend pile
            10 *f (hOmega, 10 rad/sec)
\           Transform inertial a to the body frame:
               3 identity (hC0),
               over (hOmega)
               vector: 1 (Xdd) 0 0 ; those cobs clone (ha, 1 in/sec/sec)
               4 pick (t)
               rotating 
\              dup bend 3 pick plot \ X (blu), Y (grn) vary; Z (red) = 0
            (ha) other trap (hVc) \ integrate the rotating accel into Vc
            swap (ht hOmega hVc) 
            vector: 1 0 0 ; those cols clone (r), 
            these dims null (dr), velocity, bend swap plot
            pause plotclose

}     rev *cross plus plus ;
 
   inline: watching (hC0 hOmega hV ht --- hV1) \ rotating from inertial
\     View rotating from inertial.

{     As in word rotating, spin rate vector Omega(t) is measured in the
      rotating frame, and column k of matrix Omega corresponds to the 
      time in row k of t.  But here, matrix V(t) contains vectors also 
      in the rotating frame; like Omega, column k of V corresponds to 
      the time contained in row k of t.

      This word transforms vector V into the frame where the rotating 
      frame is being observed, given by vector V1.

      Matrix C0 is the direction cosine matrix defining attitude of the
      rotating frame at the first time in t.  Its columns correspond to
      unit vectors in the frame of the rotating frame, and its rows cor-
      respond to unit vectors in the frame of V, usually--but not nec-
      essarily--an inertial frame.

      Example: This example tests both watching and rotating.  First, 
      the assumed constant inertial X acceleration of the rotating 
      frame is transformed to its components in the rotating frame with
      word rotating, showing what accelerometers in the rotating frame 
      would read.  Then these are transformed back to inertial space 
      using word watching, to verify that X acceleration is again con-
      stant, and Y and Z are zero.

         "graphdef" missing IF "plot.v" source THEN graphdef
         1 101 uniform .01 *f (t),
         3 identity (hC0),
         over rows 2 swap null, other (t) 10 *f sin bend pile
         10 *f (hOmega, rotating 10 rad/sec about Z)
         vector: 1 (Xdd) 0 0 ; those cobs clone (Ap) \ inertial X accel
         3 pick (Ap t) 
\        This shows what accelerometers on the rotating body would read
\        for inertial X acceleration and spin about Z:
         rotating (Ap) 
         2dup bend swap plot \ X (blu) and Y (grn) vary, Z (red) = 0
          
\        Now we transform the acceleration readings on the rotating 
\        body back to inertial space and verify that X is constant, 
\        Y and Z are zero:
         (t Ap) push (t), 3 identity (hC0)
         over rows 2 swap null, other (t) 10 *f sin bend pile
         10 *f (hOmega)
         pull (Ap) 3 pick
         watching bend swap plot \ see X steady at 1, Y and Z at 0
         pause plotclose
      
}     back push trapC "C" book, C cobs 1st
      DO C I catch three fold bend (C), 
         peek I catch (V) star (Vi) 
      LOOP pull cobs parkn, freed is C ;

   inline: wCOV (hX hY hLAG w d --- hR)
{     Each row in vectors X and Y is a step in time.  The covariance of
      X and Y is computed in a series of windows of w rows (time steps).
      Matrix R has number of rows equal to the rows of LAG.

      Calculation begins with the last w rows of X and Y to produce a
      vector R having rows equal to rows(LAG).  Then the window is
      translated by d rows to earlier data and the process is repeated,
      to produce another vector R.

      When all rows have been traversed, the R vectors are assembled
      with earliest columns on the left.  The number of columns in R
      is equal to integer(rows(X)/d).
}
      [ no "noise" book
        0.01 "AMP" book

      {" (hA --- hA1) 
       \ Generate random noise based on max-min range of A:
         (hA) dup maxfetch 2drop (x2)
         (hA) over minfetch 2drop (x1) - 1 max \ max - min
         AMP * dup negate swap other dims ranreal (hN)
         (hA hN) + (hA)
      "} "add_noise" macro
      ]
      "d" book "w" book hand "LAG" book 
 
      noise
      IF swap
         (hX) add_noise swap
         (hY) add_noise (hX hY)
      THEN
      "Y" book "X" book

      X rows w - 0 max 1+ X rows thrulist 
      "L" book \ (rows(X)-d+1):rows(X)

      depth push

      X rows w - 0 max d / integer 1+ (Rcols) 1st
      DO X L reach Y L reach LAG cov2 (hR)
         L d - dup 1st pry 1st <
         IF (hL) dup 1st >= rake lop THEN "L" book
      LOOP

      depth peek - revn       \ reverse columns on stack
      depth pull - parkn (hR) \ park columns, earliest on left

      purged "X" book
      purged "Y" book
      purged "L" book
      purged "LAG" book
   end

   inline: wCOVt (hX hY ht w d --- R) \ covariance R(t) for X(t), Y(t)
{     Covariance between X(t) and Y(t), using window of w steps and
      sliding step of d.  Rows of returned vector R(t) match the time
      steps in t of X and Y, and the five columns of R(t) correspond 
      to the five lags in LAG.
}
      "d" book
      "w" book
      w 6 / 5 uniform dup 3 pry - "LAG" book

      (ht) push

      (hX hY) LAG (lag) w d wCOV (hR)
      (hR) bend dup rows 2 <
      IF dup pile THEN "R" book 

   \  Expand rows of R to times in t.
      peek (ht)
   \  Make a list of rows in time t that correspond to rows of R:
      list: peek rows R rows 2 - any?
         IF (x1) 1st DO dup d - LOOP THEN 1st 
      end reversed (hRows)
      (ht hRows) reach (ht1) R park (hXY) \ table for look
      (hXY) pull R cols clone (hXY hx) look (hy)

      purged "R" book
   end

   inline: weighted (hA hW --- hB) \ weighted average
{     Columns of A align with rows of weights vector, W; for every 
      row i of A, compute the average over columns A(i,j)--each 
      column value, A(i,j), weighted by W(j)--and store into vector 
      B(i):
         B(i) = sum[A(i,j)*W(j),j=1,Acols]/sum[W(k),k=1,Wrows]
      where Acols = columns of A = Wrows = rows of W
}     these totals ontop /f, those rows clone,
      bend *by, bend totals (hB) ;

   inline: xformvec (hOp hRpb Xb type -- hXp) \ transform Xb to Xp
{     Vector Xb components expressed in frame b are transformed into
      components expressed in frame p.

      The 3-by-3 direction cosine matrix for transforming rectangular
      components from frame b to frame p is stored by columns in 9-by-1
      vector Rpb.

      The component type of incoming 3-by-1 vector Xb can be:
         rectangular=1, cylindrical=2, spherical=3.

      The component type of returned vector Xp is always rectangular,
      expressed in frame p.

      Components of Xb are first converted to rectangular type, then Xp
      is computed using:
         Xp = Rpb*Xb + Op
      where Op is the vector from the origin of frame p to the origin 
      of frame b, expressed in rectangular components of frame p.

}     [
      \ Local inlines that convert:
        "noop"   "REC" inlinex \ no conversion for rec
        "cylvec" "CYL" inlinex \ word to convert cyl to rec
        "sphvec" "SPH" inlinex \ word to convert sph to rec

      \ List of ptrs to inlines that convert:
        vector: "REC" ptr, "CYL" ptr, "SPH" ptr end makes Convert
      ]
      this one three within
      IF Convert swap ndx pryexe (hOp hRpb hXb) *dircos9 plus (hXp)
      ELSE " xformvec: invalid vector type " swap int$ cat ersys
         four dump purged (hPurged)
      THEN
   end

   inline: xortho (A Lr Li Pr Pi Gr Gi --- ) \ cross orthogonality
{     Given: 
         State transition matrix: A
         Complex frequencies: Lr Li
         Complex right hand eigenvectors: Pr Pi
         Complex left hand eigenvectors: Gr Gi

      Compute:
         Br= Gr*A*Pr - Gi*A*Pi
         Bi= Gi*A*Pr + Gr*A*Pi

      and compare (Br Bi) against (Lr Li).
}
      6 pick * "Gi*A" book
      5 roll * "Gr*A" book

      (Pr Pi) Gi*A other *
              Gr*A other * + (Bi) push

      (Pr Pi) Gr*A two roll *
              Gi*A two roll * negate + (Br)

      freed is Gr*A
      freed is Gi*A

      (Br) pull (Bi) *conj sqrt (Bmag)

      (Lr Li Bmag) rev (Bmag Lr Li)
      (Lr Li) >square swap >square swap *conj sqrt (Lmag)

      (Bmag Lmag) - abs maxfetch 2drop
      " xortho: max error =" . . nl
   end

   pull catmsg halt

\-----------------------------------------------------------------------

;  Appendix

   This is now a native word:
  _inline: -cmagphase (R P --- Ar Ai) \ magnitude-phase into real-imag
      this cos other *by "_Ar" naming, rev sin *by "_Ai" naming
   end

   A more conventional form for xformvec (and requires origin offset):
  _inline: xformvec (hRpb Xb type -- hXp) \ transform Xb to Xp using Rpb\     A 3-by-3 direction cosine matrix is stored by columns in 9-by-1
\     vector Rpb; the component type of 3-by-1 vector Xb can be:
\        rectangular=1, cylindrical=2, spherical=3.
\     The component type of returned vector Xp is always rectangular.
      its 1 = IF drop (rec)  ELSE
      its 2 = IF drop cylvec ELSE
      its 3 = IF drop sphvec ELSE
      " xformvec: invalid vector type " swap int$ cat ersys
      2drop purged return
      THEN THEN THEN *dircos9 (Xp=Rpb*Xb)
   end


