/* {{{1 GNU General Public License

sofea - the Stack Operated Finite Element Analysis program
Copyright (C) 2004  Al Danial <al.danial@gmail.com>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
}}}1 */

           /* constrain_node still wants DOF_PER_NODE */

#include <stdio.h>
#include <string.h>   /* strncpy */
#include <stk.h>
#include <malloc.h>   /* free    */
#include <main.h>
#include <inpo.h>
#include <math.h>     /* acos    */
#include <sparse.h>   /* spsum   */
#include <sql.h>
#include <tag.h>      /* MAT_ROW_IDX, MAT_COL_IDX */
#include "../elements/tri3.h"     /* elmstfmrt_, elmstfdkt_ */
#include "fea_assemble.h"

/* local headers {{{1 */
typedef struct {
    double x;
    double y;
    double z;
} XYZ;

int element_matrix(sqlite3 *dbh         ,
                   int      part_number); /* in   partition number (0 = all) */
                   /* pop EID from stack, put element [K] matrix on stack */
int get_tri3_data(sqlite3 *dbh          , /* in   database handle */
                  int      part_number  , /* in   partition number (0 = all) */
                  int      EID          , /* in       */
                  XYZ     *coord        , /* out, x[0..#nodes-1][0..2] coords */
                  int     *new_id       , /* out, canonical NID's  */
                  double  *global_dof   , /* out, global degrees of freedom */
                  double  *Young_modulus, /* out      */
                  double  *Poisson_ratio, /* out      */
                  double  *thickness    , /* out      */
                  double  *density);      /* out      */
int tri3_prop_sanity_check(XYZ    vertices[3]   ,
                           double Young_modulus , 
                           double Poisson_ratio , 
                           double thickness     , 
                           double density      );
int get_next_group_of_eids(sqlite3       *dbh             ,
                           int            partition_number,
                           int           *element_offset  ,
                           int           *elements_found  ,
                           int           *elements_remain ,
                           callback_data *sql_data);
int node_count(sqlite3 *dbh             ,
               int      partition_number);
int dof_count( sqlite3 *dbh             ,
               int      partition_number);
int get_dof_per_node( sqlite3 *dbh             ,
                      int      nNodes          ,
                      int      partition_number,
                      int     *dof_list_ptr    );
XYZ    triangle_unit_normal(XYZ vertices[3]);
double triangle_area(XYZ vertices[3]);
XYZ    Cross3(XYZ A, XYZ B);
XYZ    Unit3(XYZ A);
double Dot3(XYZ A, XYZ B);
XYZ    Sub3(XYZ A, XYZ B);
double Mag3(XYZ A);
XYZ    Rot3(XYZ Unit, XYZ V);
double Det3(XYZ A[3]);
void   rotation_3x3(XYZ Unit, double r[3][3]);
void   rotate_tri3_elem(XYZ Normal, double K[18*18]);
void   test_rotate_tri3_elem();
/* 1}}} */

int fea_assemble()      /* ( qModelDB qParnDB part_id --- hMgg hKgg ) {{{1 */
/*
 * man entry:  fea_assemble {{{2
 * ( qModelDB qParnDB part_id --- hMgg hKgg ) Load FE model from the SQLite database defined by the model and (optionally) partition database files.  Put the model's mass and stiffness matrices for partition part_id on the stack.  If part_id is 0, ignores the partition DB stack item and assemble matrices of the entire model.
Note:  this word has the side-effect of changing XBASE to 0.
 * category: FEA
 * related: db_open, db_close, sql, sparse, element_matrix 
 * 2}}}
 */
{
int DEBUG = 0;  /* takes values of 1, 2 (print each element's ID) */
    int    i, rc, elements_found, elements_remain, element_offset, 
           nNodes, nDOF, partition_number, bogus_dbh_partn,
           n_partitions, max_partition;
    callback_data sql_data;
    sqlite3 *dbh;
    double  *Mgg;
    char     query[QUERY_STR_SIZE+1];
    SparseMatrix SP;

    int    LUMPED = 1;  /* != 1 means consistent mass */

    XBASE = 0;  /* must match the database indices */
if (DEBUG) gprintf("at top of fea_assemble\n");
    query[0] = '\0';

    if (!popint(&partition_number)) return 0;
    if (partition_number) {  /* requested only a portion of the total model */
        pushstr("DB_PARTN"); /* add to stack alias for partition db */
        if (!db_open2()) {   /* open model db & attach partition db */
            stkerr(" fea_assemble: ", "failed to open model and partn dbs");
            return 0;
        }
    } else {
        drop();             /* ignore whatever is in position of partn DB */
        if (!db_open()) {   /* open model db */
            stkerr(" fea_assemble: ", "failed to open model db");
            return 0;
        }
    }
    dbh = (sqlite3 *) tos->tex;
    dup1s(); /* keep a copy of the database handle around to close it later */
    /* Now have:  ( dbh dbh ) */

    /* Make a copy of the database handle and the partition #.
     * These copies will be consumed by g_set_null when it makes empty 
     * mass and stiffness matrices.
     */
    pushint(partition_number);
    dup2s();
    /* ( dbh dbh part_id dbh part_id ) */

    nNodes = node_count(dbh, partition_number);
    nDOF   = dof_count( dbh, partition_number);

    if (partition_number) {
        strncpy(query, 
                 "select count(*) from ("
                 "  select distinct pid from element_partition);", 
                 QUERY_STR_SIZE);
        n_partitions = sql_scalar_i(dbh, query, "fea_assemble");

        strncpy(query, 
                 "select max(pid) from element_partition;"       ,
                 QUERY_STR_SIZE);
        max_partition = sql_scalar_i(dbh, query, "fea_assemble");

        if (n_partitions != max_partition) {
            stkerr(" fea_assemble: ","partition count mismatch");
            return 0;
        }

        if (partition_number > n_partitions) {
            stkerr(" fea_assemble: ", "partition does not exist");
            return 0;
        }
    }

    /*
     * Null mass matrix on stack to sum element mass matrices into.
     */
    if (LUMPED) {
        drop();
        drop(); /* ( dbh part_id --- )  */
        if (!matstk_idx(nDOF, 1, "Mgg")) return 0; 
        /* ( --- hMgg ) (Mgg is a null dense vector) */
        /* Mgg's internal row indices will be copied from Kgg later */
        Mgg = tos->mat;
        for (i = 0; i < nDOF; i++) {
            Mgg[i] = 0.0;
        }
    } else {
        rc = g_set_null(); /* Put on tos an empty sparse matrix with structure.
                              of the G set mass matrix.  
                              ( dbh part_id --- hMgg ) 
                            */
        pushstr("'Mgg' naming"); xmain(0); /* name the null matrix Mgg */
        if (!rc) { /* rc == 0 is a problem */
            stkerr(" fea_assemble: ","failure in g_set_null for Mgg");
            return 0;
        }
        set_low_tri(tos);  /* Tag Mgg as lower triangular, symmetric */
        set_symm(tos);
    }
    lpush(); /* move Mgg to local stack */

    /*
     * Null stiffness matrix on stack to sum element stiffness matrices into.
     */
    rc = g_set_null(); /* Put on tos an empty sparse matrix with structure.
                          of the G set stiffness matrix.
                          ( dbh part_id --- hKgg )
                        */

    pushstr("'Kgg' naming"); xmain(0); /* name the null matrix Kgg */
    if (!rc) { /* rc == 0 is a problem */
        stkerr(" fea_assemble: ","failure in g_set_null for Kgg");
        return 0;
    }
    set_low_tri(tos);  /* Tag Kgg as a lower triangular, symmetric matrix.  */
    set_symm(tos);

    lpull();    /* grab Mgg back from local stack, now on main stack */
    swap();     /* ( Kgg Mgg --- Mgg Kgg ) */
    if (LUMPED) {
        /* populate Mgg's internal row indices by copying them from Kgg */
        SP = sparse_overlay(tos);
        memcpy(MAT_ROW_IDX(tos-1), SP.row_idx, nDOF*sizeof(int));
    }
    /*
     * Read data for a block of elements.
     */
    elements_remain    = 1;
    element_offset     = 0;
    sql_data.row_index = 0;
    while (elements_remain) {

      /* get list of element ID's in sql_data.i[0..(nElem-1)][0] */
      get_next_group_of_eids(dbh             ,
                             partition_number,
                            &element_offset  ,
                            &elements_found  ,
                            &elements_remain ,
                            &sql_data);

if (DEBUG) gprintf("fea_assemble got %d elements:\n", elements_found);
      for (i = 0; i < elements_found; i++) {
        /* Future mod:  load nodal & material data for a block of
         *              elements instead of one at a time.
         */
if (DEBUG > 1) gprintf(" %2d. eid=[%d]\n", i, sql_data.i[i][0]);
        pushint(sql_data.i[i][0]); /* canonical element ID */
        /* ( hMgg hKgg --- hMgg hKgg eid ) */
        pushint(LUMPED);  /* 1 == lumped;   1 != consistent   */
        /* ( hMgg hKgg eid --- hMgg hKgg eid lumped ) */

        /* put this element's mass matrix, hMe, stiffness matrix, hKe,
         * and DOF sequence information, hInd, on the stack 
         */

if (DEBUG) gprintf("fea_assemble working on eid %d\n", sql_data.i[i][0]);
        if (!element_matrix(dbh, partition_number)) 
        /* ( hMgg hKgg dbh eid lumped --- hMgg hKgg hMe hKe hInd ) */
            return 0;
        /* sum the element mass and stiffness matrices into the global
         * (or partition)  mass and stiffness matrices 
         */
        pushstr("5 revn 2 pick 5 pick"); xmain(0);
                /*
                 5 revn (Mgg Kgg Me Ke Ind --- Ind Ke Me Kgg Mgg)
                 2 pick ( ... --- Ind Ke Me Kgg Mgg Me)
                 5 pick ( ... --- Ind Ke Me Kgg Mgg Me Ind)
                 */
        /* sum mass matrix Me into Mgg according to row/col indices in Ind */
        if (LUMPED) {
            pushstr("swap diag swap list: 0 ;"); xmain(0); 
                   /*
                    swap       ( ... --- Ind Ke Me Kgg Mgg Ind Me)
                    diag       ( ... --- Ind Ke Me Kgg Mgg Ind diag(Me))
                    swap       ( ... --- Ind Ke Me Kgg Mgg diag(Me) Ind)
                    list: 0 ;  ( ... --- Ind Ke Me Kgg Mgg diag(Me) Ind [0])
                    */
            /* use [0] as column vector for plusi; Mgg is just one column */
            plusi();     /* (... --- Ind Ke Me Kgg Mgg) */
        } else {
            dup1s();     /* (... --- Ind Ke Me Kgg Mgg Me Ind Ind) */
            pushint(1);  /* coupled mass goes to a sparse matrix; use
                          * lower triangle only option for spsum, lower=1
                          */
            spsum();     /* ( ... --- Ind Ke Me Kgg Mgg) */
//return 1;
//HALT();
        }

        swap();  /* ( ... --- Ind Ke Me Mgg Kgg) */

        /* sum stif. matrix Ke into Kgg according to row/col indices in Ind */
        pushstr("3 pick 5 pick dup 1"       ); xmain(0);
                /*
                 3 pick ( Ind Ke Me Mgg Kgg --- Ind Ke Me Mgg Kgg Ke)
                 5 pick ( ... --- Ind Ke Me Mgg Kgg Ke Ind)
                 dup 1  ( ... --- Ind Ke Me Mgg Kgg Ke Ind Ind 1)
                 (again lower=1 to sum into lower triangle of Kgg)
                 */
if (DEBUG) gprintf("before spsum\n");
        spsum(); /* (... --- Ind Ke Me Mgg Kgg) */
if (DEBUG) gprintf("after  spsum\n");
        pushstr("5 revn drop drop drop"     ); xmain(0);
                /*
                 5 revn ( Ind Ke Me Mgg Kgg --- Kgg Mgg Me Ke Ind)
                 drop   ( ... --- Kgg Mgg Me Ke)
                 drop   ( ... --- Kgg Mgg Me)
                 drop   ( ... --- Kgg Mgg)
                 */
        swap();

      }
    }

    if (LUMPED) {
        swap();
        spdiag();  /* convert dense Mgg vector into sparse diagonal matrix */
        pushstr("Mgg"); naming();
        SP = sparse_overlay(tos);
        /* copy Mgg's internal row indices over to its column indices */ 
        memcpy(SP.col_idx, SP.row_idx, nDOF*sizeof(int));
        swap();
    }
    pushstr("2 roll db_close"); xmain(0);  /* close the database */

    return 1;
} /* 1}}} */
int g_set_null()        /* ( dbh part_id --- hNgg) {{{1 */
/*
 * man entry:  g_set_null {{{2
 * ( dbh part_id --- hNgg ) Put on the stack a sparse lower triangular matrix with the nonzero pattern of the assembled matrix defined in the SQLite database pointed to by the database handle dbh for partition part_id.  If part_id is zero, use the entire model.  Numeric values of the sparse matrix will be null.  The sparse matrix will contain internal indices corresponding to global degrees of freedom.
 * category: FEA
 * related: db_open, db_close, sql, fea_assemble, element_matrix
 * 2}}}
 */
{
int DEBUG = 0;
    char  *name    = "_Ngg";
    int    rc, i, j, c, r, c_dof, dof, dof_row, dof_col, node_row, node_col,
           ncols, nrows, prev_col, prev_row, prev_node_row, n_conn,
           nstr, num_size, cmplx, offset, n_nodal_connections, 
           nNodes, nDOF, n_remain, n_found, node, s_ptr, n_ptr, 
           n_nodes_this_col, nDOF_this_node,
           partition_number, bogus_dbh_partn,
          *nodal_ptr, *nodal_connections, *dof_list, *dof_list_ptr;
    SparseMatrix Ngg;
    char query[QUERY_STR_SIZE+1];
    sqlite3 *dbh;
    callback_data sql_data;
    char    T[ERR_MSG_SIZE+1];

if (DEBUG) gprintf("at top of g_set_null\n");
    if (!popint(&partition_number)) return 0;
    if (!is_dbh(tos)) {
        stkerr(" g_set_null: ", "expecting an SQLite database handle");
        return 0;
    }
    dbh = (sqlite3 *) tos->tex;
    drop();

    /* Load connectivity information from node_node table {{{2 */

    /* The SQLite callback buffer (sql_data) is too small to hold the
     * complete nodal connectivity information.  This is a problem because
     * the data will be traversed a few times to create the sparse matrix
     * structure and accessing it via SQL calls will be a big performance
     * problem.  Instead malloc a temporary array, nodal_connections, to 
     * hold this information.
     *
     * nodal_connections[2*i], nodal_connections[2*i+1]
     * contain node ID's of the ith nodal connection.  (The
     * node numbering begins at 1.)  For example
     *         nodal_connections[2*i]   = 20
     *         nodal_connections[2*i+1] = 37
     * means nodes 20 and 37 are connected.  A node is always connected
     * to itself so N pairs will have the same value for both 2i and 2i+1
     * indices.  The node_node table is created with doubly 
     * sorted ascending node ID values so nodal_connections will likewise
     * contain sorted node ID's.
     *
     * The nodal_ptr array has length N+1.  It contains transition indices 
     * in nodal_connections where data changes from one node to the next
     * such that nodal_connections[ nodal_ptr[j] ] is the first entry in
     * nodal_connections for node j+1.  The last term, nodal_ptr[N+1],
     * contains the total number of nodal_connections.  This makes it easy
     * to figure out the number of nodes connected to any other node.
     * For example, the number of nodes connected to node j+1 is
     *        nodal_ptr[j+1] - nodal_ptr[j].
     *     
     */
    n_remain           = 1;
    offset             = 0;
    sql_data.row_index = 0;
    n_conn             = 0;
    if (partition_number) {
if (DEBUG) gprintf("g_set_null working on partition %d\n",
partition_number);
        snprintf(query, QUERY_STR_SIZE, 
                 "select count(nid_a) from DB_PARTN.P%02d_node_node ",
                 partition_number      );
        n_nodal_connections = sql_scalar_i(dbh, query, "g_set_null");
    } else {
        n_nodal_connections = sql_scalar_i(dbh,
                "select count(nid_a) from node_node", "g_set_null");
    }
if (DEBUG) gprintf("g_set_null n_nodal_connections = %d\n",
n_nodal_connections);

    if (!n_nodal_connections) {
        stkerr(" g_set_null n_nodal_connections"," is null");
        return 0;
    }

    if ((nodal_connections = (int *) malloc(2 * n_nodal_connections *
                                            sizeof(int))) == NULL) {
        stkerr(" g_set_null (nodal_connections): ",MEMNOT);
        return 0;
    }

    nNodes = node_count(dbh, partition_number);
    nDOF   = dof_count( dbh, partition_number);
    if (!nNodes) {
        stkerr(" g_set_null node count"," is null");
        return 0;
    }
    if (!nDOF) {
        stkerr(" g_set_null DOF count"," is null");
        return 0;
    }
    if ((nodal_ptr = (int *) malloc((nNodes+1) * sizeof(int))) == NULL) {
        stkerr(" g_set_null (nodal_ptr): ",MEMNOT);
        return 0;
    }

    while (n_remain) {
        sql_data.nRows = 0;
        /* in the connectivity query below do not use a WHERE clause    */
        /* to only pick up the lower triangular terms.  That will cause */
        /* the last column to be skipped because it has no lower        */
        /* triangular terms. */
        if (partition_number) {
            snprintf(query, QUERY_STR_SIZE, 
                     "select nid_a, nid_b from DB_PARTN.P%02d_node_node "
                     " order by nid_a, nid_b "
                     "limit %d offset %d;" ,
                     partition_number      ,
                     SQL_BLOCK_SIZE, offset);
        } else {
            snprintf(query, QUERY_STR_SIZE, 
                     "select nid_a, nid_b from node_node "
                     " order by nid_a, nid_b "
                     "limit %d offset %d;" ,
                     SQL_BLOCK_SIZE, offset);
        }
if (DEBUG) gprintf("%s\n", query);
        rc = sql_do_query(dbh, query, &sql_data);
        if ((rc != SQLITE_DONE) && (rc != SQLITE_OK)) {
            snprintf(T, ERR_MSG_SIZE,
                     "SQL error: rc=%d %s\n", rc, sql_data.err_msg);
            stkerr(" g_set_null ", T);
            return 0;
        }
        offset  += SQL_BLOCK_SIZE;
        n_found  =  sql_data.nRows;
        n_remain = (n_found == SQL_BLOCK_SIZE ? 1 : 0);
        for (i = 0; i < n_found; i++) {
if (DEBUG) gprintf("g_set_null c %3d  r %3d\n", 
sql_data.i[i][0],sql_data.i[i][1]);
            nodal_connections[n_conn++] = sql_data.i[i][0];
            nodal_connections[n_conn++] = sql_data.i[i][1];
        }
    }
    /* 2}}} */

    /* Load g-set degree of freedom indices {{{2 */

    /* Similar to the nodal_connections[] array above, load the degree
     * of freedom list into an integer(2*nDOF) array, dof_list[].
     *   dof_list[2*i], dof_list[2*i+1]
     * contain the global set canonical 0-based degree of freedom index
     * and the corresponding 0-based canonical node ID.  
     *
     * The associated array dof_list_ptr[] with integer(nNodes+1) terms
     * contains pointers into dof_list[] such that canonical node ID i has 
     *   nDOF_node_i = dof_list_ptr[i+1] - dof_list_ptr[i]
     * degrees of freedom and these degrees of freedom are
     *
     *   for (j = dof_list_ptr[i]; j < dof_list_ptr[i+1]; j++)
     *      printf(" node %i has dof %d\n", i, dof_list[2*j]);
     *
     * The dof_list[2*i+1] value is used only to enforce consistency and
     * trap errors.  In the for-j loop above, this relationship must be true:
     *
     *      dof_list[2*j+1] == j
     *
     * If it isn't the degree of freedom to node mapping is screwed up;
     * have to bail.
     */
    n_remain           = 1;
    offset             = 0;
    sql_data.row_index = 0;

    if ((dof_list = (int *) malloc(2 * nDOF * sizeof(int))) == NULL) {
        stkerr(" g_set_null (dof_list): ",MEMNOT);
        return 0;
    }
    if ((dof_list_ptr = (int *) malloc((nNodes+1) * sizeof(int))) == NULL) {
        stkerr(" g_set_null (dof_list_ptr): ",MEMNOT);
        return 0;
    }
    if (!get_dof_per_node(dbh, nNodes, partition_number, dof_list_ptr)) {
        stkerr(" g_set_null: ", "get_dof_per_node failure");
        return 0;
    }
if (DEBUG) {
for (j = 0; j < nNodes; j++) {
gprintf("g_set_null dof_list_ptr[%3d]=%d (%d dof)\n",
j, dof_list_ptr[j], dof_list_ptr[j+1] - dof_list_ptr[j]); 
}
}

    j = 0;
    while (n_remain) {
        sql_data.nRows = 0;

        if (partition_number) {
            snprintf(query, QUERY_STR_SIZE, 
                "select dof,CN.id from DB_PARTN.P%02d_canonical_nid CN,dof "
                "where dof.nid = CN.nid order by CN.id, dof;"  
                "limit %d offset %d;" ,
                 partition_number, SQL_BLOCK_SIZE, offset);
        } else {
            snprintf(query, QUERY_STR_SIZE, 
                "select dof,nid from dof "
                "order by nid, dof limit %d offset %d;" ,
                SQL_BLOCK_SIZE, offset);
        }

if (DEBUG) gprintf("%s\n", query);
        rc = sql_do_query(dbh, query, &sql_data);
        if ((rc != SQLITE_DONE) && (rc != SQLITE_OK)) {
            snprintf(T, ERR_MSG_SIZE,
                     "SQL error: rc=%d %s\n", rc, sql_data.err_msg);
            stkerr(" g_set_null ", T);
            return 0;
        }
        offset  += SQL_BLOCK_SIZE;
        n_found  =  sql_data.nRows;
        n_remain = (n_found == SQL_BLOCK_SIZE ? 1 : 0);
        for (i = 0; i < n_found; i++) {
if (DEBUG) gprintf("g_set_null global dof %3d belongs to node %3d\n", 
sql_data.i[i][0],sql_data.i[i][1]);
            dof_list[j++] = sql_data.i[i][0];
            dof_list[j++] = sql_data.i[i][1];
        }
    }
    /* 2}}} */

    /* Pass 1 to determine total number of strings and non-zero terms {{{2 */
    prev_col = -1;
    prev_row = -1;
    nstr     = 0;
    num_size = 0;
    cmplx    = 0;
    node     = 0;

if (DEBUG) gprintf("g_set_null top P1 nodal_connections[0,1]= %d %d\n", 
nodal_connections[0], nodal_connections[1] );

    prev_col = -1;
    prev_row = -1;  /* diagonal term; match column */
    for (i = 0; i < n_nodal_connections; i++) {
        node_col = nodal_connections[2*i  ];
        node_row = nodal_connections[2*i+1];
        nDOF_this_node = dof_list_ptr[node_col+1] - dof_list_ptr[node_col];
if (DEBUG) gprintf("i= %2d   col=%2d   row=%2d  ", i, node_col, node_row);
        if (node_col != prev_col) {
            /* new columns start on the diagonal due to self-  */
            /* connectivity                                    */
            /* start a new node column which means start */
            nodal_ptr[node++] = i;
            nstr     += nDOF_this_node;
            /* since we're only populating the lower triangle the */
            /* diagonal block only has half the numeric terms of  */
            /* a regular nodal block                              */
            num_size += nDOF_this_node*(nDOF_this_node + 1)/2;
            prev_row = node_col;
if (DEBUG) gprintf("new column at col=%2d", node_col);
}else{
if (DEBUG) gprintf("                    ");
        }
        if (node_row >= node_col) {
            /* only do lower triangular terms */
if (DEBUG) gprintf(" LT r=%2d/pr=%2d ", node_row, prev_row);
            if ((prev_row + 1) == node_row) {
                /* same column, continuation of previous string */
                num_size += nDOF_this_node*nDOF_this_node;
            } else {  
                /* same column but start new string */
                nstr     += nDOF_this_node;
                num_size += nDOF_this_node*nDOF_this_node;
            }
            prev_row = node_row;
}else{
if (DEBUG) gprintf(" u             ");
        }
        prev_col = node_col;
if (DEBUG) gprintf(" nStr=%2d  nnZ=%3d\n", nstr, num_size);
    }

    nodal_ptr[node] = n_nodal_connections;
    /* 2}}} */

    ncols = nDOF;
    nrows = nDOF;
if (DEBUG) {
gprintf("nNodes=%d  n_nodal_connections=%d\n", nNodes, n_nodal_connections);
for (i = 0; i <= nNodes; i++) {
gprintf("nodal_ptr[%3d] = %3d\n", i, nodal_ptr[i]); 
if (i == nNodes) break;
for (j = nodal_ptr[i]; j < nodal_ptr[i+1]; j++) {
gprintf("                      nodal_connections[%3d,%3d] = %3d %3d\n", 
                  2*j ,                   2*j + 1,
nodal_connections[2*j], nodal_connections[2*j + 1]);
}
}
gprintf("\n"); 
}
if (DEBUG) gprintf("g_set_null Ngg: %d rows, %d cols, %d strings, %d nonZ\n", 
nrows, ncols, nstr, num_size);

    if (!sparse_stk(nrows      , /* in  */
                    ncols      , /* in  */
                    nstr       , /* in  number of strings       */
                    num_size   , /* in  number of nonzero terms */
                    cmplx      , /* in  0=real  1=complex       */
                    1          , /* in  is internally indexed   */
                    name       , /* in  */
                   &Ngg)) {      /* out */
        return 0;
    }

    /* Pass 2 to fill in string and numeric pointers {{{2 */
    Ngg.S[0].start_row =  0;
    Ngg.S[0].len       =  0;
    Ngg.S[0].N_idx     =  0;
    Ngg.S_start[0]     =  0;
    Ngg.N_start[0]     =  0;
    s_ptr              = -1;
    n_ptr              =  0;

    for (c = 0; c < nNodes; c++) {

      nDOF_this_node = dof_list_ptr[c+1] - dof_list_ptr[c];
      for (c_dof = 0; c_dof < nDOF_this_node; c_dof++) {
        dof_col  = dof_list_ptr[c] + c_dof;

        /* first do the diagonal block */

        ++s_ptr;
        Ngg.S_start[dof_col]   = s_ptr;
        Ngg.N_start[dof_col]   = n_ptr;
        /* .start_row and .len values have to be adjusted to
         * account for being in the lower triangle only; the
         * diagonal start row index is the same as the column index
         */
        Ngg.S[s_ptr].start_row = dof_col;
        Ngg.S[s_ptr].len       = nDOF_this_node - c_dof;
        Ngg.S[s_ptr].N_idx     = n_ptr;
        n_ptr                 += nDOF_this_node - c_dof;

        prev_node_row = c;

        n_nodes_this_col = nodal_ptr[c+1] - nodal_ptr[c];
if (DEBUG) gprintf("g_set_null P2  c=%2d  c_dof=%1d  n_nodes_this_col=%2d\n",
c, c_dof, n_nodes_this_col);
        for (r = 0; r < n_nodes_this_col; r++) {
          node_row = nodal_connections[ 2*nodal_ptr[c] + 1 + 2*r];
if (DEBUG) gprintf("g_set_null P2  r=%2d  nod_row=%2d\n",
r, node_row);
          if (node_row > c) { /* only terms below the diagonal */
if (DEBUG) gprintf("g_set_null P2 nodal_connections[ (2* (nodal_ptr[%d]=%d) + 1 + 2*%d)=%d]=%d\n",
c, nodal_ptr[c], r, 2*nodal_ptr[c] + 1 + 2*r, 
nodal_connections[ 2*nodal_ptr[c] + 1 + 2*r]); 
if (DEBUG) gprintf("g_set_null P2 c=%d c_dof=%d r=%d  node_row=%d  dof_col=%d\n",
c, c_dof, r, node_row, dof_col);
  
            dof_row = dof_list_ptr[node_row];
  
            if (node_row == prev_node_row+1) { /* string continues from prev */
if (DEBUG) gprintf("string continuation (%d) at col=%d row=%d\n", 
s_ptr, dof_col, dof_row);
              Ngg.S[s_ptr].len += nDOF_this_node;
            } else {                           /* start new string       */
              ++s_ptr;
if (DEBUG) gprintf("new string (%d) at col=%d row=%d\n", 
s_ptr, dof_col, dof_row);
              Ngg.S[s_ptr].start_row = dof_row;
              Ngg.S[s_ptr].len   = nDOF_this_node;
              Ngg.S[s_ptr].N_idx = Ngg.S[s_ptr-1].N_idx + Ngg.S[s_ptr-1].len;
            }
  
            prev_node_row = node_row;
            n_ptr += nDOF_this_node;
          }
        }
if (DEBUG) gprintf("\n"); 
      }
    }
    Ngg.S_start[dof_list_ptr[nNodes]] = s_ptr + 1;
    Ngg.N_start[dof_list_ptr[nNodes]] = n_ptr;

    for (i = 0; i < n_ptr; i++)
        Ngg.N[i] = 0.0;

    /* 2}}} */

    /* populate row/column indices with canonical g-set DOF {{{2 */
    for (i = 0; i < nDOF; i++) {
        Ngg.row_idx[i] = dof_list[2*i];
        Ngg.col_idx[i] = dof_list[2*i];
    }
    /* 2}}} */

    free(dof_list_ptr);
    free(nodal_connections);
    free(nodal_ptr);
    free(dof_list);
    return 1;
} /* 1}}} */
int element_matrix(     /* ( dbh_m eid lumped --- hM hK hInd ) {{{1 */
                   sqlite3 *sql_dbh,
                   int      part_number)
/*
 * man entry:  element_matrix {{{2
 * ( dbh_m eid lumped --- hM hK hInd ) Stack contains an SQLite database handle, a canonical element ID (relative to the entire model), and a flag indicating lumped or coupled mass matrix.  This word queries the database for the element's nodes and material properties, then puts on the stack the element's mass and stiffness matrices and an array of indices t.  lumped=1 produces a lumped mass matrix; lumped!=1 produces a coupled or consistent mass matrix.
 * category: FEA
 * related: fea_assemble, sparse
 * 2}}}
 */
/* Future mod:  switch/case statement to call data recovery
 *              routine appropriate for each element type
 *              eg, type "tri3" calls "get_tri3_data"
 */
{
int DEBUG = 0;  /* 1, 2 (very verbose) */
    double  Young_modulus, Poisson_ratio, thickness, area, density,
            Shear_modulus, x1, x2, x3, y1, y2, y3,
            alpha = 1.5, beta = 0.5, zip,
            ek9[9*9],
           *ek,        /* ek[18*18]    */
           *em,        /* em[18*18]    */
           *em_diag,   /* em_diag[18]  */
           *global_dof_index; /* global_dof_index[18] */
    XYZ     Bx[3],     /* node coordinates in basic coordinate system */
            unit_normal, delta;
    int     ok, i, dof, node, lumped,
            *M_row_ind, *M_col_ind, *K_row_ind, *K_col_ind,
            element_id   =  0,
            plane_stress =  1,  /* >0 = plane stress
                                 * <0 = plane strain
                                 */
            node_index[3]; /* node ID for tri3 */
    char    K_name[SQL_MAX_ID_LEN], M_name[SQL_MAX_ID_LEN], 
            index_name[SQL_MAX_ID_LEN];
    char    query[QUERY_STR_SIZE+1];

    if (!popint(&lumped)) return 0;
    if (lumped != 1)  /* 1 = lumped mass;  2 = consistent mass */
        lumped = 2;

    if (!popint(&element_id)) return 0;

    snprintf(M_name,     SQL_MAX_ID_LEN, "_tri3_M_%d",   element_id);
    snprintf(K_name,     SQL_MAX_ID_LEN, "_tri3_K_%d",   element_id);
    snprintf(index_name, SQL_MAX_ID_LEN, "_tri3_dof_%d", element_id);

    if (lumped == 1) {
        /* For lumped mass formulation only put the diagonal of the mass
         * matrix on the stack.  However, still need a dense 18x18 working
         * matrix to call elmmascst(), elmmasmrt(), elmmasplt().
         */

        /* put full 18x18 even for lumped mass matrix */
        if (!matstk_idx(18, 18, M_name))     return 0;
        em = tos->mat;
    } else {
        if (!matstk_idx(18, 18, M_name))     return 0;
        em = tos->mat;
    }
    M_row_ind = MAT_ROW_IDX(tos);
    if (lumped != 1) {
        M_col_ind = MAT_COL_IDX(tos);
    }

    if (!matstk_idx(18, 18, K_name))     return 0;
    ek = tos->mat;
    K_row_ind = MAT_ROW_IDX(tos);
    K_col_ind = MAT_COL_IDX(tos);

    if (!matstk(18,  1, index_name)) return 0;
    global_dof_index = tos->mat;

    /* enable the index used tag */
    set_index_used(tos-1); /* [element K] */
    set_index_used(tos-2); /* [element M] */

if (DEBUG) gprintf("element_matrix before get_tri3_data for EID %d\n", element_id);
    ok = get_tri3_data(sql_dbh         , /* in , database handle */
                       part_number     , /* in       */
                       element_id      , /* in , EID */
                       Bx              , /* out, x[0..2][0..2] node coords */
                       node_index      , /* out, NID's          */
                       global_dof_index, /* out, degrees of freedom */
                      &Young_modulus   , /* out      */
                      &Poisson_ratio   , /* out      */
                      &thickness       , /* out      */
                      &density);         /* out      */
if (DEBUG) gprintf("element_matrix after  get_tri3_data for EID %d\n", element_id);
    if (!ok) {
        stkerr(" element_matrix, no such element: ",STRSNOT);
        return 0;
    }
    ok = tri3_prop_sanity_check(Bx           , 
                                Young_modulus, 
                                Poisson_ratio, 
                                thickness    , 
                                density      );
    if (!ok) {
        snprintf(index_name, SQL_MAX_ID_LEN, 
                 "canonical element id %d", element_id);
        stkerr(" tri3 failed sanity check ", index_name);
        return 0;
    }

    /* insert indexing terms into the matrix */
    for (i = 0; i < 18; i++) {
        M_row_ind[i] = (int) global_dof_index[i];
        if (lumped != 1) {
            M_col_ind[i] = (int) global_dof_index[i];
        }
        K_row_ind[i] = (int) global_dof_index[i];
        K_col_ind[i] = (int) global_dof_index[i];
    }

if (DEBUG) gprintf("\nelement_matrix 3\n");
    area        = triangle_area(Bx);
if (DEBUG) gprintf("element_matrix 4\n");
    unit_normal = triangle_unit_normal(Bx);
if (DEBUG) gprintf("element_matrix 5\n");
if (DEBUG) {
gprintf("element_matrix [%2d] E=%10.3e nu=%10.3e t=%10.3e area=%10.3e\n", 
element_id, Young_modulus, Poisson_ratio, thickness, area);
}
if (DEBUG) {
gprintf("Global coords\n");
gprintf("  n[0]: % 10.3e % 10.3e % 10.3e\n", Bx[0].x, Bx[0].y, Bx[0].z);
gprintf("  n[1]: % 10.3e % 10.3e % 10.3e\n", Bx[1].x, Bx[1].y, Bx[1].z);
gprintf("  n[2]: % 10.3e % 10.3e % 10.3e\n", Bx[2].x, Bx[2].y, Bx[2].z);
gprintf("  Un  : % 10.3e % 10.3e % 10.3e\n", unit_normal.x,
                                             unit_normal.y,
                                             unit_normal.z);
}
    x1 = Bx[0].x;
    y1 = Bx[0].y;

    delta       = Rot3(unit_normal, Sub3(Bx[1], Bx[0]) );
if (DEBUG) gprintf("1->2: % 10.3e % 10.3e % 10.3e\n",delta.x,delta.y,delta.z);
    x2 = Bx[0].x + delta.x;
    y2 = Bx[0].y + delta.y;

    delta       = Rot3(unit_normal, Sub3(Bx[2], Bx[0]) );
if (DEBUG) gprintf("1->3: % 10.3e % 10.3e % 10.3e\n",delta.x,delta.y,delta.z);
    x3 = Bx[0].x + delta.x;
    y3 = Bx[0].y + delta.y;

    Shear_modulus = Young_modulus/(2.0*(1.0 + Poisson_ratio));
    zip           = (thickness*thickness*thickness)/12.0/
                    (1.0 - Poisson_ratio*Poisson_ratio);

/* Values from ../elements/Cdriver.c for testing */
/* pl0   */ /* plane_stress  =  1;               */ 
/* e0    */ /* Young_modulus =  10000000.0;      */
/* g0    */ /* Shear_modulus =  4000000.0;       */
/* t0    */ /* thickness     =  1.0;             */
/* alpha */ /* alpha         =  1.50;            */
/* beta  */ /* beta          =  0.5;             */
/* area  */ /* area          =  0.00017119963;   */
/* xb1   */ /* x1            =  0.221399993;     */
/* xb2   */ /* x2            =  0.200000003;     */
/* xb3   */ /* x3            =  0.200000003;     */
/* yb1   */ /* y1            = -0.792400002;     */
/* yb2   */ /* y2            = -0.77640003;      */
/* yb3   */ /* y3            = -0.792400002;     */
/* zip0  */ /* zip           =  1.0;             */

    for (i = 0; i < 18*18; i++) {
        ek[i] = 0.0;
        em[i] = 0.0;
    }

if (DEBUG > 1) {
gprintf("element_matrix inputs to mrt, dkt:\n"); 
gprintf("E=%12.6e G=%12.6e t=%12.6e\n",Young_modulus, Shear_modulus, thickness);
gprintf("plane_stress=%d  \n", plane_stress);
gprintf("alpha=%12.6e beta=%12.6e area=%12.6e zip=%12.6e\n",
        alpha, beta, area, zip);
}
if (DEBUG) {
gprintf("Local coords\n");
gprintf("  n[0]: % 10.3e % 10.3e\n", x1, y1);
gprintf("  n[1]: % 10.3e % 10.3e\n", x2, y2);
gprintf("  n[2]: % 10.3e % 10.3e\n", x3, y3);
}
    /* element stiffness matrix */
    elmstfmrt_(&Young_modulus, &Shear_modulus, &thickness, 
               &plane_stress, &alpha, &beta, &area,
               &x1, &x2, &x3, 
               &y1, &y2, &y3, ek, ek9);
if (DEBUG > 1) {
gprintf("element_matrix ek after elmstfmrt_=\n"); 
for (i = 0; i < 18*18; i++) gprintf(" %6d  %6d % 14.8e\n",(i%18)+1,(i/18)+1,ek[i]); 
}

    elmstfdkt_(&Young_modulus, &Shear_modulus, &thickness, &zip, &area,
               &x1, &x2, &x3, &y1, &y2, &y3, ek, ek9 );
if (DEBUG > 1) {
gprintf("element_matrix ek after elmstfdkt_, before rotation=\n"); 
for (i = 0; i < 18*18; i++) gprintf(" %6d  %6d % 14.8e\n",(i%18)+1,(i/18)+1,ek[i]); 
}
    /* element mass matrix */
    elmmascst_(&density, &area, &thickness, em, &lumped);
if (DEBUG > 1) {
gprintf("element_matrix em after elmmascst_\n"); 
for (i = 0; i < 18*18; i++) {
if (fabs(em[i]) > EPSILON) gprintf(" em[%3d,%3d]= % e\n", i/18, i%18, em[i]); 
}
}
    elmmasmrt_(&density, &area, &thickness, em,  
               &x1, &x2, &x3, &y1, &y2, &y3,    &lumped);
if (DEBUG > 1) {
gprintf("element_matrix em after elmmasmrt_\n"); 
for (i = 0; i < 18*18; i++) {
if (fabs(em[i]) > EPSILON) gprintf(" em[%3d,%3d]= % e\n", i/18, i%18, em[i]); 
}
}
    elmmasplt_(&density, &area, &thickness, em,  
               &x1, &x2, &x3, &y1, &y2, &y3,    &lumped);
if (DEBUG > 1) {
gprintf("element_matrix em after elmmasplt_\n"); 
for (i = 0; i < 18*18; i++) {
if (fabs(em[i]) > EPSILON) gprintf(" em[%3d,%3d]= % e\n", i/18, i%18, em[i]); 
}
}

    /* rotate element matrices back to the global (basic) coordinate system */
    rotate_tri3_elem(unit_normal, em);
if (DEBUG > 1) {
gprintf("element_matrix em after rotation\n"); 
for (i = 0; i < 18*18; i++) {
if (fabs(em[i]) > EPSILON) gprintf(" em[%3d,%3d]= % e\n", i/18, i%18, em[i]); 
}
}
    /* after mass matrix values debugged, go back to lumped formulation
    if (lumped == 1) {
        for (i = 0; i < 18; i++) {
            em_diag[i] = em[i + 18*i];
        }
        free(em);
    }
    */

    rotate_tri3_elem(unit_normal, ek);

if (DEBUG > 1) {
gprintf("element_matrix ek after rotation=\n"); 
for (i = 0; i < 18*18; i++) gprintf(" %6d  %6d % 14.8e\n",(i%18)+1,(i/18)+1,ek[i]); 
}

    return 1;
} /* 1}}} */

int get_next_group_of_eids(sqlite3       *dbh             , /* {{{1 */
                           int            partition_number,
                           int           *element_offset  ,
                           int           *elements_found  ,
                           int           *elements_remain ,
                           callback_data *sql_data) {
    char query[QUERY_STR_SIZE+1];
    int  rc;
    char T[ERR_MSG_SIZE+1];
int DEBUG = 0;
if (DEBUG) gprintf("top of get_next_group_of_eids\n");

    (*sql_data).nRows = 0;
    /* Future mod:  also get back element type, eg, "tri3" */
    /* after query, data.a[][] will contain element ID's */
    if (partition_number) {
        snprintf(query, QUERY_STR_SIZE, 
                "select eid from DB_PARTN.element_partition "
                "where pid=%d "
                "order by eid limit %d offset %d;" ,
                partition_number ,
                SQL_BLOCK_SIZE, *element_offset);
    } else {
        snprintf(query, QUERY_STR_SIZE, 
                "select seq_no from element "
                "order by seq_no limit %d offset %d;" ,
                SQL_BLOCK_SIZE, *element_offset);
    }
if (DEBUG) gprintf("%s\n", query);
    rc = sql_do_query(dbh, query, sql_data);
    *element_offset += SQL_BLOCK_SIZE;
    if ((rc != SQLITE_DONE) && (rc != SQLITE_OK)) {
        snprintf(T, ERR_MSG_SIZE,
                 "SQL error: rc=%d %s\n", rc, (*sql_data).err_msg);
        stkerr(" get_next_group_of_eids ", T); 
        return 0;
    }
    *elements_found  = (*sql_data).nRows;
    *elements_remain = (*elements_found == SQL_BLOCK_SIZE ? 1 : 0);
}
/* 1}}} */
int get_tri3_data(sqlite3 *dbh          , /* in   database handle {{{1 */
                  int      part_number  , /* in   partition number (0 = all) */
                  int      EID          , /* in   (canonical) */
                  XYZ     *coord        , /* out, x[0..#nodes-1] node coords */
                  int     *new_id       , /* out, NID's */
                  double  *global_dof   , /* out, global degrees of freedom */
                  double  *Young_modulus, /* out      */
                  double  *Poisson_ratio, /* out      */
                  double  *thickness    , /* out      */
                  double  *density)       /* out      */
{
int DEBUG = 0;
    int   i, rc;
    char query[QUERY_STR_SIZE+1];
    callback_data sql_data;
    char T[ERR_MSG_SIZE+1];

if (DEBUG) gprintf("get_tri3_data for normalized eid=[%d]\n", EID);
    /*
     * Get node ID's and coordinates for this element.
     */
    sql_data.nRows = 0;

    if (part_number) {
        snprintf(query, QUERY_STR_SIZE, 
               "select C.id, x1, x2, x3 from "
               "    (select N.seq_no as seq_no, "
               "            N.x1 as x1, N.x2 as x2, N.x3 as x3 "
               "        from element_node EN, node N "
               "            where EN.eid=%d and EN.nid = N.seq_no "
               "            order by EN.seq_no ) "
               "    as foo inner join DB_PARTN.P%02d_canonical_nid C "
               "    on C.nid = foo.seq_no; ",
                 EID        ,
                 part_number);
    } else {
        snprintf(query, QUERY_STR_SIZE, 
            "select N.seq_no, N.x1, N.x2, N.x3 "
            "from element_node EN, node N "
            "  where EN.eid=%d and N.seq_no=EN.nid "
            "        order by EN.seq_no;" , 
                 EID);
    }
if (DEBUG) gprintf("%s\n", query);
    rc = sql_do_query(dbh, query, &sql_data);
    if ((rc != SQLITE_DONE) && (rc != SQLITE_OK)) {
        snprintf(T, ERR_MSG_SIZE,
                 "SQL error: rc=%d %s\n", rc, sql_data.err_msg);
        stkerr(" get_tri3_data ", T);
        return 0;
    }
    if (!sql_data.nRows) {
        stkerr(" get_tri3_data "," node coordinate query returns empty ");
        return 0;
    }
    if (sql_data.nRows != 3) {
        stkerr(" get_tri3_data "," did not get three nodal coordinates");
        return 0;
    }
    for (i = 0; i < sql_data.nRows; i++) { /* .nRows = # nodes this element */
        new_id[i]  = sql_data.i[i][0];  /* seq_no starts at 0 */
        coord[i].x = sql_data.x[i][1];
        coord[i].y = sql_data.x[i][2];
        coord[i].z = sql_data.x[i][3];
    }
if (DEBUG) {
for (i = 0; i < 3; i++) {
gprintf("get_tri3_data  % 14d: x,y,z =% 12.6e,% 12.6e,% 12.6e\n", 
        new_id[i], coord[i].x, coord[i].y, coord[i].z);
}
}
    /* get the global degrees of freedom associated with the vertices */
    sql_data.nRows = 0;
    snprintf(query, QUERY_STR_SIZE, 
            "select dof from element_node EN, dof where eid=%d "
            "   and EN.nid=dof.nid order by EN.seq_no;" , EID);
    rc = sql_do_query(dbh, query, &sql_data);
    if ((rc != SQLITE_DONE) && (rc != SQLITE_OK)) {
        snprintf(T, ERR_MSG_SIZE,
                 "SQL error: rc=%d %s\n", rc, sql_data.err_msg);
        stkerr(" get_tri3_data: ", T);
        return 0;
    }
    if (sql_data.nRows != 18) {
        snprintf(T, ERR_MSG_SIZE, 
                 "expected 18 dof for canonical element %d, found %d", 
                 EID, sql_data.nRows);
        stkerr(" get_tri3_data: ", T);
        return 0;
    }
    for (i = 0; i < sql_data.nRows; i++) {
        global_dof[i] = (double) sql_data.i[i][0];
    }

    /*
     * Find Young's modulus, Poisson's ratio, thickness for this element.
     */
    sql_data.nRows = 0;

if (DEBUG) gprintf("get_tri3_data: %s\n", query);
    snprintf(query, QUERY_STR_SIZE, 
            "select M.E, M.nu, M.rho, P.thick "
                "from tri3 T, shell_prop P, material M "
                "where T.id=%d and T.shell_prop=P.id "
                "and M.id=P.material_id;" ,
             EID);
if (DEBUG) gprintf("get_tri3_data: \n%s\n", query);
    rc = sql_do_query(dbh, query, &sql_data);
    if (!sql_data.nRows) {
        stkerr(" get_tri3_data "," no hits for E,nu,rho query ");
        return 0;
    }
    if ((rc != SQLITE_DONE) && (rc != SQLITE_OK)) {
        snprintf(T, ERR_MSG_SIZE,
                 "SQL error: rc=%d %s\n", rc, sql_data.err_msg);
        stkerr(" get_tri3_data ", T);
        return 0;
    }
    *Young_modulus  = sql_data.x[0][0];
    *Poisson_ratio  = sql_data.x[0][1];
    *density        = sql_data.x[0][2];
    *thickness      = sql_data.x[0][3];

if (DEBUG) {
gprintf("get_tri3_data  E=%12.6e nu=%12.6e t=%12.6e rho=%12.6e\n", 
         *Young_modulus, *Poisson_ratio, *thickness, *density);
for (i = 0; i < 3; i++) {
gprintf("get_tri3_data  % 14d: x,y,z =% 12.6e,% 12.6e,% 12.6e\n", 
        new_id[i], coord[i].x, coord[i].y, coord[i].z);
}
}
    return 1;
} /* 1}}} */
int tri3_prop_sanity_check(XYZ    vertices[3]   ,  /* {{{1 */
                           double Young_modulus , 
                           double Poisson_ratio , 
                           double thickness     , 
                           double density      ) {
int DEBUG = 0;
    double angle_1, angle_2, angle_3; /* will be in degrees */
    int    ok = 1;

    double INTERIOR_ANGLE_MIN =   5.0;   /* degrees */
    double INTERIOR_ANGLE_MAX = 150.0;   /* degrees */

if (DEBUG) {
gprintf("tri3_prop_sanity_check E=%12.6e, nu=%8.3e, t=%8.3e, rho=%8.3e\n",
Young_modulus, Poisson_ratio, thickness, density);
gprintf("n1 xyz= % 12.6e  % 12.6e  % 12.6e\n",
vertices[0].x, vertices[0].y, vertices[0].z);
gprintf("n2 xyz= % 12.6e  % 12.6e  % 12.6e\n",
vertices[1].x, vertices[1].y, vertices[1].z);
gprintf("n3 xyz= % 12.6e  % 12.6e  % 12.6e\n",
vertices[2].x, vertices[2].y, vertices[2].z);
}

    angle_1  = acos( Dot3( Unit3( Sub3(vertices[1], vertices[0]) ), 
                           Unit3( Sub3(vertices[2], vertices[0]) ) ) ) * DPR; 
    angle_2  = acos( Dot3( Unit3( Sub3(vertices[0], vertices[1]) ), 
                           Unit3( Sub3(vertices[2], vertices[1]) ) ) ) * DPR; 
    angle_3  = 180.0 - angle_1 - angle_2;

if (DEBUG) {
gprintf("tri3_prop_sanity_check ang_1=% 8.3e ang_2=% 8.3e ang_3=% 8.3e (deg)\n",
angle_1, angle_2, angle_3);
}

    if        (Young_modulus <= 0.0   ) {
        gprintf("tri3_prop_sanity_check:  E <= 0\n");
        ok = 0;
    } else if (Poisson_ratio <  0.0   ) {
        gprintf("tri3_prop_sanity_check:  nu < 0\n");
        ok = 0;
    } else if (Poisson_ratio >= 0.51  ) {
        gprintf("tri3_prop_sanity_check:  nu >= 0.51\n");
        ok = 0;
    } else if (thickness     <= 0.0   ) {
        gprintf("tri3_prop_sanity_check:  thickness <= 0\n");
        ok = 0;
    } else if (thickness     >  1.0e3 ) {
        gprintf("tri3_prop_sanity_check:  thickness > 1000\n");
        ok = 0;
    } else if (density       <= 0.0   ) {
        gprintf("tri3_prop_sanity_check:  rho <= 0\n");
        ok = 0;
    } else if ((angle_1 < INTERIOR_ANGLE_MIN) || 
               (angle_1 > INTERIOR_ANGLE_MAX) ||
               (angle_2 < INTERIOR_ANGLE_MIN) || 
               (angle_2 > INTERIOR_ANGLE_MAX) ||
               (angle_3 < INTERIOR_ANGLE_MIN) || 
               (angle_3 > INTERIOR_ANGLE_MAX)) {
        gprintf("tri3_prop_sanity_check:  element aspect ratio too high\n");
        ok = 0;
    }
    return ok;
} /* 1}}} */
int node_count(sqlite3 *dbh             , /* {{{1 */
               int      partition_number) {
int DEBUG = 0;
    int  nNodes = 0;
    char query[QUERY_STR_SIZE+1];
    callback_data sql_data;

if (DEBUG) gprintf("top of node_count\n");

    sql_data.nRows = 0;
    if (partition_number) {
        snprintf(query, QUERY_STR_SIZE, 
                 "select count(id) from DB_PARTN.P%02d_canonical_nid ",
                 partition_number);
if (DEBUG) gprintf("node_count query[%s]\n", query);
        nNodes = sql_scalar_i(dbh, query, "node_count");
    } else {
        nNodes = sql_scalar_i(dbh, 
                              "select count(seq_no) from node", "node_count");
    }
    return nNodes;
}
/* 1}}} */
int dof_count( sqlite3 *dbh             , /* {{{1 */
               int      partition_number) {
int DEBUG = 0;
    char query[QUERY_STR_SIZE+1];
    callback_data sql_data;
    int  nDOF = 0;

if (DEBUG) gprintf("top of dof_count\n");

    sql_data.nRows = 0;
    if (partition_number) {
        snprintf(query, QUERY_STR_SIZE, 
                 "select count(dof) from ("
                 "select dof from dof, DB_PARTN.P%02d_canonical_nid as CN "
                 "where dof.nid = CN.nid);",
                 partition_number);
if (DEBUG) gprintf("dof_count query[%s]\n", query);
        nDOF = sql_scalar_i(dbh, query, "dof_count");
    } else {
        nDOF = sql_scalar_i(dbh, "select count(dof) from dof", "dof_count");
    }
    return nDOF;
}
/* 1}}} */
int get_dof_per_node( sqlite3 *dbh             , /* {{{1 */
                      int      nNodes          ,
                      int      partition_number,
                      int     *dof_list_ptr    ) {
int DEBUG = 0;
    char          query[QUERY_STR_SIZE+1];
    callback_data sql_data;
    char          T[ERR_MSG_SIZE+1];
    int rc, i;
    int offset   = 0;
    int node     = 0;
    int n_found  = 0;
    int n_remain = 1;

if (DEBUG) gprintf("top of get_dof_per_node\n");

    dof_list_ptr[node++] = 0;
    sql_data.row_index   = 0;
    while (n_remain) {
        sql_data.nRows = 0;
        if (partition_number) {
            snprintf(query, QUERY_STR_SIZE, 
                     "select count(dof), CN.id from dof, "
                     "DB_PARTN.P%02d_canonical_nid as CN "
                     "   where dof.nid = CN.nid group by CN.id "
                     "limit %d offset %d;" ,
                     partition_number      ,
                     SQL_BLOCK_SIZE, offset);
        } else {
            snprintf(query, QUERY_STR_SIZE, 
                     "select count(dof), nid from dof group by nid "
                     "limit %d offset %d;" ,
                     SQL_BLOCK_SIZE, offset);
        }
if (DEBUG) gprintf("get_dof_per_node query[%s]\n", query);

        rc = sql_do_query(dbh, query, &sql_data);
        if ((rc != SQLITE_DONE) && (rc != SQLITE_OK)) {
            snprintf(T, ERR_MSG_SIZE,
                     "SQL error: rc=%d %s\n", rc, sql_data.err_msg);
            stkerr(" get_dof_per_node ", T);
            return 0;
        }

        offset  += SQL_BLOCK_SIZE;
        n_found  = sql_data.nRows;
        n_remain = (n_found == SQL_BLOCK_SIZE ? 1 : 0);
        for (i = 0; i < n_found; i++) {
            dof_list_ptr[node] = dof_list_ptr[node-1] + sql_data.i[i][0];
if (DEBUG) gprintf("get_dof_per_node  count(dof)=%d  nid=%3d  dof_list_ptr[%2d]=%d\n", 
sql_data.i[i][0],sql_data.i[i][1], node, dof_list_ptr[node]);
            node++;
        }
       n_remain = (n_found == SQL_BLOCK_SIZE ? 1 : 0);
    }
    return 1;
}
/* 1}}} */
XYZ    triangle_unit_normal(XYZ vertices[3]) {  /* {{{1 */
    /*
     * Returns a unit vector perpendicular to the triangle defined
     * by vertices[0], vertices[1], vertices[2] (in that order).
     */
    XYZ     C;
    double  len;

    C    = Cross3( Sub3(vertices[1], vertices[0]), 
                   Sub3(vertices[2], vertices[0]) ); 
    len  = Mag3(C);
    C.x /= len;
    C.y /= len;
    C.z /= len;

    return C;

} /* 1}}} */
double triangle_area(XYZ vertices[3]) {  /* {{{1 */
    /*
     * Returns the area of the triangle defined by vertices[0], 
     * vertices[1], vertices[2].
     */

    return Mag3 ( Cross3( Sub3(vertices[1], vertices[0]), 
                          Sub3(vertices[2], vertices[0]) ) ) / 2.0; 

} /* 1}}} */
XYZ    Cross3(XYZ A, XYZ B) {  /* {{{1 */
    /*
     * Returns   A x B
     */

    XYZ     C;
    C.x = A.y * B.z  -  B.y * A.z;
    C.y = B.x * A.z  -  A.x * B.z;
    C.z = A.x * B.y  -  B.x * A.y;
    return C;

} /* 1}}} */
XYZ    Unit3(XYZ A) {  /* {{{1 */
    /*
     * Returns a unit vector pointing in the same direction as A, ie,  A/|A|
     */

    XYZ     C;
    double  len;

    len = Mag3(A);
    C.x = A.x/len;
    C.y = A.y/len;
    C.z = A.z/len;
    return C;

} /* 1}}} */
double Dot3(XYZ A, XYZ B) {  /* {{{1 */
    /*
     * Returns   A . B
     */
    return A.x * B.x  +  A.y * B.y  +  A.z * B.z;

} /* 1}}} */
XYZ    Sub3(XYZ A, XYZ B) {  /* {{{1 */
    /*
     * Returns   A - B
     */
    XYZ     C;
    C.x = A.x - B.x;
    C.y = A.y - B.y;
    C.z = A.z - B.z;
    return C;

} /* 1}}} */
double Mag3(XYZ A) {  /* {{{1 */
    /*
     * Returns the length of vector A
     */
    return sqrt( A.x*A.x  +  A.y*A.y  +  A.z*A.z );

} /* 1}}} */
XYZ    Rot3(XYZ Unit, XYZ V) {  /* {{{1 */
    /*
     * Returns offsets needed to rotate point V by rotation vector {Unit}.
     * stadyn/assm325.f::rotvec
     */
int DEBUG = 0;
    XYZ     C;
    double  R[3][3];

    rotation_3x3(Unit, R);

if (DEBUG) {
gprintf("rot in % 10.3e % 10.3e % 10.3e\n", V.x, V.y, V.z);
gprintf("rot 0: % 10.3e % 10.3e % 10.3e\n", R[0][0], R[0][1], R[0][2]);
gprintf("rot 1: % 10.3e % 10.3e % 10.3e\n", R[1][0], R[1][1], R[1][2]);
gprintf("rot 2: % 10.3e % 10.3e % 10.3e\n", R[2][0], R[2][1], R[2][2]);
}

    C.x = R[0][0]*V.x + R[0][1]*V.y + R[0][2]*V.z;
    C.y = R[1][0]*V.x + R[1][1]*V.y + R[1][2]*V.z;
    C.z = R[2][0]*V.x + R[2][1]*V.y + R[2][2]*V.z;

    return C;

} /* 1}}} */
double Det3(XYZ A[3]) {  /* {{{1 */
    /*
     * Returns the determinant of the three XYZ coordinates stored in A.
     */
    return + A[0].x * A[1].y * A[2].z  
           + A[0].y * A[1].z * A[2].x  
           + A[0].z * A[1].x * A[2].y 

           - A[0].z * A[1].y * A[2].x 
           - A[0].y * A[1].x * A[2].z 
           - A[0].x * A[1].z * A[2].y; 
} /* 1}}} */
void   rotation_3x3(XYZ U, double r[3][3]) {  /* {{{1 */
    /*
     * Given a 3D unit vector {U}, returns a 3x3 rotation matrix [R] such
     * that [R]*{x,y,z} will rotate point {x,y,z} from being in a plane
     * perpendicular to {U}, to a plane perpendicular to {0,0,1} (in other
     * words, to the X-Y plane).
     * stadyn/assm325.f::rotvec
     */
int DEBUG = 0, i, j;
    double  D;

    if        (fabs(U.z) > (1.0 - SQRT_EPSILON)) {

        r[0][0] =  U.z;        r[0][1] =  0.0;       r[0][2] =  0.0;
        r[1][0] =  0.0;        r[1][1] =  1.0;       r[1][2] =  0.0;
        r[2][0] =  0.0;        r[2][1] =  0.0;       r[2][2] =  U.z;

    } else if (fabs(U.x) > (1.0 - SQRT_EPSILON)) {
       
        r[0][0] =  0.0;        r[0][1] =  0.0;       r[0][2] = -1.0;
        r[1][0] =  0.0;        r[1][1] =  U.x;       r[1][2] =  0.0;
        r[2][0] =  U.x;        r[2][1] =  0.0;       r[2][2] =  0.0;

    } else {

        D   = sqrt(1.0 - U.z*U.z);

        r[0][0] =  U.x*U.z/D;  r[0][1] =  U.y*U.z/D;  r[0][2] = -D;
        r[1][0] = -U.y/D;      r[1][1] =  U.x/D;      r[1][2] =  0.0;
        r[2][0] =  U.x;        r[2][1] =  U.y;        r[2][2] =  U.z;

    }
if (DEBUG) {
gprintf("rotation_3x3 input vector: % 12.6e % 12.6e % 12.6e\n", U.x, U.y, U.z);
gprintf("rotation_3x3 rotation matrix:\n");
for (i = 0; i < 3; i++) {
for (j = 0; j < 3; j++) { gprintf(" % 12.6e", r[i][j]); }
gprintf("\n");
}
}

} /* 1}}} */
void   rotate_tri3_elem(XYZ Normal, double K[18*18]) {  /* {{{1 */
    /*
     * Rotates the 18x18 element matrix to be normal to the given vector.
     * stadyn/assm325.f::rotate
     */
#define  I18(row,col) ((row)+(col)*18)
int DEBUG = 0;
    double  ekb[18][18], ektemp[18][18], R[3][3], Rt[3][3];
    int     i, j, ii, jj, kk, j1, j2;

    for (i = 0; i < 18; i++) {
        for (j = 0; j < 18; j++) {
            ekb[i][j]    = K[I18(i,j)];
            ektemp[i][j] = 0.0;
        }
    }

    rotation_3x3(Normal, R);

    for (i = 0; i < 3; i++) 
        for (j = 0; j < 3; j++) 
            Rt[i][j] = R[j][i];

    /*
     * Compute [Rtrans][k][R]
     * [k] is sectioned off into six 3x3's
     */

    if        (fabs(1.0 - Normal.z) < SQRT_EPSILON) {
        for (i = 0; i < 6; i++) {
            for (j = 0; j < 6; j++) {
                ii = i*3;
                jj = j*3;
                K[I18(ii  ,jj  )] =  ekb[ii  ][jj  ];
                K[I18(ii  ,jj+1)] =  ekb[ii  ][jj+1]*R[0][0];
                K[I18(ii  ,jj+2)] =  ekb[ii  ][jj+2];
                K[I18(ii+1,jj  )] =  ekb[ii+1][jj  ]*R[0][0];
                K[I18(ii+1,jj+1)] =  ekb[ii+1][jj+1];
                K[I18(ii+1,jj+2)] =  ekb[ii+1][jj+2]*R[0][0];
                K[I18(ii+2,jj  )] =  ekb[ii+2][jj  ];
                K[I18(ii+2,jj+1)] =  ekb[ii+2][jj+1]*R[0][0];
                K[I18(ii+2,jj+2)] =  ekb[ii+2][jj+2];
            }
        }

    } else if (fabs(1.0 - Normal.x) < SQRT_EPSILON) {

        for (i = 0; i < 6; i++) {
            for (j = 0; j < 6; j++) {
                ii=i*3;
                jj=j*3;
                K[I18(ii  ,jj  )] =  ekb[ii+2][jj+2];
                K[I18(ii  ,jj+1)] =  ekb[ii+2][jj+1];
                K[I18(ii  ,jj+2)] = -ekb[ii+2][jj  ]*R[1][1];
                K[I18(ii+1,jj  )] =  ekb[ii+1][jj+2];
                K[I18(ii+1,jj+1)] =  ekb[ii+1][jj+1];
                K[I18(ii+1,jj+2)] = -ekb[ii+1][jj  ]*R[1][1];
                K[I18(ii+2,jj  )] = -ekb[ii  ][jj+2]*R[1][1];
                K[I18(ii+2,jj+1)] = -ekb[ii  ][jj+1]*R[1][1];
                K[I18(ii+2,jj+2)] =  ekb[ii  ][jj  ];
            }
        }

    } else {

        for (i = 0; i < 6; i++) {
            for (j = 0; j < 6; j++) {
                j1=i*3;
                j2=j*3;

                /* [k][R] */
                for (kk = 0; kk < 3; kk++) {
                    for (ii = 0; ii < 3; ii++) {
                        ektemp[j1+kk][j2+ii] = 0.0;
                        for (jj = 0; jj < 3; jj++) {
                            ektemp[j1+kk][j2+ii] += ekb[j1+kk][j2+jj]*R[jj][ii];
                        }
                    }
                }

                /* [Rt][K] */
                for ( kk = 0; kk < 3; kk++) {
                    for ( ii = 0; ii < 3; ii++) {
                        K[I18(j1+kk,j2+ii)] = 0.0;
                        for ( jj = 0; jj < 3; jj++) {
                            K[I18(j1+kk,j2+ii)] += Rt[kk][jj]*ektemp[j1+jj][j2+ii];
                        }
                    }
                }
            }
        }

    }

} /* 1}}} */
void   test_rotate_tri3_elem() {  /* {{{1 */
    /*
     * Call rotate_tri3_elem with a few test inputs.
     */
#define  I18(row,col) ((row)+(col)*18)
int DEBUG = 0;
    XYZ     unit_normal;
    double  K[18*18];
    int     r, c;

    for (c = 0; c < 18; c++) 
        for (r = 0; r < 18; r++) {
            K[r+c*18] = 100000.0 + 1000.0*r + c;
            gprintf("test_rotate in  K[r=%2d, c=%2d]= %12.1f\n", 
                     r, c, K[r+c*18]);
        }

    unit_normal.x = 1.0;
    unit_normal.y = 0.0;
    unit_normal.z = 0.0;

    unit_normal.x = -3.574065e-01;
    unit_normal.y = -8.628564e-01;
    unit_normal.z =  3.574065e-01;

    unit_normal.x = 0.0;
    unit_normal.y = 0.0;
    unit_normal.z = 1.0;

    unit_normal.x = 0.0;
    unit_normal.y = 1.0;
    unit_normal.z = 0.0;

    rotate_tri3_elem(unit_normal, K);

    for (c = 0; c < 18; c++) {
        for (r = 0; r < 18; r++) {
            gprintf("test_rotate out K[r=%2d, c=%2d]= %12.1f\n", 
                     r, c, K[r+c*18]);
        }
    }

} /* 1}}} */
