/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2013  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

/* exe.c  April 1999

Copyright (c) 1999  D. R. Williamson
*/

#include <ctype.h>
#include <malloc.h>
#include <stdio.h>
#include <string.h>

#include "main.h"
#include "stk.h"

#include "ctrl.h"
#include "exe.h"
#include "inpo.h"
#ifdef KEYBOARD
   #include "key.h"
#endif
#include "lib.h"
#include "mem.h"
#include "mat.h"
#include "math1.h"
#include "prs.h"
#include "sys.h"
#ifdef NET
   #include "term.h"
#endif
#include "tex.h"
#include "tag.h"

catitem (**inlptr)(); /* vector of inline addr pointers */

enum inlflags {xIF=2,xELSE,xDO,xBEGIN,xWHILE,xLIST}; /* inline flags */

#define INLMAX 4096 /* max number of addr per inline function */

long IP; /* running ptr within inlptr vector of inline addr pointers */
long IPend; /* ptr to 1st ending null in vector of inline pointers */

/* Variables for inline run levels: */
catitem *Oncat[NBUF];       /* array of oncat */
catitem (**Inlptr[NBUF])(); /* array of inlptr */
long Ip[NBUF];              /* array of IP */
long Ipend[NBUF];           /* array of IPend */

/* Return stack for DO ... LOOP and list: ... end: */
#define DEPRSTK 32 
struct inlret { 
/* Used by DO: */
   long index; /* no floats: using integers for speed */
   long indexJ; 
   long limit;
   long delta;
   long slope;
   long jmp;
/* Used by list: */
   long list; 
   long depth; 
} retstk[NDBUF*DEPRSTK], *tosret, *retbase[NBUF], *Tosret[NBUF]; 

int _BEGIN(unsigned long *catptr) /* ( --- ptr xBEGIN) */
/* Pushes jump IP for _UNTIL to stack. */
{
   return(pushptr1((unsigned long (**)())(IP)) &&
      pushint(xBEGIN)
   );
}
int _BOOK(unsigned long *catptr)
/* If not already in the catalog, places into the catalog a quote-string
   having the name just read ahead of word book, as if book() had exe-
   cuted to store a string.  For example, if have
      99 "A" book
   the quote string A will go into the catalog as a string if there is
   not already a catalog item called A, even though later we want A to
   not be a string at all, but rather the number 99.

   This is done to satisfy future references only while inline code is
   created.  When the code is later executed, actual stack items will
   be booked to this catalog name, perhaps over and over and of various
   types.  For example in the above, catalog item A will hold the num-
   ber 99 when the phrase executes at runtime.  (But before the phrase
   runs, A in this example will appear as a string when the catalog is
   viewed (as with word catbins), which is somewhat confusing.) */
{
   char nambuf[DEFNC+1];
   int len;

   oncat=(catitem *)*(catptr+IP-1);

   if(oncat->typ!=STRI) {
   /* If the previous text item was not a string, put one of these
      phrases into the list of executable addresses:
         "book" main (if PUBLIC)
         "book" local (if not PUBLIC)
   */
      if(PUBLIC) mainstr("book",(unsigned long *)catptr);
      else localstr("book",(unsigned long *)catptr);

      return 1;
   }
   memcpy(nambuf,oncat->stk->tex,(len=MIN(DEFNC,oncat->stk->col)));
   *(nambuf+len)='\0';

/* If not already in local library, a string will be placed with the
   same name as quote-string in stk.tex of previous inline item at
   IP-1.  Later references to this name during inline creation will
   now be satisfied: */

   if(!caton(tagged(nambuf,*(cattag+onbuf)))) {
      if(!(
         pushq2(nambuf,len) && /* string to stack */
         dup1s() && /* its library name will be the same */
         book(TRACE) /* putting string into the catalog */
      )) return 0;
   }
   if(caton(tagnative("book"))) { /* installing code for book() */
      *(catptr+IP)=(unsigned long)oncat;

      if(TRACE) {
         gprintf(" IP %lX: word %s at addr %lX, exe %lX",\
            IP,oncat->nam,oncat,oncat->exe); nc();
      }
      IP++;
      return 1;
   }
   stkerr(" _BOOK: book ",RUNNOT);
   return 0;
}

int _BRAK()
/* Interrupts inline assembly to run a phrase in brackets. */
{
   return(brak());
}

int catexe(char *word)
/* Fetches catalog word and runs its exe function.
   Error if word not found and STRICT.
   Ok if word not found and not STRICT: pushes unknown string stack. */
{
/* Searching with function for this run level, onbuf: */
   if((*(unsigned long (*)()) *(catsearch+onbuf))(word)) {

   /* Running exe function of catalog item oncat: */
      return((*(unsigned long (*)()) oncat->exe)());
   }
   if(STRICT) {
      if(INFIX) stkerr(" ",CATNOTX);
      else stkerr(" ",CATNOT);
      return 0;
   }
   return(pushq2(word,strlen(word)));
}

int _DEF() 
/* Interrupts inline assembly to create a definition.  This is kind of
   silly--why not do it earlier?--but shows it can be done.

   Don't get carried away with this.  With new checks being done it may
   not continue to work; and weird stuff can happen if the definition
   does an inline too: inline1() is not designed to be recursive. */
{
   return(define());
}

int _DO(unsigned long *catptr)
/* Storing address of runtime do and allowing space for word EXIT
   jmp stored by _LOOP . */
{
   long Jmpto;

   if(caton(tagnative("DO,,"))) { /* native "DO,," in word.p */
      *(catptr+IP)=(unsigned long)oncat;
      if(TRACE) {
         gprintf(" IP %lX: runtime %s at addr %lX, exe %lX",\
            IP,oncat->nam,oncat,oncat->exe); nc();
      }
      if(tosret==(struct inlret *)(*(retbase+onrun)+DEPRSTK-1)) {
         stkerr(" DO: ","return stack is full");
         return 0;
      }
      tosret++;
      IP++; 
      Jmpto=IP;
      IP++; /* allowing space for jmp IP set by _LOOP, used by EXIT */
      return(pushptr1((unsigned long (**)())(Jmpto)) && 
         pushint(xDO)
      );
   }
   stkerr(" DO: DO,, ",RUNNOT);
   return 0;
}

int _do() /* DO (n2 n1 --- ) */
/* Runtime code for DO, pointed to by native "DO,," in word.p. 

   Here (and in _loop()) is where assembler could speed things up, but
   so what?  In a very rough test (linux on a 486 DX2-50), an inlined
   empty loop (the phrase 8E6 1 DO LOOP) takes 18 seconds while the
   same loop in pure C (the phrase for(i=0;i<8000000;i++) { }) takes 4
   seconds.  In either case, these are probably trivial compared to the
   elapsed time when a meaningful task is placed inside the loop--a task
   that is being done 8 million times. 

   Here is running this test on the 486 mentioned above, and taking 
   17 seconds:

      [tops@gutter] ready > "8e6 1 DO LOOP" "looptest" inlinex
       word looptest into catalog

      [tops@gutter] ready > time push looptest time pull - .i
       17
      [tops@gutter] ready > 

 
   [Here is the C code to run a loop test like the one mentioned; it 
   is not quite as easy to run as the one above.

   (time() gives time only to the nearest second):
      #include <time.h>
      time_t t1,t2;
      int i;
      time(&t1);
      printf(" t1: %d\n\r",t1);
      for(i=0;i<8000000;i++) { }
      time(&t2);
      printf(" t2: %d\n\r",t2);
      printf(" t2-t1: %d\n\r",t2-t1);
   ]
*/
{
   long n1,n2;

   if(tos->typ==NUM && (tos-1)->typ==NUM) {

      n1=(long)tos->real;
      drop();
      n2=(long)tos->real;
      drop();

      tosret++;
      tosret->jmp=1+IP;
      tosret->index=n1-XBASE; /* 1st 0-based index */
      tosret->indexJ=(tosret-1)->index; 

      if(n1<=n2) { /* forward loop */
         tosret->limit=n2-1; /* last 0-based index */
         tosret->delta=1; /* default forward */
         tosret->slope=1;
      }
      else { /* backward loop */
         tosret->limit=n2-XBASE; /* last 0-based index */
         tosret->delta=-1; /* default backward */
         tosret->slope=-1;
      }
      IP++; /* bumping past EXIT jmp addr stored next */
      return 1;
   }
   stkerr(" do: ",NEEDLIM);
/* If here, we're lost. */
   return(_endp()); /* the only thing to do is jump to end of inline */
}

/* Function _DOq() is nearly the same as _DO(), and _doq() is about the
   same as _do().  How important would it be to eliminate them and make
   _DO() and _do() dual purpose?

   Not very: the executable file is 645 bytes longer with the addition
   of _DOq() and _doq(), but there is no question that they will run
   faster than any dual purpose function.

   And maintenance?  Forget it.  This stuff is the foundation (bricks
   and mortar--or better, concrete and rebar) where no one needs to look
   again except to learn how something works.  And like a building, if
   the foundation ever needs major work it might be better to tear it 
   down and start over: its just software, mere pieces of writing that
   can become better another time around. */

int _DOq(unsigned long *catptr)
/* Storing address of runtime doq and allowing space for word EXIT jmp
   stored by _LOOP . */
{
   long Jmpto;

   if(caton(tagnative("?DO,,"))) { /* native "doq," in word.p */
      *(catptr+IP)=(unsigned long)oncat;
      if(TRACE) {
         gprintf(" IP %lX: runtime %s at addr %lX, exe %lX",\
            IP,oncat->nam,oncat,oncat->exe); nc();
      }
      if(tosret==(struct inlret *)(*(retbase+onrun)+DEPRSTK-1)) {
         stkerr(" ?DO: ","return stack is full");
         return 0;
      }
      tosret++;
      IP++;
      Jmpto=IP;
      IP++; /* allowing space for jmp IP set by _LOOP, used by EXIT */
      return(pushptr1((unsigned long (**)())(Jmpto)) &&
         pushint(xDO)
      );
   }
   stkerr(" ?DO: ?DO,, ",RUNNOT);
   return 0;
}

int _doq() /* ?DO (n2 n1 --- ) */
/* Runtime code for ?DO, pointed to by native "?DO,," in word.p. */
{
   long n1,n2;

   if(tos->typ==NUM && (tos-1)->typ==NUM) {

      n1=(long)tos->real;
      drop();
      n2=(long)tos->real;
      drop();

      tosret++;
      tosret->jmp=1+IP;

      if(n1==n2) return(_exit1()); /* skipping loop if n1=n2 */

      tosret->index=n1-XBASE; /* 1st 0-based index */
      tosret->indexJ=(tosret-1)->index; 

      if(n1<=n2) { /* forward loop */
         tosret->limit=n2-1; /* last 0-based index */
         tosret->delta=1; /* default forward */
         tosret->slope=1;
      }
      else { /* backward loop */
         tosret->limit=n2-XBASE; /* last 0-based index */
         tosret->delta=-1; /* default backward */
         tosret->slope=-1;
      }
      IP++; /* bumping past EXIT jmp addr stored next */
      return 1;
   }
   stkerr(" ?do: ",NEEDLIM);
/* If here, we're lost. */
   return(_endp()); /* the only thing to do is jump to end of inline */
}

int _ELSE(unsigned long *catptr) /* (ptr xIF --- ptr1 xELSE) */
/* Sets the jump for the preceding IF, and prepares the jump for the
   upcoming THEN. 

   Sets jump IP that is 1 more than the current one.  When run() adds
   its 1, the total jump will be two past the current IP, to step over
   this IP and IP+1 where the jump step to THEN will be stored. */
{
   char *tok;
   long len;
   const int delta=1;
   long IP0,Jmpto;
   catitem *p;

   if(stkdepth()>1 && pop()->real==xIF) {
      IP0=(long)popptr(&len,&tok);
      *(catptr+IP0)=delta+IP; /* storing jump to here from IP0 */
      if(TRACE) {
         p=(catitem *)*(catptr+IP0-1);
         gprintf(" IP %lX: runtime %s jump to IP %lX",\
            IP0,p->nam,1+delta+IP); nc();
      }
      if(caton(tagnative("Jmp,"))) {
         *(catptr+IP)=(unsigned long)oncat;
         if(TRACE) {
            gprintf(" IP %lX: runtime %s at addr %lX, exe %lX",\
               IP,oncat->nam,oncat,oncat->exe); nc();
         }
         IP++; 
         Jmpto=IP;
         IP++; /* allowing space for jmp IP */
         return(pushptr1((unsigned long (**)())(Jmpto)) && 
            pushint(xELSE)
         );
      }
      stkerr(" ELSE: Jmp, ",RUNNOT);
      return 0;
   }
   stkerr(" ELSE: ",BADBRANCH);
   return 0;
}

int _END1P(unsigned long *catptr)  /* (flag --- ) */
/* Installing runtime end1p for list. */
{
   if(tosret->list==0) return 0; /* not an end for list: */
      
   if(stkdepth()>0 && (int)pop()->real==xLIST) {
      if(!caton(tagnative("end1p,,"))) {
         stkerr(" list: end1p,, ",RUNNOT);
         return 0;
      }
      *(catptr+IP)=(unsigned long)oncat;
      if(TRACE) {
         gprintf(" IP %lX: runtime %s at addr %lX, exe %lX",\
            IP,oncat->nam,oncat,oncat->exe); nc();
      }
      tosret->list=0;
      tosret--;
      IP++;
      return 1;
   }
   stkerr(" list: ",BADBRANCH);
   return 0;
}

int _end1p() /* end ( --- ) */
/* At runtime, putting items on stack into list.  Return stack, tos-
   ret.depth, contains depth when word list: started (pushed previously
   by runtime function _list()). */
{
   unsigned int d;

   d=stkdepth()-tosret->depth;
   tosret--;
   return(pushint(d) && listfin());
}

int _ENDP(unsigned long *catptr)
/* Installing runtime endp. */
{
   if(caton(tagnative("endp,,"))) { /* native "endp,," in word.p */
      *(catptr+IP)=(unsigned long)oncat;
      if(TRACE) {
         gprintf(" IP %lX: runtime %s at addr %lX, exe %lX",\
            IP,oncat->nam,oncat,oncat->exe); nc();
      }
      IP++;
      return 1;
   }
   stkerr(" return: endp,, ",RUNNOT);
   return 0;
}

int _endp() /* return ( --- ) */
/* Bump IP to end of executing inline, after cleaning up. */
{
   tosret=*(retbase+onrun); /* clearing return stack */
   IP=IPend;
   return 1;
}

int exe() /* exe (... ptr --- ...) */
/* Running exe function of catalog item pointed to by pointer NUM ptr.
   Result is unpredictable if ptr does not represent a valid catalog 
   pointer.  

   Locators to valid pointers stored in global pattern table, ptrnum,
   were stored in the least significant 4-bytes of the 8-byte number on 
   the stack that was created by word ptr (see function ptr()). */
{ 
   catitem *c;
   union {
      double x;
      int i[sizeof(double)];
   } X={0};
   int k=-1;

   if(tos->typ!=NUM) {
      stkerr(" exe: ",NUMNOT);
       return 0;
   }
   popd(&X.x);

/* Offsets kplus and oplus to the two 4-byte integers stored in X de-
   pend upon endian.  They were determined at start up in runinit(), 
   and used by ptr() to store the location of a word's catalog pointer,
   oncat, in the global pointer pattern table called ptrnum.

   Offset oplus was used to store the 32-bit table row number, and off-
   set kplus was used to store a 32-bit authentication pattern into the 
   64-bit ptr NUM:
      oplus is 0 for little endian, 1 for big, offset to 32-bit loc
      kplus is 1 for little endian, 0 for big, offset to kzero */

/* Fetching pattern of catitem pointer oncat; oncat may be a 64-bit
   pointer: */
   c=(catitem *)patget((int)*(X.i+oplus),ptrnum);

   k=*(X.i+kplus); /* k=kzero should be null except for sign bit */

   if(k!=kzero || c==NULL) {
      stkerr(" exe: ",PTRCNOT);
      return 0;
   }
   memcpy(&oncat,c,sizeof(catitem *));

   if(oncat==0) {
      stkerr(" exe: ",PTRCNOT);
      return 0;
   }
   return((*(unsigned long (*)()) oncat->exe)());
}

#ifdef NET
int exe_remote() /* exe_remote (qW1 qW2 --- ) */
/* When running remotely, that is, when SOCKFD is not equal to -1, run 
   word W2 in place of W1.  SOCKFD is the socket file descriptor that
   is returned by word remotefd. */
{
   catitem *oncat1=NULL,*oncat2=NULL;
   unsigned long (*EXE)()=NULL; 

   if(tos->typ!=STR || (tos-1)->typ!=STR) {
      stkerr(" exe_remote: ",STRNOT2);
      return 0;
   }
   if(!caton(tagnative("exe_remote,"))) {
      stkerr(" exe_remote: ",RUNNOT);
      return 0;
   }
   EXE=oncat->exe; /* EXE of the runtime code */

   strchop();
   swap();
   strchop();
   swap();

   if(caton(tagnative((tos-1)->tex))) oncat1=oncat; /* W1 oncat */
   if(caton(tagnative(tos->tex)))     oncat2=oncat; /* W2 oncat */

   if(!oncat1) {
      gprintf(" exe_remote: %s not found in catalog",(tos-1)->tex);
      nc();
   }
   if(!oncat2) {
      gprintf(" exe_remote: %s not found in catalog",tos->tex);
      nc();
   }
   if(!oncat1 || !oncat2) {
      stkerr("","");
      return 0;
   }
/* Does W1 exe already equal EXE? */
   if(oncat1->exe==EXE && oncat1->exe1!=NULL) {
   /* Put things back to original and reenter: */
      oncat1->exe=oncat1->exe1;
      oncat1->exe1=NULL;
      oncat1->nam2=NULL;

      return(exe_remote());
   }
   oncat1->exe1=oncat1->exe;                /* W1 exe to W1 exe1 */
   oncat1->nam2=memgetn(tos->tex,tos->col); /* W2 nam to W1 nam2 */
   oncat1->exe=EXE;                         /* EXE to W1 exe */
/*
   if(WTRACE) {
      gprintf(" exe_remote: %s will run for %s when running remotely",
         untagged(oncat1->nam2),untagged(oncat1->nam));
      nc();
   }
*/
   return(drop2());
}
#endif

#ifdef NET
int EXE_REMOTE() /* (... --- ...) */
/* Runtime code for exe_remote. */
{
/*
   if(KEYS || SOCKFD==-1) (*(unsigned long (*)()) oncat->exe1)();
*/
   if(SOCKFD==-1) (*(unsigned long (*)()) oncat->exe1)();
   else catexe(oncat->nam2);
   return 1;
}
#endif

int exe1(double ptr)
/* Running exe function of catalog item pointed to by ptr.  No checking
   is done, and result is unpredictable if ptr is not a valid catalog 
   pointer. 

   Example (hash.c): 

      double ptrVALS;
      popd(&ptrVALS); // from stack, ptr to catalog function //

      for(;k<rows;k++) {
         ...
         exe1(ptrVALS); // fires the catalog function //
      } */
{
   union {
      double x;
      int i[sizeof(double)];
   } X={0};

   X.x=ptr;

/* Fetching pattern of catitem pointer oncat; oncat may be a 64-bit
   pointer: */
   memcpy(&oncat,(catitem *)patget((int)*(X.i+oplus),ptrnum),
      sizeof(catitem *));

   return((*(unsigned long (*)()) oncat->exe)());
}

int exec(unsigned long (**function)(), int n)
/* Runs in sequence the n function pointers in array *function. 

   Each function pointed to has the same header, given by "int name(),"
   and should return 1 if successful, 0 if error.  
   Returns the sum of n returns divided by n: 1 if all successful, 0 if
   not.  Here's a loop to print the function pointers:
      for(;i<n;i++) {
      gprintf(" i,function+i: %d, %lX",i,*(function+i)); nc(); }
*/
{  register int i=0,ret=0;
   for(;i<n;i++) ret+=(*(function+i))(); return ret/n;
}

void exec0(unsigned long (**function)(), int n)
/* Runs in sequence the n function pointers in array *function. 

   Each function pointed to has the same header, given by "int name()."
   The ints returned from the functions are ignored.
   Rough timings show this function runs about 15% faster than exec. */
{
   register int i=0;
   for(;i<n;i++) (*(function+i))();
}

int exec1(unsigned long (**function)())
/* Runs a function pointed to by *function and having the header "int
   function()."  

   Returns the int returned by function. 
   Here's a line to print the function pointer:
      gprintf(" function: %lX",*(function)); nc();
*/
{  
   return(*function)(); 
}

int execa(unsigned long (*address)())
/* Runs a function at address that has the header "int fname()." 

   Returns the int returned by function.  Here's a line to print 
   the function address:
      gprintf(" address: %lX",(address)); nc();
*/
{
   return(address)();
}

int execute() /* execute (hT or qS --- f) */
/* Runs the words in a volume or quote-string, using the main library
   as the local library.  Pushes success flag to stack. */
{
   return(xmain(1));
}

int exedefn()
/* Interprets and runs the words in the current catalog item, oncat, 
   where the words reside in oncat's volume, oncat.stk.tex. */
{
   char *p;
   int err[NBUF],len,ret;

   if((p=(char *)memgetn(oncat->stk->tex, \
      (len=oncat->stk->row*oncat->stk->col)))==NULL) {
      return 0;
   }
   if(TRACE) {
      gprintf(" exec definition %s, lib tag %s", \
         oncat->nam,oncat->stk->tok); 
      nc();
   }
   if(bufjump(p,len)) { /* going up one run level, to onbuf */
   /* Setting catalog context for this run level: */
      context(oncat->stk->tok);

      *(err+onbuf)=stkerrcount;

      ret=setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */

      if(!*(jmpready+onbuf)) {

         *(jmpready+onbuf)=ENDDEF;
         perform();
      }
      else {
         if(ret==ABORT) {
            /* jumping out cleanly: */
            bufreturn();
            longjmp(abortbuf,ABORT);
         }
      }
      ret=(*(err+onbuf)==stkerrcount);
      bufreturn();
      return ret;
   }
   return 0;
}

int exefree()
/* Report running a word no longer in the catalog. */
{
   stkerr(" exefree: ",WORDFREE);
   return 0;
}

int exeinline()
/* Runtime function: pushes to stack the PTR stkitem of the current
   INLI catitem, oncat, and immediately runs it. */
{
   stkitem *s;
   int ret=0;

   s=oncat->stk;

   CNT=s->cnt;
   CNTC=s->cntc;
   CTAG=s->tag;
   CIMAG=s->imag;
   NAM=oncat->nam;
   
   ret=push(s->typ,s->tok,NOTAG,s->real,s->mat,s->tex,s->row,s->col,
      s->ptr);

   CNT=NULL;
   CNTC=NULL;
   CTAG=0;
   CIMAG=0;
   NAM=NULL;
   
   if(TRACE) return(ret && runt());

   else return(ret && run());
}

int exenumber() 
/* Runtime function: pushes to stack the number in the stkitem of the
   current catitem, oncat. */
{
   stkitem *s;
   int ret=0;

   s=oncat->stk;

   CNT=s->cnt;
   CNTC=s->cntc;
   CTAG=s->tag;
   CIMAG=s->imag;
   NAM=oncat->nam;
   
   ret=push(NUM,s->tok,NOTAG,s->real,NULL,NULL,0,0,NULL);

   CNT=NULL;
   CNTC=NULL;
   CTAG=0;
   CIMAG=0;
   NAM=NULL;

   return(ret);
}

int exeq() /* exe? (ptr --- f) */
/* Functionally equivalent to exe(), but leaves a success flag on the
   stack and internally always returns 1 with no error message.

   Running exe function of catalog item pointed to by pointer NUM ptr.
   Result is unpredictable if ptr does not represent a valid catalog
   pointer.

   Locators to valid pointers stored in global pattern table, ptrnum,
   were stored in the least significant 4-bytes of the 8-byte number on
   the stack that was created by word ptr (see function ptr()). */
{
   int ret;
   catitem *c;
   union {
      double x;
      int i[sizeof(double)];
   } X={0};
   int k=-1;

   if(tos->typ!=NUM) {
      if(stkdepth()) drop();
      pushint(xFALSE);
      return 1;
   }
   popd(&X.x);

/* kplus and oplus are offsets to the two 32-bit numbers in X, and
   depend upon endian.  They were determined at start up in runinit(). 

   Function ptr() used oplus to store a table locator to catalog pointer
   oncat, and kplus to store a pattern, kzero, to verify a ptr NUM. */

/* Fetching pattern of catitem pointer oncat; oncat may be a 64-bit
   pointer: */
   c=(catitem *)patget((int)*(X.i+oplus),ptrnum);

   k=*(X.i+kplus); /* k=kzero should be null except for sign bit */

   if(k!=kzero || c==NULL) {
      pushint(xFALSE);
      return 1;
   }
   memcpy(&oncat,c,sizeof(catitem *));

   if(oncat==0) {
      pushint(xFALSE);
      return 1;
   }
   ret=((*(unsigned long (*)()) oncat->exe)());
   if(ret) {
      pushint(xTRUE);
      return 1;
   }
   else {
      pushint(xFALSE);
      return 1;
   }
}

int exestkitem() 
/* Runtime function: pushes to stack the stkitem of the current
   catitem, oncat. */
{
   stkitem *s;
   int ret=0;

   s=oncat->stk;

   CNT=s->cnt;
   CNTC=s->cntc;
   CTAG=s->tag;
   CIMAG=s->imag;
   NAM=oncat->nam;

   ret=push(s->typ,s->tok,s->tag,s->real,s->mat,s->tex,s->row,s->col,
      s->ptr);

   CNT=NULL;
   CNTC=NULL;
   CTAG=0;
   CIMAG=0;
   NAM=NULL;

   return(ret);
}

int exestring()
/* Runtime function: pushes to stack the string in the stkitem of the
   current catitem, oncat. */
{
   stkitem *s;
   int ret=0;

   s=oncat->stk;

   CNT=s->cnt;
   CNTC=s->cntc;
   CTAG=s->tag;
   CIMAG=s->imag;
   NAM=oncat->nam;

   ret=push(STR,s->tok,s->tag,0,NULL,s->tex,1,s->col,NULL);

   CNT=NULL;
   CNTC=NULL;
   CTAG=0;
   CIMAG=0;
   NAM=NULL;

   return(ret);
}

int _EXIT(unsigned long *catptr)
/* Places runtime function for EXIT into list of catalog addresses. */
{
   if(caton(tagnative("exit1,"))) {
      if(tosret>*(retbase+onrun)) {
         *(catptr+IP)=(unsigned long)oncat;
         if(TRACE) {
            gprintf(" IP %lX: runtime %s at addr %lX, exe %lX",\
               IP,oncat->nam,oncat,oncat->exe); nc();
         }
         IP++;
         return 1;
      }
      stkerr(" EXIT: ",BADLOOP);
      return 0;
   }
   stkerr(" EXIT: exit1, ",RUNNOT);
   return 0;
}

int _exit1() /* EXIT ( --- ) */
/* Runtime function for EXIT. */
{
   IP=(long)*(inlptr+(tosret->jmp)); /* jumping IP */
   tosret--; /* popping return stack */
   return 1;
}

int _EXTERN(unsigned long *catptr) /* extern ( --- ) */
/* Puts the next source word into the catalog as a quote-string so it
   will be pushed to the stack at runtime, then follows it with the 
   address for main1() that will execute the string.  

   Allows the code for word to be placed into the catalog after this
   inline code has been created and any time prior to running it.  

   Runtime behavior will mimic the phrase: "word" main (and in fact,
   the phrase "word" main is equivalent and extern is really unneces-
   sary), where word is here hidden in a quote-string so it does not 
   need to exist in the catalog until just before the phrase is seen
   (perhaps placed there by some define: that has yet to be seen). */
{
   if(!_FEED(catptr)) return 0;
   
   if(caton(tagnative("main"))) { /* native "main" in word.p */
      *(catptr+IP)=(unsigned long)oncat;
      if(TRACE) {
         gprintf(" IP %lX: runtime %s at addr %lX, exe %lX",\
            IP,oncat->nam,oncat,oncat->exe); nc();
      }
      IP++;
      return 1;
   }
   stkerr(" _EXTERN: main ",RUNNOT);
   return 0;
}

int extern1()
/* Reporting that word extern is only for phrases being inlined.  The
   phrase '"xxx" main' is completely equivalent to 'extern xxx' and 
   can be used anywhere. */
{
   stkerr(" extern: ",INLONLY);
   return 0;
}

int _FEED(unsigned long *catptr)
/* Performs the job of word feed during creation of inline pointers, 
   putting the resulting unnamed quote-string into the catalog and 
   adding the catalog address to the list of inline pointers.

   Later, at runtime, the quote-string will be pushed to the stack as
   word feed would have done. 

   Only feed is linked to this function (see word.p).  It will not be
   used for word feed.

   Word feed is for phrases that work with keyed input, and will feed
   text later from the key input stream, not immediately from the text
   of words being inlined. */
{
   char *token[NBUF];
   int len;

/* Feed next token: */
   if((*(token+onbuf)=tokenget())!=NULL) {
      if(!tokenq) {
         if((tokenq=(char *)malloc(
            3+(len=strlen(*(token+onbuf)))))==NULL) {
            stkerr(" _FEED: ",MEMNOT);
            return 0;
         }
         memcpy((tokenq+1),*(token+onbuf),len);
         *(tokenq)='\"';
         *(tokenq+len+1)='\"';
         *(tokenq+len+2)='\0';
      }
      return(_UQUOTE(catptr));
   }
   stkerr(" _FEED: ",FEEDNOT);
   return 0;
}

int fetchif() /* fetchif (qA --- hA) */
/* Fetch A from the catalog.  If A is not tagged, it is tagged with
   the current context (running word or main). */
{
   char *word;

   if(tos->typ!=STR) {
      stkerr(" fetchif: ",STRNOT);
      return 0;
   }
   if(tagon(tos->tex)) word=tos->tex;
   else word=tagged(tos->tex,*(cattag+onbuf));

   if(caton(word)) {
      drop(); /* oncat has been set by caton(), so ok to drop */

   /* Running exe function of catalog item oncat found by caton(): */
      return((*(unsigned long (*)()) oncat->exe)());
   }

/* No match in local context.  Look for a function to run: */
   word=tagged(tos->tex,"CODE__");
   if(caton(word)) {
   /* Running something like loadop4("A.op4",1,0)[1:5,3:6]. */ 

      drop(); /* oncat has been set by caton(), so ok to drop */

   /* Running exe function of catalog item oncat found by caton(): */
      return((*(unsigned long (*)()) oncat->exe)());
   }
   return(
      drop() &&
      pushstr("")
   );
}

int _I(unsigned long *catptr)
/* Places runtime function for loop index word I into list of catalog
   addresses. */
{
   if(caton(tagnative("I,,"))) {
      if(tosret>*(retbase+onrun)) {
         *(catptr+IP)=(unsigned long)oncat;
         if(TRACE) {
            gprintf(" IP %lX: runtime %s at addr %lX, exe %lX",\
               IP,oncat->nam,oncat,oncat->exe); nc();
         }
         IP++;
         return 1;
      }
      stkerr(" I: ",INDEXNOT);
      return 0;
   }
   stkerr(" I: I,, ",RUNNOT);
   return 0;
}

int _i() /* I ( --- I) */
/* Runtime function for I.  Pushes current DO loop index to stack. */
{
   return(
      push(NUM,NULL,NOTAG,(tosret->index)+XBASE,NULL,NULL,0,0,NULL)
   );
}

int _IF(unsigned long *catptr) /* ( --- ptr xIF) */
/* Prepares the jump for an upcoming ELSE or THEN. 

   Inlining below is more forgiving with the flag for IF than is the
   interpretive version of IF (file ctrl.c).  Below, Jz, jump on zero,
   is used to jump to ELSE or THEN (the false branch, when there is a 
   zero on the stack). 

   This means that any nonzero value will work for true, not just the
   rigid -1 (bool) required by interpretive IF.  Is this good?  Or bad?
   Doesn't matter?  Time will tell: this is just a heads-up.

   The pro: It certainly is more efficient just testing for 0.  And
   Forth has always done it this way with no problems.

   The con: Don't have one, except that for consistency both interpre-
   tive and inline IFs should work the same.  But maybe its the inter-
   pretive one that should be changed to work this way. */
{
   long Jmpto;

   if(caton(tagnative("Jz,"))) {
      *(catptr+IP)=(unsigned long)oncat;
      if(TRACE) {
         gprintf(" IP %lX: runtime %s at addr %lX, exe %lX",\
            IP,oncat->nam,oncat,oncat->exe); nc();
      }
      IP++; 
      Jmpto=IP;
      IP++; /* space for jmp IP to be stored by _ELSE or _THEN */
      return(pushptr1((unsigned long (**)())(Jmpto)) && 
         pushint(xIF)
      );
   }
   stkerr(" IF: Jz, ",RUNNOT);
   return 0;
}

int inline1() /* inline (hT qName --- ) */
/* Makes a vector of inline executable addresses for the words in a 
   volume or quote-string, and places it into the catalog. 

   This function is reentered when inlinex is run inside brackets
   as a word is being inlined.  IN_WORD needs to be preserved.

   This function does not use realloc(). */
{
   int addon=0,catnot=0,defn=0,err=1,len,typ;
   const int pad=4;
   unsigned long *c1,(**catptr)(); 
   stkitem *savtos;
   char name[DEFNC+1],*p,tag[DEFNC+1];
   char *tok,*token,*tokenlast=NULL; 
   double x;
   int in_word_sav;

   if(tos->typ!=STR) {
      stkerr(" inline: ",STRNOT);
      return 0;
   }
   if(!runup()) return 0; /* jump to new inline level, onrun */

   len=strlen1(tos->tex,tos->col);
   memcpy(name,tos->tex,len=MIN(DEFNC,len)); *(name+len)='\0';
   drop();

   if((typ=tos->typ)!=VOL && typ!=STR) {
      stkerr(" inline: ",STRORVOLNOT);
      rundn();
      return 0;
   }
   in_word_sav=IN_WORD;

   if(strlen(DEFNAM)) {
      IN_WORD=1;
   /* The example define below is a case of making an inlinex inside
      a definition.  It is sensed here by testing for nonzero string 
      length of DEFNAM.  

      Flag IN_WORD now set to 1 is used in book() (lib.c) and below in 
      several places, including the test on STRICT to make the inlinex 
      in abc really behave as a -inlinex to ignore the unresolved ref-
      erence to A.  Word "inline:" already does that, so now the two 
      behave the same.

         define: abc [ "A 2 + . nl" "B" inlinex ] 10 is A B ;
         inline: def [ "A 2 + . nl" "B" inlinex ] 10 is A B ;
   */
   }
   else {
      IN_WORD=0;
      strcpy(DEFNAM,name); /* needed if word defname is fired */
      *(DEFNAM+len)='\0';
   }
   if(TRACE && IN_WORD) {
      gprintf(" Inlining inside definition %s",DEFNAM); nc(); 
   }
   if(PUBLIC && IN_WORD && strcmp(name,DEFNAM)==0) {
      stkerr(" inline: ",NAMCLASH);
      gprintf(
        " hint: when public is in effect, an inline's name cannot");
      nc();
      gprintf(
        "       match the name of the definition it is within");
      nc();
      rundn();
      return 0;
   }
   if(PUBLIC && caton(tagnative(name))) { 
   /* Checking for conflict with natives: */
      if(oncat->typ==NATI) {
         stkerr(" inline: ",NATICANT);
         rundn();
         return 0;
      }
      if(oncat->typ==CONS) {
         stkerr(" inline: ",CONSCANT);
         rundn();
         return 0;
      }
   }
   if((catptr=(unsigned long (**)()) malloc(
      (INLMAX+pad)*sizeof(unsigned long)))==NULL) {
      stkerr(" inline: ",MEMNOT);
      rundn();
      return 0;
   }
   memset(catptr,0,(INLMAX+pad)*sizeof(unsigned long));

   if((p=(char *)memgetnNL(tos->tex,tos->row,tos->col,&len))==NULL) {
      stkerr(" inline: ",MEMNOT);
      rundn();
      return 0;
   }
   if(bufjump(p,len)) {
      drop(); /* finished with text on stack; text is now at *p */ 
      *tag=0;

   /* This puts inline items into local lib of the definition
      that is being made underneath this level: */
      if(IN_WORD) {
         tagmake(tag,DEFNAM);
         defn=1; /* define() is testing local stack balance */
      }
      else { /* making standalone inline: */
         tagmake(tag,name);
         tdepth1=0; /* testing local stack balance */
         defn=0;
      }
      context(tag);

      savtos=tos;
      err=stkerrcount;

      while(err==stkerrcount && (token=tokenget())!=NULL) {
/*
         if(TRACE) {
            gprintf(" inline next word: %s",token); nc();
         }
*/
         if(number(token,&x)) { /* if error, stkerrcount will change */

         /* Put number into local library of NUMBERS__ and fetch its 
            resulting catitem ptr: */
            variable((tok=tagged(token,"NUMBERS__")),x); 
            addon=caton(tok); /* fetch catitem ptr to add to IP list*/
            goto tokendone;
         }
         if((addon=caton2(token))) {

         /* Getting token+comma inline function from main if any,
            and negating addon: */
            if(caton(tagnative((char *)strcat(token,",")))) {
            /* Running a comma-function that adds runtime addresses to 
               the IP list: */
               (*(unsigned long (*)()) oncat->exe)(catptr);

               addon=0; /* no need to do add on below; running comma-
                           function has done it */
            }
         }
         else {
            if(!STRICT || IN_WORD) {
               mainword(token,(unsigned long *)catptr);
               addon=0;
            }
            else {
               if(INFIX) stkerr(" ",CATNOTX);
               else stkerr(" ",CATNOT);
               catnot=1;
            }
         }
         tokendone: 

         if(IP>=INLMAX) {
            stkerr(" inline: ","items exceed INLMAX");
         }
         if(addon && err==stkerrcount) {
         /* Adding a word address to the IP list: */
            *(catptr+IP)=(unsigned long (*)())oncat;
            if(TRACE) {
               gprintf(" IP %lX: word %s at addr %lX, exe %lX", \
                  IP,oncat->nam,*(catptr+IP),oncat->exe); nc();
            }
            IP++;
         }
         tokenlast=token;
         token=NULL;

      } /* bottom of while loop */  
      
      if(!IN_WORD) *(DEFNAM)='\0';

      if(tosret!=*(retbase+onrun)) {
         stkerr(" inline: ",INLBADLP);
      }
      if(tos!=savtos) {
         stkerr(" inline: ",INLBADSTK);
      }
      if(IP>=INLMAX) {
         stkerr(" inline: ","items exceed INLMAX");
      }
      if(tdepth1 && STKBAL && !defn) {
         stkerr(" inline: ",LOCSTKNOT);
         tdepth1=0;
      }
      if(err!=stkerrcount) {
         while(tos>savtos) { 
            drop(); /* clearing inline debris off stack when error */
         }
         report(tokenlast); /* last token */
         rundn();
         bufreturn();
         stkerr(" inline ",INLINERR);
         if(catnot) {
            gprintf("  unresolved reference to a probable future "); 
            gprintf("word or to a cataloged"); nc(); 
            gprintf("  stack item not visible; possible remedies "); 
            gprintf("are:"); nc();
            gprintf("    1. -inlinex allows unresolved references ");
            gprintf("to words yet to be defined"); nc();
            gprintf("    2. [ ... ] allows view to main library ");
            gprintf("cataloged stack items"); nc();
            gprintf("       during creation of this word"); nc();
            gprintf("    3. public allows references to main library ");
            gprintf("cataloged stack items"); nc();
            gprintf("       by making this word's library the main "); 
            gprintf("library"); nc();
         }
         mallfree((void *)&catptr);
         return 0;
      }
      if(TRACE) {
         gprintf(" IP %lX: runtime 1st ending null",IP); nc();
         gprintf(" IP %lX: runtime 2nd ending null",1+IP); nc();
      }
      *(catptr+IP)=NULL; IP++;
      *(catptr+IP)=NULL; IP++;
      bufreturn();

      if((c1=(unsigned long *)malloc(1+(len=IP* \
         sizeof(unsigned long))))==NULL) {
         stkerr(" inline: ",MEMRESIZE);
         mallfree((void *)&catptr);
         rundn();
         return 0;
      }
      memcpy(c1,(unsigned long *)catptr,len);
      mallfree((void *)&catptr);
      catptr=(unsigned long (**)())c1;

      if(*tag) p=(char *)memgetn(tag,strlen(tag)); /* for PTR stkitem */
      else p=NULL;

      pushptr(catptr,IP,p); /* PTR to stack */
      pushstr(name); 

      if(!book(1)) {
         gprintf(" inline %s not completed due to error",name); 
         nc();
         stkerr("","");

      /* Dropping ptr still on stk from book() error ret: */
         drop();
      /* Warning: mallocked memory for catptr and p will be freed
         by drop(), but mallocked memory for strings or arrays of
         the aborted inline are not freed, and remain useless in 
         the catalog with the tag of the aborted inline.
         Do not come here often. */
      }
      rundn();
      IN_WORD=in_word_sav;
      return 1; /* enough errors; return 1 even if had one */
   }
   rundn();
   mallfree((void *)&catptr);
   return 0;
}

int inlinex() /* inlinex (hT qName --- ) */
/* Makes an inline that runs immediately: does not require word run. */
{
   int IMMsav,ret=0;

   IMMsav=IMMEDIATE;
   IMMEDIATE=1;

   ret=inline1();

   IMMEDIATE=IMMsav;
   return(ret);
}

int inlinex1() /* -inlinex (hT qName --- ) */
/* Makes inlinex() without strict. */
{
   int ret,STRsav;

   STRsav=STRICT;
   strictoff();

   ret=inlinex();

   STRICT=STRsav;
   return(ret);
}

int _INTO(unsigned long *catptr)
/* Inline function for word into. */
{
   if(_FEED(catptr)) return(_BOOK(catptr));
   return 1;
}

int _J(unsigned long *catptr)
/* Places runtime function for J into list of catalog addresses. */
{
   if(caton(tagnative("J,,"))) {
      if(tosret>*(retbase+onrun)) {
         *(catptr+IP)=(unsigned long)oncat;
         if(TRACE) {
            gprintf(" IP %lX: runtime %s at addr %lX, exe %lX",\
               IP,oncat->nam,oncat,oncat->exe); nc();
         }
         IP++;
         return 1;
      }
      stkerr(" J: ",INDEXNOT);
      return 0;
   }
   stkerr(" J: J,, ",RUNNOT);
   return 0;
}

int _j() /* J ( --- J) */
/* Runtime function for J. */
{
   return(push(NUM,NULL,NOTAG,XBASE+(tosret->indexJ),NULL,NULL, \
      0,0,NULL));
}  

int _Jmp()
/* Runtime function: jumps unconditionally to IP stored at IP+1. */
{
   IP=(long)*(inlptr+IP+1); /* jumping IP */
   return 1;
}

int _Jz() /* (f --- ) */
/* Runtime function: jumps to IP stored at IP+1 if value popped from
   stack is zero. */
{
   double xr;

   if(tos->typ==NUM) {
      if((int)tos->real) {
         IP++; /* no jump; bumping IP to next */
      }
      else {
         IP=(long)*(inlptr+IP+1); /* jumping IP */
      }
      drop();
      return 1;
   }
   else {
      if(tos->typ==MAT) {
      /* Shortened version of what ontop() does: */
         xr=*tos->mat; /* taking the first element of the matrix */
         drop();
         push(NUM,NULL,NOTAG,xr,NULL,NULL,0,0,NULL); /* NUM to stk */
         return(_Jz());
      }
   }
   stkerr(" Jz: ",FLAGNOT);
/* If here, we're lost; trying to continue will produce a segmentation 
   fault.  Here's an inline that gets us to here:
      inline: badloop
         "some string" IF " bad loop if" ELSE " bad loop else" THEN
         nl . ;
*/
   return(_endp()); /* the only thing to do is jump to end of inline */
}

int _K(unsigned long *catptr)
/* Places runtime function for K into list of catalog addresses. */
{
   if(caton(tagnative("K,,"))) {
      if(tosret-2>*(retbase+onrun)) {
         *(catptr+IP)=(unsigned long)oncat;
         if(TRACE) {
            gprintf(" IP %lX: runtime %s at addr %lX, exe %lX",\
               IP,oncat->nam,oncat,oncat->exe); nc();
         }
         IP++;
         return 1;
      }
      stkerr(" K: ",INDEXNOT);
      return 0;
   }
   stkerr(" K: K,, ",RUNNOT);
   return 0;
}

int _k() /* K ( --- K) */
/* Runtime function for K. */
{
   return(push(NUM,NULL,NOTAG,XBASE+((tosret-1)->indexJ),NULL,NULL, \
      0,0,NULL));
}

int _LIST(unsigned long *catptr) /* ( --- xLIST) */
/* Sets up runtime code for word list:; uses retstk to keep nesting
   straight. */
{
   if(caton(tagnative("list:,,"))) {
      *(catptr+IP)=(unsigned long)oncat;
      if(TRACE) {
         gprintf(" IP %lX: runtime %s at addr %lX, exe %lX",\
            IP,oncat->nam,oncat,oncat->exe); nc();
      }
      if(tosret==(struct inlret *)(*(retbase+onrun)+DEPRSTK-1)) {
         stkerr(" list: ","return stack is full");
         return 0;
      }
      tosret++;
      IP++;
      tosret->list=1;
      return(pushint(xLIST));
   }
   stkerr(" list: list:,, ",RUNNOT);
   return 0;
}

int _list() 
/* Runtime code for list:.  Runtime function end1p(), to be seen, will
   complete the list. */
{
   tosret++;
   tosret->depth=stkdepth(); /* top of stack before list items */
   tosret->index=(tosret-1)->index; /* needed for word J */
   tosret->indexJ=(tosret-1)->indexJ; /* needed for word K */
   return 1;
}

int local() /* local (hT or qS --- ) */
/* Runs the words in a volume or quote-string that is sitting on the
   stack.  Phrases use as their library the local library of the word 
   in which they are running.  Unpredictable if used outside a defined
   word, where word main should be used instead; see main1()). */
{
   char *p;
   int len,ret,typ;

   if((typ=tos->typ)!=VOL && typ!=STR) {
      stkerr(" local: ",STRORVOLNOT);
      return 0;
   }
   spaced();

   if((p=(char *)memgetn(tos->tex, \
      (len=(tos->row)*(tos->col))))==NULL) {
      return 0;
   }
   if(TRACE) {
      gprintf(" local quote: "); dup1s(); dot(); nc();
   }
   if(bufjump(p,len)) {
      context(*(cattag+onbuf-1)); /* catalog context same as prev lev */

      ret=setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */

      if(!*(jmpready+onbuf)) {

         *(jmpready+onbuf)=ENDSOU;
         drop(); /* may free tos.tex, but words are in p */
         perform();
      }
      else {
         if(ret==ABORT) {
            /* jumping out cleanly: */
            bufreturn();
            longjmp(abortbuf,ABORT);
         }
      }
      bufreturn();
      return 1; /* always returning 1 */
   }
   return 0;
}

int localstr(char *string, unsigned long *catptr)
/* Storing string as unnamed quote-string, followed by address of func-
   tion local(), so quote-string will be executed later.
*/
{
   int len;

   if(tokenq) { /* Verifying there's not one already: */
      stkerr(" localstr: ",TOKNOT);
      return 0;
   }
/* Strings from source are in mallocked tokenq when tokenq!=NULL. */
   if((tokenq=(char *)malloc(3+(len=strlen(string))))==NULL) {
      stkerr(" localstr: ",MEMNOT);
      return 0;
   }
/* Making tokenq into a quote-string for _UQUOTE: */
   memcpy((tokenq+1),string,len);
   *(tokenq)='\"';
   *(tokenq+len+1)='\"';
   *(tokenq+len+2)='\0';

   if(!_UQUOTE(catptr)) return 0;

   if(caton(tagnative("local"))) { /* word "local" in word.p, linked 
                                      to local() */
      *(catptr+IP)=(unsigned long)oncat;
      if(TRACE) {
         gprintf(" IP %lX: runtime %s at addr %lX, exe %lX",\
            IP,oncat->nam,oncat,oncat->exe); nc();
      }
      IP++;
      return 1;
   }
   stkerr(" localstr: local ",RUNNOT);
   return 0;
}

int _LOOP(unsigned long *catptr) /* (ptr xDO --- ) */
/* Installs runtime code _loop used for word LOOP.

   Sets jump IP for EXIT that is this IP.  When run() adds its 1, the
   total jump will be to the next IP, to the word that follows LOOP. */
{
   char *tok;
   long IP0;
   const int delta=0;
   catitem *p;
   long len;

   if(stkdepth()>1 && (int)pop()->real==xDO) {

      if(!caton(tagnative("LOOP,,"))) {
         stkerr(" LOOP: LOOP,, ",RUNNOT);
         return 0;
      }
      *(catptr+IP)=(unsigned long)oncat;

      if(TRACE) {
         gprintf(" IP %lX: runtime %s at addr %lX, exe %lX",\
            IP,oncat->nam,oncat,oncat->exe); nc();
      }
      tosret--;
      IP0=(long)popptr(&len,&tok);
      *(catptr+IP0)=delta+IP; /* storing jmp for EXIT at IP0 */

      if(TRACE) {
         p=(catitem *)*(catptr+IP0-1);
         gprintf(" IP %lX: runtime %s EXIT jump to IP %lX",\
            IP0,p->nam,1+delta+IP); nc();
      }
      IP++;
      return 1;
   }
   stkerr(" LOOP: ",BADLOOP);
   return 0;
}

int _loop() /* LOOP ( --- ) */
/* Runtime code for word LOOP. */
{
   tosret->index+=tosret->delta;

   if(((tosret->index)-(tosret->limit))*(tosret->slope)>0) {
      tosret--; /* looping is complete: popping return stack */
   }
   else {
      IP=tosret->jmp; /* IP jumping back */
   }
   return 1;
}

int _LOOPp(unsigned long *catptr) /* (ptr xDO --- ) */
/* Installs runtime code _loopp used for word +LOOP.

   Sets jump IP for EXIT that is this IP.  When run() adds its 1, the
   total jump will be to the next IP, to the word that follows +LOOP. */
{
   char *tok;
   long IP0;
   const int delta=0;
   catitem *p;
   long len;

   if(stkdepth()>1 && (int)pop()->real==xDO) {

      if(!caton(tagnative("LOOPP,,"))) {
         stkerr(" +LOOP: LOOPP,, ",RUNNOT);
         return 0;
      }
      *(catptr+IP)=(unsigned long)oncat;

      if(TRACE) {
         gprintf(" IP %lX: runtime %s at addr %lX, exe %lX",\
            IP,oncat->nam,oncat,oncat->exe); nc();
      }
      tosret--;
      IP0=(long)popptr(&len,&tok);
      *(catptr+IP0)=delta+IP; /* storing jmp for EXIT at IP0 */

      if(TRACE) {
         p=(catitem *)*(catptr+IP0-1);
         gprintf(" IP %lX: runtime %s EXIT jump to IP %lX",\
            IP0,p->nam,1+delta+IP); nc();
      }
      IP++;
      return 1;
   }
   stkerr(" +LOOP: ",BADLOOP);
   return 0;
}

int _loopp() /* +LOOP (n --- ) */
/* Runtime code for word +LOOP. */
{
   if(tos->typ==NUM) {
      tosret->index+=(long)tos->real;
      drop();

      if(((tosret->index)-(tosret->limit))*(tosret->slope)>0) {
         tosret--; /* looping is complete: popping return stack */
      }
      else {
         IP=tosret->jmp; /* IP jumping back */
      }
      return 1;
   }
   stkerr(" +LOOP ",BADLOOP);
/* If here, we're lost. */
   return(_endp()); /* the only thing to do is jump to end of inline */
}

int main1() /* main (hT or qS --- ) */
/* Runs the words in a volume or quote-string that is sitting on the
   stack.  Phrases use the main library as their local library, even if
   they are running inside a defined word (see word local, local() for
   other behavior). */
{
   return(xmain(0));
}

#ifdef KEYBOARD
int main2() /* UP (qS n --- ) */
/* Running at a run level popped from the stack, until BYE.  String S
   is the prompt ("" gives no prompt). */ 
{
   int err[NBUF],flag;
   char *p[NBUF],*s;

   if(tos->typ!=NUM) {
      stkerr(" main2: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=STR) {
      stkerr(" main2: ",STRNOT);
      return 0;
   }
   if((s=(char *)memgetn((tos-1)->tex,(tos-1)->col))==NULL) {
      return 0;
   }
   lop();
   *(err+onbuf)=stkerrcount;

   if(bufascend()) { /* popping run level, n, from stack */
      *(p+onbuf)=s;

      while(!BYE) { /* keyboard input loop until Esc-q sets BYE */
         nc();
         pushstr(*(p+onbuf));
         accept_keys(); 
         popint(&flag);
         if(flag) {
            nc();
            main1(); /* running the text */
         }
      }
      mallfree((void *)(p+onbuf));

      bufdescend();

      stkerrcount=*(err+onbuf); /* masking errors */

      BYE=0; /* turning off so levels below this won't exit */

      return 1; 
   }
   return 0;
}
#endif

int mainif() /* mainif (... qS --- ...) */
/* Run word S if it is found, otherwise silently return. */
{
   if(tos->typ!=STR) {
      stkerr(" mainif: ",STRNOT);
      return 0;
   }
   strchop();
   if(_exists(tos->tex)) return(xmain(0));
   else return(drop());
}

int mainstr(char *string, unsigned long *catptr) 
/* Storing string as unnamed quote-string, followed by address of func-
   tion main() (really main1()), so quote-string will be executed later.
*/
{
   int len;

   if(tokenq) { /* Verifying there's not one already: */
      stkerr(" mainstr: ",TOKNOT);
      return 0;
   }
/* Strings from source are in mallocked tokenq when tokenq!=NULL. */
   if((tokenq=(char *)malloc(3+(len=strlen(string))))==NULL) {
      stkerr(" mainstr: ",MEMNOT);
      return 0;
   }
/* Making tokenq into a quote-string for _UQUOTE: */
   memcpy((tokenq+1),string,len);
   *(tokenq)='\"';
   *(tokenq+len+1)='\"';
   *(tokenq+len+2)='\0';

   if(!_UQUOTE(catptr)) return 0;

   if(caton(tagnative("main"))) { /* word "main" in word.p, linked 
                                     to main1() */
      *(catptr+IP)=(unsigned long)oncat;
      if(TRACE) {
         gprintf(" IP %lX: runtime %s at addr %lX, exe %lX",\
            IP,oncat->nam,oncat,oncat->exe); nc();
      }
      IP++;
      return 1;
   }
   stkerr(" mainstr: main ",RUNNOT);
   return 0;
}

int mainword(char *name, unsigned long *catptr)
/* Storing name as unnamed quote-string, followed by address of func-
   tion wexe(), so quote-string name will be executed later as a word.
*/
{
   int len;

   if(tokenq) { /* Verifying there's not one already: */
      stkerr(" mainword: ",TOKNOT);
      return 0;
   }
/* Strings from source are in mallocked tokenq when tokenq!=NULL. */
   if((tokenq=(char *)malloc(3+(len=strlen(name))))==NULL) {
      stkerr(" mainword: ",MEMNOT);
      return 0;
   }
/* Making tokenq into a quote-string for _UQUOTE: */
   memcpy((tokenq+1),name,len);
   *(tokenq)='\"';
   *(tokenq+len+1)='\"';
   *(tokenq+len+2)='\0';

   if(!_UQUOTE(catptr)) return 0;

   if(caton(tagnative("wexe,"))) { /* word "wexe," in word.p will 
                                      run wexe() */
      *(catptr+IP)=(unsigned long)oncat;
      if(TRACE) {
         gprintf(" IP %lX: runtime %s at addr %lX, exe %lX",\
            IP,oncat->nam,oncat,oncat->exe); nc();
      }
      IP++;
      return 1;
   }
   stkerr(" mainword: wexe, ",RUNNOT);
   return 0;
}

int _NOINLINE()
/* Reporting that word cannot be used in an inline function. */
{
   gprintf(" %s: "INLINENOT,token); nc();
   stkerr("","");
   return 0;
}

int noop()
/* No operation. */
{
   return 1;
}

int pmain() /* pmain (hT or qS --- )  eval (hT or qS --- ) */
/* After parsing the infix text in volume T or string S on the stack, 
   run the resulting postfix words just as main does.

   Phrases use the main library as their local library, even if they
   are running inside a defined word (see word local, local() for
   other behavior). */
{
   return(xmain_parse(0));
}

int pryexe() /* pryexe (hV n --- ...) */
/* Fetch element n of vector V, a ptr that points to a catalog item,
   and execute the catalog item's exe function.  The stack item or
   items that result depend upon the type of catalog item. 

   This function combines functions pry() and exe().  Type checking of
   V and bounds checking of n that is done in pry() have been removed 
   here, and testing for ptr done in exe() has been removed.

   Word pryexe is functionally equivalent to phrase "pry exe," but the
   latter is recommended during debugging. 

   Use pryexe for speed with structures where offsets are automated and
   the chance of bad stack setup is small. */
{
   union {
      double x;
      int i[sizeof(double)];
   } X={0};
   unsigned long n;

   if(!popuint(&n)) return 0;
   n-=XBASE;

   X.x=*(tos->mat+n); /* ptr */

   drop(); /* V off the stack */

/* Fetching pattern of catitem pointer oncat; oncat may be a 64-bit
   pointer: */
   memcpy(&oncat,(catitem *)patget((int)*(X.i+oplus),ptrnum),
      sizeof(catitem *));

   return((*(unsigned long (*)()) oncat->exe)());
}

int _PULL(unsigned long *catptr)
/* Decrements count of local stack depth and installs runtime pull(). */
{
   tdepth1--;

   if(caton(tagnative("pull"))) { /* native "pull" in word.p */
      *(catptr+IP)=(unsigned long)oncat;
      if(TRACE) {
         gprintf(" IP %lX: runtime %s at addr %lX, exe %lX",\
            IP,oncat->nam,oncat,oncat->exe); nc();
      }
      IP++;
      return 1;
   }
   stkerr(" _PULL: pull ",RUNNOT);
   return 0;
}

int _PUSH(unsigned long *catptr)
/* Increments count of local stack depth and installs runtime push(). */
{
   tdepth1++;

   if(caton(tagnative("push"))) { /* native "push" in word.p */
      *(catptr+IP)=(unsigned long)oncat;
      if(TRACE) {
         gprintf(" IP %lX: runtime %s at addr %lX, exe %lX",\
            IP,oncat->nam,oncat,oncat->exe); nc();
      }
      IP++;
      return 1;
   }
   stkerr(" _PUSH: push ",RUNNOT);
   return 0;
}

int _REPEAT(unsigned long *catptr) /* (p1 p2 xWHILE --- ) */
/* Installs runtime code _repeat used for word REPEAT, sets jump IP
   from here back to BEGIN, and sets jump IP up to here from WHILE. */
{
   char *tok;
   long IP0;
   catitem *p;
   const int delta2=0;
   const int delta1=-1;
   long len;

   if(stkdepth()>2 && (int)pop()->real==xWHILE) {

      if(!caton(tagnative("Jmp,"))) {
         stkerr(" REPEAT: Jmp, ",RUNNOT);
         return 0;
      }
      *(catptr+IP)=(unsigned long)oncat;

      if(TRACE) {
         gprintf(" IP %lX: runtime %s at addr %lX, exe %lX",\
            IP,oncat->nam,oncat,oncat->exe); nc();
      }
      swap();


      IP0=(long)popptr(&len,&tok);
      IP++;
      *(catptr+IP)=IP0+delta1; /* storing jmp to BEGIN at next IP */
      if(TRACE) {
         p=(catitem *)*(catptr+IP0);
         gprintf(" IP %lX: runtime %s jump to IP %lX",\
            IP0,p->nam,1+delta1+IP0); nc();
      }
      IP0=(long)popptr(&len,&tok);
      *(catptr+IP0)=IP+delta2; /* storing jmp from WHILE to here */

      if(TRACE) {
         p=(catitem *)*(catptr+IP0-1);
         gprintf(" IP %lX: while jump to IP %lX",\
            IP0,p->nam,1+delta2+IP); nc();
      }
      IP++;
      return 1;
   }
   stkerr(" REPEAT: ",BADLOOP);
   return 0;
}

int run() /* run (... hP --- ...) */
/* Runs a vector of inline addresses that ends with 0 0. */
{
   int err[NBUF],ret;
   struct inlret *tosret1[NBUF];
   char *tag;

   if(!runup()) return 0; /* jump to new inline level, onrun */

/* inlptr, IP, IPend, and oncat are new (at this point) global vari-
   ables shared by runtime functions at this run level, onrun. */

   if((inlptr=(catitem (**)())popptr(&IPend,&tag))==NULL) {
      rundn();
      return 0;
   }
/* Returned IPend is len; make it point to 1st of two ending 0 */
   IPend--; 
   IPend--; 

   *(tosret1+onrun)=tosret;
   *(err+onrun)=stkerrcount;

   if(bufup1()) {

      context(tag); /* giving onbuf the library context of inlptr */

      ret=setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */

      if(!*(jmpready+onbuf)) {

         *(jmpready+onbuf)=ENDPTR;

         while((oncat=(catitem *)*(inlptr+IP))!=NULL) {

            (*(unsigned long (*)())oncat->exe)();
            IP++;
         }
      }
      else {
         if(ret==ABORT) {
        /*  jumping out cleanly: */
            bufdn1();
            rundn();
            longjmp(abortbuf,ABORT);
         }
      }
      bufdn1();

   /* Ups and downs of loops and branches must all cancel out, so
      the current tosret equals tosret1 saved at the beginning: */
      if(onrun>0) ret=(tosret==*(tosret1+onrun));

      else ret=1; /* lax on run level 0, since it may be an abort */

      if(!ret) {
         gprintf(" run: %s, onbuf: %d, onrun: %d",RETNOT,onbuf,onrun);
         nc();
         stkerr("","");
      }
      ret=(
         ret &&
         *(err+onrun)==stkerrcount
      );
      rundn();
      return ret;
   }
   rundn();
   return 0;
}

void rundn()
{
   if(TRACE) {
      if(onrun>0) {
         gprintf("---- Down to inline level %d; stack depth = %d ---", \
         onrun-1,stkdepth()); nc();
      }
      else {
         gprintf("---- End inline level %d; stack depth = %d ----", \
         onrun,stkdepth()); nc();
      }
   }
   inlptr=*(Inlptr+onrun);
   oncat=*(Oncat+onrun);
   IP=*(Ip+onrun);
   IPend=*(Ipend+onrun);
   tosret=*(Tosret+onrun);

   onrun--;
}

void runinit()
/* Initializes the pointers for run levels of inline functions, and
   byte order flags used by exe(). */
{
   int i=0;
   int *k1=NULL;

   if(TRACE) {
      gprintf(" initializing run"); nc();
   }
   IMMEDIATE=0;

   for(;i<NBUF;i++)  {
      *(Inlptr+i)=NULL; 
      *(Oncat+i)=NULL;
      *(Ip+i)=0;
      *(Ipend+i)=0;
      *(retbase+i)=(retstk+locvec(DEPRSTK,i));
   }
   onrun=-1;

/* Offsets used in exe() and ptr(): */
   if(byteorder(&k1)==1234) { /* little endian */
     kplus=1;
     oplus=0;
   }
   else { /* big endian and pdp endian (4321 and 3412) */
     kplus=0;
     oplus=1;
   }
   kzero=0x80000000;
}

int runt() /* runt (... hP --- ...) */
/* Runs, with trace, a vector of inline addresses that ends with 0 0. */
{
   int err[NBUF],len,ret;
   struct inlret *tosret1[NBUF];
   char *tag;
   char str[20]={'\0','\0','\0','\0','\0','\0','\0','\0',
                 '\0','\0','\0','\0','\0','\0','\0','\0',
                 '.','.','.','\0'};

   if(!runup()) return 0; /* jump to new inline level, onrun */

/* inlptr, IP, IPend, and oncat are new (at this point) global vari-
   ables shared by runtime functions at this run level, onrun. */

   if((inlptr=(catitem (**)())popptr(&IPend,&tag))==NULL) {
      rundn();
      return 0;
   }
   oncat=(catitem *)*(inlptr+IP);
   gprintf(" runt. inlptr: %lX, oncat: %lX, tag: %s",inlptr,oncat,tag);
   nc();

/* Returned IPend is len; make it point to 1st of two ending 0 */
   IPend--; 
   IPend--; 

   *(tosret1+onrun)=tosret;
   *(err+onrun)=stkerrcount;

   if(bufup1()) {

      context(tag); /* giving onbuf the library context of inlptr */

      ret=setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */

      if(!*(jmpready+onbuf)) {

         *(jmpready+onbuf)=ENDPTR;

         while((oncat=(catitem *)*(inlptr+IP))!=NULL) {

            if(oncat->typ==STRI) {
               memcpy(str,oncat->stk->tex,\
                  (len=MIN(oncat->stk->col,16)));
               *(str+len)='\0';
               gprintf(" runt. onrun: %d, IP: %lX, string: %s",onrun,\
                  IP,str); nc();
            }
            else {
               gprintf(" runt. onrun: %d, IP: %lX, word: %s",onrun,IP,\
                 oncat->nam); nc();
            }
            (*(unsigned long (*)())oncat->exe)();
            IP++;
         }
      }
      else {
         if(ret==ABORT) {
         /* jumping out cleanly: */
            bufdn1();
            rundn();
            longjmp(abortbuf,ABORT);
         }
      }
      bufdn1();

   /* Ups and downs of loops and branches must all cancel out, so
      the current tosret equals tosret1 saved at the beginning: */
      if(onrun>0) ret=(tosret==*(tosret1+onrun)); 

      else ret=1; /* lax on run level 0, since it may be an abort */

      if(!ret) {
         gprintf(" runt: %s, onbuf: %d, onrun: %d",RETNOT,onbuf,onrun);
         nc();
         stkerr("","");
      }
      ret=(
         ret &&
         *(err+onrun)==stkerrcount 
      );
      rundn();
      return ret;
   }
   rundn();
   return 0;
}

int runup()
{
   if(1+onrun>=NBUF-1) {
      stkerr(" runup: ",MAXRUNLEV);
      return 0;
   }
   onrun=MAX(onrun,-1);

   if(TRACE) {
      if(1+onrun>0) {
         gprintf("---- Up to inline level %d; stack depth = %d --", \
         1+onrun,stkdepth()); nc();
      }
      else {
         gprintf("---- Begin inline level %d; stack depth = %d --", \
         1+onrun,stkdepth()); nc();
      }
   }
   onrun++;

   /* saving pointers of current inline run level */
   *(Inlptr+onrun)=inlptr;
   *(Oncat+onrun)=oncat;
   *(Ip+onrun)=IP;
   *(Ipend+onrun)=IPend;
   *(Tosret+onrun)=tosret;

   /* initializing new inline run level */
   IP=0;
   tosret=*(retbase+onrun);
   tosret->list=0;

   return 1;
}

int _THEN(unsigned long *catptr) /* (ptr flag --- ) */
/* Sets the jump for the preceding IF or ELSE. 

   Sets jump IP that is 1 less than the current one.  When run() adds
   its 1, the total jump will be to this IP. */
{
   char *tok=NULL;
   long IP0;
   const int delta=-1;
   catitem *p;
   long flag;

   if(stkdepth()>1 && (((flag=(int)pop()->real)==xIF) || flag==xELSE)) {
      IP0=(long)popptr(&flag,&tok);
      *(catptr+IP0)=delta+IP; /* storing jump to here from IP0 */

      if(TRACE) {
         p=(catitem *)*(catptr+IP0-1);
         gprintf(" IP %lX: runtime %s jump to IP %lX",\
            IP0,p->nam,1+delta+IP); nc();
      }
      return 1;
   }
   stkerr(" THEN: ",BADBRANCH);
   return 0;
}

int _UNTIL(unsigned long *catptr) /* (ptr xBEGIN --- ) */
/* Installs runtime Jz and sets jump IP for return to beginning of
   BEGIN ... UNTIL loop. */
{
   char *tok;
   long IP0,len;
   const int delta=-1;
   catitem *p;

   if(stkdepth()>1 && (int)pop()->real==xBEGIN) {

      if(!caton(tagnative("Jz,"))) {
         stkerr(" UNTIL: Jz, ",RUNNOT);
         return 0;
      }
      *(catptr+IP)=(unsigned long)oncat;

      if(TRACE) {
         gprintf(" IP %lX: runtime %s at addr %lX, exe %lX",\
            IP,oncat->nam,oncat,oncat->exe); nc();
      }
      IP0=(long)popptr(&len,&tok);
      IP++;
      *(catptr+IP)=IP0+delta; /* storing jmp to BEGIN at next IP */

      if(TRACE) {
         p=(catitem *)*(catptr+IP0);
         gprintf(" IP %lX: runtime %s jump to IP %lX",\
            IP,oncat->nam,1+delta+IP0); nc();
      }
      IP++;
      return 1;
   }
   stkerr(" UNTIL: ",BADLOOP);
   return 0;
}

int _UQUOTE(unsigned long *catptr) /* inlining " or ' */
/* Puts unnamed quote-string, currently in tokenq, into the local
   library of the catalog, and puts its catalog address into the 
   list of inline pointers. */
{
   int chmax=16,len;  
   char *fX="%X#";  
   char name[2*16]; /* when use addr, name looks like: 80696381# */
   char str[20]={'\0','\0','\0','\0','\0','\0','\0','\0',
                 '\0','\0','\0','\0','\0','\0','\0','\0',
                 '.','.','.','\0'}; 

/* Strings from source are in mallocked tokenq when tokenq!=NULL. */

   if(strlen(tokenq)-2>chmax) sprintf(name,fX,tokenq); 

   else { /* using the string (with surrounding quotes) for name: */
      memcpy(name,tokenq,(len=strlen(tokenq))); /* len < chmax */
      *(name+len)='\0';
   }
   if(quoted(name,tokenq)) {
      if(caton(tagged(name,*(cattag+onbuf)))) {

      /* Adding a word address to the list: */
         *(catptr+IP)=(unsigned long)oncat;

         if(TRACE) {
         /* Note: quoted() removed surrounding quotes from tokenq 
                  before placing it into the catalog */
            memcpy(str,tokenq,(len=MIN(16,strlen(tokenq))));
            gprintf(" IP %lX: string %s at addr %lX, exe %lX", \
               IP,str,*(catptr+IP),oncat->exe); nc();
         }
         IP++;
         return 1;
      }
   }
   stkerr(" _UQUOTE: ",STRNOTC);
   return 0;
}

int wexe() /* (... qS --- ...) */
/* This function finds unresolved references at run time and modifies
   inline code to reflect the found reference.  A noticeable speed 
   improvement was obtained with this function on the 486 DX2-50 
   machine where this program began in 1999.

   July 2009: the demo given below was reverified.

   Find booked item S in the local library of the running word, or
   a native word, constant or defined word S in the main library,
   and run it.

   Will not find a booked item that is not in the local library.

   The first time they are executed, this function modifies the point-
   ers in the list of pointers created by function inline1().

   It replaces an unresolved reference denoted by a string and requir-
   ing a catalog lookup, with the execution address that results from 
   the lookup.  

   Additional executions of the same list of pointers will then use the
   newly planted execution address directly, and not come here again. */
{
   unsigned long p;
   char *S;
   int len,m,n,ret=0,typ;

   if((typ=tos->typ)!=VOL && typ!=STR) {
      stkerr(" wexe: ",STRORVOLNOT);
      return 0;
   }
   if((S=(char *)memgetn(tos->tex,(len=1+tos->col)))==NULL) return 0;

   n=strspn(S," "); /* offset to 1st char in S */
   m=n;
   while(!isspace((int)*(S+m)) && m<len) m++;
   *(S+m)='\0';

   if(caton2(S+n)) { /* caton2() is the default search strategy */
      drop();
      p=(unsigned long)oncat;
   /* Running exe function of catalog item oncat: */
      if((*(unsigned long (*)()) oncat->exe)()) {

      /* Substituting found catitem p for the string of the unre-
         solved reference at IP-1, followed by the address of the
         no-operation function at the address, IP, of this function.

         This operation is demonstrated by running a word made using 
         word inline (these require word "run" to make them go) and 
         having unresolved references (need to use -strict) and run-
         ning with "runt" instead of "run" to show a trace. 

         Here is a set of words to demonstrate the substitution.  Be-
         cause efg and ghi follow the creation of abc, they are un-
         resolved references in abc.  All abc knows is to come here
         to find items efg and ghi in the catalog and run them.

            "efg ghi" "abc" -strict inline strict
            define: efg " efg here" . nl ;
            define: ghi " ghi here" . nl ;

         After pasting the three lines above at the ready prompt:

            Before it is ever run, here are the addresses in word abc: 
              [tops@clacker] ready > abc (hPTR) dot
               8AB1FC0 8AB9000 8BC8CB8 8AB9000 0 0

            Here is running abc (using runt (run with trace) instead
            of run) for the first time.  Note that efg and ghi are 
            strings (at IP 0 and 2), and that this word, wexe, is run 
            (at IP 1 and 3):
              [tops@clacker] ready > abc (hPTR) runt
               runt. inlptr: 8B10A40, oncat: 8AB1FC0, tag: abc
               runt. onrun: 1, IP: 0, string: efg
               runt. onrun: 1, IP: 1, word: wexe,
               efg here
               runt. onrun: 1, IP: 2, string: ghi
               runt. onrun: 1, IP: 3, word: wexe,
               ghi here

            Here are the addresses in word abc after it has been run
            once.  They are not the same as the ones shown previously:
              [tops@clacker] ready > abc (hPTR) dot
               8AB6680 8ACD220 8BCB528 8ACD220 0 0

            Here is running abc again.  Note that efg and ghi are now
            words being run (at IP 0 and 2) due to the addresses stored
            by wexe the first time abc was run, and noop is run where 
            wexe ran previously (at IP 1 and 3):
              [tops@clacker] ready > abc (hPTR) runt
               runt. inlptr: 8B10A40, oncat: 8AB6680, tag: abc
               runt. onrun: 1, IP: 0, word: efg
               efg here
               runt. onrun: 1, IP: 1, word: noop
               runt. onrun: 1, IP: 2, word: ghi
               ghi here
               runt. onrun: 1, IP: 3, word: noop

         In each run, words efg and ghi ran as noted by their outputs:
         "efg here" and "ghi here."

         The first time word abc is run, it will come here, to function
         wexe().  On the second and all subsequent times it will not 
         because the addresses stored at IP-1 and IP, by the following
         lines of code, will be run instead. */ 

      /* Storing address of catalog item p in IP-1, where string
         for p is currently stored: */
         *(inlptr+IP-1)=(catitem (*)())p;

      /* Storing no-operation at IP, where address of this function 
         is currently stored: */
         caton(tagnative("noop")); 
         *(inlptr+IP)=(catitem (*)())oncat;
         ret=1;
      }
   }
   else {
      gprintf(" wexe: word %s not found",S); nc();
      stkerr("","");
      mallfree((void *)&S);
      if(rlconsole()>-1) /* HALT when interactive program */
         gprintf(" wexe: halt and return to interactive prompt"); nc();
         HALT(); /* stop everything if keyboard() (key.c) */
      return 0;
   }
   mallfree((void *)&S);
   return ret;
}

int _WHILE(unsigned long *catptr) /* (p1 xBEGIN --- p1 p2 xWHILE) */
/* Installs runtime Jz used for word WHILE and makes room for jump IP
   to be stored by _REPEAT. */
{
   long Jmpto;

   if(stkdepth()>1 && (int)pop()->real==xBEGIN) {

      if(!caton(tagnative("Jz,"))) {
         stkerr(" WHILE: Jz, ",RUNNOT);
         return 0;
      }
      *(catptr+IP)=(unsigned long)oncat;

      if(TRACE) {
         gprintf(" IP %lX: runtime %s at addr %lX, exe %lX",\
            IP,oncat->nam,oncat,oncat->exe); nc();
      }
      IP++;
      Jmpto=IP;
      IP++; /* space for jmp IP to be stored by _REPEAT */
      return(pushptr1((unsigned long (**)())(Jmpto)) &&
         pushint(xWHILE)
      );
   }
   stkerr(" WHILE: ",BADBRANCH);
   return 0;
}

int xcatlib() /* catlib (hT nLib --- ) */ 
/* Execute text T, a VOL or STR, using the functions in catalog library
   nLib. */
{
   int len[NBUF],NLIB=-1,nlib[NBUF],ret,typ;
   char *T[NBUF];

   if(!popint(&NLIB)) return 0;

   if((typ=tos->typ)!=VOL && typ!=STR) {
      stkerr(" catlib: ",STRORVOLNOT);
      return 0;
   }
   if((*(T+onbuf)=(char *)memgetnNL(tos->tex,tos->row,tos->col,
      (len+onbuf)))==NULL) return 0;

   if(TRACE) {
      gprintf(" catlib phrase: "); dup1s(); dot(); nc();
   }
   *(nlib+onbuf)=onlib; /* record lib from */

   if(bufjump(*(T+onbuf),*(len+onbuf))) {

      drop(); /* drop stack item T */
      *(nlib+onbuf)=NLIB; /* record lib new */

      if(!nativewords(*(nlib+onbuf))) { /* get functions, set lib new */
         bufreturn();
         return 0;
      }
      ret=setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */
      if(!*(jmpready+onbuf)) {
         *(jmpready+onbuf)=ENDSOU;
 
         switch(onlib) {
            case LIB0:
               perform();
            break;

            default:
               perform();
            break;
         }
      }
      else {
         if(ret==ABORT) {
         /* Jumping out cleanly: */
            bufreturn();
            libset(*(nlib+onbuf)); /* set lib from */
            longjmp(abortbuf,ABORT);
         }
      }
      bufreturn();
      return(libset(*(nlib+onbuf))); /* set lib from */
   }
   mallfree((void *)&T);
   return 0;
}

int xmain(int flag) 
/* Runs the words in a volume or quote-string T stack.  Phrases use the
   main library as their local library, even if they are running inside
   a defined word (see word local, local() for other behavior). */
{
   int err[NBUF],infix_save[NBUF],len[NBUF],ret,typ,xMAIN;
   char *p[NBUF];

   if((typ=tos->typ)!=VOL && typ!=STR) {
      stkerr(" main: ",STRORVOLNOT);
      return 0;
   }
   if((*(p+onbuf)=(char *)memgetnNL(tos->tex,tos->row,tos->col,
      (len+onbuf)))==NULL) return 0;

   if(TRACE) {
      gprintf(" main phrase: "); dup1s(); dot(); nc();
   }
   if(bufjump(*(p+onbuf),*(len+onbuf))) {
      drop(); /* may free tos.tex, but words are in p */
      *(infix_save+onbuf)=INFIX;
      *(err+onbuf)=stkerrcount;

      ret=setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */

      if(!*(jmpready+onbuf)) {

         *(jmpready+onbuf)=ENDSOU;
         perform();
      }
      else {
         if(ret==ABORT) {
         /* Jumping out cleanly: */

         /* Restore old INFIX flag if HALT: */
            if(*(infix_save+onbuf)) infix(); else postfix();

            xMAIN=(*(err+onbuf)==stkerrcount);
            if(flag) {
               if(xMAIN) pushint(xTRUE);
               else pushint(xFALSE);
            }
            bufreturn();
            longjmp(abortbuf,ABORT);
         }
      }
      xMAIN=(*(err+onbuf)==stkerrcount);
      bufreturn();

#ifdef KEYBOARD
      if(!xMAIN && onbuf!=RLKEY) report(NULL);
#endif

      if(flag) {
         if(xMAIN) pushint(xTRUE);
         else pushint(xFALSE);
      }
      return 1; /* always returning 1 */
   }
   if(flag) pushint(xFALSE);
   mallfree((void *)(p+onbuf));
   return 0;
}

int xmain_parse(int flag) /* (hT or qS --- ) */
/* After parsing the infix text in volume T or string S on the stack, 
   run the resulting postfix words just as main does.

   Phrases use the main library as their local library, even if they
   are running inside a defined word (see word local, local() for
   other behavior). */
{  
   if(tos->typ!=VOL && tos->typ!=STR) {
      stkerr(" main_parse: ",STRORVOLNOT);
      return 0;
   }
   return(
   /* Stack friendly but takes a little longer: */
      pushstr("scratch save scratch psource scratch deleteif") &&
      xmain(flag)

   /* Not stack friendly:   
      parse() &&
      xmain(flag)
   */
   );
}  
