/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

/* main.c  March 1999

   Copyright (c) 1999   D. R. Williamson

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "word.h"

int main(int argc, char *argv[])
{
   if(
   /* Starting: */
      start(argc,argv)) {

   /* Running the input file: */
      source(); 

   /* Stopping: */
      quit();
   }
   else {
      OK_EXIT=1;
      INFIX=0;
      exit(EXIT_FAILURE);
   }
   return 0;
}

void perform()
/* Fetching and running words at run level onbuf, until end. 

   This is the heart of the program, and works as follows:

     - fetch next token from source (tokenget())
     - if it is a number (number()), push it to the stack (pushd())
     - otherwise, find a match in the catalog and run it (catexe())
     - otherwise, report an error (report())
     - repeat until source ends (and never say "die") */
{
   char *token=NULL;
   double x;

   while(!*(endbuf+onbuf) && (token=tokenget())!=NULL) {

      if(TRACE) {
         gprintf(" next word: %s",token);
         nc();
      }
      if(number(token,&x)) {
      /* Pushing to stack if number: */
         if(!pushd(x)) report(token);
      }
      else {
      /* Executing a catalog item if matched: */
         if(!catexe(token)) report(token);
      }
      token=NULL;
   }
   if(!*(endbuf+onbuf)) end1(); /* closing out jump not performed */

   if(TRACE) {
      gprintf(" Returning from perform on word %s; stack depth = %d", \
         token,stkdepth());
      nc();
   }
}

int quit() /* exit, quit ( --- ) */
{
   int code=EXIT_SUCCESS;

   INFIX=0;

   OK_EXIT=1;
   libset(LIB0); /* back to start up library for exit words */

   if(!_exists("clean")) {
      set_stdout();
      exit(EXIT_FAILURE);
   }
   unload("clean");
   if(stkerrabs) code=EXIT_FAILURE;

   exit(code);
}

static int doing_keys;

int source() /* source (qFile --- ) */
/* Running phrases in file named on the stack.

   Files sourced can source other files, so this function is reentrant,
   but only to level NBUF--about 128 (see main.h). */
{
   char *filename,*fname[NBUF],infix_save[NBUF],*s[NBUF];
   int err[NBUF],f[NBUF];

   if(tos->typ!=STR && tos->typ!=VOL) { 
      stkerr(" source: ",STRORVOLNOT); 
      return 0; 
   } 
/* If VOL instead of STR, uses the first line. */ 
   strchop();
   filename=memgetn(tos->tex,tos->col);

   if(TRACE) {
      gprintf(" Sourcing file %s",filename); 
      nc();
   }
   filefound(); /* look in known places and add path */
   popint(f+onbuf); /* popping found flag from stack */
   if(!*(f+onbuf)) { 
      gprintf(" source: file %s not found",filename); 
      nc();
      stkerr("","");
      mallfree((void *)&filename);
      return 0;
   }
/* At this point, file name with path is on the stack.

   If running INFIX and the file extension is not .v, run psource() to 
   do parsing.  

   A file being parsed by psource() can contain symbol << to switch to 
   postfix and >> to switch back to parsing infix.  This is not true of
   a file being interpreted by source().  

   The convention is becoming .v for postfix files, but any extension
   is ok as long as the program is not in parsing mode (no parsing when
   INFIX equal to zero).

   Extension .n for infix files is being tried. */

   if(INFIX && *(tos->tex+tos->col-1)!='v') {
      if(_exists("psource")) {
         mallfree((void *)&filename);
         return(
            pushstr("psource") && /* running psource() instead */
            xmain(0)
         );
      }
   }
/* This region always runs at start up to do the .v files (if there is
   a -i argv switch for parsing, it hasn't been read yet so INFIX=0): */
   if((*(s+onbuf)=fileload(tos->tex))==NULL) { /* name includes path */
      mallfree((void *)&filename);
      return 0;
   }
   drop(); /* File name off stack */
   if(bufjump(*(s+onbuf),strlen(*(s+onbuf)))) { /* to next run level */
      *(infix_save+onbuf)=INFIX;
      *(err+onbuf)=stkerrcount;

      *(f+onbuf)=setjmp(*(jmpenv+onbuf)); /* <<-- longjmp lands here */

      if(!*(jmpready+onbuf)) { /* falls through first time */
         *(jmpready+onbuf)=ENDSOU;
         *(fname+onbuf)=filename; /* file name into fname(onbuf) */

         perform();
      }
      else {
         if(*(f+onbuf)==ABORT) {

            if(*(infix_save+onbuf)) infix(); else postfix();

            mallfree((void *)(fname+onbuf));
            bufreturn(); /* *(s+onbuf) is freed in bufreturn() */
            longjmp(abortbuf,ABORT);
         }
      }
      if(*(infix_save+onbuf)) infix(); else postfix();

      if(TRACE) {
         gprintf(" source: end of file %s",*(fname+onbuf)); 
         nc();
      }
      *(f+onbuf)=(*(err+onbuf)==stkerrcount);
      mallfree((void *)(fname+onbuf));
      bufreturn(); /* *(s+onbuf) is freed in bufreturn() */
      return(*(f+onbuf+1)); /* +1 because bufreturn decrements onbuf */
   }
   return 0;
}

int start(int argc, char *argv[]) /* ( --- qF | empty) */
/* Starting the program.  If successful, returns 1 and file name is 
   on the stack ready to be sourced; otherwise returns 0 and the stack 
   is empty. */
{
   int ok,und_have=0;
   char *ARGV="ARGV its rows .i ' ARGV rows:' . nl 4 indent . nl";

#  ifdef KEYBOARD
      char *keyf=NULL;
#  endif
   char *make_logname="INFIX << push \
                      '_bin' 'filelog' localref libpry \
                      pull infix_set";
   char *nullenv=NULL;
   char *undenv;

/* Initial values: */
   BIN=xTRUE;
   BYE=0;
   CATMSG=1;
   DELTA_T=0;
   doing_keys=0;
   DONEOFF=0;
   errTHRESHOLD=9;
   HLMAX=50;
   INDENT=0;
   KEYS=0;
   LOCKED=0;
   LOGKEEP=0;
   NEW=xTRUE;
   NULLENV=&nullenv;
   OK_EXIT=0;
   PUBLIC=0;
   SBIN=memgetn("",0);
   SH=memgetn("",0);
   SHELLENV=PROGENV; /* for shell cmd env; use shellset to change */
   STKBAL=1;
   STRICT=1;
   SYSOUT=NULL;
   sysout=stdout;
   TRACE=0;
   TTY1=NULL;
   USEC=40000; /* microseconds per cycle at 25 Hz */
   WIDTH=80;
   XBASE=1;

/* The order of these initializing calls is important: */
   INFIX=0; /* start up commands are postfix */
   inpoinit();
   stkinit();
   bufinit();
   runinit();
   mathinit();
   catinit();

   if(!nativewords(LIB0)) {
      fprintf(stderr," start: failed to load native words\n");
      return 0;
   }
   if(!setargs(argc,argv)) { /* all argv into word ARGV */
      fprintf(stderr," start: failed to load command line arguments\n");
      return 0;
   }
#  ifdef KEYBOARD
      keyf="key.v"; /* file of keyboard words */
#  endif

   ok=(bootfile() && source()); /* sourcing boot file */

   if(!_exists(".s")) { /* sourcing was cut off, as with -h or -v */
      return 0;         /* no error message; return 0 to exit main() */
   }
   ok=(ok &&
#     ifdef X11
         xterminit() &&
#     endif
      terminit() 
   );
   if(ok) { 
   /* Source uboot.v to create user words at start up. */
      pushstr("uboot.v");
      filefound(); /* look in known places and add path */
      popint(&ok); /* popping found flag from stack */
      if(ok) {
         source();
         pushstr("yes 'uboot_done' 'ans' bank");
         xmain(0);
      }
   /* Initialization and sourcing of bootfile is complete.
      Next, decide what file to source.

      Logic:
         IF argv[0] matches the underscore environment variable (with
         paths stripped off), we know this is not a script being run.

            IF last argv is a path/file, source it.

               Here is why last argv is designated to be the file to 
               source:

                  Want to be able to enter the file name at the command
                  prompt, and argvs given at the command prompt follow
                  any that might be given on an alias invoking the pro-
                  gram.  Thus the only way to allow many argvs for any
                  purpose is to make the file name be the last one.

                  Making the file name be first, for example, followed
                  by other argvs, would require an alias to contain the
                  file name, and thus preclude giving the file name on
                  the fly as the program is invoked by the alias.

               Where this can go wrong: 
                  When running interactively, no file to source is spec-
                  ified.  But if the last argv is a file name, perhaps 
                  for some other use, the program will source it anyway.

                  So, the last argv should be some non-file symbol.

            ELSE keyboard mode; source keyf.

            THEN

         ELSE when argv[0] does not match the underscore environment
            variable, argv[0] is the program being run from the #! line
            in a script, and the underscore environment variable 'might'
            be the file name of the script.

            IF the underscore variable matches argv[1], source it
            ELSE file to source is either argv[1] or argv[2]; see
               the code below where this is sorted out.
            THEN
         THEN */

   /* Fetch the underscore variable: */
      ok=(getenv("_")!=NULL);
      if(ok) {
         undenv=getenv("_");
         und_have=1;
      }
      else {
         und_have=0;
         undenv="(null)";
      }
      if(sysout!=stdout) {
      /* Write ARGV list and underscore env variable to log: */
         pushstr(ARGV);
         xmain(0);
         gprintf(" underscore (_) environment variable: %s",undenv);
         nc();
      }
      ok=0;
      if(und_have && argc>1) { /* see if argv[0] and undenv 
                                  match (sans paths): */
         pushstr(argv[0]);
         pushstr("-path");
         xmain(0);
         pushstr(undenv);
         pushstr("-path");
         xmain(0);
         pushstr("alike");
         xmain(0);
         popint(&ok);
      }
   /* In searches to find a file to source, filefound() is used.  It 
      restricts searching to directories the program knows about by
      looking locally, then along usrpath, then along syspath, then 
      along docpath. */

      if(argc==1 || ok) {
      /* If here, the underscore env variable and argv[0] match. */
         if(argc>1) {
            pushstr(argv[argc-1]); /* last argv */
            filefound();           /* seeing if last argv is a file */
            popint(&ok);           /* popping file ok flag from stack */
         }
         else ok=0; /* case of no argvs past prog name, argv[0] */

#ifdef KEYBOARD
         if(!ok) {
            pushstr(keyf);
            filefound(); /* looking for keyf file */
            popint(&ok); /* popping file ok flag from stack */
            doing_keys=ok;
         }
#endif
      }
      else {
      /* If here, the underscore env variable and argv[0] do not
         match, and a script file is being run.  Must figure out the
         name of the script file being run so it can be sourced.

         The underscore env variable cannot be relied upon, since it
         might be something like /usr/bin/nohup for a script being 
         run in background.  But if argv[1] or argv[2] do match it, 
         then it is the script to source. */

         if(und_have && 
            ((undenv && strcmp(undenv,argv[1])==0) || (argc>2 &&
            (undenv && strcmp(undenv,argv[2])==0)))) {
            pushstr(undenv);
            filefound(); /* seeing if underscore env variable is file */
            popint(&ok); /* popping file ok flag from stack */
         }
         else {
         /* If here, the underscore env variable does not name the
            script file to source.  Instead, it is either argv[1] or
            argv[2]. */

            if(argc==2 || (argc>2 && strcmp(argv[1],argv[2])==0)) {
               pushstr(argv[1]);
               filefound(); /* seeing if argv[1] is a file */
               popint(&ok); /* popping file ok flag from stack */
            }
            else {
            /* Try the last argv if more than 2: */
               if(argc>2) {
                  pushstr(argv[argc-1]); /* last */
                  filefound(); /* seeing if last argv is a file */
                  popint(&ok); /* popping file ok flag from stack */
               }
               if(!ok) {
               /* Need to choose between argv[1] and argv[2].  This 
                  is a place we don't want to be, and can be avoided by 
                  requiring a sole argv on the #! line to not be a file 
                  name when running something like nohup.

                  Just take a shot: if argv[1] is a file, source it;
                  otherwise, try argv[2]. */

                  pushstr(argv[1]);
                  filefound(); /* seeing if argv[1] is a file */
                  popint(&ok); /* popping file ok flag from stack */

                  if(!ok) {
                     pushstr(argv[2]);
                     filefound(); /* seeing if argv[2] is a file */
                     popint(&ok); /* popping file ok flag from stack */
                  }
               }
            }
         }
      }
      if(ok) { /* If here, have file name on stack, ready to source.

         If doing keys, file name keyf is on the stack to be sourced 
         and begin running interactively; otherwise, not doing keyboard
         and name of file to be sourced in batch mode is on the stack. 
         */
         if(doing_keys) { /* Have additional start up to do: sourcing
            keyf to start the keyboard.  Leave log file open and let 
            keyf, to be sourced next, delete it when start up is 
            complete and interactive running begins. */
         }
         else { /* Not doing keyboard; start up is nearly complete. */

         /* Set infix or postfix: */
            pushstr("'-i' argv chars 0>");
            xmain(0);
            popint(&ok);
            if(ok) infix();

            pushstr("'-p' argv chars 0>");
            xmain(0);
            popint(&ok);
            if(ok) postfix();

         /* Setting fence to prevent redefinition of words so far: */
            pushstr("fence");
            xmain(0); 

            gprintf(" info: start up complete");
            nc();

         /* Opening STDOUT (which closes the log file first): */
            set_stdout();

         /* Deleting the log file if no start up errors: */
            ercnt();
            popint(&ok);
            if(ok==0) {
            /* log file name to stack: */
               pushstr(make_logname);
               xmain(0); /* log file name on stack */

               pushstr("-l");
               pushstr("argv chars any");
               xmain(0);
               popint(&ok); /* keep log file if there is a -l argv */
               if(ok || LOGKEEP) {
                  gprintf(" log file kept: %s",tos->tex);
                  nc();
                  drop();
               }
               else {
                  delete(); /* deleting log file */
               }
            }
         }
      /* Initialize random seed */
         pushint(SEED0);
         seedset();

         return 1; /* returning to source the file on the stack */
      }
      else {
         fprintf(stderr," no input file found \n");
         if(sysout!=stdout) {
            gprintf(" main: no input file found");
            nc();
         }
      }
   }
/* If here, never got through basic initialization functions and the 
   sourcing of boot file; or, no input file was found to source. */

   if(sysout!=stdout) {
      gprintf(" info: start up failed"); /* writing on log file */
      nc();
   }
   set_stderr(); /* vector output to STDERR */

   gprintf(" Start up failed. ");
   pushstr(make_logname);
   xmain(0); /* log file name on stack */
   dup1s();
   fileq();
   popint(&ok); /* show name of log file if it exists */
   if(ok) {
      gprintf(" See log file: %s",tos->tex);
   }
   nc();
   drop();
   return 0;
}
