/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2012  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

/* mat.c  April 1999

Copyright (c) 1999  D. R. Williamson

Contributions from A. Danial, November 2001:
   tril(), triu()
*/

#ifndef __STRICT_ANSI__
   #define __STRICT_ANSI__
#endif
#define _XOPEN_SOURCE 500 /* snprintf */
#include <stdio.h>

#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "main.h"
#include "stk.h"
#include "lib.h"

#include "ctrl.h"
#include "exe.h"
#include "inpo.h"
#include "mat.h"
#include "mem.h"
#include "math1.h"
#include "sparse.h"
#include "sys.h"
#include "tag.h"
#include "term.h"
#include "tex.h"

int bang() /* bang (x hA --- ) */
/* Putting x into the 1st element of matrix A.  If A is not in the
   catalog or elsewhere on the stack or on the local stack, it will
   be lost when it drops from the stack. */
{
   double x;

   if(!(over() && popd(&x))) return 0;

   if(tos->typ!=MAT) {
      stkerr(" bang: ",MATNOT);
      return 0;
   }
   if((tos->row*tos->col)<1) {
      stkerr(" bang: ",OUTMAT);
      return 0;
   }
   *(tos->mat)=x;
   return(drop2());
}

int bend() /* bend (hA --- hA') */
/* Transposing matrix A. */
{
   register double *A,*Bi;
   double *B;
   register int rA,cA,i,j=0;
   char *name="_bend";

   if (is_sparse(tos)) { /* is sparse */
      return spbend();
   }

   if(is_complex(tos)) {
      return(
         cmplxdbl() &&
         bend() && swap() &&
         bend() && swap() &&
         dblcmplx() &&
         pushq2(name,strlen(name)) && naming()
      );
   }
   if(tos->typ!=MAT) {
      stkerr(" bend: ",MATNOT); return 0;
   }
   A=tos->mat; 
   rA=tos->row;
   cA=tos->col;
   
   if(!matstk(cA,rA,"_bend")) return 0;
   B=tos->mat;

   if(rA==1 || cA==1) {
      memcpy(B,A,rA*cA*sizeof(double));
   }
   else {
      for(;j<cA;j++) {
         Bi=B+j;
         for(i=0;i<rA;i++) {
            *Bi=*A;
            A++;
            Bi+=cA;
         }
      }
   }
   return(lop());
}

int byte() /* byte (qS n --- c) */
/* Fetching nth byte (character value) from string S; S can also be
   an 8-byte number. */
{
   union { 
      double x; 
      unsigned char c[sizeof(double)];
   } u;
   unsigned int c;
   unsigned long n;
   int typ;

   if(!popuint(&n)) return 0;
   n-=XBASE;

   if(tos->typ==VOL) { /* take first string from VOL */
      pushint(XBASE); 
      quote(); 
   }
   if((typ=tos->typ)!=STR && typ!=NUM) {
      stkerr(" byte: ",NUMORSTRNOT);
      return 0;
   }
   if((typ==NUM && n>7) || (typ==STR && !(n<tos->col))) {
      stkerr(" byte: ",OUTCHAR);
      return 0;
   }
   if(typ==STR) c=*((tos->tex)+n);
   else {
      u.x=tos->real;
      c=*(u.c+n);
   }
   return(
      drop() &&
      pushint(c)
   );
}

int bytestore() /* byte! (qS c n --- qS1) */
/* Storing c into string S at nth byte (character value) location; 
   S can also be an 8-byte number. */
{
   union {
      double x;
      unsigned char c[sizeof(double)];
   } u;
   unsigned long c,n;
   int typ;

   if(!popuint(&n)) return 0;
   n-=XBASE;

   if(!popuint(&c)) return 0;

   if(tos->typ==VOL) { /* take first string from VOL */
      pushint(XBASE);
      quote();
   }
   if((typ=tos->typ)!=STR && typ!=NUM) {
      stkerr(" byte!: ",NUMORSTRNOT);
      return 0;
   }
   if((typ==NUM && n>7) || (typ==STR && !(n<tos->col))) {
      stkerr(" byte!: ",OUTCHAR);
      return 0;
   }
   if(typ==STR) *((tos->tex)+n)=c;
   else {
      popd(&u.x);
      *(u.c+n)=c;
      pushd(u.x);
   }
   return(1);
}

int catch() /* catch (hA n | hA hCols --- hB) */
/* From matrix A, fetching a selected column or list of columns; 
   or, from volume A, fetching a selected character column or list 
   of character columns, from all rows. */ 
{
   stkitem *Amat,*Cmat;
   register int acols,bytes,cols,i,j,jmax,k=0,rows;
   register char *A,*B;
   register double *C;
   int isquote,ret=0,TAG=NOTAG;
   char *name="_catch";

   isquote=((tos-1)->typ==STR);

   if(!(
      hand() &&
      swap() &&
      hand() &&
      swap()
   )) return 0;

   if(is_sparse(tos-1)) return(spcatch());

   Cmat=tos;
   if(Cmat->typ!=MAT) {
      stkerr(" catch: ",NUMORMATNOT);
      return 0;
   }
   C=Cmat->mat;
   cols=(Cmat->row*Cmat->col);

   Amat=tos-1;
   TAG=(tos-1)->tag;
   rows=Amat->row;
   jmax=(Amat->col)-1;

   if(Amat->typ==MAT) {
      bytes=sizeof(double)*cols;
      A=(char *)Amat->mat;
   }
   else {
      bytes=cols;
      A=(char *)Amat->tex;
   }
   if((B=(char *)malloc(1+rows*bytes))==NULL) {
      stkerr(" catch: ", MEMNOT);
      return 0;
   }
   switch(Amat->typ) {

      case MAT:
         bytes=rows*sizeof(double);
         for(;k<cols;k++) {
            j=*(C+k)-XBASE;
            if(j>jmax || j<0) {
               stkerr(" catch: ",OUTCOL);
               return 0;
            }
            /* column of terms from matrix stored by columns */
            memcpy((double *)B+locvec(k,rows), \
               (double *)A+locvec(j,rows),bytes);
         }
         return(
            drop2() &&
            push(MAT,(char *)memgetn(name,strlen(name)),TAG,0, \
               (double *)B,NULL,rows,cols,NULL)
         );

      case VOL:
         acols=Amat->col;
         for(;k<cols;k++) {
            j=*(C+k)-XBASE;
            if(j>jmax || j<0) {
               stkerr(" catch: ",OUTCHAR);
               return 0;
            } /* column of characters from volume stored by rows */
            for(i=0;i<rows;i++) {
               *(B+loclin(i,cols)+k)=*(A+loclin(i,acols)+j);
            }
         }
         ret=(
            drop2() &&
            push(VOL,(char *)memgetn(name,strlen(name)),NOTAG,0,NULL, \
               B,rows,bytes,NULL)
         );
         if(isquote) {
            ret=(ret && pushint(XBASE) && quote() &&
                pushq2(name,strlen(name)) && naming()
            );
         }
         return(ret);

      default:
         stkerr(" catch: ",MATORVOLNOT);
         return 0;
      break;
   }
}

int cats() /* cats (qS n --- qS1) */
/* Stringing together n copies of characters S to make S1; if S
   is type VOL, only the first row is used. */
{
   register char *S,*S1;
   register int i=0,len;
   int n;

   if(!popint(&n)) return 0;
   n=MAX(0,n);

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" cats: ",STRORVOLNOT);
      return 0;
   }
   S=(char *)tos->tex;
   len=tos->col;

   if(!strstk(len*n,"_cats")) return 0;
   S1=tos->tex;

   for(;i<n;i++) {
      memcpy(S1,S,len);
      S1+=len;
   }
   return(lop());
}

int chain() /* chain (hA --- hV) */
/* Columns of matrix A strung into a chain. */
{
   register double *A;
   register unsigned int len;

   if(tos->typ!=MAT) {
      stkerr(" chain: ",MATNOT);
      return 0;
   }
   A=tos->mat;
   len=tos->row*tos->col;

   if(!matstk(len,1,"_chain")) return 0;

   memcpy(tos->mat,A,len*sizeof(double));

   return(lop());
}

int chainr() /* chainr (hA --- hV) */
/* Columns of matrix A strung into a chain, in place. */
{
   if(tos->typ!=MAT) {
      stkerr(" chain: ",MATNOT);
      return 0;
   }
   tos->row=(tos->row)*(tos->col);
   tos->col=1;

   return 1;
}

int character() /* character (qS k --- qS1) */
/* Putting the kth character of S on the stack. */
{
   int k;

   if((tos-1)->typ!=STR && (tos-1)->typ!=VOL) {
      stkerr(" character: ",STRORVOLNOT);
      return 0;
   }
   if(!popint(&k)) return 0;
   k-=XBASE;

   if(k>((tos->row*tos->col)-1) || k<0) {
/*
      stkerr(" character: ",OUTCHAR);
      return 0;
*/
   /* Out of bounds.  Push empty string and silently return: */
      return(
         pushq2("",0) &&
         lop()
      );
   }
   return(
      pushq2((tos->tex+k),1) &&
      lop()
   );
}

int chop() /* chop (qS --- qS1) */
/* Chop surrounding blanks from string or volume. */
{
   pushint(xFALSE); /* left */
   return (
      justify() &&
      notrailing()
   );
}

int claw() /* claw (hA hRake --- hA0 hA1) */
/* Separating columns of A according to column vector Rake: A0 holds 
   columns of A where rows of Rake were equal to 0, and A1 holds columns
   of A where rows of Rake were not zero.  Given Rake, function mesh()
   can put A0 and A1 back together again. */
{
   register double *C0,*C1,*Rake;
   register int cols,i=0,k0=0,k1=0;
   char *claw0="_claw0",*claw1="_claw1";
   int TAG=NOTAG;

   if(!(hand() && swap() && hand() && swap())) return 0;

   if(is_sparse(tos-1)) return(spcomb());

   if(tos->typ!=MAT) {
      stkerr(" claw: ",MATNOT);
      return 0;
   }
   if(tos->row!=(tos-1)->col) {
      stkerr(" claw: ",COLSNOTV);
      return 0;
   }
   if((C0=(double *)memget((cols=tos->row),1))==NULL ||
      (C1=(double *)memget(cols,1))==NULL) {
      stkerr(" claw: ",MEMNOT);
      return 0;
   }
   TAG=(tos-1)->tag;

   Rake=tos->mat;

   for(;i<cols;i++) {
      if(*(Rake+i)) {
         *(C1+k1)=i+XBASE;
         k1++;
      }
      else {
         *(C0+k0)=i+XBASE;
         k0++;
      }
   }
   return(
      drop() &&
      dup1s() &&
      push(MAT,NULL,TAG,0,(double *)C0,NULL,k0,1,NULL) &&
      catch() &&
      pushq2(claw0,strlen(claw0)) && naming() &&
      swap() &&
      push(MAT,NULL,TAG,0,(double *)C1,NULL,k1,1,NULL) &&
      catch() &&
      pushq2(claw1,strlen(claw1)) && naming()
   );
}

int clone() /* clone (hA n --- hB) */
/* Forming matrix B by parking n copies of matrix A. */
{
   register int i=0,len,lens;
   register double *A,*B;
   int cols,n,ret=1,rows;
   char *name="_clone";

   if(tos->typ!=NUM) {
      stkerr(" clone: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=MAT && (tos-1)->typ!=VOL && (tos-1)->typ!=NUM && \
      (tos-1)->typ!=STR) {
      stkerr(" clone: ",MATORVOLNOT);
      return 0;
   }
   popint(&n);
   hand();

   if(n<=1) {
      cop(); /* just returning a copy */
      return 1;
   }
   if(tos->typ==VOL) {
      dup1s();
      n--;

      while(i<n && ret) {
         ret=(over() && park());
         i++;
      }
      return(ret && pushq2(name,strlen(name)) && naming() && lop());
   }
   else { /* parking is just too slow; here's a better way for mats: */
      rows=tos->row;
      cols=tos->col;
      len=rows*cols;
      lens=len*sizeof(double);
      A=tos->mat;

      if(!matstk(rows,n*cols,name)) return 0;
      B=tos->mat;
      tos->tag=(tos-1)->tag;

      for(;i<n;i++) {
         memcpy(B,A,lens);
         B+=len;
      }
      return(lop());
   }
}

int compint(const void *i, const void *j)
/* Comparing ints for qsort(). */
{
   return(*(int *)i - *(int *)j);
}

int cram() /* cram (hB hC hA --- ) */
/* Storing all of B within A at columns listed in C. 

   Sizes: A rows and B rows must be equal and A must have more cols
   than B; C rows must equal B cols. */
{
   stkitem *Astk,*Bstk,*Cstk;
   register char *A,*B;
   register double *C;
   register int acols,bytes,colmax,cols,rows;
   register int i=0,j=0,k=0;

   if(!(hand() && rot() && hand() && rot() && hand() && rot())
   ) return 0;

   Astk=tos;
   Bstk=tos-2;
   Cstk=tos-1;

   if(!(Astk->typ==MAT || Astk->typ==VOL)) {
      stkerr(" cram target array: ",ARRAYNOT);
      return 0;
   }
   if(!(Bstk->typ==MAT || Bstk->typ==VOL)) {
      stkerr(" cram source array: ",ARRAYNOT);
      return 0;
   }
   if(Bstk->typ!=Astk->typ) {
      stkerr(" cram source and target: ",MATCHNOT);
      return 0;
   }
   if(Cstk->typ!=MAT) {
      if(Cstk->typ==NUM) {
         swap(); hand(); swap();
      }
      else {
         stkerr(" cram specifier matrix: ",MATNOT);
         return 0;
      }
   }
   if(is_complex(Astk)) {
      rot(); /* (hC hA hB) */
      cmplxmatch();    
      rev(); /* (hB hC hA) */
   }
   else {
      if(is_complex(Bstk)) {
         stkerr(" cram: cannot put complex B into real A","");
         return 0;
      }
   }
   if((rows=Astk->row)!=Bstk->row) {
      stkerr(" cram source and target: ",ROWSNOT);
      return 0;
   }
   if((acols=Astk->col)<(cols=Bstk->col)) {
      stkerr(" cram: fewer target columns than source columns","");
      return 0;
   }
   if(Bstk->col!=Cstk->row) {
      stkerr(" cram: column list not size of source columns","");
      return 0;
   }
   colmax=acols-1; /* max col offset */

   switch(Astk->typ) {

      case MAT:
         if(is_complex(Astk)) {
            rot(); /* (hC hA hB) */
            cmplxmatch();  
            rev(); /* (hB hC hA) */
         }
         else {
            if(is_complex(Bstk)) {
               stkerr(" cram: cannot put complex B into real A","");
               return 0;
            }
         }
         A=(char *)Astk->mat;
         B=(char *)Bstk->mat;
         C=Cstk->mat;
         bytes=rows*sizeof(double);
         for(;k<cols;k++) {
            j=*(C+k)-XBASE;
            if(j>colmax) {
               stkerr(" cram: ",OUTCOL);
               return 0;
            }
         /* Column of terms from matrix stored by columns: */
            memcpy((double *)A+locvec(j,rows), \
               (double *)B+locvec(k,rows),bytes);
         }
         return(drop2() && drop());

      case VOL:
         A=Astk->tex;
         B=Bstk->tex;
         C=Cstk->mat;
         for(;k<cols;k++) {
            j=*(C+k)-XBASE;
            if(j>colmax) {
               stkerr(" cram: ",OUTCHAR);
               return 0;
            } 
         /* Column of characters from volume stored by rows: */
            for(i=0;i<rows;i++) {
               *(A+loclin(i,acols)+j)=*(B+loclin(i,cols)+k);
            }
         }
         return(drop2() && drop());

      default:
         stkerr(" cram: ",MATORVOLNOT);
         return 0;
      break;
   }
}

int crop() /* crop (hT n1 n2 --- hT1) */
/* Fetching from T character columns n1 through n2, inclusive. */
{
   unsigned long n1,n2;

   if(!popuint(&n2) || !popuint(&n1)) {
      stkerr(" crop: ",CROPNOT);
      return 0;
   }
   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" crop: ",STRORVOLNOT);
      return 0;
   }
   return(
      pushint(n1) &&
      pushint(n2-n1+1) &&
      items() &&
      catch()
   );
}

int export1() /* export1 (hA --- hT) */
/* Convert numbers in A into 1-byte integer number patterns in string
   T. */
{
   signed char *T;
   double *A;
   int bytes,k=0;

   if(tos->typ!=MAT && tos->typ!=NUM) {
      stkerr(" export1: ",NUMORMATNOT);
      return 0;
   }
   hand();
   A=tos->mat;
   bytes=(tos->row)*(tos->col);

   if(!strstk(bytes,"_export1")) return 0;
   T=(signed char *)tos->tex;

   for(;k<bytes;k++) {
      *T=*A;
      A++;
      T++;
   }
   return(lop());
}

int export2() /* export2 (hA toendian --- hT) */
/* Convert 8-byte fp numbers in A into 2-byte integer numbers of endian
   type toendian.  Returned volume T holds a matrix of 2-byte integers
   with bytes in toendian byte order.  

   Elements in T follow the same storage order as A--see discussion in
   export().

   Note: two-byte integer range is not checked for overflow. */
{
   int toendian;

   if(!popint(&toendian)) return 0;

   if(tos->typ!=MAT && tos->typ!=NUM) {
      stkerr(" export2: ",NUMORMATNOT);
      return 0;
   }
   hand();
   return(export(2,toendian));
}

int export4() /* export4 (hA toendian --- hT) */
/* Convert 8-byte fp numbers in A into 4-byte integer numbers of endian
   type toendian.  Returned volume T holds a matrix of 4-byte integers
   with bytes in toendian byte order. 

   Elements in T follow the same storage order as A--see discussion in
   export().

   Note: four-byte integer range is not checked for overflow. */
{
   int toendian;

   if(!popint(&toendian)) return 0;

   if(tos->typ!=MAT && tos->typ!=NUM) {
      stkerr(" export4: ",NUMORMATNOT);
      return 0;
   }
   hand();
   return(export(4,toendian));
}

int export8() /* export8 (hA toendian --- hT) */
/* Convert 8-byte fp numbers in A into 8-byte fp numbers of endian type
   toendian.  Returned volume T holds a matrix of 8-byte numbers with
   bytes in toendian byte order.  

   Elements in T follow the same storage order as A--see discussion in
   export().

   To convert volume T into matrix type, use the phrase

      (hT) these rows swap vol2mat chain swap fold (hA)
*/
{
   int k,toendian;

   if(!popint(&toendian)) return 0;

   if(tos->typ!=MAT && tos->typ!=NUM) {
      stkerr(" export8: ",NUMORMATNOT);
      return 0;
   }
   hand();

   k=byteorder(NULL);

   if(k!=toendian) return(export(8,toendian));
   else
      return(
         typmat2vol() &&
         pushq2("_export8",8) &&
         naming()
      );
}

int fetch() /* fetch (hA i j --- x) */
/* Putting the value of A(i,j) on stack.  Bound is checked. */
{
   int cols,i,j,nwords=1,rows;
   double *A,xi,xr;

   if(!(popint(&j) && popint(&i))) return 0;
   i-=XBASE;
   j-=XBASE;

   if(is_sparse(tos)) dense();

   if(tos->typ!=MAT) {
      stkerr(" fetch: ",MATNOT);
      return 0;
   }
   if((rows=tos->row)<1 || (cols=tos->col)<1) {
      stkerr(" fetch: ",MATPURG);
      return 0;
   }
   nwords+=is_complex(tos);
   i*=nwords;

   if(i>=rows || j>=cols || i<0 || j<0) {
      stkerr(" fetch: ",OUTMAT);
      return 0;
   }
   if(nwords==2) {
      xr=*(A=(tos->mat+(i+j*rows)));
      A++;
      xi=*A;
      return(drop() && pushdx(xr,xi));
   }
   xr=*(tos->mat+(i+j*rows));
   return(drop() && pushd(xr));
}

int fold() /* fold (hV r --- hA) */
/* Create matrix A the same size as V but with r rows; use this func-
   tion when terms in V are stored by column (see matrix()). */
{
   int c1,c2,r1,r2,size;

   if(tos->real<1) {
      stkerr(" fold: ","specified row size, r, is less than one");
      return 0;
   }
   if(!popint(&r2)) return 0;

   if(tos->typ!=MAT) {
      stkerr(" fold: ",MATNOT);
      return 0;
   }
   r1=tos->row;
   c1=tos->col;
   size=r1*c1;

   c2=size/r2;

   if(r2*c2!=size) {
      stkerr(" fold: ","terms will be lost");
      return 0;
   }
   if(!matstk(r2,c2,"_fold")) return 0;

   memcpy(tos->mat,(tos-1)->mat,size*sizeof(double));

   return(lop());
}

int foldr() /* foldr (hB r --- hA) */
/* Create matrix A in place of B simply changing the dimensions of B 
   from R-by-C to r-by-r, where c=R*C/r.  Will not work if R*C does 
   not equal r*c. */
{
   int c,C,r,R,size;

   if(tos->real<1) {
      stkerr(" foldr: ","specified row size, r, is less than one");
      return 0;
   }
   if(!popint(&r)) return 0;

   if(tos->typ!=MAT) {
      stkerr(" foldr: ",MATNOT);
      return 0;
   }
   R=tos->row;
   C=tos->col;
   size=R*C;

   c=size/r;
   if(r*c!=size) {
      stkerr(" foldr: ","terms will be lost");
      return 0;
   }
   tos->row=r;
   tos->col=c;

   return 1;
}

int goose() /* goose (d hV n --- ) */
/* Adding d to value of element n of matrix V. */
{
   int len,n;

   if(!popint(&n)) return 0;

   if(tos->typ!=MAT) {
      stkerr(" goose: ",MATNOT);
      return 0;
   }
   n-=XBASE;
   len=(tos->row)*(tos->col);
   if(n>=len || n<0) {
      stkerr(" goose: ",OUTMAT);
      return 0;
   }
   if((tos-1)->typ!=NUM) {
      stkerr(" goose: ",NUMNOT);
      return 0;
   }
   *(tos->mat+n)+=(tos-1)->real;

   return(drop2());
}

int gram_schmidt() /* gram-schmidt (hA --- hG) */
/* Applying the modified Gram-Schmidt orthonormalization procedure to 
   matrix A, where A contains n non-null, linearly independent vectors 
   (columns) of length m, and n is less-than or equal-to m. 

   Reference: http://planetmath.org/encyclopedia/
                     GramSchmidtOrthogonalization.html */
{
   double *A,*Aj,*Ak,D;
   int dj,dk,i,j,k=0,m,n;

   if(tos->typ!=MAT) {
      stkerr(" gram-schmidt: ",MATNOT);
      return 0;
   }
   n=tos->col;
   m=tos->row;

   if(n>m) {
      stkerr(" gram-schmidt: ","number of columns cannot exceed rows");
      return 0;
   }

   cop(); /* this copy of A will become G */
   A=tos->mat; 

   for(;k<n;k++) {
      dk=locvec(k,m); /* offset to column k */
      Ak=A+dk;
      D=0;
      for(i=0;i<m;i++) {
         D+=(*Ak)*(*Ak); /* dot product of column k with itself */
         Ak++;
      }
      Ak=A+dk;
      D=sqrt(D);
      for(i=0;i<m;i++) {
         *(Ak)/=D; /* normalizing column k to unity magnitude */
         Ak++;
      }
      for(j=k+1;j<n;j++) {
         Ak=A+dk;
         dj=locvec(j,m); /* offset to column j */
         Aj=A+dj;
         D=0;
         for(i=0;i<m;i++) {
            D+=(*Ak)*(*Aj); /* dot product of columns k and j */
            Aj++;
            Ak++;
         }
         Ak=A+dk;
         Aj=A+dj;
         for(i=0;i<m;i++) {
            *Aj-=D*(*Ak); /* A(:,j)=A(:,j) - D*A(:,k) */
            Aj++;
            Ak++;
         }
      }
   }
   return(pushq2("_gram-schmidt",13) && naming());
}

int hand() /* hand (x --- hA) or (qS --- hT) */
/* Gives a number or string a handle by converting the number into 
   a 1x1 matrix or the string into 1-row volume. */
{
   int typ;

   if((typ=tos->typ)==NUM) return(typnum2mat());
   if(typ==STR) return(typstr2vol());

   return 1; /* if did nothing its ok */
}

int I3x3() /* I3x3 ( --- hI3x3) */
/* Pushes a fresh 3x3 identity matrix to the stack. */
{
   const int len=9*sizeof(double);
   double *A;

   const char *name="_I3x3";
   const int nlen=strlen(name);
   char *p;

   A=(double *)calloc(1,len);
 
   *A=1;
   *(A+4)=1;
   *(A+8)=1;

   p=malloc(1+nlen);
   memcpy(p,name,1+nlen);

   return(push(MAT,(char *)p,NOTAG,0,A,NULL,3,3,NULL));
}

int import1() /* import1 (hT --- hA) */
/* Convert 1-byte signed integer numbers in T into 8-byte floating
   point numbers in matrix A.  Returned A has one column and as many
   rows as there are bytes in T. */
{
   signed char *T;
   double *A;
   int k=0,rows;

   if(tos->typ!=VOL && tos->typ!=STR) {
      stkerr(" import1: ",STRORVOLNOT);
      return 0;
   }
   T=(signed char *)tos->tex;
   rows=(tos->row)*(tos->col);

   if(!matstk(rows,1,"_import1")) return 0;
   A=tos->mat;

   for(;k<rows;k++) {
      *A=(double)*T;
      A++;
      T++;
   }
   return(lop());
}
   
int import2() /* import2 (hT fromendian --- hA) */
/* Convert 2-byte integer numbers in T, of endian type fromendian, into 
   8-byte fp numbers of the machine's endian type. */
{
   int fromendian;

   if(!popint(&fromendian)) return 0;

   if(tos->typ!=VOL) {
      stkerr(" import2: ",VOLNOT);
      return 0;
   }
   return(import(2,fromendian));
}

int import4() /* import4 (hT fromendian --- hA) */
/* Convert 4-byte integer numbers in T, of endian type fromendian, into 
   8-byte fp numbers of the machine's endian type. */
{
   int fromendian;

   if(!popint(&fromendian)) return 0;

   if(tos->typ!=VOL) {
      stkerr(" import4: ",VOLNOT);
      return 0;
   }
   return(import(4,fromendian));
}

int import4f() /* import4f (hT fromendian --- hA) */
/* Convert 4-byte floating point numbers in T, of endian type 
   fromendian, into 8-byte fp numbers of the machine's endian type. */
{
   int fromendian;

   if(!popint(&fromendian)) return 0;

   if(tos->typ!=VOL) {
      stkerr(" import4f: ",VOLNOT);
      return 0;
   }
   return(import(-4,fromendian));
}

int import8() /* import8 (hT fromendian --- hA) */
/* Convert 8-byte fp numbers in T, of endian type fromendian, into 
   8-byte fp numbers of the machine's endian type. */
{
   int fromendian,k;

   if(!popint(&fromendian)) return 0;

   if(tos->typ!=VOL) {
      stkerr(" import8: ",VOLNOT);
      return 0;
   }
   k=byteorder(NULL);

   if(k!=fromendian) return(import(8,fromendian));
   else
      return(
         typvol2mat() &&
         pushq2("_import8",8) &&
         naming()
      );
}

int inclusive() /* inclusive (hA x1 x2 --- hR) */
/* Column j of R is a rake of zeroes and ones for column j of A, where
   zero rakes values of A that are between x1 and x2 inclusive. */
{
   double y1,y2;
   register double *A,Ai,*R,x1,x2;
   register unsigned int i=0,len;
   int rows,cols;
   
   if(!popd(&y2) || !popd(&y1)) return 0;

   if(tos->typ!=MAT) {
      stkerr(" inclusive: ",MATNOT);
      return 0;
   }
   A=tos->mat;
   rows=tos->row;
   cols=tos->col;
   if((R=(double *)memget(rows,cols))==NULL) {
      stkerr(" inclusive: ",MEMNOT);
      return 0;
   }
   x1=MIN(y1,y2);
   x2=MAX(y1,y2);

   len=rows*cols;
   for(;i<len;i++) {
      if((Ai=*(A+i))>=x1 && Ai<=x2) *(R+i)=0;
      else *(R+i)=1;
   }
   drop();
   return(push(MAT,(char *)memgetn("_inclusive",10),NOTAG,0, \
      R,NULL,rows,cols,NULL));
}

int identity() /* identity (n --- hA) */
/* Putting an n-by-n identitiy matrix on the stack. */
{
   register double *A;
   register int i=0,rows;

   if(!(dup1s() && null())) return 0;

   A=tos->mat;
   rows=tos->row;

   for(;i<rows;i++) *(A+locvec(i,rows)+i)=1;

   return(
      pushq2("_identity",9) && naming()
   );
}

int items() /* items (n0 N --- hList) */
/* Providing a list of N consecutive integer values starting with n0. */
{
   register double *n;
   register int i=0,N,n0;
   int m,M;

/* If n0 is a MAT, take its first number: */
   if((tos-1)->typ==MAT) {
      swap();
      ontop();
      swap();
   }
   if(!(
      popint(&M) &&
      popint(&m)
   )) return 0;

   N=MAX(M,0);
   n0=m;

   if((n=(double *)memget(N,1))==NULL) return 0;

   for(;i<N;i++) *(n+i)=i+n0;

   return(
      push(MAT,(char *)memgetn("_items",6),NOTAG,0,n,NULL,N,1,NULL)
   );
}

int limited() /* limited (hA Amin Amax --- hA1) */
/* Values in A limited to range Amin to Amax. */
{
   double *A,Amax,Amin,temp;
   int i=0,len;

   if(tos->typ!=NUM || (tos-1)->typ!=NUM) {
      stkerr(" limited: ",NUMNOT);
      return 0;
   }
   if(!((tos-2)->typ==MAT || (tos-2)->typ==NUM)) {
      stkerr(" limited: ",NUMORMATNOT);
      return 0;
   }
   popd(&Amax);
   popd(&Amin);

   if(tos->typ==NUM) {
      if(tos->real >= Amax) return(drop() && pushd(Amax));
      if(tos->real <= Amin) return(drop() && pushd(Amin));
      return(cop());
   }
   cop();
   A=tos->mat;
   len=(tos->row)*(tos->col);

   if(Amax<Amin) {
      temp=Amax;
      Amax=Amin;
      Amin=temp;
   }
   for(;i<len;i++) {
      outside(A,Amin,Amax);
      A++;
   }
   return(
      pushq2("_limited",8) && naming()
   );
}

int listfin() /* listn ( ... n --- hA) */
/* Gathers n numbers already on the stack into a column matrix, or n 
   quote-strings already on the stack into a single-row string.

   After count n is popped, the element on top of the stack--which is
   the last in list, either a number or a quote-string--defines the
   character of the list:

      If a number is on top of stack, all elements in the list are in-
      terpreted as numbers in a column matrix, with any strings getting
      a value of 0;

      If the top element is a string, numbers are numbers and strings
      are strings in the text row matrix created (.hex can be used to
      view the list bytes).

   (Tip: an empty (zero-length) quote-string, "",  can be placed at the
   end of a list to force the second type if the last element is not a
   string.)

   All elements in the created list begin on sizeof(double) boundaries.

   At least one blank separates each element when the list is a text
   row, so sizeof(double) blanks will separate each 8-byte number or
   string of eight bytes. */
{
   int complex=0,d;
   const int lWIDTH=80;
   register int i,k=0,len=0;

   double *p,*pi;

   char *listname="_list",*q;

   union {
      double x;
      unsigned char c[sizeof(double)];
   } xc;

   if(!popint(&d)) return 0;

   if(!d) {
      return(pushint(0) && pushint(1) && null());
   }
   if(tos->typ==NUM || tos->typ==MAT || tos->typ==VOL) {
      if((p=memget0(d,1))==NULL) {
         stkerr(" list: ",LISTNOT);
         return 0;
      }
      if((pi=memget0(d,1))==NULL) {
         stkerr(" list: ",LISTNOT);
         return 0;
      }
      i=d;
      while(i) { /* for other than NUM, MAT or VOL, list element is 0 */
         i--;
         switch(tos->typ) {

            case NUM:
               *(p+i)=tos->real;
               if(is_complex(tos)) {
                  *(pi+i)=tos->imag;
                  complex=1;
               }
            break;

            case MAT:
            case VOL:
               matptr(); /* turning handle into ptr number */
               *(p+i)=tos->real;
            break;
         }
         drop();
      }
      if(complex) {
         return(
            pushmat((char *)memgetn("_lr",2),p,d,1) &&
            pushmat((char *)memgetn("_li",2),pi,d,1) &&
            dblcmplx() &&
            pushq2(listname,strlen(listname)) &&
            naming()
         );
      }
      mallfree((void *)&pi);
      return(pushmat((char *)memgetn(listname,strlen(listname)),p,d,1));
   }
   else if(tos->typ==STR) {
      if(!pushq2("",0) || /* always pushing empty quote-string */
         (q=memgetc(1,(1+d)*(MAX(sizeof(double),lWIDTH))))==NULL) {
         stkerr(" list: ",LISTNOT);
         return 0;
      }
      d++; /* +1 for pushq2, above, that fixes bug with long string */
      i=d;
      while(i) {
         i--;
         switch((tos-i)->typ) {

            case STR:
               len=MIN(lWIDTH,(tos-i)->col);
               if(len) memcpy((q+k),(tos-i)->tex,len);
               else { /* skipping zero len string */
                  len=sizeof(double); 
                  len=-len;
               }
            break;

            case MAT:
               pushint(i);
               pick();
               matptr();
               xc.x=tos->real;
               memcpy((q+k),xc.c,len);
               drop();
            break;

            case NUM:
               len=sizeof(double);
               xc.x=(tos-i)->real;
               memcpy((q+k),xc.c,len);
            break;

            default:
               stkerr(" list: ",LISTBAD);
               mallfree((void *)&q);
               return 0;
         }
         k=k+(1+len/sizeof(double))*sizeof(double);
      }
      if((q=realloc(q,k))==NULL) {
         stkerr(" list: ",MEMRESIZE);
         return 0;
      }
      while(d) {
         drop();
         d--;
      }
      return(
         pushq(q,k) &&
         pushq2("_list",5) && naming()
      );
   }
   stkerr(" list: ",LISTNEED);
   return 0;
}

int matrix() /* matrix (hV r --- hA) */
/* Create matrix A the same size as V but with r rows; use this func-
   tion when terms in V are stored by row (see fold()). */
{
   int c1,c2,r1,r2,size;

   if(tos->real<1) {
      stkerr(" matrix: ","specified row size, r, is less than one");
      return 0;
   }
   if(!popint(&r2)) return 0;

   if(tos->typ!=MAT) {
      stkerr(" matrix: ",MATNOT);
      return 0;
   }
   r1=tos->row;
   c1=tos->col;
   size=r1*c1;

   c2=size/r2;

   if(r2*c2!=size) {
      stkerr(" matrix: ","terms will be lost");
      return 0;
   }
   return(
      pushint(c2) && fold() && bend()
   );
}

int mesh() /* mesh (hA0 hA1 hRake --- hA) */
/* Assembling columns of A0 and A1 into A according to column vector
   Rake: A0 will go into A columns corresponding to zeroes in rows of 
   Rake; A1 will go into columns of A where Rake rows are not zero.
   This reverses the job done by claw(). */
{
   stkitem *A0,*A1,*R;
   char *A;
   register double *C0,*C1,*Rake;
   register int col0,col1,cols,i=0,k0=0,k1=0;
   int bytes,ret,rows,TAG;
   char *name="_mesh";

   if(!(
      hand() && rot() && hand() && rot() && hand() && rot()
   )) return 0;

   if(is_sparse(tos-1)) return(spmesh());

   A0=tos-2;
   A1=tos-1;
   R=tos;
   
   if(R->typ!=MAT) {
      stkerr(" mesh column specifier: ",MATNOT);
      return 0;
   }
   if(A0->typ!=A1->typ) {
      stkerr(" mesh matrices to combine: ",MATCHNOT);
      return 0;
   }
   if(!(A0->typ==MAT || A0->typ==VOL)) {
      stkerr(" mesh matrices: ",ARRAYNOT);
      return 0;
   }
   if((rows=A0->row)!=A1->row) {
      stkerr(" mesh source matrices: ",ROWSNOT); 
      return 0;
   }
   if((C0=(double *)memget((col0=A0->col),1))==NULL ||
      (C1=(double *)memget((col1=A1->col),1))==NULL) {
      stkerr(" mesh: ",MEMNOT);
      return 0;
   }
   TAG=(tos-2)->tag;
   Rake=tos->mat;
   cols=R->row;

   for(;i<cols;i++) {
      if(*(Rake+i)) {
         if(k1==col1) {
            stkerr(" mesh one specifier exceeds columns in A1","");
            return 0;
         }
         *(C1+k1)=i+XBASE;
         k1++;
      }
      else {
         if(k0==col0) {
            stkerr(" mesh zero specifier exceeds columns in A0","");
            return 0;
         }
         *(C0+k0)=i+XBASE;
         k0++;
      }
   }
   ret=drop();

   if(k0!=col0) {
      stkerr(" mesh zeroes specify less than A0 columns","");
      ret=0;
   }
   if(k1!=col1) {
      stkerr(" mesh ones specify less than A1 columns","");
      ret=0;
   }
   if(ret==0) return 0;

   if(A0->typ==MAT) bytes=sizeof(double)*cols;
   else bytes=cols;

   if((A=(char *)malloc(1+rows*bytes))==NULL) {
      stkerr(" mesh: ", MEMNOT);
      return 0;
   }
   if(A0->typ==MAT) 
      push(MAT,NULL,TAG,0,(double *)A,NULL,rows,cols,NULL);
   else
      push(VOL,NULL,NOTAG,0,NULL,A,rows,bytes,NULL);
   
   return(
      rot() && 

      push(MAT,NULL,TAG,0,C0,NULL,col0,1,NULL) &&
      other() && cram() && swap() && 

      push(MAT,NULL,TAG,0,C1,NULL,col1,1,NULL) &&
      other() && cram() &&

      pushq2(name,strlen(name)) && naming()
   );
}

int move1() /* move (hB hA --- ) */
/* Move bytes of matrix B into matrix A.  No bounds or type checking. */
{
   register double *A,*B;

   B=(tos-1)->mat;
   A=tos->mat;
   memcpy(A,B,sizeof(double)*((tos-1)->row)*((tos-1)->col));
   return(drop2());
}

int movec() /* movec (hV hA hCols --- ) */
/* Move bytes of vector V into column of matrix A listed in matrix
   Cols.  Rows of V must not be greater than rows of A--not checked.  
   No bounds or type checking. 

   NOTE: word cram works like this word and does bounds checking. */
{
   register double *A,*C,*V;
   int Arows,c,j=0,Vbytes,Vcols,Vrows;

   if(tos->typ==NUM) hand();

   C=tos->mat;

   A=(tos-1)->mat;
   Arows=(tos-1)->row;

   V=(tos-2)->mat;
   Vcols=(tos-2)->col;
   Vrows=(tos-2)->row;
   Vbytes=sizeof(double)*Vrows;

   for(;j<Vcols;j++) {
      c=*(C+j)-XBASE;
      memcpy(A+locvec(c,Arows),V,Vbytes);
      V+=Vrows;
   }
   return(drop2() && drop());
}

int nodupes() /* nodupes (hA --- hA1) */
/* Removing rows that have duplicate values in the first column.  

   Rows remain in original order, keeping the first in each set of 
   duplicates.  Note that the sine extractor relies on keeping the
   first of each duplicate. */
{
   double *A,*A1;
   register int i=1,i0=0,j=0,n;
   int cols,*p,rows;

   if(tos->typ!=MAT) {
      stkerr(" nodupes: ",MATNOT);
      return 0;
   }
   rows=tos->row;

   if(rows<2) {
      if(rows<1) return 1;
      cop();
      return 1;
   }
   A=tos->mat;
   cols=tos->col;

   if((A1=malloc(1+(n=rows*sizeof(double))))==NULL) {
      stkerr(" nodupes: ",MEMNOT);
      return 0;
   }
   memcpy(A1,A,n);
   p=qsort2(A1,rows,1);

   n=0;
   for(i=1;i<rows;i++) {
      while(*(A1+i)==*(A1+i0) && i<rows) {

      /* Taking first duplicate; the sine extractor expects this: */
         *(p+i0)=MIN(*(p+i0),*(p+i));

         *(p+i)=-1;
         i++;
         n++;
      }
      i0=i;
   }
   mallfree((void *)&A1);
   qsort(p,rows,sizeof(int),compint);

   if(!matstk(rows-n,cols,"_nodupes")) return 0;
   A1=tos->mat;

   for(;j<cols;j++) {
      i=n;
      for(;i<rows;i++) {
        *A1=*(A+*(p+i));
         A1++;
      }
      A+=rows;
   }
   mallfree((void *)&p);
   return(lop());
}

int nodupes1() /* nodupes1 (hA --- hA1) */
/* Removing rows that have duplicate values in the first column.  

   Rows remain in original order, keeping the last in each set of 
   duplicates.  Note that word dup_sum relies on keeping the last
   of each duplicate. */
{
   double *A,*A1;
   register int i=1,i0=0,j=0,n;
   int cols,*p,rows;

   if(tos->typ!=MAT) {
      stkerr(" nodupes1: ",MATNOT);
      return 0;
   }
   rows=tos->row;

   if(rows<2) {
      if(rows<1) return 1;
      cop();
      return 1;
   }
   A=tos->mat;
   cols=tos->col;

   if((A1=malloc(1+(n=rows*sizeof(double))))==NULL) {
      stkerr(" nodupes1: ",MEMNOT);
      return 0;
   }
   memcpy(A1,A,n);
   p=qsort2(A1,rows,1);

   n=0;
   for(i=1;i<rows;i++) {
      while(*(A1+i)==*(A1+i0) && i<rows) {

      /* Taking last duplicate; word dup_sum expects this: */
         *(p+i0)=MAX(*(p+i0),*(p+i));

         *(p+i)=-1;
         i++;
         n++;
      }
      i0=i;
   }
   mallfree((void *)&A1);
   qsort(p,rows,sizeof(int),compint);

   if(!matstk(rows-n,cols,"_nodupes1")) return 0;
   A1=tos->mat;

   for(;j<cols;j++) {
      i=n;
      for(;i<rows;i++) {
        *A1=*(A+*(p+i));
         A1++;
      }
      A+=rows;
   }
   mallfree((void *)&p);
   return(lop());
}

int nullc() /* nullc (hA --- hNcols) */
/* Returns column vector with zero at row k if column k of A is null,
   +1 otherwise. */
{
   register double *A,*Aj,*N;
   register int cols,i=0,j=0,rows,zero;

   if(tos->typ!=MAT) {
      stkerr(" nullc: ",MATNOT);
      return 0;
   }
   A=tos->mat;
   rows=tos->row;
   cols=tos->col;

   if((N=(double *)memget0(cols,1))==NULL) return 0;

   for(;j<cols;j++) {
      i=0;
      zero=0;
      Aj=A+locvec(j,rows);

      while(i<rows && !zero) {
         if(*(Aj+i)!=0) {
            *(N+j)=1;
            zero=!zero;
         }
         i++;
      }
   }
   return(
      drop() &&
      push(MAT,(char *)memgetn("_nullc",6),NOTAG,0,N,NULL,cols,1,NULL)
   );
} 

int nullr() /* nullr (hA --- hNrows) */
/* Returns column vector with zero at row k if row k of A is null, +1
   otherwise. */
{
   register double *A,*N;
   register int cols,i=0,i2,j=0,k,nw=1,rows,zero;

   if(tos->typ!=MAT) {
      stkerr(" nullr: ",MATNOT);
      return 0;
   }
   nw+=is_complex(tos);

   A=tos->mat;
   rows=(tos->row)/nw;
   cols=tos->col;
   
   if((N=(double *)memget0(rows,1))==NULL) return 0;
   
   if(nw==1) {
      for(;i<rows;i++) {
         j=0;
         zero=0;
    
         while(j<cols && !zero) {
            if(*(A+locvec(j,rows)+i)!=0) {
               *(N+i)=1;
               zero=!zero;
            }
            j++;
         }
      }
   }
   else { /* A is complex */
      for(;i<rows;i++) {
         i2=i*2;
         j=0;
         zero=0;
   
         while(j<cols && !zero) {
            k=locvec(j,nw*rows);
            if(*(A+k+i2)!=0 ||
               *(A+k+i2+1)!=0) {
               *(N+i)=1;
               zero=!zero;
            }
            j++;
         }
      }
   }
   return(
      drop() &&
      push(MAT,(char *)memgetn("_nullr",6),NOTAG,0,N,NULL,rows,1,NULL)
   );
}

int ontop() /* ontop (hA --- x) */
/* Putting the first element of matrix A on the stack. */
{
   double *A,xi,xr;
   int nw=1;

   if(tos->typ==NUM) return 1;

   if(is_sparse(tos)) dense();

   nw+=is_complex(tos);

   if(tos->typ!=MAT) {
      stkerr(" ontop: ",MATNOT);
      return 0;
   }
   if((tos->row*tos->col)<1) {
      stkerr(" ontop: ",OUTMAT);
      return 0;
   }
   if(nw==2) {
      xr=*(A=(tos->mat));
      A++;
      xi=*A;
      return(drop() && pushdx(xr,xi));
   }
   xr=*tos->mat;
   drop();
   return(push(NUM,NULL,NOTAG,xr,NULL,NULL,0,0,NULL)); 
}

int origin() /* Origin ( --- null3x1) */
/* Pushes a fresh 3x1 null matrix to the stack. */
{
   const int len=3*sizeof(double);
   double *A;

   const char *name="_Origin";
   const int nlen=strlen(name);
   char *p;

   A=(double *)calloc(1,len);

   p=malloc(1+nlen);
   memcpy(p,name,1+nlen);

   return(push(MAT,(char *)p,NOTAG,0,A,NULL,3,1,NULL));
}

int park() /* park (hA hB --- hC) */
/* Matrix C contains A followed, on the right, by B.  Naturally, rows
   of A and B must be equal. */
{
   if(stkdepth()>1) {

      if(tos->typ==MAT && (tos-1)->typ==MAT) return parkm();

      if(is_sparse(tos) || is_sparse(tos-1)) {
         if(!is_sparse(tos)) sparse();

         if(!is_sparse(tos-1)) {
            swap();
            sparse();
            swap();
         }
         return(
            pushint(2) && spparkn()
         );
      }
      if(tos->typ==STR || tos->typ==VOL) return(cat());

      return(parkm());
   }
   stkerr(" park: ",NEEDTWO);
   return 0;
}

int parkm() /* park (hA hB --- hC) */
/* Matrix B is parked on the right of matrix A. */
{
   stkitem *Amat,*Bmat;
   register double *A,*B,*C;
   register int rA,cA,cB,cC;
   int TAG=NOTAG;

   if((tos-1)->typ==NUM) {
      swap();
      hand();
      swap();
   }
   if(tos->typ==NUM) hand();

   if((tos-1)->typ==MAT && tos->typ==MAT) {
      cmplxmatch();
      TAG=tos->tag;

      Amat=tos-1; 
      A=Amat->mat; cA=Amat->col; rA=Amat->row;

      Bmat=tos;
      B=Bmat->mat; cB=Bmat->col;

      if(rA!=Bmat->row) {
         stkerr(" park matrix rows: ",MATSNOTC); return 0;
      }
      if((C=(double *)memget(rA,(cC=cA+cB)))==NULL) return 0;
      memcpy(C,A,rA*cA*sizeof(double));
      memcpy((C+locvec(cA,rA)),B,rA*cB*sizeof(double));
      return(
         drop2() &&
         push(MAT,(char *)memgetn("_park",5),TAG,0,C,NULL,rA,cC,NULL)
      );
   }
   stkerr(" park: ",STKNOTC); return 0;
}

int parkn() /* parkn (hA1 ... hAn n --- hA) */
/* Parking into one the n matrices or volumes on the stack. */
{
   register double *A;
   register char *Ti;
   register int colk,cols,i=0,k=0,rowb,rows;
   int n,nmax,match=0,typ;

   if(!popint(&n)) return 0;
   if(n<2) return 1;

   nmax=stkdepth();

   if(n>nmax) {
      stkerr(" parkn: ",INSUFITM);
      return 0;
   }
/* Checking for sparse: */
   while(!match && k<n) {
      match=is_sparse(tos-k);
      k++;
   }
   if(match) { 
      for(k=0;k<n;k++) {
         pushint(n-1);
         roll();
         if(!is_sparse(tos)) sparse();
      }
      pushint(n);
      return(spparkn());
   }
   if((typ=tos->typ)!=MAT && typ!=VOL && typ!=STR) {
      stkerr(" parkn: ",MATORVOLNOT);
      return 0;
   }
   cols=tos->col;
   rows=tos->row;

   for(k=1;k<n;k++) {
      if((tos-k)->typ!=typ) {
         if(typ==MAT) stkerr(" parkn: ",MATNOT);
         if(typ==VOL && (tos-k)->typ!=STR) stkerr(" parkn: ",VOLNOT);
         if(typ==STR && (tos-k)->typ!=VOL) stkerr(" parkn: ",STRNOT);
         return 0;
      }
      if((tos-k)->row!=rows) {
         stkerr(" parkn: ",STKNOTC);
         return 0;
      }
      cols+=(tos-k)->col;
   }
   if(typ==MAT) { /* matrices are stored by columns: */
      if(!matstk(rows,cols,"_parkn")) return 0;
      A=tos->mat;
      tos->tag=(tos-n)->tag;

      rowb=rows*sizeof(double);
      for(k=-n;k<0;k++) {
         memcpy(A,(tos+k)->mat,rowb*(cols=(tos+k)->col));
         A+=rows*cols;
      }
   }
   else { /* volumes are stored by rows: */
      if(!volstk(rows,cols,"_parkn")) return 0;
      Ti=tos->tex;
      for(;i<rows;i++) {
         for(k=-n;k<0;k++) {
            colk=(tos+k)->col;
            memcpy(Ti,loclin(i,colk)+(tos+k)->tex,colk);
            Ti+=colk;
         }
      }
   }
   for(k=0;k<n;k++) lop();
   return 1;
}

int pile() /* pile (hA hB --- hC) */
/* Matrix C holds A followed, below, by B.  A and B must have the 
   same number of columns. */
{
   if(stkdepth()>1) {

      if(tos->typ==MAT && (tos-1)->typ==MAT) return pilem();

      if(is_sparse(tos) || is_sparse(tos-1)) {
         if(!is_sparse(tos)) {
            hand();  
            sparse();
         }
         if(!is_sparse(tos-1)) {
            swap();
            hand(); 
            sparse();
            swap();
         }
         return(
            pushint(2) && sppilen()
         );
      }
      if(tos->typ==STR || tos->typ==VOL) return(pilev());

      return pilem();
   }
   stkerr(" pile: ",NEEDTWO);
   return 0;
}

int pilem() /* pile (hA hB --- hC) */
/* Matrix A is piled on top of matrix B. */
{
   stkitem *Amat,*Bmat;
   register double *A,*B,*C;
   register int rA,cA,rB,rC,rAlen,rBlen,c0,j=0;
   int TAG=NOTAG;

   if((tos-1)->typ==NUM) {
      swap();
      hand();
      swap();
   }
   if(tos->typ==NUM) hand();
 
   if((tos-1)->typ==MAT && tos->typ==MAT) {
      cmplxmatch();
      TAG=tos->tag;

      Amat=tos-1; 
      A=Amat->mat; rA=Amat->row; cA=Amat->col;

      Bmat=tos;
      B=Bmat->mat; rB=Bmat->row;

      if(cA!=Bmat->col) { 
         stkerr(" pile matrix columns: ",MATSNOTC); return 0;
      }
      if((C=(double *)memget((rC=rA+rB),cA))==NULL) return 0;
      rAlen=rA*sizeof(double); rBlen=rB*sizeof(double);
      for(;j<cA;j++) {
         c0=locvec(j,rC);
         memcpy(C+c0,A+locvec(j,rA),rAlen);
         memcpy(C+c0+rA,B+locvec(j,rB),rBlen);
      }
      return(
         drop2() &&
         push(MAT,(char *)memgetn("_pile",5),TAG,0,C,NULL,rC,cA,NULL)
      );
   }
   stkerr(" pile: ",STKNOTC); 
   return 0;
}

int pilen() /* pilen (hA1 ... hAn n --- hA) */
/* Piling into one the n matrices or volumes on the stack. */
{
   register double *A,*Ak;
   register char *T,*Tk;
   register int i=0,j,k=0,rowb,rowlen,rows,width;
   int cols,match=0;
   double *A0;
   char *T0;
   int n,nmax,typ;

   if(!popint(&n)) return 0;
   if(n<2) return 1;

   nmax=stkdepth();

   if(n>nmax) {
      stkerr(" pilen: ",INSUFITM);
      return 0;
   }
/* Checking for sparse: */
   while(!match && k<n) {
      match=is_sparse(tos-k);
      k++;
   }
   if(match) {
      for(k=0;k<n;k++) {
         pushint(n-1);
         roll();
         if(!is_sparse(tos)) sparse();
      }
      pushint(n);
      return(sppilen());
   }
   if((typ=tos->typ)!=MAT && typ!=VOL && typ!=STR) {
      stkerr(" pilen: ",VOLSTRMATNOT);
      return 0;
   }
   cols=tos->col;
   rows=tos->row;
   width=cols;

   for(k=1;k<n;k++) {
      if((tos-k)->typ!=typ) {
         if(typ==MAT) {
            stkerr(" pilen: ",MATNOT);
            return 0;
         }
         if((typ==VOL || typ==STR) &&
            ((tos-k)->typ!=VOL && (tos-k)->typ!=STR)) {
            stkerr(" pilen: ",VOLNOT);
            return 0;
         }
      }
      if((tos-k)->col!=cols && typ==MAT) {
         stkerr(" pilen: ",STKNOTC);
         return 0;
      }
      rows+=(tos-k)->row;
      width=MAX(width,(tos-k)->col);
   }
   if(typ==MAT) { /* matrices are stored by columns: */
      if(!matstk(rows,cols,"_pilen")) return 0;
      A0=tos->mat;
      tos->tag=(tos-n)->tag;

      for(k=-n;k<0;k++) {
         A=A0;
         Ak=(tos+k)->mat;
         
         rowlen=(tos+k)->row;
         rowb=sizeof(double)*rowlen;

         for(j=0;j<cols;j++) {
            memcpy(A,Ak,rowb);
            A+=rows;
            Ak+=rowlen;
         }
         A0+=rowlen;
      }
   }
   else { /* volumes are stored by rows: */
      if(!volstk(rows,width,"_pilen")) return 0;
      T0=tos->tex;
      memset(T0,' ',rows*width);

      for(k=-n;k<0;k++) {
         T=T0;
         Tk=(tos+k)->tex;

         rowlen=(tos+k)->col;

         for(i=0;i<(tos+k)->row;i++) {
            memcpy(T,Tk,rowlen);
            T+=width;
            Tk+=rowlen;
         }
         T0+=width*(tos+k)->row;
      }
   }
   for(k=0;k<n;k++) lop();
   return 1;
}

int pileof() /* pileof (qS n --- hT) */
/* T holds n copies of string S piled on top of each other. */
{
   int n,ret=1;
   register int bytes,i=0,rows;
   register char *S,*T;

   if(!popint(&n)) return 0;

   if(tos->typ==VOL) { /* if VOL, getting 1st row */
      ret=(pushint(XBASE) && quote());
   }
   if(!ret || tos->typ!=STR) {
      stkerr(" pileof: ",STRORVOLNOT);
      return 0;
   }
   rows=n;
   bytes=tos->col;
   S=tos->tex;

   if((T=(char *)malloc(1+rows*bytes))==NULL) {
      stkerr(" pileof: ", MEMNOT);
      return 0;
   }
   for(;i<rows;i++) {
      memcpy(T+loclin(i,bytes),S,bytes);
   }
   drop();
   return(push(VOL,(char *)memgetn("_pileof",7),NOTAG,0,NULL,T, \
      rows,bytes,NULL));
}

int place() /* place (hB hA r c --- ) */
/* Placement of all of B within A, with B(1,1) located at A(r,c).
   Values of B will override values of A.

   Sizes: A rows starting at r must accomodate all rows of B, and 
   A columns starting at c must accomodate all columns of B. 

   Sun May 22 07:06:31 PDT 2011.  Check for negative rows and cols. */
{
   double *A,*B;
   int Arows,Acols,Brows,Bcols,bytes,c,j=0,r;

   if(!popint(&c)) return 0;
   c-=XBASE;

   if(!popint(&r)) return 0;
   r-=XBASE;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" place: ", MATNOT2);
      return 0;
   }
   if(is_complex(tos)) {
      cmplxmatch();
      r+=r;
   }
   else {
      if(is_complex(tos-1)) {
         stkerr(" place: cannot place complex B into real A","");
         return 0;
      }
   }
   A=tos->mat;
   Arows=tos->row;
   Acols=tos->col;

   B=(tos-1)->mat;
   Brows=(tos-1)->row;
   Bcols=(tos-1)->col;

   if(Arows<0 || Acols<0 || Brows<0 || Bcols<0) {
      gprintf(
         " place: A(%d,%d) B(%d,%d) rows or cols cannot be negative",\
         Arows,Acols,Brows,Bcols);
      nc();
      stkerr("","");
      return 0;
   }
   if(r+Brows>Arows) {
      stkerr(" place: rows of B will overflow rows of A","");
      return 0;
   }
   if(c+Bcols>Acols) {
      stkerr(" place: columns of B will overflow columns of A","");
      return 0;
   }
   A+=locvec(c,Arows)+r;
   bytes=Brows*sizeof(double);

   for(;j<Bcols;j++) {
      memcpy(A,B,bytes);
      A+=Arows;
      B+=Brows;
   }
   return(drop2());
}

int place1() /* place1 (hA hR hC hB op --- hA1) */
/* The parser (prs.c) writes a lot of postfix that uses this function. 

   Place matrix B into matrix A.  If hA is a string, the string is 
   executed to cause A to appear on the stack.

   Integer op defines the operation when B is placed into A:
     op   operation
      0  equate: A=B
      1  add: A+B
      2  subtract: A-B
      3  term-by-term multiply: A*B
      4  term-by-term divide: A/B

   Matrix A1 is created by this function, and original matrix A is 
   unchanged.

   If B is not purged:

      Place all of A into A1 at A1[1,1], and place all of B into A1 at
      the rows of A1 listed in R and the columns of A1 listed in C.  

      Example: Term B[1,4] will be placed at A1[R(1),C(4)].

      The number of terms in R must match the number of rows in B, and
      the number of terms in C must match the number of cols in B.

      If R is purged, all rows of B are placed starting at row 1 of A1;
      if C is purged, all cols of B are placed starting at col 1 of A1.

      New matrix A1 encompasses all possible rows and columns implied
      by A, R, and C, and contains all terms of A starting at A1(1,1),
      and nulls outside the range of A, and terms overwritten by terms 
      of B.

   If B is purged:

      Place into A1 the submatrix of A having rows defined by R and 
      columns defined by C.

      If R is purged, all rows of A are placed starting at row 1 of A1; 
      if C is purged, all cols of A are placed starting at col 1 of A1.

      When B is purged, A can be a VOL and R operates on rows and C
      operates on characters.
*/
{
   char *name="_A1";
   double *A,*A1,*A1j,a,*B,b,*C,G,*R;
   int colA,colA1,colB,rowA,rowA1,rowB,termC,termR;
   int Apurged=0,Atyp,Bpurged=0,Btyp,i,j,k,nwA=1,nwA1=1,nwB=1,op;

   if(!popint(&op)) return 0;
   if(op < 0 || op > 4) {
      stkerr(" place1: ","placement option is out of range");
      return 0;
   }
/* Incoming A can be any of the possible items: MAT, VOL, NUM, STR.
   But if A is a VOL, B must be purged. */

   if(!((tos-1)->typ==NUM || (tos-1)->typ==MAT) ||
      !((tos-2)->typ==NUM || (tos-2)->typ==MAT)) {
      stkerr(" place1: ","expect number or matrix for R and C");
      return 0;
   }
   if(!(tos->typ==NUM || tos->typ==MAT)) {
      stkerr(" place1: ","expect number or matrix for B");
      return 0;
   }
   Btyp=tos->typ;
   hand();
   B=tos->mat; /* A R C B */
   nwB+=is_complex(tos);
   colB=tos->col;
   rowB=tos->row/nwB;
   Bpurged=(rowB==0 || colB==0);
   lpush(); /* B to temp stack; stack: A R C */

   rot(); /* A to tos: R C A */

   Atyp=tos->typ;

   if(Atyp==NUM && Btyp==NUM) { /* do NUMs now */
      lop();
      lop();
      lpull(); /* A B */
      ontop();
      switch(op) {
         case 0: return(lop());
         case 1: return(plusn()); 
         case 2: return(minusn());
         case 3: return(starn());
         case 4: return(slashn()); 
      }
   }
   if(tos->typ==STR) {
      if(!tos->col) {
         drop(); /* empty string means A will be a purged MAT */
         pushstr("purged");
         xmain(0); 
      }
      else { /* string is name of MAT to fetch from library */
         fetchif();
         if(!tos->col) { /* empty STR if not found */
            drop(); /* empty string means A will be a purged MAT */
            pushstr("purged");
            xmain(0); 
         }
         if(!(tos->typ==NUM || tos->typ==MAT)) {
         /* Force a purged MAT for A if the above did not produce MAT
            or NUM on stack, under the assumption that the STR is due
            to a function running an as-yet unresolved MAT name in 
            quotes (see note in exe.c, function _BOOK()): */
            drop();
            pushstr("purged");
            xmain(0); 
         }
      }
   }
   hand(); /* NUM to 1-by-1 */
   if(!Bpurged && tos->typ==VOL) {
      gprintf(" place1: only for MAT A; cannot store into VOL A;"); 
      gprintf(" use functional\n   forms 'ram(B, R, A)'");
      gprintf(" or 'cram(B, R, A)' to store into VOL");
      nc();
      stkerr("","");
      lpull(); /* B off temp stack */

   /* It is important to force a halt here.  A loop can correct itself 
      and run to completion with little indication that there was an 
      error. */
      HALT();

      return 0;
   }
   A=tos->mat;
   nwA+=is_complex(tos); 
   colA=tos->col;
   rowA=tos->row/nwA;
   Apurged=(rowA==0 || colA==0);

   swap(); /* C to tos: R A C */
   hand();
   cop();
   C=tos->mat;
   if(!Bpurged && (!tos->row || !tos->col)) { 
   /* Use all cols of B if C is purged: */
      drop();
      if(!matstk(colB,1,"_C")) {
         lpull(); /* B off temp stack */
         return 0;
      }
      C=tos->mat;
      for(i=0;i<colB;i++) {
         *C=i+XBASE;
         C++;
      }
      C=tos->mat;
   }
   termC=tos->row*tos->col;

   rot(); /* R to tos: A C R */ 
   hand();
   cop();
   R=tos->mat;
   if(!Bpurged && (!tos->row || !tos->col)) { 
   /* Use all rows of B if R is purged: */
      drop();
      if(!matstk(rowB,1,"_R")) {
         lpull(); /* B off temp stack */
         return 0;
      }
      R=tos->mat;
      for(i=0;i<rowB;i++) {
         *R=i+XBASE;
         R++;
      }
      R=tos->mat;
   }
   termR=tos->row*tos->col;
   lpull(); /* B off temp stack and on to tos; stack: A C R B */

   if(!Bpurged) { /* main branch when B is not purged */

   /* Making B complex if A is complex: */
      if(nwA==2 && nwB==1) {
         if(!matstk(nwA*rowB,colB,"_B")) return 0;
         A1=tos->mat; /* using name A1 temporarily */
         for(i=0;i<rowB*colB;i++) {
            *A1=*B;
            A1++;
            *A1=0;
            A1++;
            B++;
         }
         lop(); /* B off stack */
         B=tos->mat; /* new, complex B */
         nwB=2;
      }

   /* Here is where B is stored into regions of A, making A1. */
      
      if(rowB!=termR) {
         if(INFIX) {
            gprintf(\
               " place1: terms in R must match %d rows of RHS B",rowB);
         }
         else {
            gprintf(" place1: terms in R must match %d rows of B",rowB);
         }
         nc();
         stkerr("","");
         return 0;
      }
      if(colB!=termC) {
         if(INFIX) {
            gprintf(\
               " place1: terms in C must match %d cols of RHS B",colB);
         }
         else {
            gprintf(" place1: terms in C must match %d cols of B",colB);
         }
         nc();
         stkerr("","");
         return 0;
      }
      for(i=0;i<termR;i++) *(R+i)-=XBASE; /* into 0-based list */
      for(i=0;i<termC;i++) *(C+i)-=XBASE; /* into 0-based list */

   /* A1 has rows equal to the MAX(rowA, 1+max(R)) (1+max(R) because 
      R is zero-based): */
      rowA1=-1;
      for(i=0;i<termR;i++) rowA1=MAX(rowA1,(int)*(R+i));
      rowA1=MAX(1+rowA1,rowA);

   /* A1 has columns equal the max of MAX(colA, 1+max(C)): */
      colA1=-1;
      for(i=0;i<termC;i++) colA1=MAX(colA1,(int)*(C+i));
      colA1=MAX(1+colA1,colA);
   
   /* Allocated memory for A1 is initialized to zero: */
      nwA1=MAX(nwA,nwB);
      if((A1=calloc(1,(1+nwA1*rowA1*colA1)*sizeof(double)))==NULL) {
         stkerr(" place1: ",MEMNOT);
         return 0;
      }
   /* A1 to tos */
      push(MAT,(char *)memgetn(name,strlen(name)),NOTAG,0,A1,NULL, \
         nwA1*rowA1,colA1,NULL);
      A1=tos->mat;
      if(nwA1==2) set_complex(tos);

   /* Place A into A1 at A1(1,1): */
      if(nwA==nwA1) {
         A1j=A1;
         for(j=0;j<colA;j++) {
            memcpy(A1j,A,nwA*rowA*sizeof(double));
            A+=nwA*rowA;
            A1j+=nwA*rowA1;
         }
      }
      else { /* A is real and A1 is complex */
         A1j=A1;
         for(j=0;j<rowA*colA;j++) {
            *(A1j)=*A;
            A1j++;
            A1j++;
            A++;
         }
      }
   /* Place B according to op into A1 at rows and cols defined by R
      and C: */
      if(nwA1==1) { /* A1 and B are real */
         switch(op) {

            case 0: /* equate */
               for(j=0;j<termC;j++) {
                  A1j=A1+locvec((int)*(C+j),rowA1);
                  for(i=0;i<termR;i++) {
                     *(A1j+(int)*(R+i))=*B;
                     B++;
                  }
               }
            break;

            case 1: /* add */
               for(j=0;j<termC;j++) {
                  A1j=A1+locvec((int)*(C+j),rowA1);
                  for(i=0;i<termR;i++) {
                     *(A1j+(int)*(R+i))+=*B;
                     B++;
                  }
               }
            break;

            case 2: /* subtract */
               for(j=0;j<termC;j++) {
                  A1j=A1+locvec((int)*(C+j),rowA1);
                  for(i=0;i<termR;i++) {
                     *(A1j+(int)*(R+i))-=*B;
                     B++;
                  }
               }
            break;

            case 3: /* multiply */
               for(j=0;j<termC;j++) {
                  A1j=A1+locvec((int)*(C+j),rowA1);
                  for(i=0;i<termR;i++) {
                     *(A1j+(int)*(R+i))*=*B;
                     B++;
                  }
               }
            break;

            case 4: /* divide */
               for(j=0;j<termC;j++) {
                  A1j=A1+locvec((int)*(C+j),rowA1);
                  for(i=0;i<termR;i++) {
                     *(A1j+(int)*(R+i))/=*B;
                     B++;
                  }
               }
            break;
         }
      }
      else { /* A1 and B are complex */
         switch(op) {

            case 0: /* equate */
               for(j=0;j<termC;j++) {
                  A1j=A1+locvec((int)*(C+j),nwA1*rowA1);
                  for(i=0;i<termR;i++) {
                     k=nwA1*(int)*(R+i);
                     *(A1j+k)=*B;
                     B++;
                     k++;
                     *(A1j+k)=*B;
                     B++;
                  }
               }
            break;

            case 1: /* add */
               for(j=0;j<termC;j++) {
                  A1j=A1+locvec((int)*(C+j),nwA1*rowA1);
                  for(i=0;i<termR;i++) {
                     k=nwA1*(int)*(R+i);
                     *(A1j+k)+=*B;
                     B++;
                     k++;
                     *(A1j+k)+=*B;
                     B++;
                  }
               }
            break;

            case 2: /* subtract */
               for(j=0;j<termC;j++) {
                  A1j=A1+locvec((int)*(C+j),nwA1*rowA1);
                  for(i=0;i<termR;i++) {
                     k=nwA1*(int)*(R+i);
                     *(A1j+k)-=*B;
                     B++;
                     k++;
                     *(A1j+k)-=*B;
                     B++;
                  }
               }
            break;

            case 3: /* multiply complex */

            /* Notes from math.c, function starby():

               C = (a+ib)*(c+id) = Cr + iCi
               where
                  Cr = a*c - b*d
                  Ci = b*c + a*d
               and
                  a=*A, b=*(A+1)
                  c=*B, d=*(B+1)

               for(;k<rA*cA;k++) {
                  *(C+k)=*A*(*B) - *(A+1)*(*(B+1)); // Cr at row k
                  k++;
                  *(C+k)=*(A+1)*(*B) + *A*(*(B+1)); // Ci at row k+1
                  A++; A++;
                  B++; B++;
               }
               Make this code do the same thing: */

               for(j=0;j<termC;j++) {
                  A1j=A1+locvec((int)*(C+j),nwA1*rowA1);
                  for(i=0;i<termR;i++) {
                     k=nwA1*(int)*(R+i);
                     a=*(A1j+k);
                     b=*(A1j+k+1);
                     *(A1j+k  )=a*(*B) - b*(*(B+1));
                     *(A1j+k+1)=b*(*B) + a*(*(B+1));
                     B++;
                     B++;
                  }
               }
            break;

            case 4: /* divide complex */

            /* Notes from math.c, function slashby():

               C = (a+ib)/(c+id) = Cr + iCi
               where
                  G = (c+id)*(c-id) = c*c + d*d
                  Cr = (a*c + b*d)/G
                  Ci = (b*c - a*d)/G
               and
                  a=*A, b=*(A+1)
                  c=*B, d=*(B+1)

               for(;k<rA*cA;k++) {
                  G=*B*(*B) + *(B+1)*(*(B+1));
                  *(C+k)=(*A*(*B) + *(A+1)*(*(B+1)))/G; // Cr at row k
                  k++;
                  *(C+k)=(*(A+1)*(*B) - *A*(*(B+1)))/G; // Ci at row k+1
                  A++; A++;
                  B++; B++;
               }
               Make this code do the same thing: */

               for(j=0;j<termC;j++) {
                  A1j=A1+locvec((int)*(C+j),nwA1*rowA1);
                  for(i=0;i<termR;i++) {
                     k=nwA1*(int)*(R+i);
                     G=*B*(*B) + *(B+1)*(*(B+1));
                     a=*(A1j+k);
                     b=*(A1j+k+1);
                     *(A1j+k  )=(a*(*B) + b*(*(B+1)))/G;
                     *(A1j+k+1)=(b*(*B) - a*(*(B+1)))/G;
                     B++;
                     B++;
                  }
               }
            break;
         }
      }
      if(Atyp==NUM && tos->col==1 && tos->row==nwA1) {
         if(is_complex(tos)) pushdx(*tos->mat,*(tos->mat+1));
         else pushd(*tos->mat);
         lop();
      }
      return(
         lpush() && /* push A1 to temp stack */
         drop2() && /* drop B and R */
         drop2() && /* drop C and A */
         lpull()    /* A1 back on stack */
      );
   }
   else { /* Stack: hA hC hR hB */

   /* Here is where columns and rows of A are fetched, making A1. */

      drop(); /* drop purged B */

      if(Apurged) { /* A is purged */
         return( /* clean up the stack and return purged MAT */
            drop() &&
            drop() &&
            drop() &&
            pushstr("purged") &&
            xmain(0)
         );
      }
      if(!termC && !termR) { /* nothing to fetch; return all of A */
         return( /* clean up the stack and return a copy of A */
            drop() &&
            drop() &&
            cop()
         );
      }
      lpush(); /* push hR to local (temp) stack */

      if(termC) catch(); /* selected columns from A */
      else drop();

      if(nwA==2) set_complex(tos); /* A1 type to complex */

      lpull(); /* hR back on stack */

      if(termR) return(reach()); /* rows from A done last */
      else return(drop());
   }
}

int placevec() /* placevec (hA hC --- hB) */
/* From row i of real matrix A, place the term at column C(i), i.e.,
   A(i,C(i)), into B(i) of vector B. */
{
   double *A,*B,*C;
   int cols,i=0,j,rows;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" placevec: ", MATNOT2);
      return 0;
   }
   C=tos->mat;
   rows=tos->row;

   if((tos-1)->row!=rows) {
      stkerr(" placevec: ",MATSNOTC);
      return 0;
   }
   A=(tos-1)->mat;
   cols=(tos-1)->col;

   if(!matstk(rows,1,"_B")) return 0;
   B=tos->mat;

   for(;i<rows;i++) {
      j=*C-XBASE;
      if(j>cols-1 || j<0) {
         stkerr(" placevec: ",OUTMAT);
         return 0;
      }
      *B=*(A+i+locvec(j,rows));
      B++;
      C++;
   }
   return(lop() && lop());
}

int poke() /* poke (x hA n --- ) */
/* Putting x into the nth element of matrix A, A(n).  If A is not in 
   the catalog or elsewhere on the stack or on the local stack, it 
   will be lost when it drops from the stack. */
{
   int n,nwords=1,rows;
   double *A,xi,xr;

   if(!(popint(&n) && swap() && popdx(&xr,&xi))) return 0;
   n-=XBASE;

   if(tos->typ!=MAT) {
      stkerr(" poke: ",MATNOT);
      return 0;
   }
   if((rows=tos->row)<1) {
      stkerr(" poke: ",MATPURG);
      return 0;
   }
   nwords+=is_complex(tos);
   n*=nwords;

   if(n>((tos->row*tos->col)-1) || n<0) {
      stkerr(" poke: ",OUTMAT);
      return 0;
   }
   if(nwords==2) {
      *(A=(tos->mat+n))=xr;
      A++;
      *A=xi;
   }
   else *(tos->mat+n)=xr;

   return(drop());
}

int pry() /* pry (hA n --- x) */
/* Putting the nth element of matrix A on the stack. */
{
   int n,nwords=1,rows;
   double *A,xi,xr;

   if((tos-1)->typ!=MAT) {

      if(is_sparse(tos-1)) {
         swap();
         dense();
         swap();
      }
      else {
         if((tos-1)->typ==VOL) return quote();
          
         else {
            stkerr(" pry: ",MATNOT);
            return 0;
         }
      }
   }
   if(!popint(&n)) return 0;
   n-=XBASE;

   if((rows=tos->row)<1) {
      stkerr(" pry: ",MATPURG);
      return 0;
   }
   nwords+=is_complex(tos);
   n*=nwords;

   if(n>((tos->row*tos->col)-1) || n<0) {
      stkerr(" pry: ",OUTMAT);
      return 0;
   }
   if(nwords==2) {
      xr=*(A=(tos->mat+n));
      A++;
      xi=*A;
      return(drop() && pushdx(xr,xi));
   }
   xr=*(tos->mat+n);
   return(drop() && pushd(xr));
}

int rake() /* rake (hA hRake --- hA0 hA1) */
/* Separating rows of A according to column vector Rake: A0 holds the
   rows of A where rows of Rake equal 0, and A1 holds the rows of A 
   where rows of Rake are not zero.  Function tier() can put A0 and A1
   back together again. */
{
   register double *R0,*R1,*Rake;
   register int i=0,k0=0,k1=0,rows;
   char *rake0="_rake0",*rake1="_rake1";

   if(!(hand() && swap() && hand() && swap())) return 0;

   if(is_sparse(tos-1)) return(sprake());

   if(tos->typ!=MAT) {
      stkerr(" rake: ",MATNOT);
      return 0;
   }
   if(is_complex(tos-1)) {
         
      return(

         lpush() && cmplxdbl() &&

         lpeek() && rake() && rot() &&
         lpull() && rake() && rot() &&


         dblcmplx() && pushq2(rake1,strlen(rake1)) && naming() &&
         lpush() && swap() && 

         dblcmplx() && pushq2(rake0,strlen(rake0)) && naming() &&
         lpull()
      );
   }
   if(tos->row!=(tos-1)->row) {
      stkerr(" rake: ",ROWSNOT);
      return 0;
   }
   if((R0=(double *)memget((rows=tos->row),1))==NULL ||
      (R1=(double *)memget(rows,1))==NULL) {
      stkerr(" rake: ",MEMNOT);
      return 0;
   }
   Rake=tos->mat;

   for(;i<rows;i++) {
      if(*(Rake+i)) {
         *(R1+k1)=i+XBASE;
         k1++;
      }
      else {
         *(R0+k0)=i+XBASE;
         k0++;
      }
   }
   return(
      drop() &&
      dup1s() &&
      push(MAT,NULL,NOTAG,0,(double *)R0,NULL,k0,1,NULL) &&
      reach() &&
      pushq2("_rake0",6) && naming() &&
      swap() &&
      push(MAT,NULL,NOTAG,0,(double *)R1,NULL,k1,1,NULL) &&
      reach() &&
      pushq2("_rake1",6) && naming()
   );
}

int ram() /* ram (hB hC hA --- ) */
/* Storing all of B within A at rows listed in C.
   
   Sizes: A cols and B cols must be equal and A must have more rows
   than B; C rows must equal B rows. */
{  
   stkitem *Astk,*Bstk,*Cstk;
   register char *A,*B;
   register double *C;
   register int arows,brows,cols,dorows,rowmax;
   register unsigned int i=0,j=0,k=0;

   int nw=1;

   if(!(hand() && rot() && hand() && rot() && hand() && rot())
   ) return 0;

   Astk=tos;
   Bstk=tos-2;
   Cstk=tos-1;

   if(!(Astk->typ==MAT || Astk->typ==VOL)) {
      stkerr(" ram target array: ",ARRAYNOT);
      return 0;
   }
   if(!(Bstk->typ==MAT || Bstk->typ==VOL)) {
      stkerr(" ram source array: ",ARRAYNOT);
      return 0;
   }
   if(Bstk->typ!=Astk->typ) {
      stkerr(" ram source and target: ",MATCHNOT);
      return 0;
   }
   if(Cstk->typ!=MAT) {
      if(Cstk->typ==NUM) {
         swap(); hand(); swap();
      }
      else {
         stkerr(" ram specifier matrix: ",MATNOT);
         return 0;
      }
   }
   if((cols=Astk->col)!=Bstk->col) {
      stkerr(" ram source and target: ",COLSNOT);
      return 0;
   }
   if(is_complex(Astk)) {
      rot(); /* (hC hA hB) */
      cmplxmatch();          
      rev(); /* (hB hC hA) */
      nw=2;
   }
   else {
      if(is_complex(Bstk)) {
         stkerr(" ram: cannot put complex B into real A","");
         return 0;
      }
   }
   if((arows=Astk->row)<(brows=Bstk->row)) {
      stkerr(" ram: fewer target rows than source rows","");
      return 0;
   }
   if(Bstk->row!=nw*(dorows=Cstk->row)) {
      stkerr(" ram: row list not size of source rows","");
      return 0;
   }
   rowmax=arows-1; /* max col offset */

   switch(Astk->typ) {

      case MAT:
         A=(char *)Astk->mat;
         B=(char *)Bstk->mat;
         C=Cstk->mat;
         for(;k<dorows;k++) {
            i=nw*(*(C+k)-XBASE);
            if(i>rowmax) {
               stkerr(" ram: ",OUTROW);
               return 0;
            }
         /* Row of terms from matrix stored by columns: */
            for(j=0;j<cols;j++) {
               *((double *)A+i+locvec(j,arows)) \
                  =*((double *)B+nw*k+locvec(j,brows));
            }
         }
         return(drop2() && drop());

      case VOL:
         A=Astk->tex;
         B=Bstk->tex;
         C=Cstk->mat;
         for(;k<dorows;k++) {
            i=*(C+k)-XBASE;
            if(i>rowmax) {
               stkerr(" ram: ",OUTROW);
               return 0;
            } 
         /* Row of characters from volume stored by rows: */
            memcpy(A+loclin(i,cols),B+loclin(k,cols),cols);
         }
         return(drop2() && drop());

      default:
         stkerr(" ram: ",MATORVOLNOT);
         return 0;
      break;
   }
}

int reach() /* reach (hA n | hA hRows --- hB) */
/* Fetching a selected row, or list of rows, from matrix A or volume A.
*/ {
   stkitem *Amat,*Rmat;
   register int arows,bytes,cols,i,imax,j,k=0,rows;
   register char *A,*B;
   register double *R,*R1;
   char *name="_reach";
   int ret;
   static int cflag=1;

   if(!(
      hand() && swap() && hand() && swap())
   ) return 0;

   if(is_sparse(tos-1)) return(spreach());

   if(cflag && is_complex(tos-1)) {
      if(tos->typ==MAT) {

         R=tos->mat;
         rows=tos->row;

         matstk(rows*2,1,"_R");
         R1=tos->mat;

         for(i=0;i<rows;i++) {
           *R1=2*(*R)-XBASE;
           *(R1+1)=(*R1)+1;
           R1++;
           R1++;
           R++;
         }
         cflag=0;

         ret=(lop() && reach());

         set_complex(tos);

         return ret;
      }
      else {
         stkerr(" reach: ",MATNOT);
         return 0;
      }
   }
   cflag=1;

   Rmat=tos;
   if(Rmat->typ!=MAT) {
      stkerr(" reach: ",NUMORMATNOT);
      return 0;
   }
   R=Rmat->mat;
   rows=(Rmat->row*Rmat->col);

   Amat=tos-1;
   cols=Amat->col;
   imax=(Amat->row)-1;

   if(Amat->typ==MAT) {
      bytes=sizeof(double)*cols;
      A=(char *)Amat->mat;
   }
   else {
      bytes=cols;
      A=(char *)Amat->tex;
   }
   if((B=(char *)malloc(1+rows*bytes))==NULL) {
      stkerr(" reach: ", MEMNOT);
      return 0;
   }
   switch(Amat->typ) {

      case MAT:
         arows=Amat->row;
         for(;k<rows;k++) {
            i=*(R+k)-XBASE;
            if(i>imax || i<0) {
               stkerr(" reach: ",OUTROW);
               return 0;
            }
         /* Row of terms from matrix stored by columns: */
            for(j=0;j<cols;j++) {
               *((double *)B+k+locvec(j,rows)) \
                  =*((double *)A+i+locvec(j,arows));
            }
         }
         return(
            drop2() &&
            push(MAT,(char *)memgetn(name,strlen(name)),NOTAG,0, \
               (double *)B,NULL,rows,cols,NULL)
         );
       
      case VOL:
         for(;k<rows;k++) {
            i=*(R+k)-XBASE;
            if(i>imax || i<0) {
               stkerr(" reach: ",OUTROW);
               return 0;
            } 
         /* Row of characters from volume stored by rows */
            memcpy(B+loclin(k,bytes),A+loclin(i,bytes),bytes);
         }
         return(
            drop2() &&
            push(VOL,(char *)memgetn(name,strlen(name)),NOTAG,0,NULL, \
               B,rows,bytes,NULL)
         );

      default:
         stkerr(" reach: ",MATORVOLNOT);
         return 0;
      break;
   }
}

int reach1() /* reach1 (hA hR --- hB) */
/* Sat Sep 28 06:06:28 PDT 2013

   Incoming A is a vector, and incoming R is a matrix that has n 
   columns holding various index vectors for reaching a subset of
   A rows in some order.

   Returned matrix B matches the size of R.

   Column k of returned B matches a reach operation on vector A using 
   column k of R: Bk = reach(A, R[*, k]).

   Example:

      1 10 uniform (hA) dup bend itext neat                      \
      " Vector A: " . . nl (hA)                                  \
      3 5 items reversed (hR1) (R1: A(3):A(7) in reversed order) \
      1 10 5 1 ranint (hR2)    (R2: random set from A)           \
      8 5 1 fill (hR3)         (R3: row 8 of A)                  \
      (hA hR1 hR2 hR3) 3 parkn (hR) dup nl .m nl reach1 nl .m 

      These results are correct by inspection:

         Vector A: 0 1 2 3 4 5 6 7 8 9

                      R1       R2       R3
         Row 1:        7        6        8
         Row 2:        6        4        8
         Row 3:        5        1        8
         Row 4:        4        3        8
         Row 5:        3        9        8

                      B1       B2       B3
         Row 1:        6        5        7
         Row 2:        5        3        7
         Row 3:        4        0        7
         Row 4:        3        2        7
         Row 5:        2        8        7 */
{
   double *A,*B,*R;
   int cols,i,j=0,k,rowsA,rowsR;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" reach1: ",MATNOT2);
      return 0;
   }
   R=tos->mat;
   rowsR=tos->row;
   cols=tos->col;

   A=(tos-1)->mat;
   rowsA=(tos-1)->row;

   if(!matstk(rowsR,cols,"_B")) return 0;
   B=tos->mat;

   for(;j<cols;j++) {
      for(k=0;k<rowsR;k++) {
         i=(int)(*R)-XBASE;
         if(i>rowsA || i<0) {
            stkerr(" reach1: ",OUTROW);
            return 0;
         }
         *B=*(A+i);
         B++;
         R++;
      }
   }
   return(lop() && lop());
}

int reach2() /* reach2 (hA hR --- hB) */
/* Sat Sep 28 09:47:12 PDT 2013

   Each column of incomng R holds a vector of indices for reaching a
   subset of A rows of the corresponding column of A in some order.

   Returned matrix B matches the size of R.

   Column k of returned B matches a reach operation on column k of A
   using column k of R: Bk = reach(A[*, k], R[*, k]).

   Example:

      1 9 uniform (hA) 3 clone (hA) dup .m nl (hA)               \
      3 5 items reversed (hR1) (R1: A(3):A(7) in reversed order) \
      1 10 5 1 ranint (hR2)    (R2: random set from A)           \
      8 5 1 fill (hR3)         (R3: row 8 of A)                  \
      (hA hR1 hR2 hR3) 3 parkn (hR) dup nl .m nl reach2 nl .m

      These results are correct by inspection:

                      A1       A2       A3
         Row 1:        0        0        0
         Row 2:        1        1        1
         Row 3:        2        2        2
         Row 4:        3        3        3
         Row 5:        4        4        4
         Row 6:        5        5        5
         Row 7:        6        6        6
         Row 8:        7        7        7
         Row 9:        8        8        8

                      R1       R2       R3
         Row 1:        7        6        8
         Row 2:        6        4        8
         Row 3:        5        1        8
         Row 4:        4        3        8
         Row 5:        3        9        8

                      B1       B2       B3
         Row 1:        6        5        7
         Row 2:        5        3        7
         Row 3:        4        0        7
         Row 4:        3        2        7
         Row 5:        2        8        7 */
{
   double *A,*B,*R,*X;
   int cols,i,j=0,k,rowsA,rowsR;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" reach2: ",MATNOT2);
      return 0;
   }
   R=tos->mat;
   rowsR=tos->row;
   cols=tos->col;

   A=(tos-1)->mat;
   rowsA=(tos-1)->row;

   if((tos-1)->col!=cols) {
      gprintf(" reach2: columns of incoming A and R do not match");
      stkerr("","");
      nc();
      return 0;
   }
   if(!matstk(rowsR,cols,"_B")) return 0;
   B=tos->mat;

   for(;j<cols;j++) {
      X=A+j*rowsA;
      for(k=0;k<rowsR;k++) {
         i=(int)(*R)-XBASE;
         if(i>rowsA || i<0) {
            stkerr(" reach2: ",OUTROW);
            return 0;
         }
         *B=*(X+i);
         B++;
         R++;
      }
   }
   return(lop() && lop());
}

int repeat1() /* repeat (hA n --- hB) */
/* Forming matrix B by piling n copies of A. */
{
   register int i=0,j=0;
   register double *A,*B;
   int cols,n,ret=1,rows;
   char *name="_repeat";

   if((tos-1)->typ==STR || ((tos-1)->typ==VOL && (tos-1)->row==1)) 
      return(pileof()); /* this is faster for STR */
   
   if(tos->typ!=NUM) {
      stkerr(" repeat: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=MAT && (tos-1)->typ!=VOL && (tos-1)->typ!=NUM) {
      stkerr(" repeat: ",MATORVOLNOT);
      return 0;
   }
   popint(&n);
   hand();

   if(n<=1) {
      cop(); /* just returning a copy */
      return 1;
   }
   if(tos->typ==VOL) {
      dup1s();
      n--;

      while(i<n && ret) {
         ret=(over() && pile());
         i++;
      }
      return(ret && pushq2(name,strlen(name)) && naming() && lop());
   }
   else { /* piling is ok for text, but here's a faster way for mats: */
      rows=tos->row;
      cols=tos->col;

      if(rows<=0 || cols<=0) { /* matrix is purged */
         stkerr(" repeat: ","incoming matrix is purged");
         HALT();
         return 0;
      }
      A=tos->mat;

      if(!matstk(n*rows,cols,name)) return 0;
      B=tos->mat;
      tos->tag=(tos-1)->tag;

      for(;j<cols;j++) {
         for(i=0;i<n;i++) {
            memcpy(B,A,rows*sizeof(double));
            B+=rows;
         }
         A+=rows;
      }
      return(lop());
   }
}

int replace() /* replace (hA hOff d --- hB) */
/* Replacing terms in A at offsets listed in Off with value d. */
{
   double d1;
   int cA,rA;
   register double *B,d,*Off;
   register int i=0,k,kmax,rOff;

   if(!(popd(&d1) && hand() && swap() && hand() && swap())) return 0;

   if(stkset(" replace: ")!=MAT) {
      stkerr(" replace: ",MATNOT2);
      return 0;
   }
   if((B=(double *)memget((rA=(tos-1)->row),(cA=(tos-1)->col)))==NULL) {
      stkerr(" replace: ",MEMNOT); 
      return 0;
   }
   memcpy(B,(tos-1)->mat,rA*cA*sizeof(double));

   kmax=rA*cA-1;
   d=d1;

   Off=tos->mat;
   rOff=tos->row;

   for(;i<rOff;i++) {
      k=*(Off+i)-XBASE;
      if(k>kmax) {
         stkerr(" replace: ",OUTMAT);
         return 0;
      }
      *(B+k)=d;
   }
   return(
      drop2() &&
      push(MAT,(char *)memgetn("_replace",8),NOTAG,0, \
         (double *)B,NULL,rA,cA,NULL)
   );
}

int reversed() /* reversed (hA --- hB) */
/* Reversing the order of rows in matrix or volume A. */
{
   register double *Ai,*B,*Bk;
   int cols,rows;
   register int i,j=0;

   if(!hand()) return 0;

   if(tos->typ==VOL) return(reversed1());
   
   rows=tos->row;
   cols=tos->col;
   Ai=tos->mat;

   if(!matstk(rows,cols,"_reversed")) return 0;
   B=tos->mat;
   tos->tag=(tos-1)->tag;

   if(is_complex(tos)) {
      for(;j<cols;j++) {
         Bk=B+locvec((1+j),rows);
         for(i=0;i<rows;i+=2) {
            Bk--;
            Ai++;
            *Bk=*Ai;

            Ai--;
            Bk--;
            *Bk=*Ai;

            Ai++;
            Ai++;
         }
      }
   }
   else {
      for(;j<cols;j++) {
         Bk=B+locvec((1+j),rows);
         for(i=0;i<rows;i++) {
            Bk--;
            *Bk=*Ai;
            Ai++;
         }
      }
   }
   return(lop());
}

int reversed1() /* (hA --- hB) */
/* Reversing the order of rows in matrix or volume A. */
{
   register int i=0,k,N;
   register double *R;

   if(!hand()) return 0;

   N=tos->row;

   if((R=(double *)memget(N,1))==NULL) return 0;

   k=N-1+XBASE;
   for(;i<N;i++) {
      *(R+i)=k;
      k--;
   }
   return(
      push(MAT,NULL,NOTAG,0,R,NULL,N,1,NULL) &&
      reach() &&
      pushq2("_reversed",9) && naming()
   );
}

int sling() /* sling (hV --- hRake) */
/* Rake for last of zero or more consecutive duplicates in V.

   Rake has 1 at location of last in series of zero or more equal
   values in V, zeroes elsewhere; example:
         list: 12 12 11 11 15 15 17 20 20 ; this sling park .i

   Designed for use with data evolving in time, ending 20 20 are not 
   considered consecutive duplicates--upcoming data is needed.

   To force ending duplicates to be considered, pile a different
   value on the end, like:
      list: 12 12 11 11 15 15 17 20 20 ; INF pile this sling park .i
*/
{
   register double *V,*R;
   register int i=0;
   int rows;
   
   if(tos->typ!=MAT) {
      stkerr(" sling: ",MATNOT);
      return 0;
   }
   rows=tos->row;
   V=tos->mat;

   if(!matstk(rows,1,"_sling")) return 0;
   R=tos->mat;
   memset(R,0,rows*sizeof(double));

   for(;i<rows-1;i++) if(*(V+i+1)!=*(V+i)) *(R+i)=1;
   return(lop());
}

int sling1() /* sling1 (hV --- hRake) */
/* Rake for first of zero or more consecutive duplicates in V.

   Rake has 1 at location of first in series of zero or more equal
   values in V, zeroes elsewhere; first value in Rake is always 1.
   Example:
         list: 11 12 12 11 11 15 15 17 12 12 ; this sling1 park .i
*/
{
   register double *V,*R;
   register int i=1;
   int rows;
  
   if(tos->typ!=MAT) {
      stkerr(" sling1: ",MATNOT);
      return 0;
   }
   rows=tos->row;
   V=tos->mat;

   if(!matstk(rows,1,"_sling1")) return 0;
   R=tos->mat;
   memset(R,0,rows*sizeof(double));
   *R=1;

   for(;i<rows;i++) if(*(V+i-1)!=*(V+i)) *(R+i)=1;
   return(lop());
}

int sq1() /* sq (hV --- hA) vector V into square, diagonal A */
{
   register double *A,*V;
   register int i=0;
   int cols,nw=1,rows;
 
   if(tos->typ!=MAT) {
      stkerr(" sq: ",MATNOT);
      return 0;
   }
   V=tos->mat;
   rows=tos->row;

   if(is_complex(tos)) nw=2;
   cols=rows/nw;

   if(!matstk(rows,cols,"_sq")) return 0;
   A=tos->mat;
   tos->tag=(tos-1)->tag;

   memset(A,0,rows*cols*sizeof(double));
   rows++;

   if(nw==2) {
      for(;i<cols;i++) {
         *A=*V;
         V++;
         A++;
         *A=*V;
         V++;
         A+=rows;
      }
   }
   else {
      for(;i<cols;i++) {
         *A=*V;
         V++;
         A+=rows;
      }
   }
   return(lop());
}

int store() /* store (x hA i j --- ) */
/* Putting the value x into A(i,j).  Bound is checked. */
{
   int cols,i,j,nwords=1,rows;
   double *A,xi,xr;

   if(!(popint(&j) && popint(&i) && swap() && popdx(&xr,&xi))) return 0;
   i-=XBASE;
   j-=XBASE;

   if(tos->typ!=MAT) {
      stkerr(" store: ",MATNOT);
      return 0;
   }
   if((rows=tos->row)<1 || (cols=tos->col)<1) {
      stkerr(" store: ",MATPURG);
      return 0;
   }
   nwords+=is_complex(tos);
   i*=nwords;

   if(i>=rows || j>=cols || i<0 || j<0) {
      stkerr(" store: ",OUTMAT);
      return 0;
   }
   if(nwords==2) {
      *(A=(tos->mat+(i+j*rows)))=xr;
      A++;
      *A=xi;
   }
   else *(tos->mat+(i+j*rows))=xr;

   return(drop()); 
}

int stretch() /* stretch (hA r --- hB) */
/* Mon Jul 22 07:47:20 PDT 2013

   Stretch matrix A to number of rows equal to r, where r>rows(A).

   Test case: Paste the following four lines at the ready prompt:
      >> \
      A = [0:5, [0.0657; 0.8089; 0.2681; 0.0709; 0.3328; 0.5098]]; \
      B = stretch(A, 11); \
      << A .m nl nl B .m nl

         [tops@kaffia] ready > 

          Row 1:        0   0.0657
          Row 2:        1   0.8089
          Row 3:        2   0.2681
          Row 4:        3   0.0709
          Row 5:        4   0.3328
          Row 6:        5   0.5098
 
           Row 1:        0   0.0657
           Row 2:      0.5   0.4373
           Row 3:        1   0.8089
           Row 4:      1.5   0.5385
           Row 5:        2   0.2681
           Row 6:      2.5   0.1695
           Row 7:        3   0.0709
           Row 8:      3.5   0.2019
           Row 9:        4   0.3328
          Row 10:      4.5   0.4213
          Row 11:        5   0.5098

         [tops@kaffia] ready >

   The original 6-by-2 matrix has been stretched to an 11-by-2. 
   Results are correct by inspection. */
{
   double *A,*B,dx,*tA;
   int cols,j=0,k=0,r,rows;

   if((tos-1)->typ!=MAT) {
      stkerr(" stretch: ",MATNOT);
      return 0;
   }
   if(!popint(&r)) return 0;

   A=tos->mat;
   rows=tos->row;
   cols=tos->col;

   if(r<rows) {
      gprintf(" stretch: r %d is less than %d rows of A",r,rows);
      stkerr("","");
      nc();
      return 0;
   }
   dx=(double)(rows-1)/(double)(r-1); /* stretch */

   if(!matstk(rows,1,"_tA")) return 0;
   tA=tos->mat;
   for(;k<rows;k++) *(tA+k)=k;

   if(!matstk(r,cols,"_B")) return 0;
   B=tos->mat;

   for(;j<cols;j++) {
      for(k=0;k<r;k++) lerp(INTERP,tA,A,rows,k*dx,(B+k));
      A+=rows;
      B+=r;
   }
   return(lop() && lop());
}

void stretch4(float *A, double *B, int len)
/* Stretch floating point numbers from 4 to 8 bytes. */
{
   register int i;
   for(i=len-1;i>-1;i--) *(B+i)=*(A+i);
}

int sym_unpack() /* sym_unpack (hV --- hA) */
/* From chain of real numbers in vector V containing the lower triangle
   of symmetric c-by-c matrix A, reconstruct the complete matrix A.

   V contains all terms in the lower triangular portion of A.  Every
   column of A is strung one after another starting at the diagonal. */
{
   int c,i,k=0,n;
   double *A,*A0,*V;
   const double eps=0.05;

   if(tos->typ!=MAT) {
      stkerr(" sym_unpack: ",MATNOT);
      return 0;
   }
   V=tos->mat;
   n=tos->row;
   c=(int)(eps + pow((0.25 + n + n),0.5) - 0.5);

   if(c*(c+1)/2 != n) {
      stkerr(" sym_unpack: ",\
         "number of V rows is invalid for a symmetric lower triangle");
      return 0;
   }
   if(!matstk(c,c,"_Asym")) return 0;
   A=tos->mat;
   A0=A;

   i=c+1;
   n=c;
/* Create the lower triangle of A, including the diagonal: */
   for(;k<c;k++) {
      memcpy(A,V,n*sizeof(double));
      A+=i;
      V+=n;
      n--;
   }
/* Fill the upper triangle of A, above the diagonal: */
   for(k=1;k<c;k++) {
      A=A0+locvec(k,c); /* top of column k */
      for(i=0;i<k;i++) {
         *A=*(A0+locvec(i,c)+k);
         A++;
      }
   }
   return(lop());
}

int thereat() /* thereat (hA hX --- hRake) */
/* Rake for rows of A where A(i,j) = X(j) for all columns j, i.e.,
   rows of A that match vector X. */
{
   register double *A,*R,*X;
   register int i=0,j,ok;
   int cols,rows;

   hand();

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" thereat: ",MATNOT2);
      return 0;
   }
   if((cols=tos->row)!=(tos-1)->col) {
     stkerr(" thereat: vector length does not match matrix columns","");
     return 0;
   }
   rows=(tos-1)->row;
   A=(tos-1)->mat;
   X=tos->mat;

   if(!matstk(rows,1,"_0_at_matched_rows")) return 0;
   R=tos->mat;

   for(;i<rows;i++) {
      ok=1;
      j=0;
      while(ok && j<cols) {
         if(*(A+locvec(j,rows)+i)!=*(X+j)) ok=0;
         j++;
      }
      if(ok) *(R+i)=0;
      else *(R+i)=1;
   }
   return(lop() && lop());
}

int thrulist() /* thrulist (n1 nK --- hN) or : (n1 nK --- hN) */
/* List of consecutive integer values from n1 to nK. */
{
   double *N;
   int i,L,n1,nK;

/* If n1 or nK is a MAT, take its first number: */
   if(tos->typ==MAT) ontop();
   if((tos-1)->typ==MAT) {
      swap();
      ontop();
      swap();
   }
   if(!(
      popint(&nK) &&
      popint(&n1)
   )) return 0;

   if(n1>nK) {
      return(
         pushstr("MAT tpurged") &&
         xmain(0)
      );
   }
   if(!matstk((L=nK-n1+1),1,"_thrulist")) return 0;
   N=tos->mat;

   for(i=n1;i<L+n1;i++) {
      *N=i;
      N++;
   }
   return 1;
}

int tier() /* tier (hA0 hA1 hRake --- hA) */
/* Assembling rows of A0 and A1 into A according to column vector
   Rake: A0 will go into A rows correspondng to zeroes in rows of
   Rake; A1 will go into rows of A where Rake rows are not zero.
   This reverses the job done by rake(). */
{
   stkitem *A0,*A1,*R;
   char *A;
   register double *R0,*R1,*Rake;
   register int i=0,k0=0,k1=0,row0,row1,rows;
   int bytes,ret,cols;
   char *name="_tier";

   if(!(
      hand() && rot() && hand() && rot() && hand() && rot()
   )) return 0;

   if(is_sparse(tos-1)) 
   return(

      rot() && spbend() &&
      rot() && spbend() &&
      rot() &&
      spmesh() &&
      spbend() &&

      pushq2(name,strlen(name)) && naming()
   );

   if(is_complex(tos-1)) {
 
      return(

         lpush() &&
         cmplxdbl() && rot() &&
         cmplxdbl() && rot() &&

         lpeek() && tier() && rev() && swap() &&
         lpull() && tier() && swap() && dblcmplx() &&

         pushq2(name,strlen(name)) && naming()
      );
   }

   A0=tos-2;
   A1=tos-1;
   R=tos;

   if(R->typ!=MAT) {
      stkerr(" tier row specifier: ",MATNOT);
      return 0;
   }
   if(A0->typ!=A1->typ) {
      stkerr(" tier matrices to combine: ",MATCHNOT);
      return 0;
   }
   if(!(A0->typ==MAT || A0->typ==VOL)) {
      stkerr(" tier matrices: ",ARRAYNOT);
      return 0;
   }
   if((cols=A0->col)!=A1->col) {
      stkerr(" tier source matrices: ",COLSNOT);
      return 0;
   }
   if((R0=(double *)memget((row0=A0->row),1))==NULL ||
      (R1=(double *)memget((row1=A1->row),1))==NULL) {
      stkerr(" tier: ",MEMNOT);
      return 0;
   }
   Rake=tos->mat;
   rows=R->row;

   for(;i<rows;i++) {
      if(*(Rake+i)) {
         if(k1==row1) {
            stkerr(" tier one specifier exceeds rows in A1","");
            return 0;
         }
         *(R1+k1)=i+XBASE;
         k1++;
      }
      else {
         if(k0==row0) {
            stkerr(" tier zero specifier exceeds rows in A0","");
            return 0;
         }
         *(R0+k0)=i+XBASE;
         k0++;
      }
   }
   ret=drop();

   if(k0!=row0) {
      stkerr(" tier zeroes specify less than A0 rows","");
      ret=0;
   }
   if(k1!=row1) {
      stkerr(" tier ones specify less than A1 rows","");
      ret=0;
   }
   if(ret==0) return 0;

   if(A0->typ==MAT) bytes=sizeof(double)*cols;
   else bytes=cols;

   if((A=(char *)malloc(1+rows*bytes))==NULL) {
      stkerr(" tier: ", MEMNOT);
      return 0;
   }
   if(A0->typ==MAT)
      push(MAT,NULL,NOTAG,0,(double *)A,NULL,rows,cols,NULL);
   else
      push(VOL,NULL,NOTAG,0,NULL,A,rows,bytes,NULL);

   return(
      rot() &&
      push(MAT,NULL,NOTAG,0,R0,NULL,row0,1,NULL) &&
      pushint(2) && pick() &&
      ram() &&
      swap() &&
      push(MAT,NULL,NOTAG,0,R1,NULL,row1,1,NULL) &&
      pushint(2) && pick() &&
      ram() &&
      pushq2(name,strlen(name)) && naming()
   );
}

int tovector() /* diag (hA --- hV) diagonal of A into vector */
{
   register double *A,*V;
   register int i=0;
   int *col_idx_V = 0;
   int cols,nw=1,rows, indexed = 0;
 
   if(is_sparse(tos)) return(spdiag());

   if(tos->typ!=MAT) {
      stkerr(" diag: ",MATNOT);
      return 0;
   }
   rows=tos->row;
   cols=tos->col;

   if(is_complex(tos)) nw=2;
   if(is_indexed(tos)) indexed = 1;

   if(rows!=nw*cols) {
      stkerr(" diag: ",SQUNOT);
      return 0;
   }
   A=tos->mat;

   if (indexed) {
       if(!matstk_idx(rows,1,"_diag")) return 0;
   } else {
       if(!matstk(    rows,1,"_diag")) return 0;
   }
   V=tos->mat;
   tos->tag=(tos-1)->tag;
   if (indexed) {
      memcpy(MAT_ROW_IDX(tos), MAT_ROW_IDX(tos-1), (rows/nw)*sizeof(int));
      col_idx_V = MAT_COL_IDX(tos);
      col_idx_V[0] = 0;
   }

   memset(V,0,rows*sizeof(double));
   rows++;
   
   if(nw==2) {
      for(;i<cols;i++) {
         *V=*A;
         V++;
         A++;
         *V=*A;
         V++;
         A+=rows;
      }
   }
   else {
      for(;i<cols;i++) {
         *V=*A;
         V++;
         A+=rows;
      }
   }
   return(lop());
}

int tril()       /* tril (hA n --- hL) {{{1 */
/* 
 * man entry:  tril {{{2
 * (hA n --- hL) Stack has a matrix and an integer n where n < nRows.  Returns a matrix L containing the lower triangle of A beginning n terms below the diagonal. If n = 0, L will contain A's diagonal (and everything below it).

use> 2 seedset 3 5 random (X) 1 tril .m nl
produces this matrix
 [      0        0        0        0        0 ]
 [ 0.2631        0        0        0        0 ]
 [ 0.5112   0.4379        0        0        0 ]
 * category: math::matrix::partitioning
 * related:  triu, trilp, triup
 * 2}}}
 */
{
    char   *name = "_tril";
    int     n, r, c, nRows, nCols;
    double *A, *L;

    if (!popint(&n)) return 0;
    if (tos->typ != MAT) {
        stkerr(" tril: ",MATNOT);
        return 0;
    }
    if (is_complex(tos)) {
        return(
            cmplxdbl() &&
            pushint(n) && tril() && swap() &&
            pushint(n) && tril() && swap() &&
            dblcmplx() &&
            pushq2(name, strlen(name)) && naming()
        );
    }
    nRows = tos->row;
    nCols = tos->col;
    A     = tos->mat;
    if (n >= nRows) {
        stkerr(" tril: ", OUTMAT);
        return 0;
    }

    /* allocate a nRows x nCols dense matrix and put it on the stack */
    if (!matstk(nRows, nCols, name)) return 0;
    L = tos->mat;
    for (c = 0; c < nCols; c++) {
        for (r = 0; r < nRows; r++) {
            if ((r-n) >= c) { /* am at or below requested diagonal slice */
                L[r+c*nRows] = A[r+c*nRows];
            } else {
                L[r+c*nRows] = (double) 0.0;
            }
        }
    }

    lop(); /* drop A from stack */
    return 1;
} /* 1}}} */

int trilp() /* trilp (hA --- hL) */
/*  Store symmetric, real N-by-N matrix A in lower-triangular packed 
    mode, used by the scientific software library.  Each column, from 
    the diagonal and below, is continuous in returned L.  N cells 
    (initialized to 0) follow L for optional use by library functions. 
*/
{
   double *A,*L;
   int i,j=0,rows,size;

   if(tos->typ!=MAT) {
      stkerr(" trilp: ",MATNOT);
      return 0;
   }
   if((rows=tos->row)!=tos->col) {
      stkerr(" trilp: ",SQUNOT);
      return 0;
   }
   A=tos->mat;

   size=rows*(1+rows)/2 + rows;
   if(!matstk(size,1,"_Atrilp")) return 0;
   L=tos->mat;

   for(;j<rows;j++) {
      for(i=j;i<rows;i++) {
         *L=*(A+i+locvec(j,rows));
         L++;
      }
   }
   for(i=0;i<rows;i++) {
      *L=0;
      L++;
   }
   return(lop());
}

int triu()       /* triu (hA n --- hU) {{{1 */
/* 
 * man entry:  triu {{{2
 * (hA n --- hU) Stack has a matrix and an integer n where n < nCols.  Returns a matrix U containing the upper triangle of A beginning n terms above the diagonal. If n = 0, U will contain A's diagonal (and everything above it).

use> 2 seedset 4 3 random (X) 1 triu .m nl
produces this matrix
 [ 0  0.06553   0.3586 ]
 [ 0        0   0.8694 ]
 [ 0        0        0 ]
 [ 0        0        0 ]
 * category: math::matrix::partitioning
 * related:  tril, trilp, triup
 * 2}}}
 */
{
    char   *name = "_triu";
    int     n, r, c, nRows, nCols;
    double *A, *U;

    if (!popint(&n)) return 0;
    if (tos->typ != MAT) {
        stkerr(" triu: ",MATNOT);
        return 0;
    }
    if (is_complex(tos)) {
        return(
            cmplxdbl() &&
            pushint(n) && triu() && swap() &&
            pushint(n) && triu() && swap() &&
            dblcmplx() &&
            pushq2(name, strlen(name)) && naming()
        );
    }
    nRows = tos->row;
    nCols = tos->col;
    A     = tos->mat;
    if (n >= nCols) {
        stkerr(" triu: ", OUTMAT);
        return 0;
    }

    /* allocate a nRows x nCols dense matrix and put it on the stack */
    if (!matstk(nRows, nCols, name)) return 0;
    U = tos->mat;
    for (c = 0; c < nCols; c++) {
        for (r = 0; r < nRows; r++) {
            if ((r+n) <= c) { /* am at or above requested diagonal slice */
                U[r+c*nRows] = A[r+c*nRows];
            } else {
                U[r+c*nRows] = (double) 0.0;
            }
        }
    }

    lop(); /* drop A from stack */
    return 1;
} /* 1}}} */

int triup() /* triup (hA --- hU) */
/*  Store symmetric, real N-by-N matrix A in upper-triangular packed
    mode, used by the scientific software library.  Each column, from
    the top row to the diagonal, is continuous in returned U.  N cells 
    (initialized to 0) follow U for optional use by library functions. 
*/
{
   double *A,*U;
   int i,j=0,rows,size;

   if(tos->typ!=MAT) {
      stkerr(" triup: ",MATNOT);
      return 0;
   }
   if((rows=tos->row)!=tos->col) {
      stkerr(" triup: ",SQUNOT);
      return 0;
   }
   A=tos->mat;

   size=rows*(1+rows)/2 + rows;
   if(!matstk(size,1,"_Atriup")) return 0;
   U=tos->mat;

   for(;j<rows;j++) {
      for(i=0;i<=j;i++) {
         *U=*(A+i+locvec(j,rows));
         U++;
      }
   }
   for(i=0;i<rows;i++) {
      *U=0;
      U++;
   }
   return(lop());
}

int uimport1() /* uimport1 (hT --- hA) */
/* Convert 1-byte unsigned integer numbers in T into 8-byte floating
   point numbers in matrix A.  Returned A has one column and as many
   rows as there are bytes in T. */
{
   unsigned char *T;
   double *A;
   int k=0,rows;

   if(tos->typ!=VOL && tos->typ!=STR) {
      stkerr(" uimport1: ",STRORVOLNOT);
      return 0;
   }
   T=(unsigned char *)tos->tex;
   rows=(tos->row)*(tos->col);

   if(!matstk(rows,1,"_uimport1")) return 0;
   A=tos->mat;

   for(;k<rows;k++) {
      *A=(double)*T;
      A++;
      T++;
   }
   return(lop());
}

int uimport2() /* uimport2 (hT fromendian --- hA) */
/* Convert unsigned 2-byte integer numbers in T, of endian type from-
   endian, into 8-byte fp numbers of the machine's endian type. */
{
   int fromendian;

   if(!popint(&fromendian)) return 0;

   if(tos->typ!=VOL) {
      stkerr(" uimport2: ",VOLNOT);
      return 0;
   }
   return(uimport(2,fromendian));
}

int uimport4() /* uimport4 (hT fromendian --- hA) */
/* Convert unsigned 4-byte integer numbers in T, of endian type from-
   endian, into 8-byte fp numbers of the machine's endian type. */
{
   int fromendian;

   if(!popint(&fromendian)) return 0;

   if(tos->typ!=VOL) {
      stkerr(" uimport4: ",VOLNOT);
      return 0;
   }
   return(uimport(4,fromendian));
}

int uniform() /* uniform (x n --- hN) */
/* Uniform numbers from 0, separated by x. */
{
   register int i=0;
   int n;
   register double *N,x;
   double y;

   if(tos->typ!=NUM) {
      stkerr(" uniform: ",NUMNOT);
      return 0;
   }
   if(tos->real<0) {
      stkerr(" uniform: ","n cannot be negative");
      return 0;
   }
   popint(&n);

   if(!popd(&y)) return 0;
   x=y;

   if((N=(double *)memget(n,1))==NULL) {
      stkerr(" uniform: ",MEMNOT);
      return 0;
   }
   for(;i<n;i++) *(N+i)=i*x;

   return(push(MAT,(char *)memgetn("_uniform",8),NOTAG,0, \
      (double *)N,NULL,n,1,NULL));
}

int within() /* within (x x1 x2 --- f) */
{
   double x,y1,y2;
   register double x1,x2;
   register double *F,*X,*X1,*X2;
   stkitem *sX,*sX1,*sX2;
   int cols,len,rows;
   register int k=0;

   if(tos->typ==NUM) {
      if(!popd(&y2) || !popd(&y1) || !popd(&x)) {
         stkerr(" within: expect three numbers on stack","");
         return 0;
      }
      x1=MIN(y1,y2);
      x2=MAX(y1,y2);
      if(x<=x2 && x>=x1) pushint(xTRUE);
      else pushint(xFALSE);
      return 1;
   }
   if(tos->typ==MAT) {
      if((tos-1)->typ!=MAT || (tos-2)->typ!=MAT) {
         stkerr(" within: expect three matrices on stack","");
         return 0;
      }
      sX2=tos;
      sX1=tos-1;
      sX=tos-2;
      rows=sX->row;
      cols=sX->col;

      if(sX2->row!=rows || sX1->row!=rows ||
         sX2->col!=cols || sX1->col!=cols) {
         stkerr(" within: ",MATSNOTC);
         return 0;
      }
      if(!matstk(rows,cols,"_within")) return 0;
      F=tos->mat;

      X2=sX2->mat;
      X1=sX1->mat;
      X=sX->mat;

      len=rows*cols;

      for(;k<len;k++) {
         x1=MIN(*(X1+k),*(X2+k));
         x2=MAX(*(X1+k),*(X2+k));
         if(*(X+k)<=x2 && *(X+k)>=x1) *(F+k)=xTRUE;
         else *(F+k)=xFALSE;
      }
      return(lop() && lop() && lop());
   }
   stkerr(" within: ",STKNOT);
   return 0;
}

int Xvec() /* Xvec ( --- h3x1) */
/* Pushes a fresh 3x1 X unit vector to the stack. */
{
   const int len=3*sizeof(double);
   double *A;

   const char *name="_Xvec";
   const int nlen=strlen(name);
   char *p;

   A=(double *)calloc(1,len);
   *A=1;

   p=malloc(1+nlen);
   memcpy(p,name,1+nlen);

   return(push(MAT,(char *)p,NOTAG,0,A,NULL,3,1,NULL));
}

int Yvec() /* Yvec ( --- h3x1) */
/* Pushes a fresh 3x1 Y unit vector to the stack. */
{
   const int len=3*sizeof(double);
   double *A;

   const char *name="_Yvec";
   const int nlen=strlen(name);
   char *p;

   A=(double *)calloc(1,len);
   *(A+1)=1;

   p=malloc(1+nlen);
   memcpy(p,name,1+nlen);

   return(push(MAT,(char *)p,NOTAG,0,A,NULL,3,1,NULL));
}

int Zvec() /* Zvec ( --- h3x1) */
/* Pushes a fresh 3x1 Z unit vector to the stack. */
{
   const int len=3*sizeof(double);
   double *A;

   const char *name="_Zvec";
   const int nlen=strlen(name);
   char *p;

   A=(double *)calloc(1,len);
   *(A+2)=1;

   p=malloc(1+nlen);
   memcpy(p,name,1+nlen);

   return(push(MAT,(char *)p,NOTAG,0,A,NULL,3,1,NULL));
}

int inflate()       /* inflate (hA hRind hCind R C --- hB) {{{1 */
/* 
 * man entry:  inflate {{{2
 * (hA hRind hCind R C --- hB) Creates a null matrix [B] of size R x C, then inserts terms of [A] into [B] according to the row and column index arrays {Rind} and {Cind}.  The size of {Rind} must equal the number of rows in [A], and the each term in {Rind} must fall between 1 and the number of rows in [B] (for 1 based indexing).  Similarly, the size of {Cind} must equal the number of columns in [A], and each term in {Cind} must fall between 1 and the number of columns in [B].  [A] must be dense, may be real or complex.  In matlab, [B] would be created with these commands:  B = zeros(R,C); B(Rind,Cind) = A;
use> 3 seedset 2 3 random (A) list: 5 1 ; (Rind) list: 2 6 4 ; (Cind) 5 (R) 6 (C) inflate
  creates these inputs
   [A]    = 
   [  0.3946    0.376   0.6569 ]
   [  0.2668   0.5983   0.1411 ]
   {Rind} = {5, 1}
   {Cind} = {2, 6, 4}
  to produce this [B] matrix:
    [ 0   0.2668    0   0.1411     0   0.5983 ]
    [ 0        0    0        0     0        0 ]
    [ 0        0    0        0     0        0 ]
    [ 0        0    0        0     0        0 ]
    [ 0   0.3946    0   0.6569     0    0.376 ]

 * category: math::matrix::partitioning
 * related:  place, ram, cram, spinflate
 * 2}}}
 */
{
int DEBUG = 0;
    char   *name = "_inflate";
    int     ri, ci, R_size, C_size, cmplex = 0,
            rA = 0, cA = 0, nRowsA = 0, nColsA = 0, 
            rB = 0, cB = 0, nRowsB = 0, nColsB = 0;
    double *A, *B, *C_ind, *R_ind;
#define size_T 160
    char      T[size_T+1];

    if (!popint(&nColsB)) return 0;
    if (!popint(&nRowsB)) return 0;
    if (tos->typ != MAT) {
        stkerr(" inflate: Cind ",MATNOT);
        return 0;
    }
    if ((tos-1)->typ != MAT) {
        stkerr(" inflate: Rind ",MATNOT);
        return 0;
    }
    if ((tos-2)->typ != MAT || is_sparse(tos-2)) {
        stkerr(" inflate: A ",MATNOT);
        return 0;
    }
    nRowsA = (tos-2)->row;
    nColsA = (tos-2)->col;
    A      = (tos-2)->mat;
    cmplex = is_complex(tos-2);
if (DEBUG) gprintf("inflate [B]: R=%d  C=%d [A]: R=%d  C=%d\n", nRowsB, nColsB, nRowsA, nColsA);
    if (cmplex)
        nRowsA /= 2;

    C_ind  = (tos-0)->mat;
    C_size = (tos-0)->row*(tos-0)->col;
    if (C_size != nColsA) {
        stkerr(" inflate: ", COLSNOTV);
        return 0;
    }

    R_ind  = (tos-1)->mat;
    R_size = (tos-1)->row*(tos-1)->col;
    if (R_size != nRowsA) {
        stkerr(" inflate: ", ROWSNOTV);
        return 0;
    }
if (DEBUG) gprintf("inflate size {Rind}=%d {Cind}=%d\n", R_size, C_size);

    /* allocate a nRows x nCols dense matrix and put it on the stack, zero it out */
    if (cmplex) {
        if (!matstk(2*nRowsB, nColsB, name)) return 0;
        set_complex(tos);
    } else {
        if (!matstk(  nRowsB, nColsB, name)) return 0;
    }
    B = tos->mat;
    for (cB = 0; cB < nColsB; cB++)
        for (rB = 0; rB < nRowsB; rB++)
            if (cmplex) {
                B[2*(rB + cB*nRowsB)    ] = (double) 0.0;
                B[2*(rB + cB*nRowsB) + 1] = (double) 0.0;
            } else {
                B[   rB + cB*nRowsB     ] = (double) 0.0;
            }

    /* make sure row and column indices are in range */
    for (ci = 0; ci < C_size; ci++) {
        cA = ((int) C_ind[ci]) - XBASE;
        if ( (cA < 0) ||  (cA >= nColsB)) {
            snprintf(T, size_T, 
                     " inflate: Cind[%d] = %d is not a valid column index; must be >= %d and <= # columns [B]=%d", 
                     ci+XBASE, cA+XBASE, XBASE, nColsB);
            stkerr(T, ""); 
            return 0;
        }
    }
    for (ri = 0; ri < R_size; ri++) {
        rA = ((int) R_ind[ri]) - XBASE;
        if ( (rA < 0) ||  (rA >= nRowsB)) {
            snprintf(T, size_T, 
                     " inflate: Rind[%d] = %d is not a valid row index; must be >= %d and <= rows [B]=%d)", 
                     ri+XBASE, rA+XBASE, XBASE, nRowsB);
            stkerr(T, ""); 
            return 0;
        }
    }

    /* insert terms of [A] into [B] */
    for (ci = 0; ci < C_size; ci++) {
        cA = ((int) C_ind[ci]) - XBASE;
        for (ri = 0; ri < R_size; ri++) {
            rA = ((int) R_ind[ri]) - XBASE;
            if (cmplex) {
                B[2*(rA + cA*nRowsB)    ] = A[2*(ri + ci*nRowsA)    ];
                B[2*(rA + cA*nRowsB) + 1] = A[2*(ri + ci*nRowsA) + 1];
            } else {
                B[   rA + cA*nRowsB     ] = A[   ri + ci*nRowsA     ];
            }
        }
    }

    lop(); /* drop {Cind} from stack */
    lop(); /* drop {Rind} from stack */
    lop(); /* drop [A]    from stack */
    return 1;
} /* 1}}} */
int shuffle()       /* shuffle (hA --- hB) {{{1 */
/* 
 * man entry:  shuffle {{{2
 * (hA --- hB) Randomly rearrange entries of the matrix A using the Fisher-Yates algorithm.  Useful for creating a random list that has no duplicate entries:
 use> 1 4 uniform (A) shuffle (B) .m nl
 shows a vector with values 0,1,2 & 3 in random order.
 Ref: http://www.stanford.edu/~blp/writings/clc/shuffle.html
 * category: math::matrix::partitioning, math::matrix::operator
 * related:  ranint
 * 2}}}
 */
{
    char   *name = "_shuffle";
    int     i, j, n;
    double *A, temp;

    cop();
    if (tos->typ != MAT || is_sparse(tos)) {
        stkerr(" shuffle: ",MATNOT);
        return 0;
    }
    A = tos->mat;
    n = tos->row * tos->col;

    for (i = 0; i < n - 1; i++) {
        j    = i + rand() / (RAND_MAX / (n - i) + 1);
        temp = A[j];
        A[j] = A[i];
        A[i] = temp;
    }
    return pushstr(name) && naming();
} /* 1}}} */
