/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2013  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

/* sys.c  April 1999

Copyright (c) 1999  D. R. Williamson

   Linking object functions from outside sources into the program can
   make it difficult to keep it running autonomously, since functions 
   written by others may take over alarm signals or attempt to exit if
   they encounter a problem.  Some safeguards against this have been 
   built-in:

      Function exit() allows recovery if functions linked from outside 
      sources call exit() when they have a problem.  Placing long jumps
      in exit() back to locations in the program where these outside 
      functions were called is a means of preventing unnecessary pro-
      gram exits.  

      As an example, see setjmp() settings in vmo.c, and their counter-
      parts in exit(), that deal with possible time-outs of voice modem
      events in mgetty object functions that are used as-is by the pro-
      gram.  Also see dgemm1() in lapack.c, where program exit by BLAS
      function dgemm() (calling XERBLA()) is blocked. */

#define _POSIX_SOURCE /* GNU C Lib: features.h will set __USE_POSIX */

#define _XOPEN_SOURCE_EXTENDED 1 /* 1 for AIX, for time.h timeval */
#include <time.h> /* must appear before stdio.h in AIX */
#include <stdio.h>
 
#include <sys/utsname.h>
#include <ctype.h>

#undef  __USE_BSD
#define __USE_BSD

#undef  __EXTENSIONS__
#define __EXTENSIONS__ /* for stat.h, Sun */
#include <sys/stat.h>

#undef  __USE_SVID 
#define __USE_SVID 

#undef  __EXTENSIONS__
#define __EXTENSIONS__ /* for stdlib.h, Sun */
#include <stdlib.h>

#include <string.h>
#include <unistd.h>
#include <sys/wait.h>

#include "main.h"
#include "stk.h"

#include "ctrl.h"
#include "exe.h"
#include "inpo.h"
#include "key.h"
#include "lib.h"
#include "mat.h"
#include "math1.h"
#include "mem.h"
#include "prs.h"
#include "sys.h"
#include "term.h"
#include "tex.h"

/* Phrase for restoring the terminal line settings (odd interactive 
   behavior may be seen otherwise): */
char* stty_reset="INFIX << push 'stty' exists? \
                  IF 'stty' 'clean' localrun THEN \
                  pull infix_set";

int abort1() /* abort (n --- ) */
/* Stop execution with error n to the exit function.  When new lines
   are added to this program's function exit(), this word is useful for
   checking the handling of its received exit codes (for n not equal
   to 1). 
   For immediate exit to the system's _exit() function, n=1. */
{
   int n;

   if(tos->typ!=NUM) {
      stkerr(" abort: ",NUMNOT);
      return 0;
   }
   popint(&n);

   if(n==1) {
      pushstr(stty_reset);
      main1();
      _exit(1); 
   }
   if(n==8) { /* 8 abort makes a clean exit */
#     ifdef KEYBOARD
         gprintf(" Press a key to exit..."); 
         key(); 
         nc();
#     endif
#     ifdef NET
         serverclose();
#     endif
      set_stdout();
      pushstr(stty_reset);
      main1();

      OK_EXIT=1; 
   }
   exit(n);
   return 1;
}

#ifdef KEYBOARD
int accept1() /* accept ( --- false or qA true) */
/* Accepting key input on current line, with vi-style editing. 
   This function has been superseded by accept_keys(), file key.c. */
{
   int len;

/* Leftmost margin, INDENT, is set here only, and usually reflects 
   the offset due to the prompt. */
   INDENT=MIN(chout,GOMAX); 
   kset(kBEG);

   while(keys(0)); /* runs until NL key or ESC-q */
    
   kset(kEND);

/* The keyed line image, same as displayed on screen--including the 
   prompt--is in GOUT: */
   len=MAX(0,(int)(strlen(GOUT)-INDENT));

   if(!len || BYE) {
      /* NL key on empty line or ESC-q for BYE: */
      return(pushint(xFALSE)); /* no mail */
   }
   return(
      pushq2(GOUT+INDENT,len) /* pushing text without prompt to stack */
      && pushint(xTRUE) /* you have mail */
   );
}
#endif

int argv() /* argv (qName --- qVal) */
/* Return the value of command line argument associated with Name. 
   At start up, command line arguments were stored in the catalog 
   in a VOL called ARGV.

   In ARGV, line N contains Name and line N+1 contains Val. 

   Notes: 

   Intended for use with paired Name:Val, and returns the string 
   associated with Val, the very next argv following Name.  

   But works ok for single Name--just test for chars in returned Val.  
   If Name is a found argv, then chars in Val will not be zero, even 
   if Name is the last argv.  Caveat: this single usage will give false
   indication if some Val happens to match a single Name being tested.
*/
{
/* Running this string (using main1()) puts on the stack the line 
   following Name, or an empty string if Name not found: */
   char *getval="(hRows) any? \
      IF (hRows) ontop tic ARGV rows min quote strchop \
      ELSE (ARGV) drop '' \
      THEN";

   if(tos->typ!=STR) {
      stkerr(" argv: ",STRNOT);
      return 0;
   }
   strchop();

   pushq2("ARGV",4); main1();
   dup1s();
   rot(); /* stk: ARGV ARGV Name */

   grepe(); /* (hRows): line numbers containing Name exactly */

   pushstr(getval); main1();
   return 1;
}

int ascii() /* ascii ( --- ) */
/* Next files to be opened are ascii. */
{
   BIN=xFALSE;
   return 1;
}

int base0() /* 0based ( --- ) */
/* Set index base to 0. */
{ 
   XBASE=0;
   return 1; 
}

int base1() /* 1based ( --- ) */
/* Set index base to 1. */
{ 
   XBASE=1;
   return 1; 
}

int beep1() /* beep ( --- ) */
/* Sounding the system bell. */
{
   fprintf(sysout,"%c",0x7);
   return 1;
}

int binary() /* binary ( --- ) */
/* Next files to be opened are binary. */
{
   BIN=xTRUE;
   return 1;
}
 
char *binif(double opt)
/* Obtaining character string for opt from word _binh (file boot.v). */
{
   char *p;

   pushd(opt); /* firing opt in word _binh */
   pushq2("_binh",5);
   main1();

   p=(char *)memgetn(tos->tex,tos->col);
   drop();

   return p; /* returning mallocked ptr */
}

int boot() /* boot (... opt --- ) */
/* Responding to an option during start up (see file boot.v). */
{
   int opt;

/*       1          2    3 */
   enum {bSYSCRCH=1,bSAY,bSYSPATH};
   
   popint(&opt);

   switch(opt) {

      case bSYSCRCH:
         SYSCRCH=(char *)memgetn(tos->tex,tos->col);
         drop();
      break;

      case bSAY: /* (qMessage --- ) */
         SAY=(char *)memgetn(tos->tex,tos->col);
         proghello();
         drop();
      break;

      case bSYSPATH:
         SYSPATH=(char *)memgetn(tos->tex,tos->col);
         drop();
      break;
   }
   return 1;
}

int bye() /* bye ( --- ) */
{
   int code=EXIT_SUCCESS,keys,ret=0;

   OK_EXIT=1;

   if(!_exists("clean")) { 
      set_stdout();
      exit(EXIT_FAILURE);
   }
   keys=KEYS;
   unload("clean");

   if(keys) {
      gprintf("%d keys",kHIT); nc();
      ret=(
         pushstr("Good-bye") &&
         datesys() &&
         neat() &&
         pile() &&
         dup1s() &&
         cols() &&
         pushstr("center") &&
         main1() &&
         dot() &&
         nc()
      );
   }
   if(stkerrabs || !ret) code=EXIT_FAILURE;
   exit(code);
}

int catmsg() /* catmsg (f --- ) */
/* Enable or disable message when word goes into catalog or is 
   redefined; true f enables. */
{
   int f;

   if(!popint(&f)) return 0;

   CATMSG=(f!=0);
   return 1;
}

int catmsgflag() /* CATMSG ( --- f) */
/* Current setting of catmsg flag. */
{
   return(pushint(xTRUE*CATMSG));
}

int catpath() /* catpath (qPath qFile --- qFile1) */
/* Add path to file name, ensuring / between them.
   Note: trailing blanks on qFile are not removed. */
{
   char *F;
   int k=0,rows,width;

   if(stkdepth()<2) {
      stkerr(" catpath: ",NEEDTWO);
      return 0;
   }
   if(!(tos->typ==STR || tos->typ==VOL)|| (tos-1)->typ!=STR) {
      stkerr(" catpath: ",STRNOT);
      return 0;
   }
   F=tos->tex;
   width=tos->col;
   rows=tos->row;

   swap();
   strchop();
   if(tos->col==0) return(drop());

   if(*(tos->tex+(tos->col)-1)!='/') {
      pushq2("/",1);
      cat();
   }
   lpush();

   for(;k<rows;k++) {
      lpeek();
      pushq2(F,width);

      pushint(0); justify(); /* left justify */

      if(*(tos->tex)=='/') {
         pushint(-1); indent(); /* no leading slash */
      }
      cat();
      F+=width;
   }
   return(
      lpull() &&
      drop() &&
      pushint(rows) &&
      pilen() &&
      pushq2("_catpath",8) &&
      naming() &&
      lop()
   );
}

int chdir1() /* chdir (qS --- ) */
/* Change the current directory. */
{
   int len=0;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" chdir: ",STRORVOLNOT);
      return 0;
   }
   if(!unquoted()) return 0;

   len=stradv(tos->tex,"#");
   if(len) {
      *(tos->tex+len-1)='\0';
      tos->col=strlen(tos->tex);
   }
   len=stradv(tos->tex,"\\");
   if(len) {
      *(tos->tex+len-1)='\0';
      tos->col=strlen(tos->tex);
   }
   pushq2(" ",1);
   catpath();
   strchop();

   if(!tos->col) {
      drop();
      pushq2("~",1);
   }
   pushq2("dup 1st catch",13); main1();

   if((strcmp(tos->tex,"~")==0)) {
      drop();
      pushq2("-1 indent",9); main1();
      pushq2("HOME",4);
      env();
      pushq2("swap catpath",12); main1();
   }
   else drop();

   if(chdir(tos->tex)) {
      stkerr(" chdir: ","directory not changed");
      return 0;
   }
   return(drop());
}

int chmodf() /* chmod (qFile Mode --- ) */
/* Change file access permissions. */
{
   FILE *fp=NULL;
   mode_t Mode;
   int fd;

   if((tos-1)->typ!=STR && (tos-1)->typ!=VOL) {
      stkerr(" chmodf: ",STRORVOLNOT);
      return 0;
   }
   fp=fopen((tos-1)->tex,"ab+"); /* can open for read and write? */

   if(fp) {
      fd=fileno(fp);
      Mode=(mode_t)tos->real;
      if(fchmod(fd,Mode)) {
         stkerr(" chmodf: ","error changing file mode");
         fclose(fp);
         return 0;
      }
      fclose(fp);
      return(drop2());
   }
   else {
      stkerr(" chmodf: ",FILERR1);
      return 0;
   }
}

int chperm() /* chperm ( --- ) */
/* Display typical permissions for chmod */
{
   nc();
   gprintf(" Typical Mode permissions (decimal) for word chmod:"); nc();
   nc();
   gprintf("      Read by owner: %d",S_IRUSR); nc();
   gprintf("     Write by owner: %d",S_IWUSR); nc();
   gprintf("   Execute by owner: %d",S_IXUSR); nc();
   gprintf("       All by owner: %d",S_IRWXU); nc();
   nc();
   gprintf("      Read by group: %d",S_IRGRP); nc();
   gprintf("     Write by group: %d",S_IWGRP); nc();
   gprintf("   Execute by group: %d",S_IXGRP); nc();
   gprintf("       All by group: %d",S_IRWXG); nc();
   nc();
   gprintf("     Read by others: %d",S_IROTH); nc();
   gprintf("    Write by others: %d",S_IWOTH); nc();
   gprintf("  Execute by others: %d",S_IXOTH); nc();
   gprintf("      All by others: %d",S_IRWXO); nc();
   nc();
   gprintf(
    "  man chperm shows how these values can be summed to make others");
   nc();
   return 1;
}

int ctime1() /* ctime (hA --- hT) */
/* System times in A into readable text, local date string. */
{
   time_t mtime;
   struct tm *tm1;
   int TIMLEN=33;

   char *name="_ctime";
   double *A;
   char *T;
   int i=0,rows;

   if(tos->typ!=NUM && tos->typ!=MAT) {
      stkerr(" ctime: ",NUMORMATNOT);
      return 0;
   }
   hand();
   A=tos->mat;
   rows=tos->row;

   if(!volstk(rows,TIMLEN,name)) return 0;
   T=tos->tex;
   memset(T,' ',rows*TIMLEN);

   for(;i<rows;i++) {
      mtime=*A;

      tm1=localtime(&mtime);
      strftime(T,TIMLEN,"%a %b %e %H:%M:%S %Z %Y",tm1);

      T+=TIMLEN;
      A++;
   }
   return(
      lop() &&
      notrailing() &&
      pushstr(name) &&
      naming()
   );
}

char strtime[33];
#define LEN 32

int datesys() /* date ( --- qS) */
/* Putting system date and time string on the stack with the program's
   GMT adjustment. */
{
   return(pushstr(datetime()));
}

int datesys1() /* date1 ( --- qS) */
/* Putting machine date and time string on the stack, without the
   program's GMT adjustment. */
{
   struct tm *loc;
   time_t ltime;
   int len;

   ltime=time(&ltime);

   loc=localtime(&ltime);
   len=strftime(strtime,LEN,"%a %b %e %H:%M:%S %Z %Y",loc);

   *(strtime+len)='\0';
   return(pushstr(strtime));
}

int datesys2() /* date2 ( --- qS) */
/* Putting system Ymd date string on stack. */
{
   struct tm *loc;
   time_t ltime;
   int len;

   ltime=time(&ltime)+DELTA_T;

   loc=localtime(&ltime);
   len=strftime(strtime,LEN,"%Y%m%d",loc);

   *(strtime+len)='\0';
   return(pushstr(strtime));
}

char *datetime()
/* Date and time string with the program's GMT adjustment. */
{
   struct tm *loc;
   time_t ltime;
   int len;

   ltime=time(&ltime)+DELTA_T;

   loc=localtime(&ltime);
   len=strftime(strtime,LEN,"%a %b %e %H:%M:%S %Z %Y",loc);

   *(strtime+len)='\0';
   return(strtime);
}

int delete() /* delete (qS --- ) */
/* Deleting file named S. */
{
   char *permission="INFIX << push dup fallow pull infix_set";
   int f;

   if(tos->typ!=STR) {
      stkerr(" delete: ",STRNOT);
      return 0;
   }
   unquoted();

   dup1s();
   fileq(); /* does file exist? */
   popint(&f);
   if(f) {
      pushstr(permission); 
      main1(); /* permission to write? */
      popint(&f);
      if(f) { 
         if(remove(tos->tex)) {
            if(f) {
               gprintf(" delete: failed to delete %s",tos->tex);
               nc();
               stkerr("","");
               return 0;
            }
         }
         else return(drop());
      }
      else {
         gprintf(" delete: permission denied: %s",tos->tex);
         nc();
         stkerr("","");
         return 0;
      }
   }
   gprintf(" delete: file not found: %s",tos->tex);
   stkerr("","");
   nc();
   drop();
   return 0;
}

int env() /* env (qName --- qEnv) */
/* Return the value of environment Name. */
{
   char *env;

   if(tos->typ!=STR) {
      stkerr(" env: ",STRNOT);
      return 0;
   }
   strchop();

   env=getenv(tos->tex);
   if(env) {
      strstk(strlen(env),"_env");
      memcpy(tos->tex,env,strlen(env));
   }
   else pushq2("",0);
   return(lop());
}

int ercnt() /* ercnt ( --- k) */
/* Push current error count to stack. */
{  
   return(pushint(stkerrabs));
}

int ertrip() /* ertrip (n --- ) */
/* Set error threshold that trips the runaway error monitor.
   Typical values: 9 for slow machine, 30 for fast one. */
{
   return(popint(&errTHRESHOLD));
}

void exit(int code)
/* Exit the program if flag OK_EXIT is set.

   This function provides some insulation against functions, linked 
   from elsewhere, that call exit() when they have a problem.  Because 
   they call exit, this function is called exit.  To do a system exit, 
   this function calls _exit(). */
{
   char *jmp_ready_name=" ";

/* Doing a normal exit: */
   if(OK_EXIT) _exit(code);

/*--------------------------------------------------------------------*/

/* Build special jumps in this section; see enum ctrlret in ctrl.h to 
   add names of jmpready values.  At this point, the program is on run 
   level onbuf.  */

/* Long jmp info (this name is only needed if it is to be optionally
   displayed below; enter BL (" ") for no display): */
   if(*(jmpready+onbuf)==VOICE) jmp_ready_name=" ";
   else if(*(jmpready+onbuf)==PROMPTER) jmp_ready_name=" ";
   else if(*(jmpready+onbuf)==KEY) jmp_ready_name=" ";

/* Add new long jmp info above using this template:
   else if(*(jmpready+onbuf)==xxxx) jmp_ready_name="yyyy";
   */

/* Displaying optional long jmp info (if name is not BL): */
   if(strcmp(jmp_ready_name," ")!=0) {
      gprintf(" exit: jumping to %s",jmp_ready_name); 
      nc();
   }
/*--------------------------------------------------------------------*/

/* Make longjmps from here: */

/* Case of a function in mgetty calling EXIT: */
   if(*(jmpready+onbuf)==VOICE) {
      gprintf(" exit: jumping to VOICE"); 
      nc();
      jmptable();
      longjmp(*(jmpenv+onbuf),VOICE);
   }
/* Case of Blas function xerbla.f calling STOP: */
   if(*(jmpready+onbuf)==BLASX) {
      gprintf(" exit: jumping to BLASX"); 
      nc();
      longjmp(*(jmpenv+onbuf),BLASX);
   }
#ifdef KEYBOARD
/* Case of ^C waiting for a key: */
   if(*(jmpready+onbuf)==KEY) {
      gprintf(" exit: jumping to KEY"); 
      nc();
      jmptable();

      keywin(kEND);
      keyboard();

      if(_exists("cprompt")) {
         nc();
         pushq2("cprompt .",9); main1();
      }
      longjmp(*(jmpenv+onbuf),KEY);
   }
#endif

/*--------------------------------------------------------------------*/

/* Add new long jmps above using this template:

   if(*(jmpready+onbuf)==xxxx) longjmp(*(jmpenv+onbuf),yyyy);

   where xxxx=yyyy in most cases.
*/
/*--------------------------------------------------------------------*/
 
/* Default: Unwinding to the nearest-below "running of words" level
   that got us here. */

   datesys1(); /* machine date, to correlate with system log times */
   gprintf(" exit: code %d received on level %d, %s",\
      code,onbuf,tos->tex);
   nc();
   drop();

   jmptable();
   gprintf(" exit: attempting to return to lower levels");
   nc();

#ifdef KEYBOARD
   keywin(kEND);
   keyboard();
#endif

   return1();

#ifdef KEYBOARD
   if(KEYS) {
      gprintf(" exit: return failed; press a key to exit..."); 
      key();
      nc();
   }
   else gprintf(" exit: return failed; program exit"); 
#else
   gprintf(" exit: return failed; program exit"); 
#endif

   quit(); /* clean, then re-enter this function to OK_EXIT */

   _exit(code); /* never gets here, but compiler is happier */
}

int extract1(char *libname, char *wordname) /* (... --- ...) */
/* Speedy version of extract, for wordname and libname already in
   memory.  Not for general use: assumes maximum string length for
   catalog item name. */
{
   char p[256]; /* assumed max len for wordname+LIBID+libname */

   *p='\0';

/* Making catalog name like wordname+LIBID+libname: */
   strncat(p,wordname,256);
   strncat(p,LIBID,256);
   strncat(p,libname,256);

   return(
      pushstr(p) &&
      ptr() &&
      exe() /* firing the catalog item */
   );
}

int flush_sysout() /* flush_sysout ( --- ) */
/* Write buffered data to sysout and leave sysout open. */
{
   fflush(sysout);
   return 1;
}

int getenv1() /* getenv ( --- hT) */
/* Return the program environment strings in the rows of T; T is 
   purged if no program environment is defined. */
{
   char *env;
   int k=1,len;

   pushq2("",0);
   hand();

   env=*PROGENV;

   if(env==NULL) {
      return(
         noblanklines() &&
         pushq2("_getenv",7) &&
         naming()
      );
   }
   len=strlen(env);

   while(len) { 

      pushq2(env,len); 
      env+=(1+len);
      len=strlen(env);

      k++; 
   }
   return(
      pushint(k) &&
      pilen() &&
      noblanklines() &&
      pushq2("_getenv",7) &&
      naming()
   );
}

int getlogin1() /* getlogin ( --- qS) */
/* Avoiding the getlogin() function.  In RH 7.2 distribution of
   linux, man says:
      BUGS
        Unfortunately, it is often rather easy to fool getlogin().
        Sometimes  it  does  not work at all, because some program
        messed up the utmp file. ...

      DESCRIPTION
        ...
        For most purposes, it is more useful to use  the  environ-
        ment  variable  LOGNAME to find out who the user is. 

   Having this function not work in window manager rxvt under RH 7.3, 
   after working in previous versions, LOGNAME is what we'll use. */
{
   return(
      pushq2("LOGNAME",7) &&
      env()
   );
}

int getpid1() /* getpid ( --- n) */
{
   int n;
   n=getpid();
   return(pushint(n));
}

int getpwd() /* pwd ( --- qS) */
/* Get present working directory. */
{
   char *p=NULL,pwdstr[256];

   p=getcwd(pwdstr,255);
   if(p)
      return(
         pushstr(pwdstr) &&
         pushq2("",0) &&
         catpath()
      );
   else return(pushq2("",0));
}

int getuid1() /* getuid ( --- n) */
{
   int n;
   n=getuid();
   return(pushint(n));
}

int GMTdelta() /* GMTdelta (s --- ) */
/* Set true GMT minus the machine's GMT to be s (seconds).  This dif-
   ference will be applied to time functions used by the program, like 
   ALARMs and the time pushed to the stack by words time and date.  

   It will not affect the system time used for things like dates of
   files. */
{
   int s;
   if(!popint(&s)) return 0;
   DELTA_T=s;
   return 1;
}

int gmtime1() /* gmtime (hA --- hT) */
/* System times in A into readable text, date string for Greenwich,
   England. */
{
   time_t mtime;
   struct tm *tm1;
   int TIMLEN=33;

   char *name="_gmtime";
   double *A;
   char *T;
   int i=0,rows;

   if(tos->typ!=NUM && tos->typ!=MAT) {
      stkerr(" gmtime: ",NUMORMATNOT);
      return 0;
   }
   hand();
   A=tos->mat;
   rows=tos->row;

   if(!volstk(rows,TIMLEN,name)) return 0;
   T=tos->tex;
   memset(T,' ',rows*TIMLEN);

   for(;i<rows;i++) {
      mtime=*A;

      tm1=gmtime(&mtime);
      strftime(T,TIMLEN,"%a %b %e %H:%M:%S %Z %Y",tm1);

      T+=TIMLEN;
      A++;
   }
   return(
      lop() &&
      notrailing() &&
      pushstr(name) &&
      naming()
   );
}

int hide(int opt)
/* Disable or enable the trace flag. */
{
   static int hiding=0,TRsav=0;

   if(opt) {
      if(!hiding) {
      /* Saving TRACE, then turning TRACE off: */
         hiding=1;
         TRsav=TRACE;
         TRACE=0;
      }
   }
   else {
      if(hiding) {
         TRACE=TRsav; /* restoring TRACE */
         hiding=0;
      }
   }
   return 1;
}

int hideoff() /* nohide ( --- ) */
/* Restoring the saved TRACE flag. */
{
   return(hide(0));
}

int hideon() /* hide ( --- ) */
/* Hiding the TRACE flag, turning TRACE off. */
{
   return(hide(1));
}

int hlmax() /* HLMAX (L --- ) */
/* Set maximum lines (at least 1) in history buffer. */
{
   double dL;
   const double Lmax=1000000.; /* so INF HLMAX is just 1 million */
   int L;

   if(!popd(&dL)) return 0;
   L=MIN(dL,Lmax);

   HLMAX=MAX(1,L);

   return 1;
}

int hostname() /* _host ( --- qS) */
/* Fetching host name.  Uses only the first characters that precede
   a dot, so that host gutter.abc.com is simply gutter. */
{
   char ch,str[133]={0};
   int i=1,len=0,ret;

   ret=(gethostname(str,132)==0);

   if(ret) {
      pushstr(str);
      len=strlen(tos->tex);
      while(i<len && (ch=*(tos->tex+i))!='.' && ch!=' ') i++;
      len=i;
   }
   return(
      ret &&
      pushq2(tos->tex,len) &&
      notrailing() &&
      lowercase() &&
      lop()
   );
}

int implant1() /* implant1 (d qW qA --- ) */
/* Store number d in scalar A in the library of word W.  Equivalent to
   the phrase: (d qW qA) localpry bang. */
{
   char *p;
   int len,lensep=strlen(LIBID),ret=0;

   if(tos->typ!=STR || (tos-1)->typ!=STR) {
      stkerr(" implant1: ",STRSNOT2);
      return 0;
   }
   strchop();
   swap();
   strchop();
   swap();

   len=lensep+tos->col+(tos-1)->col;
   if((p=malloc(1+len))==NULL) {
      stkerr(" implant1: ",MEMNOT);
      return 0;
   }
   *p='\0';

/* Making catalog name like A+LIBID+W: */
   strcat(p,tos->tex);
   strcat(p,LIBID);
   strcat(p,(tos-1)->tex);

   ret=(
      drop2() &&
      pushq2(p,len) &&
      ptr() &&
      exe() &&
      bang()
   );
   mallfree((void *)&p);
   return(ret);
}

int infix() /* >> ( --- ) */
{
   INFIX=1;
   return 1;
}

int infixflag() /* INFIX ( --- f) */
/* Current setting of infix flag. */
{
   return(pushint(xTRUE*INFIX));
}

int infix_set() /* infix_set (f --- ) */
{
   int f;

   if(!popint(&f)) return 0;

   INFIX=(f!=0);
   return 1;
}

int keyq() /* keys? ( --- f) */
{
   return(pushint(xTRUE*KEYS));
}

int mktime1() /* ltime (YYYDDMM HHMMSS --- n) */

/* From: /usr/src/linux-2.0.36/arch/i386/kernel/time.c */

/* Converts Gregorian date to seconds since 1970-01-01 00:00:00.
 * Assumes input in normal date format, i.e. 1980-12-31 23:59:59
 * => year=1980, mon=12, day=31, hour=23, min=59, sec=59.
 *
 * [For the Julian calendar (which was used in Russia before 1917,
 * Britain & colonies before 1752, anywhere else before 1582,
 * and is still in use by some communities) leave out the
 * -year/100+year/400 terms, and add 10.]
 *
 * This algorithm was first published by Gauss (I think).
 *
 * WARNING: this function will overflow on 2106-02-07 06:28:16 on
 * machines where long is 32-bit! (However, as time_t is signed, we
 * will already get problems at other places on 2038-01-19 03:14:08)
 */
{
        unsigned long year; unsigned long mon;
        unsigned long day; unsigned long hour;
        unsigned long min; unsigned long sec;

        long long ltimel;
        double ltime;

        if(tos->typ!=NUM || (tos-1)->typ!=NUM) {
           stkerr(" ltime: ",NUMS2NOT);
           return 0;
        }
        pushint(100); 
        slashmod();
        pushint(100); 
        slashmod();
        popuint(&hour);
        popuint(&min);
        popuint(&sec);

        pushint(100); 
        slashmod();
        pushint(100); 
        slashmod();
        popuint(&year);
        year+=1900;
        popuint(&mon);
        popuint(&day);

        if (0 >= (int) (mon -= 2)) {    /* 1..12 -> 11,12,1..10 */
           mon += 12;      /* Puts Feb last since it has leap day */
           year -= 1;
        }
        ltimel = ((( (long long)
               (year/4 - year/100 + year/400 + 367*mon/12 + day) +
              year*365 - 719499
            )*24 + hour /* now have hours */
           )*60 + min /* now have minutes */
          )*60 + sec; /* finally seconds */
        
        ltime=ltimel;
        return(pushd(ltime));
}

int ndx() /* ndx (n --- n1) */
/* Converts n from a quantity to an index in the current index base:
      if index base is 1, n1=n; 
      if index base is 0, n1=n-1. */
{
   if(XBASE==1) return 1;

   if(tos->typ==NUM) {
      tos->real--;
      return 1;
   }
   if(tos->typ==MAT) {
      pushint(-1);
      return(plusd());
   } 
   stkerr(" ndx: ",NUMORMATNOT);
   return 0;
}   

int new() /* new ( --- ) */
/* Next files opened will be new. */
{
   NEW=xTRUE;
   return 1;
}

void nl_prompt()  
/* Perform nl and show the interactive prompt in a case where the Enter
   key may not have been hit. */
{
   if(KEYS && !OK_EXIT) {
      if(REMOTE_PROMPT) { /* showing remote prompt */
         nc();
         gprintf("%s",REMOTE_PROMPT);
         return;
      }
      else {
         if(_exists("cprompt")) { /* showing ready prompt */
            nc();
            pushstr("cprompt .");
            xmain(0);
            return;
         }
      }
   }
   if(chout) nc();
}

int nullshell() /* nullshell (hT --- ) */
/* Run system commands contained in lines of volume T in a shell
   of minimum (NULL) environment. */
{
   char *envsave=NULL;
   int cols,ret=0,rows,typ;

   shellget(); /* current shell environment to stack */

   cols=tos->col;
   rows=tos->row;
   typ=tos->typ;

   if((envsave=(char *)memgetn(tos->tex,cols*rows))==NULL) return 0;

   ret=(
      drop() &&

      pushq2("",0) &&
      shellset() && /* setting a NULL shell environment */

      shell() && /* nullshell running word shell */

      pushq2(envsave,cols*rows) /* saved environment to stack */
   );
   if(envsave) {
      mallfree((void *)&envsave);
      envsave=NULL;
   }
   if(ret) {
      tos->typ=typ;
      tos->row=rows;
      tos->col=cols;
   }
   return(ret && shellset()); /* setting shell environment to saved */
}

int old() /* old ( --- ) */
/* Next files opened will be old. */
{
   NEW=xFALSE;
   return 1;
}

int os() /* os ( --- qS) */
{
   struct utsname Name;
   if(uname(&Name)) return(pushstr("unknown_os"));
   return(
      pushstr(Name.sysname) &&
      lowercase()
   );
}

int os_ver() /* os_ver ( --- qS) */
{
   struct utsname Name;
   if(uname(&Name)) return(pushstr("unknown_os"));
   return(
      pushstr(Name.sysname) &&
      pushstr(".") &&
      pushstr(Name.release) &&
      cat() &&
      cat() &&
      lowercase()
   );
}

int postfix() /* << ( --- ) */
{
   INFIX=0;
   return 1;
}

int proghello() /* proghello ( --- ) */
{
   char *logfile="INFIX << push \
                  '_bin' 'filelog' extract \
                  pull infix_set";
   int f;

   if(!(sizeof(long)==8)) {
      gprintf("         %s",SAY);
   /*          .........Tops 1.1.1.........
               Fri Sep  3 04:36:56 PDT 1999 */
   }
   else {
      gprintf("       %s (64)",SAY); 
   /*          .......Tops 1.1.1 (64)......
               Fri Sep  3 04:36:56 PDT 1999 */
   }
   nc();
   gprintf("%s",datetime());
   if(!KEYS) nc();

   if(stkerrabs) {
      nc();
      stkerr(" hello: ",ERRSBOOT);
      pushstr(logfile); main1();
      fileq();
      popint(&f);
      if(f) {
         LOGKEEP=1;
         gprintf("   Please refer to file ");
         pushstr(logfile); main1();
         dot();
         nc();
         gprintf("   at occurrences of word 'fault'");
         nc();
      }
      else {
         gprintf("   Restart program with -l option to save");
         gprintf(" error log\n");
      }
   }
   return 1;
}

int public() /* public ( --- ) */
{
   PUBLIC=1;

   context('\0'); /* forcing context of this run level to MAINLIB */

/* Warning: context is not reset when -public is fired.  It is 
   assumed that by then work on this run level is complete. */

   return 1;
}

int publicoff() /* -public ( --- ) */
{
   PUBLIC=0;
   return 1;
}

int putenv1() /* setenv (qS --- ) */
/* Set the value of an environment variable with string S of the form 
   Name=Value.   

   Resetting PWD=XXX causes *environ (equals *PROGENV) to change, which
   is evidenced by running word getenv before and after.  

   This seems like a bug in putenv(), since the pointer to the environ-
   ment should never change.  Save and restore *PROGENV so this cannot 
   happen. */
{
   static char *envsave;
   char *S;

   envsave=*PROGENV;

   if(tos->typ!=STR) {
      stkerr(" setenv: ",STRNOT);
      return 0;
   }
   strchop();

   if((S=memgetn(tos->tex,tos->col))==NULL) return 0;

   if(putenv(S)) {
      stkerr(" setenv: ","cannot further enlarge environment");
      mallfree((void *)&S);
      *PROGENV=envsave;
      return 0;
   }
   *PROGENV=envsave;
   return(drop());
}

int qdx() /* qdx (n --- q) */
/* Converts n from an index in the current index base to a quantity:
      if index base is 1, q=n;
      if index base is 0, q=n+1. */
{
   if(XBASE==1) return 1;

   if(tos->typ==NUM) {
      tos->real++;
      return 1;
   }
   if(tos->typ==MAT) {
      pushint(1);
      return(plusd());
   }
   stkerr(" qdx: ",NUMORMATNOT);
   return 0;
}

int sbin() /* SBIN ( --- qS) */
/* String for subdirectory containing Unix /bin commands. */
{
   return(pushstr(SBIN));
}

int set_keyout() /* set_keyout ( --- ) */
/* Set sysout to stderr, used for keyboard. */
{
   pushstr("SYSKEY");
   return(set_sysout());
}

int set_sbin() /* SBIN_set (qS --- ) */
/* Set the directory for the Unix sh command. */
{
   if(tos->typ!=STR) {
      stkerr(" SBIN_set: ",STRNOT);
      return 0;
   }
   strchop();
   if(SBIN) mallfree((void *)&SBIN);
   SBIN=memgetn(tos->tex,tos->col); 
   return(drop());
}

int set_sh() /* SH_set (qS --- ) */
/* Set the file for the Unix sh command. */
{
   if(tos->typ!=STR) {
      stkerr(" SH_set: ",STRNOT);
      return 0;
   }
   strchop();
   if(SH) mallfree((void *)&SH);
   SH=memgetn(tos->tex,tos->col);
   return(drop());
}

int set_stderr() /* set_stderr ( --- ) */
/* Close sysout and set sysout to STDERR. */
{
   pushstr("STDERR");
   return(set_sysout());
}

int set_stdout() /* set_stdout ( --- ) */
/* Close sysout and set sysout to STDOUT. */
{
   pushstr("STDOUT");
   return(set_sysout());
}

int set_sysout() /* set_sysout (qFile --- ) */
/* Open file for program output.  

   If qFile is STDOUT or stdout, or qFile is an empty quote,
      uses system stdout;
   if qFile is STDERR, uses system stderr;
   if qFile is SYSKEY, uses system stderr. 

   Otherwise, opens a real file where all output (from words using
   gprintf()) will go. 
   Note that the interactive keyboard display will continue to be the 
   screen (word set_keyout--see use in word console, file key.v). */
{
   static int SYSOUT_is_open=0;
   int f=0;

   strchop();

   if(SYSOUT_is_open && sysout!=stdout && sysout!=stderr) {
      if(sysout) fclose(sysout);
      SYSOUT_is_open=0;
   }
   if(SYSOUT) mallfree((void *)&SYSOUT);

   if(tos->col) {
      SYSOUT=(char *)memgetn(tos->tex,tos->col);
   }
   else {
      SYSOUT=(char *)memgetn("STDOUT",6);
   }
   if(!strcmp(SYSOUT,"STDOUT") || !strcmp(SYSOUT,"stdout")) {
      sysout=stdout;
   }
   else {
      if(!strcmp(SYSOUT,"STDERR") || !strcmp(SYSOUT,"SYSKEY")) {
         sysout=stderr;
      }
      else {
         pushq2(SYSOUT,strlen(SYSOUT)); /* SYSOUT to tos */
         rmpath(); /* remove path from tos->tex = SYSOUT */
      /* Reenter without path if file name is one of these: */
         if(!strcmp(tos->tex,"STDOUT") ||
            !strcmp(tos->tex,"STDERR") ||
            !strcmp(tos->tex,"SYSKEY")) {
            lop();
            return(set_sysout()); /* reenter without path */
         }
         drop();
         pushq2(SYSOUT,strlen(SYSOUT)); /* SYSOUT to tos */
         fallow(); /* check write permission of SYSOUT */
         popint(&f);
         if(!f) {
            getpwd(); /* pwd to tos */
            gprintf(" set_sysout: %s access denied in directory %s",
               SYSOUT,tos->tex);
            nc();
            drop(); /* drop pwd */
            stkerr("","");
            SYSOUT=(char *)memgetn("STDERR",6);
            sysout=stderr;
            return 0;
         }
         sysout=fopen(SYSOUT,"ar+");
         SYSOUT_is_open=1;
      }
   }
   return(drop());
}

int setargs(int argc, char *argv[])
/* Putting command line arguments into volume called ARGS, and banking
   it into inlinex word ARGV in the main library.  Word ARGV is created
   right here and volume ARGS is banked into it.

   Later, running word ARGV will cause volume ARGS, the array of all
   command line arguments, to be placed on the stack.  Function start()
   runs word ARGV below to get the last argument, the name of the file
   to be sourced.

   Each blank-terminated string in the command line is a row in the
   volume returned by ARGV, so returned ARGS has number of rows equal
   to the argument count, argc.

   This is the first C function written that creates an inline.  Until
   now, all inlines have been created by text source files using words
   inline, inline:, inlinex and -inlinex.

   Below, word -inlinex (the dash means "not strict" about unresolved
   forward references) is fired from here to create an inline from a
   source text string and a name that have been placed on the stack by
   calls to pushstr().

   Word -inlinex is used instead of inlinex to invoke -strict and allow
   unresolved forward references in the creation of word ARGV.  In this
   case, the unresolved forward reference is to the VOL called ARGS.
   ARGS is banked into ARGV's library right after it is made, thus re-
   solving the reference. */
{
   int args=0,bad_args=0,catmsg,k=0,ok=0,ret=0;
   char *command,*undenv=NULL;
   
   os();
   if(!strcmp((char *)tos->tex,"aix")) {
      os_ver();
   /* Getting around the argv bug in AIX 5.5 for #! line argvs. */
      if(!strcmp((char *)tos->tex,"5.3")) bad_args=1;
      drop();
   }
   drop();

   if(argc>1) {
      if(bad_args) {
      /* Fetch the underscore variable: */
         ok=(getenv("_")!=NULL);
         if(ok) undenv=getenv("_");
      }
      if(undenv) { 
         command=*argv;
         if(strlen(command)>strlen(undenv)) { 
         /* Make command same length as undenv: */
            memcpy(command,\
               command+(strlen(command)-strlen(undenv)),\
               strlen(undenv));
            *(command+strlen(undenv))='\0';
         }
      /* If command is different, and not "tops," this may be a script
         file (hardcoded names like "tops" are bad, but this is just a 
         workaround): */
         if(strcmp(command,"tops") && strcmp(undenv,command)) {
            pushstr(undenv); /* file name on stack */
            strchop();
            k=tos->col-5;
            if(!strcmp((char *)(tos->tex+k),"nohup")) {
            /* If file name is nohup, the actual file to run is the
               last name in the otherwise goofed up argv list: */
               drop(); /* drop name "/usr/bin/nohup" */
               pushstr(*(argv+argc-1)); /* push name of file to run */
            }
            dup1s();
            fileq(); /* is it a file? */
            popint(&ok);
            if(ok) { /* read the #! line from the script file */
               asciiload(); /* file on stack */
               pushint(XBASE);
               quote();
               words(); /* array of #! line words on stack */
               args=tos->row;
               lpush(); /* push the array of strings to local stack */

            /* Pushing each command line argv string to the stack: */
               for(k=0;k<args;k++) { 
                  lpeek(); /* peek at array of argv strings */
                  pushint(k+XBASE);
                  quote(); /* fetch kth argv to stack */
                  strchop();
               }

               lpull();
               drop();

            /* Nohup and rsh add argvs that follow the first two;
               put them on the stack too: */
               for(k=2;k<argc;k++) {
                  pushstr(*(argv+k));
                  args++;
               }
            }
            else {
               drop(); /* file name dropped */
               undenv=NULL;
            }
         }
         else undenv=NULL;
      }
   }
/* Pushing each command line argv string to the stack: */
   if(!undenv) {
      args=argc;
      for(k=0;k<args;k++) pushstr(*(argv+k));
   }
   catmsg=CATMSG;
   CATMSG=0; /* disable report of word into catalog */

/* Piling the argv strings into a VOL, then making inlinex word, ARGV,
   and banking the VOL of args into its local library.  The C code that
   follows is equivalent to these phrases:
      (str1 str2 ... strN) N pilen (hT) \ pile of argvs, T
      words hand (hT)                   \ one argv word per row
      'ARGS' 'ARGV' -inlinex            \ making inlinex called ARGV
      (hT) 'ARGV' 'ARGS' bank           \ bank T into ARGV, called ARGS
*/
   ret=(
      pushint(args) && /* N */
      pilen() &&
      words() && /* each argv is now a separate line */
      hand() &&  /* if one line STR, hand() makes into VOL handle */

   /* The job of word ARGV is to push ARGS, a VOL (to be) in its local
      library, to the stack: */
      pushstr("ARGS") &&
      pushstr("ARGV") &&
      inlinex1() && /* running phrase: 'ARGS' 'ARGV' -inlinex */

   /* Banking (hT) into the local library of ARGV, calling it ARGS: */
      pushstr("ARGV") &&
      pushstr("ARGS") &&
      implant() /* running phrase: (hT) 'ARGV' 'ARGS' implant */
   );
   CATMSG=catmsg; /* reinstates catalog reporting option */

   if(!caton(tagged("ARGV","CODE__"))) {
      stkerr(" setargs: ","catitem ARGV not found");
      return 0;
   }
   oncat->typ=NATI; /* changing INLI type to NATI type */

   return ret;
}

int setenv1() /* setenv1 (qName qValue --- ) */
/* Set the value of an environment variable. */
{
   int ret=-1;

   if(tos->typ!=STR && (tos-1)->typ!=STR) {
      stkerr(" setenv1: ",STRNOT);
      return 0;
   }
   swap();
   strchop();
   swap();
   strchop();

   ret=setenv((tos-1)->tex,tos->tex,1);

   if(ret) {
      stkerr(" setenv1: ","cannot further enlarge environment");
      return 0;
   }
   return(drop() && drop());
}

int setxbase() /* indexbase (0 or 1 --- ) */
/* Set the index base. */
{
   int i;
   if(popint(&i)) {
      if(i==0 || i==1) {
         XBASE=i; 
         return 1;
      }
      stkerr(" setxbase: ","index base is 0 or 1");
   }
   return 0;
}

int sh1() /* SH ( --- qS) */
/* String for file that runs the command shell. */
{
   return(pushstr(SH));
}

int shell() /* shell (hT --- ), system (hT --- ) */
/* Run system commands contained in lines of volume T. */
{
   char q[512]={0},*s;
   static int qsize=511;
   register int width,i=0,k;
   char *bat="/dev/null";
   char *to;

   if(tos->typ!=VOL) {
      if(!typstr2vol()) {
         stkerr(" shell: ",VOLNOT);
         return 0;
      }
   }
   if(TTY1) to=TTY1;
   else to=bat;

   chop();
   s=tos->tex;
   width=MIN(tos->col,qsize);
   
   for(;i<tos->row;i++) {
      memcpy(q,s+loclin(i,width),width);
      *(q+width)='\0';

      k=0; /* looking for simple redirection: */
      while(*(q+k)!='>' && k<width) k++;

      if(k>=width) { /* add the "to" str if no other redirection: */
         strncat(q," > ",qsize); 
         strncat(q,to,qsize); 
      }
      system_env(q,SHELLENV);
   }
   return(drop());
}

int shellset() /* shellset (hT --- ) */
/* Each line of T defines a NAME=VALUE pair for the environment to be 
   used in upcoming command shells.  

   Lines are converted to an array of null-terminated strings which 
   make up the environment string, and the pointer to the string array 
   is placed in SHELLENV that is used for the command shells run by 
   word shell. 

   To set a null shell environment, run the phrase: "" shellset
*/
{
   char *T;
   static char *env=NULL;
   int k=0,len;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" shellset: ",STRORVOLNOT);
      return 0;
   }
   if(env) {
      mallfree((void *)&env);
      env=NULL;
   }
   if(tos->row==0 || tos->col==0) {
      SHELLENV=NULLENV;
      return(drop());
   }
   textput(); /* puts new-line \n at the end of each chopped line */
   T=tos->tex;
   len=tos->col;

   for(;k<len;k++) {
      if(*T=='\n') *T='\0'; /* replacing \n new-lines with \0 */
      T++;
   }
   env=(char *)memgetn(tos->tex,len);
   SHELLENV=&env;

   return(drop());
}

int shellget() /* shellget ( --- hT) */
/* Return in T the shell environment strings (SHELLENV) being used for 
   command shells run by this program.  T is purged if a NULL shell 
   environment is being used. */
{
   char *env;
   int k=1,len;

   pushq2("",0);
   hand();

   env=*SHELLENV;

   if(env==NULL) {
      return(
         noblanklines() &&
         pushq2("_shellget",9) &&
         naming()
      );
   }
   len=strlen(env);

   while(len) {

      pushq2(env,len);
      env+=(1+len);
      len=strlen(env);

      k++;
   }
   return(
      pushint(k) &&
      pilen() &&
      noblanklines() &&

      pushq2("_shellget",9) &&
      naming()
   );
}

int stkbal() /* stkbal ( --- ) */ 
/* Track the balance of push and pull ops */
{ 
   STKBAL=1; 
   return 1;
}

int stkbaloff() /* -stkbal ( --- ) */ 
/* Ignore the balance of push and pull ops */
{ 
   STKBAL=0; 
   return 1;
}

int strict() /* strict ( --- ) */ 
/* Unknown strings cause error. */
{ 
   STRICT=1; 
   return 1;
}

int strictoff() /* -strict ( --- ) */ 
/* Unknown strings are pushed to the stack. */
{ 
   STRICT=0; 
   return 1; 
}

int stty1()
/* Run word stty in key.v to set options for standard input. */
{
   char *s="stty";

   if(_exists(s)) {
      return(
         pushstr(s) && main1()
      );
   }
   return 1;
}

int sysout1() /* SYSOUT ( --- qFile) */
/* Push to stack the name of current output file. */
{
   return(pushstr(SYSOUT));
}

int _syspath() /* _syspath ( --- qS) */
/* Getting syspath.

   Gets path from command line -s argument, then from TOPS_SYSPATH env
   variable, then assumes local. */
{
/* TOPS_SYSPATH_DEFAULT */
   char *text="INFIX << push '-s' argv any? not \
         IF 'TOPS_SYSPATH' env any? not \
            IF '/usr/local/tops/sys' THEN \
         THEN (qS) ' ' catpath pull infix_set";

   pushstr(text); main1();
   strchop();
   return 1;
}

int system_env(char *command,  char **myenviron) 
/* Running a command shell using the environment pointed to by the
   pointer in myenviron.

   Returns -1 if unsuccessful; usually returns 0 if successful, but
   some functions may return values above 0.

   This function is used in place of the system's function, system(),
   to allow calls by word shell to be made with greater security.  

   It runs a file called SBIN/SH, where directory SBIN can be set at 
   start up with command line argument -d and file name SH can be set 
   at start up with command line argument -c.

   If not set on the command line, SBIN can be defined in boot.v, most
   likely as /bin/ where the regular sh file is may be: sh -> bash.

   Word SBIN_set lets SBIN be reset while the program runs, and word 
   SH_set lets file sh be reset.  But note that certain words defined 
   at start up to use the initial settings will not change (some are
   in boot.v and sys.v).

   Reference for system(): 
      Linux Programmer's Manual entry for system()--execute a shell 
      command (GNU 1998-05-11 SYSTEM(3)).  

   To view Reference on line, run: man system; it says the following 
   about system():

      BUGS [in system(), not this program!]
        It is extremely unfortunate that the libc version of sys
        tem() ignores interrupts.  This makes programs that call
        it from a loop uninterruptable.  This means that for such
        purposes one should not use system() but a private version
        like ... .

        Do not use system() from a program with suid or sgid priv
        ileges, because strange values for some environment vari
        ables might be used to subvert system integrity.

   This substitution for system() is adapted from mysystem() given
   in the Reference.  In addition, it receives a pointer to an envi-
   ronment to use in the call to execve().  For shelling to common
   system functions, a null environment pointer will suffice.

   From man execve:

      A maximum line length of 127 characters is allowed for the
      first line in a #! executable shell script.
*/
{
   static char *envsave;
   char file[512]={0};
   static int lencpy=511;
   char *argv[4]={NULL,"-c",NULL,NULL};
   int pid,status=0;

   if(command==0) return(status);

   strncpy(file,SBIN,lencpy);
   strncat(file,SH,lencpy);

   if(TRACE) {
      gprintf(" system_env    file: %s\n",file);
      gprintf(" system_env command: %s\n",command);
      gprintf(" system_env environ: %lX\n",*myenviron);
   }
   pid=fork();

   if(pid==-1) {
      gprintf(" system_env: fork failed, errno %d  %s\n",errno,
         datetime());

      gprintf(" system_env    file: %s\n",file);
      gprintf(" system_env command: %s\n",command);
      gprintf(" system_env environ: %lX\n",*myenviron);

   /* Wait 5 seconds and try again: */
      gprintf(" system_env: wait 5 seconds and try fork() again\n");
      sleep(5);

      pid=fork();
      
      if(pid==-1) {
         gprintf(" system_env: fork second try failed, errno %d  %s\n",
            errno,datetime());
         return -1;
      }
      else {
         gprintf(" system_env fork: second try succeeded  %s\n",
            datetime());
      }
   }
   envsave=*PROGENV;
   *PROGENV=*myenviron;

   if(pid==0) {
      *(argv  )=(char *)SH;
      *(argv+2)=(char *)command;

      execve(file,argv,(char **)PROGENV);

      gprintf(" system_env failure: execve() returned for:\n");
      gprintf("       file: %s\n",file);
      gprintf("    command: %s\n",command);
      gprintf("    environ: %lX\n",*myenviron);

      *PROGENV=envsave;
      return -1;
   }
   while(1) {
      if(waitpid(pid,&status,0)==-1) {
         if(errno!=EINTR) {
            gprintf(" system_env failure: errno: %d\n",errno);

            *PROGENV=envsave;
            return -1;
         }
      } 
      else {
         if(TRACE) 
            gprintf(" system_env return status: %d\n",status);

         *PROGENV=envsave;
         return(status);
      }
   } 
}

double timed() 
{
   struct timeval tim;
   gettimeofday(&tim,NULL);

   return((double)(1E6*tim.tv_sec + tim.tv_usec)/1E6);
}

int timeprobe() /* timprobe ( --- ) */
/* Write the change in machine time, microseconds, since the last 
   call to this word. */
{
   double t1;
   static double t0=0;
   struct timeval tim;

   gettimeofday(&tim,NULL);

   t1=1E6*(tim.tv_sec) + tim.tv_usec;
   gprintf(" %0.0f ",t1-t0);
   t0=t1;
   return 1;
}

void timeprobe1(int k)
/* Write the change in the kth timer, microseconds, since the last 
   call to this word for the kth timer. */
{
   double t1[10];
   static double t0[10]={0,0,0,0,0,0,0,0,0,0};
   struct timeval tim;

   if(k>9) {
      gprintf(" timeprobe1: timer k=%d is out of range\n",k);
      return;
   }
   gettimeofday(&tim,NULL);

   *(t1+k)=1E6*(tim.tv_sec) + tim.tv_usec;
   gprintf(" %0.0f ",*(t1+k)-*(t0+k));
   *(t0+k)=*(t1+k);
}

int timesys() /* time ( --- S) */
/* S is the program time in seconds to the nearest microsecond.  It 
   includes the integer correction DELTA_T (seconds) needed to make 
   the machine time match real world time to within one second.  

   Word GMTdelta is used to set integer offset DELTA_T to a value 
   given on the stack.

   Word NISTdelta (file net.v) is an example of a word that fetches 
   GMT from an outside source and creates a correction suitable for
   word GMTdelta. */
{  
   struct timeval tim;
   gettimeofday(&tim,NULL);

   return(pushd(_int(1E6*(DELTA_T + tim.tv_sec) + tim.tv_usec)/1E6));
}

int timesys1() /* time1 ( --- S) */
/* S is the machine time in seconds to the nearest microsecond, from
   the machine clock. */
{
   struct timeval tim;
   gettimeofday(&tim,NULL);
   return(pushd(_int(1E6*(tim.tv_sec) + tim.tv_usec)/1E6));
}

int trace1() /* trace ( --- ) */
/* Trace intermediate operations. */
{ 
   TRACE=1; 
   return 1; 
}

int traceflag() /* TRACE ( --- f) */
/* True if trace flag is on. */
{
   if(TRACE) return(pushint(xTRUE)); 
   return(pushint(xFALSE)); 
}

int traceoff1() /* -trace ( --- ) */
{ 
   TRACE=0;
   return 1; 
}

int type1() /* type (hA --- n) */
/* Pushes type of stack item as defined in this enumeration:
      enum stktyp {DEF=1,NUM,MAT,STR,VOL,PTR}; 
   Constants NUM,MAT,STR,VOL, and PTR can be used to compare with n. */
{
   register stkitem *stk;

   if((stk=(peek()))!=NULL) {
      return(pushint((int)pop()->typ));
   }
   stkerr(" ",CANNOTPOP);
   return 0;
}

void unload(char *clean)
/* Close server, reset stty, and run word clean. */
{
   traceoff1();
#ifdef NET
   ntraceoff();
#endif
   wtraceoff();

#ifdef NET
   serverclose();
#endif
   set_stdout();
   pushstr(stty_reset);
   main1();

   pushstr(clean); 
   main1();

   if(ptrnum) {
      mallfree((void *)ptrnum);  /* free p->pat */
      mallfree((void *)&ptrnum); /* free p itself */
   }
}

int usleep1() /* usleep (n --- ) */
/* Sun Jun  2 10:20:49 PDT 2013
   Run Unix usleep(), where n is a number of microseconds. 

      Example: sleeping for one second, 1,000,000 microseconds:

         [dale@kaffia] /opt/tops/tops/src > tops
                  Tops 3.2.1
         Sun Jun  2 10:45:37 PDT 2013
         [tops@kaffia] ready > time 1E6 usleep time swap -

          stack elements:
                0 number: 1.0018901825
          [1] ok!
         [tops@kaffia] ready > exit
         [dale@kaffia] /opt/tops/tops/src > */
{
   int n;

   if(tos->typ!=NUM) {
      stkerr(" usleep: ",NUMNOT);
      return 0;
   }
   popint(&n);
   usleep(n);
   return 1;
}

int xbase() /* xbase ( --- 0 or 1) 1st ( --- 0 or 1) */
/* Push index base to stack. */
{
   return(pushint(XBASE));
}

int xbase1() /* 2nd ( --- 1 or 2) */
/* Push 1+index base to stack. */
{
   return(pushint(1+XBASE));
}

int xbase2() /* 3rd ( --- 2 or 3) */
/* Push 2+index base to stack. */
{
   return(pushint(2+XBASE));
}

int xbase3() /* 4th ( --- 3 or 4) */
/* Push 3+index base to stack. */
{
   return(pushint(3+XBASE));
}
