/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License 
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

/* vt.c  November 2005

Copyright (c) 2005  D. R. Williamson

Words for parsing terminal strings.

References:

   (All http links tested ok 2-20-2006.)

   1. Williams, Paul, "A Parser for DEC's ANSI-compatible video
      terminals," last update September 2001:
         http://www.vt100.net/emu/dec_ansi_parser

   2. Bourke, Paul, "ANSI Standard (X3.64) Control Sequences for
      Video Terminals and Peripherals," March 1989:
         http://astronomy.swin.edu.au/~pbourke/dataformats/ansi.html

   3. _______, "ANSI/VT100 Terminal Control Escape Sequences:"
         http://www.termsys.demon.co.uk/vtansi.htm

   4. Smith, J., "Summary of ANSI Standards for ASCII Terminals,"
      May 1984:
         http://www.inwap.com/pdp10/ansicode.txt

----------------------------------------------------------------------*/

#ifdef NET
#if defined(VT) || defined(VTAIX)

#define _XOPEN_SOURCE 500 /* snprintf */
#include <stdio.h>
#include <string.h>

#include "main.h"
#include "stk.h"

#include "exe.h"
#include "inpo.h"
#include "mat.h"
#include "mem.h"
#include "sys.h"
#include "tag.h"
#include "tex.h"
#include "vt.h"

/* The following includes are needed for dispattr() and getattr(), 
   which are utilities and not critical for running vtparser. */
#ifdef VT
/* For portability, do not define HAVE_TERMINFO.  Terminfo requires
   setupterm() which is not portable.   
   When HAVE_TERMINFO is not defined, Unix termcap is used. */

#define HAVE_TERMINFO

#ifdef HAVE_TERMINFO
/* Using terminfo, Linux */
   #include <termios.h>
   #include <term.h>
   #define TGETSTR(p, NULL) tigetstr(p)

#else
/* Using termcap, Unix */
   #include <termcap.h>
   #define TGETSTR(p, NULL) tgetstr(p, NULL)

#endif
#endif /* VT */

/*--------------------------------------------------------------------*/

/* Local prototypes. */

vtparse_t parser;

void parser_callback1(vtparse_t *parser, vtparse_action_t action,
   unsigned char ch);
void parser_callback2(vtparse_t *parser, vtparse_action_t action,
   unsigned char ch);
void parser_callback3(vtparse_t *parser, vtparse_action_t action,
   unsigned char ch);

int pushattr(char *name, char *string);

int action_prev; /* parser previous action */
int SOCK_CALL; /* client socket for immediate commands */
char *vt; /* address of parsed string */ 
int VTERR;
char *vtmax; /* max vt address while building parsed string */

char /* see man terminfo and file /etc/termcap */ 
   *AB,  /* setab  AB  set background color to #1, using ANSI escape */
   *AF,  /* setaf  AF  set foreground color to #1, using ANSI escape */
   *al,  /* il1    al  insert line */
   *ce,  /* el     ce  clear to end of line */
   *cd,  /* ed     cd  clear to end of screen */
   *cl,  /* clear  cl  clear screen and home cursor */
   *cm,  /* cup    cm  move cursor to row #1 column #2 */
   *cs,  /* csr    cs  change region to line #1 to line #2 */
   *DC,  /* dch    DC  delete #1 characters */
   *dc,  /* dch1   dc  delete character */
   *DL,  /* dl     DL  delete #1 lines */
   *dl,  /* dl1    dl  delete line */
   *do1, /* cud1   do  down one line */
   *ei,  /* rmir   ei   exit insert mode */
   *ho,  /* home   ho  home cursor (if no cup) */
   *IC,  /* ich    IC  insert #1 characters */
   *ic,  /* ich1   ic  insert character */
   *le1, /* cub1   le  move left one space */
   *ll,  /* ll     ll  last line, first column (if no cup) */
   *me,  /* sgr0   me  turn off all attributes */
   *nd,  /* cuf1   nd  non-destructive space (move rt one space) */
   *op,  /* op     op  set default pair to its original value */
   *rc,  /* rc     rc  restore cursor to position of last save_cursor */
   *RI,  /* cuf    RI  move cursor #1 characters to the right */
   *sc,  /* sc     sc  save current cursor position */
   *se,  /* rmso   se  exit standout mode */
   *sf,  /* ind    sf  scroll text up */
   *so,  /* begin standout mode */
   *sr,  /* ri     sr  scroll text down */
   *te,  /* string to end termcap */
   *ti,  /* string to start termcap */
   *up,  /* cursor one line up */
   *ve,  /* cnorm  ve  make cursor appear normal (undo civis/cvvis) */
   *vi,  /* civis  vi  make cursor invisible */
   *vs,  /* cvvis  vs  make cursor very visible */
*ZZLAST;

enum vt_action {
   CLEAR = 1,
   COLLECT = 2,
   CSI_DISPATCH = 3,
   ESC_DISPATCH = 4,
   EXECUTE = 5,
   HOOK = 6,
   IGNORE = 7,
   OSC_END = 8,
   OSC_PUT = 9,
   OSC_START = 10,
   PARAM = 11,
   PRINT = 12,
   PUT = 13,
   START = 0,
   UNHOOK = 14,
   RESERVED = 15 /* created by vtreserved(), not vtparser */
}; 

/*--------------------------------------------------------------------*/

#ifdef VT

int dispattr() /* dispattr (qS --- ) */
/* Display the terminal attribute strings from getattr */
{
   int k,len=0,more=1;
   char attrlen,name[3]={0,0,0},*S;

   if(tos->typ!=STR) {
      stkerr(" dispattr: ",STRNOT);
      return 0;
   }
#ifdef HAVE_TERMINFO
   gprintf(" Terminal attributes using terminfo:");
#else
   gprintf(" Terminal attributes using termcap:");
#endif
   nc();

   S=tos->tex;
   attrlen=tos->col;

   while(more) {
      memcpy(name,S,2);
      S+=2;
      if(*name) {
         len=(int)*S;
         S++;

         gprintf("    %s: ",name);
         if(len) {
            for(k=0;k<len;k++) {
               if(*S<32) gprintf("%x",*S);
               else gprintf("%c",*S);
               S++;
            }
         }
         else gprintf("no attributes found");

         nc();
      }
      else more=0;
   }
   return(drop());
}

int getattr() /* getattr ( --- qS) */
/* Get terminal attributes.

   The format of each attribute string in S is (see pushattr()):
      Name (2 bytes)
      Len (1 byte)
      String (Len bytes)

   S ends with two null bytes: Name = 00 00. */
{
#ifdef HAVE_TERMINFO
   setupterm(NULL,1,NULL);

   AB = TGETSTR("setab",NULL);
   AF = TGETSTR("setaf",NULL);
   al = TGETSTR("il1",NULL);
   cd = TGETSTR("ed",NULL);
   ce = TGETSTR("el",NULL);
   cl = TGETSTR("clear",NULL);
   cm = TGETSTR("cup",NULL);
   cs = TGETSTR("csr",NULL);
   DC = TGETSTR("dch",NULL);
   dc = TGETSTR("dch1",NULL);
   DL = TGETSTR("dl",NULL);
   dl = TGETSTR("dl1",NULL);
   do1= TGETSTR("cud1",NULL);
   ei = TGETSTR("rmir",NULL);
   ho = TGETSTR("home",NULL);
   IC = TGETSTR("ich",NULL);
   ic = TGETSTR("ich1",NULL);
   le1= TGETSTR("cub1",NULL);
   ll = TGETSTR("ll",NULL);
   me = TGETSTR("sgr0",NULL);
   nd = TGETSTR("cuf1",NULL);
   op = TGETSTR("op",NULL);
   rc = TGETSTR("rc",NULL);
   RI = TGETSTR("cuf",NULL);
   sc = TGETSTR("sc",NULL);
   se = TGETSTR("rmso",NULL);
   sf = TGETSTR("ind",NULL);
   so = TGETSTR("smso",NULL);
   sr = TGETSTR("ri",NULL);
   te = TGETSTR("rmcup",NULL);
   ti = TGETSTR("smcup",NULL);
   up = TGETSTR("cuu1",NULL);
   ve = TGETSTR("cnorm",NULL);
   vi = TGETSTR("civis",NULL);
   vs = TGETSTR("cvvis",NULL);
#else
   AB = TGETSTR("AB",NULL);
   AF = TGETSTR("AF",NULL);
   al = TGETSTR("al",NULL);
   cd = TGETSTR("cd",NULL);
   ce = TGETSTR("ce",NULL);
   cl = TGETSTR("cl",NULL);
   cm = TGETSTR("cm",NULL);
   cs = TGETSTR("cs",NULL);
   DC = TGETSTR("DC",NULL);
   dc = TGETSTR("dc",NULL);
   DL = TGETSTR("DL",NULL);
   dl = TGETSTR("dl",NULL);
   do1= TGETSTR("do",NULL);
   ei = TGETSTR("ei",NULL);
   ho = TGETSTR("ho",NULL);
   IC = TGETSTR("IC",NULL);
   ic = TGETSTR("ic",NULL);
   le1= TGETSTR("le",NULL);
   ll = TGETSTR("ll",NULL);
   me = TGETSTR("me",NULL);
   nd = TGETSTR("nd",NULL);
   op = TGETSTR("op",NULL);
   rc = TGETSTR("rc",NULL);
   RI = TGETSTR("RI",NULL);
   sc = TGETSTR("sc",NULL);
   se = TGETSTR("se",NULL);
   sf = TGETSTR("sf",NULL);
   so = TGETSTR("so",NULL);
   sr = TGETSTR("sr",NULL);
   te = TGETSTR("te",NULL);
   ti = TGETSTR("ti",NULL);
   up = TGETSTR("up",NULL);
   ve = TGETSTR("ve",NULL);
   vi = TGETSTR("vi",NULL);
   vs = TGETSTR("vs",NULL);
#endif

   pushattr("AB",AB);
   pushattr("AF",AF); cat();
   pushattr("al",al); cat();
   pushattr("cd",cd); cat();
   pushattr("ce",ce); cat();
   pushattr("cl",cl); cat();
   pushattr("cm",cm); cat();
   pushattr("cs",cs); cat();
   pushattr("DC",DC); cat();
   pushattr("dc",dc); cat();
   pushattr("DL",DL); cat();
   pushattr("dl",dl); cat();
   pushattr("do",do1); cat();
   pushattr("ei",ei); cat();
   pushattr("ho",ho); cat();
   pushattr("IC",IC); cat();
   pushattr("ic",ic); cat();
   pushattr("le",le1); cat();
   pushattr("ll",ll); cat();
   pushattr("me",me); cat();
   pushattr("nd",nd); cat();
   pushattr("op",op); cat();
   pushattr("rc",rc); cat();
   pushattr("RI",RI); cat();
   pushattr("sc",sc); cat();
   pushattr("se",se); cat();
   pushattr("sf",sf); cat();
   pushattr("so",so); cat();
   pushattr("sr",sr); cat();
   pushattr("te",te); cat();
   pushattr("ti",ti); cat();
   pushattr("up",up); cat();
   pushattr("ve",ve); cat();
   pushattr("vi",vi); cat();
   pushattr("vs",vs); cat();

/* Two null ending bytes: */
   nullbyte(); cat();
   nullbyte(); cat();
   return 1;
}

int pushattr(char *name, char *string) 
/* Push terminal attribute name and its string to the stack while
   getattr() is processing attributes.

   The format of each attribute string is:
      name (2 bytes)
      len (1 byte hex)
      string (Len bytes) 

   Example: property cm, 16 bytes long:

      cm 10 E[%i%p1%d;%p2%dH
      |  |  |
      |  |  |__ string (E=1b), length=len
      |  |  
      |  |__ len (hex) of string, 1 byte (255 max)
      |
      |__ name, 2 bytes */
{
   int len=0;

   pushq2(name,2); 

   if(string!=NULL) len=strlen(string); 

   if(len>255) {
      gprintf(
         " pushattr: ","attribute %s len is greater than 255 bytes");
      stkerr("","");
      return 0;
   }
   pushint(len); 
   export1();
   cat();
   if(len) {
      pushstr(string); 
      cat();
   }
   return 1;
}
#endif /* VT */

void parser_callback1(vtparse_t *parser, vtparse_action_t action, 
   unsigned char ch)
/* Used for study.
   This shows vtparser action names received and associated intermediate
   characters and parameters, and is used by word vtparse1. */
{
   int i;

    gprintf("Received action %s (%d), char=0x%02x (%c)\n",
      ACTION_NAMES[action],action,ch,ch);
    gprintf("Intermediate chars: '%s'\n", parser->intermediate_chars);
    gprintf("%d Parameters:\n", parser->num_params);
    for(i = 0; i < parser->num_params; i++)
        gprintf("\t%d\n", parser->params[i]);
    gprintf("\n");
}

void parser_callback2(vtparse_t *parser, vtparse_action_t action, 
   unsigned char ch)
/* Used for study.  Not all responses may be here.
   For incoming parser action, this callback shows what the program's 
   response will be.  It is used by word vtparse2. */
{
   int nchars,nparam;

   nchars=strlen((char *)parser->intermediate_chars);
   nparam=parser->num_params;

   switch(action) {

      case CSI_DISPATCH:
         switch(ch) {

            case '@':
               if(nparam==1) gprintf(" IC: insert %d characters",
                  *parser->params);
               if(nparam==0) gprintf(" IC: insert chars unknown");
               nc();
            break;

            case 'C':
               if(nparam==1)
                  gprintf(" RI: move cursor right %d",*parser->params);
               if(nparam==0) gprintf(" RI: move cursor unknown");
               nc();
            break;

            case 'H':
               if(nparam==2) gprintf(" cm: cursor to %d %d",
                  *parser->params,*(parser->params+1));
               if(nparam==1) gprintf(" cm: cursor to %d",
                  *parser->params);
               if(nparam==0) gprintf(" ho: cursor to home");
               nc();
            break;

            case 'h':
               if(nparam) {
                  switch(*parser->params) {
                     case 25:
                        gprintf(" ve: 25h cursor normal");
                        nc();
                     break;

                     default:
                        gprintf(" CSI_DISPATCH: unknown action for %dh",
                           *parser->params);
                        nc();
                     break;
                  }  
               }
               else {
                  gprintf(" CSI_DISPATCH: unknown action for h");
                  nc();
               }
            break;

            case 'J': 
               gprintf(" cd: clear to end of screen");
               nc();
            break;

            case 'K':
               gprintf(" ce: clear to end of line");
               nc();
            break;

            case 'L':
               if(nparam==1)
                  gprintf(" al: insert line at %d",*parser->params);
               if(nparam==0) gprintf(" al: insert line at unknown");
               nc();
            break;

            case 'l':
               if(nparam) {
                  switch(*parser->params) {
                     case 25:
                        gprintf(" ve: 25l cursor invisible");
                        nc();
                     break;

                     default:
                        gprintf(" CSI_DISPATCH: unknown action for %dl",
                           *parser->params);
                        nc();
                     break;
                  }
               }
               else { 
                  gprintf(" CSI_DISPATCH: unknown action for l");
                  nc();
               }
            break;   

            case 'M':
               if(nparam==1) gprintf(" DL: delete %d lines",
                  *parser->params);
               if(nparam==0) gprintf(" DL: delete lines unknown");
               nc();
            break;

            case 'P':
               if(nparam==1) gprintf(" DC: delete %d characters",
                  *parser->params);
               if(nparam==0) gprintf(" DC: delete chars unknown");
               nc();
            break;

            case 'r':
               if(nparam==2) 
                  gprintf(" cs: region is line %d to line %d",
                  *parser->params,*(parser->params+1));
               if(nparam==1) gprintf(" cs: region is line %d",
                  *parser->params);
               if(nparam==0) gprintf(" cs: region is unknown");
               nc();
            break;

            default:
            gprintf(
               " CSI_DISPATCH: unknown action for %x (%c) nparam: %d",
               ch,ch,nparam);
            nc();
            break;
         }
      break;

      case ESC_DISPATCH:
         switch(ch) {

            case '7':
               gprintf(" sc save cursor position");
               nc();
            break;

            default:
            gprintf(
               " ESC_DISPATCH: unknown action for %x (%c) nparam: %d",
               ch,ch,nparam);
            nc();
            break;
         }
      break;

      case EXECUTE:
         switch(ch) {
            case  7:   /* 07 BEL: bell */
            case  8:   /* 08 BS: backspace */
            case  9:   /* 09 HT: tab */
            case 0x0A: /* 0A LF: new line */
            case 0x0D: /* 0D CR: carriage return */
            case 0x0E: /* 0F SO: activate the G1 character set */
            case 0x0F: /* 0F SI: activate the G0 character set */
               gprintf("%c",ch); 
            break;

            default:
               gprintf(" EXECUTE: unknown action for %x %c%d",
                  ch,ch,*parser->params);
               nc();
            break;
         }
      break;

      case IGNORE:
         default:
      break;

      case HOOK:
         gprintf(" HOOK: unknown action for %x",ch);
         nc();
      break;

      case OSC_PUT:
         gprintf(" OSC_PUT: unknown action for %x",ch);
         nc();
      break;

      case OSC_END:
         gprintf(" OSC_END: unknown action for %x",ch);
         nc();
      break;

      case PRINT:
         gprintf("%c",ch);
      break;

      case PUT:
         gprintf(" PUT: unknown action for %x",ch);
         nc();
      break;

      case START:
         gprintf(" START: unknown action for %c",ch);
         nc();
      break;

      case UNHOOK:
         gprintf(" HOOK: unknown action for %c",ch);
         nc();
      break;
   }
}

void parser_callback3(vtparse_t *parser, vtparse_action_t action, 
   unsigned char ch) 
/* This function fills array vt with the parsed VT response to send 
   to a client.  This function is used by word vtparse3. */
{
   #define CANCEL {vt--; vt--;}
   static char *count_addr;
   static int CMAX=128,count,pad=128;
   int k,nparam;

   if(vt+pad>vtmax) {
      VTERR=1;
      return;
   }
   if(action==PRINT) { /* 0x0C */
      if(action_prev!=PRINT || !(count<CMAX)) {
      /* Start a PRINT counted string of bytes to print.  

         A counted string of bytes to print has this form:

            PR N b1 b2 ... bN 

         where header byte PR=action(PRINT) and N is the count of 
         bytes that follow (N<CMAX).

         When displayed, all bytes in a counted string are given the 
         color attribute of the color attribute command that preceded 
         it, and the color bytes that accompany this action are ignored.

         When using bytes for counts, JavaScript will produce erroneous
         byte values above 127.  

         The only JavaScript function found for extracting byte values 
         is String.charCodeAt().  Above 127, String.charCodeAt() pro-
         duces Unicode values that are useless for doing math (like 
         counting things).

         The Unicode values for bytes 0-127 simply equal 0-127, but byte
         128, for example, does not equal 128.  For this reason, CMAX 
         here restricts the count byte, N, to 127 instead of 255.  When
         the count reaches 127, a new counted string is started. */

         *vt=PRINT;     /* header */
         vt++;         
         count_addr=vt; 
         vt++;    
         count=0;
      }
      *vt=ch;            /* store ch into string */
      vt++;              /* bump addr */

      count++;           /* bump count */
      *count_addr=count; /* store count */

      action_prev=PRINT;
      return;
   }
   action_prev=action;
   nparam=parser->num_params;

   *vt=action; /* header byte */
   vt++;
   *vt=ch;     /* action byte */
   vt++;

   switch(action) {

   /* Each case for action ch sets its own number of parameter bytes 
      in string vt. */

      case CSI_DISPATCH: /* 0x03 */

         switch(ch) { /* set parameter bytes: */

            case '@': /* IC: insert %1 characters */

            case 'A': /* CUU: move cursor up %1 rows */
            case 'B': /* CUD: move cursor down %1 rows */
            case 'C': /* CUF: move cursor right %1 columns */
            case 'D': /* CUB: move cursor left %1 columns */
            case 'E': /* CNL: move cursor down %1 to column 1 */
            case 'F': /* CPL: move cursor up %1 to column 1 */
            case 'G': /* CHA: move cursor to column %1 in current row */
            case 'L': /* IL: insert %1 blank lines */
            case 'M': /* DL: delete %1 lines */
            case 'P': /* DCH: delete %1 characters on current line */
            case 'X': /* ECH: erase %1 characters on current line */

            case 'a': /* HPR: move cursor right %1 columns */
            case 'd': /* VPA: move cursor to %1 row in current column */
            case 'e': /* VPR: move cursor down %1 rows */
            case 'h': /* SM: set cursor mode %1 */
            case 'l': /* RM: reset cursor mode %1 */
            case 'n': /* DSR: status report %1 */
               if(nparam) {
                  *vt=*parser->params; /* sending byte value */
                  vt++;
               } 
               else CANCEL
            break;
           
            case 'c': /* DA: device attributes %1 */

               /* Send this device attribute response to SOCK_CALL right
                  now, and do not return 'c' command to client:
                     VT100 (mem for 24-by-80, inverse video (Ref 4)):
                        ESC[?1;0c 
                        pushstr(\
                           "'1B 5B 3F 31 3B 30 63 0D 00' hexbytes");

                     VT100 (132 col capability, bold+blink+underline
                        +inverse (Ref 4)):
                        ESC[?1;2c 
                        pushstr(\
                           "'1B 5B 3F 31 3B 32 63 0D 00' hexbytes");

                     VT102 (printer port, 132 col, ins/del (Ref 4)):
                        ESC[?6c 
                        pushstr("'1B 5B 3F 36 63 0D 00' hexbytes");
               */
               if(SOCK_CALL>-1) {
                  pushstr("'1B 5B 3F 31 3B 30 63 0D 00' hexbytes");
                  xmain(0);
                  pushint(SOCK_CALL);
                  pushstr("remoteputf");
                  xmain(0);
               }
               CANCEL
            break;

            case 'f': /* HVP: cursor to %1,%2 */
            case 'H': /* CUP: cursor to %1,%2; ho: cursor to home */
               if(nparam==2) { /* cursor to %1,%2 */
                  *vt='2';
                  vt++;
                  for(k=0;k<nparam;k++) {
                     *vt=*(parser->params+k); /* sending byte value */
                     vt++;
                  }
               }
               else { /* cursor to home */
                  *vt='0';
                  vt++;
               }
            break;

            case 'J': /* ED: clear to end of screen */
            case 'K': /* EL: erase in line */
               if(nparam) snprintf(vt,2,"%d",*parser->params);
               else snprintf(vt,2,"%d",0);
               vt++;
            break;

            case 'm': /* AB: set background to color %d 
                         AF: set foreground to color %d

               These are the attributes for m (also see Appendix):
                         AB: 1b[4%p1%dm
                         AF: 1b[3%p1%dm */

               if(nparam>0) {
                  snprintf(vt,2,"%d",nparam); /* N */
                  vt++;
                  for(k=0;k<nparam;k++) {
                     *vt=*(parser->params+k); /* sending byte value */
                     if(*vt==0) *vt=255; /* set zero to 255 */
                     vt++;
                  }
               }
               else { /* case of m with one or no parameters 

The third line below shows [m preceding the text [dale@clacker] which
is the prompt.  It will be assumed that [m is the same as [00m, since
that appears to be its role here.

3790  20 4A 75 6E 20 31 32 20 32 32 3A 30 39 20 1B 5B   Jun 12 22:09 .[
3792  30 30 6D 79 61 6D 6C 5F 69 6E 73 74 61 6C 6C 2E  00myaml_install.
3794  74 78 74 1B 5B 30 30 6D 0D 0A 1B 5B 6D 5B 64 61  txt.[00m...[m[da
3796  6C 65 40 63 6C 61 63 6B 65 72 5D 20 2F 68 6F 6D  le@clacker] /hom
3798  65 2F 64 61 6C 65 20 3E 20 00 00 00 00 00 00 00  e/dale > .......
*/
               /* These outputs are for [00m and [m: */
                  *vt='0'; /* N */
                  vt++;
               }
            break;

            case 'r': /* cs: region is line %d to line %d */
               if(nparam==2) { /* region is %1,%2 */
                  *vt='2'; /* N */
                  vt++;
                  for(k=0;k<nparam;k++) {
                     *vt=*(parser->params+k);
                     vt++;
                  }
               }
               else {
                  CANCEL
                  gprintf(" CSI_DISPATCH: unknown region for r");
                  nc();
               }
            break;

            case 's': /* ?: save cursor location */
            case 'u': /* ?: restore cursor location */
            break;

            default:
            CANCEL
            gprintf(
               " CSI_DISPATCH: unknown action for %x (%c) nparam: %d",
               ch,ch,nparam);
            nc();
            break;
         }
      break;

      case ESC_DISPATCH: /* 0x04 */

         switch(ch) { /* set parameter bytes: */

            case '=': /* ei: exit insert mode (seen as E[? 1 h E=) */
            case '>': /* se: exit standout mode (seen as E[? 1 l E>) */
            case '7': /* sc: save cursor position */
            case '8': /* rc: restore cursor position to last-saved */
            case 'D': /* IND: linefeed */
            case 'E': /* NEW: newline */
            case 'H': /* HTS: set tab stop at current column */
            case 'M': /* RI: reverse linefeed */
            break;

            case 'h': /* mode switch to %1 */
            case 'n': /* status report %1 */
               if(nparam) {
                  *vt=*parser->params; /* sending byte value */
                  vt++;
               } 
            break;

            default:
            CANCEL
            gprintf(
               " ESC_DISPATCH: unknown action for %x (%c) nparam: %d",
               ch,ch,nparam);
            nc();
            break;
         }
      break;

      case EXECUTE: /* 0x05 */

         switch(ch) { /* set parameter bytes: */

            case    7: /* 07 BEL: bell */
            case    8: /* 08 BS: backspace */
            case    9: /* 09 HT: tab */
            case 0x0A: /* 0A LF: new line */
            case 0x0D: /* 0D CR: carriage return */
            case 0x0E: /* 0E SO: activate the G1 character set */
            case 0x0F: /* 0F SI: activate the G0 character set */
               gprintf("%c",ch);
            break;

            case    0: /* 14 DC4: begin raw mode */
               vt--;
               *vt=0x14;
               gprintf("%c",*vt);
               vt++;
            break;

            default:
            CANCEL
            gprintf(
               " EXECUTE: unknown action for %x (%c) nparam: %d",
               ch,ch,nparam);
            nc();
            break;
         }
      break;

      default:
         CANCEL
         gprintf(" parser_callback3: unknown action %d cancelled",ch);
         nc();
      break;
   }
   #undef CANCEL
}

int vtparse1() /* vtparse1 (qT --- qT) */
/* This uses a callback that shows the program's response to parsed
   actions.  T is unchanged.
   Usage:
      (hT) "vtparse" >stk eview */
{
   char cr=13,null=0;

/* Replace 00 with 0D: */
   chrrpl(tos->tex,tos->row*tos->col,null,cr);

   vtparse_init(&parser, parser_callback1);
   vtparse(&parser,(unsigned char *)tos->tex,tos->col);
   return 1;
}

int vtparse2() /* vtparse2 (qT --- qT) */
/* This uses a callback that shows the parser response.  T is unchanged.
   Usage:
      (hT) "vtparse2" >stk eview */
{
   char cr=13,null=0;

/* Replace 00 with 0D: */
   chrrpl(tos->tex,tos->row*tos->col,null,cr);

   vtparse_init(&parser, parser_callback2);
   vtparse(&parser,(unsigned char *)tos->tex,tos->col);
   return 1;
}

int vtparse3() /* vtparse3 (qS nSocket --- qS1) */
/* This uses parser_callback3() to format S for sending to a client in 
   the form of S1.
   Immediate commands are sent directly to client Socket and are not 
   returned in S1.  For example, see parser_callback3(), case 'c'. */
{
   const int BUFSIZE=8192;

   if(!popint(&SOCK_CALL)) return 0;

/* Remove 00: */
/*   tos->col=rmnull(tos->tex,tos->col); */

/* Put STR S1 on stack to receive parsed data to be sent to client. */
   if(!strstk(MAX(BUFSIZE,2*tos->col),"_vtparse3")) return 0;
   vt=tos->tex;
   vtmax=vt+tos->col;

   action_prev=-1;
   VTERR=0;

   vtparse_init(&parser, parser_callback3);
   vtparse(&parser,(unsigned char *)(tos-1)->tex,(tos-1)->col);

   if(VTERR) {
      stkerr(" vtparse3: ","insufficient space for parsed string");
      return 0;
   }
   if(vt+1<vtmax) {
      tos->col=vt-tos->tex;
      *(tos->tex+tos->col)='\0';
   }
   return(lop());
}

int vtreserved() /* vtreserved (qS n --- qS1) */
/* Place a RESERVED byte and an action byte on the left of S so it can 
   be sent along with, and later distinguished from, other vtparser 
   strings. 
   Example (where RESERVED=15=0x0F): 
      "Status message" 1 vtreserved .hex
      0F 01 53 74 61 74 75 73 20 6D 65 73 73 61 67 65 */
{
   return(
      export1() &&
      pushint(RESERVED) &&
      export1() &&
      swap() &&
      rot() &&
      cat() && 
      cat()
   );
}

#endif
#endif

/*--------------------------------------------------------------------*/

/* Appendix */

/*

This is terminfo for screen:
/packages/screen-3.9.15/terminfo/screencap:

SC|screen|VT 100/ANSI X3.64 virtual terminal:\
        :am:xn:ms:mi:G0:km:\
        :DO=\E[%dB:LE=\E[%dD:RI=\E[%dC:UP=\E[%dA:bs:bt=\E[Z:\
        :cb=\E[1K:cd=\E[J:ce=\E[K:cl=\E[H\E[J:cm=\E[%i%d;%dH:ct=\E[3g:\
        :do=^J:nd=\E[C:pt:rc=\E8:rs=\Ec:sc=\E7:st=\EH:up=\EM:\
        :le=^H:bl=^G:cr=^M:it#8:ho=\E[H:nw=\EE:ta=^I:is=\E)0:\
        :li#24:co#80:us=\E[4m:ue=\E[24m:so=\E[3m:se=\E[23m:\
        :mb=\E[5m:md=\E[1m:mr=\E[7m:me=\E[m:sr=\EM:al=\E[L:\
        :AL=\E[%dL:dl=\E[M:DL=\E[%dM:cs=\E[%i%d;%dr:dc=\E[P:\
        :DC=\E[%dP:im=\E[4h:ei=\E[4l:IC=\E[%d@:\
        :ks=\E[?1h\E=:ke=\E[?1l\E>:vb=\Eg:\
        :ku=\EOA:kd=\EOB:kr=\EOC:kl=\EOD:kb=^H:\
        :k1=\EOP:k2=\EOQ:k3=\EOR:k4=\EOS:k5=\E[15~:k6=\E[17~:\
        :k7=\E[18~:k8=\E[19~:k9=\E[20~:k;=\E[21~:F1=\E[23~:F2=\E[24~:\
        :kh=\E[1~:kI=\E[2~:kD=\E[3~:kH=\E[4~:@7=\E[4~:kP=\E[5~:\
        :kN=\E[6~:eA=\E(B\E)0:as=^N:ae=^O:ti=\E[?1049h:te=\E[?1049l:\
        :vi=\E[?25l:ve=\E[34h\E[?25h:vs=\E[34l:\
        :Co#8:pa#64:AF=\E[3%dm:AB=\E[4%dm:op=\E[39;49m:AX:\
        :ac=``aaffggjjkkllmmnnooppqqrrssttuuvvwwxxyyzz{{||}}~~..--++,,hhII00:
SB|screen-bce|VT 100/ANSI X3.64 virtual terminal with bce:\
        :ut:tc=screen:
SH|screen-s|VT 100/ANSI X3.64 virtual terminal with hardstatus line:\
        :ts=\E_:fs=\E\\:ds=\E_\E\\:tc=screen:


Miscellaneous Notes.

Set Display Attributes

Set Attribute Mode	<ESC>[{attr1};...;{attrn}m

    * Sets multiple display attribute settings. The following lists standard attributes:

0	Reset all attributes
1	Bright
2	Dim
4	Underscore	
5	Blink
7	Reverse
8	Hidden

	Foreground Colours
30	Black
31	Red
32	Green
33	Yellow
34	Blue
35	Magenta
36	Cyan
37	White

	Background Colours
40	Black
41	Red
42	Green
43	Yellow
44	Blue
45	Magenta
46	Cyan
47	White

               When multiplied by 10, the 4 and 3 are used to as 
               offsets as learned from the following:

                  From http://linuxgazette.net/issue65/padala.html:

                     fg is one of the following:
                        30 Black
                        31 Red
                        32 Green
                        33 Yellow
                        34 Blue
                        35 Magenta
                        36 Cyan
                        37 White

                     bg is one of the following:
                        40 Black
                        41 Red
                        42 Green
                        43 Yellow
                        44 Blue
                        45 Magenta
                        46 Cyan
                        47 White

               These are colors from man terminfo:

                  Color       #define       Value       RGB
                  black     COLOR_BLACK       0     0, 0, 0
                  red       COLOR_RED         1     max,0,0
                  green     COLOR_GREEN       2     0,max,0
                  yellow    COLOR_YELLOW      3     max,max,0
                  blue      COLOR_BLUE        4     0,0,max
                  magenta   COLOR_MAGENTA     5     max,0,max
                  cyan      COLOR_CYAN        6     0,max,max
                  white     COLOR_WHITE       7     max,max,max

               Table fg shows that foreground colors are the values in
               the table above plus 30; Table bg shows that background
               colors are the values in the table above plus 40.

               This shows color escape sequences in a directory listing
               (taken from vi, ^[ is an escape (1b)):

                   2866 Nov 29 2002 ^[[00mbabble.doc^[[00m^M
                  53237 Oct  9 2003 ^[[01;35mbaby goo 10-7-3.jpg^[[00m^M
                   2813 Jan  3 2005 ^[[00mbackspace.txt^[[00m^M

               From the parser, the second entry gives:

                  Received action CSI_DISPATCH 3, char=0x6d m
                  Intermediate chars: ''
                  2 Parameters:
                   1
                   35 

                The 35 agrees with magenta in the fg table above.

                All entries use 1b[00m just before ^M at the end of each
                line.  From the parser, 1b[00m appears as one parameter
                equal to zero:

                   Received action CSI_DISPATCH 3, char=0x6d m
                   Intermediate chars: ''
                   1 Parameters:
                    0

                This is taken to mean a cancellation of colors set 
                previously, or a resumption of standard colors (re-
                search is continuing to resolve this).  

                Send color parameters in a color string, and allow for
                N parameters by sending a count:

                   CSI ch N p1 p2 ... bN 

                where CSI=header=action(CSI_DISPATCH)
                      ch='m'
             */


/*

Tab Control

Set Tab 		<ESC>H

    * Sets a tab at the current position.

Clear Tab 		<ESC>[g

    * Clears tab at the current position.

Clear All Tabs 		<ESC>[3g

    * Clears all tabs.



Erasing Text

Erase End of Line	<ESC>[K

    * Erases from the current cursor position to the end of the current line.

Erase Start of Line	<ESC>[1K

    * Erases from the current cursor position to the start of the current line.

Erase Line		<ESC>[2K

    * Erases the entire current line.

Erase Down		<ESC>[J

    * Erases the screen from the current line down to the bottom of the screen.

Erase Up		<ESC>[1J

    * Erases the screen from the current line up to the top of the screen.

Erase Screen		<ESC>[2J

    * Erases the screen with the background colour and moves the cursor to home
*/
