// Distance.hpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

#ifndef TOVERO_MATH_DISTANCE_HPP_
#define TOVERO_MATH_DISTANCE_HPP_

#include <tovero/math/geometry/Quantity.hpp>
#include <tovero/math/geometry/Unitless.hpp>

namespace Roan_trail
{
  namespace Tovero_math
  {
    class Distance : public Quantity
    {
    public:
      // constructors
      Distance() : Quantity(0.0) {}
      //   meant for creating a distance from a value and a unit
      Distance(double value, const Distance& unit) : Quantity(value * unit.value()) {}
      static inline Distance from_value(double value);
      //  Note: use compiler generated copy constructor and destructor
      // operators
      inline Distance& operator=(const Distance& q);
      Distance operator+(const Distance& q) const { return Distance::from_value(value() + q.value()); }
      Distance operator-(const Distance& q) const { return Distance::from_value(value() - q.value()); }
      Distance operator-() const { return Distance::from_value(-value()); }
      Distance operator*(const Unitless& factor) const { return Distance::from_value(factor.value() * value()); }
      Distance operator*(double factor) const { return Distance::from_value(factor * value()); }
      Unitless operator/(const Distance &divisor) const
      { return Unitless::from_value(value() / divisor.value()); }
      Distance operator/(const Unitless &divisor) const
      { return Distance::from_value(value() / divisor.value()); }
      Distance operator/(double divisor) const { return Distance::from_value(value() / divisor); }
      inline Distance& operator+=(const Distance& addend);
      inline Distance& operator-=(const Distance& subtrahend);
      inline Distance& operator*=(const Unitless& factor);
      inline Distance& operator*=(double factor);
      inline Distance& operator/=(const Unitless& factor);
      inline Distance& operator/=(double factor);
      bool operator<(const Distance& q) const { return (value() < q.value()); }
      bool operator>(const Distance& q) const { return (value() > q.value()); }
      // class constants
      //   SI
      static const Distance meter;
      static const Distance millimeter;
      //   US
      static const Distance foot;
      static const Distance inch;
    };

    // free functions

    inline Distance operator*(const Unitless& u, const Distance& d)
    { return Distance::from_value(d.value() * u.value()); }

    inline Distance operator*(double u, const Distance& d) { return Distance::from_value(d.value() * u); }

    //
    // Inline definitions
    //

    //
    // Constructors
    //

    inline Distance Distance::from_value(double value)
    {
      Distance d;
      d.set_value(value);

      return d;
    }

    //
    //   Operators
    //

    inline Distance& Distance::operator=(const Distance& d)
    {
      if (this != &d)
      {
        set_value(d.value());
      }

      return *this;
    }

    inline Distance& Distance::operator+=(const Distance& addend)
    {
      set_value(value() + addend.value());

      return *this;
    }

    inline Distance& Distance::operator-=(const Distance& subtrahend)
    {
      set_value(value() - subtrahend.value());

      return *this;
    }

    inline Distance& Distance::operator*=(const Unitless& factor)
    {
      set_value(value() * factor.value());

      return *this;
    }

    inline Distance& Distance::operator*=(double factor)
    {
      set_value(value() * factor);

      return *this;
    }

    inline Distance& Distance::operator/=(const Unitless& factor)
    {
      set_value(value() / factor.value());

      return *this;
    }

    inline Distance& Distance::operator/=(double factor)
    {
      set_value(value() / factor);

      return *this;
    }
  }
}

#endif // TOVERO_MATH_DISTANCE_HPP_
